/*
 * jNPad v0.3 - jNPad's an Simple Text Editor written in Java
 *
 * Copyright (C) 2014-2017  rgs
 *
 * Require JDK 1.6 (or later)
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 *
 *
 * Info, Questions, Suggestions & Bugs Report to rgsevero@gmail.com
 */

package jnpad.ui.toolbar;

import java.awt.Color;
import java.awt.Container;
import java.awt.Graphics;
import java.awt.event.ActionEvent;
import java.awt.event.MouseEvent;
import java.util.logging.Level;
import java.util.logging.Logger;

import javax.swing.Icon;
import javax.swing.ImageIcon;
import javax.swing.JComponent;
import javax.swing.JPopupMenu;
import javax.swing.SwingUtilities;
import javax.swing.event.MouseInputAdapter;

import jnpad.JNPadBundle;
import jnpad.ui.plaf.LAFUtils;

/**
 * The Class PopupButton2.
 *
 * @version 0.3
 * @since   jNPad v0.1
 */
public class PopupButton2 extends ToolBarButton {
  private JPopupMenu          popupMenu;
  private Icon                _standardIcon;
  private Icon                _realIcon;
  private boolean             showSplitter;
  private String              dropDownToolTip;
  private String              tooltip;

  /** Logger */
  private static final Logger LOGGER           = Logger.getLogger(PopupButton2.class.getName());

  /** UID */
  private static final long   serialVersionUID = -7604550184286894909L;

  /**
   * Instantiates a new popup button2.
   */
  public PopupButton2() {
    super();
    MouseHandler mouseHandler = new MouseHandler();
    addMouseMotionListener(mouseHandler);
    addMouseListener(mouseHandler);
  }

  /**
   * Instantiates a new popup button2.
   *
   * @param popup JPopupMenu
   * @param icon Icon
   */
  public PopupButton2(JPopupMenu popup, Icon icon) {
    this();
    popupMenu = popup;
    setIcon(icon);
  }

  /**
   * Gets the popup menu.
   *
   * @return JPopupMenu
   */
  public JPopupMenu getPopupMenu() {return popupMenu;}

  /**
   * Sets the popup menu.
   *
   * @param popup JPopupMenu
   */
  public void setPopupMenu(JPopupMenu popup) {
    popupMenu = popup;
  }

  /**
   * Sets the icon.
   *
   * @param icon Icon
   * @see javax.swing.AbstractButton#setIcon(javax.swing.Icon)
   */
  @Override
  public final void setIcon(Icon icon) {
    _realIcon = icon;
    if (icon != null && icon instanceof ImageIcon)
      super.setIcon(_standardIcon = new DropDownIcon( (ImageIcon) icon));
    else
      super.setIcon(icon);
  }

  /**
   * Show popup.
   */
  protected void showPopup() {
    if (popupMenu != null) {
      SwingUtilities.updateComponentTreeUI(popupMenu);
      popupMenu.pack();
      popupMenu.show(this, 0, getHeight());
    }
  }

  /**
   * Paint.
   *
   * @param g Graphics
   * @see javax.swing.JComponent#paint(java.awt.Graphics)
   */
  @Override
  public void paint(Graphics g) {
    super.paint(g);
    try {
      if (showSplitter) {
        showSplitter(LAFUtils.getControlDkShadow(), g, getSplitterPosn(), 1, getHeight() - 4);
        showSplitter(LAFUtils.getControlHighlight(), g, getSplitterPosn() + 1, 1, getHeight() - 4);
      }
    }
    catch (Exception ex) {
      LOGGER.log(Level.WARNING, ex.getMessage(), ex);
    }
  }

  /**
   * Gets the splitter posn.
   *
   * @return int
   */
  private int getSplitterPosn() {
    return getIconPosn() + _realIcon.getIconWidth() + 3;
  }

  /**
   * Gets the icon posn.
   *
   * @return int
   */
  private int getIconPosn() {
    int x = (this.getWidth() - _standardIcon.getIconWidth()) / 2;
    return x;
  }

  /**
   * Sets the drop down tool tip text.
   *
   * @param dropDownToolTip String
   */
  public void setDropDownToolTipText(String dropDownToolTip) {
    this.dropDownToolTip = dropDownToolTip;
  }

  /**
   * Show splitter.
   *
   * @param c Color
   * @param g Graphics
   * @param x int
   * @param y int
   * @param height int
   */
  private void showSplitter(Color c, Graphics g, int x, int y, int height) {
    g.setColor(c);
    g.drawLine(x, y + 0, x, y + height);
  }

  /**
   * Show splitter.
   *
   * @param show boolean
   */
  private void showSplitter(boolean show) {
    if (show && !showSplitter) {
      showSplitter = true;
      repaint();
      String tt = null;
      Container parent = getParent();
      if (parent instanceof JComponent)
        tt = dropDownToolTip;
      if (tt == null)
        tt = JNPadBundle.getString("PopupButton.select"); //$NON-NLS-1$
      super.setToolTipText(tt);
    }
    else if (!show && showSplitter) {
      showSplitter = false;
      repaint();
      super.setToolTipText(tooltip);
    }
  }

  /**
   * Sets the tool tip text.
   *
   * @param text String
   * @see javax.swing.JComponent#setToolTipText(java.lang.String)
   */
  @Override
  public void setToolTipText(String text) {
    super.setToolTipText(text);
    tooltip = text;
  }

  /**
   * Fire action performed.
   *
   * @param event ActionEvent
   * @see javax.swing.AbstractButton#fireActionPerformed(java.awt.event.ActionEvent)
   */
  @Override
  protected void fireActionPerformed(ActionEvent event) {
    if (showSplitter) {
      showPopup();
    }
    else {
      super.fireActionPerformed(event);
    }
  }

  //////////////////////////////////////////////////////////////////////////////
  /**
   * The Class MouseHandler.
   */
  private class MouseHandler extends MouseInputAdapter {
    /**
     * Mouse moved.
     *
     * @param e MouseEvent
     * @see java.awt.event.MouseAdapter#mouseMoved(java.awt.event.MouseEvent)
     */
    @Override
    public void mouseMoved(final MouseEvent e) {
      if (_standardIcon != null)
        showSplitter(e.getX() >= getSplitterPosn());
    }

    /**
     * Mouse entered.
     *
     * @param e MouseEvent
     * @see java.awt.event.MouseAdapter#mouseEntered(java.awt.event.MouseEvent)
     */
    @Override
    public void mouseEntered(final MouseEvent e) {
      if (_standardIcon != null)
        showSplitter(e.getX() >= getSplitterPosn());
    }

    /**
     * Mouse exited.
     *
     * @param e MouseEvent
     * @see java.awt.event.MouseAdapter#mouseExited(java.awt.event.MouseEvent)
     */
    @Override
    public void mouseExited(final MouseEvent e) {
      if (_standardIcon != null)
        showSplitter(false);
    }
  }
  //////////////////////////////////////////////////////////////////////////////

}
