/**
 * $Id: FileSeeker.java,v 1.12 2001/10/09 02:45:41 groomed Exp $
 *
 * Copyright (C) 1998-2001 groomed <groomed@users.sourceforge.net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

package redlight.client;
 
import java.awt.*;
import java.awt.event.*;
import java.util.Stack;
import java.util.Vector;
import java.io.IOException;

import javax.swing.*;
import javax.swing.event.*;
import javax.swing.table.*;

import redlight.hotline.*;
import redlight.utils.DebuggerOutput;
import redlight.utils.BytesFormat;
import redlight.utils.TextUtils;
import redlight.graphics.Spinner;

/**
 * Renderer for the table cells.
 */

class FileNameTableCellRenderer extends DefaultTableCellRenderer {

    static Color idleColor = new Color(0xababab);

    FileNameTableCellRenderer() {

	setOpaque(true);

    }
    
    public Component getTableCellRendererComponent(JTable table,
                                                   Object value,
                                                   boolean sel,
                                                   boolean hasFocus,
                                                   int row,
                                                   int col) {

        HLProtocol.FileListComponent file = 
            (HLProtocol.FileListComponent) value;

	setBackground(sel ? table.getSelectionBackground() :
		      table.getBackground());

        setText(file.fileName);

        if(Main.iconFile != null) 
            setIcon(Main.iconFile.getIconForFile(file));

	return this;

    }

}



/**
 * This is the interface for the file search tool, which
 * recursively descends into directories looking for
 * files that match some specified search string.
 */
public class FileSeeker extends AbstractTableModel
    implements Runnable, 
	       ActionListener, 
	       WindowListener, 
	       Child, 
	       MouseListener {
    Machine rlm;
    Container contentPane;
    Vector results;
    JLabel info;
    JTable list;
    JTextField find;
    JButton stop;
    JFrame f;
    Spinner spinner;
    String path, search;
    Thread seeker;
    Stack visitedPaths;
    DepthFirstFileSearch fileSearch;

    public FileSeeker(Machine machine, String pa) {
	rlm = machine;
	path = pa;
	results = new Vector();

	Font guiFont = (Font) Main.rlo.getProperty("Font.gui");

	f = new JFrame("Search (" + rlm.getServerName() + ")");
	contentPane = f.getContentPane();
	GridBagLayout gbl = new GridBagLayout();
	GridBagConstraints gbc = new GridBagConstraints();
	JPanel p = new JPanel();
	JLabel l1 = new JLabel("Search for:");
	JLabel l2 = new JLabel("in " + path);
	find = new JTextField("", 12);
	stop = new JButton("", new ImageIcon(Main.rlo.getImageSet("FunctionIcons")[Options.STOP_ICON]));
	JPanel spinnerPanel = new JPanel();
	list = new JTable(this);
	FileTCTableCellRenderer ftcTableCellRenderer = 
	    new FileTCTableCellRenderer();
	info = new JLabel("Idle.");
	spinner = new Spinner(Main.rlo.getImageSet("Spinner"), 
			      100, 16, 16);

	contentPane.setLayout(gbl);
	p.setLayout(new FlowLayout(FlowLayout.LEFT));
	spinnerPanel.setLayout(new FlowLayout(FlowLayout.RIGHT));
	stop.setEnabled(false);
        stop.setActionCommand("Stop");
	info.setFont((Font) Main.rlo.getProperty("Font.info"));
	list.setFont((Font) Main.rlo.getProperty("Font.list"));
	list.getColumnModel().getColumn(0).setPreferredWidth(600);
	list.getColumnModel().getColumn(1).setPreferredWidth(150);
	list.getColumnModel().getColumn(2).setPreferredWidth(300);
	list.getColumn("Size").
	    setCellRenderer(new FileSizeTableCellRenderer());
	list.getColumn("Name").
	    setCellRenderer(new FileNameTableCellRenderer());
	list.setBackground(rlm.getSchemeColor("background"));
	list.setForeground(rlm.getSchemeColor("foreground"));
	list.setShowVerticalLines(false);
	list.setShowHorizontalLines(false);
        list.setShowGrid(false);
        list.setRowHeight(18);
        list.setIntercellSpacing(new Dimension(0, 0));
	find.setCaretPosition(0);
	l1.setFont(guiFont);
	l2.setFont(guiFont);
	stop.setFont(guiFont);

	list.addMouseListener(this);
	stop.addActionListener(this);
	find.addActionListener(this);
	f.addWindowListener(this);

	p.add(stop);
	p.add(l1);
	p.add(find);
	if(!path.equals(""))
	    p.add(l2);
	gbc.weightx = 1;
	gbc.fill = GridBagConstraints.HORIZONTAL;
	contentPane.add(p, gbc);
	spinnerPanel.add(spinner);
	gbc.gridx = 1;
	gbc.weightx = 0;
	gbc.fill = GridBagConstraints.NONE;
	contentPane.add(spinnerPanel, gbc);
	gbc.gridx = 0; gbc.gridy = 1;
	gbc.gridwidth = 2;
	gbc.weightx = 1; gbc.weighty = 1;
	gbc.insets = 
	    (Insets) Main.rlo.getProperty("BorderInsets");
	gbc.fill = GridBagConstraints.BOTH;
        JScrollPane scrollPane = new JScrollPane(list);
	contentPane.add(scrollPane, gbc);
        scrollPane.getViewport().setBackground(list.getBackground());

	gbc.gridy = 2;
	gbc.weightx = 0; gbc.weighty = 0;
	gbc.fill = GridBagConstraints.HORIZONTAL;
	contentPane.add(info, gbc);

	f.pack();
	f.setSize(440, 340);

	Dimension d = Toolkit.getDefaultToolkit().getScreenSize();
	Dimension w = f.getSize();

	f.setLocation((d.width / 2) - (w.width / 2), 
		      (d.height / 2) - (w.height / 2));
        find.requestFocus();
	f.show();

	rlm.getInterface().registerChild(this);
    }

    void itemActivated() {
	int index = list.getSelectedRow();

	if(index != -1) {

	    HLProtocol.FileListComponent file = 
		((Item) results.elementAt(index)).file;
	    String path = ((Item) results.elementAt(index)).path;

            String fullname = path + HLProtocol.DIR_SEPARATOR + file.fileName;
            DebuggerOutput.debug("FileSeeker: opening " + fullname);

	    if(file.fileType.equals("fldr"))
		DirOpener.open(rlm, fullname);
	    else
		new DownloadInterface(rlm, 
                                      fullname, 
                                      file);

	}

    }

    public void run() {

        final String findString = search.toUpperCase();
        final FileSeeker seek = this;

	results.removeAllElements();
	stop.setEnabled(true);
        
	try {
            
	    visitedPaths = new Stack();
	    spinner.start();
            FileSearchAction finder = new FileSearchAction() {

                    public void fileFound(String path, HLProtocol.FileListComponent file) {
                        
                        info.setText(path);
                        
                        if(file.fileName.toUpperCase().indexOf(findString) != -1) {
                            
                            results.addElement(new Item(path, file));
                            SwingUtilities.invokeLater(new UpdateList(new TableModelEvent(seek)));
                            
                        }
                        
                    }
                    
                };
            
            fileSearch = new DepthFirstFileSearch();
            fileSearch.descend(rlm.getHLC(), path, 0, 0, finder);

	} catch(InterruptedException e) {
	} catch(IOException e) {
        } catch(HLException e) {
	} finally {

	    spinner.stop();
	    stop.setEnabled(false);
	    info.setText("Idle" + (results.size() > 0 ? 
			 " (" + results.size() + " items found)." :
			 "."));

	}

    }

    class Item {

	protected String path;
	protected HLProtocol.FileListComponent file;

	Item(String p, HLProtocol.FileListComponent f) {

	    path = p;
	    file = f;

	}

    }

    class UpdateList implements Runnable {

	TableModelEvent tev;

	UpdateList(TableModelEvent t) {

	    tev = t;

	}

	public void run() {

	    list.tableChanged(tev);
	    list.repaint();

	}

    }

    /**
     * Following methods extend AbstractTreeTableModel.
     */
    public int getColumnCount() { 

	return 3; 

    }

    public int getRowCount() { 

	return results.size(); 

    }

    public Object getValueAt(int row, int col) {	

	Item i = (Item) results.elementAt(row);

	switch(col) {

	case 0:
	    return i.file;
	case 1:
	    return i.file.fileType.equals("fldr") ? 
		"(" + i.file.fileSize + ")" : 
                    " " + BytesFormat.format(i.file.fileSize);
	case 2:
	    return i.path;
	}

	return null;

    }

    public Class getColumnClass(int col) {

        switch(col) {
        case 0:
            return HLProtocol.FileListComponent.class;
        default:
            return String.class;

        }

    }

    public String getColumnName(int col) {

	if(col == 0) return "Name";
	if(col == 1) return "Size";
	if(col == 2) return "Where";
	return "error: col = "+col;

    }

    /**
     * Following method implements ActionListener.
     */
    public void actionPerformed(ActionEvent e) {

	if(e.getSource() == find) {
            
	    search = find.getText().trim();

	    if(seeker != null) {

		if(seeker.isAlive()) {

		    fileSearch.interrupt();
                    
		    try {

			seeker.join();

		    } catch(InterruptedException ex) {}

		}

            }

	    if(!search.trim().equals("")) {

                if(fileSearch != null)
                    fileSearch.interrupt();

		seeker = new Thread(this);
		seeker.start();

	    }
	
	} else if(e.getActionCommand().equals("Stop")) {

	    if(seeker.isAlive()) {

                if(fileSearch != null)
                    fileSearch.interrupt();

		try {

		    seeker.join();

		} catch(InterruptedException ex) {}

		seeker = null;

	    }

	}

    }

    /**
     * Following methods implement WindowListener.
     */
    public void windowActivated(WindowEvent e) {}
    public void windowDeactivated(WindowEvent e) {}
    public void windowClosed(WindowEvent e) { }
    public void windowClosing(WindowEvent e) {

	rlm.getInterface().unregisterChild(this);
	close();

    }
    public void windowIconified(WindowEvent e) {}
    public void windowDeiconified(WindowEvent e) {}
    public void windowOpened(WindowEvent e) {}

    /**
     * Following methods implement MouseListener.
     */
    public void mouseClicked(MouseEvent e) {

	if(e.getClickCount() == 2)
	    itemActivated();

    }
    public void mouseEntered(MouseEvent e) {}
    public void mouseExited(MouseEvent e) {}
    public void mousePressed(MouseEvent e) {}
    public void mouseReleased(MouseEvent e) {}

    /**
     * Following methods implement Child.
     */
    public void displayPropertyChanged(String what, final Object property) {

	if(what.equals("listfont")) {

            SwingUtilities.invokeLater(new Runnable() {

                    public void run() {

                        list.setFont((Font) property);

                    }

                });

        } else {

            SwingUtilities.invokeLater(new Runnable() {

                    public void run() {

                        list.setBackground(rlm.getSchemeColor("background"));
                        list.setForeground(rlm.getSchemeColor("foreground"));

                    }
                    
                });

	}

    }

    public void close() {

	if(seeker != null)
            if(fileSearch != null)
                fileSearch.interrupt();

	f.dispose();

    }

}





