/**
 * $Id: JTreeTable.java,v 1.7 2001/10/06 19:59:02 groomed Exp $
 *
 * Copyright (C) 1998-2001 groomed <groomed@users.sourceforge.net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

package redlight.client;

import javax.swing.*;
import javax.swing.event.*;
import javax.swing.tree.*;
import javax.swing.table.*;
import javax.swing.border.*;

import java.awt.*;
import java.awt.dnd.*;
import java.awt.datatransfer.*;

import java.awt.event.MouseEvent;

import java.util.EventObject;
import java.io.File;
import java.io.IOException;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.ArrayList;

import redlight.hotline.HLProtocol;
import redlight.utils.DebuggerOutput;

/**
 * This example shows how to create a simple JTreeTable component, 
 * by using a JTree as a renderer (and editor) for the cells in a 
 * particular column in the JTable.  
 *
 * @version 1.2 10/27/98
 *
 * @author Philip Milne
 * @author Scott Violet
 */
public class JTreeTable extends JTable 
    implements DropTargetListener, DragSourceListener, DragGestureListener, Autoscroll

{

    Machine rlm;
    String rootPath;
    DropTarget dropTarget = new DropTarget (this, this);
    DragSource dragSource = DragSource.getDefaultDragSource();
    public int AutoscrollSensitivity = 60;

    /** A subclass of JTree. */
    protected TreeTableCellRenderer tree;

    public JTreeTable(Machine machine, String rootPath, TreeTableModel treeTableModel) {
	super();
        this.rootPath = rootPath;
        this.rlm = machine;

        this.dragSource.createDefaultDragGestureRecognizer(this, DnDConstants.ACTION_COPY, this);

	// Create the tree. It will be used as a renderer and editor. 
	tree = new TreeTableCellRenderer(treeTableModel);

	// Install a tableModel representing the visible rows in the tree. 
	super.setModel(new TreeTableModelAdapter(treeTableModel, tree));

	// Force the JTable and JTree to share their row selection models. 
	ListToTreeSelectionModelWrapper selectionWrapper = new 
            ListToTreeSelectionModelWrapper();
	tree.setSelectionModel(selectionWrapper);
	setSelectionModel(selectionWrapper.getListSelectionModel()); 

	// Install the tree editor renderer and editor. 
	setDefaultRenderer(TreeTableModel.class, tree); 
	setDefaultEditor(TreeTableModel.class, new TreeTableCellEditor());

	// No grid.
	setShowGrid(false);

	// No intercell spacing
	setIntercellSpacing(new Dimension(0, 0));	

        tree.setScrollsOnExpand(true);

	// And update the height of the trees row to match that of
	// the table.
	if (tree.getRowHeight() < 1) {
	    // Metal looks better like this.
            //	    setRowHeight(18);
	}
    }

    public Insets getAutoscrollInsets() {
        
        return new Insets(-AutoscrollSensitivity, -AutoscrollSensitivity, getSize().height + AutoscrollSensitivity, getSize().width + AutoscrollSensitivity);

    }

    public void autoscroll(Point cursorLoc) {

        Rectangle r, visR = getVisibleRect();
        int rowNum = -1;
        
        if(cursorLoc.y < visR.y + AutoscrollSensitivity) 
            rowNum = rowAtPoint(cursorLoc) - 1;
        else if(cursorLoc.y > visR.height - AutoscrollSensitivity)
            rowNum = rowAtPoint(cursorLoc) + 1;
        
        if(rowNum != -1) {
            
            r = getCellRect(rowNum, 1, false);
            scrollRectToVisible(r);
            
        }

    }

    public void dragDropEnd(DragSourceDropEvent dragSourceDropEvent) {
        //DebuggerOutput.debug("dragDropEnd: " + dragSourceDropEvent);
        DragSourceContext dsc = dragSourceDropEvent.getDragSourceContext();
        Transferable tr = dsc.getTransferable();

        DataFlavor[] flavors = tr.getTransferDataFlavors();
        
        //DebuggerOutput.debug("dragdropEnd: transferDataFlavors = " + tr);
        
        for(int i = 0; i < flavors.length; i++) 
            //DebuggerOutput.debug("dragDropEnd: transferDataFlavor[" + i + "]: " + flavors[i]);
        //DebuggerOutput.debug("dragDropEnd: exiting");

        this.setBorder( BorderFactory.createEmptyBorder() );

    }
    public void dragEnter(DragSourceDragEvent dragSourceDragEvent){}
    public void dragExit(DragSourceEvent dragSourceEvent){}
    public void dragOver(DragSourceDragEvent dragSourceDragEvent) {
    }
    public void dropActionChanged(DragSourceDragEvent dragSourceDragEvent){

        //DebuggerOutput.debug("dropActionChanged: dragSourceDragEvent = " + dragSourceDragEvent);

    }
    
    public void dragEnter(DropTargetDragEvent e) {
        //DebuggerOutput.debug("dragEnter(DropTargetDragEvent): accepting " + e.getDropAction());
        //       e.rejectDrag();
        e.acceptDrag(e.getDropAction());
    }

    public void dragExit (DropTargetEvent dropTargetEvent) {}

    public void dragOver (DropTargetDragEvent dropTargetDragEvent) {
        Point dropLocation = dropTargetDragEvent.getLocation();

        int selRow = getTree().getClosestRowForLocation(dropLocation.x, dropLocation.y);
        TreePath selPath = getTree().getPathForRow(selRow);
            
        Object o = ((DefaultMutableTreeNode) selPath.getLastPathComponent()).getUserObject();

        if(!(o instanceof HLProtocol.FileListComponent))
            return;

        HLProtocol.FileListComponent hlf = (HLProtocol.FileListComponent) o;

        if(hlf.fileType.equals("fldr")) {

            setRowSelectionInterval(selRow, selRow);
            dropTargetDragEvent.acceptDrag(dropTargetDragEvent.getDropAction());
            this.setBorder( BorderFactory.createEmptyBorder() );
            //getTree().expandPath(selPath);

        } else {

            getTree().setSelectionPath(selPath.getParentPath());

            if(selPath.getParentPath().getPathCount() == 1)
                this.setBorder(BorderFactory.createLineBorder(SystemColor.textHighlightText, 4));

            dropTargetDragEvent.acceptDrag(dropTargetDragEvent.getDropAction());

        }

    }

    public void dropActionChanged (DropTargetDragEvent dropTargetDragEvent) {}

    public synchronized void drop (DropTargetDropEvent dropTargetDropEvent) {

        try {

            Point dropLocation = dropTargetDropEvent.getLocation();

            int selRow = getTree().getClosestRowForLocation(dropLocation.x, dropLocation.y);
            TreePath selPath = getTree().getPathForRow(selRow);
            
            HLProtocol.FileListComponent hlf = (HLProtocol.FileListComponent) ((DefaultMutableTreeNode) selPath.getLastPathComponent()).getUserObject();
            
            String target = FilesInterface.treePathToFilePath(rootPath, selPath);
            Transferable tr = dropTargetDropEvent.getTransferable();

            DataFlavor[] flavors = tr.getTransferDataFlavors();

            //DebuggerOutput.debug("drop: tr = " + tr);

            for(int i = 0; i < flavors.length; i++) 
                //DebuggerOutput.debug(flavors[i]);

            this.setBorder( BorderFactory.createEmptyBorder() );

            if(dropTargetDropEvent.isLocalTransfer()) {

                String from = (String) tr.getTransferData(DataFlavor.stringFlavor);

                //DebuggerOutput.debug("drop: is local transfer from " + from + " to " + target);

                if(!hlf.fileType.equals("fldr")) {

                    target = target.substring(0, target.lastIndexOf(HLProtocol.DIR_SEPARATOR));
                    //DebuggerOutput.debug("drop: target is a file; truncated to " + target);

                }

                rlm.getHLC().requestFileMove(from, target);
                dropTargetDropEvent.acceptDrop(DnDConstants.ACTION_COPY_OR_MOVE);
                dropTargetDropEvent.dropComplete(true);
                return;

            }

            if(tr.isDataFlavorSupported(DataFlavor.javaFileListFlavor)) {

                if(!hlf.fileType.equals("fldr"))
                    target = target.substring(0, target.lastIndexOf(HLProtocol.DIR_SEPARATOR));
                
                dropTargetDropEvent.acceptDrop(DnDConstants.ACTION_COPY_OR_MOVE);

                //DebuggerOutput.debug("Getting transfer data ...");
                java.util.List fileList = (java.util.List)
                    tr.getTransferData(DataFlavor.javaFileListFlavor);
                Iterator iterator = fileList.iterator();
                
                while(iterator.hasNext()) {
                    
                    File file = (File) iterator.next();
                    //DebuggerOutput.debug("Got file: " + file.toString());
                    
                    if(!file.toString().equals("")) {

                        new UploadInterface(rlm, 
                                            target, 
                                            file);

                    }
                    
                }
                

                dropTargetDropEvent.dropComplete(true);
                return;
                
            } else if(tr.isDataFlavorSupported(DataFlavor.stringFlavor)) {

                String moveTo = (String) tr.getTransferData(DataFlavor.stringFlavor);
                //DebuggerOutput.debug("moveTo: " + moveTo);
                
            } else {

                System.err.println ("Rejected");
                dropTargetDropEvent.rejectDrop();

            }

        } catch (IOException io) {

            io.printStackTrace();
            dropTargetDropEvent.rejectDrop();

        } catch (UnsupportedFlavorException ufe) {

            ufe.printStackTrace();
            dropTargetDropEvent.rejectDrop();

        }

    }

    public void dragGestureRecognized(DragGestureEvent dragGestureEvent) {

        //DebuggerOutput.debug(dragGestureEvent);

        Point dropLocation = dragGestureEvent.getDragOrigin();

        int selRow = getTree().getClosestRowForLocation(dropLocation.x, dropLocation.y);
        TreePath selPath = getTree().getPathForRow(selRow);

        final HLProtocol.FileListComponent hlf = (HLProtocol.FileListComponent) ((DefaultMutableTreeNode) selPath.getLastPathComponent()).getUserObject();

        final String filePath = FilesInterface.treePathToFilePath(rootPath, selPath);
        
        Transferable fileList = new Transferable() {
                
                public Object getTransferData(DataFlavor flavor) {
                    
                    //DebuggerOutput.debug("Transferable.getTransferData: flavor = " + flavor);
                    
                    if(flavor.equals(DataFlavor.stringFlavor))
                        return filePath;

                    if(hlf.fileType.equals("fldr")) {
                        
                        new FolderDownloadInterface(rlm,
                                                    filePath,
                                                    hlf);
                                                    
                    } else {
 
                        DownloadInterface dli = 
                            new DownloadInterface(rlm, 
                                                  filePath,
                                                  hlf);
                        
                        File f = dli.getLocalFile();

                        try {

                            Thread.currentThread().sleep(1000);

                        } catch(InterruptedException e) {}

                        if(f == null) 
                            return null;

                        ArrayList al = new ArrayList();
                        al.add(f);
                        return al;

                    }

                    return null;

                }

                public DataFlavor[] getTransferDataFlavors() {

                    DataFlavor[] flavors = { DataFlavor.javaFileListFlavor, DataFlavor.stringFlavor };
                    return flavors;

                }

                public boolean isDataFlavorSupported(DataFlavor flavor) {

                    if(flavor.equals(DataFlavor.javaFileListFlavor) ||
                       flavor.equals(DataFlavor.stringFlavor))
                        return true;
                    
                    return false;
                    
                }

            };
        

        // start the dragging
        dragSource.startDrag(dragGestureEvent, DragSource.DefaultCopyDrop, fileList, this);

    }

    /**
     * Overridden to message super and forward the method to the tree.
     * Since the tree is not actually in the component hieachy it will
     * never receive this unless we forward it in this manner.
     */
    public void updateUI() {
	super.updateUI();
	if(tree != null) {
	    tree.updateUI();
	}
	// Use the tree's default foreground and background colors in the
	// table. 
        LookAndFeel.installColorsAndFont(this, "Tree.background",
                                         "Tree.foreground", "Tree.font");
    }

    /* Workaround for BasicTableUI anomaly. Make sure the UI never tries to 
     * paint the editor. The UI currently uses different techniques to 
     * paint the renderers and editors and overriding setBounds() below 
     * is not the right thing to do for an editor. Returning -1 for the 
     * editing row in this case, ensures the editor is never painted. 
     */
    public int getEditingRow() {
        return (getColumnClass(editingColumn) == TreeTableModel.class) ? -1 :
            editingRow;  
    }

    /**
     * Overridden to pass the new rowHeight to the tree.
     */
    public void setRowHeight(int rowHeight) { 
        super.setRowHeight(rowHeight); 
	if (tree != null && tree.getRowHeight() != rowHeight) {
            tree.setRowHeight(getRowHeight()); 
	}
    }

    /**
     * Returns the tree that is being shared between the model.
     */
    public JTree getTree() {
	return tree;
    }


    
    
    /**
     * A TreeCellRenderer that displays a JTree.
     */
    public class TreeTableCellRenderer extends JTree implements TableCellRenderer {
        /** Last table/tree row asked to renderer. */
        protected int visibleRow;

        public TreeTableCellRenderer(TreeModel model) {
            super(model); 
        }

        /**
	 * updateUI is overridden to set the colors of the Tree's renderer
	 * to match that of the table.
	 */
        public void updateUI() {
            super.updateUI();
            // Make the tree's cell renderer use the table's cell selection
            // colors. 
            TreeCellRenderer tcr = getCellRenderer();
            if (tcr instanceof DefaultTreeCellRenderer) {
                DefaultTreeCellRenderer dtcr = ((DefaultTreeCellRenderer)tcr); 
                // For 1.1 uncomment this, 1.2 has a bug that will cause an
                // exception to be thrown if the border selection color is
                // null.
                // dtcr.setBorderSelectionColor(null);
                dtcr.setTextSelectionColor(UIManager.getColor
                                           ("Table.selectionForeground"));
                dtcr.setBackgroundSelectionColor(UIManager.getColor
                                                 ("Table.selectionBackground"));
            }
        }

        /**
	 * Sets the row height of the tree, and forwards the row height to
	 * the table.
	 */
        public void setRowHeight(int rowHeight) { 
            if (rowHeight > 0) {
                super.setRowHeight(rowHeight); 
                if (JTreeTable.this != null &&
                    JTreeTable.this.getRowHeight() != rowHeight) {
                    JTreeTable.this.setRowHeight(getRowHeight()); 
                }
            }
        }

        /**
	 * This is overridden to set the height to match that of the JTable.
	 */
        public void setBounds(int x, int y, int w, int h) {
            super.setBounds(x, 0, w, JTreeTable.this.getHeight());
        }

        /**
	 * Sublcassed to translate the graphics such that the last visible
	 * row will be drawn at 0,0.
	 */
        public void paint(Graphics g) {
            g.translate(0, -visibleRow * getRowHeight());
            super.paint(g);
        }

        /**
	 * TreeCellRenderer method. Overridden to update the visible row.
	 */
        public Component getTableCellRendererComponent(JTable table,
                                                       Object value,
                                                       boolean isSelected,
                                                       boolean hasFocus,
                                                       int row, int column) {
            if(isSelected)
                setBackground(table.getSelectionBackground());
            else
                setBackground(table.getBackground());

            visibleRow = row;
            return this;
        }
    }


    /**
     * TreeTableCellEditor implementation. Component returned is the
     * JTree.
     */
    public class TreeTableCellEditor extends AbstractCellEditor implements
                                                                                                                                TableCellEditor {
        public Component getTableCellEditorComponent(JTable table,
                                                     Object value,
                                                     boolean isSelected,
                                                     int r, int c) {
            return tree;
        }

        /**
         * Overridden to return false, and if the event is a mouse event
         * it is forwarded to the tree.<p>
         * The behavior for this is debatable, and should really be offered
         * as a property. By returning false, all keyboard actions are
         * implemented in terms of the table. By returning true, the
         * tree would get a chance to do something with the keyboard
         * events. For the most part this is ok. But for certain keys,
         * such as left/right, the tree will expand/collapse where as
         * the table focus should really move to a different column. Page
         * up/down should also be implemented in terms of the table.
         * By returning false this also has the added benefit that clicking
         * outside of the bounds of the tree node, but still in the tree
         * column will select the row, whereas if this returned true
         * that wouldn't be the case.
         * <p>By returning false we are also enforcing the policy that
         * the tree will never be editable (at least by a key sequence).
         */
        public boolean isCellEditable(EventObject e) {
            if (e instanceof MouseEvent) {
                for (int counter = getColumnCount() - 1; counter >= 0;
                     counter--) {
                    if (getColumnClass(counter) == TreeTableModel.class) {
                        MouseEvent me = (MouseEvent)e;
                        MouseEvent newME = new MouseEvent(tree, me.getID(),
                                                          me.getWhen(), me.getModifiers(),
                                                          me.getX() - getCellRect(0, counter, true).x,
                                                          me.getY(), me.getClickCount(),
                                                          me.isPopupTrigger());
                        tree.dispatchEvent(newME);
                        break;
                    }
                }
            }
            return false;
        }
    }


    /**
     * ListToTreeSelectionModelWrapper extends DefaultTreeSelectionModel
     * to listen for changes in the ListSelectionModel it maintains. Once
     * a change in the ListSelectionModel happens, the paths are updated
     * in the DefaultTreeSelectionModel.
     */
    class ListToTreeSelectionModelWrapper extends DefaultTreeSelectionModel { 
        /** Set to true when we are updating the ListSelectionModel. */
        protected boolean         updatingListSelectionModel;

        public ListToTreeSelectionModelWrapper() {
            super();
            getListSelectionModel().addListSelectionListener
                (createListSelectionListener());
        }

        /**
         * Returns the list selection model. ListToTreeSelectionModelWrapper
         * listens for changes to this model and updates the selected paths
         * accordingly.
         */
        ListSelectionModel getListSelectionModel() {
            return listSelectionModel; 
        }

        /**
         * This is overridden to set <code>updatingListSelectionModel</code>
         * and message super. This is the only place DefaultTreeSelectionModel
         * alters the ListSelectionModel.
         */
        public void resetRowSelection() {
            if(!updatingListSelectionModel) {
                updatingListSelectionModel = true;
                try {
                    super.resetRowSelection();
                }
                finally {
                    updatingListSelectionModel = false;
                }
            }
            // Notice how we don't message super if
            // updatingListSelectionModel is true. If
            // updatingListSelectionModel is true, it implies the
            // ListSelectionModel has already been updated and the
            // paths are the only thing that needs to be updated.
        }

        /**
         * Creates and returns an instance of ListSelectionHandler.
         */
        protected ListSelectionListener createListSelectionListener() {
            return new ListSelectionHandler();
        }

        /**
         * If <code>updatingListSelectionModel</code> is false, this will
         * reset the selected paths from the selected rows in the list
         * selection model.
         */
        protected void updateSelectedPathsFromSelectedRows() {
            if(!updatingListSelectionModel) {
                updatingListSelectionModel = true;
                try {
                    // This is way expensive, ListSelectionModel needs an
                    // enumerator for iterating.
                    int        min = listSelectionModel.getMinSelectionIndex();
                    int        max = listSelectionModel.getMaxSelectionIndex();

                    clearSelection();
                    if(min != -1 && max != -1) {
                        for(int counter = min; counter <= max; counter++) {
                            if(listSelectionModel.isSelectedIndex(counter)) {
                                TreePath     selPath = tree.getPathForRow
                                    (counter);

                                if(selPath != null) {
                                    addSelectionPath(selPath);
                                }
                            }
                        }
                    }
                }
                finally {
                    updatingListSelectionModel = false;
                }
            }
        }

        /**
         * Class responsible for calling updateSelectedPathsFromSelectedRows
         * when the selection of the list changse.
         */
        class ListSelectionHandler implements ListSelectionListener {
            public void valueChanged(ListSelectionEvent e) {
                updateSelectedPathsFromSelectedRows();
            }
        }
    }
}
