/**
 * $Id: UserInfoInterface.java,v 1.5 2001/10/04 02:43:54 groomed Exp $
 *
 * Copyright (C) 1998-2001 groomed <groomed@users.sourceforge.net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

package redlight.client;
 
import java.io.IOException;
import java.awt.*;
import java.awt.event.*;
import javax.swing.*;

import redlight.hotline.*;
import redlight.utils.DebuggerOutput;
import redlight.graphics.Spinner;

/**
 * Displays user info for a user with optional
 * periodic refresh.
 */
public class UserInfoInterface implements Child {

    Machine rlm;
    JFrame f;
    UserInfoPanel userInfoPanel;
    String userInfo;
    Thread getterThread;
    User user;

    public UserInfoInterface(Machine rlm, User user) {
        
        this.rlm = rlm;
        this.user = user;

	f = new JFrame("User info (" + rlm.getServerName() + ")");
	rlm.getInterface().registerChild(this);
        userInfoPanel = new UserInfoPanel(rlm, user);
        
        userInfoPanel.setBorder(BorderFactory.createCompoundBorder(BorderedPanel.emptyBorder, userInfoPanel.getBorder()));

	Container contentPane = f.getContentPane();
        contentPane.setLayout(new BorderLayout());
        contentPane.add(userInfoPanel, BorderLayout.CENTER);

        f.addWindowListener(new WindowAdapter() {

                public void windowClosing(WindowEvent e) {

                    close();

                }
                
            });

        userInfoPanel.autoRefresh.addActionListener(new ActionListener() {

                public void actionPerformed(ActionEvent e) {

                    if(e.getActionCommand().equals("ActivateAutoRefresh")) {

                        if(userInfoPanel.autoRefresh.isSelected() &&
                           !getterThread.isAlive()) {

                            getterThread = new Thread(getter);
                            getterThread.start();

                        }

                    }

                }

            });

        userInfoPanel.toolBar.reload.addActionListener(new ActionListener() {

                public void actionPerformed(ActionEvent e) {

                    if(e.getActionCommand().equals("ActivateReload")) {
                        
                        getterThread.interrupt();

                        try {

                            getterThread.join();
                            getterThread = new Thread(getter);
                            getterThread.start();

                        } catch(InterruptedException ex) {}
                        
                    }

                }

            });

        KeyStroke ks = KeyStroke.getKeyStroke(KeyEvent.VK_ESCAPE, 0);
        userInfoPanel.registerKeyboardAction(new ActionListener() {
                
                public void actionPerformed(ActionEvent e) {

                    if(e.getActionCommand().equals("Cancel"))
                        close();

                }

            }, "Cancel", ks, JComponent.WHEN_IN_FOCUSED_WINDOW);

	f.pack();

	Dimension d = Toolkit.getDefaultToolkit().getScreenSize();
	Dimension w = f.getSize();

        f.setLocation((d.width / 2) - (w.width / 2), 
                      (d.height / 2) - (w.height / 2));

        f.show();

	getterThread = new Thread(getter);
        getterThread.start();

    }

    public void displayPropertyChanged(final String what, 
                                       final Object property) {
        
        SwingUtilities.invokeLater(new Runnable() {

                public void run() {

                    if(what.equals("textfont"))
                        userInfoPanel.textArea.setFont((Font) property);

                }

            });

    }

    public void close() {

        rlm.getInterface().unregisterChild(this);
        
        getterThread.interrupt();

        try {

            getterThread.join();

        } catch(InterruptedException e) {}

        f.dispose();
        
    }

    Runnable getter = new Runnable() {
            
            public void run() {
                
                Thread.currentThread().setName("UserInfoGetterThread[ " + user + "]");                
                DebuggerOutput.debug("UserInfoInterface.getterThread: started");

                try {
                    
                    do {
                        
                        userInfoPanel.spinner.start();
                        
                        SwingUtilities.invokeLater(new Runnable() {
                                
                                public void run() {
                                    
                                    userInfoPanel.toolBar.reload.setEnabled(false);
                                
                                }
                                
                            });
                        
                        try {
                            
                            userInfo = rlm.getHLC().getUserInfo(user.sock);
                            userInfo = userInfo.replace('\r', '\n');
                            
                        } catch(Exception e) {
                            
                            if(!(e instanceof IOException ||
                                 e instanceof HLException))
                                throw (InterruptedException) e;
                            
                            String reason = e.toString();
                            
                            if(e.getMessage() != null)
                                reason = e.getMessage();
                            
                            userInfo = reason;
                            userInfoPanel.autoRefresh.setSelected(false);
                            
                        } finally {
                            
                            userInfoPanel.spinner.stop();
                            
                            SwingUtilities.invokeLater(new Runnable() {
                                    
                                    public void run() {
                                        
                                        userInfoPanel.textArea.setText(userInfo);
                                        userInfoPanel.textArea.setCaretPosition(0);
                                        userInfoPanel.toolBar.reload.setEnabled(true);
                                        
                                    }
                                    
                                });
                            
                        }
                        
                        if(!Thread.currentThread().isInterrupted() &&
                           userInfoPanel.autoRefresh.isSelected()) {
                            
                            DebuggerOutput.debug("UserInfoInterface.getterThread: sleeping ...");
                            Thread.currentThread().sleep(15000);
                            
                        }
                        
                    } while(!Thread.currentThread().isInterrupted() &&
                            userInfoPanel.autoRefresh.isSelected());
                    
                } catch(InterruptedException e) {}
                
                DebuggerOutput.debug("UserInfoInterface.getterThread: exiting");
            }
            
        };
    
    
}

class UserInfoPanel extends BorderedPanel {

    JTextArea textArea;
    UserToolBar toolBar;
    Spinner spinner;
    JCheckBox autoRefresh;

    UserInfoPanel(Machine rlm, User user) {

        super("User info");

        GridBagConstraints gbc = new GridBagConstraints();
        toolBar = new UserToolBar(rlm, user);
        textArea = new JTextArea(15, 45);
        spinner = new Spinner(Main.rlo.getImageSet("Spinner"), 
			      100, 16, 16);
        JLabel renderedUser = new UserCellRenderer(user, false);
        JPanel headerPanel = new JPanel();
        autoRefresh = new JCheckBox("Autorefresh");

        autoRefresh.setActionCommand("ActivateAutoRefresh");
        autoRefresh.setToolTipText("Enable/disable 15 sec. refresh");
        toolBar.info.setVisible(false);
        textArea.setFont((Font) Main.rlo.getProperty("Font.text"));
        headerPanel.setLayout(new GridBagLayout());
        setLayout(new GridBagLayout());

        gbc.anchor = gbc.WEST;
        headerPanel.add(new JLabel("Info for: "), gbc);
        gbc.fill = gbc.HORIZONTAL;
        gbc.gridx = 1;
        gbc.weightx = 1;
        headerPanel.add(renderedUser, gbc);

        gbc.gridx = 0;
        gbc.gridwidth = 2;
        gbc.anchor = gbc.WEST;
        gbc.weightx = 1;
        Insets oldInsets = gbc.insets;
        gbc.insets = new Insets(0, 0, 8, 0);
        add(headerPanel, gbc);
        gbc.insets = oldInsets;
        
        gbc.gridwidth = 1;
        gbc.gridx = 0;
        gbc.gridy = 1;
	gbc.fill = gbc.NONE;
	gbc.anchor = gbc.WEST;
        add(toolBar, gbc);
        gbc.gridx = 1;
        gbc.anchor = gbc.EAST;
        gbc.weightx = 1;
        add(spinner, gbc);

        gbc.gridx = 0;
        gbc.gridy = 2;
        gbc.gridwidth = 2;
        gbc.weighty = 1;
        gbc.fill = gbc.BOTH;
        add(new JScrollPane(textArea), gbc);
        gbc.weighty = 0;
        gbc.anchor = gbc.WEST;
        gbc.fill = gbc.NONE;
        gbc.ipady = 4;
        gbc.gridy = 3;
        add(autoRefresh, gbc);

    }


}

