/**
 * $Id: HLTracker.java,v 1.1.1.1 2001/07/22 02:44:45 groomed Exp $
 *
 * Copyright (C) 1998-2001 groomed <groomed@users.sourceforge.net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

package redlight.hotline;

import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.IOException;
import java.io.ByteArrayOutputStream;
import java.net.Socket;
import java.net.DatagramPacket;
import java.net.DatagramSocket;
import java.net.InetAddress;

import redlight.hotline.HLProtocol;
import redlight.utils.DebuggerOutput;
import redlight.utils.ToArrayConverters;

/**
 * This class provides a means to retrieve a server list 
 * from a tracker and to send server information to a tracker.
 */
public class HLTracker {

    InetAddress host;
    int port;

    /**
     * Create a HLTracker for the specified
     * tracker and the default port.
     * @param server the tracker name
     */
    public HLTracker(String server) throws IOException {

	this(InetAddress.getByName(server), HLProtocol.HTRK_TCPPORT);

    }

    /**
     * Create a HLTracker for the specified
     * tracker and the default port.
     * @param host the InetAddress of the tracker
     */
    public HLTracker(InetAddress host) throws IOException {

	this(host, HLProtocol.HTRK_TCPPORT);

    }

    /**
     * Create a HLTracker for the specified
     * tracker and the specified port.
     * @param server the tracker name
     * @param port the port number
     */
    public HLTracker(String server, int port) throws IOException {

	this(InetAddress.getByName(server), port);

    }

    /**
     * Create a HLTracker for the specified
     * tracker and the specified port.
     * @param h the InetAddress of the tracker
     * @param p the port number
     */
    public HLTracker(InetAddress h, int p) throws IOException {

	host = h;
	port = p;

    }

    /**
     * Retrieves a list of servers from the tracker.
     * @return array of HLProtocol.ServerInfo objects
     */
    public HLProtocol.ServerInfo[] getServerList() throws IOException {

	HLProtocol hlp = new HLProtocol();
	Socket socket = new Socket(host, port);
	DataOutputStream output = 
            new DataOutputStream(socket.getOutputStream());
	DataInputStream input = 
            new DataInputStream(socket.getInputStream());

        /* Write tracker magic. */

        output.writeBytes(HLProtocol.HTRK_MAGIC);

        /* Read tracker magic. */

        byte[] magic = new byte[HLProtocol.HTRK_MAGIC_LEN];
        input.read(magic, 0, HLProtocol.HTRK_MAGIC_LEN);

        if(!new String(magic).equals(HLProtocol.HTRK_MAGIC))
            throw new IOException("Bad magic, probably not a tracker.");

        /* Read packet header. */

        HLProtocol.TrackerPacketHeader th = 
            hlp.new TrackerPacketHeader(input);

        HLProtocol.ServerInfo[] servers = 
            new HLProtocol.ServerInfo[th.nservers];

        short nservers = th.nservers;
        int i = 0, pack, total = pack = th.pack;

        /* Read servers. */

        while(nservers > 0) {

            while(i < total && i < th.nservers) {

                servers[i++] = hlp.new ServerInfo(input);
                nservers--;

            }

            if(nservers!=0) {

                th = hlp.new TrackerPacketHeader(input);
                pack = th.pack;

            }

            total += pack;

        }

        socket.close();

	return servers;

    }

    /**
     * Sends server information to the tracker. If the arguments
     * are longer than allowed, they are silently truncated.
     * @param serverPort the port where the server is running.
     * @param name the name of the server (max. 255 characters).
     * @param description description of the server (max 255 characters).
     * @param nusers the number of users on the server.
     * @param password the password for the tracker (may be null, max
     * 255 characters).  
     */
    public void sendServerInfo(int serverPort, 
                               String name,
                               String description,
                               short nusers,
                               String password) throws IOException {

        if(name == null || description == null)
            throw new IllegalArgumentException("null argument");

        if(name.length() > 255)
            name = name.substring(0, 255);

        if(description.length() > 255)
            description = description.substring(0, 255);

        if(password != null)
            if(password.length() > 255)
                password = password.substring(0, 255);

        ByteArrayOutputStream bos = new ByteArrayOutputStream(500);
        DataOutputStream output = new DataOutputStream(bos);

        output.writeShort(1);
        output.writeShort((char) serverPort);
        output.writeShort(nusers);
        output.writeShort(0);
        output.writeInt(0xefe44d00);
        output.writeByte(name.length());
        output.writeBytes(name);
        output.writeByte(description.length());
        output.writeBytes(description);
        
        if(password != null) {

            output.writeByte(password.length());
            output.writeBytes(password);

        } else {

            output.writeByte(0);

        }

        byte[] data = bos.toByteArray();

        DatagramSocket ds = new DatagramSocket();
        ds.send(new DatagramPacket(data, data.length, host, this.port));
        ds.close();

    }

    /**
     * Returns the InetAddress for this tracker.
     * @return InetAddress for this tracker
     */
    public InetAddress getInetAddress() {

	return host;

    }

    /**
     * Returns the port number for this tracker.
     * @return port number for this tracker
     */
    public int getPort() {

	return port;

    }

}

