package redlight.hotline.iconlib;

import redlight.utils.DebuggerOutput;

/*
                Read the license.txt file for license information!
                If you don't have it, get it at www.MarkusStengel.de.
*/

/**
 *          an abstract class with various useful operations, such as conversions etc.
 *          @author     Markus Stengel
 *          @see          <a href="http://www.MarkusStengel.de">developer site</a>
 */

public abstract class HLUtils {

     /**
      *          converts a signed byte to an unsigned int
      *          @param     b the byte to be converted
      *          @return     unsigned byte value
      */
     public static int signedToUnsigned(byte b) {
          if (b < 0) {
               return 128+(128+b);
          }
          return b;
     }

     /**
      *          converts a signed short to an unsigned int
      *          @param     s the byte to be converted
      *          @return     unsigned byte value
      */
     public static int signedToUnsigned(short s) {
          if (s < 0) {
               return 32768 + (32768 + s);
          }
          return s;
     }
     /**
      *          extracts an integer in big endian format from a byte field at a given position
      *          @param     data byte field where the integer is stored in big endian format
      *          @param     pos index of the byte field where the integer starts
      *          @return     extracted int
      */
     public static int extractInt(byte[] data, int pos) {                                   // extracts an int from the bytefield
          int i;
          i  = signedToUnsigned(data[pos+00]) << 24;
          i += signedToUnsigned(data[pos+01]) << 16;
          i += signedToUnsigned(data[pos+02]) << 8;
          i += signedToUnsigned(data[pos+03]);
          return i;
     }
     /**
      *          extracts a given number of bytes at a given index from a byte field
      *          @param     data source byte field
      *          @param     pos index in data
      *          @param     length number of bytes
      *          @return     extracted bytes
      */
     public static byte[] extractBytes(byte[] data, int pos, int length) {     // extracts given number of bytes from a given byte field
          int i;
          byte[] ret = new byte[length];

          for (i=0; i<length; i++) {
               ret[i] = data[pos+i];
          }

          return ret;
     }

     /**
      *          extracts the identity, size and offset of an icon/gif/... from the given bytefield
      *          which should be a LVT entry from a hotline.dat file
      *          @param     data byte field containing a LVT row
      *          @param     pos index in data
      *          @param     length number of bytes
      *          @return     extracted bytes
      */
     public static HLData extractDataFieldHLData(byte[] data) {
          HLData ret = new HLData();

          ret.identity           = extractInt(data, 0x00);
          ret.dataSize           = extractInt(data, 0x08);
          ret.dataOffset      = extractInt(data, 0x0C);

          return ret;
     }

     /**
      *          extracts the block name and row count from the bytefield which should have the form of
      *          a HLDataHdr table, i. e. the first 8 bytes of a LVT entry
      *          @param     data byte field containing a LVT entry
      *          @return     object with extracted data
      */
     public static HLDataHdr extractDataFieldHLDataHdr(byte[] data) {
          HLDataHdr ret = new HLDataHdr();
          ret.blockName = new String(extractBytes(data, 0, 4));
          ret.rowCount  = extractInt(data, 4);
          return ret;
     }

     /**
      *          converts an integer to a byte field, bytes are in big endian order
      *          @param     val integer value
      *          @return     byte field in big endian order
      */
     public static byte[] intToBytes(int val) {
          byte[]     b = new byte[4];

          b[0]     = (byte) (val / 0x1000000);
          val  -= b[0] * 0x1000000;
          b[1]     = (byte) (val / 0x10000);
          val  -= b[1] * 0x10000;
          b[2]     = (byte) (val >> 8);
          val  -= b[2] * 0x100;
          b[3]     = (byte) val;

          return b;
     }

     /**
      *          copies bytes from one data field into another
      *          @see HLUtils#storeReversed
      *          @param dest          destination byte field
      *          @param src          source byte field
      *          @param destOff     offset in the destination byte field
      *          @param srcOff     offset in the source byte field
      *          @param len          number of bytes to be copied
      */
     public static void copyBytes(byte[] dest, byte[] src, int destOff, int srcOff, int len) {
          int i;
          for (i=0; i<len; i++) {
               dest[destOff+i] = src[srcOff+i];
          }
     }

     /**
      *          copies bytes reversed from one data field into another
      *          @see HLUtils#copyBytes
      *          @param dest          destination byte field
      *          @param src          source byte field
      *          @param destOff     offset in the destination byte field
      *          @param srcOff     offset in the source byte field
      *          @param len          number of bytes to be copied
      */
     public static void storeReversed ( byte[] dest, byte[] src, int destOff, int srcOff, int len) {
          int i;
          for (i=0; i<len; i++) {
               if ((srcOff+len-1-i) < 0) return;                                        // we have done as much as possible, ISSSUE: maybe someone wants to know about this?
               if ((destOff+i) < 0) return;                                                  // we have done as much as possible, ISSSUE: maybe someone wants to know about this?
               dest[destOff+i] = src[srcOff+len-1-i];                                   // ok, store it
          }
     }

     /**
      *          prints the unsigned byte values of a byte field
      *          @param field     byte field to be printed
      */
     public static void printField(byte[] field) {
          int i;
          for (i=0; i< field.length; i++) {                                             // check each one
               System.out.print(((int) HLUtils.signedToUnsigned(field[i])) + "     ");
          }
          DebuggerOutput.debug("");                                                                 // close the last line
     }

     /**
      *          prints the int field
      *          @param field     int field to be printed
      */
     public static void printFieldInt(int[] field) {
          int i;
          for (i=0; i< field.length; i++) {                                             // check each one
               System.out.print(field[i] + "     ");
          }
          DebuggerOutput.debug("");                                                                 // close the last line
     }

     /**
      *      compare the beginning of a (Text-) bytefield with a string
      *          @param     field          byte field
      *          @param     c               String
      *          @return     <b>true</b> if the first c.length bytes of c and field are equal
      *                         - when field is considered as a String - <b>false</b> otherwise
      */
     public static boolean compareBeginStr(byte[] field, String c) {
          if (c.compareTo(new String(extractBytes(field, 0, c.length()))) == 0) {
               return true;                                                                           // beginnings are equal
          }
          else {
               return false;                                                                           // no, they are not
          }
     }
}
