/**
 * $Id: AccountsTable.java,v 1.4 2001/10/06 14:31:30 groomed Exp $
 *
 * Copyright (C) 1998-2001 groomed <groomed@users.sourceforge.net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

package redlight.server;

import java.io.IOException;
import java.io.LineNumberReader;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.InputStreamReader;
import java.io.File;
import java.util.Enumeration;
import java.util.Hashtable;

import redlight.hotline.HLServer;
import redlight.hotline.HLServerAccountsTable;
import redlight.hotline.HLProtocol;
import redlight.crypto.UnixCrypt;
import redlight.utils.DebuggerOutput;

/**
 * This class provides persistent storage to a file for the server
 * accounts table.<p>
 *
 * The accounts file describes the accounts known to the server. It
 * is structured after the Unix /etc/passwd file.  The accounts
 * file has one entry per line, and each line has the following
 * format (the brackets [ and ] indicate that the home directory
 * field, including the colon (:), are optional):<p>
 *
 * <tt>login:nickname:privileges:password[:home directory]</tt><p>
 *
 * <tt>login</tt> the login for the account.<br>
 * <tt>nickname</tt> the standard nickname for the account.<br>
 * <tt>privileges</tt> a 64-bit bitfield describing the privileges
 * for this account (2's complement, so -1 is 0xFFFFFFFFL).
 * ({@see redlight.hotline.HLProtocol.AccountInfo#privileges}).<br>
 * <tt>password</tt> the encrypted password for the account ({@see
 * redlight.utils.UnixCrypt}).<br>
 * <tt>home directory</tt> optional, the home directory for the
 * account (that is, the root folder).<p>
 *
 * If an error is encountered at any point during parsing, the
 * program exits with a {@link #EXIT_ACCOUNTS_INVALID} error
 * code.  
 */
public class AccountsTable extends HLServerAccountsTable {

    HLProtocol hlp;
    File accountsFile;
    long lastModified;

    /**
     * Creates an accounts table which uses the specified
     * file for persistent storage.
     * @param file the file to use for persistent storage.
     */
    public AccountsTable(HLServer h, File file) {

        super(h);
        hlp = new HLProtocol();
        accountsFile = file;        

    }

    /**
     * Called whenever the accounts table changes.
     */
    public void hasChanged() {

        write(accountsFile);

    }

    /**
     * Load the component from disk if it has changed.
     */
    public void checkModified() {

        if(accountsFile.lastModified() != lastModified) {

            hls.log("Accounts file " + accountsFile.toString() + " changed, reloading.");
            load();

        }

    }

    /**
     * Loads the accounts table from persistent storage. Expects the
     * accounts table to be locked (or otherwise safe from multiple 
     * threads trying to access it) on entry.
     */
    public void load() {

        load(accountsFile);
        lastModified = accountsFile.lastModified();

    }

    /**
     * Loads the accounts table from the given InputStream. Expects the
     * accounts table to be locked (or otherwise safe from multiple 
     * threads trying to access it) on entry.
     * @param in the InputStream.
     */
    public void load(InputStream in) {

        Hashtable tempTable = new Hashtable();
        LineNumberReader lr = null;
        
        try {

            lr = new LineNumberReader(new InputStreamReader(in));
            
            for(String line = lr.readLine(); line != null; line = lr.readLine()) {

                /* Check whether the line contains at least three
                   colons.  Otherwise, the file format is invalid. */

                int nc = -1, colonPos = 0;
                for(nc = -1; colonPos != -1; colonPos = line.indexOf(':', colonPos + 1), nc++);

                if(nc > 2) {

                    /* Parse the line into it's 4 or 5 component
                       fields. */

                    int offset = 0;
                    String login = 
                        line.substring(offset, line.indexOf(':'));
                    offset += login.length() + 1;

                    String nick = 
                        line.substring(offset, line.indexOf(':', offset));
                    offset += nick.length() + 1;

                    String privsAsString = 
                        line.substring(offset, line.indexOf(':', offset));
                    offset += privsAsString.length() + 1;

                    String password;
                    String homedir = null;

                    if(nc > 3) {

                        /* This is a 5 component line, where the fifth
                           component specifies a home directory for
                           the account. */

                        password = 
                            line.substring(offset, line.indexOf(':', offset));
                        offset += password.length() + 1;

                        homedir = line.substring(offset);

                        /* Verify that the home directory is not empty
                           (or all spaces); otherwise, the home
                           directory is assumed unspecified. */

                        if(homedir.trim().equals(""))
                            homedir = null;
                        
                    } else {

                        /* No home directory specified for this
                           account. */

                        password = line.substring(offset);

                    }

                    if(password.equals(""))
                        password = UnixCrypt.crypt("aa", "");

                    /* This will throw an exception if the privileges
                       field does not contain a number, ie. if the
                       file format is invalid. */

                    long privileges = Long.parseLong(privsAsString);

                    /* OK, all fields valid, now add it to the
                       accounts table. */

                    tempTable.put(login, hlp.new AccountInfo(login, nick, password, privileges, homedir));

                    DebuggerOutput.debug("AccountsTable.parseAccountsFile: login = " + login + ", nick = " + nick + ", password = " + password + ", privileges = " + privileges + ", homedir = " + homedir);

                } else {
                    
                    throw new RuntimeException("Format invalid");
                    
                }
                
            }

        } catch(NumberFormatException e) {

            tempTable = null;

            String error = "Accounts file " + " has invalid privilege field '" + e.getMessage() + "'";

            if(lr != null)
                error += " at line " + lr.getLineNumber();

            error += ".";

            hls.log(error);
            hls.shutdown(error);
            
        } catch(Exception e) {

            tempTable = null;

            String error = "Accounts file " + " has invalid format";

            if(lr != null)
                error += " at line " + lr.getLineNumber();

            error += ".";

            hls.log(error);
            hls.shutdown(error);

        }

        if(tempTable != null) {

            table.clear();
            table = tempTable;

        }

    }

    /**
     * Writes the accounts table to persistent storage. Expects the
     * accounts table to be locked (or otherwise safe from multiple 
     * threads trying to access it) on entry.
     */
    public void write() {

        write(accountsFile);
        
    }

    /**
     * Writes the accounts table to the given OutputStream. Expects the
     * accounts table to be locked (or otherwise safe from multiple 
     * threads trying to access it) on entry.
     * @param out the OutputStream to write to.
     */
    public void write(OutputStream out) {
        
        try {
            
            DebuggerOutput.debug("AccountsTable.write: writing");
            
            for(Enumeration en = table.elements(); en.hasMoreElements(); ) {
                
                HLProtocol.AccountInfo account = (HLProtocol.AccountInfo) en.nextElement();
                
                String line = account.login + ":" + account.nick + ":" + account.privileges + ":" + account.password;
                
                if(account.homeDirectory != null)
                    line += ":" + account.homeDirectory;
                
                line += System.getProperty("line.separator");
                
                out.write(line.getBytes());
                
            }
            
        } catch(IOException e) {}
        
    }

}
