/*
 * JOCL - Java bindings for OpenCL
 *
 * Copyright (c) 2009 Marco Hutter - http://www.jocl.org
 * 
 * Permission is hereby granted, free of charge, to any person
 * obtaining a copy of this software and associated documentation
 * files (the "Software"), to deal in the Software without
 * restriction, including without limitation the rights to use,
 * copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following
 * conditions:
 * 
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
 * OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
 * HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
 * WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
 * OTHER DEALINGS IN THE SOFTWARE.
 */

package org.jocl;

import java.io.*;
import java.util.Locale;

/**
 * Utility class for detecting the operating system and architecture
 * types, and automatically loading the matching native library
 * as a resource or from a file. <br />
 * <br />
 * The architecture and OS detection has been adapted from 
 * http://javablog.co.uk/2007/05/19/making-jni-cross-platform/
 * and extended with http://lopica.sourceforge.net/os.html 
 */
final class LibUtils
{
    /**
     * Enumeration of common operating systems, independent of version 
     * or architecture. 
     */
    public static enum OSType
    {
    	APPLE, LINUX, SUN, WINDOWS, UNKNOWN
    }
    
    /**
     * Enumeration of common CPU architectures.
     */
    public static enum ARCHType
    {
        PPC, PPC_64, SPARC, X86, X86_64, ARM, MIPS, RISC, UNKNOWN
    }
    
    /**
     * Loads the specified library. The full name of the library
     * is created by calling {@link LibUtils#createLibName(String)}
     * with the given argument. The method will attempt to load
     * the library using the usual System.loadLibrary call,
     * and, if this fails, it will try to load it as a as a 
     * resource (for usage within a JAR).
     *    
     * @param baseName The base name of the library
     * @throws UnsatisfiedLinkError if the native library 
     * could not be loaded.
     */
    public static void loadLibrary(String baseName)
    {
        String libName = LibUtils.createLibName(baseName);

        Throwable throwable = null;
        try
        {
            System.loadLibrary(libName);
            return;
        }
        catch (Throwable t) 
        {
        	throwable = t;
        }
        
        try
        {
            loadLibraryResource(libName);
        	return;
        }
        catch (Throwable t)
        {
            StringWriter sw = new StringWriter();
            PrintWriter pw = new PrintWriter(sw);
            
            pw.println("Error while loading native library \"" +
	        		libName + "\" with base name \""+baseName+"\"");
            pw.println("Operating system name: "+
	        		System.getProperty("os.name"));
            pw.println("Architecture         : "+
	        		System.getProperty("os.arch"));
            pw.println("Architecture bit size: "+
	        		System.getProperty("sun.arch.data.model"));
	        
            pw.println("---(start of nested stack traces)---");
            pw.println(
	            "Stack trace from the attempt to " +
	            "load the library as a file:");
	        throwable.printStackTrace(pw);
	        
	        pw.println(
        		"Stack trace from the attempt to " +
        		"load the library as a resource:");
	        t.printStackTrace(pw);
            pw.println("---(end of nested stack traces)---");
	        
	        pw.close();
	        throw new UnsatisfiedLinkError(sw.toString());
        }
    }

    /**
     * Load the library with the given name from a resource. 
     * The extension for the current OS will be appended.
     * 
     * @param libName The library name
     * @throws Throwable If the library could not be loaded
     */
    private static void loadLibraryResource(String libName) throws Throwable
    {
        // Build the full name of the library 
        String libPrefix = createLibPrefix();
    	String libExtension = createLibExtension();
    	String fullName = libPrefix + libName;
    	String fullNameWithExt = fullName + "." + libExtension;

    	// If a temporary file with the resulting name
    	// already exists, it can simply be loaded
        String tempDirName = System.getProperty("java.io.tmpdir");
        String tempFileName = tempDirName + File.separator + fullNameWithExt;
        File tempFile = new File(tempFileName);
        if (tempFile.exists())
        {
            //System.out.println("Loading from existing file: "+tempFile);
            System.load(tempFile.toString());
            return;
        }
    	
        // No file with the resulting name exists yet. Try to write
        // the library data from the JAR into the temporary file, 
        // and load the newly created file.
        String resourceName = "/lib/" + fullNameWithExt;
        InputStream inputStream = 
        	LibUtils.class.getResourceAsStream(resourceName);
        if (inputStream == null)
        {
        	throw new NullPointerException(
        			"No resource found with name '"+resourceName+"'");
        }
        OutputStream outputStream = null;
        try
        {
        	outputStream = new FileOutputStream(tempFile);
	        byte[] buffer = new byte[8192];
	        while (true)
	        {
	        	int read = inputStream.read(buffer);
	        	if (read < 0)
	        	{
	        		break;
	        	}
	        	outputStream.write(buffer, 0, read);	
	        }
	        outputStream.flush();
	        outputStream.close();
	        outputStream = null;
	        
            //System.out.println("Loading from newly created file: "+tempFile);
            System.load(tempFile.toString());
        }
        finally 
        {
        	if (outputStream != null)
        	{
        		outputStream.close();
        	}
        }
    }


    /**
     * Returns the extension for dynamically linked libraries on the
     * current OS. That is, returns "jnilib" on Apple, "so" on Linux
     * and Sun, and "dll" on Windows.
     * 
     * @return The library extension
     */
    private static String createLibExtension()
    {
        OSType osType = calculateOS();
        switch (osType) 
        {
            case APPLE:
                return "jnilib";
            case LINUX:
                return "so";
            case SUN:
                return "so";
            case WINDOWS:
                return "dll";
        }
        return "";
    }

    /**
     * Returns the prefix for dynamically linked libraries on the
     * current OS. That is, returns "lib" on Apple, Linux and Sun, 
     * and the empty String on Windows.
     * 
     * @return The library prefix
     */
    private static String createLibPrefix()
    {
        OSType osType = calculateOS();
        switch (osType) 
        {
            case APPLE:
            case LINUX:
            case SUN:
                return "lib";
            case WINDOWS:
                return "";
        }
        return "";
    }
    
    
    /**
     * Creates the name for the native library with the given base
     * name for the current operating system and architecture.
     * The resulting name will be of the form<br />
     * baseName-OSType-ARCHType<br />
     * where OSType and ARCHType are the <strong>lower case</strong> Strings
     * of the respective enum constants. Example: <br />
     * JOCL-windows-x86<br /> 
     * 
     * @param baseName The base name of the library
     * @return The library name
     */
    public static String createLibName(String baseName)
    {
        OSType osType = calculateOS();
        ARCHType archType = calculateArch();
        String libName = baseName;
        libName += "-" + osType.toString().toLowerCase(Locale.ENGLISH);
        libName += "-" + archType.toString().toLowerCase(Locale.ENGLISH);
        return libName;
    }
    
    /**
     * Calculates the current OSType
     * 
     * @return The current OSType
     */
    public static OSType calculateOS()
    {
        String osName = System.getProperty("os.name");
        osName = osName.toLowerCase(Locale.ENGLISH);
        if (osName.startsWith("mac os"))
        {
            return OSType.APPLE;
        }
        if (osName.startsWith("windows"))
        {
            return OSType.WINDOWS;
        }
        if (osName.startsWith("linux"))
        {
            return OSType.LINUX;
        }
        if (osName.startsWith("sun"))
        {
            return OSType.SUN;
        }
        return OSType.UNKNOWN;
    }


    /**
     * Calculates the current ARCHType
     * 
     * @return The current ARCHType
     */
    public static ARCHType calculateArch()
    {
        String osArch = System.getProperty("os.arch");
        osArch = osArch.toLowerCase(Locale.ENGLISH);
        if (osArch.equals("i386") || 
            osArch.equals("x86")  || 
            osArch.equals("i686"))
        {
            return ARCHType.X86; 
        }
        if (osArch.startsWith("amd64") || osArch.startsWith("x86_64"))
        {
            return ARCHType.X86_64;
        }
        if (osArch.equals("ppc") || osArch.equals("powerpc"))
        {
            return ARCHType.PPC;
        }
        if (osArch.startsWith("ppc"))
        {
            return ARCHType.PPC_64;
        }
        if (osArch.startsWith("sparc"))
        {
            return ARCHType.SPARC;
        }
        if (osArch.startsWith("arm"))
        {
            return ARCHType.ARM;
        }
        if (osArch.startsWith("mips"))
        {
            return ARCHType.MIPS;
        }
        if (osArch.contains("risc"))
        {
            return ARCHType.RISC;
        }
        return ARCHType.UNKNOWN;
    }    

    /**
     * Private constructor to prevent instantiation.
     */
    private LibUtils()
    {
    }
}




///**
// * Return an up-to-date library file, based on the given temporary file.
// * 
// * @param tempFile The temporary file
// * @param fullName The full library file name (without path)
// * @param libExtension The library file extension
// * @param retry Whether the update process may be retried
// * @return The up-to-date library file
// */
//private static File updateLibraryFile(
//    File tempFile, String fullName, String libExtension, boolean retry)
//{
//    // Implementation note: On Windows, files marked for deletion using
//    // File#deleteOnExit will not be deleted when they are 'open'. 
//    // (See http://bugs.sun.com/bugdatabase/view_bug.do?bug_id=4171239 )
//    // A file that was loaded via System#load is 'open', and will remain
//    // so until the JVM exits. Thus, the temporary files for the libraries
//    // can not be cleaned up properly. This method tries to provide
//    // a workaround for this...:
//
//    // Check if the library file with the expected name already exists
//    String fileNameWithExt = fullName + "." + libExtension;
//    String presentTempFileName = 
//        tempFile.getParent() + File.separator + fileNameWithExt;
//    File presentTempFile = new File(presentTempFileName);
//    if (!presentTempFile.exists())
//    {
//        System.out.println("Li");
//        // If no library file exists yet, then use the newly created file
//        boolean wasRenamed = tempFile.renameTo(presentTempFile);
//        if (!wasRenamed)
//        {
//            throw new UnsatisfiedLinkError(
//                "Could not rename temporary file");
//        }
//        return tempFile;
//    }
//        
//    // So the library file already exists. If the present file has the 
//    // same length (in bytes) as the file currently extracted from the
//    // JAR, then assume that it is the same file. Delete the newly 
//    // created file on exit, and use the present file instead. 
//    if (presentTempFile.length() == tempFile.length())
//    {
//        tempFile.deleteOnExit();
//        return presentTempFile;
//    }
//    
//    
//    // The present file has a different length. Assume that it is an 
//    // older version, and try to delete the present file. This could 
//    // probably be solved more cleanly by including a version number 
//    // in the native library name.  
//    boolean wasDeleted = presentTempFile.delete();
//    if (!wasDeleted)
//    {
//        // The present file could not be deleted. Although this
//        // should hardly ever happen, try the same procedure
//        // once more with a new file name
//        if (!retry)
//        {
//            throw new UnsatisfiedLinkError(
//                "Could not create updated temporary file");
//        }
//        return updateLibraryFile(
//            presentTempFile, fullName + "_new", libExtension, false);
//    }
//    
//    // The present, out-dated tempFile was deleted. Rename
//    // the new tempFile to have the expected name, and 
//    // return it
//    boolean wasRenamed = tempFile.renameTo(presentTempFile);
//    if (!wasRenamed)
//    {
//        throw new UnsatisfiedLinkError(
//            "Could not rename temporary file");
//    }
//    return tempFile;
//}
