package dks.src.checkboardPanel;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.Rectangle;
import java.awt.image.BufferedImage;

import javax.swing.JPanel;

import dks.src.utils.pictures.CPicture;

/**
 * This panel is a checkboard panel : it can display green and white squares to simulate a transparent background
 * <br>
 * date : 3 sept. 07
 * @author DarK Sidious
 */
public class CheckboardPanel extends JPanel {

	private static final long serialVersionUID = 2923588433984451486L;

	/**
	 * The default size of the squares
	 */
	public static final int TILE_SIZE = 7;

	/**
	 * The default color of the dark squares
	 */
	public static final Color DARK_COLOR = new Color(204, 204, 204);

	/**
	 * The default color of the light squares
	 */
	public static final Color LIGHT_COLOR = new Color(255, 255, 255);

	protected int _tileSize = 7;
	protected Color _darkColor = new Color(204, 204, 204);
	protected Color _lightColor = new Color(255, 255, 255);
	protected transient BufferedImage _buffer;

    public CheckboardPanel() {
    	this(TILE_SIZE, DARK_COLOR, LIGHT_COLOR);
    }

    /**
     * @param tileSize the size of the squares
     */
    public CheckboardPanel(int tileSize) {
    	this(tileSize, DARK_COLOR, LIGHT_COLOR);
    }

    /**
     * @param darkColor the dark color of the squares
     * @param lightColor the light color of the squares
     */
    public CheckboardPanel(Color darkColor, Color lightColor) {
    	this(TILE_SIZE, darkColor, lightColor);
    }

    /**
     * @param tileSize the size of the squares
     * @param darkColor the dark color of the squares
     * @param lightColor the light color of the squares
     */
    public CheckboardPanel(int tileSize, Color darkColor, Color lightColor) {
    	_tileSize = tileSize;
    	_darkColor = darkColor;
    	_lightColor = lightColor;
    }

    /**
     * @see javax.swing.JComponent#getPreferredSize()
     */
    public Dimension getPreferredSize() {
        Dimension size = super.getPreferredSize();
        size.setSize(size.width + 50, size.height + 50);
        return size;
    }

    /**
     * @see javax.swing.JComponent#paintComponent(java.awt.Graphics)
     */
    public void paintComponent(Graphics g) {
        if (_buffer == null || getWidth() != _buffer.getWidth() || getHeight() != _buffer.getHeight()) {
            createCheckboard();
        }
        final Rectangle bounds = g.getClipBounds();
        g.drawImage(_buffer.getSubimage(bounds.x, bounds.y, bounds.width, bounds.height), bounds.x, bounds.y, null);
    }

    private void createCheckboard() {
        final int width = getWidth();
        final int height = getHeight();
        _buffer = CPicture.resizeAndClearImage(_buffer, width, height);
        final Graphics2D g = _buffer.createGraphics();

        int boundsStart = 0;
        int boundsEnd = boundsStart + width;
        final int startX = boundsStart - (boundsStart % _tileSize);
        final int endX = boundsEnd + (boundsEnd % _tileSize);

        boundsStart = 0;
        boundsEnd = boundsStart + height;
        final int startY = boundsStart - (boundsStart % _tileSize);
        final int endY = boundsEnd + (boundsEnd % _tileSize);

        for (int x = startX; x < endX; x += _tileSize) {
            final int xdark = x % (_tileSize << 1) == 0 ? _tileSize : 0;
            for (int y = startY; y < endY; y += _tileSize) {
                final boolean dark = ((y + xdark) % (_tileSize << 1)) == 0;
                g.setColor(dark ? _darkColor : _lightColor);
                g.fillRect(x, y, _tileSize, _tileSize);
            }
        }
        g.dispose();
    }

    public String toString() {
    	return "dks.src.checkboardPanel[tileSize=" + _tileSize + ",darkColor=" + _darkColor + ",lightColor=" + _lightColor + "]";
    }
}
