package dks.src.shadowEditor;

import java.awt.AlphaComposite;
import java.awt.Color;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.image.BufferedImage;
import java.awt.image.ConvolveOp;
import java.awt.image.Kernel;
import java.io.ObjectStreamException;
import java.io.Serializable;

import javax.swing.event.ChangeListener;

import org.jdom.Attribute;
import org.jdom.Element;
import org.jdom.JDOMException;

import dks.src.utils.XML.XMLWritable;
import dks.src.utils.XML.XMLWriter;
import dks.src.utils.listener.CChangeListenerDelegate;
import dks.src.utils.listener.Changeable;
import dks.src.utils.pictures.CPicture;

/**
 * This class implement a shadow which can be saved in a XML format <br> date : 3 sept. 07
 * @author   DarK Sidious
 */
public class CShadow implements XMLWritable, Serializable, Changeable {

	private static final long serialVersionUID = 225696894867631404L;

	protected static final String ERROR_SHADOW_OPACITY_NOT_FOUND_DESCRIPTION = "Le fichier xml est invalide : l'opacit de l'ombre n'a pas t trouve";
	protected static final String ERROR_SHADOW_ANGLE_NOT_FOUND_DESCRIPTION = "Le fichier xml est invalide : l'angle de l'ombre n'a pas t trouve";
	protected static final String ERROR_SHADOW_DISTANCE_NOT_FOUND_DESCRIPTION = "Le fichier xml est invalide : la distance de l'ombre n'a pas t trouve";
	protected static final String ERROR_SHADOW_SIZE_NOT_FOUND_DESCRIPTION = "Le fichier xml est invalide : la taille de l'ombre n'a pas t trouve";
	protected static final String ERROR_SHADOW_COLOR_NOT_FOUND_DESCRIPTION = "Le fichier xml est invalide : la couleur de l'ombre n'a pas t trouve";
	protected static final String XML_COLOR_PROPERTY = "color";
	protected static final String XML_SIZE_PROPERTY = "size";
	protected static final String XML_DISTANCE_PROPERTY = "distance";
	protected static final String XML_ANGLE_PROPERTY = "angle";
	protected static final String XML_OPACITY_PROPERTY = "opacity";
	protected static final String XML_VISIBLE_PROPERTY = "visible";

	protected static final float OPACITY = 0.5f;
	protected static final int ANGLE = 315;
	protected static final int SIZE = 10;
	protected static final int DISTANCE = 10;
	protected static final Color COLOR = new Color(128, 128, 128);

	protected Color _color = COLOR;
	protected int _size = SIZE;
	protected int _distance = DISTANCE;
	protected int _angle = ANGLE;
	protected float _opacity = OPACITY;
	protected boolean _visible;

	protected transient Double _Xdistance = 0.0;
	protected transient Double _Ydistance = 0.0;
	protected transient BufferedImage _image;
	protected transient BufferedImage _shadow = null;
	protected transient BufferedImage _subject;
	protected transient boolean _positionChanged;
    protected transient boolean _shadowChanged;
    protected transient CChangeListenerDelegate _changeListeners;

    /**
     * @param image the image to used to define the shadow
     * @param color the color of the shadow
     * @param opacity the opacity of the shadow
     * @param angle the angle of the shadow
     * @param size the size of the shadow
     * @param distance the distance between the shadow and the image
     */
	public CShadow(BufferedImage image, Color color, float opacity, int angle, int size, int distance) {
		super();
		_color = color;
		_opacity = opacity;
		_angle = angle;
		_size = size;
		_distance = distance;
		_visible = true;
		_positionChanged = true;
		_shadowChanged = true;
		_changeListeners = new CChangeListenerDelegate();
		init(image);
	}

	protected void init(BufferedImage image) {
		setImage(image);
	}

	/**
     * @param image the image to used to define the shadow
     * @param color the color of the shadow
     * @param opacity the opacity of the shadow
     * @param angle the angle of the shadow
     * @param size the size of the shadow
     */
	public CShadow(BufferedImage image, Color color, float opacity, int angle, int size) {
		this(image, color, opacity, angle, size, DISTANCE);
	}

	/**
     * @param image the image to used to define the shadow
     * @param color the color of the shadow
     * @param opacity the opacity of the shadow
     * @param angle the angle of the shadow
     */
	public CShadow(BufferedImage image, Color color, float opacity, int angle) {
		this(image, color, opacity, angle, SIZE, DISTANCE);
	}

	/**
     * @param image the image to used to define the shadow
     * @param color the color of the shadow
     * @param opacity the opacity of the shadow
	 */
	public CShadow(BufferedImage image, Color color, float opacity) {
		this(image, color, opacity, ANGLE, SIZE, DISTANCE);
	}

	/**
     * @param image the image to used to define the shadow
     * @param color the color of the shadow
	 */
	public CShadow(BufferedImage image, Color color) {
		this(image, color, OPACITY, ANGLE, SIZE, DISTANCE);
	}

	/**
     * @param image the image to used to define the shadow
	 */
	public CShadow(BufferedImage image) {
		this(image, COLOR, OPACITY, ANGLE, SIZE, DISTANCE);
	}

	public CShadow() {
		this(null, COLOR, OPACITY, ANGLE, SIZE, DISTANCE);
	}

	/**
	 * @return the angle of the shadow
	 */
	public int getAngle() {
		return _angle;
	}

	/**
	 * @param angle the angle of the shadow
	 */
	public void setAngle(int angle) {
		if (_angle != angle) {
			_angle = angle;
			_positionChanged = true;
			_changeListeners.notifyChanged();
		}
	}

	/**
	 * @return the color of the shadow
	 */
	public Color getColor() {
		return _color;
	}

	/**
	 * @param color the color of the shadow
	 */
	public void setColor(Color color) {
		if (!_color.equals(color)) {
			_color = color;
			_shadowChanged = true;
			_changeListeners.notifyChanged();
		}
	}

	/**
	 * @return the distance between the image and the shadow
	 */
	public int getDistance() {
		return _distance;
	}

	/**
	 * @param distance the distance between the image and the shadow
	 */
	public void setDistance(int distance) {
		if (_distance != distance) {
			_distance = distance;
			_positionChanged = true;
			_changeListeners.notifyChanged();
		}
	}

	/**
	 * @return the image used to define the shadow
	 */
	public BufferedImage getImage() {
		return _image;
	}

	/**
	 * @param image the image used to define the shadow
	 */
	public void setImage(BufferedImage image) {
		if (_image != image) {
			_image = image;
			_shadowChanged = true;
			_changeListeners.notifyChanged();
		}
	}

	/**
	 * @return the opacity of the shadow
	 */
	public float getOpacity() {
		return _opacity;
	}

	/**
	 * @param opacity the opacity of the shadow
	 */
	public void setOpacity(float opacity) {
		if (_opacity != opacity) {
			_opacity = opacity;
			_shadowChanged = true;
			_changeListeners.notifyChanged();
		}
	}

	/**
	 * @return the size of the shadow
	 */
	public int getSize() {
		return _size;
	}

	/**
	 * @param size the size of the shadow
	 */
	public void setSize(int size) {
		if (_size != size) {
			_size = size;
			_shadowChanged = true;
			_changeListeners.notifyChanged();
		}
	}

	/**
	 * @see java.lang.Object#hashCode()
	 * @return the hashCode of the object
	 */
	@Override
	public int hashCode() {
		final int PRIME = 31;
		int result = 1;
		result = PRIME * result + _angle;
		result = PRIME * result + ((_color == null) ? 0 : _color.hashCode());
		result = PRIME * result + _distance;
		result = PRIME * result + Float.floatToIntBits(_opacity);
		result = PRIME * result + _size;
		result = PRIME * result + (_visible ? 1231 : 1237);
		return result;
	}

	/**
	 * @see java.lang.Object#equals(java.lang.Object)
	 * @param obj The object to compare
	 * @return The equality of the object
	 */
	@Override
	public boolean equals(Object obj) {
		if (obj == null) {
			return false;
		}
		if (!(obj instanceof CShadow)) {
			return false;
		}
		final CShadow other = (CShadow) obj;
		if (_angle != other._angle) {
			return false;
		}
		if (_color == null) {
			if (other._color != null) {
				return false;
			}
		} else if (!_color.equals(other._color)) {
			return false;
		}
		if (_distance != other._distance) {
			return false;
		}
		if (Float.floatToIntBits(_opacity) != Float.floatToIntBits(other._opacity)) {
			return false;
		}
		if (_size != other._size) {
			return false;
		}
		if (_visible != other._visible) {
			return false;
		}
		return true;
	}

	/**
	 * the visibility of the shadow
	 * @return the visibility of the shadow
	 */
	public boolean isVisible() {
		return _visible;
	}

	/**
	 * the visibility of the shadow
	 * @param visible the visibility of the shadow
	 */
	public void setVisible(boolean visible) {
		_visible = visible;
	}

	/**
	 * @param g the graphics used to display the shadow
	 * @param x the x position of the shadow
	 * @param y the y position of the shadow
	 */
	public void draw(Graphics g, int x, int y) {
		draw(g, x, y, -1, -1);
    }

	/**
	 * @param g the graphics used to display the shadow
	 * @param x the x position of the shadow
	 * @param y the y position of the shadow
	 * @param width the width of the shadow
	 * @param height the height of the shadow
	 */
	public void draw(Graphics g, int x, int y, int width, int height) {
		draw(g, x, y, width, height, -1, -1, -1, -1);
    }

	/**
	 * @param g the graphics used to display the shadow
	 * @param x the x position of the shadow
	 * @param y the y position of the shadow
	 * @param width the width of the shadow
	 * @param height the height of the shadow
	 * @param srcX the x position of the source picture
	 * @param srcY the y position of the source picture
	 * @param srcWidth the width of the source picture
	 * @param srcHeight the height of the source picture
	 */
	public void draw(Graphics g, int x, int y, int width, int height, int srcX, int srcY, int srcWidth, int srcHeight) {
		if (_positionChanged) {
			computeShadowPosition();
			_positionChanged = false;
		}
		if (_shadowChanged) {
			refreshShadow();
			_shadowChanged = false;
		}
		if (_shadow != null) {
			if (srcX == -1 && srcY == -1 && srcWidth == -1 && srcHeight == -1) {
				if (width == -1 && height == -1) {
					g.drawImage(_shadow, (int) (x + _Xdistance), (int) (y + _Ydistance), null);
				} else {
					g.drawImage(_shadow, (int) (x + _Xdistance), (int) (y + _Ydistance), width, height, null);
				}
			} else {
				g.drawImage(_shadow, (int) (x + _Xdistance), (int) (y + _Ydistance), width, height, srcX, srcY, srcWidth, srcHeight, null);
			}

        }

        if (_image != null) {
        	if (srcX == -1 && srcY == -1 && srcWidth == -1 && srcHeight == -1) {
				if (width == -1 && height == -1) {
					g.drawImage(_image, (int) x, (int) y, null);
				} else {
					g.drawImage(_image, (int) x, (int) y, width, height, null);
				}
        	} else {
        		g.drawImage(_image, (int) (x + _Xdistance), (int) (y + _Ydistance), width, height, srcX, srcY, srcWidth, srcHeight, null);
        	}
        }
    }

	/**
	 * @see dks.src.utils.XML.XMLWritable#XMLload(org.jdom.Element)
	 * @param root the XML DOM element used to load the properties of the shadow
	 */
	public void XMLload(Element root) throws JDOMException {
		_color = XMLWriter.getColorElement(XML_COLOR_PROPERTY, root);
		if (_color == null) {
			throw new JDOMException(ERROR_SHADOW_COLOR_NOT_FOUND_DESCRIPTION);
		}

		String size = root.getAttributeValue(XML_SIZE_PROPERTY);
		if (size == null) {
			throw new JDOMException(ERROR_SHADOW_SIZE_NOT_FOUND_DESCRIPTION);
		}
		_size = Integer.parseInt(size);

		String distance = root.getAttributeValue(XML_DISTANCE_PROPERTY);
		if (distance == null) {
			throw new JDOMException(ERROR_SHADOW_DISTANCE_NOT_FOUND_DESCRIPTION);
		}
		_distance = Integer.parseInt(distance);

		String angle = root.getAttributeValue(XML_ANGLE_PROPERTY);
		if (angle == null) {
			throw new JDOMException(ERROR_SHADOW_ANGLE_NOT_FOUND_DESCRIPTION);
		}
		_angle = Integer.parseInt(angle);

		String opacity = root.getAttributeValue(XML_OPACITY_PROPERTY);
		if (opacity == null) {
			throw new JDOMException(ERROR_SHADOW_OPACITY_NOT_FOUND_DESCRIPTION);
		}
		_opacity = Float.parseFloat(opacity);

		Attribute attribute = root.getAttribute(XML_VISIBLE_PROPERTY);
		if (attribute != null) { // Fonctionnalits ajoute dans la version 1.0.0 => compatibilit ascendante avec les fichiers de la version 1.0.0
			_visible = Boolean.parseBoolean(attribute.getValue());
		}
	}

	/**
	 * @see dks.src.utils.XML.XMLWritable#XMLsave(org.jdom.Element)
	 * @param root the XML Dom element used to save the properties of the shadow
	 */
	public void XMLsave(Element root) {
		root.addContent(XMLWriter.createColorElement(XML_COLOR_PROPERTY, _color));
		root.setAttribute(XML_SIZE_PROPERTY, Integer.valueOf(_size).toString());
		root.setAttribute(XML_DISTANCE_PROPERTY, Integer.valueOf(_distance).toString());
		root.setAttribute(XML_ANGLE_PROPERTY, Integer.valueOf(_angle).toString());
		root.setAttribute(XML_OPACITY_PROPERTY, Float.valueOf(_opacity).toString());
		root.setAttribute(XML_VISIBLE_PROPERTY, Boolean.valueOf(_visible).toString());
	}

	protected void computeShadowPosition() {
		double angleRadians = Math.toRadians(_angle);
        _Xdistance = (Math.cos(angleRadians) * _distance);
        _Ydistance = (Math.sin(angleRadians) * _distance);
        if (_Xdistance < 0) {
        	_Xdistance *= 2;
        }
        if (_Ydistance < 0) {
        	_Ydistance *= 2;
        }
    }

	protected void createDropShadow(BufferedImage image) {
		int width = image.getWidth();
		int height = image.getHeight();
		_subject = CPicture.resizeAndClearImage(_subject, width + _size, height + _size);
		int subjectWidth = width + _size;
		int subjectHeight = height + _size;
		Graphics2D g2d = (Graphics2D) _subject.getGraphics();
		g2d.setBackground(new Color(0, 0, 0, 0));
		g2d.clearRect(0, 0, subjectWidth, subjectHeight);
		g2d.drawImage(image, _size, _size, null);
		g2d.dispose();

		_shadow = CPicture.resizeAndClearImage(_shadow, subjectWidth, subjectHeight);

		createShadowMask(_subject);
		getLinearBlurOp(_size).filter(_subject, _shadow);
    }

	protected ConvolveOp getLinearBlurOp(int size) {
        float[] data = new float[size * size];
        float value = 1.0f / (float) (size * size);
        for (int i = 0; i < data.length; i++) {
            data[i] = value;
        }
        return new ConvolveOp(new Kernel(size, size, data));
    }

	protected void createShadowMask(BufferedImage image) {
        Graphics2D g2d = (Graphics2D) image.getGraphics();
        g2d.setComposite(AlphaComposite.getInstance(AlphaComposite.SRC_IN, _opacity));
        g2d.setColor(_color);
        g2d.fillRect(0, 0, image.getWidth(), image.getHeight());
        g2d.dispose();
    }

	protected void refreshShadow() {
        if (_image != null) {
            createDropShadow(_image);
        }
    }

	/**
	 * @param listener
	 * @see dks.src.utils.listener.CListenerDelegate#addListener(java.lang.Object)
	 */
	public void addChangeListener(ChangeListener listener) {
		_changeListeners.addListener(listener);
	}

	/**
	 * @param listener
	 * @see dks.src.utils.listener.CListenerDelegate#removeListener(java.lang.Object)
	 */
	public void removeChangeListener(ChangeListener listener) {
		_changeListeners.removeListener(listener);
	}

	public String toString() {
		return "src.dks.shadowEditor.CShadow[color=" + _color + ",size=" + _size + ",distance=" + _distance + ",angle=" + _angle + ",opacity=" + _opacity + ",visible=" + _visible + "]";
	}

	protected Object readResolve() throws ObjectStreamException {
		_Xdistance = 0.0;
		_Ydistance = 0.0;
		_positionChanged = true;
	    _shadowChanged = true;
	    _changeListeners = new CChangeListenerDelegate();
		return this;
	}
}
