package dks.src.textureEditor;

import java.awt.AlphaComposite;
import java.awt.Graphics2D;
import java.awt.Image;
import java.awt.Paint;
import java.awt.Rectangle;
import java.awt.TexturePaint;
import java.awt.Toolkit;
import java.awt.image.BufferedImage;
import java.awt.image.FilteredImageSource;
import java.io.File;
import java.io.IOException;
import java.io.ObjectStreamException;
import java.io.Serializable;

import javax.imageio.ImageIO;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;

import org.jdom.Element;
import org.jdom.JDOMException;

import dks.src.utils.XML.XMLWritable;
import dks.src.utils.listener.CChangeListenerDelegate;
import dks.src.utils.listener.Changeable;
import dks.src.utils.pictures.CPicture;

/**
 * This class implements a texture composed by 2 layers which can be saved in a XML format <br> date : 4 sept. 07
 * @author   DarK Sidious
 */
public class CTexture implements XMLWritable, Serializable, Changeable {

	private static final long serialVersionUID = -5203818560744297996L;

	protected static final String XML_WIDTH_PROPERTY = "width";
	protected static final String XML_HEIGHT_PROPERTY = "height";
	protected static final String XML_TEXTURE_FILENAME_PROPERTY = "texture";
	protected static final String XML_MASK_FILENAME_PROPERTY = "mask";
	protected static final String XML_BACKGROUND_LAYER_PROPERTY = "backgroundLayer";
	protected static final String XML_FOREGROUND_LAYER_PROPERTY = "foregroundLayer";

	protected CLayer _backgroundLayer;
	protected CLayer _foregroundLayer;
	protected int _width;
	protected int _height;
	protected String _maskFileName;
	protected String _textureFileName;

	protected transient BufferedImage _texture;
	protected transient BufferedImage _mask;
	protected transient Image _maskImage;
	protected transient BufferedImage _maskImageLayer;
	protected transient BufferedImage _maskImageLayerTemp;
	protected transient BufferedImage _result;
	protected transient BufferedImage _foregroundImage;
	protected transient BufferedImage _textureImage;
	protected transient boolean _changed;
	protected transient boolean _maskChanged;
	protected transient Change _changeListener;
	protected transient CChangeListenerDelegate _changeListenerDelegate;

	public CTexture() {}

	/**
	 * @param backgroundLayer the background layer of the texture
	 * @param foregroundLayer the foreground layer of the texture
	 * @param width the width of the texture
	 * @param height the height of the texture
	 * @param texture the texture picture
	 * @param mask the mask picture
	 */
	public CTexture(CLayer backgroundLayer, CLayer foregroundLayer, Integer width, Integer height, BufferedImage texture, BufferedImage mask) {
		_changeListener = new Change();
		_changeListenerDelegate = new CChangeListenerDelegate();
		_backgroundLayer = backgroundLayer;
		_backgroundLayer.addChangeListener(_changeListener);
		_foregroundLayer = foregroundLayer;
		_foregroundLayer.addChangeListener(_changeListener);
		_texture = texture;
		_mask = mask;
		_maskChanged = true;
		_changed = true;
		_width = width;
		_height = height;
	}

	/**
	 * @return the background layer of the texture
	 */
	public CLayer getBackgroundLayer() {
		return _backgroundLayer;
	}

	/**
	 * @param backgroundLayer the background layer of the texture
	 */
	public void setBackgroundLayer(CLayer backgroundLayer) {
		if (_backgroundLayer == null || !_backgroundLayer.equals(backgroundLayer)) {
			_changed = true;
		}
		_backgroundLayer = backgroundLayer;
	}

	/**
	 * @return the foreground layer of the texture
	 */
	public CLayer getForegroundLayer() {
		return _foregroundLayer;
	}

	/**
	 * @param foregroundLayer the foreground layer of the texture
	 */
	public void setForegroundLayer(CLayer foregroundLayer) {
		if (_foregroundLayer == null || !_foregroundLayer.equals(foregroundLayer)) {
			_changed = true;
		}
		_foregroundLayer = foregroundLayer;
	}

	/**
	 * @return the mask picture of the texture
	 */
	public BufferedImage getMask() {
		return _mask;
	}

	/**
	 * @param mask the mask picture of the texture
	 */
	public void setMask(BufferedImage mask) {
		if (_mask == null || !_mask.equals(mask)) {
			_changed = true;
			_maskChanged = true;
			_mask = mask;
		}
	}

	/**
	 * @return the texture picture
	 */
	public BufferedImage getTexture() {
		return _texture;
	}

	/**
	 * @param texture the texture picture
	 */
	public void setTexture(BufferedImage texture) {
		if (_texture == null || !_texture.equals(texture)) {
			_changed = true;
		}
		_texture = texture;
	}

	/**
	 * @return the fileName of the mask picture
	 */
	public String getMaskFileName() {
		return _maskFileName;
	}

	/**
	 * @param maskFileName the filename of the mask picture
	 */
	public void setMaskFileName(String maskFileName) {
		_maskFileName = maskFileName;
	}

	/**
	 * @return the filename of the texture picture
	 */
	public String getTextureFileName() {
		return _textureFileName;
	}

	/**
	 * @param textureFileName the filename of the texture picture
	 */
	public void setTextureFileName(String textureFileName) {
		_textureFileName = textureFileName;
	}

	/**
	 * @return the height of the texture
	 */
	public int getHeight() {
		return _height;
	}

	/**
	 * @param height the height of the texture
	 */
	public void setHeight(int height) {
		if (_height != height) {
			_changed = true;
			_height = height;
		}
	}

	/**
	 * @return the width of the texture
	 */
	public int getWidth() {
		return _width;
	}

	/**
	 * @param width the width of the texture
	 */
	public void setWidth(int width) {
		if (_width != width) {
			_changed = true;
			_width = width;
		}
	}

	protected void computeTexture() {
		_result = CPicture.resizeAndClearImage(_result, _width, _height);
		final Graphics2D graphics = (Graphics2D) _result.getGraphics();
		if (_backgroundLayer != null) {
			switch (_backgroundLayer.getType()) {
			case ColoredLayer:
				graphics.setColor(_backgroundLayer.getColor());
				graphics.fillRect(0, 0, _width, _height);
				break;
			case ImageLayer:
				graphics.drawImage(_backgroundLayer.getImage(), 0, 0, null);
				break;
			case GradientLayer:
				Paint paint = _backgroundLayer.getGradient().getPaint(new Rectangle(0, 0, _width, _height));
				graphics.setPaint(paint);
				graphics.fillRect(0, 0, _width, _height);
				break;
			default:
			}
		}
		if (_foregroundLayer != null) {
			_foregroundImage = CPicture.resizeAndClearImage(_foregroundImage, _width, _height);
			_textureImage = CPicture.resizeAndClearImage(_textureImage, _width, _height);
			final Graphics2D graphicsForeground = _textureImage.createGraphics();
			switch (_foregroundLayer.getType()) {
			case ColoredLayer:
				graphicsForeground.setColor(_foregroundLayer.getColor());
				graphicsForeground.fillRect(0, 0, _width, _height);
				break;
			case ImageLayer:
				graphicsForeground.drawImage(_foregroundLayer.getImage(), 0, 0, null);
				break;
			case GradientLayer:
				Paint paint = _foregroundLayer.getGradient().getPaint(new Rectangle(0, 0, _width, _height));
				graphicsForeground.setPaint(paint);
				graphicsForeground.fillRect(0, 0, _width, _height);
				break;
			default:
				break;
			}
			graphicsForeground.dispose();

			if (_texture != null) {
				final Graphics2D g2 = _foregroundImage.createGraphics();
				g2.drawImage(_textureImage, 0, 0, null);
				g2.setComposite(AlphaComposite.getInstance(AlphaComposite.SRC_OVER, 0.5f));
				g2.drawImage(_texture, 0, 0, null);
				g2.dispose();
			} else {
				final Graphics2D g2 = _foregroundImage.createGraphics();
				g2.drawImage(_textureImage, 0, 0, null);
				g2.dispose();
			}

			if (_mask != null) {
				if (_maskChanged) {
					_maskImage = Toolkit.getDefaultToolkit().createImage(new FilteredImageSource(_mask.getSource(), new CMaskFilter()));
					_maskImageLayer = CPicture.resizeAndClearImage(_maskImageLayer, _width, _height);
					_maskImageLayer.getGraphics().drawImage(_maskImage, 0, 0, null);
					_maskImageLayerTemp = CPicture.resizeAndClearImage(_maskImageLayerTemp, _width, _height);
					final Graphics2D g2 = _maskImageLayerTemp.createGraphics();
					g2.drawImage(_foregroundImage, 0, 0, null);
					g2.setComposite(AlphaComposite.DstOut);
					g2.drawImage(_maskImageLayer, null, 0, 0);
					g2.dispose();
				}
				graphics.drawImage(_maskImageLayerTemp, 0, 0, null);
			} else {
				graphics.drawImage(_foregroundImage, 0, 0, null);
			}
		}
		graphics.dispose();
	}

	/**
	 * @return the paint to use for applying the texture
	 */
	public Paint getPaint() {
		if (_changed) {
			computeTexture();
		}
		return new TexturePaint(_result, new Rectangle(0, 0, _width, _height));
	}

	protected class Change implements ChangeListener {
		public void stateChanged(ChangeEvent arg0) {
			_changed = true;
		}
	}

	/**
	 * @see dks.src.utils.XML.XMLWritable#XMLload(org.jdom.Element)
	 * @param root the XML Dom Element to use to load the properties of the texture
	 */
	public void XMLload(Element root) throws JDOMException {
		_backgroundLayer.XMLload(root.getChild(XML_BACKGROUND_LAYER_PROPERTY));
		_foregroundLayer.XMLload(root.getChild(XML_FOREGROUND_LAYER_PROPERTY));

		final String width = root.getAttributeValue(XML_WIDTH_PROPERTY);
		if (width == null) {
			throw new JDOMException("Le fichier xml est invalide : la largeur de la texture n'a pas t trouve");
		}
		_width = Integer.parseInt(width);

		final String height = root.getAttributeValue(XML_HEIGHT_PROPERTY);
		if (height == null) {
			throw new JDOMException("Le fichier xml est invalide : la hauteur de la texture n'a pas t trouve");
		}
		_height = Integer.parseInt(height);
		_maskFileName = root.getAttributeValue(XML_MASK_FILENAME_PROPERTY);
		_textureFileName = root.getAttributeValue(XML_TEXTURE_FILENAME_PROPERTY);
		if (_maskFileName != null) {
			final File temp = new File(_maskFileName);
			if (temp.exists()) {
				try {
					final BufferedImage image = ImageIO.read(temp);
					setMask(image);
				}
				catch (IOException e) {
					e.printStackTrace();
				}
			} else {
				setTexture(null);
			}
		}
		if (_textureFileName != null) {
			final File temp = new File(_textureFileName);
			if (temp.exists()) {
				try {
					final BufferedImage image = ImageIO.read(temp);
					setTexture(image);
				} catch (IOException e) {}
			} else {
				setTexture(null);
			}
		}
	}

	/**
	 * @see dks.src.utils.XML.XMLWritable#XMLsave(org.jdom.Element)
	 * @param root the XML DOM Element used to save the properties of the texture
	 */
	public void XMLsave(Element root) {
		root.setAttribute(XML_WIDTH_PROPERTY, Integer.valueOf(_width).toString());
		root.setAttribute(XML_HEIGHT_PROPERTY, Integer.valueOf(_height).toString());
		if (_textureFileName != null) {
			root.setAttribute(XML_TEXTURE_FILENAME_PROPERTY, _textureFileName);
		}
		if (_maskFileName != null) {
			root.setAttribute(XML_MASK_FILENAME_PROPERTY, _maskFileName);
		}
		final Element backgroundLayer = new Element(XML_BACKGROUND_LAYER_PROPERTY);
		root.addContent(backgroundLayer);
		_backgroundLayer.XMLsave(backgroundLayer);
		final Element foregroundLayer = new Element(XML_FOREGROUND_LAYER_PROPERTY);
		root.addContent(foregroundLayer);
		_foregroundLayer.XMLsave(foregroundLayer);
	}

	/**
	 * @param listener the change listener to add which is invoke when the texture change
	 * @see dks.src.utils.listener.CListenerDelegate#addListener(java.lang.Object)
	 */
	public void addChangeListener(ChangeListener listener) {
		_changeListenerDelegate.addListener(listener);
	}

	/**
	 * @param listener the change listener to remove
	 * @see dks.src.utils.listener.CListenerDelegate#removeListener(java.lang.Object)
	 */
	public void removeChangeListener(ChangeListener listener) {
		_changeListenerDelegate.removeListener(listener);
	}

	/**
	 * @see java.lang.Object#hashCode()
	 * @return the hashcode of the object
	 */
	public int hashCode() {
		final int PRIME = 31;
		int result = 1;
		result = PRIME * result + ((_backgroundLayer == null) ? 0 : _backgroundLayer.hashCode());
		result = PRIME * result + ((_foregroundLayer == null) ? 0 : _foregroundLayer.hashCode());
		result = PRIME * result + _height;
		result = PRIME * result + ((_maskFileName == null) ? 0 : _maskFileName.hashCode());
		result = PRIME * result + ((_textureFileName == null) ? 0 : _textureFileName.hashCode());
		result = PRIME * result + _width;
		return result;
	}

	/**
	 * @see java.lang.Object#equals(java.lang.Object)
	 * @param obj the object to compare
	 * @return the equality of the object
	 */
	public boolean equals(Object obj) {
		if (obj == null) {
			return false;
		}
		if (!(obj instanceof CTexture)) {
			return false;
		}
		final CTexture other = (CTexture) obj;
		if (_backgroundLayer == null) {
			if (other._backgroundLayer != null) {
				return false;
			}
		} else if (!_backgroundLayer.equals(other._backgroundLayer)) {
			return false;
		}
		if (_foregroundLayer == null) {
			if (other._foregroundLayer != null) {
				return false;
			}
		} else if (!_foregroundLayer.equals(other._foregroundLayer)) {
			return false;
		}
		if (_height != other._height) {
			return false;
		}
		if (_maskFileName == null) {
			if (other._maskFileName != null) {
				return false;
			}
		} else if (!_maskFileName.equals(other._maskFileName)) {
			return false;
		}
		if (_textureFileName == null) {
			if (other._textureFileName != null) {
				return false;
			}
		} else if (!_textureFileName.equals(other._textureFileName)) {
			return false;
		}
		if (_width != other._width) {
			return false;
		}
		return true;
	}

	public String toString() {
		return "dks.src.textureEditor.CTexture[backgroundLayer=" + _backgroundLayer + ",foregroundLayer=" + _foregroundLayer + ",width=" + _width + ",height=" + _height + ",maskFileName=" + _maskFileName + ",textureFileName=" + _textureFileName + "]";
	}

	protected Object readResolve() throws ObjectStreamException {
		_changeListener = new Change();
		_changeListenerDelegate = new CChangeListenerDelegate();
		_changed = true;
		_maskChanged = true;
		return this;
	}
}
