
/**
 *  Classroom Scheduler
 *  Copyright (C) 2004 Colin Archibald, Ph.D.
 *  https://sourceforge.net/projects/cr-scheduler/
 *
 *  Licensed under the Academic Free License version 2.0
 */

package application;

import java.awt.*;
import java.awt.print.*;
import java.io.*;
import java.text.DateFormat;
import java.util.*;

import resources.*;

/**
 *	The Schedule maintains the data for a particular Schedule being manipulated
 *	by the application.
 *
 *	Uses Singleton model.  When someone asks for it, the method will either
 *	create a new one or give the existing one.
 *
 */

public class Schedule extends Observable
implements Constants, Serializable, Printable, Pageable, Observer {
    static final long serialVersionUID = 911;
    
    public static final String DEFAULT_FILE_NAME = "Untitled.crs";
    
    private static Schedule schedule = null;
    private static boolean changed = false; // used for save warning.
    private Date lastUpdated = new Date();
    
    private String scheduleName;
    private String directory;    // Directory for this document
    private ArrayList professors;
    private ArrayList classrooms;
    private ArrayList timeSlots;
    private ArrayList courses;
    private ArrayList schedCourses;
    private ArrayList conflicts;
    private ArrayList books;    
    private Summary summary = null; // The Summary object for printing and display
    
    /**
     * There is no public constructor for a schedule.  Only the static
     * singleton method "getSchedule" can create a schedule using the
     * default constructor for a Schedule
     */
    
    private Schedule() {
        scheduleName = DEFAULT_FILE_NAME;
        directory = ".";    // Directory for this document
        professors = new ArrayList(20);
        classrooms = new ArrayList(20);
        timeSlots = new ArrayList(20);
        courses = new ArrayList(20);
        schedCourses = new ArrayList(20);
        conflicts = new ArrayList(20);
        books = new ArrayList(20);     
        
        addTheTBAs();        
    }
    
    /**
     * Any time the schedule changes, we will set a changed flag.  The
     * schedule will be an observer of itself.
     *
     */
   
    synchronized public void update(Observable self, Object obj)  {
        changed = true;
        lastUpdated = new Date();
    }
    private void setLastUpdated(Date d) {
        lastUpdated = d;
    }
    private Date getLastUpdated() {
        return lastUpdated;
    }
    public void setChanged(boolean b) {
        changed = b;
    }
    public boolean getChanged() {
        return changed;
    }
    public String updated() {
        DateFormat df =
        DateFormat.getDateTimeInstance(DateFormat.MEDIUM,DateFormat.SHORT);
        return "   Updated: " + df.format(lastUpdated);
    }
    
    synchronized public void resetSchedule(Schedule localSchedule, String name) {
        if (localSchedule != null) {
            schedule.setScheduleName(name);
            schedule.setDirectory(localSchedule.getDirectory());
            schedule.setProfessors(localSchedule.getProfessors());
            schedule.setClassrooms(localSchedule.getClassrooms());
            schedule.setTimeSlots(localSchedule.getTimeSlots());
            schedule.setCourses(localSchedule.getCourses());
            schedule.setBooks(localSchedule.getBooks());
            schedule.setSchedCourses(localSchedule.getSchedCourses());
            schedule.setLastUpdated(localSchedule.getLastUpdated());            
            schedule.setChanged(false);
        } else {
            scheduleName = DEFAULT_FILE_NAME;
            directory = ".";    // Directory for this document
            professors = new ArrayList(20);
            classrooms = new ArrayList(20);
            timeSlots = new ArrayList(20);
            courses = new ArrayList(20);
            books = new ArrayList(20);
            schedCourses = new ArrayList(20);
            conflicts = new ArrayList(20);            
            summary = new Summary();
            addTheTBAs();
            
            setChanged();	// Mark the document as changed
            notifyObservers();	// Tell everybody
        }        
    }
    
    private void addTheTBAs() {
        professors.add(new Professor(" ", " STAFF", Professor.ADJUNCT));
        classrooms.add(new Classroom(" ", "TBA", 0));
        timeSlots.add( new TimeSlot().setTba() );
        timeSlots.add( new TimeSlot().setOnline() );

    }
    
    /** Allows access to the schedule to anyone who wants it without needing to pass
     * it around as a parameter.
     */
    synchronized public static Schedule getSchedule() {
        if (schedule == null) {
            schedule = new Schedule();            
            return schedule;
        } else {
            return schedule;
        }
    }
    
    synchronized public String getScheduleName() {
        return scheduleName;
    }
    
    synchronized public void setScheduleName(String scheduleName) {
        this.scheduleName = scheduleName;
    }
    
    // Get the directory where the document is stored
    
    synchronized public String getDirectory() {
        return directory;
    }
    
    // Set the directory for the document
    
    /**
     * Sets the directory for the document
     * @param String directory the directory
     */
    
    synchronized public void setDirectory(String directory) {
        this.directory = directory;
    }
    
    
    /**
     * Gets the instructors.
     * @return a <code>Vector</code> specifying the instructors
     */
    
    public ArrayList getProfessors() {
        return professors;
    }
    
    synchronized public void setProfessors(ArrayList professors) {
        this.professors = professors;
        setChanged();					    // Mark the document as changed
        notifyObservers();					// Tell everybody
    }
    
    /**
     * Gets the class rooms.
     * @return a <code>Vector</code> specifying the class rooms.
     */
    
    synchronized public ArrayList getClassrooms() {
        return classrooms;
    }
    
    synchronized public void setClassrooms(ArrayList classrooms) {
        this.classrooms = classrooms;
        setChanged();					    // Mark the document as changed
        notifyObservers();					// Tell everybody
    }
    
    /**
     * Gets the Courses.
     * @return  Vector specifying the course sections.
     */
    
    synchronized public ArrayList getCourses() {
        return courses;
    }
    
    synchronized public void setCourses(ArrayList courses) {
        this.courses = courses;
        setChanged();					    // Mark the document as changed
        notifyObservers();					// Tell everybody
    }
    
    /**
     * Gets the time slots.
     * @return  Vector specifying the course sections.
     */
    
    public ArrayList getTimeSlots() {
        return timeSlots;
    }
    
    synchronized public void setTimeSlots(ArrayList timeSlots) {
        this.timeSlots = timeSlots;
        setChanged();					    // Mark the document as changed
        notifyObservers();					// Tell everybody
    }
    
    public ArrayList getBooks(){
        return books;
    }
    public void setBooks(ArrayList books){
        this.books = books;
        setChanged();
        notifyObservers();
    }
    /**
     * Gets the Scheduled Courses.
     * @return   Vector   specifying the scheduled courses.
     */
    
    public ArrayList getSchedCourses() {
        return schedCourses;
    }
    
    synchronized public void setSchedCourses(ArrayList schedCourses) {
        this.schedCourses = schedCourses;
        setChanged();					    // Mark the document as changed
        notifyObservers();					// Tell everybody
    }        
    
    /**
     * Gets the Conflicts.
     * @return Vector specifying the conflicts.
     */
    
    synchronized public ArrayList getConflicts() {
        return conflicts;
    }
    
    synchronized public Summary getSummary() {
        if(summary == null) {
            summary = new Summary();            
        }
        return summary;        
    }
    
    /**
     * Adds an Professor to the document
     * @param Professor to be added
     */
    
    synchronized public void addProfessor(Professor prof) {
        if(!professors.contains(prof)){
            professors.add(prof);
            Collections.sort(professors);
            setChanged();					    // Mark the document as changed
            notifyObservers();					// Tell everybody
        }
    }
    
    /**
     * Removes a Professor from the document
     * @param Professor to be removed
     */
    
    synchronized public void removeProfessor(Professor prof) {
        professors.remove(prof);
        setChanged();
        notifyObservers();
    }
    
    /**
     * Adds a Classroom to the document
     * @param Classroom to be added
     */
    
    synchronized public void addClassroom(Classroom classroom) {
        if(!classrooms.contains(classroom)){
            classrooms.add(classroom);
            Collections.sort(classrooms);
            setChanged();				// Mark the document as changed
            notifyObservers();				// Tell everybody
        }
    }
    
    /**
     * Removes a Classroom from the document
     * @param Classroom to be removed
     */
    
    synchronized public void removeClassroom(Classroom classroom) {
        classrooms.remove(classroom);
        setChanged();
        notifyObservers();
    }
    
    /**
     * Adds a TimeSlot to the document
     * @param TimeSlot to be added
     */
    
    synchronized public void addTimeSlot(TimeSlot timeSlot) {
        if(!timeSlots.contains(timeSlot)){
            timeSlots.add(timeSlot);
            Collections.sort(timeSlots);
            setChanged();			// Mark the document as changed
            notifyObservers();			// Tell everybody
        }
    }
    
    /**
     * Removes a TimeSlot from the document
     * @param TimeSlot to be removed
     */
    
    synchronized public void removeTimeSlot(TimeSlot timeSlot) {
        timeSlots.remove(timeSlot);
        setChanged();
        notifyObservers();
    }
    
    /**
     * Adds a Course to the document
     * @param Course to be added
     */
    
    synchronized public void addCourse(Course course) {
        if(!courses.contains(course)){
            courses.add(course);   	// Add the resource
            Collections.sort(courses);
            setChanged();			// Mark the document as changed
            notifyObservers();			// Tell everybody
        }
    }
    
    /**
     * Removes a Course from the document
     * @param Course to be removed
     */
    
    synchronized public void removeCourse(Course course) {
        courses.remove(course);
        setChanged();
        notifyObservers();
    }
    
    synchronized public void addBook(Textbook book){
        if(!books.contains(book)){
            books.add(book);
            //Collections.sort(books);
            setChanged();
            notifyObservers();
        }
    }
    synchronized public void removeBook(Textbook book){
        books.remove(book);
        setChanged();
        notifyObservers();
    }
    
    /**
     * Adds a SchedCourse to the document
     * @param SchedCourse to be added
     */
    
    synchronized public void addSchedCourse(SchedCourse schedCourse) {
         if(!schedCourses.contains(schedCourse)){
            schedCourse.getCourse().setScheduled(true);
            schedCourses.add(schedCourse);   	// Add the resource
            Collections.sort(schedCourses);
            setChanged();			// Mark the document as changed
            notifyObservers();			// Tell everybody
         }
    }
    
    /**
     * Removes a SchedCourse from the document
     * @param SchedCourse to be removed
     */
    
    synchronized public void removeSchedCourse(SchedCourse schedCourse) {
        schedCourse.getCourse().setScheduled(false);
        schedCourses.remove(schedCourse);
        setChanged();
        notifyObservers();
    }
    
    synchronized public void sort(int key) {
        //CPAVectorSort.sortVector(getSchedCourses(), key);
        setChanged();
        notifyObservers();
    }
    
    synchronized public void changed() {
        setChanged();
        notifyObservers();
    }
    
    /**
     * update the conflicts Vector
     */
    
    synchronized public void findConflicts() {
        // start fresh and find all conflicts
        conflicts.removeAll(conflicts);  // this seems strange to me.
        
        // remove conflict data from the SchedCourse objects
        Iterator it = schedCourses.iterator();
        while (it.hasNext())
            ((SchedCourse) it.next()).setConflict(null);
        
        for (int i = 0; i < schedCourses.size(); i++) {
            for (int j = i + 1; j < schedCourses.size(); j++) {
                SchedCourse sched1 = (SchedCourse) schedCourses.get(i);
                SchedCourse sched2 = (SchedCourse) schedCourses.get(j);
                
                if (sched1.overlap(sched2)) {
                    // same time same prof
                    Professor i1 = sched1.getProfessor();
                    Professor i2 = sched2.getProfessor();
                    if (i1.getLastName().equals(i2.getLastName())) {
                        if (!i1.getLastName().trim().equals("STAFF")) {
                            Conflict c = new Conflict((SchedCourse) schedCourses.get(i),
                            (SchedCourse) schedCourses.get(j),
                            PROF_DOUBLE_BOOKED);
                            conflicts.add(c);
                            sched1.setConflict(c);
                            sched2.setConflict(c);
                            continue; // stop looking when one is found  this
                            //  Favors professor conflicts
                            //System.out.println("Found a conflict \n" + c);
                        }
                    }
                    
                    // same time, same room
                    Classroom cr1 = ((SchedCourse) schedCourses.get(i)).getClassroom();
                    Classroom cr2 = ((SchedCourse) schedCourses.get(j)).getClassroom();
                    
                    if (cr1.getRoomName().equals(cr2.getRoomName()) &&
                    cr1.getBuilding().equals(cr2.getBuilding())) {
                        
                        if (!cr1.getRoomName().trim().equals("TBA")) {
                            Conflict c = new Conflict((SchedCourse) schedCourses.get(i),
                                (SchedCourse) schedCourses.get(j),
                            ROOM_DOUBLE_BOOKED);
                            conflicts.add(c);
                            sched1.setConflict(c);
                            sched2.setConflict(c);
                            
                            continue; // stop looking when one is found
                            //System.out.println("Found a conflict \n" + c);
                        }
                    }
                    
                }
            }    // for j
        }       // for i
    }        
    
    /**
     * Put the schedule in a file that can be read by Excel
     */
    
    synchronized public void exportCsv(File outCsv) throws IOException {
        
        PrintWriter pw = new PrintWriter(
                            new BufferedWriter(
                                new FileWriter(outCsv)));
        
        // put the headers on the cols
        pw.println("CRN,Name,Course Title, Course #, Section #,Credit Hrs,"
        +  "Bldg,Room,Time,MTWRFSN,Cap,Note/Conflict");
        
        Iterator it = schedCourses.iterator();
        
        while (it.hasNext()) {
            SchedCourse item = (SchedCourse) it.next();
            pw.print(item.getProfessor().getLastName().toUpperCase() + ","
                + item.getCrn() + ","
                + item.getCourse().getCourseName() + ","
                + item.getCourse().getField()
                + item.getCourse().getCourseNumber() + ","
                + item.getCourse().getSectionPrefix()
                + item.getCourse().getSectionNumber() + ","                
                + "," // credits
           
                + item.getClassroom().getBuilding() + ","
                + item.getClassroom().getRoomName() + ","
                + item.getTimeSlot().toTimeString() + ","
                + item.getTimeSlot().toDaysString()+ ","
                + item.getClassroom().getCapacity() + ","
                + item.getNote()
                + "  "); 
            
            Conflict conflict = item.getConflict();
            
            if (conflict != null) {
                String strConflict = "Conflict: ";
                SchedCourse inCon = null;
                // show which one is in conflict
                if (conflict.getSched1().equals(item))
                    inCon = conflict.getSched2();
                else
                    inCon = conflict.getSched1();
                
                strConflict += inCon.getCourse().getField()
                + inCon.getCourse().getCourseNumber() + " "
                + inCon.getCourse().getSectionPrefix()
                + inCon.getCourse().getSectionNumber();
                pw.print(strConflict);
            }
            pw.println();
        }
        pw.close();
    }
    
    
    /**
     * Print some text schedcourses on each page
     */
    
    synchronized public int print(Graphics g, PageFormat pageFormat, int pageIndex) {
        if (pageIndex > getNumberOfPages()) {
            return NO_SUCH_PAGE;    // done printing
        }
        
        draw(g, pageFormat, pageIndex,
        new Rectangle((int)(pageFormat.getPaper().getImageableX()),
        (int)(pageFormat.getPaper().getImageableY()),
        (int)(pageFormat.getPaper().getImageableWidth()),
        (int)(pageFormat.getPaper().getImageableHeight())));
        
        return PAGE_EXISTS;
    }
    
    /**
     * Used as a helper for printing.  Changes the margins of the paper to get
     * more stuff on the page.
     *
     */
    
    
    /**
     * draw the schedule for a page for printing
     *
     */
    
    synchronized private void draw(Graphics g, PageFormat pageFormat, int pageIndex,
    Rectangle r) {
        
        Graphics g2d = g;
        ((Graphics2D)g).setStroke( new BasicStroke(0.5F)); // make thin lines
        
        Font	    titleFont = new Font("Monospaced", Font.BOLD, 10);
        Font	    regFont = new Font("Monospaced", Font.PLAIN, 8);
        Font	    regFontBold = new Font("Monospaced", Font.BOLD, 8);
        FontMetrics fm = g2d.getFontMetrics(titleFont);     // Get font metrics
        String      header = "Class Schedule:  " + scheduleName + updated();
        String      footer = "Total of: " + schedule.getSchedCourses().size()
        + " classes.";
        int	    titleWidth = fm.stringWidth(header);    // Get title width
        int	    titleHeight = fm.getHeight();	    // Get title height
        int	    lineHeight = fm.getHeight();	    // Get line height
        int	    linesPerPage = 40;
        
        //		System.out.println("Lines per page = " + linesPerPage);
        //		System.out.println("There are: " + schedule.getSchedCourses().size()
        //			+ " records\n"
        //			+ "Calc 1.0 * schedule.getSchedCourses().size() / linesPerPage:"
        //			+ 1.0 * schedule.getSchedCourses().size()
        //			/ linesPerPage + "\npageIndex " + pageIndex);
        
        
        g.setFont(titleFont);
        g.drawString(header,
          (int) pageFormat.getImageableX(),
          ((int) pageFormat.getImageableY() + titleHeight));
        
        String pageNumberString = (pageIndex + 1) + "/" + getNumberOfPages();
        //        System.out.println("Page#String:" + pageNumberString + ":to here");
        //        System.out.println("is this wide: " + fm.stringWidth(pageNumberString));
        //        System.out.println("r is: " + r);
        g.drawString(pageNumberString,
        (int) (r.width - fm.stringWidth(pageNumberString) +
        pageFormat.getImageableX()),
        (int) pageFormat.getImageableY() + titleHeight);
        
        fm = g.getFontMetrics(regFont);    // Get font metrics
        
        int ypos = (int) pageFormat.getImageableY() + titleHeight * 2;    // ypos for first line
        
        g.setFont(regFont);
        
        String prev = "";
        int currentRecord = pageIndex * linesPerPage;
        int currentLine = 0;
        
        //		System.out.println("Current record \n" + currentRecord);
        
        g.drawLine(r.x, ypos + 5 - lineHeight,
        r.x + r.width, ypos + 5 - lineHeight);
        
        for (int i = currentRecord; i < currentRecord + linesPerPage; i++) {
            SchedCourse sc = (SchedCourse)(schedule.getSchedCourses().get(i));
            
            if(sc.getConflict() != null)
                g.setFont(regFontBold);
            else
                g.setFont(regFont);            
            g.drawString(sc.toString(), (int) pageFormat.getImageableX(), ypos);
            if ((i+1) % 4 == 0)
                g.drawLine(r.x, ypos + 5, r.x + r.width, ypos + 5);
            
            if(!(sc.getNote().equals("")) && sc.getNote() != null){ // Print the note
                ypos += lineHeight;
                g.drawString("   " + sc.getNote(),(int) pageFormat.getImageableX(), ypos);
                linesPerPage--;  // this is ugly
            }            
            ypos += lineHeight;
            
            // just printed the last record
            if (i == schedule.getSchedCourses().size() - 1) {
                
                ypos += lineHeight;		    // space
                
                g.setFont(titleFont);
                g.drawString(footer, (int) pageFormat.getImageableX(), ypos);
                //				System.out.println("Printing footer at: " + ypos);
                //				System.out.println("Printing last record: i = " + i);
                //				System.out.println(" at location x = "
                //					+ (int) pageFormat.getImageableX()
                //					+ " \n ypos is " + ypos);
                
                break;
            }
        }
    }
    
    // Always 40 courses per page
    synchronized public int getNumberOfPages() {
        return schedCourses.size() / 41 + 1; // for 40 records this is 1.
    }
    
    // Return the Printable object that will render the page
    synchronized public Printable getPrintable(int pageIndex) {
        return this;
    }
    
    // required to implement Pageable  sets the paper to half inch margins and
    //  and imageable area of 7.5 x 10 for max paper useage
    
    synchronized public PageFormat getPageFormat(int pageIndex) {
        // Get the default page format and its Paper object.
        PageFormat pageFormat = PrinterJob.getPrinterJob().defaultPage();
        Paper paper = pageFormat.getPaper();
        pageFormat.setPaper(paper);   // Restore the paper
        return pageFormat;            // Return the page format
    }
}
