/* $Id: MsqlPooledConnection.java,v 2.1 1999/01/19 06:36:55 borg Exp $ */
/* Copyright  1999 George Reese, All Rights Reserved */
package com.imaginary.sql.msql;

import java.sql.CallableStatement;
import java.sql.Connection;
import java.sql.DatabaseMetaData;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.SQLWarning;
import java.sql.Statement;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.Map;
import javax.sql.ConnectionEvent;
import javax.sql.ConnectionEventListener;
import javax.sql.PooledConnection;

/**
 * Represents a logical database connection using pooled connection
 * resources.  The JDBC 2.0 Standard Extension API is very liberal about
 * the underlying mechanisms it allows for the support of connection
 * pooling.  This interpretation of that API is based on an established
 * data source stored in a JNDI directory under the
 * <CODE>MsqlPooledDataSource</CODE> class name.
 * <BR>
 * Last modified $Date: 1999/01/19 06:36:55 $
 * @version $Revision: 2.1 $
 * @author George Reese
 */
public class MsqlPooledConnection implements Connection, PooledConnection {
    private Connection           connection = null;
    private MsqlPooledDataSource dataSource = null;
    private ArrayList            listeners  = new ArrayList();
    private boolean              open       = true;

    MsqlPooledConnection(MsqlPooledDataSource ds) {
	super();
	dataSource = ds;
    }

    public void addConnectionEventListener(ConnectionEventListener cel) {
	synchronized( listeners ) {
	    listeners.add(cel);
	}
    }
    
    public void clearWarnings() throws SQLException {
	try {
	    getConnection().clearWarnings();
	}
	catch( SQLException e ) {
	    notifyListeners(new ConnectionEvent(this, e));
	    throw e;
	}
    }
    
    public void close() throws SQLException {
	if( isClosed() ) {
	    throw new MsqlException("Connection is already closed.");
	}
	open = false;
	connection = null;
	notifyListeners(new ConnectionEvent(this));
    }

    public void commit() throws SQLException {
	try {
	    getConnection().commit();
	}
	catch( SQLException e ) {
	    notifyListeners(new ConnectionEvent(this, e));
	    throw e;
	}
    }

    public Statement createStatement() throws SQLException {
	try {
	    return getConnection().createStatement();
	}
	catch( SQLException e ) {
	    notifyListeners(new ConnectionEvent(this, e));
	    throw e;
	}
    }

    public Statement createStatement(int rstype, int rsconcur)
    throws SQLException {
	try {
	    return getConnection().createStatement(rstype, rsconcur);
	}
	catch( SQLException e ) {
	    notifyListeners(new ConnectionEvent(this, e));
	    throw e;
	}
    }

    public boolean getAutoCommit() throws SQLException {
	try {
	    return getConnection().getAutoCommit();
	}
	catch( SQLException e ) {
	    notifyListeners(new ConnectionEvent(this, e));
	    throw e;
	}
    }

    public String getCatalog() throws SQLException {
	try {
	    return getConnection().getCatalog();
	}
	catch( SQLException e ) {
	    notifyListeners(new ConnectionEvent(this, e));
	    throw e;
	}
    }
    
    public Connection getConnection() throws SQLException {
	if( isClosed() ) {
	    throw new MsqlException("Illegal attempt to reuse a closed, " +
				    "pooled connection.");
	}
	if( connection != null ) {
	    return connection;
	}
	connection = dataSource.popConnection();
	return connection;
    }

    public DatabaseMetaData getMetaData() throws SQLException {
	try {
	    return getConnection().getMetaData();
	}
	catch( SQLException e ) {
	    notifyListeners(new ConnectionEvent(this, e));
	    throw e;
	}
    }

    public int getTransactionIsolation() throws SQLException {
	try {
	    return getConnection().getTransactionIsolation();
	}
	catch( SQLException e ) {
	    notifyListeners(new ConnectionEvent(this, e));
	    throw e;
	}
    }

    public Map getTypeMap() throws SQLException {
	try {
	    return getConnection().getTypeMap();
	}
	catch( SQLException e ) {
	    notifyListeners(new ConnectionEvent(this, e));
	    throw e;
	}
    }

    public SQLWarning getWarnings() throws SQLException {
	try {
	    return getConnection().getWarnings();
	}
	catch( SQLException e ) {
	    notifyListeners(new ConnectionEvent(this, e));
	    throw e;
	}
    }
    
    public boolean isClosed() {
	return !open;
    }

    public boolean isReadOnly() throws SQLException {
	try {
	    return getConnection().isReadOnly();
	}
	catch( SQLException e ) {
	    notifyListeners(new ConnectionEvent(this, e));
	    throw e;
	}
    }

    public String nativeSQL(String sql) throws SQLException {
	try {
	    return getConnection().nativeSQL(sql);
	}
	catch( SQLException e ) {
	    notifyListeners(new ConnectionEvent(this, e));
	    throw e;
	}
    }

    private void notifyListeners(ConnectionEvent event) {
	SQLException exception = event.getSQLException();
	
	synchronized( listeners ) {
	    Iterator it = listeners.iterator();

	    while( it.hasNext() ) {
		ConnectionEventListener cel;

		cel = (ConnectionEventListener)it.next();
		if( exception != null ) {
		    cel.connectionErrorOccurred(event);
		}
		else {
		    cel.connectionClosed(event);
		}
	    }
	}
    }
    
    public CallableStatement prepareCall(String sql) throws SQLException {
	try {
	    return getConnection().prepareCall(sql);
	}
	catch( SQLException e ) {
	    notifyListeners(new ConnectionEvent(this, e));
	    throw e;
	}
    }

    public CallableStatement prepareCall(String sql, int type, int concur)
    throws SQLException {
	try {
	    return getConnection().prepareCall(sql, type, concur);
	}
	catch( SQLException e ) {
	    notifyListeners(new ConnectionEvent(this, e));
	    throw e;
	}
    }

    public PreparedStatement prepareStatement(String sql) throws SQLException{
	try {
	    return getConnection().prepareStatement(sql);
	}
	catch( SQLException e ) {
	    notifyListeners(new ConnectionEvent(this, e));
	    throw e;
	}
    }
    
    public PreparedStatement prepareStatement(String sql, int type, int conc)
    throws SQLException{
	try {
	    return getConnection().prepareStatement(sql, type, conc);
	}
	catch( SQLException e ) {
	    notifyListeners(new ConnectionEvent(this, e));
	    throw e;
	}
    }
    
    public void removeConnectionEventListener(ConnectionEventListener cel) {
	synchronized( listeners ) {
	    listeners.remove(cel);
	}
    }
    
    public void rollback() throws SQLException {
	try {
	    getConnection().rollback();
	}
	catch( SQLException e ) {
	    notifyListeners(new ConnectionEvent(this, e));
	    throw e;
	}
    }

    public void setAutoCommit(boolean ac) throws SQLException {
	try {
	    getConnection().setAutoCommit(ac);
	}
	catch( SQLException e ) {
	    notifyListeners(new ConnectionEvent(this, e));
	    throw e;
	}
    }

    public void setCatalog(String cat) throws SQLException {
	try {
	    getConnection().setCatalog(cat);
	}
	catch( SQLException e ) {
	    notifyListeners(new ConnectionEvent(this, e));
	    throw e;
	}
    }

    public void setReadOnly(boolean ro) throws SQLException {
	try {
	    getConnection().setReadOnly(ro);
	}
	catch( SQLException e ) {
	    notifyListeners(new ConnectionEvent(this, e));
	    throw e;
	}
    }

    public void setTransactionIsolation(int level) throws SQLException {
	try {
	    getConnection().setTransactionIsolation(level);
	}
	catch( SQLException e ) {
	    notifyListeners(new ConnectionEvent(this, e));
	    throw e;
	}
    }

    public void setTypeMap(Map map) throws SQLException {
	try {
	    getConnection().setTypeMap(map);
	}
	catch( SQLException e ) {
	    notifyListeners(new ConnectionEvent(this, e));
	    throw e;
	}
    }
}
