/* $Id: MsqlPooledDataSource.java,v 2.2 1999/01/21 01:48:16 borg Exp $ */
/* Copyright  1999 George Reese, All Rights Reserved */
package com.imaginary.sql.msql;

import java.sql.Connection;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Properties;
import java.util.Stack;
import javax.naming.NamingException;
import javax.naming.Reference;
import javax.naming.Referenceable;
import javax.naming.StringRefAddr;
import javax.sql.ConnectionEvent;
import javax.sql.ConnectionEventListener;
import javax.sql.ConnectionPoolDataSource;
import javax.sql.PooledConnection;

/**
 * The mSQL-JDBC implementation of the JDBC 2.0 standard extension
 * connection pooling API. Connection pooling is actually something
 * transparent to application developers.  It is configured in the
 * JNDI directory storing this class instead of <CODE>MsqlDataSource</CODE>.
 * The result is a data source with a pool of connections up to a specified
 * <CODE>maxPoolSize</CODE> value.  If a connection is requested after the
 * pool has reached its max size, the <CODE>MsqlPooledDataSource</CODE>
 * implementation will wait until one of the in-use connections is
 * released.  In addition to <CODE>maxPoolSize</CODE>, you should also
 * configure the directory entry with the attributes of a
 * <CODE>MsqlDataSource</CODE>.
 * <BR>
 * Last modified $Date: 1999/01/21 01:48:16 $
 * @version $Revision: 2.2 $
 * @author George Reese (borg@imaginary.com)
 */
public final class MsqlPooledDataSource extends MsqlDataSource
implements ConnectionPoolDataSource, ConnectionEventListener {
    // the pool of available connections
    private transient Stack       available            = new Stack();
    // the connection properties
    private transient Properties  connectionProperties = null; 
    // the pool of connections currently in use
    private transient ArrayList   inUse                = new ArrayList();
    // the maximum size of the pool
    private           int         maxPoolSize          = 1;

    /**
     * Constructs a new pooled data source.
     */
    public MsqlPooledDataSource() {
	super();
    }

    /**
     * This method is called whenever a pooled connection that is in
     * used gets closed.
     * @param event the closing event
     */
    public synchronized void connectionClosed(ConnectionEvent event) {
	MsqlPooledConnection con;
	Connection actual;
	
	con = (MsqlPooledConnection)event.getSource();
	try {
	    actual = con.getConnection();
	}
	catch( SQLException e ) {
	    return;
	}
	if( !inUse.contains(actual) ) {
	    return;
	}
	inUse.remove(actual);
	available.push(actual);
	notifyAll();
    }

    /**
     * This method is called whenever errors occur to a pooled
     * connection.
     * @param event the error event
     */
    public synchronized void connectionErrorOccurred(ConnectionEvent event) {
    }

    /**
     * @return a pooled connection using the specified properties
     * @throws java.sql.SQLException a connection error occurred
     */
    protected Connection getConnection(Properties p)
    throws SQLException {
	connectionProperties = p;
	return new MsqlPooledConnection(this);
    }

    /**
     * @return the maximum size of the connection pool
     */
    public int getMaxPoolSize() {
	return maxPoolSize;
    }

    /**
     * Constructs a pooled connection using the default user/password.
     * @return a database connection
     * @throws java.sql.SQLException a connection error occurred
     */
    public PooledConnection getPooledConnection() throws SQLException {
	return (MsqlPooledConnection)getConnection();
    }

    /**
     * @param uid the user ID for the connection
     * @param pw the password for the connection
     * @return a database connection
     * @throws java.sql.SQLException a connection error occurred
     */
    public PooledConnection getPooledConnection(String uid, String pw)
    throws SQLException {
	return (MsqlPooledConnection)getConnection(uid, pw);
    }

    /**
     * Pulls a pooled connection off the stack of available connections.
     * If no connection is available, it will create a new one unless
     * the max pool size limit has been hit.  If the limit has been hit,
     * processing will halt until a connection becomes available.
     * @return a database connection
     * @throws java.sql.SQLException a database error occurred
     */
    synchronized Connection popConnection() throws SQLException {
	Connection con;
	
	if( !available.empty() ) {
	    con = (Connection)available.pop();
	    inUse.add(con);
	    return con;
	}
	if( inUse.size() < getMaxPoolSize() ) {
	    con = super.getConnection(connectionProperties);
	    inUse.add(con);
	    return con;
	}
	while( available.empty() ) {
	    try { wait(); }
	    catch( InterruptedException e ) { }
	}
	con = (Connection)available.pop();
	inUse.add(con);
	return con;
    }

    /**
     * Creates a reference to this object.
     * @return the reference
     * @throws javax.naming.NamingException a naming error occurred
     */
    public Reference getReference() throws NamingException {
	Reference ref = super.getReference();

	ref.add(new StringRefAddr("maxPoolSize", "" + getMaxPoolSize()));
	return ref;
    }

    /**
     * Sets the maximum size for the connection pool.
     * @param sz the max size
     */
    public void setMaxPoolSize(int sz) {
	maxPoolSize = sz;
    }
}
