/* $Id: MsqlResultSet.java,v 2.6 1999/03/03 03:27:12 borg Exp $ */
/* Copyright (c) 1997-1999 George Reese */
package com.imaginary.sql.msql;

import com.imaginary.util.Encoder;
import com.imaginary.util.NoSuchEncoderException;
import java.io.ByteArrayOutputStream;
import java.io.CharArrayReader;
import java.io.InputStream;
import java.io.IOException;
import java.io.ObjectOutputStream;
import java.io.Reader;
import java.io.Serializable;
import java.io.UnsupportedEncodingException;
import java.math.BigDecimal;
import java.math.BigInteger;
import java.sql.Array;
import java.sql.Blob;
import java.sql.Clob;
import java.sql.DatabaseMetaData;
import java.sql.Date;
import java.sql.Ref;
import java.sql.SQLData;
import java.sql.SQLException;
import java.sql.SQLInput;
import java.sql.SQLWarning;
import java.sql.Statement;
import java.sql.ResultSet;
import java.sql.ResultSetMetaData;
import java.sql.Time;
import java.sql.Timestamp;
import java.sql.Types;
import java.text.SimpleDateFormat;
import java.text.ParseException;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Locale;
import java.util.Map;

/**
 * As of mSQL-JDBC 2.0, MsqlResultSet is an abstract base class for all
 * of the different kinds of result sets supported by mSQL-JDBC.
 * <BR>
 * Last modified $Date: 1999/03/03 03:27:12 $
 * @version $Revision: 2.6 $
 * @author George Reese (borg@imaginary.com)
 */
public abstract class MsqlResultSet implements ResultSet {
    // Column types for updateable result sets
    private   HashMap       columnTypes      = new HashMap();
    // The JDBC 2.0 result set concurrency 
    private   int           concurrency      = ResultSet.CONCUR_READ_ONLY;
    // The direction in which fetches occur
    private   int           fetchDirection   = ResultSet.FETCH_FORWARD;
    // The logging object for this result set
    protected MsqlLog       log              = null;
    // Primary Key info for updateable result sets
    private   HashMap       pkInfo           = null;
    // The Statement that owns this ResultSet
    private   MsqlStatement statement        = null;
    // Table info for updateable result sets
    private   HashMap       tableInfo        = null;
    // The result set type
    private   int           type             = ResultSet.TYPE_FORWARD_ONLY;

    MsqlResultSet(MsqlStatement stmt, int ll) throws SQLException {
	super();
	statement = stmt;
	log = new MsqlLog(ll, this);
	if( stmt != null ) {
	    fetchDirection = stmt.getFetchDirection();
	    concurrency = stmt.getResultSetConcurrency();
	    type = stmt.getResultSetType();
	}
    }
	
    /**
     * Moves the result set to after the last row.
     * @exception java.sql.SQLException a database error occurred or the
     * result set is TYPE_FORWARD_ONLY
     */
    public void afterLast() throws SQLException {
	absolute(-1);
	next();
    }

    /**
     * Moves the result set to before the first row.
     * @exception java.sql.SQLException a database error occurred or the
     * result set is TYPE_FORWARD_ONLY
     */
    public void beforeFirst() throws SQLException {
	absolute(1);
	previous();
    }
    
    /**
     * This feature is not currently supported.
     * @exception java.sql.SQLException this is never thrown
     */
    public void cancelRowUpdates() throws SQLException {
	log.log("cancelRowUpdates()", MsqlLog.JDBC,
		"Row updates cancelled.");
	if( getConcurrency() != ResultSet.CONCUR_UPDATABLE ) {
	    log.log("cancelRowUpdates()", MsqlLog.ERROR,
			   "Result set is not updatable.");
	    throw new MsqlException("Result set is not updatable.");
	}
    }
    
    /**
     * mSQL does not generate warnings, so this method is a NO-OP
     * @exception java.sql.SQLException this is never thrown
     */
    public void clearWarnings() throws SQLException {
	log.log("clearWarnings()", MsqlLog.JDBC, "Clear warnings.");
    }

    /**
     * A NO-OP for this base class.
     * @exception java.sql.SQLException this is never thrown
     */
    public void close() throws SQLException {
	log.log("close()", MsqlLog.JDBC, "Closing result set.");
	log.close();
    }

    /**
     * This method is called to mark a result set as complete.
     */
    public void complete() {
	log.log("complete()", MsqlLog.DRIVER, "Result set complete.");
	if( statement != null ) {
	    statement.completeLoad();
	}
    }
    
    /**
     * The default implementation throws an exception.
     * @exception java.sql.SQLException result set is not updatable
     */
    public void deleteRow() throws SQLException {
	log.log("deleteRow()", MsqlLog.JDBC, "Delete current row.");
	if( getConcurrency() != ResultSet.CONCUR_UPDATABLE ) {
	    log.log("deleteRow()", MsqlLog.ERROR,
		    "Result set is not updatable.");
	    throw new MsqlException("Result set is not updatable.");
	}
    }
    
    /**
     * Moves the result set to the first row.
     * @exception java.sql.SQLException a database error occurred or
     * the result set TYPE_FORWARD_ONLY
     */
    public boolean first() throws SQLException {
	return absolute(1);
    }
    
    /**
     * mSQL does not support array columns.
     * @param cname the name of the desired column
     * @return the column as an Array
     * @exception java.sql.SQLException this is always thrown
     */
    public Array getArray(String cname) throws SQLException {
	return getArray(findColumn(cname));
    }
    
    /**
     * mSQL does not support array columns.
     * @param column the number of the desired column
     * @return the column as an Array
     * @exception java.sql.SQLException this is always thrown
     */
    public Array getArray(int column) throws SQLException {
	log.log("getArray()", MsqlLog.ERROR, "Arrays not supported.");
	throw new MsqlException("This operation is not supported.");
    }
    
   /**
     * For performance reasons, you should get values by column
     * number when at all possible.
     * @param cname the name of the desired column
     * @return an ASCII input stream for the column
     * @exception java.sql.SQLException thrown when the column cannot be read
     */
    public InputStream getAsciiStream(String cname) throws SQLException {
	return getAsciiStream(findColumn(cname));
    }

    /**
     * Avoid getting columns by name whenever possible.
     * @param cname the name of the column being sought
     * @return the specified column as a BigDecimal
     * @exception java.sql.SQLException thrown when the column cannot be read
     */
    public BigDecimal getBigDecimal(String cname) throws SQLException {
	return getBigDecimal(findColumn(cname));
    }
    
    /**
     * @param column the column number of the desired column
     * @return the specified column as a BigDecimal
     * @exception java.sql.SQLException thrown when the column cannot be read
     */
    public BigDecimal getBigDecimal(int column) throws SQLException {
	String tmp = getString(column);

	if( tmp == null ) {
	    return new BigDecimal(new BigInteger("0"));
	}
	else {
	    return new BigDecimal(new BigInteger(tmp));
	}
    }

    /**
     * Avoid getting columns by name whenever possible.
     * @param cname the name of the column being sought
     * @param scale the scale with which the BigDecimal should be constructed
     * @return the specified column as a BigDecimal
     * @exception java.sql.SQLException thrown when the column cannot be read
     * @deprecated
     */
    public BigDecimal getBigDecimal(String cname, int scale)
    throws SQLException {
	return getBigDecimal(findColumn(cname), scale);
    }
    
    /**
     * @param column the column number of the desired column
     * @param scale the scale with which the BigDecimal should be constructed
     * @return the specified column as a BigDecimal
     * @exception java.sql.SQLException thrown when the column cannot be read
     * @deprecated
     */
    public BigDecimal getBigDecimal(int column, int scale)throws SQLException {
	String tmp = getString(column);

	if( tmp == null ) {
	    return new BigDecimal(new BigInteger("0"), scale);
	}
	else {
	    return new BigDecimal(new BigInteger(tmp), scale);
	}
    }

    /**
     * Avoid getting columns by name whenever possible.
     * @param cname the name of the desired column
     * @return the column as an InputStream
     * @exception java.sql.SQLException thrown in the event of an error
     * reading the column
     */
    public InputStream getBinaryStream(String cname) throws SQLException {
	return getBinaryStream(findColumn(cname));
    }
    
    /**
     * mSQL does not support BLOBs.
     * @param cname the name of the desired column
     * @return the column as a Blob
     * @exception java.sql.SQLException this is always thrown
     */
    public Blob getBlob(String cname) throws SQLException {
	return getBlob(findColumn(cname));
    }
    
    /**
     * mSQL does not support BLOBs.
     * @param column the number of the desired column
     * @return the column as a Blob
     * @exception java.sql.SQLException this is always thrown
     */
    public Blob getBlob(int column) throws SQLException {
	log.log("getBlob()", MsqlLog.ERROR, "Blob objects are not supported.");
	throw new MsqlException("This operation is not supported.");
    }
    
    /**
     * Avoid getting columns by name whenever possible.
     * @param cname the name of the desired column
     * @return the column as a boolean
     * @exception java.sql.SQLException thrown in the event of an error
     * reading the column
     */
    public boolean getBoolean(String cname) throws SQLException {
	return getBoolean(findColumn(cname));
    }
    
    /**
     * Avoid getting columns by name whenever possible.
     * @param cname the name of the desired column
     * @return the column as a byte
     * @exception java.sql.SQLException thrown in the event of an error
     * reading the column
     */
    public byte getByte(String cname) throws SQLException {
	return getByte(findColumn(cname));
    }
    
    /**
     * Avoid getting columns by name whenever possible.
     * @param cname the name of the desired column
     * @return the column as a byte array
     * @exception java.sql.SQLException thrown in the event of an error
     * reading the column
     */
    public byte[] getBytes(String cname) throws SQLException {
	return getBytes(findColumn(cname));
    }
    
    /**
     * Avoid getting columns by name whenever possible.
     * @param cname the name of the desired column
     * @return the column as a Reader
     * @exception java.sql.SQLException thrown in the event of an error
     * reading the column
     */
    public Reader getCharacterStream(String cname) throws SQLException {
	return getCharacterStream(findColumn(cname));
    }
    
    /**
     * @param column the number of the desired column
     * @return the column as a Reader
     * @exception java.sql.SQLException thrown in the event of an error
     * reading the column
     */
    public Reader getCharacterStream(int column) throws SQLException {
	return new CharArrayReader(getString(column).toCharArray());
    }
  
    /**
     * mSQL does not support Character large objects.
     * @param cname the name of the desired column
     * @return the column as a clob
     * @exception java.sql.SQLException this is always thrown
     */
    public Clob getClob(String cname) throws SQLException {
	return getClob(findColumn(cname));
    }
    
    /**
     * mSQL does not support Character large objects
     * @param column the number of the desired column
     * @return the column as a Clob
     * @exception java.sql.SQLException this is always thrown
     */
    public Clob getClob(int column) throws SQLException {
	log.log("getClob()", MsqlLog.ERROR, "Clobs not supported.");
	throw new MsqlException("This operation is not supported.");
    }
    
    /**
     * @return the result set concurrency
     * @exception java.sql.SQLException this is never thrown
     */
    public int getConcurrency() throws SQLException {
	return concurrency;
    }
    
    /**
     * @return the cursor name
     * @exception java.sql.SQLException this is always thrown as mSQL does
     * not support named cursors
     */
    public String getCursorName() throws SQLException {
	log.log("getCursorName()", MsqlLog.ERROR, "Cursors not supported.");
	throw new SQLException("mSQL does not support cursors.");
    }

    /**
     * Avoid getting columns by name whenever possible.
     * @param cname the name of the desired column
     * @return the column as a Date
     * @exception java.sql.SQLException thrown in the event of an error
     * reading the column
     */
    public Date getDate(String cname) throws SQLException {
	return getDate(findColumn(cname));
    }

    /**
     * mSQL 1.0 does not support a date type, but 2.0 does.  This method
     * allows any version of mSQL to store dates as a CHAR type.  It also
     * allows mSQL 2.0 to store them as TEXT or DATE types.  If you choose to
     * store your date as a CHAR or TEXT type for any reason, you must store
     * the date as a long converted into a CHAR or in DD-Mon-YYYY.
     * @param column the number of desired column
     * @return a Date representation of the specified column
     * @exception java.sql.SQLException a database error occurred
     */
    public Date getDate(int column) throws SQLException {
	String tmp = getString(column);  

	if( tmp == null ) {
	    return null;
	}
	try {
	    SimpleDateFormat fmt = new SimpleDateFormat("dd-MMM-yyyy",
							Locale.US);
	    java.util.Date date = fmt.parse(tmp);

	    return new Date(date.getTime());
	}
	catch( ParseException e ) {
	    try {
		return new Date(Long.parseLong(tmp));
	    }
	    catch( NumberFormatException e2 ) {
		log.log("getDate()", MsqlLog.ERROR,
			"Invalid date format: " + tmp);
		throw new SQLException("Invalid date format.");
	    }
	}
    }

    /**
     * Avoid getting columns by name whenever possible.
     * @param cname the name of the desired column
     * @param cal the Calendar to use in constructing the Date
     * @return the column as a Date
     * @exception java.sql.SQLException thrown in the event of an error
     * reading the column
     */
    public Date getDate(String cname, Calendar cal) throws SQLException {
	return getDate(findColumn(cname), cal);
    }

    /**
     * mSQL 1.0 does not support a date type, but 2.0 does.  This method
     * allows any version of mSQL to store dates as a CHAR type.  It also
     * allows mSQL 2.0 to store them as TEXT or DATE types.  If you choose to
     * store your date as a CHAR or TEXT type for any reason, you must store
     * the date as a long converted into a CHAR or in DD-Mon-YYYY.
     * @param column the number of desired column
     * @param cal the Calendar to use in constructing the date
     * @return a Date representation of the specified column
     * @exception java.sql.SQLException a database error occurred
     */
    public Date getDate(int column, Calendar cal) throws SQLException {
	String tmp = getString(column);  

	if( tmp == null ) {
	    return null;
	}
	try {
	    SimpleDateFormat fmt = new SimpleDateFormat("dd-MMM-yyyy",
							Locale.US);
	    java.util.Date date;

	    fmt.setCalendar(cal);
	    date = fmt.parse(tmp);
	    return new Date(date.getTime());
	}
	catch( ParseException e ) {
	    try {
		return new Date(Long.parseLong(tmp));
	    }
	    catch( NumberFormatException e2 ) {
		log.log("getDate()", MsqlLog.ERROR,
			"Invalid date format: " + tmp);
		throw new MsqlException("Invalid date format.");
	    }
	}
    }

    /**
     * Avoid getting columns by name whenever possible.
     * @param cname the name of the desired column
     * @return the column as a double
     * @exception java.sql.SQLException thrown in the event of an error
     * reading the column
     */
    public double getDouble(String cname) throws SQLException {
	return getDouble(findColumn(cname));
    }
    
    /**
     * @param column the number of the desired column
     * @return the column as a double
     * @exception java.sql.SQLException thrown in the event of an error
     * reading the column
     */
    public double getDouble(int column) throws SQLException {
	String tmp = getString(column);

	if( tmp == null ) {
	    return 0;
	}
	try {
	    return Double.valueOf(tmp).doubleValue();
	}
	catch( NumberFormatException e ) {
	    log.log("getDouble()", MsqlLog.ERROR,
		    "Invalid number format: " + tmp + ".");
	    throw new MsqlException(e);
	}
    }  

    public String getEncoding() {
	if( statement != null ) {
	    return statement.getEncoding();
	}
	else {
	    return "8859_1";
	}
    }
    
    /**
     * @return the direction in which results might be fetched
     * @exception java.sql.SQLException this is never thrown
     */
    public int getFetchDirection() throws SQLException {
	return fetchDirection;
    }

    /**
     * mSQL-JDBC always fetches all rows.  This is required by the mSQL
     * network protocol and cannot be modified.
     * @return 0
     * @exception java.sql.SQLException this is never thrown
     */
    public int getFetchSize() throws SQLException {
	return 0;
    }
    
    /**
     * Avoid getting columns by name whenever possible.
     * @param cname the name of the desired column
     * @return the column as a float
     * @exception java.sql.SQLException thrown in the event of an error
     * reading the column
     */
    public float getFloat(String cname) throws SQLException {
	return getFloat(findColumn(cname));
    }
    
    /**
     * @param column the number of the desired column
     * @return the column as a float
     * @exception java.sql.SQLException thrown in the event of an error
     * reading the column
     */
    public float getFloat(int column) throws SQLException {
	String tmp = getString(column);

	if( tmp == null ) {
	    return 0;
	}
	try {
	    return Float.valueOf(tmp).floatValue();
	}
	catch( NumberFormatException e ) {
	    log.log("getFloat()", MsqlLog.ERROR,
		    "Invalid number format: " + tmp);
	    throw new MsqlException(e);
	}
    }  

    /**
     * Avoid getting columns by name whenever possible.
     * @param cname the name of the desired column
     * @return the column as an int
     * @exception java.sql.SQLException thrown in the event of an error
     * reading the column
     */
    public int getInt(String cname) throws SQLException {
	return getInt(findColumn(cname));
    }
    
    /**
     * @param column the number of the desired column
     * @return the column as an int
     * @exception java.sql.SQLException thrown in the event of an error
     * reading the column
     */
    public int getInt(int column) throws SQLException {
	String tmp = getString(column);

	if( tmp == null ) {
	    return 0;
	}
	try {
	    return Integer.parseInt(tmp);
	}
	catch( NumberFormatException e ) {
	    log.log("getInt()", MsqlLog.ERROR,
		    "Invalid number format: " + tmp + ".");
	    throw new MsqlException(e);
	}
    }

    /**
     * Avoid getting columns by name whenever possible.
     * @param cname the name of the desired column
     * @return the column as a long
     * @exception java.sql.SQLException thrown in the event of an error
     * reading the column
     */
    public long getLong(String cname) throws SQLException {
	return getLong(findColumn(cname));
    }
    
    /**
     * @param column the number of the desired column
     * @return the column as a long
     * @exception java.sql.SQLException thrown in the event of an error
     * reading the column
     */
    public long getLong(int column) throws SQLException {
	String tmp = getString(column);

	if( tmp == null ) {
	    return 0;
	}
	try {
	    return Long.parseLong(tmp);
	}
	catch( NumberFormatException e ) {
	    log.log("getLong()", MsqlLog.ERROR, 
		    "Invalid numver format: " + tmp + ".");
	    throw new MsqlException(e);
	}
    }  

    /**
     * Avoid getting columns by name whenever possible.
     * @param cname the name of the desired column
     * @return the column as a Java object based on the standard
     * SQL -> Java mapping unless the SQL type is a custom SQL type, in
     * which case the getTypeMap() value in the MsqlConnection will be used
     * @exception java.sql.SQLException thrown in the event of an error
     * reading the column
     * @see java.sql.Connection#getTypeMap
     */
    public Object getObject(String cname) throws SQLException {
	return getObject(findColumn(cname));
    }
  
    /**
     * @param column the number of the desired column
     * @return the column as a Java object based on the standard
     * SQL -> Java mapping unless the SQL type is a custom SQL type, in
     * which case the getTypeMap() value in the MsqlConnection will be used
     * @exception java.sql.SQLException thrown in the event of an error
     * reading the column
     * @see java.sql.Connection#getTypeMap
     */
    public Object getObject(int column) throws SQLException {
	ResultSetMetaData meta = getMetaData();
	int type = meta.getColumnType(column);

	switch(type) {
	case Types.BIT:
	    return new Boolean(getBoolean(column));
	    
	case Types.TINYINT:
	    return new Character((char)getInt(column));
	    
	case Types.SMALLINT:
	    return new Integer(getShort(column));
	    
	case Types.INTEGER:
	    return new Integer(getInt(column));
	    
	case Types.BIGINT:
	    return new Long(getLong(column));
	    
	case Types.FLOAT:
	    return new Float(getFloat(column));
	    
	case Types.REAL:
	    return new Float(getFloat(column));
	    
	case Types.DOUBLE:
	    return new Double(getDouble(column));
	    
	case Types.NUMERIC:
	    return getBigDecimal(column, 0);

	case Types.DECIMAL:
	    return getBigDecimal(column, 0);

	case Types.CHAR:
	    return getString(column);
	    
	case Types.VARCHAR:
	    return getString(column);
	    
	case Types.LONGVARCHAR:
	    return getString(column);
	    
	case Types.DATE:
	    return getDate(column);
	    
	case Types.TIME:
	    return getTime(column);
	    
	case Types.TIMESTAMP:
	    return getTimestamp(column);
	    
	case Types.BINARY:
	    return getBytes(column);
	    
	case Types.VARBINARY:
	    return getBytes(column);
	    
	case Types.LONGVARBINARY:
	    return getBytes(column);
	    
	case Types.OTHER: case Types.JAVA_OBJECT: case Types.DISTINCT:
	case Types.STRUCT: case Types.ARRAY: case Types.BLOB:
	case Types.CLOB: case Types.REF:
	    return getObject(column, statement.getConnection().getTypeMap());
	}
	log.log("getObject()", MsqlLog.ERROR, "Unknown SQL type " + type +".");
	throw new MsqlException("Unknown SQL type: " + type);
    }

    /**
     * This method exists for the sake of compliance with JDBC 2.0 and
     * possible future UDTs in mSQL.  Try to avoid getting columns by name
     * whenever possible.
     * @param cname the name of the desired column
     * @return the column as a Java object based on the standard
     * SQL -> Java mapping unless the SQL type is a custom SQL type, in
     * which case the getTypeMap() value in the MsqlConnection will be used
     * @exception java.sql.SQLException thrown in the event of an error
     * reading the column
     * @see java.sql.Connection#getTypeMap
     */
    public Object getObject(String cname, Map types) throws SQLException {
	return getObject(findColumn(cname), types);
    }
    
    /**
     * This method exists for the sake of compliance with JDBC 2.0 and
     * possible future UDTs in mSQL.
     * @param column the number of the desired column
     * @return the column as a Java object based on the standard
     * SQL -> Java mapping unless the SQL type is a custom SQL type, in
     * which case the getTypeMap() value in the MsqlConnection will be used
     * @exception java.sql.SQLException thrown in the event of an error
     * reading the column
     * @see java.sql.Connection#getTypeMap
     */
    public Object getObject(int column, Map types) throws SQLException {
	ResultSetMetaData meta = getMetaData();
	String type = meta.getColumnTypeName(column);
	SQLInput is = null;
	SQLData ob;
	Class cls;

	if( !types.containsKey(type) ) {
	    switch( meta.getColumnType(column) ) {
	    case Types.OTHER: case Types.JAVA_OBJECT: case Types.DISTINCT:
	    case Types.STRUCT: case Types.ARRAY: case Types.BLOB:
	    case Types.CLOB: case Types.REF:
		{
		    log.log("getObject()", MsqlLog.ERROR,
			    "No type mapping defined for " + type + ".");
		    throw new MsqlException("No type mapping defined for" +
					    type + ".");
		}
	    default:
		{
		    return getObject(column);
		}
	    }
	}
	try {
	    cls = (Class)types.get(type);
	}
	catch( ClassCastException e ) {
	    log.log("getObject()", MsqlLog.ERROR,
		    "Class not found: " + e.getMessage());
	    throw new MsqlException(e);
	}
	try {
	    ob = (SQLData)cls.newInstance();
	}
	catch( Exception e ) {
	    log.log("getObject()", MsqlLog.ERROR,
		    "Could not create object: " + e.getMessage());
	    throw new MsqlException(e);
	}
	// this is a placeholder in case mSQL ever supports UDTs
	ob.readSQL(is, type);
	return ob;
    }

    /**
     * mSQL-JDBC does not currently support references
     * @param cname the name of the desired column
     * @return a ref for the specified column
     * @exception java.sql.SQLException this is always thrown
     */
    public Ref getRef(String cname) throws SQLException {
	return getRef(findColumn(cname));
    }
    
    /**
     * mSQL-JDBC does not currently support references
     * @param column the desired column number
     * @return a ref for the specified column
     * @exception java.sql.SQLException this is always thrown
     */
    public Ref getRef(int column) throws SQLException {
	log.log("getRef()", MsqlLog.ERROR, "Ref objects not supported.");
	throw new MsqlException("This operation is not supported.");
    }
    
    /**
     * Avoid getting columns by name whenever possible.
     * @param cname the name of the desired column
     * @return the column as a short
     * @exception java.sql.SQLException thrown in the event of an error
     * reading the column
     */
    public short getShort(String cname) throws SQLException {
	return getShort(findColumn(cname));
    }
    
    /**
     * @param column the number of the desired column
     * @return the column as a short
     * @exception java.sql.SQLException thrown in the event of an error
     * reading the column
     */
    public short getShort(int column) throws SQLException {
	String tmp = getString(column);

	if( tmp == null ) {
	    return 0;
	}
	try {
	    return (short)Short.parseShort(tmp);
	}
	catch( NumberFormatException e ) {
	    log.log("getShort()", MsqlLog.ERROR,
		    "Invalid number format: " + tmp + ".");
	    throw new MsqlException(e);
	}
    }

    /**
     * @return the Statement that generated this result set
     * @exception java.sql.SQLException this is never thrown
     */
    public Statement getStatement() throws SQLException {
	return statement;
    }
    
    /**
     * Avoid getting columns by name whenever possible.
     * @param cname the name of the desired column
     * @return the column as a Java String
     * @exception java.sql.SQLException thrown in the event of an error
     * reading the column
     */
    public String getString(String cname) throws SQLException {
	return getString(findColumn(cname));
    }
    
    /**
     * Avoid getting columns by name whenever possible.
     * @param cname the name of the desired column
     * @return the column as a Time object
     * @exception java.sql.SQLException thrown in the event of an error
     * reading the column
     */
    public Time getTime(String cname) throws SQLException {
	return getTime(findColumn(cname));
    }
    
    /**
     * @param column the number of the desired column
     * @return the column as a Java String
     * @exception java.sql.SQLException thrown in the event of an error
     * reading the column
     */
    public Time getTime(int column) throws SQLException {
	String tmp = getString(column);  

	if( tmp == null ) {
	    return null;
	}
	try {
	    SimpleDateFormat fmt = new SimpleDateFormat("HH:mm:ss",
							Locale.US);
	    java.util.Date time = fmt.parse(tmp);

	    return new Time(time.getTime());
	}
	catch( ParseException e ) {
	    try {
		return new Time(Long.parseLong(tmp));
	    }
	    catch( NumberFormatException e2 ) {
		log.log("getTime()", MsqlLog.ERROR,
			"Invalid time format: " + tmp + ".");
		throw new SQLException("Invalid time format.");
	    }
	}
    }

    /**
     * Avoid getting columns by name whenever possible.
     * @param cname the name of the desired column
     * @param cal the Calendar to use in creating the Time object
     * @return the column as a Time object
     * @exception java.sql.SQLException thrown in the event of an error
     * reading the column
     */
    public Time getTime(String cname, Calendar cal) throws SQLException {
	return getTime(findColumn(cname), cal);
    }
    
    /**
     * @param column the number of the desired column
     * @param cal the Calendar to use in creating the Time object
     * @return the column as a Java String
     * @exception java.sql.SQLException thrown in the event of an error
     * reading the column
     */
    public Time getTime(int column, Calendar cal) throws SQLException {
	String tmp = getString(column);  

	if( tmp == null ) {
	    return null;
	}
	try {
	    SimpleDateFormat fmt = new SimpleDateFormat("HH:mm:ss",
							Locale.US);
	    java.util.Date time;

	    fmt.setCalendar(cal);
	    time = fmt.parse(tmp);
	    return new Time(time.getTime());
	}
	catch( ParseException e ) {
	    try {
		return new Time(Long.parseLong(tmp));
	    }
	    catch( NumberFormatException e2 ) {
		log.log("getTime()", MsqlLog.ERROR,
			"Invalid time format: " + tmp + ".");
		throw new SQLException("Invalid time format.");
	    }
	}
    }

    /**
     * <P>
     * Timestamps are not really supported in mSQL.  To make use of
     * them, store timestamps in char fields.
     * </P>
     * <P>
     * Avoid getting columns by name whenever possible.
     * </P>
     * @param cname the name of the desired column
     * @return the column as a Timestamp
     * @exception java.sql.SQLException thrown in the event of an error
     * reading the column
     */
    public Timestamp getTimestamp(String cname) throws SQLException {
	return getTimestamp(findColumn(cname));
    }

    /**
     * <P>
     * Timestamps are not really supported in mSQL.  To make use of
     * them, store timestamps in char fields.
     * </P>
     * @param column the number of the desired column
     * @return the column as a Timestamp
     * @exception java.sql.SQLException thrown in the event of an error
     * reading the column
     */
    public Timestamp getTimestamp(int column) throws SQLException {
	String tmp = getString(column);

	if( tmp == null ) {
	    return null;
	}
	else {
	    return new Timestamp(Long.parseLong(tmp));
	}
    }

    /**
     * <P>
     * Timestamps are not really supported in mSQL.  To make use of
     * them, store timestamps in char fields.
     * </P>
     * <P>
     * Avoid getting columns by name whenever possible.
     * </P>
     * @param cname the name of the desired column
     * @param cal this is currently ignored
     * @return the column as a Timestamp
     * @exception java.sql.SQLException thrown in the event of an error
     * reading the column
     */
    public Timestamp getTimestamp(String cname, Calendar cal)
    throws SQLException {
	return getTimestamp(findColumn(cname), cal);
    }

    /**
     * <P>
     * Timestamps are not really supported in mSQL.  To make use of
     * them, store timestamps in char fields.
     * </P>
     * @param column the number of the desired column
     * @param c this is currently ignored
     * @return the column as a Timestamp
     * @exception java.sql.SQLException thrown in the event of an error
     * reading the column
     */
    public Timestamp getTimestamp(int column, Calendar c) throws SQLException {
	String tmp = getString(column);

	if( tmp == null ) {
	    return null;
	}
	else {
	    return new Timestamp(Long.parseLong(tmp));
	}
    }

    /**
     * @return the result set type
     * @exception java.sql.SQLException this is never thrown
     */
    public int getType() throws SQLException {
	return type;
    }
    
    /**
     * Avoid getting columns by name whenever possible.
     * @param cname the name of the desired column
     * @return the column as an InputStream
     * @exception java.sql.SQLException thrown in the event of an error
     * reading the column
     * @deprecated use the getCharacterStream()
     */
    public InputStream getUnicodeStream(String cname) throws SQLException {
	return getUnicodeStream(findColumn(cname));
    }

    /**
     * This method does nothing. Subclasses should extend this method.
     * @param col the desired column
     * @return the value (unescaped for strings) to be updated
     * @throws java.sql.SQLException a database error occurred
     */
    protected String getUpdate(int col) throws SQLException {
	log.log("getUpdate()", MsqlLog.DRIVER,
		"Getting update value for column " + col + ".");
	return null;
    }
    
    /**
     * mSQL does not generate warnings, so this method always returns null.
     * @return null
     * @exception java.sql.SQLException this is never thrown
     */
    public SQLWarning getWarnings() throws SQLException {
	return null;
    }

    /**
     * This is a NO-OP. Implementing classes should extend this method.
     * @exception java.sql.SQLException this is never thrown
     */
    public void insertRow() throws SQLException {
	log.log("insertRow()", MsqlLog.JDBC, "Inserting row.");
	if( getConcurrency() != ResultSet.CONCUR_UPDATABLE ) {
	    log.log("insertRow()", MsqlLog.ERROR,
		    "Result set not updatable.");
	    throw new MsqlException("Result set not updatable.");
	}
    }
    
    /**
     * Determines if the driver is positioned after the last row.
     * @return true if the result set is positioned after the last row
     * @exception java.sql.SQLException this is never thrown
     */
    public boolean isAfterLast() throws SQLException {
	return (getRow() == -2);
    }
    
    /**
     * Because of the way the driver works, isAfterLast() == isBeforeFirst().
     * @return true if the result set is positioned before the first row
     * @exception java.sql.SQLException this is never thrown
     */
    public boolean isBeforeFirst() throws SQLException {
	return (getRow() == 0);
    }
    
    /**
     * @return true if the current row is the first row
     * @exception java.sql.SQLException this is never thrown
     */
    public boolean isFirst() throws SQLException {
	return (getRow() == 1);
    }
    
    /**
     * Moves the result set to the last row.
     * @exception java.sql.SQLException a database error occurred or
     * the result set TYPE_FORWARD_ONLY
     */
    public boolean last() throws SQLException {
	return absolute(-1);
    }
    
    /**
     * Moves the cursor to the current row.  This is a NO-OP since result set
     * inserts are not yet supported.
     * @exception java.sql.SQLException this is never thrown
     */
    public void moveToCurrentRow() throws SQLException {
	log.log("moveToCurrentRow()", MsqlLog.JDBC,
		"Moving to the current row.");
    }
    
    /**
     * This is a NO-OP. Implementing classes should extend this.
     * @exception java.sql.SQLException this is never thrown
     */
    public void moveToInsertRow() throws SQLException {
	log.log("moveToInsertRow()", MsqlLog.JDBC,
		"Moving to the insert row.");
	if( getConcurrency() != ResultSet.CONCUR_UPDATABLE ) {
	    log.log("moveToInsertRow()", MsqlLog.ERROR,
		    "Result set not updatable.");
	    throw new MsqlException("Result set not updatable.");
	}
    }

    private synchronized void prepareModification() throws SQLException {
	ResultSetMetaData rsmd;
	DatabaseMetaData dmd;
	Iterator tables;
	int cols;

	log.log("prepareModification()", MsqlLog.DRIVER,
		"Preparing result set modifications.");
	if( pkInfo != null ) {
	    return;
	}
	dmd = statement.getConnection().getMetaData();
	rsmd = getMetaData();
	tableInfo = new HashMap();
	cols = rsmd.getColumnCount();
	for(int i=1; i<(cols+1); i++) {
	    String col = rsmd.getColumnName(i);
	    String table = rsmd.getTableName(i);
	    ArrayList columns;

	    if( tableInfo.containsKey(table) ) {
		columns = (ArrayList)tableInfo.get(table);
	    }
	    else {
		columns = new ArrayList();
	    }
	    columns.add(col);
	    tableInfo.put(table, columns);
	}
	// Walk over the list of tables and make sure that all the keys
	// are there...
	tables = tableInfo.keySet().iterator();
	pkInfo = new HashMap();
	while( tables.hasNext() ) {
	    String table = (String)tables.next();
	    ArrayList columns = (ArrayList)tableInfo.get(table);
	    ArrayList pk = new ArrayList();
	    ResultSet rs;

	    // get the primary keys
	    rs = dmd.getPrimaryKeys(null, null, table);
	    // first row is the index type, ignore it
	    rs.next();
	    while( rs.next() ) {
		String col = rs.getString(1);
		
		if( !columns.contains(col) ) {
		    log.log("prepareModifications()", MsqlLog.ERROR,
			    "JDBC requires keys to be part of an updatable " +
			    "result set.");
		    throw new MsqlException("The keys must be included in " +
					    "any result set you wish to " +
					    "modify in place.");
		}
		pk.add(col);
	    }
	    pkInfo.put(table, pk);
	    rs.close();
	    // Get column type info
	    rs = dmd.getColumns("", "", table, "");
	    while( rs.next() ) {
		String name = rs.getString("NAME");
		String type = rs.getString("TYPE");

		columnTypes.put(table + "." + name, type);
	    }
	}
    }

    /**
     * This method is a NO-OP. Implementing classes should extend this.
     * @throws java.sql.SQLException result set is not updatable
     */
    public void refreshRow() throws SQLException {
	log.log("refreshRow()", MsqlLog.JDBC, "Refreshing row.");
	if( getConcurrency() != ResultSet.CONCUR_UPDATABLE ) {
	    log.log("refreshRow()", MsqlLog.ERROR,
		    "Result set is not updatable.");
	    throw new MsqlException("Result set is not updatable.");
	}
    }
	

    /**
     * The default implementation here returns false.  Subclasses
     * that supported updatable cursors will provide a meaningful answer.
     * @return false
     * @exception java.sql.SQLException this is never thrown
     */
    public boolean rowDeleted() throws SQLException {
	return false;
    }
    
    /**
     * The default implementation here returns false.  Subclasses
     * that supported updatable cursors will provide a meaningful answer.
     * @return false
     * @exception java.sql.SQLException this is never thrown
     */
    public boolean rowInserted() throws SQLException {
	return false;
    }

    /**
     * The default implementation here returns false.  Subclasses
     * that supported updatable cursors will provide a meaningful answer.
     * @return false
     * @throws java.sql.SQLException not thrown by this class
     */
    public boolean rowUpdated() throws SQLException {
	return false;
    }

    /**
     * This method checks only that the result set is updatable.
     * Implementing classes should extend this to provide behavior.
     * @param col the column number to set
     * @param val the value to set the column to 
     * @throws java.sql.SQLException this result set is not updatable
     */
    protected void setColumn(int col, String val) throws SQLException {
	log.log("setColumn()", MsqlLog.DRIVER,
		"Setting column " + col + " to \"" + val + "\".");
	log.log("setColumn()", MsqlLog.ERROR,
		"Result set not updatable.");
	throw new MsqlException("Result set is not updatable.");
    }
    
    /**
     * Sets the concurrency.
     * @param concur the concurrency level
     */
    void setConcurrency(int concur) {
	log.log("setConcurrency()", MsqlLog.JDBC,
		"Setting concurrency to " + concur + ".");
	concurrency = concur;
    }
    
    /**
     * Provide a hint to mSQL-JDBC as to which direction results should
     * be fetched in.  It is really never used since mSQL cannot
     * be optimized for a fetch direction.
     * @param dir the direction
     * @exception java.sql.SQLException this is never thrown
     */
    public void setFetchDirection(int dir) throws SQLException {
	log.log("setFetchDirection()", MsqlLog.JDBC,
		"Setting fetch direction to " + dir + ".");
	fetchDirection = dir;
    }

    /**
     * This is a NO-OP since no matter what you suggest, mSQL-JDBC
     * is forced by the mSQL network protocol to download all rows.
     * @param unused the suggested fetch size
     * @exception java.sql.SQLException this is never thrown
     */
    public void setFetchSize(int unused) throws SQLException {
	log.log("setFetchSize()", MsqlLog.JDBC,
		"Setting fetch size to " + unused + ".");
    }

    /**
     * Sets the result set type.
     * @param t the type
     */
    public void setType(int t) {
	log.log("setType()", MsqlLog.JDBC, "Result set type set to " + t +".");
	type = t;
    }
    
    /**
     * Performs an in-place modification of the specified column.
     * @param column the column number for the update
     * @param is the stream to update the column with
     * @param len the length of the stream
     * @exception java.sql.SQLException an error occurred reading the stream
     * or an invalid column number
     */
    public void updateAsciiStream(int col, InputStream is, int len)
    throws SQLException {
        byte[] str;

	str = new byte[len];
	try {
	    is.read(str, 0, len);
	    setColumn(col, new String(str, "8859_1"));
	}
	catch( IOException e ) {
	    throw new MsqlException(e);
	}
    }
    
    /**
     * Performs an in-place modification of the specified column.
     * @param column the name of the column to update
     * @param is the stream to update the column with
     * @param len the length of the stream
     * @exception java.sql.SQLException a database error occurred
     */
    public void updateAsciiStream(String cname, InputStream is, int len)
    throws SQLException {
	updateAsciiStream(findColumn(cname), is, len);
    }
    
    /**
     * Does an in-place update of the specified column in the current row.
     * @param column the column number for the update
     * @param d the value to use in the update
     * @exception java.sql.SQLException a database error occurred
     */
    public void updateBigDecimal(int column, BigDecimal d)
    throws SQLException {
	setColumn(column, d.toString());
    }
    
    /**
     * Does an in-place update of the specified column in the current row.
     * @param column the name of the column to update
     * @param d the value to use in the update
     * @exception java.sql.SQLException a database error occurred
     */
    public void updateBigDecimal(String cname, BigDecimal d)
    throws SQLException {
	updateBigDecimal(findColumn(cname), d);
    }
    
    /**
     * Does an in-place update of the specified column in the current row.
     * @param column the column number for the update
     * @param is the stream to update the column with
     * @param len the length of the stream
     * @exception java.sql.SQLException a database error occurred
     */
    public void updateBinaryStream(int column, InputStream is, int len)
    throws SQLException {
	byte[] data = new byte[len];
	
	try {
	    is.read(data, 0, len);
	}
	catch( IOException e ) {
	    throw new MsqlException(e);
	}
	updateBytes(column, data);
    }
    
    /**
     * Performs an in-place modification of the specified column.
     * @param column the name of the column to update
     * @param is the stream to update the column with
     * @param len the length of the stream
     * @exception java.sql.SQLException a database error occurred
     */
    public void updateBinaryStream(String cname, InputStream is, int len)
    throws SQLException {
	updateBinaryStream(findColumn(cname), is, len);
    }

    /**
     * Performs an in-place modification of a Blob column.
     * @param column the name of the column to update
     * @param b the Blob value
     * @exception java.sql.SQLException a database error occurred
     */
    public void updateBlob(int column, Blob b) throws SQLException {
	long len = b.length();

	if( len > Integer.MAX_VALUE ) {
	    throw new MsqlException("Binary length too long for mSQL.");
	}
	updateBinaryStream(column, b.getBinaryStream(), (int)len);
    }
    
    /**
     * Performs an in-place modification of the specified column.
     * @param column the column number for the update
     * @param b the value to use in the update
     * @exception java.sql.SQLException a database error occurred
     */
    public void updateBoolean(int column, boolean b) throws SQLException {
	setColumn(column, (b ? "1" : "0"));
    }
    
    /**
     * Performs an in-place modification of the specified column.
     * @param column the name of the column to update
     * @param b the value to use in the update
     * @exception java.sql.SQLException a database error occurred
     */
    public void updateBoolean(String cname, boolean b) throws SQLException {
	updateBoolean(findColumn(cname), b);
    }
    
    /**
     * Performs an in-place modification of the specified column.
     * @param column the column number for the update
     * @param b the value to use in the update
     * @exception java.sql.SQLException a database error occurred
     */
    public void updateByte(int column, byte b) throws SQLException {
	setColumn(column, "" + b);
    }
    
    /**
     * Performs an in-place modification of the specified column.
     * @param column the name of the column to update
     * @param b the value to use in the update
     * @exception java.sql.SQLException a database error occurred
     */
    public void updateByte(String cname, byte b) throws SQLException {
	updateByte(findColumn(cname), b);
    }
    
    /**
     * Performs an in-place modification of the specified column.
     * @param column the column number for the update
     * @param data the value to use in the update
     * @exception java.sql.SQLException a database error occurred
     */
    public void updateBytes(int column, byte[] data) throws SQLException {
	String value;
	
	try {
	    Encoder enc = Encoder.getInstance(Encoder.BASE64);
	    
	    // encode the binary data using base 64
	    data = enc.encode(data);
	    // now, convert the encoded array to a String
	    // base 64 is an ASCII encoding
	    value = new String(enc.encode(data), "8859_1");
	}
	catch( UnsupportedEncodingException e ) {
	    throw new MsqlException(e);
	}
	catch( NoSuchEncoderException e ) {
	    throw new MsqlException(e);
	}
	setColumn(column, value);
    }
    
    /**
     * Performs an in-place modification of the specified column.
     * @param column the name of the column to update
     * @param b the value to use in the update
     * @exception java.sql.SQLException a database error occurred
     */
    public void updateBytes(String cname, byte[] b) throws SQLException {
	updateBytes(findColumn(cname), b);
    }
    
    /**
     * Performs an in-place modification of the specified column.
     * @param column the column number for the update
     * @param in the reader to use for the update
     * @param len the length of the stream
     * @exception java.sql.SQLException a database error occurred
     */
    public void updateCharacterStream(int column, Reader in, int len)
    throws SQLException {
        char[] str = new char[len];

	try {
	    in.read(str, 0, len);
	}
	catch( IOException e ) {
	    throw new MsqlException(e);
	}
	setColumn(column, new String(str));
    }
    
    /**
     * Performs an in-place modification of the specified column.
     * @param column the name of the column to update
     * @param r the reader to use for the update
     * @param len the length of the stream
     * @exception java.sql.SQLException a database error occurred
     */
    public void updateCharacterStream(String cname, Reader r, int len)
    throws SQLException {
	updateCharacterStream(findColumn(cname), r, len);
    }
    
    /**
     * Performs an in-place modification of the specified column.
     * @param column the column number for the update
     * @param d the value to use in the update
     * @exception java.sql.SQLException a database error occurred
     */
    public void updateDate(int column, Date d) throws SQLException {
        SimpleDateFormat fmt;

	fmt = new SimpleDateFormat("dd-MMM-yyyy", Locale.US);
	setColumn(column, fmt.format(d));
    }
    
    /**
     * Performs an in-place modification of the specified column.
     * @param column the name of the column to update
     * @param d the value to use in the update
     * @exception java.sql.SQLException a database error occurred
     */
    public void updateDate(String cname, Date d) throws SQLException {
	updateDate(findColumn(cname), d);
    }
    
    /**
     * Performs an in-place modification of the specified column.
     * @param column the column number for the update
     * @param d the value to use in the update
     * @exception java.sql.SQLException a database error occurred
     */
    public void updateDouble(int column, double d) throws SQLException {
	setColumn(column, "" + d);
    }
    
    /**
     * Performs an in-place modification of the specified column.
     * @param column the name of the column to update
     * @param d the value to use in the update
     * @exception java.sql.SQLException a database error occurred
     */
    public void updateDouble(String cname, double d) throws SQLException {
	updateDouble(findColumn(cname), d);
    }
    
    /**
     * Performs an in-place modification of the specified column.
     * @param column the column number for the update
     * @param f the value to use in the update
     * @exception java.sql.SQLException a database error occurred
     */
    public void updateFloat(int column, float f) throws SQLException {
	setColumn(column, "" + f);
    }
    
    /**
     * Performs an in-place modification of the specified column.
     * @param column the name of the column to update
     * @param f the value to use in the update
     * @exception java.sql.SQLException a database error occurred
     */
    public void updateFloat(String cname, float f) throws SQLException {
	updateFloat(findColumn(cname), f);
    }
    
    /**
     * Performs an in-place modification of the specified column.
     * @param column the column number for the update
     * @param x the value to use in the update
     * @exception java.sql.SQLException a database error occurred
     */
    public void updateInt(int column, int x) throws SQLException {
	setColumn(column, "" + x);
    }
    
    /**
     * Performs an in-place modification of the specified column.
     * @param column the name of the column to update
     * @param x the value to use in the update
     * @exception java.sql.SQLException a database error occurred
     */
    public void updateInt(String cname, int x) throws SQLException {
	updateInt(findColumn(cname), x);
    }
    
    /**
     * Performs an in-place modification of the specified column.
     * @param column the column number for the update
     * @param l the value to use in the update
     * @exception java.sql.SQLException a database error occurred
     */
    public void updateLong(int column, long l) throws SQLException {
	setColumn(column, "" + l);
    }
    
    /**
     * Performs an in-place modification of the specified column.
     * @param column the name of the column to update
     * @param l the value to use in the update
     * @exception java.sql.SQLException a database error occurred
     */
    public void updateLong(String cname, long l) throws SQLException {
	updateLong(findColumn(cname), l);
    }
    
    /**
     * Performs an in-place modification of the specified column.
     * @param column the column number for the update
     * @exception java.sql.SQLException a database error occurred
     */
    public void updateNull(int column) throws SQLException {
	setColumn(column, "NULL");
    }
    
    /**
     * Performs an in-place modification of the specified column.
     * @param cname the name of the column to be updated
     * @exception java.sql.SQLException a database error occurred
     */
    public void updateNull(String cname) throws SQLException {
	updateNull(findColumn(cname));
    }
    
    /**
     * Performs an in-place modification of the specified column.
     * @param column the column number for the update
     * @param ob the value to use in the update
     * @exception java.sql.SQLException a database error occurred
     */
    public void updateObject(int column, Object ob) throws SQLException {
        if( ob instanceof Date ) {
	    updateDate(column, (Date)ob);
	}
	else if( ob instanceof String ) {
	    updateString(column, ob.toString());
	}
	else if( ob instanceof StringBuffer ) {
	    updateString(column, ob.toString());
	}
	else if( ob instanceof Blob ) {
	    updateBlob(column, (Blob)ob);
	}
	else if( ob instanceof byte[] ) {
	    updateBytes(column, (byte[])ob);
	}
	else if( ob instanceof Serializable ) {
	    try {
		ByteArrayOutputStream baos;
		ObjectOutputStream oos;
		
		baos = new ByteArrayOutputStream();
		oos = new ObjectOutputStream(baos);
		oos.writeObject(ob);
		oos.flush();
		updateBytes(column, baos.toByteArray());
	    }
	    catch( IOException e ) {
		throw new MsqlException(e);
	    }
	}
	else {
	    throw new MsqlException("Invalid binary object type.");
	}
    }
    
    /**
     * Performs an in-place modification of the specified column.
     * @param column the column number for the update
     * @param ob the value to use in the update
     * @param scale the scale to use for numeric types
     * @exception java.sql.SQLException a database error occurred
     */
    public void updateObject(int column, Object ob, int scale)
    throws SQLException {
	updateObject(column, ob);
    }
    
    /**
     * Performs an in-place modification of the specified column.
     * @param column the name of the column to update
     * @param ob the value to use in the update
     * @exception java.sql.SQLException a database error occurred
     */
    public void updateObject(String cname, Object ob) throws SQLException {
	updateObject(findColumn(cname), ob);
    }
    
    /**
     * Performs an in-place modification of the specified column.
     * @param column the name of the column to update
     * @param ob the value to use in the update
     * @param scale the scale for numeric types
     * @exception java.sql.SQLException a database error occurred
     */
    public void updateObject(String cname, Object ob, int scale)
    throws SQLException {
	updateObject(findColumn(cname), ob, scale);
    }

    /**
     * Sends changes to the current row to the database.  The changes will not
     * be committed without an explicit commit unless the connection is
     * auto-commit.
     * @throws java.sql.SQLException a database error occurred
     */
    public void updateRow() throws SQLException {
	Iterator tables;

	if( getConcurrency() != ResultSet.CONCUR_UPDATABLE ) {
	    throw new MsqlException("Result set is not updatable.");
	}
	prepareModification();
	tables = tableInfo.keySet().iterator();
	while( tables.hasNext() ) {
	    String table = (String)tables.next();
	    Iterator columns = ((ArrayList)tableInfo.get(table)).iterator();
	    ArrayList pk = (ArrayList)pkInfo.get(table);
	    StringBuffer where = null;
	    StringBuffer set = null;

	    while( columns.hasNext() ) {
		String col = (String)columns.next();
		String type = (String)columnTypes.get(table + "." + col);
		String val;

		if( type.equals("CHAR") || type.equals("TEXT") ) {
		    String tmp = getUpdate(findColumn(col));

		    val = "'" + MsqlPreparedStatement.fixString(tmp) + "'";
		}
		else {
		    val = getUpdate(findColumn(col));
		}
		if( pk.contains(col) ) {
		    if( where == null ) {
			where = new StringBuffer();
			where.append(" WHERE ");
		    }
		    else {
			where.append(" AND ");
		    }
		    where.append(col + "=" + val);
		}
		else {
		    if( set == null ) {
			set = new StringBuffer();
			set.append(" SET ");
		    }
		    else {
			set.append(", ");
		    }
		    set.append(col + "=" + val);
		}
	    }
	    statement.executeUpdate("UPDATE " + table + " " + set + " " +
				    where);
	}
    }
    
    /**
     * Performs an in-place modification of the specified column.
     * @param column the column number for the update
     * @param s the value to use in the update
     * @exception java.sql.SQLException a database error occurred
     */
    public void updateShort(int column, short s) throws SQLException {
	setColumn(column, "" + s);
    }
    
    /**
     * Performs an in-place modification of the specified column.
     * @param column the name of the column to update
     * @param s the value to use in the update
     * @exception java.sql.SQLException a database error occurred
     */
    public void updateShort(String cname, short s) throws SQLException {
	updateShort(findColumn(cname), s);
    }
    
    /**
     * Performs an in-place modification of the specified column.
     * @param column the column number for the update
     * @param str the value to use in the update
     * @exception java.sql.SQLException a database error occurred
     */
    public void updateString(int column, String str) throws SQLException {
	setColumn(column, str);
    }
    
    /**
     * Performs an in-place modification of the specified column.
     * @param column the name of the column to update
     * @param str the value to use in the update
     * @exception java.sql.SQLException a database error occurred
     */
    public void updateString(String cname, String str) throws SQLException {
	updateString(findColumn(cname), str);
    }
    
    /**
     * Performs an in-place modification of the specified column.
     * @param column the column number for the update
     * @param t the value to use in the update
     * @exception java.sql.SQLException a database error occurred
     */
    public void updateTime(int column, Time t) throws SQLException {
        SimpleDateFormat fmt;

	fmt = new SimpleDateFormat("HH:mm:ss", Locale.US);
	setColumn(column, fmt.format(t));
    }
    
    /**
     * Performs an in-place modification of the specified column.
     * @param column the name of the column to update
     * @param t the value to use in the update
     * @exception java.sql.SQLException a database error occurred
     */
    public void updateTime(String cname, Time t) throws SQLException {
	updateTime(findColumn(cname), t);
    }
    
    /**
     * Performs an in-place modification of the specified column.
     * @param column the column number for the update
     * @param ts the value to use in the update
     * @exception java.sql.SQLException a database error occurred
     */
    public void updateTimestamp(int column, Timestamp ts)throws SQLException {
	updateLong(column, ts.getTime());
    }
    
    /**
     * Performs an in-place modification of the specified column.
     * @param column the name of the column to update
     * @param ts the value to use in the update
     * @exception java.sql.SQLException a database error occurred
     */
    public void updateTimestamp(String cname, Timestamp ts)
    throws SQLException {
	updateTimestamp(findColumn(cname), ts);
    }
}
