/* $Id: MsqlRow.java,v 2.3 1999/07/19 17:40:09 borg Exp $ */
/* Copyright  1999 George Reese, All Rights Reserved */
package com.imaginary.sql.msql;

import java.io.UnsupportedEncodingException;
import java.sql.ResultSet;
import java.sql.SQLException;

/**
 * Represents a row from an mSQL database.  This class keeps track
 * of the data in a row in a result set and any in-place modifications
 * as well as the state of the row.
 * <BR>
 * Last modified $Date: 1999/07/19 17:40:09 $
 * @version $Revision: 2.3 $
 * @author George Reese (borg@imaginary.com)
 */
public class MsqlRow {
    /**
     * The unmodified state for a row from the database.
     */
    static public final int UNMODIFIED = 1;
    /**
     * The modified state for a row that has one or more columns updated.
     */
    static public final int MODIFIED   = 2;
    /**
     * The new state for a row that is to be inserted.
     */
    static public final int NEW        = 3;
    /**
     * The delted state for a row that is to be deleted.
     */
    static public final int DELETED    = 4;
    
    // The encoding for the data
    private String       encoding      = "8859_1";
    // Any modifications to any fields
    private String[]     modifications = null;
    // The row data as a tokenizer
    private ParsedRow    row           = null; 
    // The current state of the row
    private int          state         = MsqlRow.UNMODIFIED;

    /**
     * Constructs a new <CODE>MsqlRow</CODE> having the specified
     * number of columns.
     * @param sz the number of columns in the row
     * @param enc the encoding to be used for strings
     */
    MsqlRow(int sz, String enc) {
        super();
        modifications = new String[sz];
        encoding = enc;
    }

    /**
     * Constructs a new <CODE>MsqlRow</CODE> having the specified
     * initial values.
     * @param r the parsed row representing this row's data
     * @param enc the encoding to be used for strings
     */
    MsqlRow(ParsedRow r, String enc) {
        super();
        row = r;
        encoding = enc;
    }

    /**
     * Provides the raw form of the specified column <EM>index</EM>.
     * Unlike other column-operating methods, this refers to a
     * zero-based index and not a column number.
     * @param col the zero-based index of the desired column
     * @param type the result set type to determine change visibility
     * @throws java.sql.SQLException the result set type does not allow
     * visibility of an inserted row
     */
    String getColumn(int col, int type) throws SQLException {
        if( state != MsqlRow.UNMODIFIED ) {
            if( type == ResultSet.TYPE_SCROLL_SENSITIVE ) {
                return modifications[col];
            }
            else {
                if( row == null ) {
                    throw new MsqlException("Invalid row.");
                }
                else {
                    return row.get(col);
                }
            }
        }
        else {
            return row.get(col);
        }
    }

    /**
     * Marks the row for deletion.
     */
    void delete() {
        state = MsqlRow.DELETED;
    }

    /**
     * @return true if the row has been marked for deletion
     */
    boolean isDeleted() {
        return (state == MsqlRow.DELETED);
    }

    /**
     * @return true if the row is a new insert
     */
    boolean isInserted() {
        return (state == MsqlRow.NEW);
    }

    /**
     * @return true if the row has been modified
     */
    boolean isUpdated() {
        return (state == MsqlRow.MODIFIED);
    }

    /**
     * Clears any modifications made to this row.
     */
    void refresh() throws SQLException {
        modifications = null;
        if( state == MsqlRow.MODIFIED ) {
            state = state = MsqlRow.UNMODIFIED;
        }
    }
    
    /**
     * Assigns a new value to the specified column.
     * @param col the zero-based column index
     * @param val the new value
     */
    void setColumn(int col, String val) throws SQLException {
        String tmp;

        if( col < 0 ) {
            throw new MsqlException("Negative array index.");
        }
        if( modifications == null ) {
            modifications = new String[row.size()];
            for(int i=0; i<modifications.length; i++) {
                modifications[i] = row.get(i);
            }
        }
        if( col >= modifications.length ) {
            throw new IndexOutOfBoundsException();
        }
        tmp = modifications[col];
        if( tmp != null && tmp.equals(val) ) {
            return;
        }
        modifications[col] = val;
        if( state == MsqlRow.UNMODIFIED ) {
            state = MsqlRow.MODIFIED;
        }
    }

    /**
     * Sets the row state to <CODE>MsqlRow.UNMODIFIED</CODE> and
     * clears any modifications.
     */
    void update() throws SQLException {
        state = MsqlRow.UNMODIFIED;
        for(int i=0; i<row.size(); i++) {
            row.set(i, modifications[i]);
        }
        modifications = null;
    }
}
