// --------------------------------------------------------------------------
// The jSyncManager Project -- Source File.
// Copyright (c) 1998 - 2003 Brad BARCLAY <bbarclay@jsyncmanager.org>
// --------------------------------------------------------------------------
// OSI Certified Open Source Software
// --------------------------------------------------------------------------
//
// This library is free software; you can redistribute it and/or modify it
// under the terms of the GNU Lesser General Public License as published 
// by the Free Software Foundation; either version 2.1 of the License, or 
// (at your option) any later version.
//
// This library is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of 
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU 
// Lesser General Public License for more details.
//
// You should have received a copy of the GNU Lesser General Public 
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
//
// --------------------------------------------------------------------------
// $Id: JSerialTransport.java,v 1.5 2003/11/20 19:57:08 yaztromo Exp $
// --------------------------------------------------------------------------

package org.jSyncManager.Transport;

import org.jSyncManager.JSerial.*;
import org.jSyncManager.API.Transport.*;
import java.io.*;
import java.util.*;
import javax.swing.*;
import java.awt.event.*;
import java.awt.*;

// ==========================================================================

/** JSerial Transport implementation of the J-HotSync SLPTransport Interface. This class defines the transport
  * interface based on the J-HotSync Protocol stack's SLPTransportInterface Interface, providing hardware serial
  * port access through the Java Serial Handler, written by Brad BARCLAY. 
  * @author Brad BARCLAY &lt;bbarclay@jsyncmanager.org&gt;
  * @author Last modified by: $Author: yaztromo $ on $Date: 2003/11/20 19:57:08 $.
  * @version $Revision: 1.5 $
  */
  
public class JSerialTransport extends SerialTransportInterface implements ActionListener {
   private transient SerialInterface serial = null;
   private transient DataInputStream dis = null;
   private transient DataOutputStream dos = null;
   private transient boolean openState = false;

   /** A combo box to hold the communications port names.            */
   protected transient JComboBox comSelectBox = null;

   /** A combo box to hold the serial port speeds.                   */
   protected transient JComboBox speedSelectBox = null;

   /** The name of the port selected for I/O.                        */
   protected String portName = null;

// ==========================================================================

/** Construct a new instance of the JSerial Transport. 
  * @exception TransportException thrown if any problems occur during construction.
  */
public JSerialTransport() throws TransportException {
   super();
   try {
      Class.forName("org.jSyncManager.JSerial.SerialInterface");
   } /* endtry */
   catch (Exception e) {
      throw new TransportException(e);
   } /* endcatch */
} // end-constructor

// ==========================================================================

/** Closes the JSerial Transport. 
  * This method closes the JSerial Transport.
  * @exception TransportException this just throws any exceptions that occur during closure.
  */
public void close() throws TransportException {
   try {
      openState = false;
      dos.close();
      dis.skip(dis.available());
      dis.close();
   } /* endtry */
   catch (Exception e) {
      throw new TransportException(e);
   } /* endcatch */
} // end-method

// --------------------------------------------------------------------------

/** Flushes the input buffer of any remaining data. 
  * @exception TransportException thrown when a problem occurs with flushing the stream.
  */
public void flush() throws TransportException {
   try {
      dis.skip(dis.available());
   } /* endtry */
   catch (Exception e) {
      throw new TransportException(e);
   } /* endcatch */
} // end-method

// --------------------------------------------------------------------------

/** Returns the name of this J-Serial implementation. This method will return the platform name string for
  * the JSerial handler. 
  * @return the name of the local JSerial handler.
  */
public String getTransportName() {
   return getResourceBundle().getString("jserial.transport.name");
} // end-method

// --------------------------------------------------------------------------

/** Opens a read/write connection to the JSerial Transport. This method will open the serial port for
  * communications.
  * @exception TransportException throws any exceptions encountered while opening the port.
  */
public void open() throws TransportException {
   if (serial==null) initialize();
   try {
      if(!openState) serial.grabPort();
      openState = true;
   } /* endtry */
   catch (Exception e) {
      throw new TransportException(e);
   } /* endcatch */
} // end-method

// --------------------------------------------------------------------------

/** Read a single byte from the underlying data stream. This method will read and return the next available
  * byte in the stream.
  * @return the next byte from the underlying serial buffer/port.
  */
public byte readByte() {
   if(!openState) return 0;
   try {
      return dis.readByte();
   } catch(IOException ex) {
      return 0;
   }
} // end-method

// --------------------------------------------------------------------------

/** Changes the speed of the serial port. This method will tell the JSerial handler to change to the specified
  * speed. 
  * @param speed The speed to set the transport to.
  * @exception TransportException any exception encountered while setting the speed is rethrown.
  */
protected void setSpeed(int speed) throws TransportException {
   try {
      flush();
      serial.setCurrentSpeed(speed);
   } /* endtry */
   catch (Exception e) {
      throw new TransportException(e);
   } /* endcatch */
} // end-method

// --------------------------------------------------------------------------

/** Write an array of bytes to the underlying data stream. This method call will write an array of bytes to
  * the underlying serial data stream.
  * @param data the array of bytes to be written.
  */
public void writeBytes(byte data[]) {
   if(!openState) return;
   try {
      dos.write(data, 0, data.length);
   } catch(IOException ex) {}
} // end-method

// --------------------------------------------------------------------------

/** Constructs the configuration JPanel for this transport.
  * @return the configuration JPanel for this transport.
  */
protected JPanel constructConfigPanel() {
   JPanel ret=new JPanel();

   // Settings for this panel:   - a JComboBox containing the COM ports.
   //                            - a JComboBox containing the serial speeds.
   //                            - a JLabel for the COM port selection.
   //                            - a JLabel for the serial speed selection.
   
   JLabel comPortLabel = new JLabel(getResourceBundle().getString("transport.jserial.portlabel"));
   JLabel comSpeedLabel = new JLabel(getResourceBundle().getString("transport.jserial.speedlabel"));

   Vector v=new Vector();
   v.addElement("9600");
   v.addElement("19200");
   v.addElement("57600");
   v.addElement("115200");
   v.addElement("230400");

   Vector pnV=new Vector();
   pnV.addElement("Serial 0");
   pnV.addElement("Serial 1");
   pnV.addElement("Serial 2");
   pnV.addElement("Serial 3");
   pnV.addElement("Serial 4");
   pnV.addElement("Serial 5");
   pnV.addElement("Serial 6");
   pnV.addElement("Serial 7");

   comSelectBox = new JComboBox(pnV);
   speedSelectBox = new JComboBox(v);

   comSelectBox.addActionListener(this);
   speedSelectBox.addActionListener(this);

   // Layout the components into the JPanel
   ret.setLayout(new GridLayout(5, 1));
   ret.add(comPortLabel);
   ret.add(comSelectBox);
   ret.add(comSpeedLabel);
   ret.add(speedSelectBox);

   // Initialize the fields
   if (portName==null) portName=(String)pnV.elementAt(0);

   for (int i=0;i<speedSelectBox.getItemCount();i++) {
      try {
         if (Integer.parseInt((String)speedSelectBox.getItemAt(i))==serialSpeed) {
            speedSelectBox.setSelectedIndex(i);
         } /* endif */
      } /* endtry */
      catch (NumberFormatException e) {
         // Won't ever happen
      } /* endcatch */
   } /* endfor */

   for (int i=0;i<comSelectBox.getItemCount();i++) {
      try {
         if (((String)comSelectBox.getItemAt(i)).equals(portName)) {
            comSelectBox.setSelectedIndex(i);
         } /* endif */
      } /* endtry */
      catch (NumberFormatException e) {
         // Won't ever happen
      } /* endcatch */
   } /* endfor */

   return ret;
} // end-method

// --------------------------------------------------------------------------

/** Method that is called whenever an action is performed.
  * @param e the action that was performed.
  */
public void actionPerformed(ActionEvent e) {
   if (!(e.getSource() instanceof JComboBox)) return;
   JComboBox cb = (JComboBox)e.getSource();

   if (cb==comSelectBox) {
      portName=(String)comSelectBox.getSelectedItem();
      return;
   } /* endif */

   if (cb==speedSelectBox) {
      try {
         serialSpeed = Integer.parseInt((String)speedSelectBox.getSelectedItem());
      } /* endtry */
      catch (NumberFormatException ex) {
         // This should never happen, but if it does, we'll just use the default of 9600bps.
         System.err.println("*** A JSerialTransport parse error has occurred. Forcing 9600bps..."); 
         serialSpeed=INITIAL_SERIAL_SPEED;
      } /* endcatch */
   } /* endif */
} // end-method

// --------------------------------------------------------------------------

/** Initializes this transport.
  * @exception TransportException thrown if any initialization problems occur.
  */
public void initialize() throws TransportException {
   if (portName==null) {
      // If the port name isn't set, tell the implementing application that 
      // it should request the user to setup the transport.
      throw new TransportException(TransportException.TRANSPORT_NOT_ENOUGH_INFO);
   } /* endif */

   try {
      byte temp = Byte.valueOf(portName.substring(7)).byteValue();
      serial = new SerialInterface(temp);
      serial.setRTSCTS(true);

      dis = new DataInputStream(serial.getInputStream());
      dos = new DataOutputStream(serial.getOutputStream());
   } /* endtry */
   catch (Throwable t) {
      // We ran into an error -- rethrow the exception so the calling app can process.
      throw new TransportException(t);
   } /* endcatch */
} // end-method

// --------------------------------------------------------------------------

/** Initializes this transport from a properties object.
  * @param properties the Properties object containing the settings for this transport.
  * @param id the ID number for this transports settings.
  * @exception TransportException thrown if any initialization problems occur.
  */
public void initialize(Properties properties, int id) throws TransportException {
   // First, get the port name.
   portName=properties.getProperty("jsyncman.server.port."+id);

   try {
      serialSpeed = Integer.parseInt(properties.getProperty("jsyncman.server.speed."+id));
   } /* endtry */
   catch (NumberFormatException e) {
      // We're unable to parse the serial speed.
      throw new TransportException(new Exception(getResourceBundle().getString("transport.jserial.server.portspeed_error")
      +"jsyncman.server.speed."+id));
   } /* endcatch */

   initialize();
} // end-method

// --------------------------------------------------------------------------

/** Retreives descripitive text for a given transport exception.
  * @param ex the TransportException to examine.
  * @return descripitive text for a given transport exception.
  */
public String getTransportExceptionText(TransportException ex) {
   if (ex==null) return null;

   if (ex.getReasonCode()==TransportException.TRANSPORT_NOT_ENOUGH_INFO) {
      // This exception was thrown due to insufficient transport information.
      return getResourceBundle().getString("transport.jserial.insufficientinfo");
   } /* endif */

   if (ex.getReasonCode()==TransportException.TRANSPORT_EXCEPTION_OCCURRED) {
      // If we get here, it's because something threw an exception.
      if (ex.getException() instanceof IOException) {
         // We got here because we encountered an IOException in the close or flush method.
         return getResourceBundle().getString("transport.jserial.ioexceptiontext");
      } /* endif */

      if (ex.getException() instanceof NoClassDefFoundError) {
         // This one is serious.  It will generally mean that the Java Comm API can't be located at all.
         return getResourceBundle().getString("transport.jserial.noapiexceptiontext");
      } /* endif */

      // If we get here. it's because we've encountered an unknown exception.
      return getResourceBundle().getString("transport.jserial.unknownexception");
   } /* endif */

   // If we get here, it's for an unknown reason.  This should never happen.
   return getResourceBundle().getString("transport.jserial.undefinederror");
} // end-method

// --------------------------------------------------------------------------

/** Retreives the resource bundle name for use with this transport.
  * @return the resource bundle name for use with this transport.
  */
public java.lang.String getResourceBundleName() {
   return "org.jSyncManager.Transport.Resources.Text.JSerialTransportResources";
} // end-method

// --------------------------------------------------------------------------

/** Retreives the description for this transport.
  * @return the resource bundle name for use with this transport.
  */
public java.lang.String getTransportDescription() {
   return getResourceBundle().getString("transport.jserial.description");
} // end-method

// --------------------------------------------------------------------------

/** Retreives the preferred (user-specified) sync speed to use with this transport.
  * @return the user preferred sync speed.
  */
public int getPreferredSyncSpeed() {
   return serialSpeed;
} // end-method

// ==========================================================================

} // end-class

