/*
 * 2004  Abacus Research AG , St. Gallen , Switzerland . All rights reserved.
 * Terms of Use under The GNU GENERAL PUBLIC LICENSE Version 2
 *
 * THIS SOFTWARE IS PROVIDED BY ABACUS RESEARCH AG ``AS IS'' AND ANY EXPRESS 
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
 * WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, OR 
 * NON-INFRINGEMENT, ARE DISCLAIMED. IN NO EVENT SHALL ABACUS RESEARCH AG BE 
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
 * POSSIBILITY OF SUCH DAMAGE.
 *
 */

/*
 * JARadioGroup.java
 *
 * Creator:
 * 21.07.2003 10:35:32 lauchenauerst
 *
 * Maintainer:
 * 21.07.2003 10:35:32 lauchenauerst
 *
 * Last Modification:
 * $Id: JARadioGroup.java,v 1.9 2005/05/21 18:37:06 vegh Exp $
 *
 * Copyright (c) 2003 ABACUS Research AG, All Rights Reserved
 */

package ch.abacus.lib.ui;

import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.util.Iterator;
import java.util.Vector;

import javax.swing.ButtonGroup;

import ch.abacus.lib.ui.vorgaben.ApplicationSettingsInterface;

/**
 * This class is a container for dynamically creating a group of radio buttons.  The base class in a Panel where radio
 * buttons can be added and displayed.  The radio buttons can be added to the JARadioGroup component with the {@link
 * #addRadioButton addRadioButtons()} method. The index of the selected radio button can be set and retrieved with the
 * methods {@link #setSelectedIndex setSelectedIndex()} and {@link #getSelectedIndex getSelectedIndex()},
 * respectively.<br><br>
 * <p/>
 * If necessary, an ActionListener can also be assigned to detect when the state of the radio buttons in the
 * JARadioGroup change.<br><br>
 * <p/>
 * <b>Example of Use :</b><pre>
 *    // Declarations of variables for the user interface from the Design Cockpit *.proz.decl file
 *    private JARadioGroup RadioGroupTest_rdgTestRadioGroup;
 * <p/>
 *    // Assignments for the user interface from the Design Cockpit *.proz.decl file
 *    public void getReferences(){
 *        RadioGroupTest_rdgTestRadioGroup = (JARadioGroup) m_AbaRenderer.getObject(&quot;rdgTestRadioGroup&quot;);
 *    }
 * <p/>
 *    public void initializeUiComponents() {
 *        // If you do not not want to use the default vertical layout you can set another layout
 *        // before adding the radio buttons.
 *        // RadioGroupTest_rdgTestRadioGroup.setLayout(new java.awt.GridLayout());
 *        int maxRadioButtons = 2;
 *        for( int count = 0; count &lt; maxRadioButtons; count++ ) {
 *            // The text from the radio buttons would norally come from a NLS file.
 *            RadioGroupTest_rdgTestRadioGroup.addRadioButton(&quot;Option [&quot; + count + &quot;]&quot;);
 *        }
 *        // Select the first radio button
 *        RadioGroupTest_rdgTestRadioGroup.setSelectedIndex(0);
 *        // Add an action listener (see actionPerformed method below)
 *        RadioGroupTest_rdgTestRadioGroup.addActionListener(this);
 *    }
 * <p/>
 *    // Action listener for the JARadioGroup component
 *    public void actionPerformed(ActionEvent e) {
 *        int rdbIndex = RadioGroupTest_rdgTestRadioGroup.getSelectedIndex();
 *        if ( rdbIndex &gt;= 0 ) {
 *            // If you want more info from the JARadioButton you could do the following - normally the index is enough
 *            if ( e.getSource().equals(RadioGroupTest_rdgTestRadioGroup.getRadioButton(rdbIndex)) ) {
 *                System.out.println(&quot;Selected Index=&quot; + rdbIndex + &quot; Text=&quot; +
 *                                  RadioGroupTest_rdgTestRadioGroup.getRadioButton(rdbIndex).getText());
 * <p/>
 *            }
 *        }
 *    }</pre>
 */
public class JARadioGroup extends JAPanel implements ActionListener, ApplicationSettingsInterface {

  /**
   * Internal member class container to hold the radio buttons.
   */
  private static class PrivateButtonGroup extends ButtonGroup {
    public Vector getVector() {
      return buttons;
    }
  }

  /**
   * Internal member class container to hold the radio buttons.
   */
  private PrivateButtonGroup mBG = new PrivateButtonGroup();

  /**
   * Internal ActionListener to handle radio button actions.
   */
  private EventListenerContainer mActionListenerContainer;

  /**
   * Internal name of the component.
   */
  private String mSettingname = "radiolist";

  /**
   * Constructor for ApplicationSettingsRegistry
   */
  public JARadioGroup() {
  }

  /**
   * ActionListener implemented function
   */
  public void actionPerformed(ActionEvent _e) {
    fireActionPerformed(_e);
  }

  /**
   * Adds and ActionListener to the JARadioGroup components to receive events when the radio buttons state is changed
   * (i.e. selected).  The selected radio buttons can be determined with the {@link #getSelectedIndex
   * getSelectedIndex()} method.
   *
   * @param _al ActionListener the specified action listener where the action events are channelled
   */
  public void addActionListener(ActionListener _al) {
    if (mActionListenerContainer == null) {
      mActionListenerContainer = new EventListenerContainer();
    }
    mActionListenerContainer.addListener(_al);
  }

  /**
   * Removes the specified Action Listener from the component
   *
   * @param l the specified ActionListener to remove.
   */
  public void removeActionListener(ActionListener l) {
    if (mActionListenerContainer != null)
      mActionListenerContainer.removeListener(l);
  }

  /**
   * Adds a radio button with a caption to the end of the existing radio buttons list.
   *
   * @param _caption the text sting to be used for the radio button caption
   */
  public JARadioButton addRadioButton(String _caption) {
    return insertRadioButton(mBG.getButtonCount(), _caption);
  }

  /**
   * fireActionPerformed
   *
   * @param _e ActionEvent
   */
  private void fireActionPerformed(ActionEvent _e) {
    if (mActionListenerContainer == null) return;
    Iterator i = mActionListenerContainer.iterator();
    while (i.hasNext()) {
      ((ActionListener) i.next()).actionPerformed(_e);
    }
  }

  /**
   * Returns the radiobutton-element with the Index _pos
   *
   * @param _pos the zero-based index of the radio button (normally, in the order they were added)
   *
   * @return JARadioButton returns the JARadioButton component at the specified index
   */
  public JARadioButton getRadioButton(int _pos) {
    if (mBG.getVector().size() > _pos)
      return (JARadioButton) mBG.getVector().get(_pos);
    return null;
  }

  /**
   * Returns the current zero-based index of the selected radio button in the group
   *
   * @return int returns the zero-based index of the selected radio button in the group
   */
  public int getSelectedIndex() {
    int index = 0;
    while (index < mBG.getVector().size()) {
      if (((JARadioButton) mBG.getVector().get(index)).isSelected()) return index;
      index++;
    }
    return -1;
  }

  /**
   * Returns the setting name for the component
   *
   * @return String returns the setting name for the component
   */
  public String getSettingName() {
    return mSettingname;
  }

  /**
   * Inserts a radio button at the specified zero-based index position
   *
   * @param _position the zero-based index position to insert the radio button
   * @param _caption  the text sting to be used for the radio button caption
   */
  public JARadioButton insertRadioButton(int _position, String _caption) {
    JARadioButton rb = new JARadioButton(_caption);
    rb.addActionListener(this);
    rb.setEnabled(isEnabled());
    mBG.add(rb);
    add(rb, _position);
    return rb;
  }

  /**
   * Inserts a radio button at the specified zero-based index position
   * @param position the zero-based index position to insert the radio button
   * @param rb  an existing radiobutton
   */
  public void insertRadioButton(int position, JARadioButton rb) {
    rb.addActionListener(this);
    rb.setEnabled(isEnabled());
    mBG.add(rb);
    add(rb, position);
  }

  /**
   * Removes an existing button at the specified zero-based index from the JAButtonGroup component
   *
   * @param _index removes an existing button at the specified zero-based index
   */
  public void removeRadioButton(int _index) {
    if (getRadioButton(_index) != null) {
      remove(getRadioButton(_index));
      mBG.getVector().remove(_index);
    }
  }

  /**
   * Enables (or disables) the RadioGroup component and all contained RadioButtons.
   *
   * @param _b true to enable, otherwise false
   */
  public void setEnabled(boolean _b) {
    super.setEnabled(_b);
    int index = 0;
    while (index < mBG.getVector().size()) {
      ((JARadioButton) mBG.getVector().get(index)).setEnabled(_b);
      index++;
    }
  }

  /**
   * Selects the radio button at the specified zero-based index
   *
   * @param _i the specified zero-based index for the radio button to select
   */
  public void setSelectedIndex(int _i) {
    if (getRadioButton(_i) != null)
      getRadioButton(_i).setSelected(true);
  }

  /**
   * Set the setting name for the component
   *
   * @param _name the name of the component
   */
  public void setSettingName(String _name) {
    mSettingname = _name;
  }
}