/*
 * 2004  Abacus Research AG , St. Gallen , Switzerland . All rights reserved.
 * Terms of Use under The GNU GENERAL PUBLIC LICENSE Version 2
 *
 * THIS SOFTWARE IS PROVIDED BY ABACUS RESEARCH AG ``AS IS'' AND ANY EXPRESS 
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
 * WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, OR 
 * NON-INFRINGEMENT, ARE DISCLAIMED. IN NO EVENT SHALL ABACUS RESEARCH AG BE 
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
 * POSSIBILITY OF SUCH DAMAGE.
 *
 */

package ch.abacus.lib.ui.layout;

import java.awt.*;
import java.math.*;

import ch.abacus.lib.ui.layout.AnchoringComponent;

/**
2004  Abacus Research AG , St. Gallen , Switzerland .
 * All rights reserved. Terms of Use under The ABACUS RESEARCH AG PUBLIC LICENSE Version 1.0.
  THIS SOFTWARE IS PROVIDED BY ABACUS RESEARCH AG ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, OR NON-INFRINGEMENT, ARE DISCLAIMED. IN NO EVENT SHALL ABACUS RESEARCH AG
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
 * IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * Use like any other layout manager: (example)
 *
 * Container container = new Panel(new AnchoringLayoutManager());
 * container.add(control1, "control1");
 * container.add(control2, "control2");
 *
 * or you can set anchoring.
 *
 * AnchoringLayoutManager mgr = new AnchoringLayoutManager();
 * container.setLayout(mgr);
 * container.add(control1, "control1");
 * container.add(control2, "control2");
 * mgr.setAnchoring(control1, true, true, false, false);  // anchor on left and right
 * mgr.setAnchoring(control2, false, true, false, true);  // anchor on right and bottom.
 *
 */

public class AnchoringLayoutManager implements LayoutManager2 {
    AnchoringComponent firstAnchoringComponent = null;
    AnchoringComponent lastAnchoringComponent = null;
    private static final int PREFERRED = 0;
    private static final int MINIMUM = 1;
    private static final int DEFAULT_SCALE = 1000;
    public int width;
    public int height;
    public int scale = DEFAULT_SCALE;

    /**
     * AnchoringLayoutManager - default constructor.  Will autoresize.
     */
    public AnchoringLayoutManager() {
        width = -1;
        height = -1;
    }

    /**
     * AnchoringLayoutManager - set specific size for initial layout.
     * @param w - int - width of canvas
     * @param h - int - height of canvas
     */

    public AnchoringLayoutManager(int w, int h) {
        width = w;
        height = h;
    }

    /**
     * findAnchoringComponent - returns anchoring information about a stored component.
     * @param comp - Component - Component of container.
     * @return AnchoringComponent - anchoring data for the specific component.
     */
    public AnchoringComponent findAnchoringComponent(Component comp) {
        AnchoringComponent theAnchoringComponent = firstAnchoringComponent;
        while (theAnchoringComponent != null) {
            if (theAnchoringComponent.component.equals(comp))
                break;
            theAnchoringComponent = theAnchoringComponent.nextAnchoringComponent;
        }
        return theAnchoringComponent;
    }

    /**
     * getScale - return the scaling factor of the layout
     * @return int scaling factor (1000 = 1:1)
     */
    public int getScale() {
        return scale;
    }

    /**
     * setScale - sets the scaling factor of the layout
     * @param scale - int - the rescaling factor (1000 = 1:1, 2000 = 2:1, 500 = 1:2)
     */
    public void setScale(int scale) {
        this.scale = scale;
    }

    /**
     * addLayoutComponent - This method is called usually from the container's layout method.
     * A lot depends on how the objects are added.  This class is also designed to work with
     * layered panes where you wouldn't normally use a layout.  There is a bug with setLayer
     * in that it removes the component information for a LayoutManager2 class.
     *
     * @param name - String - name of object to store.
     * @param comp - Component - component to store.
     */
    public void addLayoutComponent(String name, Component comp) {
        // make sure component isn't already here.
        AnchoringComponent theMatch = this.findAnchoringComponent(comp);
        boolean bNewComponent = false;
        if (theMatch == null) {
            theMatch = new AnchoringComponent();
            bNewComponent = true;
        }
        Anchoring theAnchoring = new Anchoring(false, false, false, false);
        theMatch.anchoring = theAnchoring;
//        System.out.println("Adding layout component "+comp.getName());
        theMatch.name = new String(name);
        theMatch.component = comp;
        theMatch.scale = scale;
        if (bNewComponent) {
            if (firstAnchoringComponent == null)
                firstAnchoringComponent = theMatch;
            else
                lastAnchoringComponent.nextAnchoringComponent = theMatch;
            theMatch.prevAnchoringComponent = lastAnchoringComponent;
            lastAnchoringComponent = theMatch;
        }
    }

    /**
     * addLayoutComponent - this is the entry point for LayoutManager2 (see note above)
     *
     * @param comp - component to store.
     * @param name - name of component
     */
    public void addLayoutComponent(Component comp, Object name) {
        // layered panes remove and add back components when they change
        // layers.  When they add them back they pass null as the
        // second parameter.  You can fix this by setting your object
        // name before you add it to a JLayeredPane.
        if (name == null) { // WORKAROUND: setName()
            name = comp.getName();
        }
        if (name instanceof String)
            addLayoutComponent((String) name, comp);
        else
            throw new IllegalArgumentException("cannot add component " + comp.toString() + " to anchoring layout: must pass component and name (a string)");
    }

    /**
     * setAnchoring - This method allows anchoring properties to be set for a component.
     * If anchor left or top then normal behavior.  Anchor bottom or right moves with that
     * side.  Anchor on top and bottom and the object grows vertically.  Anchor on left
     * and right and the object grows horizontally.
     *
     * @param comp - Component - the component displayed.
     * @param left - boolean - whether anchors on left
     * @param right - boolean - whether anchors on right.
     * @param top - boolean - whether anchors on top
     * @param bottom - boolean - whether anchors on bottom.
     */
    public void setAnchoring(Component comp, boolean left, boolean right,
                             boolean top, boolean bottom) {
//        System.out.println("Setting layout component of "+comp.getName());
        AnchoringComponent theComponent = findAnchoringComponent(comp);
        if (theComponent != null) {
            Anchoring theAnchoring = new Anchoring(left, right, top, bottom);
            theComponent.anchoring = theAnchoring;
        } else {
            String sName = comp.getName();
            if (sName != null) {
                addLayoutComponent(comp.getName(), comp);
                theComponent = findAnchoringComponent(comp); // should be last
                theComponent.anchoring = new Anchoring(left, right, top, bottom);
            }
        }
    }

    /**
     * setAnchoring - This method allows anchoring properties to be set for a component.
     * If anchor left or top then normal behavior.  Anchor bottom or right moves with that
     * side.  Anchor on top and bottom and the object grows vertically.  Anchor on left
     * and right and the object grows horizontally.
     *
     * @param comp - Component - the component displayed.
     * @param iAnchoring - int - the anchoring value (see constant definitions in Anchoring.java)
     */
    public void setAnchoring(Component comp, int iAnchoring) {
//        System.out.println("Setting layout component of "+comp.getName());
        AnchoringComponent theComponent = findAnchoringComponent(comp);
        if (theComponent != null) {
            Anchoring theAnchoring = new Anchoring(iAnchoring);
            theComponent.anchoring = theAnchoring;
        } else {
            String sName = comp.getName();
            if (sName != null) {
                addLayoutComponent(comp.getName(), comp);
                theComponent = findAnchoringComponent(comp); // should be last
                theComponent.anchoring = new Anchoring(iAnchoring);
            }
        }
    }

    /**
     * removeLayoutComponent - invoked by container to remove a component.
     * Our implementation removes the stored data for the component.
     * @param comp - the component to remove.
     */
    public void removeLayoutComponent(Component comp) {
//        System.out.println("Removing layout component of "+comp.getName());
        AnchoringComponent theComponent = findAnchoringComponent(comp);
        if (theComponent != null) {
            if (theComponent.equals(firstAnchoringComponent)) {
                firstAnchoringComponent = theComponent.nextAnchoringComponent;
                if (firstAnchoringComponent != null)
                    firstAnchoringComponent.prevAnchoringComponent = null;
            }
            if (theComponent.equals(lastAnchoringComponent)) {
                lastAnchoringComponent = theComponent.prevAnchoringComponent;
                if (lastAnchoringComponent != null)
                    lastAnchoringComponent.nextAnchoringComponent = null;
            }
            if (theComponent.prevAnchoringComponent != null)
                theComponent.prevAnchoringComponent.nextAnchoringComponent = theComponent.nextAnchoringComponent;
            if (theComponent.nextAnchoringComponent != null)
                theComponent.nextAnchoringComponent.prevAnchoringComponent = theComponent.prevAnchoringComponent;
        }
    }

    /**
     * getName - return the name stored with the component.  Simple way of seeing
     * whether the component is stored in the layout manager.
     *
     * @param comp - component to search for.
     * @return String - stored name or null
     */
    public String getName(Component comp) {
        AnchoringComponent theComponent = findAnchoringComponent(comp);
        if (theComponent != null)
            return theComponent.name;
        return null;  // name is not stored for this component.
    }

    /**
     * minimumLayoutSize - get minimum space needed in canvas to show objects.
     *
     * @param parent - Container - the container associated with the layout.
     * @return Dimension - size needed.
     */
    public Dimension minimumLayoutSize(Container parent) {
        return layoutSize(parent, MINIMUM);
    }

    /**
     * preferredLayoutSize - get preferred space needed in canvas to show objects.
     *
     * @param parent - Container - the container associated with the layout.
     * @return Dimension - size preferred.
     */
    public Dimension preferredLayoutSize(Container parent) {
        return layoutSize(parent, PREFERRED);
    }


    /**
     * Returns the maximum dimensions for this layout given the components
     * in the specified target container.
     * @param target the component which needs to be laid out
     * @see Container
     * @see #minimumLayoutSize
     * @see #preferredLayoutSize
     */
    public Dimension maximumLayoutSize(Container target) {
        return new Dimension(Integer.MAX_VALUE, Integer.MAX_VALUE);
    }

    /**
     * Returns the alignment along the x axis.  This specifies how
     * the component would like to be aligned relative to other
     * components.  The value should be a number between 0 and 1
     * where 0 represents alignment along the origin, 1 is aligned
     * the furthest away from the origin, 0.5 is centered, etc.
     */
    public float getLayoutAlignmentX(Container parent) {
        return 0.5f;
    }

    /**
     * Returns the alignment along the y axis.  This specifies how
     * the component would like to be aligned relative to other
     * components.  The value should be a number between 0 and 1
     * where 0 represents alignment along the origin, 1 is aligned
     * the furthest away from the origin, 0.5 is centered, etc.
     */
    public float getLayoutAlignmentY(Container parent) {
        return 0.5f;
    }

    /**
     * Invalidates the layout, indicating that if the layout manager
     * has cached information it should be discarded.  We don't cache
     * this kind of data.
     */
    public void invalidateLayout(Container target) {
    }

    /**
     * layoutContainer - This places and sizes the objects using anchoring.
     * @param parent - the container to layout.
     */
    public void layoutContainer(Container parent) {
        Component comp;
        Component[] comps;
        Dimension dimComponent;
        Dimension dimMinComponent;
        Dimension dimMaxComponent;
        Dimension dimParent;
        Point ptLocation;
        int x;
        int y;
        int cx;
        int cy;
        // change size and location if scale changes.
        int iNewScale = scale;
        /// get all components.
        comps = parent.getComponents();
        // adjust for insets.
        Insets insets = parent.getInsets();
        int iHorzInsets = insets.left + insets.right;
        int iVertInsets = insets.top + insets.bottom;
        // we need the current size so we can find where the components go.
        Dimension dimParentRaw = parent.getSize();
        dimParent = new Dimension((int) dimParentRaw.width - iHorzInsets,
                (int) dimParentRaw.height - iVertInsets);
        // ok, process all components.
        for (int i = 0; i < comps.length; i++) {
            comp = comps[i];
            // get various sizes.
            dimComponent = comp.getSize();
            dimMinComponent = comp.getMinimumSize();
            dimMaxComponent = comp.getMaximumSize();
            // and now get where the object is.
            ptLocation = comp.getLocation();
            x = (int) ptLocation.getX();
            y = (int) ptLocation.getY();
            cx = (int) dimComponent.width;
            cy = (int) dimComponent.height;
            // get the anchoring component.
            AnchoringComponent theAnchoring = findAnchoringComponent(comp);
            if (theAnchoring == null) {
                System.err.println("Cannot find anchoring component for component:  " + comp.toString());
                return;
            }
            // if there is anchoring...
            if (theAnchoring.anchoring != null) {
                // if we have initialization, get the place and size of the object.
                if (theAnchoring.anchoring.anchorInitialized) {
                    if ((theAnchoring.anchoring.anchoring & Anchoring.LEFT) == Anchoring.LEFT) {
                        x = theAnchoring.anchoring.anchorLeft;
                        if ((theAnchoring.anchoring.anchoring & Anchoring.RIGHT) == Anchoring.RIGHT) {
                            cx = dimParent.width - (x + theAnchoring.anchoring.anchorRight);
                        }
                    } else if ((theAnchoring.anchoring.anchoring & Anchoring.RIGHT) == Anchoring.RIGHT)
                        x = dimParent.width - (cx + theAnchoring.anchoring.anchorRight);
                    if ((theAnchoring.anchoring.anchoring & Anchoring.TOP) == Anchoring.TOP) {
                        y = theAnchoring.anchoring.anchorTop;
                        if ((theAnchoring.anchoring.anchoring & Anchoring.BOTTOM) == Anchoring.BOTTOM) {
                            cy = dimParent.height - (y + theAnchoring.anchoring.anchorBottom);
                        }
                    } else if ((theAnchoring.anchoring.anchoring & Anchoring.BOTTOM) == Anchoring.BOTTOM)
                        y = dimParent.height - (cy + theAnchoring.anchoring.anchorBottom);
                } else { // initialize anchoring with beginning values.
                    if ((dimParent.width > 0) && (dimParent.height > 0)) {
                        if ((theAnchoring.anchoring.anchoring & Anchoring.LEFT) == Anchoring.LEFT)
                            theAnchoring.anchoring.anchorLeft = x;
                        if ((theAnchoring.anchoring.anchoring & Anchoring.RIGHT) == Anchoring.RIGHT)
                            theAnchoring.anchoring.anchorRight = dimParent.width - (x + (int) dimComponent.width);
                        if ((theAnchoring.anchoring.anchoring & Anchoring.TOP) == Anchoring.TOP)
                            theAnchoring.anchoring.anchorTop = y;
                        if ((theAnchoring.anchoring.anchoring & Anchoring.BOTTOM) == Anchoring.BOTTOM)
                            theAnchoring.anchoring.anchorBottom = dimParent.height - (y + (int) dimComponent.height);
                        theAnchoring.anchoring.anchorInitialized = true;
                    }
                }
            }
            // adjust for min and max values.
            if (cx < dimMinComponent.width)
                cx = dimMinComponent.width;
            if (cy < dimMinComponent.height)
                cy = dimMinComponent.height;
            int iMaxWidth = dimMaxComponent.width;
            int iMaxHeight = dimMaxComponent.height;
            if ((iMaxWidth != 0) && (cx > iMaxWidth))
                cx = iMaxWidth;
            if ((iMaxHeight != 0) && (cy > iMaxHeight))
                cy = iMaxHeight;
            // set the object bounds.
            if (iNewScale != theAnchoring.scale) {
                int oldScale = theAnchoring.scale;
                comp.setBounds(x * iNewScale / oldScale, y * iNewScale / oldScale,
                        cx * iNewScale / oldScale, cy * iNewScale / oldScale);
                if (comp instanceof CustomScaleableInterface) {
                    ((CustomScaleableInterface) comp).scaleChanged(iNewScale, oldScale);
                } else {
                    Font font = comp.getFont();
                    if (font != null) {
                        Font newFont = new Font(font.getName(), font.getStyle(), font.getSize() * iNewScale / oldScale);
                        comp.setFont(newFont);
                    }
                }
                theAnchoring.scale = iNewScale;
            } else
                comp.setBounds(x, y, cx, cy);
        }
    }

    /**
     * layoutSize - used to determine minimum size, preferred size, and maximum size.
     * @param parent - the object to layout.
     * @param iType - the type of layout (preferred, minimum, maximum)
     * @return Dimension - space required.
     */
    protected Dimension layoutSize(Container parent, int iType) {
        int newWidth;
        int newHeight;
        if ((width == -1) || (height == -1)) {
            Component comp;
            Component[] comps;
            Dimension dimComponent;
            Dimension dimParent;
            Point pt;
            int x;
            int y;

            newWidth = newHeight = 0;
            comps = parent.getComponents();
            for (int i = 0; i < comps.length; i++) {
                comp = comps[i];
                // get anchoring data.
                AnchoringComponent theAnchoring = findAnchoringComponent(comp);
                // use object's preferred or minimum size.
                if (iType == PREFERRED)
                    dimComponent = comp.getPreferredSize();
                else
                    dimComponent = comp.getMinimumSize();
                // get parent size. (where we are laying out)
                dimParent = parent.getSize();
                // get component location within parent.
                pt = comp.getLocation();
                // Ok, find the size needed.
                x = dimComponent.width;
                if (theAnchoring != null) {
                    if ((theAnchoring.anchoring.anchoring & Anchoring.LEFT) == Anchoring.LEFT)
                        x += (int) pt.getX(); // size is object size + object location
                    if ((theAnchoring.anchoring.anchoring & Anchoring.RIGHT) == Anchoring.RIGHT)
                        x += (int) (dimParent.width - pt.getX() - dimComponent.width); // anchored on right
                }
                y = dimComponent.height;
                if (theAnchoring != null) {
                    if ((theAnchoring.anchoring.anchoring & Anchoring.TOP) == Anchoring.TOP)
                        y += (int) pt.getY();
                    if ((theAnchoring.anchoring.anchoring & Anchoring.BOTTOM) == Anchoring.BOTTOM)
                        y += (int) (dimParent.height - pt.getY() - dimComponent.height); // anchored on bottom
                }
                newWidth = Math.max(newWidth, x);
                newHeight = Math.max(newHeight, y);
            }
            // stretch to max if width and height are not set.
            if (width != -1)
                newWidth = width;
            if (height != -1)
                newHeight = height;
        } else {
            newWidth = width;
            newHeight = height;
        }
        // adjust for insets.
        Insets insets = parent.getInsets();
        return (new Dimension(newWidth + insets.left + insets.right,
                newHeight + insets.top + insets.bottom));
    }
}