/*
 * 2004  Abacus Research AG , St. Gallen , Switzerland . All rights reserved.
 * Terms of Use under The GNU GENERAL PUBLIC LICENSE Version 2
 *
 * THIS SOFTWARE IS PROVIDED BY ABACUS RESEARCH AG ``AS IS'' AND ANY EXPRESS 
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
 * WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, OR 
 * NON-INFRINGEMENT, ARE DISCLAIMED. IN NO EVENT SHALL ABACUS RESEARCH AG BE 
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
 * POSSIBILITY OF SUCH DAMAGE.
 *
 */

package ch.abacus.lib.ui.renderer.common;

/**
 * Title:        MetadataProvider
 * Description:  This is the class that allows access to the system metadata
 *               (both through the metadata xml file and java introspection.)
 * Copyright:    Copyright (c) 2001
 * Company:      Abacus Software
 * @author       Michael Gouker (Cagey Logic)
 * @version 0.1
 */

import java.lang.reflect.Constructor;
import java.lang.reflect.Field;
import java.lang.reflect.Modifier;
import java.util.ArrayList;


/**
 * MetadataProvider - This class provides access to data about the types
 * of objects that the UIFactory can support.  It uses java introspection
 * to query the system for capabilities.  It also stores the metadata XML
 * document that is used during code generation to guide its execution.
 */
public class MetadataProvider {


    /**
     * docMetadata - This is the xml document that stores metadata about the
     * objects that can be constructed by the UIFactory.
     */
    public MetaProject theDesignProject;

    /**
     * MetadataProvider - This constructor creates the MetadataProvider object.
     *
     */
    public MetadataProvider(MetaProject theDesignProject) {
        this.theDesignProject = theDesignProject;
    }

    /**
     * Get default list of propeties for generation.
     *
     * @param sObjectClass Class whose properties are to be queried.
     */

    public ArrayList getDefaultPropertyValues(String sObjectClass) {
        ArrayList theReturnValues = new ArrayList();
        MetaClass theClass = theDesignProject.findClass(sObjectClass);
        while (theClass != null) {
            MetaPropertyDescriptor theProperty = theClass.theMetadata.theFirstProperty;
            while (theProperty != null) {
                String sDefaultValue = theProperty.getDefaultValue();
                String sSetMethod = theProperty.sSetName;
                String sType = theProperty.sType;
                String sClass = theProperty.sClass;
                if (sDefaultValue != null) {
                    theReturnValues.add(sDefaultValue);
                    theReturnValues.add(sSetMethod);
                    theReturnValues.add(sType);
                    theReturnValues.add(sClass);
                }
                theProperty = theProperty.theNextProperty;
            }
            theClass = theClass.theParentClass;
        }
        return theReturnValues;
    }

    /**
     * FindMatchingFinalFieldName - This method is used to search for static final
     * values (such as names of colors or other types of objects) by object value.
     * If a color object, for example, of "cyan" is passed to the function it will
     * return a string "cyan". It can throw ReportField not found or illegal access
     * exceptions which must be trapped by the caller.
     *
     * @param obj - The value to search for - an Object.
     */
    public static String FindMatchingFinalFieldName(Object obj)
            throws IllegalAccessException {
        Class c = obj.getClass();
        Modifier mod = new Modifier();
        Field[] fields = c.getFields();
        int iFieldCount = fields.length;
        for (int i = iFieldCount - 1; i >= 0; i--) {
            Field field = fields[i];
            int iModifiers = field.getModifiers();
            // Look for a final static that matches.
            if (mod.isFinal(iModifiers) && mod.isStatic(iModifiers)) {
                Object objTest = field.get(obj);
                if (objTest.equals(obj))
                    return field.getName();
            }
        }
        return null;
    }


    /**
     * GetFieldValueAsInteger - This method searches for a ReportField and returns an
     * integral value.
     *
     * @param  c - The class that will be queried for the ReportField value.
     * @param  sFieldName - The name of the ReportField to search for.
     *
     */
    public static int GetFieldValueAsInteger(Class c, String sFieldName)
            throws NoSuchFieldException, IllegalAccessException, UIFactoryException {
        int value = 0;
        if (c != null) {
            Field fld = c.getField(sFieldName);
            if (fld != null)
                value = fld.getInt(fld);
            else
                throw new UIFactoryException("Bad Integer Value");
        }
        return value;
    }

    /**
     * GetFieldValueAsObject - This method searches for a ReportField and returns
     * the value as an object.
     *
     * @param objTest - The Object that will be queried for the ReportField value.
     * @param sFieldName - The name of the ReportField to search for.
     *
     */
    public static Object GetFieldValueAsObject(Object objTest, String sFieldName)
            throws NoSuchFieldException, IllegalAccessException {
        Class c = objTest.getClass();
        if (c != null) {
            Field fld = c.getField(sFieldName);
            if (fld != null)
                return fld.get(objTest);
        }
        return null;
    }

    public static Object createBogusObject(Class cls) {
        if (cls.equals(String.class))
            return new String("0");
        else if (cls.equals(int.class))
            return new Integer(0);
        else if (cls.equals(double.class))
            return new Double(0.0);
        else if (cls.equals(float.class))
            return new Float(0.0);
        else if (cls.equals(char.class))
            return new Character('0');
        else if (cls.equals(byte.class))
            return new Byte((byte) 0);
        else if (cls.equals(long.class))
            return new Long((long) 0);
        else if (cls.equals(boolean.class))
            return Boolean.FALSE;

        try {
            return cls.newInstance();
        } catch (Exception e1) {
            try {
                Class[] classes = new Class[1];
                classes[0] = String.class;
                Constructor constructorString = cls.getConstructor(classes);
                Object[] objects = new Object[classes.length];
                objects[0] = new String("0");
                return constructorString.newInstance(objects);
            } catch (Exception e2) {
                try {
                    Constructor[] constructors = cls.getConstructors();
                    Class[] classes = constructors[0].getParameterTypes();
                    Object[] objects = new Object[classes.length];
                    for (int i = 0; i < classes.length; i++) {
                        objects[i] = MetadataProvider.createBogusObject(classes[i]);
                    }
                    return constructors[0].newInstance(objects);
                } catch (Exception e3) {
                    return null;
                }
            }
        }

    }

    public static Object GetFieldValueAsObject(String sClassName, String sFieldName)
            throws NoSuchFieldException, IllegalAccessException, ClassNotFoundException {
        Class c = Class.forName(sClassName);
        Object objTest = MetadataProvider.createBogusObject(c);
        if (c != null) {
            Field fld = c.getField(sFieldName);
            return fld.get(objTest);
        }
        return null;
    }


    public static boolean IsSimpleClass(Class theClass) {
        try {
            // If it has a 1 parameter constructor that takes a string.
            Class[] classes = new Class[1];
            classes[0] = Class.forName("java.lang.String");
            theClass.getConstructor(classes);
            return true;
        } catch (Exception e1) {
            return false;
        }
    }

}
