/*
 * 2004  Abacus Research AG , St. Gallen , Switzerland . All rights reserved.
 * Terms of Use under The GNU GENERAL PUBLIC LICENSE Version 2
 *
 * THIS SOFTWARE IS PROVIDED BY ABACUS RESEARCH AG ``AS IS'' AND ANY EXPRESS 
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
 * WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, OR 
 * NON-INFRINGEMENT, ARE DISCLAIMED. IN NO EVENT SHALL ABACUS RESEARCH AG BE 
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
 * POSSIBILITY OF SUCH DAMAGE.
 *
 */

package ch.abacus.lib.ui.renderer.common.jdbc;

import com.sun.rowset.JdbcRowSetImpl;

import java.sql.*;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Map;

/**
 * Adam Graham, Erin Bay Associates, LLC.
 * Oct 11, 2005, 5:18:27 PM
 */
class JSSUpdatableRowSet extends JdbcRowSetImpl
{
    boolean forceNonUpdatableMode = false; // Set to True for Testing only
    private String updatableTableName;
    private String uniqueColumnName;

    private HashMap rowChanges = new HashMap();

    public void updateNull(String s) throws SQLException
    {
        if (getConcurrency() != ResultSet.CONCUR_UPDATABLE || forceNonUpdatableMode)
            rowChanges.put(s, null);
        else
            super.updateNull(s);
    }

    public void updateInt(String s, int i) throws SQLException
    {
        if (getConcurrency() != ResultSet.CONCUR_UPDATABLE || forceNonUpdatableMode)
            rowChanges.put(s, i);
        else
            super.updateInt(s, i);
    }

    public void updateLong(String s, long l) throws SQLException
    {
        if (getConcurrency() != ResultSet.CONCUR_UPDATABLE || forceNonUpdatableMode)
            rowChanges.put(s, l);
        else
            super.updateLong(s, l);
    }

    public void updateFloat(String s, float v) throws SQLException
    {
        if (getConcurrency() != ResultSet.CONCUR_UPDATABLE || forceNonUpdatableMode)
            rowChanges.put(s, v);
        else
            super.updateFloat(s, v);
    }

    public void updateDouble(String s, double v) throws SQLException
    {
        if (getConcurrency() != ResultSet.CONCUR_UPDATABLE || forceNonUpdatableMode)
            rowChanges.put(s, v);
        else
            super.updateDouble(s, v);
    }

    public void updateDate(String s, Date date) throws SQLException
    {
        if (getConcurrency() != ResultSet.CONCUR_UPDATABLE || forceNonUpdatableMode)
            rowChanges.put(s, date);
        else
            super.updateDate(s, date);
    }

    public void updateBoolean(String s, boolean b) throws SQLException
    {
        if (getConcurrency() != ResultSet.CONCUR_UPDATABLE || forceNonUpdatableMode)
            rowChanges.put(s, b);
        else
            super.updateBoolean(s, b);
    }

    public void updateString(String s, String s1) throws SQLException
    {
        if (getConcurrency() != ResultSet.CONCUR_UPDATABLE || forceNonUpdatableMode)
        {
            System.err.println("rowChanges.put(" + s + ", " + s1 + ")");
            rowChanges.put(s, s1);
        }
        else
            super.updateString(s, s1);
    }

    public JSSUpdatableRowSet(Connection conn, String param_updatableTableName, String param_uniqueColumnName) throws SQLException
    {
        super(conn);
        setUpdatableTableName(param_updatableTableName);
        setUniqueColumnName(param_uniqueColumnName);
    }

    public String getUpdatableTableName() throws SQLException
    {
        if (updatableTableName == null)
            updatableTableName = getMetaData().getTableName(1);

        return updatableTableName;
    }

    public void setUpdatableTableName(String tableName)
    {
        this.updatableTableName = tableName;
    }

    public String getUniqueColumnName()
    {
        try
        {
            if (uniqueColumnName == null)
                uniqueColumnName = getPrimaryKey(getConnection(), updatableTableName);
        }
        catch (Exception e)
        {
            e.printStackTrace();  //To change body of catch statement use File | Settings | File Templates.
        }

        return uniqueColumnName;
    }

    public void setUniqueColumnName(String uniqueColumnName)
    {
        this.uniqueColumnName = uniqueColumnName;
    }

    public void insertRow() throws SQLException
    {
        if (getConcurrency() != ResultSet.CONCUR_UPDATABLE || forceNonUpdatableMode)
        {
            int n = execSQLInsert(getUpdatableTableName(), getUniqueColumnName(), getConnection(), this, rowChanges);
            if (n > 0)
            {
                refreshRow();
                notifyRowChanged();
            }
        }
        else
            super.insertRow();
    }

    public void updateRow() throws SQLException
    {
        if (getConcurrency() != ResultSet.CONCUR_UPDATABLE || forceNonUpdatableMode)
        {
            System.err.println("execSQLUpdate(): " + rowChanges);
            int n = execSQLUpdate(getUpdatableTableName(), getUniqueColumnName(), getConnection(), this, rowChanges);
            if (n > 0)
            {
                //refreshRow();
                notifyRowChanged();
            }
        }
        else
            super.updateRow();
    }

    public void deleteRow() throws SQLException
    {
        if (getConcurrency() != ResultSet.CONCUR_UPDATABLE || forceNonUpdatableMode)
        {
            int n = execSQLDelete(getUpdatableTableName(), getUniqueColumnName(), getConnection(), this);
            if (n > 0)
            {
                refreshRow();
                notifyRowChanged();
            }
        }
        else
            super.deleteRow();
    }


    /**
     * Get quote for Strings
     */
    public static String getFieldQuote(ResultSetMetaData rsmd, String fieldName)
    {
        int i = -1;
        String fieldQuote = "";
        int fieldType = 0;

        try
        {
            int numcols = rsmd.getColumnCount();
            for (int i1 = 1; i1 <= numcols; i1++)
            {
                if (rsmd.getColumnLabel(i1).equalsIgnoreCase(fieldName))
                {
                    i = i1;
                    break;
                }
            }
            if (i == -1) return null;

            fieldType = rsmd.getColumnType(i);
            fieldQuote = "";
            if (fieldType == java.sql.Types.VARCHAR ||
                    fieldType == java.sql.Types.CHAR ||
                    fieldType == java.sql.Types.LONGVARCHAR ||
                    fieldType == java.sql.Types.TIME ||
                    fieldType == java.sql.Types.TIMESTAMP ||
                    fieldType == java.sql.Types.DATE)
            {
                fieldQuote = "'";
            }
        }
        catch (SQLException e)
        {
            ;
        }

        return fieldQuote;
    }

    public int execSQLUpdate(String paramTable, String uniqueFieldName, Connection dbconn, ResultSet rs, Map paramFields) throws SQLException
    {
        ResultSetMetaData rsmd = rs.getMetaData();
        PreparedStatement updateST = null;
        String qrystr = "";
        String valstr = "";
        String fieldQuote = "";
        ArrayList validFields = new ArrayList();

        // Build SQL
        boolean bFirstColumn = true;
        java.util.Iterator fieldnames = paramFields.keySet().iterator();
        while (fieldnames.hasNext())
        {
            Object o = fieldnames.next();
            String aname = String.valueOf(o);
            if (!isField(paramTable, aname, dbconn))
                continue;
            if (aname.equalsIgnoreCase(uniqueFieldName))
                continue;

            if (bFirstColumn)
            {
                valstr = aname + "=?";
                bFirstColumn = false;
            }
            else
            {
                valstr += "," + aname + "=?";
            }

            validFields.add(aname);
        }

        qrystr = "UPDATE " + paramTable + " SET " + valstr;

        String uniqueFieldValue = rs.getString(uniqueFieldName);
        fieldQuote = getFieldQuote(rsmd, uniqueFieldName);
        qrystr = qrystr + " WHERE " + uniqueFieldName + "=" + fieldQuote + uniqueFieldValue + fieldQuote;
System.err.println("execSQLUpdate(): " + qrystr);

        updateST = dbconn.prepareStatement(qrystr);

        for (int fld = 0; fld < validFields.size(); fld++)
        {
            String aname = (String) validFields.get(fld);
            String avalue = (String) paramFields.get(aname);
System.err.println("execSQLUpdate(): " + aname + "=" + avalue);
            updateST.setString(fld + 1, avalue);
        }

        int numUpdates = updateST.executeUpdate();
        updateST.close();

        return numUpdates;
    }

    public int execSQLInsert(String paramTable, String uniqueFieldName, Connection dbconn, ResultSet rs, Map paramFields) throws SQLException
    {
        ResultSetMetaData rsmd = rs.getMetaData();
        PreparedStatement updateST = null;
        String qrystr = "";
        String valstr = "";
        ArrayList validFields = new ArrayList();
        qrystr = "INSERT INTO " + paramTable + " (";

        // Build SQL
        boolean bFirstColumn = true;
        java.util.Iterator fieldnames = paramFields.keySet().iterator();
        while (fieldnames.hasNext())
        {
            Object o = fieldnames.next();
            String aname = String.valueOf(o);
            if (!isField(paramTable, aname, dbconn))
                continue;

            // Check if already included, can't have duplicates.
            if (("," + qrystr + ",").toLowerCase().indexOf(("," + aname + ",").toLowerCase()) > 0)
                continue;

            if (bFirstColumn)
            {
                qrystr += aname;
                valstr = "?";
                bFirstColumn = false;
            }
            else
            {
                qrystr += "," + aname;
                valstr += ",?";
            }

            validFields.add(aname);
        }
        qrystr += ") VALUES (" + valstr + ") ";

        updateST = dbconn.prepareStatement(qrystr);

        for (int fld = 0; fld < validFields.size(); fld++)
        {
            String aname = (String) validFields.get(fld);
            String avalue = (String) paramFields.get(aname);
            updateST.setString(fld + 1, avalue);
        }

        int numUpdates = updateST.executeUpdate();

        updateST.close();

        return numUpdates;
    }


    public static String getPrimaryKey(Connection con, String tableName)
    {
        String[] key = {"TABLE_NAME", "COLUMN_NAME", "KEY_SEQ", "PK_NAME"};
        DatabaseMetaData dbMeta = null;

        try
        {
            dbMeta = con.getMetaData();
        }
        catch (SQLException e)
        {
            return null;
        }

        ResultSet rsKey = null;
        try
        {
            rsKey = dbMeta.getPrimaryKeys(null, null, tableName.toUpperCase());
        }
        catch (Exception e)
        {
            try
            {
                rsKey = dbMeta.getPrimaryKeys(null, null, tableName);
            }
            catch (Exception e1)
            {
                return null;
            }
        }

        String scolname = null;
        try
        {
            if (rsKey.next())
                scolname = rsKey.getString(key[1]);

            rsKey.close();
        }
        catch (SQLException e)
        {
        }

        return scolname;
    }


    public int execSQLDelete(String paramTable, String uniqueFieldName, Connection dbconn, ResultSet rs) throws SQLException
    {
        ResultSetMetaData rsmd = rs.getMetaData();
        Statement updateST = null;
        updateST = dbconn.createStatement();
        String qrystr = "";
        String fieldQuote = "";

        qrystr = "DELETE FROM " + paramTable;

        String uniqueFieldValue = rs.getString(uniqueFieldName);
        fieldQuote = getFieldQuote(rsmd, uniqueFieldName);
        qrystr = qrystr + " WHERE " + uniqueFieldName + "=" + fieldQuote + uniqueFieldValue + fieldQuote;

        int numUpdates = updateST.executeUpdate(qrystr);
        updateST.close();

        return numUpdates;
    }

    /**
     * isField
     */
    public boolean isField(String tableName, String fieldName, Connection dbconn)
    {
        boolean foundField = false;

        try
        {
            // Get DatabaseMetaData
            DatabaseMetaData dbmd = dbconn.getMetaData();

            ResultSet columnRS = dbmd.getColumns(null, null, tableName, "%");
            if (!columnRS.next())
            {
                columnRS = dbmd.getColumns(null, null, tableName.toLowerCase(), "%");
                if (!columnRS.next())
                {
                    columnRS = dbmd.getColumns(null, null, tableName.toUpperCase(), "%");
                    if (!columnRS.next())
                    {
                        columnRS.close();
                        return false;
                    }
                }
            }

            if (columnRS == null)
                return false;

            do
            {
                String dbColumnName = columnRS.getString(4);
                if (fieldName.equalsIgnoreCase(dbColumnName))
                {
                    foundField = true;
                    break;
                }
            }
            while (columnRS.next());

            columnRS.close();
        }
        catch (Exception E)
        {
            E.printStackTrace();
        }

        return foundField;
    }

    protected void notifyCursorMoved() throws SQLException
    {
        if (getConcurrency() != ResultSet.CONCUR_UPDATABLE || forceNonUpdatableMode)
            rowChanges.clear();
        super.notifyCursorMoved();
    }

    protected void notifyRowChanged() throws SQLException
    {
        //if (getConcurrency() != ResultSet.CONCUR_UPDATABLE || forceNonUpdatableMode)
        //    rowChanges.clear();
        super.notifyRowChanged();
    }

    protected void notifyRowSetChanged() throws SQLException
    {
        //if (getConcurrency() != ResultSet.CONCUR_UPDATABLE || forceNonUpdatableMode)
        //    rowChanges.clear();
        super.notifyRowSetChanged();
    }

    @Override public boolean getBoolean(String s) throws SQLException
    {
        if ((getConcurrency() != ResultSet.CONCUR_UPDATABLE || forceNonUpdatableMode) && rowChanges.containsKey(s))
            return ((Boolean) rowChanges.get(s)).booleanValue();
        else
            return super.getBoolean(s);    //To change body of overridden methods use File | Settings | File Templates.
    }

    @Override public Date getDate(String s) throws SQLException
    {
        if ((getConcurrency() != ResultSet.CONCUR_UPDATABLE || forceNonUpdatableMode) && rowChanges.containsKey(s))
            return (Date) rowChanges.get(s);
        else
            return super.getDate(s);    //To change body of overridden methods use File | Settings | File Templates.
    }

    @Override public double getDouble(String s) throws SQLException
    {
        if ((getConcurrency() != ResultSet.CONCUR_UPDATABLE || forceNonUpdatableMode) && rowChanges.containsKey(s))
            return ((Double) rowChanges.get(s)).doubleValue();
        else
            return super.getDouble(s);    //To change body of overridden methods use File | Settings | File Templates.
    }

    @Override public float getFloat(String s) throws SQLException
    {
        if ((getConcurrency() != ResultSet.CONCUR_UPDATABLE || forceNonUpdatableMode) && rowChanges.containsKey(s))
            return ((Float) rowChanges.get(s)).floatValue();
        else
            return super.getFloat(s);    //To change body of overridden methods use File | Settings | File Templates.
    }

    @Override public int getInt(String s) throws SQLException
    {
        if ((getConcurrency() != ResultSet.CONCUR_UPDATABLE || forceNonUpdatableMode) && rowChanges.containsKey(s))
            return ((Integer) rowChanges.get(s)).intValue();
        else
            return super.getInt(s);    //To change body of overridden methods use File | Settings | File Templates.
    }

    @Override public long getLong(String s) throws SQLException
    {
        if ((getConcurrency() != ResultSet.CONCUR_UPDATABLE || forceNonUpdatableMode) && rowChanges.containsKey(s))
            return ((Long) rowChanges.get(s)).longValue();
        else
            return super.getLong(s);    //To change body of overridden methods use File | Settings | File Templates.
    }

    @Override public String getString(String s) throws SQLException
    {
        if ((getConcurrency() != ResultSet.CONCUR_UPDATABLE || forceNonUpdatableMode) && rowChanges.containsKey(s))
            return (String) rowChanges.get(s);
        else
            return super.getString(s);
    }
}

