/**
 * Title:        Comedia Beans
 * Copyright:    Copyright (c) 2001
 * Company:      Capella Development Group
 * @author Sergey Seroukhov
 * @version 1.0
 */

package org.comedia.db.view;

import java.awt.*;
import javax.swing.*;
import java.awt.event.*;
import org.comedia.ui.*;

/**
 * Implements a link between two tables on database schemas.
 * <p>
 * <img src="CTableLink.gif">
 * <p>
 * Usage examples:
 * <pre>
 * CTableBox table1 = new CTableBox();
 * CTableBox table2 = new CTableBox();
 *
 * CTableLink link = new CTableLink();
 * link.setLeftTable(table1);
 * link.setLeftLinkType(CTableLink.NORMAL_LINK);
 * link.setLeftRelation(CTableLink.ONE_RELATION);
 * link.setLeftFieldIndex(0);
 * link.setRightTable(table2);
 * link.setRightLinkType(CTableLink.JOIN_LINK);
 * link.setRightRelation(CTableLink.MANY_RELATION);
 * link.setRightFieldIndex(5);
 * link.setLocation(150, 150);
 * link.setSize(100, 100);
 * </pre>
 */
public class CTableLink extends JComponent implements FocusListener,
  MouseListener {

  /**
   * The normal link type.
   */
  public final static int NORMAL_LINK = 0;

  /**
   * The join link type.
   */
  public final static int JOIN_LINK = 1;

  /**
   * The unknown relation.
   */
  public final static int UNKNOWN_RELATION = 0;

  /**
   * The one relation.
   */
  public final static int ONE_RELATION = 1;

  /**
   * The many relation.
   */
  public final static int MANY_RELATION = 2;

  /**
   * The left table box.
   */
  private CTableBox leftTable = null;

  /**
   * The right table box.
   */
  private CTableBox rightTable = null;

  /**
   * The left table index field.
   */
  private int leftFieldIndex = -1;

  /**
   * The right table index field.
   */
  private int rightFieldIndex = -1;

  /**
   * The left link type.
   */
  private int leftLinkType = NORMAL_LINK;

  /**
   * The right link type.
   */
  private int rightLinkType = NORMAL_LINK;

  /**
   * The left relation type.
   */
  private int leftRelation = UNKNOWN_RELATION;

  /**
   * The right relation type.
   */
  private int rightRelation = UNKNOWN_RELATION;

  /**
   * The draw points.
   */
  private Point leftStart = new Point();
  private Point leftEnd = new Point();
  private Point rightStart = new Point();
  private Point rightEnd = new Point();

  /**
   * The draw flag.
   */
  private boolean draw = false;

  /**
   * The updated flag.
   */
  private boolean updated = false;

  /**
   * Constructs this link with default properties.
   */
  public CTableLink() {
    this.setSize(65, 65);
    this.setForeground(Color.gray);
    this.setOpaque(false);
    this.setRequestFocusEnabled(true);
    this.addFocusListener(this);
    this.addMouseListener(this);
  }

  /**
   * Checks is focus of the links is traversable.
   * @result <code>TRUE</code> always.
   */
  public boolean isFocusTraversable() {
    return true;
  }

  /**
   * Checks is focus of the links is managable.
   * @result <code>TRUE</code> always.
   */
  public boolean isManagingFocus() {
    return true;
  }

  /**
   * Checks is link updated.
   * @result <code>TRUE</code> if link updated and <code>FALSE</code> otherwise;
   */
  public boolean isUpdated() {
    return updated;
  }

  /**
   * Gets the left related table box.
   * @result the left related table box.
   */
  public CTableBox getLeftTable() {
    return this.leftTable;
  }

  /**
   * Sets a new left related table box.
   * @param leftTable a new left related table box.
   */
  public void setLeftTable(CTableBox leftTable) {
    if (this.leftTable != leftTable) {
      if (this.leftTable != null)
        this.leftTable.removeLink(this);
      this.leftTable = leftTable;
      if (leftTable != null)
        leftTable.addLink(this);
      updated = false;
      updateShape();
    }
  }

  /**
   * Gets the right related table box.
   * @result the right related table box.
   */
  public CTableBox getRightTable() {
    return this.rightTable;
  }

  /**
   * Sets a new right related table box.
   * @param rightTable a new right related table box.
   */
  public void setRightTable(CTableBox rightTable) {
    if (this.rightTable != rightTable) {
      if (this.rightTable != null)
        this.rightTable.removeLink(this);
      this.rightTable = rightTable;
      if (rightTable != null)
        rightTable.addLink(this);
      updated = false;
      updateShape();
    }
  }

  /**
   * Gets the related field index of left table box.
   * @result the related field index of left table box.
   */
  public int getLeftFieldIndex() {
    return this.leftFieldIndex;
  }

  /**
   * Sets a new related field index of left table box.
   * @param leftFieldIndex a new related field index of left table box.
   */
  public void setLeftFieldIndex(int leftFieldIndex) {
    if (this.leftFieldIndex != leftFieldIndex) {
      this.leftFieldIndex = leftFieldIndex;
      updateShape();
    }
  }

  /**
   * Gets the related field index of right table box.
   * @result the related field index of right table box.
   */
  public int getRightFieldIndex() {
    return this.rightFieldIndex;
  }

  /**
   * Sets a new related field index of right table box.
   * @param rightFieldIndex a new related field index of right table box.
   */
  public void setRightFieldIndex(int rightFieldIndex) {
    if (this.rightFieldIndex != rightFieldIndex) {
      this.rightFieldIndex = rightFieldIndex;
      updateShape();
    }
  }

  /**
   * Gets the related link type of left table box.
   * @result the related link type of left table box.
   */
  public int getLeftLinkType() {
    return this.leftLinkType;
  }

  /**
   * Sets a new related link type of left table box.
   * @param leftLinkType a new related link type of left table box.
   */
  public void setLeftLinkType(int leftLinkType) {
    if (this.leftLinkType != leftLinkType) {
      this.leftLinkType = leftLinkType;
      repaint();
    }
  }

  /**
   * Gets the related link type of right table box.
   * @result the related link type of right table box.
   */
  public int getRightLinkType() {
    return this.rightLinkType;
  }

  /**
   * Sets a new related link type of right table box.
   * @param rightLinkType a new related link type of right table box.
   */
  public void setRightLinkType(int rightLinkType) {
    if (this.rightLinkType != rightLinkType) {
      this.rightLinkType = rightLinkType;
      repaint();
    }
  }

  /**
   * Gets the related relation for left table box.
   * @result the related relation for left table box.
   */
  public int getLeftRelation() {
    return this.leftRelation;
  }

  /**
   * Sets a new related relation for left table box.
   * @param leftRelation a new related relation for left table box.
   */
  public void setLeftRelation(int leftRelation) {
    if (this.leftRelation != leftRelation) {
      this.leftRelation = leftRelation;
      repaint();
    }
  }

  /**
   * Gets the related relation for right table box.
   * @result the related relation for right table box.
   */
  public int getRightRelation() {
    return this.rightRelation;
  }

  /**
   * Sets selected or unselected colors for this table link.
   * @param selected <code>TRUE</code> to set selected color
   *   and <code>FALSE</code> otherwise.
   */
  private void setSelectedColor(boolean selected) {
    if (selected)
      this.setForeground(Color.black);
    else
      this.setForeground(Color.gray);
  }

  /**
   * Performs event when this table box gets a focus.
   * @param e an object which described occured event.
   */
  public void focusGained(FocusEvent e) {
    setSelectedColor(true);
  }

  /**
   * Performs event when this table box lost a focus.
   * @param e an object which described occured event.
   */
  public void focusLost(FocusEvent e) {
    setSelectedColor(false);
  }

  /**
   * Sets a new related relation for right table box.
   * @param rightRelation a new related relation for right table box.
   */
  public void setRightRelation(int rightRelation) {
    if (this.rightRelation != rightRelation) {
      this.rightRelation = rightRelation;
      repaint();
    }
  }

  /**
   * Paints the contents of this link.
   */
  public void paint(Graphics g) {
    super.paint(g);

    if (draw) {
      g.setColor(this.getForeground());
      g.drawLine(leftStart.x, leftStart.y - 1, leftEnd.x, leftEnd.y - 1);
      g.drawLine(leftStart.x, leftStart.y, leftEnd.x, leftEnd.y);
      g.drawLine(leftStart.x, leftStart.y + 1, leftEnd.x, leftEnd.y + 1);
      g.drawLine(leftEnd.x, leftEnd.y, rightStart.x, rightStart.y);
      g.drawLine(rightStart.x, rightStart.y - 1, rightEnd.x, rightEnd.y - 1);
      g.drawLine(rightStart.x, rightStart.y, rightEnd.x, rightEnd.y);
      g.drawLine(rightStart.x, rightStart.y + 1, rightEnd.x, rightEnd.y + 1);

      if (this.leftLinkType == JOIN_LINK) {
        if (leftStart.x < leftEnd.x) {
          CEncodedIcon icon = CEncodedIcon.SMALL_LEFT_ARROW;
          icon.setColor(this.getForeground());
          icon.paintIcon(this, g, leftStart.x, leftStart.y - 3);
        } else {
          CEncodedIcon icon = CEncodedIcon.SMALL_RIGHT_ARROW;
          icon.setColor(this.getForeground());
          icon.paintIcon(this, g, leftEnd.x + 11, leftEnd.y - 3);
        }
      }

      if (this.leftRelation == ONE_RELATION) {
        CEncodedIcon icon = CEncodedIcon.ONE_SIGN;
        icon.setColor(this.getForeground());
        icon.paintIcon(this, g, (leftStart.x + leftEnd.x) / 2 - 1, leftStart.y - 9);
      }

      if (this.leftRelation == MANY_RELATION) {
        CEncodedIcon icon = CEncodedIcon.MANY_SIGN;
        icon.setColor(this.getForeground());
        icon.paintIcon(this, g, (leftStart.x + leftEnd.x) / 2 - 6, leftStart.y - 9);
      }

      if (this.rightLinkType == JOIN_LINK) {
        if (rightStart.x > rightEnd.x) {
          CEncodedIcon icon = CEncodedIcon.SMALL_LEFT_ARROW;
          icon.setColor(this.getForeground());
          icon.paintIcon(this, g, rightEnd.x, rightEnd.y - 3);
        } else {
          CEncodedIcon icon = CEncodedIcon.SMALL_RIGHT_ARROW;
          icon.setColor(this.getForeground());
          icon.paintIcon(this, g, rightStart.x + 11, rightStart.y - 3);
        }
      }

      if (this.rightRelation == ONE_RELATION) {
        CEncodedIcon icon = CEncodedIcon.ONE_SIGN;
        icon.setColor(this.getForeground());
        icon.paintIcon(this, g, (rightStart.x + rightEnd.x) / 2 - 1, rightStart.y - 9);
      }

      if (this.rightRelation == MANY_RELATION) {
        CEncodedIcon icon = CEncodedIcon.MANY_SIGN;
        icon.setColor(this.getForeground());
        icon.paintIcon(this, g, (rightStart.x + rightEnd.x) / 2 - 6, rightStart.y - 9);
      }
    }
  }

  /**
   * The side line size.
   */
  private final static int LINE_SIZE = 17;

  /**
   * Presents the rectangle class.
   */
  private class Rect {
    public int left, right;
    public int top, bottom;

    public Rect(int left, int top, int right, int bottom) {
      this.left = left;
      this.right = right;
      this.top = top;
      this.bottom = bottom;
    }
  }

  /**
   * Gets a bound rect of the control in parents coordinates.
   * @param control a control.
   * @result a bounded rectangle.
   */
  private Rect getBoundRect(Component control) {
    Point cp = control.getLocationOnScreen();
    if (this.getParent() != null) {
      Point pp = this.getParent().getLocationOnScreen();
      cp.x -= pp.x;
      cp.y -= pp.y;
    }
    return new Rect(cp.x - LINE_SIZE, cp.y,
      cp.x + control.getWidth() + LINE_SIZE, cp.y + control.getHeight());
  }

  /**
   * Converts coordinates from parent to own coordinates.
   * @param point a point in parents coordinates.
   * @result a point in own coordinates.
   */
  private Point convertParentToOwn(Point point) {
    if (this.getParent() != null) {
      point.x -= this.getLocationOnScreen().x
        - this.getParent().getLocationOnScreen().x;
      point.y -= this.getLocationOnScreen().y;
    }
    return point;
  }

  /**
   * Updates the shape of this link.
   */
  public void updateShape() {
    if (!this.isShowing()) return;

    if (this.leftTable != null && this.rightTable != null) {
      Rect rect1 = getBoundRect(this.leftTable);
      Rect rect2 = getBoundRect(this.rightTable);
      Rect rect = new Rect(0, 0, 0, 0);

      if (rect1.left <= rect2.left) {
        rect.left = Math.min(rect1.right - LINE_SIZE, rect2.left);
        rect.right = Math.max(rect1.right, rect2.left + LINE_SIZE);
      } else {
        rect.left = Math.min(rect1.left, rect2.right - LINE_SIZE);
        rect.right = Math.max(rect1.left + LINE_SIZE, rect2.right);
      }

      rect.left = rect.left;
      rect.right = rect.right;
      rect.top = Math.min(rect1.top, rect2.top);
      rect.bottom = Math.max(rect1.bottom, rect2.bottom);

      this.setLocation(rect.left, rect.top);
      this.setSize(rect.right - rect.left, rect.bottom - rect.top);

      leftStart.y = leftTable.getItemPos(leftFieldIndex);
      leftEnd.y = leftStart.y;
      rightStart.y = rightTable.getItemPos(rightFieldIndex);
      rightEnd.y = rightStart.y;

      if (rect1.left <= rect2.left) {
        leftStart.x = rect1.right - LINE_SIZE;
        leftEnd.x = rect1.right;
        rightStart.x = rect2.left;
        rightEnd.x = rect2.left + LINE_SIZE;
      } else {
        leftStart.x = rect1.left + LINE_SIZE;
        leftEnd.x = rect1.left;
        rightStart.x = rect2.right;
        rightEnd.x = rect2.right - LINE_SIZE;
      }

      leftStart = convertParentToOwn(leftStart);
      leftEnd = convertParentToOwn(leftEnd);
      rightStart = convertParentToOwn(rightStart);
      rightEnd = convertParentToOwn(rightEnd);

      draw = true;
    } else
      draw = false;
    updated = true;
    this.repaint();
  }

  /**
   * The maximum distance from the line.
   */
  private final static int MAX_DIST = 3;

  /**
   * Checks is number lays between two other numbers.
   * @param start the start number of the interval.
   * @param end the end number of the interval.
   * @param value the value to check.
   * @result <code>TRUE</code> if value in inside interval
   *   and <code>FALSE</code> otherwise.
   */
  private boolean checkInterval(double start, double end, double value) {
    if (start <= end)
      return (start <= value && end >= value);
    else
      return (end <= value && start >= value);
  }

  /**
   * Checks is point lays on the line startPoint-endPoint.
   * @param start the start point of the line.
   * @param end the end point of the line.
   * @param point the check point.
   * @result <code>TRUE</code> if point lays on the line
   *   and <code>FALSE</code> otherwise.
   */
  private boolean checkLine(Point start, Point end, Point point) {
    double x1 = start.x;
    double y1 = start.y;
    double x2 = end.x;
    double y2 = end.y;
    double x3 = point.x;
    double y3 = point.y;

    double a1 = y2 - y1;
    double b1 = -(x2 - x1);
    double c1 = a1 * x2 + b1 * y2;
    double a2 = x2 - x1;
    double b2 = y2 - y1;
    double c2 = a2 * x3 + b2 * y3;

    double x = (c2 * b1 - c1 * b2) / (a2 * b1 - a1 * b2);
    double y = (b1 != 0) ? (c1 - a1 * x) / b1 : y3;

    double d = Math.sqrt(Math.pow(x3 - x, 2) + Math.pow(y3 - y, 2));
    return (d <= MAX_DIST && checkInterval(x1, x2, x) && checkInterval(y1, y2, y));
  }

  /**
   * Checks is the point is on the control.
   * @param point the point to check.
   * @result <code>TRUE</code> if point is on the control
   *   and <code>FALSE</code> otherwise.
   */
  private boolean checkOver(Point point) {
    return checkLine(leftStart, leftEnd, point) ||
      checkLine(leftEnd, rightStart, point) ||
      checkLine(rightStart, rightEnd, point);
  }

  /**
   * Checks whether this component "contains" the specified point,
   * where <code>x</code> and <code>y</code> are defined to be
   * relative to the coordinate system of this component.
   * @param     x   the <i>x</i> coordinate of the point.
   * @param     y   the <i>y</i> coordinate of the point.
   * @see       java.awt.Component#getComponentAt(int, int)
   * @since     JDK1.1
   */
  public boolean contains(int x, int y) {
    return checkOver(new Point(x, y));
  }

  /**
   * Performs event when mouse is clicked on the component.
   * @param e an object which described occured event.
   */
  public void mouseClicked(MouseEvent e) {
  }

  /**
   * Performs event when mouse is pressed on the component.
   * @param e an object which described occured event.
   */
  public void mousePressed(MouseEvent e) {
    this.requestFocus();
  }

  /**
   * Performs event when mouse is released on the component.
   * @param e an object which described occured event.
   */
  public void mouseReleased(MouseEvent e) {
  }

  /**
   * Performs event when mouse is entered to the component.
   * @param e an object which described occured event.
   */
  public void mouseEntered(MouseEvent e) {
  }

  /**
   * Performs event when mouse is exited from the component.
   * @param e an object which described occured event.
   */
  public void mouseExited(MouseEvent e) {
  }

}
