// copyright 2001-2002 by The Mind Electric

package electric.xml;

import java.io.*;
import electric.util.*;

/**
 * <tt>DocType</tt> represents an XML DOCTYPE declaration.
 *
 * @author <a href="http://www.themindelectric.com">The Mind Electric</a>
 */

final public class DocType extends Parent implements org.w3c.dom.DocumentType
  {
  static final String START = "<!DOCTYPE";
  static final String STOP = ">";
  static final String SYSTEM = "SYSTEM";
  static final String PUBLIC = "PUBLIC";

  private String name;
  private String systemId;
  private String publicId;

  // ********** CONSTRUCTION ************************************************

  /**
   * Construct a DocType with the specified name.
   * @param name The name.
   */
  public DocType( String name )
    {
    this.name = name;
    }

  /**
   * Construct a copy of the specified DocType.
   * @param docType The DocType to copy.
   */
  public DocType( DocType docType )
    {
    super( docType );
    this.name = docType.name;
    this.systemId = docType.systemId;
    this.publicId = docType.publicId;
    }

  /**
   * Construct a DocType from the specified lexical analyzer.
   * @param lex The lexical analyzer.
   * @param parent The parent of this DocType.
   * @throws IOException If an error occurs during parsing.
   */
  DocType( Lex lex, Parent parent )
    throws IOException
    {
    parent.addChild( this );
    lex.readChar( '<' );
    lex.readToken( START.substring( 1 ) );
    name = lex.readToDelimiter( "[>" );

    if( name.equals( "[" ) || name.equals( ">" ) )
      throw new IOException( "DOCTYPE is missing a name" );

    String next = lex.readToken();

    if( next.equals( SYSTEM ) )
      {
      systemId = lex.readToDelimiter( "[>", Lex.SKIP_WS | Lex.QUOTES | Lex.STRIP );
      next = lex.readToDelimiter( "[>" );
      }
    else if( next.equals( PUBLIC ) )
      {
      publicId = lex.readToDelimiter( "[>", Lex.SKIP_WS | Lex.QUOTES | Lex.STRIP );
      systemId = lex.readToDelimiter( "[>", Lex.SKIP_WS | Lex.QUOTES | Lex.STRIP );
      next = lex.readToDelimiter( "[>" );
      }

    if( next.equals( "[" ) )
      {
      while( true )
        {
        lex.skipWhitespace();
        int[] chars = new int[ 2 ];
        lex.peek( chars );
        int ch1 = chars[ 0 ];
        int ch2 = chars[ 1 ];

        if( ch1 == ']' )
          break;
        else if( ch1 == -1 ) // eof
          throw new IOException( "could not find matching ']' in DOCTYPE" );
        else if( ch1 == '%' )
          lex.readToPattern( ";", Lex.CONSUME | Lex.INCLUDE );
        else if( ch2 == '!' && lex.peekString( AttlistDecl.START ) )
          new AttlistDecl( lex, this );
        else if( ch2 == '!' && lex.peekString( ElementDecl.START ) )
          new ElementDecl( lex, this );
        else if( ch2 == '!' && lex.peekString( EntityDecl.START ) )
          new EntityDecl( lex, this );
        else if( ch2 == '!' && lex.peekString( NotationDecl.START ) )
          new NotationDecl( lex, this );
        else if( ch2 == '!' && lex.peekString( Comment.START ) )
          new Comment( lex, this );
        else if( ch2 == '?' )
          new Instruction( lex, this );
        else
          throw new IOException( "illegal entry in DOCTYPE" );
        }

      next = lex.readToken(); // consume "]"
      next = lex.readToken();
      }

    if( !next.equals( ">" ) )
      throw new IOException( "could not find matching '>' in DOCTYPE" );
    }

  // ********** CLONING *****************************************************

  /**
   * Return a clone of this DocType.
   */
  public Object clone()
    {
    return new DocType( this );
    }

  // ********** NAME ********************************************************

  /**
   * Return my name.
   */
  public String getName()
    {
    return name;
    }

  /**
   * Set my name.
   * @param name The new name value.
   */
  public void setName( String name )
    {
    this.name = name;
    }

  // ********** NAME ********************************************************

  /**
   * Return my system id.
   */
  public String getSystemId()
    {
    return systemId;
    }

  /**
   * Set my system id.
   * @param systemId The new system id value.
   */
  public void setSystemId( String systemId )
    {
    this.systemId = systemId;
    }

  /**
   * Return my public id.
   */
  public String getPublicId()
    {
    return publicId;
    }

  /**
   * Set my public id.
   * @param publicId The new public id value.
   */
  public void setPublicId( String publicId )
    {
    this.publicId = publicId;
    }

  /**
   * Set my public and system ids.
   * @param publicId The new public id value.
   * @param systemId The new system id value.
   */
  public void setIds( String publicId, String systemId )
    {
    this.publicId = publicId;
    this.systemId = systemId;
    }

  // ********** WRITING *****************************************************

  /**
   * Write myself to the specified writer starting at the specified indent level.
   * If the indent level is -1, no indentation will occur, otherwise the indent
   * level increases by two at each child node.
   * @param writer The nodeWriter.
   * @throws IOException If an I/O exception occurs.
   */
  public void write( NodeWriter writer )
    throws IOException
    {
    writer.writeIndent();
    writer.write( START );
    writer.write( ' ' );
    writer.write( name );

    if( publicId != null )
      {
      writer.write( ' ' );
      writer.write( PUBLIC );
      writer.write( " '" );
      writer.write( publicId );
      writer.write( "' '" );
      writer.write( systemId );
      writer.write( "'" );
      }
    else if( systemId != null )
      {
      writer.write( ' ' );
      writer.write( SYSTEM );
      writer.write( " '" );
      writer.write( systemId );
      writer.write( "'" );
      }

    if( !children.isEmpty() )
      {
      writer.writeEOL();
      writer.increaseIndent();
      writer.writeIndent();
      writer.write( '[' );

      for( Node node = children.first; node != null; node = node.next )
        {
        writer.writeEOL();
        writer.write( node );
        }

      writer.writeEOL();
      writer.writeIndent();
      writer.decreaseIndent();
      writer.write( ']' );
      writer.writeEOL();
      }

    writer.write( STOP );
    }

  // ********** DOM *********************************************************

  /**
   * Return DOCUMENT_TYPE_NODE.
   */
  public short getNodeType()
    {
    return DOCUMENT_TYPE_NODE;
    }

  /**
   * Return the name.
   */
  public String getNodeName()
    {
    return name;
    }

  /**
   * Not implemented yet - return null.
   */
  public String getInternalSubset()
    {
    return null;
    }

  /**
   * Not implemented yet - return null.
   */
  public org.w3c.dom.NamedNodeMap getEntities()
    {
    return null;
    }

  /**
   * Not implemented yet - return null.
   */
  public org.w3c.dom.NamedNodeMap getNotations()
    {
    return null;
    }
  }