// copyright 2001-2002 by The Mind Electric

package electric.xml;

import java.io.*;
import electric.util.*;

/**
 * <tt>Text</tt> represents a piece of XML text.
 *
 * @author <a href="http://www.themindelectric.com">The Mind Electric</a>
 */

public class Text extends CharacterData implements org.w3c.dom.Text
  {
  private static final char[] specials = new char[]{ '&', '<', '>', '\'', '"' };
  private static final String[] substitutes = new String[]{ "&amp;", "&lt;", "&gt;", "&apos;", "&quot;" };

  protected boolean raw; // true if subsitution should be disabled when writing

  // ********** CONSTRUCTION ************************************************

  /**
   * Construct a Text with the specified string.
   * @param string The comment string.
   */
  public Text( String string )
    {
    super( string );
    }

  /**
   * Construct a copy of the specified Text.
   * @param The Text to copy.
   */
  public Text( Text text )
    {
    super( text.string );
    }

  /**
   * Construct a Text from the specified lexical analyzer.
   * @param lex The lexical analyzer.
   * @param whitespace Any preceding whitespace.
   * @param parent The parent of this Text.
   * @throws IOException If an error occurs during parsing.
   */
  Text( Lex lex, StringBuffer whitespace, Element parent )
    throws IOException
    {
    super( parent );
    String token = lex.readToPattern( "<", Lex.EOF_OK | Lex.HTML );
    string = (whitespace == null ? token : whitespace.append( token ).toString());
    }

  /**
   * Construct a Text with the specified parent.
   * @param parent The parent.
   */
  Text( Parent parent )
    {
    super( parent );
    }

  // ********** CLONING *****************************************************

  /**
   * Return a clone of this Text.
   */
  public Object clone()
    {
    return new Text( this );
    }

  // ********** STRING ******************************************************

  /**
   * Return my string.
   */
  public String getString()
    {
    return string;
    }

  /**
   * Set my string.
   * @param string The new string value.
   */
  public void setString( String string )
    {
    this.string = string;
    }

  // ********** RAW MODE ****************************************************

  /**
   * Set my raw mode.
   * If true, subsitutions are disabled when I am written.
   * @param flag The new raw value.
   */
  public void setRaw( boolean flag )
    {
    raw = flag;
    }

  /**
   * Return my raw mode.
   */
  public boolean getRaw()
    {
    return raw;
    }

  // ********** WRITING *****************************************************

  /**
   * Write myself to the specified writer.
   * @param writer The nodeWriter.
   * @throws IOException If an I/O exception occurs.
   */
  public void write( NodeWriter writer )
    throws IOException
    {
    writer.writeIndent();

    if( raw )
      writer.write( string );
    else
      writeWithSubstitution( writer, string );
    }

  // ********** WRITING WITH SUBSTITUTION ***********************************

  /**
   * Return the index of the specified character into the special character
   * array, or -1 if it is not special.
   */
  private static int getSpecialIndex( char ch )
    {
    for( int i = 0; i < specials.length; i++ )
      if( specials[ i ] == ch )
        return i;

    return -1;
    }

  /**
   * Write the specified string to the specified Writer, substituting
   * the special characters &, < and >, ' and "
   * for their &amp;, &lt;, &gt;, &apos; and &quot; equivalents.
   * @param writer The nodeWriter.
   * @param string The String.
   * @throws IOException If an I/O exception occurs.
   */
  static void writeWithSubstitution( NodeWriter writer, String string )
    throws IOException
    {
    char[] chars = string.toCharArray();
    int start = 0;

    for( int i = 0; i < chars.length; i++ )
      {
      int index = getSpecialIndex( chars[ i ] );

      if( index >= 0 )
        {
        if( start < i )
          writer.write( chars, start, i - start );

        writer.write( substitutes[ index ] );
        start = i + 1;
        }
      }

    if( start < chars.length )
      writer.write( chars, start, chars.length - start );
    }

  // ********** DOM *********************************************************

  /**
   * Return TEXT_NODE.
   */
  public short getNodeType()
    {
    return TEXT_NODE;
    }

  /**
   * Return "#text".
   */
  public String getNodeName()
    {
    return "#text";
    }

  /**
   * Breaks this node into two nodes at the specified offset, keeping both
   * in the tree as siblings. After being split, this node will contain all
   * the content up to the offset point. A new node of the same type, which
   * contains all the content at and after the offset point, is returned.
   * If the original node had a parent node, the new node is inserted as the
   * next sibling of the original node. When the offset is equal to the length
   * of this node, the new node has no data.
   * @param offset The offset.
   */
  public org.w3c.dom.Text splitText( int offset )
    {
    String tail = string.substring( offset );
    string = string.substring( 0, offset );
    Text text = new Text( tail );
    setNextSiblingChild( text );
    return text;
    }
  }