/*
 * Copyright (C) 2003-2011 Karl Tauber <karl at jformdesigner dot com>
 * All Rights Reserved
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *  o Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 *  o Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 *  o Neither the name of JFormDesigner or Karl Tauber nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package com.jformdesigner.model;

import java.util.Vector;
import javax.swing.table.DefaultTableModel;

/**
 * Used for TableModels in JFormDesigner forms.
 * Extends <code>DefaultTableModel</code> by the possibility to specify
 * column types and column editable flags.
 *
 * @author Karl Tauber
 */
public class SwingTableModel
	extends DefaultTableModel
{
	private Vector<Class<?>> columnTypes;
	private Vector<Boolean> columnEditables;
	private Vector<SwingTableColumn> columnInfos;

	/**
	 * Constructs a default <code>SwingTableModel</code> that has zero
	 * columns and rows.
	 */
	public SwingTableModel() {
		init();
	}

	/**
	 * Constructs a <code>SwingTableModel</code> and initializes it with the
	 * specified data and column names.
	 * <p>
	 * See <code>javax.swing.table.DefaultTableModel</code> for details.
	 */
	public SwingTableModel( Vector<Vector<Object>> data, Vector<String> columnNames ) {
		this( data, columnNames, null, null );
	}

	/**
	 * Constructs a <code>SwingTableModel</code> and initializes it with the
	 * specified data, column names, column types and column editable flags.
	 * <p>
	 * See <code>javax.swing.table.DefaultTableModel</code> for details.
	 */
	public SwingTableModel( Vector<Vector<Object>> data, Vector<String> columnNames,
		Vector<Class<?>> columnTypes, Vector<Boolean> columnEditables )
	{
		this( data, columnNames, columnTypes, columnEditables, null );
	}

	/**
	 * Constructs a <code>SwingTableModel</code> and initializes it with the
	 * specified data, column names, column types, column editable flags
	 * and additional column infos.
	 * <p>
	 * See <code>javax.swing.table.DefaultTableModel</code> for details.
	 *
	 * @since 3.0
	 */
	public SwingTableModel( Vector<Vector<Object>> data, Vector<String> columnNames,
		Vector<Class<?>> columnTypes, Vector<Boolean> columnEditables, Vector<SwingTableColumn> columnInfos )
	{
		super( data, columnNames );

		this.columnTypes = columnTypes;
		this.columnEditables = columnEditables;
		this.columnInfos = columnInfos;

		init();
	}

	/**
	 * Constructs a <code>SwingTableModel</code> and initializes it with the
	 * specified data and column names.
	 * <p>
	 * See <code>javax.swing.table.DefaultTableModel</code> for details.
	 */
	public SwingTableModel( Object[][] data, Object[] columnNames ) {
		super( data, columnNames );
		init();
	}

	/**
	 * Copy constructor.
	 * Copies vectors, but not the real data.
	 */
	@SuppressWarnings("unchecked")
	public SwingTableModel( SwingTableModel model ) {
		int rowCount = model.dataVector.size();
		dataVector = new Vector<Vector<Object>>( rowCount );
		for( int i = 0; i < rowCount; i++ )
			dataVector.add( ((Vector<?>)model.dataVector.get(i)).clone() );

		columnIdentifiers = (Vector<String>) model.columnIdentifiers.clone();
		columnTypes = (Vector<Class<?>>) model.columnTypes.clone();
		columnEditables = (Vector<Boolean>) model.columnEditables.clone();
		columnInfos = (Vector<SwingTableColumn>) model.columnInfos.clone();
	}

	private void init() {
		if( columnTypes == null ) {
			columnTypes = new Vector<Class<?>>();
			columnTypes.setSize( getColumnCount() );
		}
		if( columnEditables == null ) {
			columnEditables = new Vector<Boolean>();
			columnEditables.setSize( getColumnCount() );
		}
		if( columnInfos == null ) {
			columnInfos = new Vector<SwingTableColumn>();
			columnInfos.setSize( getColumnCount() );
		}
	}

	/**
	 * Returns the <code>Vector</code> that contains the table's column names.
	 * Each element in the vector must be a <code>java.lang.String</code>.
	 */
	@SuppressWarnings("unchecked")
	public Vector<String> getColumnNames() {
		return columnIdentifiers;
	}

	/**
	 * Returns the <code>Vector</code> that contains the table's column types.
	 * Each element in the vector must be a <code>java.lang.Class</code>.
	 */
	public Vector<Class<?>> getColumnTypes() {
		return columnTypes;
	}

	/**
	 * Returns the <code>Vector</code> that contains the table's column editable flags.
	 * Each element in the vector must be a <code>java.lang.Boolean</code>.
	 */
	public Vector<Boolean> getColumnEditables() {
		return columnEditables;
	}

	/**
	 * Returns the <code>Vector</code> that contains the table's column infos.
	 * Each element in the vector must be a {@link SwingTableColumn}.
	 *
	 * @since 3.0
	 */
	public Vector<SwingTableColumn> getColumnInfos() {
		return columnInfos;
	}

	/**
	 * Sets the number of columns in the model.
	 * <p>
	 * See <code>javax.swing.table.DefaultTableModel.setColumnCount()</code>
	 * for details.
	 */
	@Override
	public void setColumnCount( int columnCount ) {
		columnTypes.setSize( columnCount );
		columnEditables.setSize( columnCount );
		columnInfos.setSize( columnCount );
		super.setColumnCount( columnCount );
	}

	/**
	 * Returns <code>true</code> if the cells in the specified column are editable.
	 */
	public boolean isColumnEditable( int column ) {
		Boolean editable = columnEditables.get( column );
		return (editable != null) ? editable.booleanValue() : true;
	}

	/**
	 * Returns the column info for the given column index.
	 *
	 * @since 3.0
	 */
	public SwingTableColumn getColumnInfo( int column ) {
		return columnInfos.get( column );
	}

	//---- TableModel overrides ----

	/**
	 * Returns the type of the data in the specified column.
	 * <p>
	 * See <code>javax.swing.table.TableModel.getColumnClass()</code> for details.
	 *
	 * @see #getColumnTypes()
	 */
	@Override
	public Class<?> getColumnClass( int columnIndex ) {
		Class<?> cls = columnTypes.get( columnIndex );
		return (cls != null) ? cls : Object.class;
	}

	/**
	 * Returns <code>true</code> if the specified column is editable.
	 * <p>
	 * See <code>javax.swing.table.TableModel.isCellEditable()</code> for details.
	 *
	 * @see #isColumnEditable
	 * @see #getColumnEditables
	 */
	@Override
	public boolean isCellEditable( int row, int column ) {
		return isColumnEditable( column );
	}
}
