/*
 * Copyright (C) 2003-2011 Karl Tauber <karl at jformdesigner dot com>
 * All Rights Reserved
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *  o Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 *  o Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 *  o Neither the name of JFormDesigner or Karl Tauber nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package com.jformdesigner.runtime;

import java.awt.*;
import java.beans.*;
import java.io.*;
import java.lang.reflect.*;
import java.net.URL;
import java.util.*;
import javax.swing.*;
import javax.swing.border.*;
import javax.swing.table.DefaultTableModel;
import com.jgoodies.forms.factories.Borders;
import com.jgoodies.forms.layout.CellConstraints;
import com.jformdesigner.model.*;

/**
 * @author Karl Tauber
 */
class FormXMLPersistence
{
	/** Maps built-in classes to persistence delegates (String,PersistenceDelegate). */
	private static HashMap<String, PersistenceDelegate> class2persistenceDelegateMap;

	private static Class<?> enumClass;
	private static Method enumNameMethod;
	private static Method enumGetDeclaringClassMethod;
	private static PersistenceDelegate enumPersistenceDelegate;

	/**
	 * Maps configured classes to persistence delegates (Class,PersistenceDelegate).
	 * Using a weak hash map to allow unloading of classes.
	 */
	private static WeakHashMap<Class<?>, PersistenceDelegate> confClass2persistenceDelegateMap;

	private static Properties persistenceDelegatesConf0;

	private static boolean persistenceDelegatesConfLocated;
	private static File persistenceDelegatesConfFile;
	private static Properties persistenceDelegatesConf;
	private static long persistenceDelegatesConfLastModified;

	/**
	 * Maps property value classes to persistence delegates (Class,PersistenceDelegate).
	 */
	private final static WeakHashMap<Class<?>, PersistenceDelegate> propertyPersistenceDelegatesMap = new WeakHashMap<Class<?>, PersistenceDelegate>();

	static void initialize() {
		if( class2persistenceDelegateMap == null ) {
			class2persistenceDelegateMap = new HashMap<String, PersistenceDelegate>();
			initializePersistenceDelegates( class2persistenceDelegateMap );
			loadPersistenceDelegatesConf0();
		}
		loadPersistenceDelegatesConf();
	}

	private static void initializePersistenceDelegates( HashMap<String, PersistenceDelegate> map ) {
		map.put( FormObject.class.getName(), new FormObject_PersistenceDelegate() );
		map.put( FormComponent.class.getName(), new FormComponent_PersistenceDelegate() );
		map.put( FormNonVisual.class.getName(), new DefaultPersistenceDelegate(
			new String[] { "className" } ) );
		map.put( FormContainer.class.getName(), new FormContainer_PersistenceDelegate() );
		map.put( FormWindow.class.getName(), new DefaultPersistenceDelegate(
			new String[] { "className", "layout" } ) );
		map.put( FormRoot.class.getName(), new FormRoot_PersistenceDelegate() );
		map.put( FormLayoutManager.class.getName(), new DefaultPersistenceDelegate(
			new String[] { "layoutClass" } ) );
		map.put( FormLayoutConstraints.class.getName(), new DefaultPersistenceDelegate(
			new String[] { "constraintsClass" } ) );
		map.put( FormEvent.class.getName(), new FormEvent_PersistenceDelegate() );
		map.put( FormBindingGroup.class.getName(), new FormBindingGroup_PersistenceDelegate() );
		map.put( FormBinding.class.getName(), new FormBinding_PersistenceDelegate() );
		map.put( FormReference.class.getName(), new DefaultPersistenceDelegate(
			new String[] { "name" } ) );
		map.put( FormMessage.class.getName(), new DefaultPersistenceDelegate(
			new String[] { "baseName", "key" } ) );
		map.put( FormMessageArray.class.getName(), new DefaultPersistenceDelegate(
			new String[] { "baseName", "key", "length" } ) );

		map.put( SwingBorder.class.getName(), new DefaultPersistenceDelegate( new String[] { "key" } ) );
		map.put( SwingColor.class.getName(), new DefaultPersistenceDelegate( new String[] { "key" } ) );
		map.put( SwingDerivedFont.class.getName(), new DefaultPersistenceDelegateEx( new String[] { "nameChange", "styleChange", "sizeChange", "absoluteSize" } ) );
		map.put( SwingFont.class.getName(), new DefaultPersistenceDelegate( new String[] { "key" } ) );
		map.put( SwingIcon.class.getName(), new DefaultPersistenceDelegate( new String[] { "type", "name" } ) );
		map.put( SwingTableModel.class.getName(), new DefaultPersistenceDelegate(
			new String[] { "dataVector", "columnNames", "columnTypes", "columnEditables", "columnInfos" } ) );
		map.put( SwingTableColumn.class.getName(), new DefaultPersistenceDelegate(
			new String[] { "values", "preferredWidth", "minWidth", "maxWidth", "resizable" } ) );

		// JGoodies Forms
		map.put( CellConstraints.Alignment.class.getName(), new CellConstraints_Alignment_PersistenceDelegate() );
		map.put( Borders.EmptyBorder.class.getName(), new Borders_EmptyBorder_PersistenceDelegate() );

		//---- optimized to reduce XML file size ----

		map.put( "javax.swing.SpinnerNumberModel", new SpinnerNumberModel_PersistenceDelegate() );

		//---- fix bugs in Sun's Java implementation ----

		// Sun's implementation does not save "roundedCorners"
		map.put( "javax.swing.border.LineBorder", new DefaultPersistenceDelegate(
			new String[] { "lineColor", "thickness", "roundedCorners" } ) );

		map.put( "javax.swing.border.MatteBorder", new MatteBorder_PersistenceDelegate() );
		map.put( "javax.swing.border.TitledBorder", new TitledBorder_PersistenceDelegate() );

		// Sun's implementation uses wrong property names
		map.put( "javax.swing.border.SoftBevelBorder", new DefaultPersistenceDelegate(
			new String[] { "bevelType", "highlightOuterColor", "highlightInnerColor",
						   "shadowOuterColor", "shadowInnerColor" } ) );

		map.put( "java.awt.Color", new Color_PersistenceDelegate() );
		map.put( "java.awt.ComponentOrientation", new Fields_PersistenceDelegate( ComponentOrientation.class ) );
		map.put( "java.awt.GradientPaint", new GradientPaint_PersistenceDelegate() );
		map.put( "javax.swing.KeyStroke", new KeyStroke_PersistenceDelegate() );
		map.put( "javax.swing.DefaultListModel", new DefaultListModel_PersistenceDelegate() );
		map.put( "javax.swing.table.DefaultTableModel", new DefaultTableModel_PersistenceDelegate() );
		map.put( "javax.swing.tree.DefaultTreeModel", new DefaultPersistenceDelegate(
			new String[] { "root" } ) );

		PersistenceDelegate toStringPD = new ToString_PersistenceDelegate();
		map.put( "java.math.BigDecimal", toStringPD );
		map.put( "java.math.BigInteger", toStringPD );

		// Java 5 (and earlier) does not support empty Collections
		Fields_PersistenceDelegate collectionsPD = new Fields_PersistenceDelegate( Collections.class );
		map.put( Collections.EMPTY_LIST.getClass().getName(), collectionsPD );
		map.put( Collections.EMPTY_MAP.getClass().getName(), collectionsPD );
		map.put( Collections.EMPTY_SET.getClass().getName(), collectionsPD );

		map.put( "java.util.Locale", new DefaultPersistenceDelegate(
			new String[] { "language", "country", "variant" } ) );

		// Java 6
		map.put( "javax.swing.border.EmptyBorder", new EmptyBorder_PersistenceDelegate() );

		// Java 7
		map.put( "java.awt.Dimension", new Dimension_PersistenceDelegate() );
		map.put( "java.awt.Point", new Point_PersistenceDelegate() );
		map.put( "java.awt.Rectangle", new Rectangle_PersistenceDelegate() );

		//---- fix bugs in IBM J9 VM ----

		if( "IBM J9 VM".equals( System.getProperty( "java.vm.name" ) ) ) {
			map.put( "java.awt.Font", new DefaultPersistenceDelegate( new String[] { "name", "style", "size" } ) );

			map.put( "javax.swing.border.BevelBorder", new DefaultPersistenceDelegate( new String[] { "bevelType", "highlightOuterColor", "highlightInnerColor", "shadowOuterColor", "shadowInnerColor" } ) );
			map.put( "javax.swing.border.EtchedBorder", new DefaultPersistenceDelegate( new String[]{ "etchType", "highlightColor", "shadowColor" } ) );
		}

		//---- Java 5 enum ----

		try {
			enumClass = Class.forName( "java.lang.Enum" );
			enumNameMethod = enumClass.getMethod( "name", (Class[]) null );
			enumGetDeclaringClassMethod = enumClass.getMethod( "getDeclaringClass", (Class[]) null );
			enumPersistenceDelegate = new Enum_PersistenceDelegate();
		} catch( ClassNotFoundException ex ) {
			// not Java 5
		} catch( SecurityException ex ) {
			// not Java 5
		} catch( NoSuchMethodException e ) {
			// not Java 5
		}
	}

	private static void loadPersistenceDelegatesConf0() {
		// Note: not using relative file name to avoid problems after obfuscation.
		InputStream in = FormXMLPersistence.class.getResourceAsStream(
			"/com/jformdesigner/runtime/persistenceDelegates.conf" );
		persistenceDelegatesConf0 = loadConf( in );
	}

	private static void loadPersistenceDelegatesConf() {
		if( !persistenceDelegatesConfLocated ) {
			persistenceDelegatesConfLocated = true;

			File f = new File( "persistenceDelegates.conf" );
			if( !f.exists() )
				f = new File( "config/persistenceDelegates.conf" );
			if( !f.exists() ) {
				URL url = FormXMLPersistence.class.getResource( "/config/persistenceDelegates.conf" );
				if( url != null && "file".equals( url.getProtocol() ) )
					f = new File( url.getFile() );
			}
			if( f.exists() )
				persistenceDelegatesConfFile = f;
		}

		if( persistenceDelegatesConfFile != null &&
			persistenceDelegatesConfFile.lastModified() > persistenceDelegatesConfLastModified )
		{
			confClass2persistenceDelegateMap = null;
			persistenceDelegatesConf = null;
			persistenceDelegatesConfLastModified = persistenceDelegatesConfFile.lastModified();

			try {
				FileInputStream in = new FileInputStream( persistenceDelegatesConfFile );
				persistenceDelegatesConf = loadConf( in );
			} catch( IOException ex ) {
				// ignore
			}
		}
	}

	private static Properties loadConf( InputStream in ) {
		try {
			Properties properties = new Properties();
			properties.load( in );
			in.close();

			return properties;
		} catch( IOException ex ) {
			return null;
		} finally {
			try {
				in.close();
			} catch( IOException ex1 ) {
				// ignore
			}
		}
	}

	static void initializePropertyPersistenceDelegates( FormComponent component,
														ClassLoader classLoader )
		throws Exception
	{
		String className = component.getClassName();
		if( className != null ) {
			try {
				Class<?> cls = classLoader.loadClass( className );
				BeanInfoEx beanInfo = IntrospectorEx.getBeanInfoEx( cls );
				if( beanInfo != null )
					propertyPersistenceDelegatesMap.putAll( beanInfo.getPersistenceDelegateMap() );
			} catch( ClassNotFoundException ex ) {
				// ignore
			} catch( NoClassDefFoundError ex ) {
				// ignore
			}
		}

		if( component instanceof FormContainer ) {
			FormContainer container = (FormContainer) component;
			if( container.getMenuBar() != null )
				initializePropertyPersistenceDelegates( container.getMenuBar(), classLoader );
			int count = container.getComponentCount();
			for( int i = 0; i < count; i++ )
				initializePropertyPersistenceDelegates( container.getComponent( i ), classLoader );
		}
	}

	static PersistenceDelegate getPersistenceDelegate( Class<?> type, ClassLoader cl ) {
		if( type == null || type == Object.class || type == Class.class )
			return null;

		String typeName = type.getName();

		// look for built-in persistence delegates
		PersistenceDelegate pd = class2persistenceDelegateMap.get( typeName );
		if( pd != null )
			return pd;

		if( enumClass != null && enumClass.isAssignableFrom( type ) )
			return enumPersistenceDelegate;

		// look for persistence delegates specified in BeanDescriptor
		try {
			BeanInfoEx beanInfo = IntrospectorEx.getBeanInfoEx( type );
			pd = (PersistenceDelegate) beanInfo.getBeanDescriptor().getValue( "persistenceDelegate" );
			if( pd != null )
				return pd;
		} catch( IntrospectionException ex ) {
			// ignore
		} catch( NoClassDefFoundError ex ) {
			// ignore
		}

		// look for persistence delegates specified in PropertyDescriptors
		pd = propertyPersistenceDelegatesMap.get( type );
		if( pd != null )
			return pd;

		// is a persistence delegate configured
		String confStr = null;
		if( persistenceDelegatesConf != null )
			confStr = (String) persistenceDelegatesConf.get( typeName );
		if( confStr == null && persistenceDelegatesConf0 != null )
			confStr = (String) persistenceDelegatesConf0.get( typeName );
		if( confStr == null )
			return null;

		// persistence delegate already instantiated
		if( confClass2persistenceDelegateMap != null ) {
			pd = confClass2persistenceDelegateMap.get( type );
			if( pd != null )
				return pd;
		}

		confStr = confStr.trim();
		if( confStr.startsWith( "class:" ) ) {
			String delegateClassName = confStr.substring( "class:".length() ).trim();
			try {
				Class<?> delegateClass = cl.loadClass( delegateClassName );
				pd = (PersistenceDelegate) delegateClass.newInstance();
			} catch( Exception ex ) {
				ex.printStackTrace();
				pd = null;
			}
		} else if( confStr.startsWith( "constructorArguments:" ) ) {
			String constructorPropertyNamesStr = confStr.substring( "constructorArguments:".length() ).trim();
			StringTokenizer st = new StringTokenizer( constructorPropertyNamesStr, ", " );
			int count = st.countTokens();
			String[] constructorPropertyNames = new String[count];
			for( int i = 0; i < constructorPropertyNames.length; i++ )
				constructorPropertyNames[i] = st.nextToken();
			pd = new DefaultPersistenceDelegateEx( constructorPropertyNames );

		} else if( confStr.equals( "toString" ) ) {
			pd = new ToString_PersistenceDelegate();
		}

		// add to cache
		if( confClass2persistenceDelegateMap == null )
			confClass2persistenceDelegateMap = new WeakHashMap<Class<?>, PersistenceDelegate>();
		confClass2persistenceDelegateMap.put( type, pd );

		return pd;
	}

	private static RuntimeException newUnrecognizedInstance( Object oldInstance ) {
		return new RuntimeException( "Unrecognized instance: " + oldInstance );
	}

	//---- inner class FormObject_PersistenceDelegate -------------------------

	private static class FormObject_PersistenceDelegate
		extends DefaultPersistenceDelegate
	{
		@Override
		protected void initialize( Class<?> type, Object oldInstance,
								   Object newInstance, Encoder out )
		{
			super.initialize( type, oldInstance, newInstance, out );

			String[] ignoreProps = null; // must be in alphabetic order
			if( oldInstance instanceof FormBinding ) {
				ignoreProps = new String[] { FormBinding.PROP_SOURCE, FormBinding.PROP_SOURCE_PATH,
					FormBinding.PROP_TARGET, FormBinding.PROP_TARGET_PATH };
			}

			// write properties
			FormObject formObject = (FormObject) oldInstance;
			for( Map.Entry<String, Object> entry : formObject.properties() ) {
				String name = entry.getKey();
				Object value = entry.getValue();

				if( ignoreProps != null && Arrays.binarySearch( ignoreProps, name ) >= 0 )
					continue;

				if( value == FormObject.NULL_VALUE )
					writeNullValue( out );

				if( value != null ) {
					XMLExceptionListener el = (XMLExceptionListener) out.getExceptionListener();
					int oldExceptionCount = el.getExceptionCount();

					out.writeStatement( new Statement( oldInstance, "setProperty",
											new Object[]{ name, value } ) );

					int newExceptionCount = el.getExceptionCount();
					if( newExceptionCount > oldExceptionCount && formObject instanceof FormComponent ) {
						Exception[] exceptions = el.getExceptions();
						int count = newExceptionCount - oldExceptionCount;
						Exception[] setPropExceptions = new Exception[count];
						System.arraycopy( exceptions, oldExceptionCount,
										  setPropExceptions, 0, count );

						el.exceptionThrown( new SetPropertyException(
							((FormComponent)formObject).getName(),
							"Failed to encode value of property \"" + name + "\".",
							setPropExceptions ) );
					}
				}
			}
		}

		private void writeNullValue( Encoder out ) {
			try {
				out.writeExpression( new Expression(
					FormObject.class.getField( "NULL_VALUE" ),
					"get", new Object[] { null } ) );
			} catch( Exception ex ) {
				// ignore
				ex.printStackTrace();
			}
		}
	}

	//---- inner class FormComponent_PersistenceDelegate ----------------------

	private static class FormComponent_PersistenceDelegate
		extends DefaultPersistenceDelegate
	{
		FormComponent_PersistenceDelegate() {
			super( new String[] { "className" } );
		}

		@Override
		protected void initialize( Class<?> type, Object oldInstance,
								   Object newInstance, Encoder out )
		{
			super.initialize( type, oldInstance, newInstance, out );

			// write auxiliary properties
			FormComponent component = (FormComponent) oldInstance;
			if( component.hasAuxiliary() )
				out.writeExpression( new Expression( oldInstance, "auxiliary", null ) );

			// write events
			int count = component.getEventCount();
			for( int i = 0; i < count; i++ ) {
				FormEvent event = component.getEvent( i );
				out.writeStatement( new Statement( oldInstance, "addEvent",
												   new Object[] { event } ) );
			}
		}
	}

	//---- inner class FormContainer_PersistenceDelegate ----------------------

	private static class FormContainer_PersistenceDelegate
		extends DefaultPersistenceDelegate
	{
		FormContainer_PersistenceDelegate() {
			super( new String[] { "className", "layout" } );
		}

		@Override
		protected void initialize( Class<?> type, Object oldInstance,
								   Object newInstance, Encoder out )
		{
			super.initialize( type, oldInstance, newInstance, out );

			// write child components
			FormContainer container = (FormContainer) oldInstance;
			FormLayoutManager layout = container.getLayout();
			int count = container.getComponentCount();
			for( int i = 0; i < count; i++ ) {
				FormComponent comp = container.getComponent( i );
				FormLayoutConstraints constraints = (layout != null)
											? layout.getConstraints( comp )
											: null;
				out.writeStatement( new Statement( oldInstance, "add",
						(constraints != null) ? new Object[] { comp, constraints }
											  : new Object[] { comp } ) );
			}
		}
	}

	//---- inner class FormRoot_PersistenceDelegate ---------------------------

	private static class FormRoot_PersistenceDelegate
		extends DefaultPersistenceDelegate
	{
		@Override
		protected void initialize( Class<?> type, Object oldInstance,
								   Object newInstance, Encoder out )
		{
			super.initialize( type, oldInstance, newInstance, out );

			// write binding groups
			FormRoot root = (FormRoot) oldInstance;
			int count = root.getBindingGroupCount();
			for( int i = 0; i < count; i++ ) {
				FormBindingGroup bindingGroup = root.getBindingGroup( i );
				out.writeStatement( new Statement( oldInstance, "addBindingGroup",
												   new Object[] { bindingGroup } ) );
			}
		}
	}

	//---- inner class FormEvent_PersistenceDelegate --------------------------

	private static class FormEvent_PersistenceDelegate
		extends DefaultPersistenceDelegate
	{
		FormEvent_PersistenceDelegate() {
			super( new String[] { "listener", "listenerMethod", "handler", "passParams" } );
		}

		@Override
		protected Expression instantiate( Object oldInstance, Encoder out ) {
			FormEvent event = (FormEvent) oldInstance;
			if( event.getPropertyName() != null ) {
				return new Expression( oldInstance, oldInstance.getClass(), "new",
					new Object[] { event.getListener(), event.getListenerMethod(),
						event.getHandler(), Boolean.valueOf( event.getPassParams() ),
						event.getPropertyName() } );
			} else
				return super.instantiate( oldInstance, out );
		}
	}

	//---- inner class FormBindingGroup_PersistenceDelegate -------------------

	private static class FormBindingGroup_PersistenceDelegate
		extends DefaultPersistenceDelegate
	{
		FormBindingGroup_PersistenceDelegate() {
			super( new String[] { "bindingGroupClass" } );
		}

		@Override
		protected void initialize( Class<?> type, Object oldInstance,
								   Object newInstance, Encoder out )
		{
			super.initialize( type, oldInstance, newInstance, out );

			// write bindings
			FormBindingGroup bindingGroup = (FormBindingGroup) oldInstance;
			int count = bindingGroup.getBindingCount();
			for( int i = 0; i < count; i++ ) {
				FormBinding binding = bindingGroup.getBinding( i );
				out.writeStatement( new Statement( oldInstance, "addBinding",
												   new Object[] { binding } ) );
			}
		}
	}

	//---- inner class FormBinding_PersistenceDelegate -------------------

	private static class FormBinding_PersistenceDelegate
		extends DefaultPersistenceDelegate
	{
		@Override
		protected Expression instantiate( Object oldInstance, Encoder out ) {
			FormBinding binding = (FormBinding) oldInstance;
			return new Expression( oldInstance, oldInstance.getClass(), "new",
				new Object[] { binding.getSource(), binding.getSourcePath(),
							   binding.getTarget(), binding.getTargetPath() } );
		}
	}

	//---- inner class CellConstraints_Alignment_PersistenceDelegate ----------

	private static class CellConstraints_Alignment_PersistenceDelegate
		extends PersistenceDelegate
	{
		@Override
		protected Expression instantiate( Object oldInstance, Encoder out ) {
			String fieldName = oldInstance.toString().toUpperCase();
			try {
				return new Expression( oldInstance, CellConstraints.class.getField(fieldName),
						"get", new Object[] { null } );
			} catch( Exception ex ) {
				throw newUnrecognizedInstance( oldInstance );
			}
		}
	}

	//---- inner class Borders_EmptyBorder_PersistenceDelegate ----------------

	private static class Borders_EmptyBorder_PersistenceDelegate
		extends Fields_PersistenceDelegate
	{
		public Borders_EmptyBorder_PersistenceDelegate() {
			super( Borders.class );
		}

		@Override
		protected Expression instantiate2( Object oldInstance, Encoder out ) {
			Borders.EmptyBorder border = (Borders.EmptyBorder) oldInstance;
			String encodedSizes = FormSpecCoder.encodeSize( border.top() )
						 + ", " + FormSpecCoder.encodeSize( border.left() )
						 + ", " + FormSpecCoder.encodeSize( border.bottom() )
						 + ", " + FormSpecCoder.encodeSize( border.right() );
			return new Expression( oldInstance, Borders.class, "createEmptyBorder",
									new Object[] { encodedSizes } );
		}
	}

	//---- class EmptyBorder_PersistenceDelegate ------------------------------

	private static class EmptyBorder_PersistenceDelegate
		extends PersistenceDelegate
	{
		@Override
		protected Expression instantiate( Object oldInstance, Encoder out ) {
			EmptyBorder border = (EmptyBorder) oldInstance;
			Insets insets = border.getBorderInsets();
			return new Expression( border, EmptyBorder.class, "new",
				new Object[] { new Integer( insets.top ), new Integer( insets.left ),
							   new Integer( insets.bottom ), new Integer( insets.right ) } );
		}
	}

	//---- class MatteBorder_PersistenceDelegate ------------------------------

	private static class MatteBorder_PersistenceDelegate
		extends DefaultPersistenceDelegate
	{
		@Override
		protected Expression instantiate( Object oldInstance, Encoder out ) {
			MatteBorder border = (MatteBorder) oldInstance;
			Insets insets = RuntimeUtils.getRawMatteBorderInsets( border );
			Color matteColor = border.getMatteColor();
			Icon tileIcon = border.getTileIcon();

			return new Expression( oldInstance, MatteBorder.class, "new",
				new Object[] { new Integer( insets.top ), new Integer( insets.left ),
							   new Integer( insets.bottom ), new Integer( insets.right ),
							   (matteColor != null) ? matteColor : tileIcon } );
		}
	}

	//---- class TitledBorder_PersistenceDelegate -----------------------------

	private static class TitledBorder_PersistenceDelegate
		extends DefaultPersistenceDelegate
	{
		@Override
		protected Expression instantiate( Object oldInstance, Encoder out ) {
			TitledBorder border = (TitledBorder) oldInstance;
			Border tBorder = RuntimeUtils.getRawTitledBorderBorder( border );
			String title = border.getTitle();
			int titleJustification = border.getTitleJustification();
			int titlePosition = RuntimeUtils.getRawTitledBorderPosition( border );
			Font titleFont = RuntimeUtils.getRawTitledBorderFont( border );
			Color titleColor = RuntimeUtils.getRawTitledBorderColor( border );

			// store as few properties as possible because title position has
			// different default values on different Java versions
			Object[] args;
			if( titleFont != null || titleColor != null )
				args = new Object[] { tBorder, title, new Integer( titleJustification ), new Integer( titlePosition ), titleFont, titleColor };
			else if( titleJustification != RuntimeUtils.defaultTitleJustification || titlePosition != RuntimeUtils.defaultTitlePosition )
				args = new Object[] { tBorder, title, new Integer( titleJustification ), new Integer( titlePosition ) };
			else if( tBorder != null )
				args = new Object[] { tBorder, title };
			else
				args = new Object[] { title };
			return new Expression( oldInstance, TitledBorder.class, "new", args );
		}
	}

	//---- inner class Color_PersistenceDelegate ------------------------------

	private static class Color_PersistenceDelegate
		extends DefaultPersistenceDelegate
	{
		Color_PersistenceDelegate() {
			super( new String[] { "red", "green", "blue", "alpha" } );
		}

		@Override
		protected Expression instantiate( Object oldInstance, Encoder out ) {
			try {
				Field[] fields = Color.class.getFields();
				for( int i = 0; i < fields.length; i++ ) {
					if( oldInstance == fields[i].get( null ) )
						return new Expression( oldInstance, fields[i], "get", new Object[] { null } );
				}

				return super.instantiate( oldInstance, out );
			} catch( Exception ex ) {
				throw newUnrecognizedInstance( oldInstance );
			}
		}
	}

	//---- class Dimension_PersistenceDelegate --------------------------------

	private static class Dimension_PersistenceDelegate
		extends PersistenceDelegate
	{
		@Override
		protected boolean mutatesTo( Object oldInstance, Object newInstance ) {
			return oldInstance.equals( newInstance );
		}

		@Override
		protected Expression instantiate( Object oldInstance, Encoder out ) {
			Dimension dimension = (Dimension) oldInstance;
			Object[] args = new Object[] { new Integer( dimension.width ), new Integer( dimension.height ) };
			return new Expression( dimension, dimension.getClass(), "new", args );
		}
	}

	//---- class Point_PersistenceDelegate ------------------------------------

	private static class Point_PersistenceDelegate
		extends PersistenceDelegate
	{
		@Override
		protected boolean mutatesTo( Object oldInstance, Object newInstance ) {
			return oldInstance.equals( newInstance );
		}

		@Override
		protected Expression instantiate( Object oldInstance, Encoder out ) {
			Point point = (Point) oldInstance;
			Object[] args = new Object[] { new Integer( point.x ), new Integer( point.y ) };
			return new Expression( point, point.getClass(), "new", args );
		}
	}

	//---- class Rectangle_PersistenceDelegate --------------------------------

	private static class Rectangle_PersistenceDelegate
		extends PersistenceDelegate
	{
		@Override
		protected boolean mutatesTo( Object oldInstance, Object newInstance ) {
			return oldInstance.equals( newInstance );
		}

		@Override
		protected Expression instantiate( Object oldInstance, Encoder out ) {
			Rectangle rectangle = (Rectangle) oldInstance;
			Object[] args = new Object[] { new Integer( rectangle.x ), new Integer( rectangle.y ),
										   new Integer( rectangle.width ), new Integer( rectangle.height ) };
			return new Expression( rectangle, rectangle.getClass(), "new", args );
		}
	}

	//---- class GradientPaint_PersistenceDelegate ----------------------------

	private static class GradientPaint_PersistenceDelegate
		extends DefaultPersistenceDelegate
	{
		@Override
		protected Expression instantiate( Object oldInstance, Encoder out ) {
			GradientPaint gradientPaint = (GradientPaint) oldInstance;
			return new Expression( oldInstance, GradientPaint.class, "new",
				new Object[] {
					new Float( gradientPaint.getPoint1().getX() ),
					new Float( gradientPaint.getPoint1().getY() ),
					gradientPaint.getColor1(),
					new Float( gradientPaint.getPoint2().getX() ),
					new Float( gradientPaint.getPoint2().getY() ),
					gradientPaint.getColor2(),
					Boolean.valueOf( gradientPaint.isCyclic() )
				} );
		}
	}

	//---- inner class KeyStroke_PersistenceDelegate --------------------------

	private static class KeyStroke_PersistenceDelegate
		extends DefaultPersistenceDelegate
	{
		@Override
		protected Expression instantiate( Object oldInstance, Encoder out ) {
			KeyStroke keyStroke = (KeyStroke) oldInstance;
			return new Expression( oldInstance, KeyStroke.class, "getKeyStroke",
				new Object[] { new Integer( keyStroke.getKeyCode() ),
							   new Integer( keyStroke.getModifiers() ),
							   Boolean.valueOf( keyStroke.isOnKeyRelease() )} );
		}
	}

	//---- inner class DefaultListModel_PersistenceDelegate -------------------

	private static class DefaultListModel_PersistenceDelegate
		extends DefaultPersistenceDelegate
	{
		@Override
		protected void initialize( Class<?> type, Object oldInstance,
								   Object newInstance, Encoder out )
		{
			DefaultListModel m = (DefaultListModel) oldInstance;
			for( int i = 0; i < m.getSize(); i++ ) {
				out.writeStatement( new Statement( oldInstance, "addElement",
					new Object[] { m.getElementAt( i ) } ) );
			}
		}
	}

	//---- inner class DefaultTableModel_PersistenceDelegate ------------------

	private static class DefaultTableModel_PersistenceDelegate
		extends DefaultPersistenceDelegate
	{
		@Override
		protected Expression instantiate( Object oldInstance, Encoder out ) {
			DefaultTableModel model = (DefaultTableModel) oldInstance;
			int columnCount = model.getColumnCount();
			Vector<?> data = model.getDataVector();
			Vector<Object> columnNames = new Vector<Object>( columnCount );
			for( int i = 0; i < columnCount; i++ )
				columnNames.add( model.getColumnName( i ) );

			return new Expression( oldInstance, DefaultTableModel.class, "new",
				new Object[] { data, columnNames } );
		}
	}

	//---- class SpinnerNumberModel_PersistenceDelegate -----------------------

	private static class SpinnerNumberModel_PersistenceDelegate
		extends DefaultPersistenceDelegate
	{
		SpinnerNumberModel_PersistenceDelegate() {
			super( new String[] { "value", "minimum", "maximum", "stepSize" } );
		}

		@Override
		protected Expression instantiate( Object oldInstance, Encoder out ) {
			SpinnerNumberModel model = (SpinnerNumberModel) oldInstance;
			if( model.getMinimum() == null || model.getMaximum() == null ) {
				// Use default constructor if minimum or maximum are null
				// to avoid problems with decoding on some VMs, where the
				// decoder tries to invoke the wrong constructor.
				return new Expression( oldInstance, oldInstance.getClass(), "new", new Object[0] );
			} else
				return super.instantiate( oldInstance, out );
		}
	}

	//---- class ToString_PersistenceDelegate ---------------------------------

	private static class ToString_PersistenceDelegate
		extends PersistenceDelegate
	{
		@Override
		protected boolean mutatesTo( Object oldInstance, Object newInstance ) {
			return newInstance != null &&
				   oldInstance.toString().equals( newInstance.toString() );
		}

		@Override
		protected Expression instantiate( Object oldInstance, Encoder out ) {
			return new Expression( oldInstance, oldInstance.getClass(), "new",
				new Object[] { oldInstance.toString() } );
		}
	}

	//---- class Enum_PersistenceDelegate -------------------------------------

	private static class Enum_PersistenceDelegate
		extends DefaultPersistenceDelegate
	{
		@Override
		protected Expression instantiate( Object oldInstance, Encoder out ) {
			try {
				Class<?> oldClass = (Class<?>) enumGetDeclaringClassMethod.invoke( oldInstance, (Object[]) null );
				String oldName = (String) enumNameMethod.invoke( oldInstance, (Object[]) null );
				return new Expression( enumClass, "valueOf", new Object[] { oldClass, oldName } );
			} catch( Exception ex ) {
				return null;
			}
		}

		@Override
		protected boolean mutatesTo( Object oldInstance, Object newInstance ) {
			return oldInstance == newInstance;
		}
	}

	//---- class Fields_PersistenceDelegate ---------------------

	private static class Fields_PersistenceDelegate
		extends PersistenceDelegate
	{
		private final Class<?> cls;

		Fields_PersistenceDelegate( Class<?> cls ) {
			this.cls = cls;
		}

		@Override
		protected Expression instantiate( Object oldInstance, Encoder out ) {
			try {
				Field[] fields = cls.getFields();
				for( int i = 0; i < fields.length; i++ ) {
					if( oldInstance == fields[i].get( null ) )
						return new Expression( oldInstance, fields[i], "get", new Object[] { null } );
				}

				return instantiate2( oldInstance, out );
			} catch( Exception ex ) {
				throw newUnrecognizedInstance( oldInstance );
			}
		}

		protected Expression instantiate2( Object oldInstance, Encoder out ) {
			throw newUnrecognizedInstance( oldInstance );
		}
	}

	//---- class DefaultPersistenceDelegateEx ---------------------------------

	private static class DefaultPersistenceDelegateEx
		extends DefaultPersistenceDelegate
	{
		private final String[] constructor;

		DefaultPersistenceDelegateEx( String[] constructorPropertyNames ) {
			this.constructor = constructorPropertyNames;
		}

		/**
		 * Equal to super.instantiate() but also supports "is..." methods
		 * for booleans.
		 */
		@Override
		protected Expression instantiate( Object oldInstance, Encoder out ) {
			Class<?> type = oldInstance.getClass();
			Object[] constructorArgs = new Object[constructor.length];
			for( int i = 0; i < constructor.length; i++ ) {
				String name = constructor[i];
				Field f = null;
				try {
					f = type.getDeclaredField( name );
					f.setAccessible( true );
				} catch( NoSuchFieldException ex ) {
					// ignore
				}
				try {
					if( f != null && !Modifier.isStatic( f.getModifiers() ) )
						constructorArgs[i] = f.get( oldInstance );
					else {
						Method m = getGetterMethod( type, name );
						constructorArgs[i] = m.invoke( oldInstance, (Object[]) null );
					}
				} catch( Exception ex ) {
					out.getExceptionListener().exceptionThrown( ex );
				}
			}
			return new Expression( oldInstance, oldInstance.getClass(), "new", constructorArgs );
		}

		private Method getGetterMethod( Class<?> type, String name ) throws Exception {
			String capName = capitalize( name );
			try {
				return type.getMethod( "get" + capName, (Class[]) null );
			} catch( NoSuchMethodException ex ) {
				try {
					return type.getMethod( "is" + capName, (Class[]) null );
				} catch( NoSuchMethodException ex2 ) {
					throw ex; // forward exception of "get..." method
				}
			}
		}

		private String capitalize( String str ) {
			char[] chars = str.toCharArray();
			chars[0] = Character.toUpperCase( chars[0] );
			return new String( chars );
		}
	}
}
