/*
 * Copyright (C) 2003-2011 Karl Tauber <karl at jformdesigner dot com>
 * All Rights Reserved
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *  o Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 *  o Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 *  o Neither the name of JFormDesigner or Karl Tauber nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package com.jformdesigner.runtime;

import java.beans.*;
import java.util.*;

/**
 * @author Karl Tauber
 */
public class IntrospectorEx
{
	public interface Hook {
		public BeanInfo getBeanInfo( Class<?> beanClass ) throws IntrospectionException;
		public void flushCaches();
		public void flushFromCaches( Class<?> beanClass );
		public void flushCaches( ClassLoader classLoader );
	}

	private static Hook hook;

	/** Maps bean classes to bean infos <Class,BeanInfoEx>. */
	private static Map<Class<?>, BeanInfoEx> beanInfoCache = Collections.synchronizedMap( new WeakHashMap<Class<?>, BeanInfoEx>( 100 ) );

	public static Hook getHook() {
		return hook;
	}

	public static void setHook( Hook hook ) {
		IntrospectorEx.hook = hook;
	}

	public static void useHardRefCache() {
		beanInfoCache = Collections.synchronizedMap( new HashMap<Class<?>, BeanInfoEx>( 100 ) );
	}

	public static BeanInfoEx getBeanInfoEx( Class<?> beanClass )
		throws IntrospectionException
	{
		BeanInfoEx beanInfoEx = beanInfoCache.get( beanClass );
		if( beanInfoEx == null ) {
			BeanInfo beanInfo = (hook != null)
				? hook.getBeanInfo( beanClass )
				: Introspector.getBeanInfo( beanClass );
			beanInfoEx = new BeanInfoEx( beanInfo );
			beanInfoCache.put( beanClass, beanInfoEx );
		}
		return beanInfoEx;
	}

	public static void flushCaches() {
		beanInfoCache.clear();
		if( hook != null )
			hook.flushCaches();
		else
			Introspector.flushCaches();
	}

	public static void flushFromCaches( Class<?> beanClass ) {
		beanInfoCache.remove( beanClass );
		if( hook != null )
			hook.flushFromCaches( beanClass );
		else
			Introspector.flushFromCaches( beanClass );
	}

	/**
	 * Flush all BeanInfo loaded with the specified class loader.
	 */
	public static void flushCaches( ClassLoader classLoader ) {
		synchronized( beanInfoCache ) {
			Iterator<Class<?>> it = beanInfoCache.keySet().iterator();
			while( it.hasNext() ) {
				Class<?> cls = it.next();
				if( cls.getClassLoader() != classLoader )
					continue;

				it.remove();

				// remove from Introspector cache (including superclasses)
				do {
					if( hook != null )
						hook.flushFromCaches( cls );
					else
						Introspector.flushFromCaches( cls );
					cls = cls.getSuperclass();
				} while( cls != null && cls.getClassLoader() == classLoader );
			}
		}

		if( hook != null )
			hook.flushCaches( classLoader );
	}
}
