/**
 * ColumnList.java
 *
 * Copyright 1995-1996 Active Software Inc.
 *
 * @version @(#)ColumnList.java 1.40 96/08/02
 * @author  Tilman Sporkert
 */

package sunsoft.jws.visual.rt.awt;

import sunsoft.jws.visual.rt.awt.GBLayout;
import sunsoft.jws.visual.rt.awt.GBConstraints;
import sunsoft.jws.visual.rt.awt.VJScrollbar;
import sunsoft.jws.visual.rt.awt.PanelFrame;

import java.awt.*;
import java.util.Vector;

/**
 * A Widget to display data in a row/column fashion, with scrollbars etc.
 *
 * A ColumnList has the following features/attributes:
 *  headers: An array of Strings. The size of this array determines the number
 *           of columns in the list. There will be one additional, hidden
 *           column that can be used for any Object. 
 *           The length of the header string defines the initial width of the
 *           column. To make it wider, add spaces to the end of the string.
 *           If a header field is of the form "name=header", then only the
 *           "header" part will actually be used for the visible header.
 *           The "name" part is available through getNames() or getNameColumn()
 *  selectable: If true, records can be selected with the mouse. A LIST_EVENT 
 *              gets posted.
 *  highlight_items: If true, new entries will be highlighted in orange, slowly
 *                   fading out.
 *  showHeaders: If set to false, the headers will not be shown.
 *           
 *
 * @author  Tilman Sporkert
 */
public class ColumnList extends ScrollPanel {

    // Entry attributes
    static final int HIGHLIGHT = 1;

    ColumnListCanvas canvas;
    String[] names = null;
    boolean[] keys = null;


    /**
     * create a new ColumnList with no information about the columns
     */
    public ColumnList() {
	add(canvas = new ColumnListCanvas(this));
	setCanvasBackground(Color.white);
    }

    public void setCanvasForeground(Color fg) {
        canvas.setForeground(fg);
	canvas.repaint();
    }

    public Color getCanvasForeground() {
        return canvas.getForeground();
    }
    
    public void setCanvasBackground(Color fg) {
        canvas.setBackground(fg);
	canvas.repaint();
    }
    
    public Color getCanvasBackground() {
        return canvas.getBackground();
    }
    
    public void setCanvasFont(Font font) {
        canvas.setFont(font);
	canvas.repaint();
    }
    
    public Font getCanvasFont() {
        return canvas.getFont();
    }
    
    /**
     * create a new ColumnList, with the given columns, and given visible
     * rows.
     * ...
     *
     * @param ...    ...
     * @return       ...
     * @exception    ...
     */
    public ColumnList(String[] headers,
		      boolean selectable,
		      boolean highlight_items) {
	this();
	setHeaders(headers);
	// setFormat(format);
	setSelectable(selectable);
	setHighlightItems(highlight_items);
    }
    

    /**
     * Set the desired number of rows to be displayed. This affects the 
     * minumumSize() and preferredSize() of the widget. The actual rows
     * displayed by the widget depend on how the LayoutManager interprets
     * those values.
     *
     * @param rows   number of desired rows for display
     */
    public void setVisibleRows(int rows) {
	setDisplayRows(rows);
    }


    // for backward compatibility
    public void setDisplayRows(int rows) {
	canvas.setDisplayRows(rows);
    }


    /**
     * Set the desired number of "visible" chars, i.e. the total width of
     * the column list. Defaults to "all".
     *
     * @param ...    ...
     * @return       ...
     * @exception    ...
     */
    public void setVisibleChars(int chars) {
	canvas.setVisibleChars(chars);
    }

    
    /**
     * ...
     *
     * @param ...    ...
     * @return       ...
     * @exception    ...
     */
    public void setHeaders(String[] headers) {
	String[] realHeaders = new String[headers.length];
	names = new String[headers.length];
	keys = new boolean[headers.length];
	boolean hasKeys = false;
	for (int h = 0; h < headers.length; h++) {
	    keys[h] = false;
	    int offset = headers[h].indexOf('=');
	    if (offset > 0) {
		names[h] = headers[h].substring(0, offset);
		if (names[h].charAt(0) == '*') {
		    keys[h] = true;
		    hasKeys = true;
		    names[h] = headers[h].substring(1, offset);
		}
		realHeaders[h] = headers[h].substring(offset + 1);
	    } else {
		realHeaders[h] = headers[h];
		names[h] = headers[h];
	    }
	}
	if (!hasKeys)
	    keys = null;
	canvas.setHeaders(realHeaders);
    }
    
    
    
    /**
     * ...
     *
     * @param ...    ...
     * @return       ...
     * @exception    ...
     */
    public void setSelectable(boolean selectable) {
	canvas.setSelectable(selectable);
    }
    

    /**
     * ...
     *
     * @param ...    ...
     * @return       ...
     * @exception    ...
     */
    public void setHighlightItems(boolean highlight_items) {
	canvas.setHighlightItems(highlight_items);
    }
    

    /**
     * ...
     *
     * @param ...    ...
     * @return       ...
     * @exception    ...
     */
    public void setShowHeaders(boolean showHeaders) {
	canvas.setShowHeaders(showHeaders);
    }
    
    /**
     * ...
     *
     * @param ...    ...
     * @return       ...
     * @exception    ...
     */
    public void setAutoWidth(boolean autoWidth) {
	canvas.setAutoWidth(autoWidth);
    }
    

    /**
     * Adds an item to the list. If the list has no keys defined, the item
     * will be appended at the end of the list. Otherwise, the existing entries
     * will be searched for an entry with the same key values. If there is
     * such an entry, it will get replaced. Otherwise, the record goes at the
     * end.
     *
     * @param values the record
     * @return index of new record
     */
    public int addItem(Object[] values, boolean updateView) {
	if (keys != null) {
	    for (int r = 0; r < entries(); r++)
		if (isEqualTo(values, r)) {
		    canvas.addItem(values, r);
		    if (updateView)
			updateView();
		    return r;
		}
	}
	canvas.addItem(values);
	if (updateView)
	    updateView();
	return (entries() - 1);
    }


    /**
     * ...
     *
     * @param ...    ...
     * @return       ...
     * @exception    ...
     */
    public void addItem(Object[] values) {
	addItem(values, true);
    }
    
    
    /**
     * ...
     *
     * @param ...    ...
     * @return       ...
     * @exception    ...
     */
    private boolean isEqualTo(Object[] values, int r) {
	for (int c = 0; c < getColumns(); c++) {
	    if (keys[c]) {
		if (!((String) getItem(r, c)).equals((String) values[c]))
		    return false;
	    }
	}
	return true;
    }
    

    /**
     * ...
     *
     * @param ...    ...
     * @return       ...
     * @exception    ...
     */
    public void addItem(Object[] values, int row) {
	canvas.addItem(values, row);
	updateView();
    }
    

    /**
     * ...
     *
     * @param ...    ...
     * @return       ...
     * @exception    ...
     */
    public void insertItem(Object[] values, int row) {
	canvas.insertItem(values, row);
	updateView();
    }
    

    /**
     * ...
     *
     * @param ...    ...
     * @return       ...
     * @exception    ...
     */
    public void delItems() {
	canvas.delItems();
	updateView();
    }
    

    /**
     * ...
     *
     * @param ...    ...
     * @return       ...
     * @exception    ...
     */
    public void delItems(int start, int end) {
	canvas.delItems(start, end);
	updateView();
    }
    

    /**
     * ...
     *
     * @param ...    ...
     * @return       ...
     * @exception    ...
     */
    public boolean delItem(Object[] values) {
	if (keys != null) {
	    for (int r = 0; r < entries(); r++)
		if (isEqualTo(values, r)) {
		    canvas.delItems(r, r);
		    updateView();
		    return true;
		}
	}
	return false;
    }
    
    
    /**
     * ...
     *
     * @param ...    ...
     * @return       ...
     * @exception    ...
     */
    public void highlight(int row) {
	canvas.highlight(row);
    }
 

    /**
     * ...
     *
     * @param ...    ...
     * @return       ...
     * @exception    ...
     */
    public Rectangle getListBounds() {
	return canvas.bounds();
    }
    

    /**
     * ...
     *
     * @param ...    ...
     * @return       ...
     * @exception    ...
     */
    public int entries() {
	return canvas.records;
    }
    
    
    /**
     * ...
     *
     * @param ...    ...
     * @return       ...
     * @exception    ...
     */
    public int getColumns() {
	return canvas.columns;
    }
    
    
    /**
     * ...
     *
     * @param ...    ...
     * @return       ...
     * @exception    ...
     */
    public String[] getNames() {
	return names;
    }
    
    
    /**
     * ...
     *
     * @param ...    ...
     * @return       ...
     * @exception    ...
     */
    public int getNameColumn(String name) {
	for (int n = 0; n < names.length; n++)
	    if (names[n].equals(name))
		return n;
	return -1;
    }
    
    
    /**
     * ...
     *
     * @param ...    ...
     * @return       ...
     * @exception    ...
     */
    public Object getItem(int row, int column) {
	return canvas.getItem(row, column);
    }
    
    
    /**
     * ...
     *
     * @param ...    ...
     * @return       ...
     * @exception    ...
     */
    public boolean putItem(int row, int column, Object value) {
	return canvas.putItem(row, column, value);
    }
    
    
    /**
     * swap the entries in two rows
     *
     * @param ...    ...
     * @return       ...
     */
    public boolean swapItems(int row1, int row2) {
	return canvas.swapItems(row1, row2);
    }


    /**
     * get the Y coordinate of the upper edge of a row in the column list.
     * Returns -1 if the row is not visible. Returns -2 is the list is
     * not layed out yet.
     *
     * @param row index of desired row
     * @return       Y coordinage of row
     */
    public int getRowY(int row) {
	return canvas.getRowY(row);
    }


    /**
     * get the row height of entries in the list
     *
     * @return       height of a row
     */
    public int getRowHeight() {
	return canvas.rowHeight;
    }


    /**
     * call repaint() on the ColumnListCanvas. Needed if an Object in the
     * column list has been changed directly (without going through 
     * putItem).
     */
    public void needsRepaint() {
	canvas.repaint();
    }
    

    /**
     * redraw everything, and reevaluate the need for scroll bars
     */
    public void updateView() {
	canvas.updateView();
	updateWindow();
    }
    
    /**
     * overrides updateVScrollbar in ScrollPanel
     */
    protected void updateVScrollbar() {
	super.updateVScrollbar();
	vbar.setLineIncrement(canvas.lineHeight());
	vbar.setPageIncrement(canvas.lineHeight() * canvas.disp_rows);
    }
    
    
}
