/*
 * @(#)GBConstraints.java	1.8 96/08/03 Doug Stein
 *
 * Copyright (c) 1996 Sun Microsystems, Inc. All Rights Reserved.
 *
 * Permission to use, copy, modify, and distribute this software
 * and its documentation for NON-COMMERCIAL purposes and without
 * fee is hereby granted provided that this copyright notice
 * appears in all copies. Please refer to the file "copyright.html"
 * for further important copyright and licensing information.
 *
 * SUN MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
 * THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, OR NON-INFRINGEMENT. SUN SHALL NOT BE LIABLE FOR
 * ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
 * DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 */
package sunsoft.jws.visual.rt.awt;

import java.awt.*;
import java.util.StringTokenizer;
import java.util.NoSuchElementException;

/**
 * GBConstraints is used to specify constraints for components
 * laid out using the GBLayout class.
 *
 * @see java.awt.GBLayout
 * @version 1.8, 07/30/96
 * @author Doug Stein
 */
public class GBConstraints implements Cloneable {
  public static final int RELATIVE = -1;
  public static final int REMAINDER = 0;

  public static final int NONE = 0;
  public static final int BOTH = 1;
  public static final int HORIZONTAL = 2;
  public static final int VERTICAL = 3;

  public static final int CENTER = 10;
  public static final int NORTH = 11;
  public static final int NORTHEAST = 12;
  public static final int EAST = 13;
  public static final int SOUTHEAST = 14;
  public static final int SOUTH = 15;
  public static final int SOUTHWEST = 16;
  public static final int WEST = 17;
  public static final int NORTHWEST = 18;

  public int gridx, gridy, gridwidth, gridheight;
  public double weightx, weighty;
  public int anchor, fill;

  // Regular insets and pads will shrink when space gets tight
  public Insets insets;
  public int ipadx, ipady;

  // Hard insets and pads never shrink
  public Insets hardinsets;
  public int hardipadx, hardipady;

  // Normally a component will not shrink below it minimum size.  Setting
  // shrinkx or shrinky to true indicates that the component may shrink
  // below its minimum size.
  public boolean shrinkx;
  public boolean shrinky;

  // The following variables are filled in during layout and
  // can be accessed, but should not be modified:
  public Point location;           // location of the component
  public Dimension size;           // size of the component
  public Dimension minsize;        // minimum size of the component

  int tempX, tempY;
  int tempWidth, tempHeight;
  int tinyWidth, tinyHeight;

  /**
   * Creates a set of gridbag constraints.
   */
  public GBConstraints () {
    gridx = RELATIVE;
    gridy = RELATIVE;
    gridwidth = 1;
    gridheight = 1;

    weightx = 0;
    weighty = 0;
    anchor = CENTER;
    fill = NONE;

    insets = new Insets(0, 0, 0, 0);
    ipadx = 0;
    ipady = 0;

    hardinsets = new Insets(0, 0, 0, 0);
    hardipadx = 0;
    hardipady = 0;
  }

  /**
   * Creates a set of gridbag constraints by parsing the given
   * constraints option string.  Each option has the form key=value.
   * Options are separated by semicolons (;).
   */
  public GBConstraints (String constraints) {
    this();
    parseConstraints(constraints);
  }

  public Object clone () {
      GBConstraints c;
      try { 
	  c = (GBConstraints)super.clone();
      } catch (CloneNotSupportedException e) { 
	  // this shouldn't happen, since we are Cloneable
	  throw new InternalError();
      }

      if (c.insets != null)
	c.insets = (Insets)c.insets.clone();
      if (c.hardinsets != null)
	c.hardinsets = (Insets)c.hardinsets.clone();

      return c;
  }

  private void parseConstraints(String constraints) {
    StringTokenizer st = new StringTokenizer(constraints, ";", true);

    String option_string = null;
    try {
      while (st.hasMoreTokens()) {
	option_string = st.nextToken();
	if (option_string.equals(";"))
	  continue;
	    
	StringTokenizer op = new StringTokenizer(option_string, "=", true);
	String option = op.nextToken();
	    
	if (option.equals("gridx") || option.equals("x"))
	  gridx = convertSymbolicValue(getValueToken(op));
	else if (option.equals("gridy") || option.equals("y"))
	  gridy = convertSymbolicValue(getValueToken(op));
	else if (option.equals("gridwidth") || option.equals("width"))
	  gridwidth = convertSymbolicValue(getValueToken(op));
	else if (option.equals("gridheight") || option.equals("height"))
	  gridheight = convertSymbolicValue(getValueToken(op));

	else if (option.equals("weightx")) {
	  Double x = new Double(getValueToken(op));
	  weightx = x.doubleValue();
	}
	else if (option.equals("weighty")) {
	  Double x = new Double(getValueToken(op));
	  weighty = x.doubleValue();
	}
	else if (option.equals("anchor"))
	  anchor = convertSymbolicValue(getValueToken(op));
	else if (option.equals("fill"))
	  fill = convertSymbolicValue(getValueToken(op));

	else if (option.equals("insets.top"))
	  insets.top = convertSymbolicValue(getValueToken(op));
	else if (option.equals("insets.left"))
	  insets.left = convertSymbolicValue(getValueToken(op));
	else if (option.equals("insets.bottom"))
	  insets.bottom = convertSymbolicValue(getValueToken(op));
	else if (option.equals("insets.right"))
	  insets.right = convertSymbolicValue(getValueToken(op));

	else if (option.equals("ipadx"))
	  ipadx = convertSymbolicValue(getValueToken(op));
	else if (option.equals("ipady"))
	  ipady = convertSymbolicValue(getValueToken(op));

	else if (option.equals("shrinkx")) {
	  Boolean x = new Boolean(getValueToken(op));
	  shrinkx = x.booleanValue();
	}
	else if (option.equals("shrinky")) {
	  Boolean x = new Boolean(getValueToken(op));
	  shrinky = x.booleanValue();
	}

	else if (option.equals("hardinsets.top"))
	  hardinsets.top = convertSymbolicValue(getValueToken(op));
	else if (option.equals("hardinsets.left"))
	  hardinsets.left = convertSymbolicValue(getValueToken(op));
	else if (option.equals("hardinsets.bottom"))
	  hardinsets.bottom = convertSymbolicValue(getValueToken(op));
	else if (option.equals("hardinsets.right"))
	  hardinsets.right = convertSymbolicValue(getValueToken(op));

	else if (option.equals("hardipadx"))
	  hardipadx = convertSymbolicValue(getValueToken(op));
	else if (option.equals("hardipady"))
	  hardipady = convertSymbolicValue(getValueToken(op));

	else
	  throw new NoSuchElementException();
      }
    }
    catch (Exception e) {
      throw new Error("\r\n\tSyntax error in GBConstraints string:\r\n" +
		      "\t\t" + constraints + ": " + option_string);
    }
  }

  private String getValueToken(StringTokenizer op)
    throws NoSuchElementException
  {
    if (op.hasMoreTokens()) {
      String assign = op.nextToken();
      if (assign.equals("=")) 
	if (op.hasMoreTokens()) 
	return op.nextToken();
    }
    throw new NoSuchElementException();
  }

  private int convertSymbolicValue(String value) {
    if (value.equals("relative"))
      return GBConstraints.RELATIVE;
    else if (value.equals("remainder"))
      return GBConstraints.REMAINDER;
    else if (value.equals("none"))
      return GBConstraints.NONE;
    else if (value.equals("both"))
      return GBConstraints.BOTH;
    else if (value.equals("horizontal"))
      return GBConstraints.HORIZONTAL;
    else if (value.equals("vertical"))
      return GBConstraints.VERTICAL;
    else if (value.equals("center"))
      return GBConstraints.CENTER;
    else if (value.equals("north"))
      return GBConstraints.NORTH;
    else if (value.equals("northeast"))
      return GBConstraints.NORTHEAST;
    else if (value.equals("east"))
      return GBConstraints.EAST;
    else if (value.equals("southeast"))
      return GBConstraints.SOUTHEAST;
    else if (value.equals("south"))
      return GBConstraints.SOUTH;
    else if (value.equals("southwest"))
      return GBConstraints.SOUTHWEST;
    else if (value.equals("west"))
      return GBConstraints.WEST;
    else if (value.equals("northwest"))
      return GBConstraints.NORTHWEST;

    Integer int_val = new Integer(value);
    return int_val.intValue();
  }
}
