/*
 *        Copyright (C) 1996  Active Software, Inc.
 *                  All rights reserved.
 *
 * @(#) ImageButton.java 1.13 - last change made 07/22/96
 */

package sunsoft.jws.visual.rt.awt;

import sunsoft.jws.visual.rt.base.DesignerAccess;
import java.awt.*;

/**
 * An image button (a 3D rect around an image.)  It greys itself out
 * when disabled and inverts the 3D rect and moves its image when depressed.
 *
 * @(#) ImageButton.java 1.13 - last change made 07/22/96
 */
public class ImageButton extends ImageLabel {
  private int lineWidth = 2;	  // thickness of 3D line around button
  private int pressMovement = 1;  // distance image moves when button pressed
  protected boolean depressed;

  public ImageButton(Image img) {
    this(img, 20, 20);
  }

  public ImageButton(Image img, int w, int h) {
    super(img, w, h);
    depressed = false;
    padWidth = lineWidth + 2;
  }

  public void setPadWidth(int w) {
    super.setPadWidth(w + lineWidth);
  }

  public int getPadWidth() {
    return(super.getPadWidth() - lineWidth);
  }

  public void setLineWidth(int w) {
    int oldPadWidth = getPadWidth();
    lineWidth = w;
    setPadWidth(oldPadWidth);
  }

  public int getLineWidth() {
    return(lineWidth);
  }

  public void setPressMovement(int p) {
    if (p != pressMovement) {
      pressMovement = p;
      if (depressed) {
	Graphics g = getGraphics();
	if (g != null) {
	  Dimension d = size();
	  g.setColor(getBackground());
	  g.fillRect(0, 0, d.width, d.height);
	  repaint();
	}
      }
    }
  }

  public int getPressMovement() {
    return(pressMovement);
  }

  public void paint(Graphics g) {
    if (upImg != null && (imgWidth >= 0 && imgHeight >= 0)) {
      synchronized (DesignerAccess.mutex) {
	Dimension d = size();
	Image img = isEnabled() ? upImg : disImg;
	int x = (d.width - imgWidth) / 2;
	int y = (d.height - imgHeight) / 2;
	int offset = (depressed ? pressMovement : 0);

	if (pressMovement != 0) {
	  // clear the area needed to accomodate press movement
	  g.setColor(getBackground());
	  int m = (pressMovement < 0 ? -1 : 1);
	  for (int i = 0; i < pressMovement * m; i++)
	    g.drawRect(x + i * m + (m < 0 ? -1 : 0),
		       y + i * m + (m < 0 ? -1 : 0), imgWidth, imgHeight);
	}

	// draw the image and the 3D border
	g.drawImage(img, x + offset, y + offset, getBackground(), this);
	g.setColor(getBackground());
	for (int i = 0; i < lineWidth; i++)
	  g.draw3DRect(i, i,
		       d.width - i*2 - 1,
		       d.height - i*2 - 1,
		       !depressed);
      }
    }
  }

  public boolean mouseDown(Event e, int x, int y) {
    depressed = true;
    repaint();
    return true;
  }

  public boolean mouseDrag(Event e, int x, int y) {
    if (depressed != inside(x, y)) {
      depressed = !depressed;
      repaint();
    }
    return true;
  }

  public boolean mouseUp(Event evt, int x, int y) {
    if (depressed) {
      action();
      depressed = false;
      repaint();
    }
    return true;
  }

  public void action() {
    postEvent(new Event(this, Event.ACTION_EVENT, null));
  }
}
