/*
 *        Copyright (C) 1996  Active Software, Inc.
 *                  All rights reserved.
 *
 * @(#) Attribute.java 1.25 - last change made 07/30/96
 */

package sunsoft.jws.visual.rt.base;

/**
 * Storage for a single attribute.
 *
 * @version 	1.25, 07/30/96
 */
public class Attribute implements Cloneable {
  /**
   * Name of this attribute.
   */
  private String name;

  /**
   * The full class name of the type of the value of this attribute.
   */
  private String type;

  /**
   * The actual value of this attribute.  It is of the type specified
   * by the type field.
   */
  private Object value;

  /**
   * The default value of this attribute.  It is of the type specified
   * by the type field.
   */
  private Object defaultValue;

  /**
   * Contains description flags about the nature of this attribute.
   */
  private int flags;

  /**
   * The constructor initializes the value for the attribute.
   * The initial value of the attribute is set to be the same
   * as the default value.  If the default value is not a simple type, you
   * may want to set the value again after cloning the default value,
   * otherwise, directly setting internal members of the value will change
   * the default value as well.
   */
  public Attribute(String name, String type, Object defaultValue, int flags) {
    this.name = name;
    this.type = type;

    // throwOnBadType is commented out for performance reasons.
    // Shadow class construction is 15% faster with this commented out.
    //
    // throwOnBadType(defaultValue);

    this.value = defaultValue;
    this.defaultValue = defaultValue;
    this.flags = flags;
  }

  private void throwOnBadType(Object checkee) {
    // allow null
    if (checkee == null)
      return;

    // type must be exactly the same (except for null)
    if (checkee.getClass().getName().equals(type))
      return;

    throw new Error("Illegal attribute value for \"" + name + "\"" +
		    Global.newline() + "    Expected type : " + type + "" +
		    Global.newline() + "    Actual type   : " +
		    checkee.getClass().getName() +
		    Global.newline() +"    value = \"" + checkee + "\"");
  }

  public String getName() {
    return(name);
  }

  public String getType() {
    return(type);
  }

  public Object getDefaultValue() {
    return(defaultValue);
  }

  void setDefaultValue(Object value) {
    defaultValue = value;
  }

  public boolean isModified() {
    if (value == null)
      return(defaultValue != null);
    else
      return(!value.equals(defaultValue));
  }

  /**
   * Resets the value to the default
   */
  public void reset() {
    setValue(defaultValue);
  }

  public Object getValue() {
    return(value);
  }

  /**
   * Sets the value of the attribute.
   */
  public void setValue(Object value) {
    // Commented out the "does new value equal old value" check, because:
    // 1) Setting a value to "null" can cause a null pointer exception
    // 2) Some attributes may not implement equals properly
    throwOnBadType(value);
    this.value = value;
  }
  
  public int getFlags() {
    return(flags);
  }

  public void addFlags(int flags) {
    this.flags = (this.flags | flags);
  }

  public boolean flagged(int flags) {
    return((flags & this.flags) != 0);
  }

  /**
   * Shallow clone for this attribute.  It's not a deep clone, only the
   * references to the value and default value are cloned, not the actual
   * values themselves.
   */
  public Object clone() {
    Object retval;
    try {
      retval = super.clone();
    }
    catch (CloneNotSupportedException e) {
      throw new Error(e.getMessage());
    }
    return(retval);
  }
}
