/*
 *        Copyright (C) 1996  Active Software, Inc.
 *                  All rights reserved.
 *
 * @(#) Util.java 1.28 - last change made 07/30/96
 */

package sunsoft.jws.visual.rt.base;

import java.awt.*;
import java.util.*;
import java.io.*;

import java.awt.image.RGBImageFilter;
import java.awt.image.FilteredImageSource;
import java.net.URL;
import java.net.MalformedURLException;
import java.applet.Applet;

/**
 * Utilities needed by run-time.
 *
 * @version 	1.28, 07/30/96
 */
public class Util {
  // Relief constants
  public static final int RELIEF_FLAT   = 0;
  public static final int RELIEF_RAISED = 1;
  public static final int RELIEF_SUNKEN = 2;
  public static final int RELIEF_RIDGE  = 3;
  public static final int RELIEF_GROOVE = 4;
  public static final int WIN95_RAISED = 6;
  public static final int WIN95_SUNKEN = 7;
  public static final int WIN95_FIELD_BORDER = 8;
  public static final int WIN95_WINDOW_BORDER = 9;

  // Darkness constants
  private static final double BFACTOR = 0.82;
  private static final double DFACTOR = 0.7;

  // Buffer size for reading files
  private static final int BUFSIZE = 2048;

  /**
   * Returns a brighter version of this color.
   */
  public Color brighter(Color c) {
    int r = c.getRed();
    int g = c.getGreen();
    int b = c.getBlue();

    if (r == 255)
      r = 192;
    else
      r = Math.min((int)(r * (1/BFACTOR)), 255);

    if (g == 255)
      g = 192;
    else
      g = Math.min((int)(g * (1/BFACTOR)), 255);

    if (b == 255)
      b = 192;
    else
      b = Math.min((int)(b * (1/BFACTOR)), 255);

    return new Color(r, g, b);
  }

  /**
   * Returns a darker version of this color.
   */
  public Color darker(Color c) {
    int r = c.getRed();
    int g = c.getGreen();
    int b = c.getBlue();

    if (r == 255)
      r = 128;
    else
      r = Math.max((int)(r * DFACTOR), 0);

    if (g == 255)
      g = 128;
    else
      g = Math.max((int)(g * DFACTOR), 0);

    if (b == 255)
      b = 128;
    else
      b = Math.max((int)(b * DFACTOR), 0);

    return new Color(r, g, b);
  }

  /**
   * Draw a 3D rectable with the given relief and border width.
   */
  public void draw3DRect(Graphics g, int x, int y, int w, int h,
			 int relief, int bd) {
    int bd2 = (bd+1)/2;
    for (int i=0; i<bd; i++)
      draw3DRect(g, x+i, y+i, w-2*i, h-2*i, relief, (i < bd2));
  }

  /**
   * Draw a 3D rectable with the given relief and outer boolean.
   */
  private void draw3DRect(Graphics g, int x, int y, int w, int h,
			 int relief, boolean isOuter) {
    Color color = g.getColor();

    g.setColor(getEdgeColor(color, relief, true, isOuter));
    g.drawLine(x, y, x+w, y);
    g.drawLine(x, y, x, y+h);

    g.setColor(getEdgeColor(color, relief, false, isOuter));
    g.drawLine(x, y+h, x+w, y+h);
    g.drawLine(x+w, y, x+w, y+h);

    g.setColor(color);
  }

  /**
   * Returns an adjusted color for the given edge and the given relief.
   */
  private Color getEdgeColor(Color base, int relief,
			     boolean isTopEdge, boolean isOuter) {
    Color color = null;

    switch (relief) {
    case RELIEF_RAISED:
      if (isTopEdge)
	color = brighter(base);
      else
	color = darker(base);
      break;

    case RELIEF_SUNKEN:
      if (isTopEdge)
	color = darker(base);
      else
	color = brighter(base);
      break;

    case RELIEF_RIDGE:
      if (isTopEdge) {
	if (isOuter)
	  color = brighter(base);
	else
	  color = darker(base);
      }
      else {
	if (isOuter)
	  color = darker(base);
	else
	  color = brighter(base);
      }
      break;

    case RELIEF_GROOVE:
      if (isTopEdge) {
	if (isOuter)
	  color = darker(base);
	else
	  color = brighter(base);
      }
      else {
	if (isOuter)
	  color = brighter(base);
	else
	  color = darker(base);
      }
      break;

    case WIN95_RAISED:
      if (isTopEdge) {
	if (isOuter)
	  color = Color.white;
	else
	  color = brighter(base);
      }
      else {
	if (isOuter)
	  color = Color.black;
	else
	  color = darker(base);
      }
      break;

    case WIN95_SUNKEN:
      if (isTopEdge) {
	if (isOuter)
	  color = Color.black;
	else
	  color = darker(base);
      }
      else {
	if (isOuter)
	  color = Color.white;
	else
	  color = brighter(base);
      }
      break;

    case WIN95_FIELD_BORDER:
      if (isTopEdge) {
	if (isOuter)
	  color = darker(base);
	else
	  color = Color.black;
      }
      else {
	if (isOuter)
	  color = Color.white;
	else
	  color = base; // brighter(base);
      }
      break;

    case WIN95_WINDOW_BORDER:
      if (isTopEdge) {
	if (isOuter)
	  color = brighter(base);
	else
	  color = Color.white;
      }
      else {
	if (isOuter)
	  color = Color.black;
	else
	  color = darker(base);
      }
      break;

    case RELIEF_FLAT:
    default:
      color = base;
      break;
    }

    return color;
  }

  /**
   * Get an image given a url.  If we are on Windows 95, then we
   * need to use a filter to get around the transparency bugs.
   */
  public Image getWorkaroundImage(URL url, Component comp) {
    Image image = comp.getToolkit().getImage(url);
    return getWorkaroundImage(image, comp);
  }

  /**
   * Get an image given another.  If we are on Windows 95, then we
   * need to use a filter to get around the transparency bugs.
   * Otherwise, just return the image directly.
   */
  public Image getWorkaroundImage(Image image, Component comp) {
    if (image == null)
      return null;

    if (Global.isWindows95() && Global.javaVersion() == 1.0) {
      RGBImageFilter filter = new TransFilter(comp);
      image = comp.createImage(
	new FilteredImageSource(image.getSource(), filter));
    }

    return image;
  }

  /**
   * Returns a URL based on a relative path to a file or directory.
   * If we are running under a browser, then a URL is created based
   * off of the code base.  Otherwise, a file URL will be created
   * by searching the CLASSPATH for the file.
   */
  public URL pathToURL(String path, Applet applet) {
    // First see if the path is a full URL path
    try {
      return new URL(path);
    }
    catch (MalformedURLException ex) {
    }

    // Are we running as an applet?
    if (applet != null) {
      String s = applet.getCodeBase().toExternalForm();
      if (s.charAt(s.length()-1) != '/')
	path = "/" + path;

      URL url;
      try {
	url = new URL(applet.getCodeBase(), path);
      }
      catch (MalformedURLException ex) {
	url = null;
      }
      return url;
    }

    // Search the CLASSPATH for the file
    String classpath;
    try {
      classpath = System.getProperty("java.class.path");
    }
    catch (SecurityException ex) {
      throw new Error(Global.newline() +
	"    You must provide a non-null applet parameter to \"pathToURL\"" +
        Global.newline() + "    when running as an applet.");
    }

    String delimiter, urlPrefix;
    if (Global.isWindows()) {
      delimiter = ";";
      urlPrefix = "file:/";
    }
    else {
      delimiter = ":";
      urlPrefix = "file:";
    }

    classpath = DesignerAccess.getCWD() + delimiter + classpath;

    StringTokenizer st = new StringTokenizer(classpath, delimiter);
    while (st.hasMoreTokens()) {
      String p = st.nextToken();

      if (p == "")
	p = ".";

      p = makeAbsolute(p);

      char c = p.charAt(p.length()-1);
      if (c != '/' && c != '\\')
	p = p + separator;

      p = p + path;

      if (Global.isWindows()) {
	// Java allows the use of '/' in the classpath, so we need
	// convert '/' to '\'.
	char buf[] = p.toCharArray();
	for (int i=0; i<buf.length; i++) {
	  if (buf[i] == '/')
	    buf[i] = '\\';
	}
	p = new String(buf);
      }

      File f = new File(p);
      if (f.exists()) {
	try {
	  return new URL(urlPrefix + p);
	}
	catch (MalformedURLException ex) {
	  return null;
	}
      }
    }

    return null;
  }

  private static String separator;
  private static String cwd;

  private String makeAbsolute(String path) {
    if (separator == null) {
      separator = System.getProperty("file.separator");
    }

    if (cwd == null) {
      cwd = System.getProperty("user.dir");
      if (cwd.charAt(cwd.length()-1) != separator.charAt(0))
	cwd = cwd + separator;
    }

    if (Global.isWindows()) {
      if (path.length() < 3 ||
	  (path.charAt(1) != ':' ||
	   (path.charAt(2) != '/' && path.charAt(2) != '\\'))) {
	path = cwd + path;
      }
    }
    else {
      if (path.charAt(0) != '/')
	path = cwd + path;
    }

    return path;
  }

  /**
   * Creates a string from reading a URL.  Returns null when an error occurs.
   */
  public String readURLToString(URL url)
    throws IOException
  {
    // Try to read the url as a file if possible (this is faster).
    if (url.getProtocol().equals("file")) {
      try {
	return fileToString(url.getFile());
      }
      catch (FileNotFoundException ex) {
      }
      catch (SecurityException ex) {
	System.out.println("SECURITY EXCEPTION1: " + url.getFile());
      }
    }

    // The url is not local, so read it in one byte at a time.
    StringBuffer buf = new StringBuffer();
    InputStream in = null;

    try {
      int c;
      in = url.openStream();
      while ((c = in.read()) != -1)
	buf.append((char) c);
    }
    catch (IOException ex) {
      if (in != null)
	try { in.close(); } catch (IOException e) {}
      throw ex;
    }

    return buf.toString();
  }

  /**
   * Creates a string from the contents of a file.  Returns null when
   * an error occurs, or the filename turns out to be a directory.
   */
  public String fileToString(String filename)
    throws FileNotFoundException, IOException, SecurityException
  {
    // try to read the file
    FileInputStream fis;
    StringBuffer loadString = new StringBuffer();
    byte buf[] = new byte[BUFSIZE];
    int bytes;

    fis = new FileInputStream(filename);

    try {
      while ((bytes = fis.read(buf)) > 0) {
	for (int i=0; i<bytes; i++)
	  loadString.append((char)buf[i]);
      }
    } catch (IOException ex) {
      try { fis.close(); } catch(IOException e) {}
      throw ex;
    }

    fis.close();

    return loadString.toString();
  }

  /**
   * Puts a string out to a file.  Returns true on success.
   */
  public void stringToFile(String saveString, String filename)
    throws FileNotFoundException, IOException, SecurityException
  {
    FileOutputStream fos = new FileOutputStream(filename);

    //
    // It turns out to be much faster to allocate a new byte array and
    // and write out the string in chunks.  Writing each byte out one
    // at a time is about 10 times slower.
    //
    int len = saveString.length();
    byte buf[] = new byte[BUFSIZE];

    try {
      for (int i=0; i<len; i+=BUFSIZE) {
	int readlen = Math.min(BUFSIZE, len-i);
	saveString.getBytes(i, i+readlen, buf, 0);
	fos.write(buf, 0, readlen);
      }
    }
    catch (IOException ex) {
      try { fos.close(); } catch(IOException e) {}
      throw ex;
    }

    fos.close();
  }

  /**
   * Compares two objects and returns if they are equal.  Will work with
   * null objects
   */

  public boolean isEqual(Object o1, Object o2) {
    if (o1 == null) {
      return (o2 == null);
    }
    else {
      return (o1.equals(o2));
    }
  }

  /**
   * Quicksort for strings.  Could not get James Gosling's example working
   * properly, or the "fixed" example, so wrote my own using algorithms
   * book.
   */

  public void qsort(String[] list) {
    quicksort(list, 0, list.length-1);
  }

  private void quicksort(String[] list, int p, int r) {
    if (p < r) {
      int q = partition(list,p,r);
      if (q == r) {
	q--;
      }
      quicksort(list,p,q);
      quicksort(list,q+1,r);
    }
  }

  private int partition (String[] list, int p, int r) {
    String pivot = list[p];
    int lo = p;
    int hi = r;
    
    while (true) {
      while (list[hi].compareTo(pivot) >= 0 &&
	     lo < hi) {
	hi--;
      }
      while (list[lo].compareTo(pivot) < 0 &&
	     lo < hi) {
	lo++;
      }
      if (lo < hi) {
	String T = list[lo];
	list[lo] = list[hi];
	list[hi] = T;
      }
      else return hi;
    }
  }      

  /**
   * A workaround routine for the Windows95 pack bug in 1.0.2
   *
   * @param c The component to pack
   */
  static public void pack(Window c) {
    c.pack();
    if (Global.isWindows95() || Global.isWindowsNT()) {
      Thread.yield();
      c.pack();
    }
  }
    
}


/**
 * A Work-around filter.
 *
 * Transparent gifs don't display properly on Windows 95.  The work-around
 * is to replace transparent pixels with the background color of the
 * component they're being displayed in before drawing them.
 */
class TransFilter extends RGBImageFilter {
  private Color bg;
  private Component comp;
 
  TransFilter(Component comp) {
    if (comp == null)
      throw new Error("null comp argument to TransWorkAroundFilter");

    this.comp = comp;
    canFilterIndexColorModel = false;
  }
 
  public int filterRGB(int x, int y, int rgb) {
    if (bg == null)
      bg = comp.getBackground();

    if ((rgb & 0xff000000) == 0)
      return(bg.getRGB());
    else
      return(rgb);
  }

}
