/*
 *        Copyright (C) 1996  Active Software, Inc.
 *                  All rights reserved.
 *
 * @(#) FileDialogShadow.java 1.11 - last change made 08/02/96
 */

package sunsoft.jws.visual.rt.shadow.java.awt;

import sunsoft.jws.visual.rt.type.ModeEnum;
import sunsoft.jws.visual.rt.base.Global;

import java.awt.FileDialog;
import java.io.FilenameFilter;

/**
 * Wraps an AWT widget.  The attributes available for this
 * class are listed below.  In the type column, type names beginning
 * with "sunsoft.jws.visual.rt" have been abbreviated to begin with "rt".
 *
 * <pre>
    name            type                      default value
    -----------------------------------------------------------------------
    directory       java.lang.String          null
    file            java.lang.String          null 
  + modal           java.lang.Boolean         true 
    mode            rt.type.ModeEnum	      load
  + title           java.lang.String          "Unnamed File Dialog" 
  + visible         java.lang.Boolean         false 
 * </pre>
 *
 * + = this attribute overrides one inherited from an ancestor class.
 * <p>
 * mode: can be "load" or "save", and determines whether the file dialog
 * is to be used for opening or saving a file, repectively (the text in
 * the lower left button of the file dialog will read either "Open" or
 * "Save" depending on the mode.)
 * <p>
 * Check the super class for additional attributes.
 *
 * @see FileFialog
 * @version 	1.11, 08/02/96
 */
public class FileDialogShadow extends DialogShadow {

  public FileDialogShadow() {
    attributes.add("title", "java.lang.String",
		   "File Dialog (not viewable)", NOEDITOR);
    attributes.add("directory", "java.lang.String", null, NOEDITOR);
    attributes.add("file", "java.lang.String", null, NOEDITOR);

    // REMIND: This is commented out because Java WorkShop's version of
    // the JDK does not implement the FileDialog.setFilenameFilter()
    // method.
    // attributes.add("filenameFilter", "java.io.FilenameFilter",
    //		      null, HIDDEN);

    // which mode, load or save?
    attributes.add("mode", "sunsoft.jws.visual.rt.type.ModeEnum",
		   new ModeEnum(FileDialog.LOAD), CONSTRUCTOR);

    // Always modal
    attributes.add("modal", "java.lang.Boolean", Boolean.TRUE, HIDDEN);

    // Always resizeable
    attributes.add("resizable", "java.lang.Boolean", Boolean.TRUE, HIDDEN);

    // Not visible by default
    attributes.add("visible", "java.lang.Boolean", Boolean.FALSE,
		   HIDDEN | NONBODY);
  }

  public void showComponent() {
    checkCreate();
    ((FileDialog)body).show();
    set("visible", Boolean.FALSE);
  }

  protected Object getOnBody(String key) {
    FileDialog fd = (FileDialog)body;

    if (key.equals("directory")) {
      return fd.getDirectory();
    }
    else if (key.equals("file")) {
      String file = fd.getFile();

      // WORK-AROUND: remove the .*.* that Win95 puts on filename
      if (file != null && file.endsWith(".*.*"))
	file = file.substring(0, file.length() - 4);

      return file;
    }

    // REMIND: This is commented out because Java WorkShop's version of
    // the JDK does not implement the FileDialog.setFilenameFilter()
    // method.
    // else if (key.equals("filenameFilter"))
    //   return fd.getFilenameFilter();

    else if (key.equals("mode"))
      return(new ModeEnum(fd.getMode()));
    else
      return(super.getOnBody(key));
  }

  protected void setOnBody(String key, Object value) {
    FileDialog fd = (FileDialog)body;

    if (key.equals("directory"))
      fd.setDirectory((String)value);
    else if (key.equals("file"))
      fd.setFile((String)value);

    // REMIND: This is commented out because Java WorkShop's version of
    // the JDK does not implement the FileDialog.setFilenameFilter()
    // method.
    // else if (key.equals("filenameFilter"))
    //   fd.setFilenameFilter((FilenameFilter)value);

    else if (key.equals("mode")) {
      // Do nothing ; constructor attribute
    }
    else
      super.setOnBody(key, value);
  }

  public void createBody() {
    dialogFrame = getFrame();
    String title = (String) getFromTable("title");
    int mode = ((ModeEnum) getFromTable("mode")).intValue();

    body = new FileDialog(dialogFrame, title, mode);
  }

  /**
   * Disposes of the AWT top-level window so that window system
   * resources are reclaimed.
   */
  protected void destroyBody() {
    //
    // Workaround for Motif bug during removeNotify
    //
    // java.lang.NullPointerException
    //        at sun.awt.motif.MComponentPeer.dispose(MComponentPeer.java:175)
    //        at sun.awt.motif.MDialogPeer.dispose(MDialogPeer.java:73)
    //        at java.awt.Component.removeNotify(Component.java:1037)
    //        at java.awt.Container.removeNotify(Container.java:385)
    //        at java.awt.Window.dispose(Window.java:127)
    //        at sunsoft.jws.visual.rt.shadow.java.awt.
    //           FileDialogShadow.destroyBody(FileDialogShadow.java:103)
    //
    if (!Global.isMotif())
      ((FileDialog)body).dispose();

    body = null;
  }
}
