/*
 *        Copyright (C) 1996  Active Software, Inc.
 *                  All rights reserved.
 *
 * @(#) @(#) ScrollbarShadow.java 1.29 - last change made 07/19/96 
 */

package sunsoft.jws.visual.rt.shadow.java.awt;

import sunsoft.jws.visual.rt.awt.GBConstraints;
import sunsoft.jws.visual.rt.awt.VJScrollbar;
import sunsoft.jws.visual.rt.type.OrientationEnum;

/**
 * Wraps an AWT widget.  The attributes available for this
 * class are listed below.  In the type column, type names beginning
 * with "sunsoft.jws.visual.rt" have been abbreviated to begin with "rt".
 *
 * <pre>
    name            type                      default value
    -----------------------------------------------------------------------
    lineIncrement   java.lang.Integer         10 
    maximum         java.lang.Integer         100 
    minimum         java.lang.Integer         0 
    orientation     rt.type.OrientationEnum   vertical 
    pageIncrement   java.lang.Integer         10 
    value           java.lang.Integer         0 
    visiblePageSize java.lang.Integer         50 
 * </pre>
 *
 * Check the super class for additional attributes.
 *
 * @see VJScrollbar
 * @see Scrollbar
 * @version 	1.29, 07/19/96
 */
public class ScrollbarShadow extends ComponentShadow {

  public ScrollbarShadow() {
    attributes.add("orientation",
		   "sunsoft.jws.visual.rt.type.OrientationEnum",
		   new OrientationEnum(VJScrollbar.VERTICAL),
		   CONSTRUCTOR);
    attributes.add("lineIncrement","java.lang.Integer", new Integer(10), 0);
    attributes.add("maximum","java.lang.Integer", new Integer(100), 0);
    attributes.add("minimum","java.lang.Integer",new Integer(0), 0);
    attributes.add("pageIncrement","java.lang.Integer", new Integer(10), 0);
    attributes.add("value","java.lang.Integer", new Integer(0), 0);
    attributes.add("visiblePageSize","java.lang.Integer",new Integer(50), 0);

    GBConstraints c = (GBConstraints)get("GBConstraints");
    c.fill = GBConstraints.VERTICAL;
    attributes.add("GBConstraints",
		   "sunsoft.jws.visual.rt.awt.GBConstraints", c);
  }

  /**
   * A change in the orientation of a scrollbar should also result
   * in a change of the fill mode in its GBConstraints.
   */
  private void setProperFill(int orientation) {
    GBConstraints c = (GBConstraints)get("GBConstraints");

    if (c == null)
      throw new Error("scrollbar does not have layout constraints!");
    else if ((c.fill == GBConstraints.VERTICAL
	      && orientation == VJScrollbar.HORIZONTAL) ||
	     (c.fill == GBConstraints.HORIZONTAL
	      && orientation == VJScrollbar.VERTICAL)) {
      c = (GBConstraints) c.clone();
      if (orientation == VJScrollbar.VERTICAL)
	c.fill = GBConstraints.VERTICAL;
      else 
	c.fill = GBConstraints.HORIZONTAL;
      set("GBConstraints", c);
    }
  }

  protected Object getOnBody(String key) {
    VJScrollbar sbar = (VJScrollbar)body;

    if (key.equals("orientation"))
      return(new OrientationEnum(sbar.getOrientation()));
    else if (key.equals("lineIncrement"))
      return(new Integer(sbar.getLineIncrement()));
    else if (key.equals("maximum"))
      return(new Integer(sbar.getMaximum()));
    else if (key.equals("minimum"))
      return(new Integer(sbar.getMinimum()));
    else if (key.equals("pageIncrement"))
      return(new Integer(sbar.getPageIncrement()));
    else if (key.equals("value"))
      return(new Integer(sbar.getValue()));
    else if (key.equals("visiblePageSize"))
      return(new Integer(sbar.getVisible()));
    else
      return(super.getOnBody(key));
  }

  protected void setOnBody(String key, Object value) {
    VJScrollbar sbar = (VJScrollbar)body;

    if (key.equals("lineIncrement")) {
      sbar.setLineIncrement( ((Integer) value).intValue());
    }
    else if (key.equals("maximum")) {
      sbar.setValues(sbar.getValue(), sbar.getVisible(),
		     sbar.getMinimum(), ((Integer) value).intValue());
    }
    else if (key.equals("minimum")) {
      sbar.setValues(sbar.getValue(), sbar.getVisible(),
		     ((Integer) value).intValue(), sbar.getMaximum());
    }
    else if (key.equals("pageIncrement")) {
      sbar.setPageIncrement( ((Integer) value).intValue());
    }
    else if (key.equals("value")) {
      sbar.setValue (((Integer) value).intValue());
    }
    else if (key.equals("visiblePageSize")) {
      sbar.setValues(sbar.getValue(),
		     ((Integer) value).intValue(),
		     sbar.getMinimum(), sbar.getMaximum());
    }
    else if (key.equals("orientation")) {
      setProperFill(((OrientationEnum) value).intValue());
    }
    else {
      super.setOnBody(key, value);
    }
  }

  public void createBody() {
    int orientation =
      ((OrientationEnum) getFromTable("orientation")).intValue();

    // Use the VJScrollbar because it has workarounds for the broken
    // scrollbars on Windows.
    body = new VJScrollbar(orientation);
  }
}
