/*
 *        Copyright (C) 1996  Active Software, Inc.
 *                  All rights reserved.
 *
 * @(#) AttributeListConverter.java 1.25 - last change made 07/30/96
 */

package sunsoft.jws.visual.rt.type;

import sunsoft.jws.visual.rt.base.*;
import java.util.*;

/**
 * The class converts AttributeLists to strings and back again.
 *
 * @see AttributeList
 * @version 1.25, 07/30/96
 */
public class AttributeListConverter extends Converter {

  private static AttributeConverter ac = new AttributeConverter();

  /**
   * Returns true if the attribute is one that should be placed in a
   * save file.  Include only non-default attributes (and skip the
   * name attribute.)
   */
  private boolean shouldConvertAttribute(Attribute a) {
    return(a.isModified() && !a.getName().equals("name") && !isTransient(a));
  }

  private boolean isTransient(Attribute a) {
    return ((a.getFlags() & (AttributeManager.TRANSIENT |
			     AttributeManager.READONLY)) != 0);
  }

  /**
   * Converts an AttributeList to a string.
   *
   * @param obj AttributeList to convert
   * @param buf buffer to which to add the string
   */
  public void convertToString(Object obj, StringBuffer buf) {
    if (obj != null) {
      AttributeList l = (AttributeList) obj;
      for (Enumeration e = l.elements(); e.hasMoreElements(); ) {
	Attribute a = (Attribute) e.nextElement();
	if (shouldConvertAttribute(a)) {
	  indent(buf);
	  ac.convertToString(a, buf);
	  newline(buf);
	}
      }
    }
  }

  /**
   * Call convertFromString that takes more arguments instead.
   *
   * @exception Error when called
   */
  public Object convertFromString(String s) {
    throw new Error("AttributeListConverter cannot work without a shadow object argument");
  }

  /**
   * Converts a string to an AttributeList.
   *
   * @param version description file version
   * @param mgr AttributeManager from which the attribute list came
   * @param s string to convert
   * @return string representation of AttributeList
   * @exception ParseError when there is a problem with the string
   */
  public String convertFromString(double version, AttributeManager mgr,
				  String s) {
    String children = null;
    String type = null, key = null, value = null;
    boolean isChildren;

    if (s == null || mgr == null)
      return null;

    Enumeration e;
    if (version >= 3)
      e = ListParser.getListElements(s, 3);
    else
      e = ListParser.getListElements(s, 2);

    try {
      while (e.hasMoreElements()) {
	type = null;
	key = null;
	value = null;

	if (version >= 3) {
	  type = (String)e.nextElement();
	  key = (String)e.nextElement();
	  value = (String)e.nextElement();
	  isChildren = (type.equals("child") && key.equals("list"));
	}
	else {
	  key = (String)e.nextElement();
	  value = (String)e.nextElement();
	  isChildren = (key.equals("children"));
	}
	
	if (isChildren)
	  children = value;
	else
	  ac.convertFromString(version, mgr, type, key, value);
      }
    }
    catch (NoSuchElementException ex) {
      throw new ParseException(Global.newline() +
			       "    Incomplete attribute line:" +
			       Global.newline() + "      type = " + type +
			       Global.newline() + "      key = " + key +
			       Global.newline() + "      value = " + value);
    }

    return children;
  }

  /**
   * Returns true if the attribute is one that should be placed in a
   * generated code file.  Include only non-default attributes.
   */
  private boolean shouldGenerateAttribute(Attribute a) {
    return(a.isModified() && !isTransient(a) &&
	   !a.getName().equals("operations"));
  }

  /**
   * Converts an AttributeManager's AttributeList to code.
   *
   * Skips the name attribute because it is generated separately in GenCode.
   * This is done because during initialization for the generated root, the
   * name must be set before add is called.  All the other attributes
   * must be set after add is called.
   *
   * @param amName name of the AttributeManager
   * @param list the list to convert
   * @param indent number of spaces to indent on each line
   * @param buf buffer to which to add the code
   */
  public void convertToCodeBlock(String amName, AttributeList list,
				 int indent, StringBuffer buf) {
    Enumeration e = list.elements();
    while (e.hasMoreElements()) {
      Attribute a = (Attribute) e.nextElement();
      if (shouldGenerateAttribute(a) && (!a.getName().equals("name"))) {
	ac.convertToCodeBlock(amName, a, indent, buf);
      }
    }
  }
}
