/*
 *        Copyright (C) 1996  Active Software, Inc.
 *                  All rights reserved.
 *
 * @(#) ColorStore.java 1.11 - last change made 07/18/96
 */

package sunsoft.jws.visual.rt.type;

import sunsoft.jws.visual.rt.base.Global;

import java.awt.Color;
import java.util.Hashtable;
import java.util.Enumeration;

/**
 * Stores colors by name and rgb value.  Names are always stored in
 * lower case, and searches are done after converting the search key
 * into lower case.  Two hashtables are used so colors can be accessed
 * efficiently either by color or name.
 *
 * @version 1.11, 07/18/96
 */
public class ColorStore {
  // COLOR <-> NAME HASHTABLES

  /**
   * Storage place for Color RGB values and named colors.
   * It's like a two-way hashtable.
   */
  private static ColorStore defaultColorStore;

  /**
   * Returns a reference to the single global instance of ColorStore.
   */
  public static ColorStore getDefaultColorStore() {
    initColorStore();
    return(defaultColorStore);
  }

  private static void initColorStore() {
    if (defaultColorStore != null)
      return;
    defaultColorStore = new ColorStore();
    defaultColorStore.initDefaultColorList();
  }


  /**
   * Storage place for Color RGB values (keys) and named colors (values)
   */
  private Hashtable rgbkeys;

  /**
   * Storage place for named colors (keys) Color RGB values (values)
   */
  private Hashtable namekeys;

  /**
   * A list sorted by colorName (a-z)
   */

  private String[] colorNameArray;

  public ColorStore() {
    rgbkeys = new Hashtable();
    namekeys = new Hashtable();
  }

  /**
   * Returns the name of a color.  If the color name is not in
   * our table, return #RRGGBB (a string of hex numbers).
   */
  public String getColorName(Color c) {
    return (String) rgbkeys.get(c);
  }

  /**
   * Given the name of a color, return the Color corresponding to it.
   */
  public Color getColor(String name) {
    if (name == null) {
      return null;
    }
    else {
      return((Color) namekeys.get(name.toLowerCase()));
    }
  }

  /**
   * Given a Color, return the name of the color listed in the
   * ColorStore closest to that color.
   */
  public String getClosestColorName(Color c) {

    Enumeration e;
    int difference = 255*3, newDifference;
    Color bestColor = null, cKey;
    
    if (rgbkeys.containsKey(c)) {
      return(getColorName(c));
    }
    
    e = rgbkeys.keys();
    while (e.hasMoreElements()) {
      cKey = (Color) e.nextElement();
      
      if ((newDifference = 
	   (Math.abs(cKey.getRed() - c.getRed())) +
	   (Math.abs(cKey.getGreen() - c.getGreen())) +
	   (Math.abs(cKey.getBlue() - c.getBlue()))) <=
	  difference)
	{
	  difference = newDifference;
	  bestColor = cKey;
	}
    }
    return getColorName(bestColor);
  }

  /**
   * Adds the color/name pair to the color store is it is not already
   * there and returns the name of the color.  If the color is already
   * registered, the previous name of the color is returned.
   */
  public String add(Color c, String name) {
    if (rgbkeys.containsKey(c)) {
      return(getColorName(c));
    }
    else {
      rgbkeys.put(c, name.toLowerCase());
      namekeys.put(name, c);
      colorNameArray = null;
      return(name);
    }
  }

  /**
   * Returns the list of Colors named in the ColorStore.
   */
  public String[] getColorList() {
    int index;

    if (colorNameArray == null) {
      colorNameArray = new String[rgbkeys.size()];

      index = 0;
      Enumeration e = namekeys.keys();
	
      while (e.hasMoreElements()) {
	String colorName = (String)(e.nextElement());
	colorNameArray[index] = colorName;
	index++;
      }
      
      Global.util.qsort(colorNameArray);
    }
    return colorNameArray;
  }

  private void initDefaultColorList() {
    // AWT's gray:

    add(new Color(128,128,128), "medium gray");

    // Colors from rgb.txt:

    add(new Color(133,133,133), "gray52");
    add(new Color(122,122,122), "gray48");
    add(new Color(186,186,186), "gray73");
    add(new Color(94,94,94), "gray37");
    add(new Color(196,196,196), "gray77");
    add(new Color(84,84,84), "gray33");
    add(new Color(66,66,66), "gray26");
    add(new Color(48,48,48), "gray19");
    add(new Color(92,92,92), "gray36");
    add(new Color(74,74,74), "gray29");
    add(new Color(163,163,163), "gray64");
    add(new Color(28,28,28), "gray11");
    add(new Color(46,46,46), "gray18");
    add(new Color(173,173,173), "gray68");
    add(new Color(36,36,36), "gray14");
    add(new Color(54,54,54), "gray21");
    add(new Color(194,194,194), "gray76");
    add(new Color(150,150,150), "gray59");
    add(new Color(171,171,171), "gray67");
    add(new Color(3,3,3), "gray1");
    add(new Color(181,181,181), "gray71");
    add(new Color(20,20,20), "gray8");
    add(new Color(10,10,10), "gray4");
    add(new Color(207,207,207), "gray81");
    add(new Color(99,99,99), "gray39");
    add(new Color(237,237,237), "gray93");
    add(new Color(79,79,79), "gray31");
    add(new Color(87,87,87), "gray34");
    add(new Color(204,204,204), "gray80");
    add(new Color(214,214,214), "gray84");
    add(new Color(235,235,235), "gray92");
    add(new Color(245,245,245), "gray96");
    add(new Color(8,8,8), "gray3");
    add(new Color(71,71,71), "gray28");
    add(new Color(201,201,201), "gray79");
    add(new Color(61,61,61), "gray24");
    add(new Color(51,51,51), "gray20");
    add(new Color(33,33,33), "gray13");
    add(new Color(15,15,15), "gray6");
    add(new Color(117,117,117), "gray46");
    add(new Color(31,31,31), "gray12");
    add(new Color(13,13,13), "gray5");
    add(new Color(105,105,105), "gray41");
    add(new Color(148,148,148), "gray58");
    add(new Color(252,252,252), "gray99");
    add(new Color(115,115,115), "gray45");
    add(new Color(158,158,158), "gray62");
    add(new Color(179,179,179), "gray70");
    add(new Color(240,240,240), "gray94");
    add(new Color(189,189,189), "gray74");
    add(new Color(250,250,250), "gray98");
    add(new Color(199,199,199), "gray78");
    add(new Color(145,145,145), "gray57");
    add(new Color(70,130,180), "steel blue");
    add(new Color(233,150,122), "darksalmon");
    add(new Color(100,149,237), "cornflowerblue");
    add(new Color(30,144,255), "dodgerblue");
    add(new Color(238,221,130), "lightgoldenrod");
    add(new Color(205,133,63), "peru");
    add(new Color(154,205,50), "yellow green");
    add(new Color(175,238,238), "paleturquoise");
    add(new Color(0,100,0), "darkgreen");
    add(new Color(255,0,0), "red");
    add(new Color(160,82,45), "sienna");
    add(new Color(143,188,143), "darkseagreen");
    add(new Color(255,228,196), "bisque");
    add(new Color(147,112,219), "mediumpurple");
    add(new Color(148,0,211), "dark violet");
    add(new Color(124,252,0), "lawn green");
    add(new Color(119,136,153), "lightslategray");
    add(new Color(230,230,250), "lavender");
    add(new Color(248,248,255), "ghostwhite");
    add(new Color(176,224,230), "powderblue");
    add(new Color(218,165,32), "goldenrod");
    add(new Color(255,228,181), "moccasin");
    add(new Color(255,228,225), "mistyrose");
    add(new Color(255,255,224), "light yellow");
    add(new Color(255,99,71), "tomato");
    add(new Color(245,255,250), "mintcream");
    add(new Color(255,255,0), "yellow");
    add(new Color(138,43,226), "blueviolet");
    add(new Color(32,178,170), "light sea green");
    add(new Color(255,240,245), "lavender blush");
    add(new Color(127,255,212), "aquamarine");
    add(new Color(165,42,42), "brown");
    add(new Color(219,112,147), "pale violet red");
    add(new Color(240,255,255), "azure");
    add(new Color(107,142,35), "olivedrab");
    add(new Color(47,79,79), "darkslategray");
    add(new Color(139,69,19), "saddle brown");
    add(new Color(160,32,240), "purple");
    add(new Color(186,85,211), "medium orchid");
    add(new Color(240,255,240), "honeydew");
    add(new Color(176,196,222), "lightsteelblue");
    add(new Color(0,0,0), "black");
    add(new Color(64,224,208), "turquoise");
    add(new Color(255,127,80), "coral");
    add(new Color(184,134,11), "darkgoldenrod");
    add(new Color(60,179,113), "mediumseagreen");
    add(new Color(210,180,140), "tan");
    add(new Color(255,222,173), "navajo white");
    add(new Color(46,139,87), "sea green");
    add(new Color(190,190,190), "gray");
    add(new Color(123,104,238), "mediumslateblue");
    add(new Color(250,250,210), "light goldenrod yellow");
    add(new Color(135,206,235), "sky blue");
    add(new Color(132,112,255), "lightslateblue");
    add(new Color(250,240,230), "linen");
    add(new Color(218,112,214), "orchid");
    add(new Color(0,0,128), "navy blue");
    add(new Color(253,245,230), "old lace");
    add(new Color(240,248,255), "aliceblue");
    add(new Color(72,209,204), "mediumturquoise");
    add(new Color(255,140,0), "dark orange");
    add(new Color(72,61,139), "dark slate blue");
    add(new Color(255,160,122), "light salmon");
    add(new Color(221,160,221), "plum");
    add(new Color(238,130,238), "violet");
    add(new Color(34,139,34), "forest green");
    add(new Color(0,255,127), "springgreen");
    add(new Color(85,107,47), "darkolivegreen");
    add(new Color(238,232,170), "pale goldenrod");
    add(new Color(0,255,255), "cyan");
    add(new Color(245,245,220), "beige");
    add(new Color(255,250,240), "floralwhite");
    add(new Color(255,218,185), "peach puff");
    add(new Color(50,205,50), "limegreen");
    add(new Color(152,251,152), "palegreen");
    add(new Color(240,230,140), "khaki");
    add(new Color(188,143,143), "rosybrown");
    add(new Color(244,164,96), "sandybrown");
    add(new Color(189,183,107), "darkkhaki");
    add(new Color(25,25,112), "midnight blue");
    add(new Color(255,235,205), "blanched almond");
    add(new Color(224,255,255), "light cyan");
    add(new Color(255,182,193), "lightpink");
    add(new Color(95,158,160), "cadetblue");
    add(new Color(106,90,205), "slate blue");
    add(new Color(245,222,179), "wheat");
    add(new Color(255,69,0), "orangered");
    add(new Color(127,255,0), "chartreuse");
    add(new Color(255,255,255), "white");
    add(new Color(65,105,225), "royalblue");
    add(new Color(173,216,230), "light blue");
    add(new Color(255,250,250), "snow");
    add(new Color(255,245,238), "seashell");
    add(new Color(250,128,114), "salmon");
    add(new Color(255,255,240), "ivory");
    add(new Color(211,211,211), "lightgray");
    add(new Color(255,239,213), "papaya whip");
    add(new Color(153,50,204), "dark orchid");
    add(new Color(208,32,144), "violet red");
    add(new Color(255,248,220), "cornsilk");
    add(new Color(255,105,180), "hotpink");
    add(new Color(255,165,0), "orange");
    add(new Color(255,0,255), "magenta");
    add(new Color(176,48,96), "maroon");
    add(new Color(178,34,34), "firebrick");
    add(new Color(240,128,128), "lightcoral");
    add(new Color(220,220,220), "gainsboro");
    add(new Color(216,191,216), "thistle");
    add(new Color(255,192,203), "pink");
    add(new Color(135,206,250), "light sky blue");
    add(new Color(210,105,30), "chocolate");
    add(new Color(173,255,47), "green yellow");
    add(new Color(112,128,144), "slate gray");
    add(new Color(0,0,255), "blue");
    add(new Color(0,255,0), "green");
    add(new Color(0,191,255), "deepskyblue");
    add(new Color(255,250,205), "lemon chiffon");
    add(new Color(0,206,209), "dark turquoise");
    add(new Color(222,184,135), "burlywood");
    add(new Color(199,21,133), "mediumvioletred");
    add(new Color(250,235,215), "antique white");
    add(new Color(255,215,0), "gold");
    add(new Color(255,20,147), "deep pink");
    add(new Color(205,92,92), "indianred");
    add(new Color(0,250,154), "medium spring green");
  }
}
