/*
 *        Copyright (C) 1996  Active Software, Inc.
 *                  All rights reserved.
 *
 * @(#) OpAction.java 1.14 - last change made 07/30/96
 */

package sunsoft.jws.visual.rt.type;

import sunsoft.jws.visual.rt.base.*;
import java.util.Hashtable;

/**
 * Stores the action that should be triggered in an operation whose
 * filter has matched an event or message.  This class also simultaneously
 * acts as its own converter.
 *
 * @see Op
 * @version 	1.14, 07/30/96
 */
public class OpAction extends Converter implements Cloneable {
  // Register the converter
  static {
    Converter.addConverter("sunsoft.jws.visual.rt.type.OpAction",
			   "sunsoft.jws.visual.rt.type.OpAction");
  }

  /**
   * Action is to set an attribute.
   */
  public static final int ATTRIBUTE = 0;

  /**
   * Action is to send a message.
   */
  public static final int MESSAGE = 1;

  /**
   * Action is to execute some code.
   */
  public static final int CODE = 2;

  // Action Details
  public static final int SHOW = 1;
  public static final int HIDE = 2;
  public static final int EXIT = 3;

  /**
   * Used when setting the action target, specifies that the target is
   * permanently set (i.e. always use what is stored in this objects
   * target memeber.
   *
   * @see #target
   */
  public static final int CONSTANT = 10;

  /**
   * Used when setting the action target, specifies that the target should
   * be the same as the target of the event or message that triggered this
   * operation.
   */  
  public static final int TARGET = 11;

  /**
   * Used when setting the action target, specifies that the target should
   * be whatever is in the arg variable of the message or event.
   */
  public static final int ARG = 12;

  /**
   * Action type, ATTRIBUTE, MESSAGE, or CODE.
   */
  public int actionType;

  /**
   * Action detail, used by the action editor to keep track of simple actions.
   */
  public int actionDetail;

  /**
   * Action target, used when targetSource == CONSTANT.
   */
  public AMRef target;

  /**
   * Specifies how the target or the action is determined, is either
   * CONSTANT, TARGET, or ARG.
   */
  public int targetSource;

  /**
   * Message or Attribute name.
   */
  public String name;

  /**
   * Message arg or Attribute value, used when valueSource == CONSTANT.
   */
  public Object value;

  /**
   * Where is the arg/value from for setting an attribute,
   * is either CONSTANT, TARGET, or ARG.
   */
  public int valueSource;

  /**
   * Message type, used when the action is to send a message.
   */
  public String type;

  /**
   * Message target name, used when the action is to send a message.
   */
  public String targetName;

  /**
   * The code associate with the action, in string form.
   */
  public String code;

  /**
   * Constructs an instance where targetSource and valueSource are CONSTANT.
   */
  public OpAction() {
    this.targetSource = CONSTANT;
    this.valueSource = CONSTANT;
  }

  /**
   * Constructs an instance where targetSource and valueSource are CONSTANT.
   */
  public OpAction(int actionType, AMRef target, String name, Object value) {
    this.actionType = actionType;
    this.target = target;
    this.name = name;
    this.targetSource = CONSTANT;
    this.value = value;
    this.valueSource = CONSTANT;
  }

  /**
   * Performs the action.
   *
   * @param target the target of the action if targetSource == TARGET
   * @param arg the argument of the message or event and the target of the action if targetSource == ARG
   * @param scope the root tree in which to search for the target
   */
  void invoke(Object target, Object arg, Root scope) {
    if (name == null)
      return;

    // Lookup the target
    AttributeManager mgr = lookupTarget(target, arg, scope);
    if (mgr == null)
      return;

    // Lookup the value
    Object value = lookupValue(target, arg, scope);

    // Perform the action
    performAction(mgr, value);
  }

  private AttributeManager lookupTarget(Object target, Object arg,
					Root scope) {
    AttributeManager mgr = null;

    switch (targetSource) {
    case CONSTANT:
      if (this.target != null)
	mgr = this.target.getRef(scope);
      break;

    case TARGET:
      mgr = (AttributeManager)target;
      break;

    case ARG:
      mgr = (AttributeManager)arg;
      break;
    }

    return mgr;
  }

  private Object lookupValue(Object target, Object arg, Root scope) {
    Object value = null;

    switch (valueSource) {
    case CONSTANT:
      value = this.value;
      break;

    case TARGET:
      value = target;
      break;

    case ARG:
      value = arg;
      break;
    }

    return value;
  }

  private void performAction(AttributeManager target, Object value) {
    switch (actionType) {
    case ATTRIBUTE:
      if (name != null)
	target.set(name, value);
      break;

    case MESSAGE:
      if (name != null)
	target.postMessage(new Message(target, targetName, type, name, value));
      break;

    case CODE:
      System.out.println("CODE: " + code);
      break;
    }
  }

  /**
   * Returns a new copy of this action.
   */
  public Object clone() {
    try {
      return super.clone();
    } catch (CloneNotSupportedException e) { 
      // this shouldn't happen, since we are Cloneable
      throw new InternalError();
    }
  }

  //
  // Code generation
  //

  /**
   * Appends the initialization code for this operation filter into
   * the buffer given.
   *
   * @param varname variable name of the operation filter
   * @param buf buffer onto which the code should be appended
   */
  public void genInitCode(StringBuffer buf, String varname) {
    // Action type
    buf.append("    ");
    buf.append(varname);
    buf.append(".actionType = ");
    buf.append("OpAction.");
    buf.append(constantToString(actionType));
    buf.append(";");
    Global.newline(buf);

    // Target
    if (target != null) {
      buf.append("    ");
      buf.append(varname);
      buf.append(".target = new AMRef(");
      ListParser.quote(target.getName(), buf, true);
      buf.append(");");
      Global.newline(buf);
    }

    // Target source
    if (targetSource != CONSTANT) {
      buf.append("    ");
      buf.append(varname);
      buf.append(".targetSource = ");
      buf.append("OpAction.");
      buf.append(constantToString(targetSource));
      buf.append(";");
      Global.newline(buf);
    }

    // Name
    if (name != null) {
      buf.append("    ");
      buf.append(varname);
      buf.append(".name = ");
      ListParser.quote(name, buf, true);
      buf.append(";");
      Global.newline(buf);
    }

    // Value source
    if (valueSource != CONSTANT) {
      buf.append("    ");
      buf.append(varname);
      buf.append(".valueSource = ");
      buf.append("OpAction.");
      buf.append(constantToString(valueSource));
      buf.append(";");
      Global.newline(buf);
    }

    // Value
    if (value != null) {
      // Lookup converter for value
      String valueType = value.getClass().getName();
      if (Converter.hasConverter(valueType)) {
	Converter c = Converter.getConverter(valueType);
	buf.append("    ");
	buf.append(varname);
	buf.append(".value = ");
	buf.append(c.convertToCode(value));
	buf.append(";");
	Global.newline(buf);
      }
    }

    // Message type
    if (type != null) {
      buf.append("    ");
      buf.append(varname);
      buf.append(".type = ");
      ListParser.quote(type, buf, true);
      buf.append(";");
      Global.newline(buf);
    }

    // Message target name
    if (targetName != null) {
      buf.append("    ");
      buf.append(varname);
      buf.append(".targetName = ");
      ListParser.quote(targetName, buf, true);
      buf.append(";");
      Global.newline(buf);
    }
  }

  //
  // String converters
  //

  private String constantToString(int c) {
    switch (c) {
    // Action consts
    case ATTRIBUTE:
      return "ATTRIBUTE";
    case MESSAGE:
      return "MESSAGE";
    case CODE:
      return "CODE";

    // Source consts
    case CONSTANT:
      return "CONSTANT";
    case TARGET:
      return "TARGET";
    case ARG:
      return "ARG";

    default:
      return null;
    }
  }

  private int stringToConstant(String s) {
    // Action consts
    if (s.equals("ATTRIBUTE"))
      return ATTRIBUTE;
    else if(s.equals("MESSAGE"))
      return MESSAGE;
    else if(s.equals("CODE"))
      return CODE;

    // Source consts
    else if(s.equals("CONSTANT"))
      return CONSTANT;
    else if(s.equals("TARGET"))
      return TARGET;
    else if(s.equals("ARG"))
      return ARG;

    else
      return -1;
  }

  public void convertToString(Object obj, StringBuffer buf) {
    OpAction a = (OpAction)obj;

    // Open brace
    buf.append("{");
    newline(buf);
    incrIndent();

    // Action type
    indent(buf);
    buf.append("actionType ");
    buf.append(constantToString(a.actionType));
    newline(buf);

    // Action detail
    if (a.actionDetail != 0) {
      indent(buf);
      buf.append("actionDetail ");
      buf.append(Integer.toString(a.actionDetail));
      newline(buf);
    }

    // Action target
    if (a.target != null) {
      indent(buf);
      buf.append("target ");
      buf.append(a.target.getName());
      newline(buf);
    }

    // Target source
    if (a.targetSource != CONSTANT) {
      indent(buf);
      buf.append("targetSource ");
      buf.append(constantToString(a.targetSource));
      newline(buf);
    }

    // Name
    if (a.name != null) {
      indent(buf);
      buf.append("name ");
      ListParser.quote(a.name, buf, false);
      newline(buf);
    }

    // Value
    if (a.value != null) {
      indent(buf);
      buf.append("value ");

      // Lookup converter for value
      String valueType = value.getClass().getName();
      if (Converter.hasConverter(valueType)) {
	Converter c = Converter.getConverter(valueType);
	ListParser.quote(c.convertToString(value), buf, false);
      }
      else {
	buf.append("null");
      }

      newline(buf);

      indent(buf);
      buf.append("valueType ");
      buf.append(valueType);
      newline(buf);
    }

    // Value source
    if (a.valueSource != CONSTANT) {
      indent(buf);
      buf.append("valueSource ");
      buf.append(constantToString(a.valueSource));
      newline(buf);
    }

    // Message type
    if (a.type != null) {
      indent(buf);
      buf.append("type ");
      ListParser.quote(a.type, buf, false);
      newline(buf);
    }

    // Message target name
    if (a.targetName != null) {
      indent(buf);
      buf.append("targetName ");
      ListParser.quote(a.targetName, buf, false);
      newline(buf);
    }

    // Code
    if (a.code != null) {
      indent(buf);
      buf.append("code ");
      ListParser.list(a.code, buf);
      newline(buf);
    }

    // Close brace
    decrIndent();
    indent(buf);
    buf.append("}");
  }

  public Object convertFromString(String s) {
    OpAction action = new OpAction();
    convertFromString(s, action);
    return action;
  }

  public void convertFromString(String s, OpAction a) {
    Hashtable table = ListParser.makeListTable(s);
    String val;

    // Action type
    val = (String)table.get("actionType");
    a.actionType = stringToConstant(val);

    // Action detail
    val = (String)table.get("actionDetail");
    if (val != null) {
      try {
	a.actionDetail = Integer.parseInt(val);
      }
      catch (NumberFormatException ex) {
	throw new ParseException("Number format exception: " + val);
      }
    }

    // Action target
    val = (String)table.get("target");
    if (val != null)
      a.target = new AMRef(val);

    // Target source
    val = (String)table.get("targetSource");
    if (val != null)
      a.targetSource = stringToConstant(val);

    // Name
    a.name = (String)table.get("name");

    // Value
    val = (String)table.get("value");
    if (val != null) {
      if (val.equals("null")) {
	a.value = null;
      }
      else {
	String valueType = (String)table.get("valueType");
	if (valueType == null) {
	  throw new ParseException("Got a value without a valueType: " + s);
	}

	if (!Converter.hasConverter(valueType)) {
	  throw new ParseException("Could not find converter for \"" +
				   valueType + "\"");
	}

	a.value = Converter.getConverter(valueType).convertFromString(val);
      }
    }

    // Value source
    val = (String)table.get("valueSource");
    if (val != null)
      a.valueSource = stringToConstant(val);

    // Message type
    a.type = (String)table.get("type");

    // Message target name
    a.targetName = (String)table.get("targetName");

    // Code
    a.code = (String)table.get("code");
  }

  /**
   * Returns true if this type should be displayed in an editor.
   *
   * For the attribute editor, a return value of false means that the
   * the textfield will be hidden.
   *
   * @return false
   */
  public boolean viewableAsString() {
    return(false);
  }
}
