/*-
 * Javoids -- Javoids is an asteroids based game (that look nothing like the original).
 * 
 * Copyright (C) 1999-2006 Patrick Mallette
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 * 
 * I can be reached at parickmallette@rogers.com
 */
package javoids;

import static javoids.ImageMap.Name.NO_IMAGE;
import static javoids.Shapes.Shape.STAR;

import java.awt.Color;
import java.awt.geom.Area;
import java.util.Vector;

/* Shield--------------------- */
/**
 * A sprite class to provide a shield to another sprite, protecting it from damage.
 * @author mallette
 */
public final class Shield extends Sprite // attached to sprite (goes where owner goes)
{
  /** This is the version used for serializing/deserializing (storing/retrieving) this object */
  private static final long   serialVersionUID        = 1L;
  /** The maximum number of these sprites allowed in the game */
  protected final static int  MAX_NUMBER              = 1;
  /** the maximum number of lives */
  protected final static int  MAX_DEATHS              = 3;
  /** the maximum damage elvel */
  protected final static int  MAXIMUM_DAMAGE_CAPACITY = 300;
  /** the maximum duration */
  protected final static int  MAX_DURATION            = 2000;
  /** the default duration */
  protected final static int  DEFAULT_DURATION        = 200;
  /** the default rate that the sprite ages */
  protected final static int  DEFAULT_AGING_RATE      = 1;
  /** the sprite's maximum size */
  protected final static int  MAX_SIZE                = 100;
  /** the sprite's minimum size */
  protected final static int  MIN_SIZE                = 1;
  /** the sprite's default size */
  private final static int    DEFAULT_SIZE            = 20;  // This should match the size of the image!
  /** the chance the sprite will rotate */
  private final static double CHANCE_ROTATE           = 0.35;
  /** the amount to rotate by */
  private final static double ROTATION                = 30;

  /**
   * Constructor
   * @param parent the parent of this sprite
   * @param health the health information
   * @param move the movment information
   * @param _size the size this should be
   * @param shieldShape the shield's shape
   */
  public Shield(BasicSprite parent,Health health,Move move,int _size,Shapes.Shape shieldShape)
  {
    super(parent,health,move,_size,NO_IMAGE,new Vector<Area>(),new Vector<Color>());
    if (Debug.debugShield)
      this.getAreas().addAll(Shapes.getAreas(STAR,_size,Shield.MAX_SIZE));
    else
      this.getAreas().addAll(Shapes.getAreas(shieldShape,_size,Shield.MAX_SIZE));
    this.getColors().addAll(this.getColors());
    if (Math.random() < Shield.CHANCE_ROTATE)
      this.setRotation(Math.random() * Shield.ROTATION * (Math.random() < 0.5 ? -1 : 1));
    this.setDisplayAreas(true);
  }

  /**
   * @return the maximum size
   */
  public int getMaximumSize()
  {
    return Shield.MAX_SIZE;
  }

  /**
   * @return the minimum size
   */
  public int getMinimumSize()
  {
    return Shield.MIN_SIZE;
  }

  /**
   * @return the default size
   */
  public int getDefaultSize()
  {
    return Shield.DEFAULT_SIZE;
  }

  /**
   * @param _size the size to set the sprite's size to
   */
  /*
   * (non-Javadoc)
   * @see javoids.Sprite#setSize(int)
   */
  @Override
  public void setSize(int _size)
  {
    super.setSize(_size >= Shield.MIN_SIZE ? _size <= Shield.MAX_SIZE ? _size : Shield.MAX_SIZE : Shield.MIN_SIZE);
  }

  /**
   * @return a list of colors to use for drawing the areas that represent the sprite
   */
  /*
   * (non-Javadoc)
   * @see javoids.Sprite#getColors()
   */
  @Override
  public Vector<Color> getColors()
  {
    Color color = this.getColor();
    super.getColors().clear();
    for (int i = 0;i < this.getAreas().size();i++)
      super.getColors().add(new Color(color.getRed(),color.getGreen(),color.getBlue(),(int)(255 * 0.75 * (this.getMaxDuration() - this.getDuration()) / this.getMaxDuration())));
    return super.getColors();
  }

  /**
   * @return the image name for the current image
   */
  /*
   * (non-Javadoc)
   * @see javoids.Sprite#getImageNumber()
   */
  @Override
  public ImageMap.Name getImageNumber()
  {
    return BasicSprite.getPacmanGame() ? NO_IMAGE : this.getOriginalImage();
  }
}
/* Shield--------------------- */
