/*-
 * 
 * Copyright (C) 1999-2006 Patrick Mallette
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 * 
 * I can be reached at parickmallette@rogers.com
 */
package javoids;

import static javoids.BasicSprite.Gravity.NONE;
import static javoids.Health.DurationType.NORMAL;
import static javoids.ImageMap.Name.PACMAN_PACMAN;
import static javoids.ImageMap.Name.SHIP1;
import static javoids.Shapes.Shape.LANDER;
import static javoids.Shapes.Shape.POINT;
import static javoids.Shapes.Shape.TRIANGLE30;
import static javoids.Shapes.Shape.TRIANGLE45;
import static javoids.SoundMap.Sound.PACMAN_DIE;
import static javoids.SoundMap.Sound.PACMAN_EATDOT;
import static javoids.SoundMap.Sound.PACMAN_EATFRUIT;
import static javoids.SoundMap.Sound.PACMAN_EXTRALIFE;
import static javoids.SoundMap.Sound.POWERUP;
import static javoids.SoundMap.Sound.SHIPDIE;
import static javoids.SoundMap.Sound.SHIPHIT;
import static javoids.SoundMap.Sound.SHIP_EXTRALIFE;
import static javoids.SoundMap.Sound.THRUST;

import java.util.HashMap;

/* Ship----------------------- */
/**
 * A space ship sprite class.
 * @author mallette
 */
public final class Ship extends ShieldedSprite
{
  /** This is the version used for serializing/deserializing (storing/retrieving) this object */
  private static final long       serialVersionUID        = 1L;
  /** The amount to increase the maximum speed by when using afterburners */
  protected final static int      AFTERBURNER_MULTIPLIER  = 2;
  /** The maximum number of these sprites allowed in the game */
  protected final static int      MAX_NUMBER              = 4;             // must always be greater than 1 because 1 is the player!
  /** the maximum damage level */
  protected final static int      MAXIMUM_DAMAGE_CAPACITY = 100;
  /** the sprite's maximum size */
  protected final static int      MAX_SIZE                = 100;
  /** the sprite's minimum size */
  protected final static int      MIN_SIZE                = 20;
  /** the sprite's default size */
  protected final static int      DEFAULT_SIZE            = 30;            // This should match the size of the image!
  /** the default velocity */
  protected final static int      DEFAULT_VELOCITY        = 10;
  /** the defualt number of lives */
  protected final static int      DEFAULT_LIVES           = 3;
  /** the number of points that need to be obtained before a free life is rewarded */
  protected final static int      FREE_SHIP_POINTS        = 50000;
  /** the number of these sprites already created */
  private static int              count = 0;
  /** the number of free lives rewarded */
  private int                     freeLives;
  /** a mapping of item names to items that this sprite possesses */
  private HashMap<Item.Type,Item> items;
  /** the current item (GUN1 is always available) */
  private Item.Type               currentItem             = Item.Type.GUN1;
  /** the duration for the current gravity effect */
  private int                     gravityDuration;
  /** is the afterburner sound to be played? */
  private boolean                 soundAfterburner;

  /**
   * Constructor
   * @param parent the parent of this sprite
   * @param health the health information
   * @param move the movment information
   * @param _size the size this should be
   */
  public Ship(BasicSprite parent,Health health,Move move,int _size)
  {
    super(parent,health,move,new Shield(parent,new Health(Shield.MAX_DEATHS,Shield.MAXIMUM_DAMAGE_CAPACITY,Shield.DEFAULT_DURATION,Shield.DEFAULT_AGING_RATE,NORMAL),new Move(move),(int)(_size * 1.5),TRIANGLE30),_size,SHIP1,Shapes.getAreas(TRIANGLE30,_size,Ship.MAX_SIZE),Shapes.getColors(TRIANGLE30));
    Ship.count++;
    if (Ship.count == 1)
    {
      this.setPlayer(true); // player is first ship
      if (Math.random() < 0.1)
      {
        this.setShape(LANDER);
        this.setDisplayAreas(true);
      }
      else
      {
        this.setShape(TRIANGLE30);
        this.setDisplayAreas(false);
      }
    }
    else
    {
      this.setAutomaticMove(true); // "AI" ships -- AI is used very loosly here ;)
      // let some ships be other shapes
      if (Math.random() < 0.20)
      {
        this.setShape(TRIANGLE30);
        this.setDisplayAreas(true);
      }
      else if (Math.random() < 0.40)
      {
        this.setShape(TRIANGLE45);
        this.setDisplayAreas(true);
      }
      else if (Math.random() < 0.60)
      {
        this.setShape(LANDER);
        this.setDisplayAreas(true);
      }
      else
      {
        this.setShape(TRIANGLE30);
        this.setDisplayAreas(false);
      }
    }
    this.getShield().setParent(this);
    this.getShield().getAreas().clear();
    int shape = (int)(Math.random() * Shapes.Shape.values().length * 1.5) + 1;
    if (shape == 0 || shape >= Shapes.Shape.values().length)
      this.getShield().getAreas().addAll(Shapes.getAreas(POINT,this.getShield().getSize(),Ship.MAX_SIZE));
    else
      this.getShield().getAreas().addAll(Shapes.getAreas(Shapes.getShape(shape),this.getShield().getSize(),Ship.MAX_SIZE));
    this.setMaxDamage(health.getMaxDamage() * this.getSize() / Ship.DEFAULT_SIZE);
    this.setDamage(0);
    this.items = Media.getItems();
    this.resetItems();
  }

  /**
   * @return the maximum size
   */
  public int getMaximumSize()
  {
    return Ship.MAX_SIZE;
  }

  /**
   * @return the minimum size
   */
  public int getMinimumSize()
  {
    return Ship.MIN_SIZE;
  }

  /**
   * @return the default size
   */
  public int getDefaultSize()
  {
    return Ship.DEFAULT_SIZE;
  }

  /**
   * @param _size the size to set the sprite's size to
   */
  /*
   * (non-Javadoc)
   * @see javoids.Sprite#setSize(int)
   */
  @Override
  public void setSize(int _size)
  {
    super.setSize(_size >= Ship.MIN_SIZE ? _size <= Ship.MAX_SIZE ? _size : Ship.MAX_SIZE : Ship.MIN_SIZE);
  }

  /**
   * @return a mapping of item names to items
   */
  public HashMap<Item.Type,Item> getItems()
  {
    return this.items;
  }

  /**
   * @return the currently selected item
   */
  public Item.Type getCurrentItem()
  {
    return this.currentItem;
  }

  /**
   * @param _soundAfterburner the sound for the ship's afterburner engine
   */
  public void setSoundAfterburner(boolean _soundAfterburner)
  {
    this.soundAfterburner = _soundAfterburner;
  }

  /**
   * @param item the the currently selected item
   */
  public void setCurrentItem(Item.Type item)
  {
    this.currentItem = item;
  }

  /**
   * set the current item to the best weapon the ship owns
   */
  public void setWeaponBest()
  {
    this.setCurrentItem(Item.Type.GUN1); // select the lowest powered weapon (GUN1)
    this.cycleWeapon(Item.Direction.LEFT); // cycle through the weapons left (going to the most powerful weapon possessed
  }

  /**
   * @param _gravityDuration the length of time the gravity effect on the ship will last
   */
  public void setGravityDuration(int _gravityDuration)
  {
    this.gravityDuration = _gravityDuration > 0 ? _gravityDuration : 0;
  }

  /**
   * @param points the number of points to set this sprite's point total to
   */
  /*
   * (non-Javadoc)
   * @see javoids.BasicSprite#setPoints(int)
   */
  @Override
  public void setPoints(int points)
  {
    super.setPoints(points);
    super.setPoints(this.getPoints() < 0 ? 0 : this.getPoints());
    this.checkFreeLife();
  }

  /**
   * @param points the number of points to award
   * @return the new point total
   */
  /*
   * (non-Javadoc)
   * @see javoids.BasicSprite#modifyPoints(int)
   */
  @Override
  public int modifyPoints(int points)
  {
    super.modifyPoints(points);
    this.checkFreeLife();
    return this.getPoints();
  }

  /**
   * Check to see if a free life should be awarded (takes away a death).
   */
  private void checkFreeLife()
  {
    while (this.getPoints() / (Ship.FREE_SHIP_POINTS * (Javoids.getDifficulty() + 1) / 2) > this.freeLives && this.getDeaths() < this.getMaxDeaths())
    {
      this.freeLives++; // update this just in case the points get updated very very rapidly
      this.modifyDeaths(-1);
      Media.play(BasicSprite.getPacmanGame() ? PACMAN_EXTRALIFE : SHIP_EXTRALIFE);
    }
  }

  /**
   * Go throught the weapons until one with ammunition is found
   * @param direction the direction to cycle through the list of weapons
   */
  public void cycleWeapon(Item.Direction direction)
  {
    this.currentItem = Item.cycleWeapon(this.getItems(),this.getCurrentItem(),direction);
  }

  /**
   * @param x the x coordinate
   * @param y the y coordinate
   * @param maxVelocity the maximum velocity
   * @param direction the direction the ship is facing
   */
  public void restore(double x,double y,double maxVelocity,double direction)
  {
    this.setX(x);
    this.setY(y);
    this.setMaxVelocity(maxVelocity);
    this.setDirection(direction);
    this.accelerate(0);
    this.setDamage(0);
    this.setDuration(0);
    this.getShield().modifyDamage(0);
    this.getShield().modifyDuration(0);
    this.getShield().setMove(new Move(this.getMove()));
    this.setGravity(NONE);
    this.gravityDuration = 0;
    this.items.clear();
    this.items = Media.getItems();
    this.resetItems();
  }

  /**
   * restore an item's number of uses to 0
   */
  public static void restoreCount()
  {
    Ship.count = 0;
  }

  /**
   * Push the ship forward by the amount specified
   * @param amount the amount of thrust to use
   */
  public void thrust(double amount)
  {
    super.accelerate(amount);
    Media.play(BasicSprite.getPacmanGame() ? PACMAN_EATDOT : THRUST);
  }

  /**
   * Move the sprite and check for an end to gravity effects.
   */
  /*
   * (non-Javadoc)
   * @see javoids.ShieldedSprite#move()
   */
  @Override
  public void move()
  {
    if (this.gravityDuration > 0)
    {
      this.gravityDuration -= 1 * Javoids.getDifficulty();
      if (this.gravityDuration <= 0)
        this.setGravity(NONE);
    }
    super.move();
  }

  /**
   * @param _damage the damage the sprite sustained
   * @return the new damage level
   */
  /*
   * (non-Javadoc)
   * @see javoids.BasicSprite#modifyDamage(int)
   */
  @Override
  public int modifyDamage(int _damage)
  {
    int damage = super.modifyDamage(_damage);

    if (this.isAlive())
      Media.play(SHIPHIT);
    else
    {
      Media.play(BasicSprite.getPacmanGame() ? PACMAN_DIE : SHIPDIE);
      this.freeLives = this.getPoints() / (Ship.FREE_SHIP_POINTS * (Javoids.getDifficulty() + 1) / 2); // cap point to where to begin checking for free lives
      this.modifyDeaths(1);
      this.setGravity(NONE);
      this.gravityDuration = 0;
    }
    return damage;
  }

  /**
   * Perform a sepcified action
   * @param action the action to perform in the game
   * @param ships the list of all ships in the game
   * @param bullets the list of all bullets in the game
   * @param mines the list of all mines in the game
   * @param javoids the list of all javoids in the game
   * @param explosions the list of all explosions in the game
   */
  public void performAction(KeyMap.Action action,SpriteVector<Ship> ships,SpriteVector<Bullet> bullets,SpriteVector<Mine> mines,SpriteVector<Javoid> javoids,SpriteVector<Explosion> explosions)
  {
    if (this.isAlive())
    {
      switch (action)
      {
        case ACTION_FIRE :
          synchronized (bullets)
          {
            bullets.addAll(this.useGun());
          }
          break;
        case ACTION_THRUST :
          this.thrust(Move.DEFAULT_ACCELERATE);
          break;
        case ACTION_TURN_LEFT :
          this.turn(Move.LEFT,10);
          break;
        case ACTION_TURN_RIGHT :
          this.turn(Move.RIGHT,10);
          break;
        case ACTION_CYCLE_LEFT :
          this.cycleWeapon(Item.Direction.LEFT);
          break;
        case ACTION_CYCLE_RIGHT :
          this.cycleWeapon(Item.Direction.RIGHT);
          break;
        case ACTION_FLIP :
          this.setDirection((540 + this.getDirection()) % 360);
          break;
        case ACTION_AFTERBURNER :
          this.useAfterBurner();
          break;
        case ACTION_SHIELD :
          this.useShields();
          break;
        case ACTION_JUMP :
          this.useJumps();
          break;
        case ACTION_BOMB1 :
          if (this.getItems().get(Item.Type.BOMB1).getCount() > 0)
          {
            this.getItems().get(Item.Type.BOMB1).modifyCount(-1);
            this.useBomb(javoids,explosions);
          }
          break;
        case ACTION_BOMB2 :
          if (this.getItems().get(Item.Type.BOMB2).getCount() > 0)
          {
            this.getItems().get(Item.Type.BOMB2).modifyCount(-1);
            this.useMegaBomb(javoids,explosions);
            this.useMegaBomb(mines,explosions);
            this.useMegaBomb(ships,explosions);
          }
          break;
        case ACTION_SELECT_WEAPON_BEST :
          this.setWeaponBest();
          break;
        case ACTION_SELECT_WEAPON_01 :
          this.currentItem = Item.setWeapon(Item.Type.GUN1);
          break;
        case ACTION_SELECT_WEAPON_02 :
          this.currentItem = this.items.get(Item.Type.GUN2).getCount() > 0 ? Item.setWeapon(Item.Type.GUN2) : Item.setWeapon(Item.Type.GUN1);
          break;
        case ACTION_SELECT_WEAPON_03 :
          this.currentItem = this.items.get(Item.Type.GUN3).getCount() > 0 ? Item.setWeapon(Item.Type.GUN3) : Item.setWeapon(Item.Type.GUN1);
          break;
        case ACTION_SELECT_WEAPON_04 :
          this.currentItem = this.items.get(Item.Type.MACHINEGUN1).getCount() > 0 ? Item.setWeapon(Item.Type.MACHINEGUN1) : Item.setWeapon(Item.Type.GUN1);
          break;
        case ACTION_SELECT_WEAPON_05 :
          this.currentItem = this.items.get(Item.Type.MACHINEGUN2).getCount() > 0 ? Item.setWeapon(Item.Type.MACHINEGUN2) : Item.setWeapon(Item.Type.GUN1);
          break;
        case ACTION_SELECT_WEAPON_06 :
          this.currentItem = this.items.get(Item.Type.MACHINEGUN3).getCount() > 0 ? Item.setWeapon(Item.Type.MACHINEGUN3) : Item.setWeapon(Item.Type.GUN1);
          break;
        case ACTION_SELECT_WEAPON_07 :
          this.currentItem = this.items.get(Item.Type.MULTIGUN1).getCount() > 0 ? Item.setWeapon(Item.Type.MULTIGUN1) : Item.setWeapon(Item.Type.GUN1);
          break;
        case ACTION_SELECT_WEAPON_08 :
          this.currentItem = this.items.get(Item.Type.MULTIGUN2).getCount() > 0 ? Item.setWeapon(Item.Type.MULTIGUN2) : Item.setWeapon(Item.Type.GUN1);
          break;
        case ACTION_SELECT_WEAPON_09 :
          this.currentItem = this.items.get(Item.Type.MULTIGUN3).getCount() > 0 ? Item.setWeapon(Item.Type.MULTIGUN3) : Item.setWeapon(Item.Type.GUN1);
          break;
        case ACTION_SELECT_WEAPON_10 :
          this.currentItem = this.items.get(Item.Type.MULTIGUN4).getCount() > 0 ? Item.setWeapon(Item.Type.MULTIGUN4) : Item.setWeapon(Item.Type.GUN1);
          break;
        case ACTION_SELECT_WEAPON_11 :
          this.currentItem = this.items.get(Item.Type.MULTIGUN5).getCount() > 0 ? Item.setWeapon(Item.Type.MULTIGUN5) : Item.setWeapon(Item.Type.GUN1);
          break;
        case ACTION_SELECT_WEAPON_12 :
          this.currentItem = this.items.get(Item.Type.MULTIGUN6).getCount() > 0 ? Item.setWeapon(Item.Type.MULTIGUN6) : Item.setWeapon(Item.Type.GUN1);
          break;
        case ACTION_SELECT_WEAPON_13 :
          this.currentItem = this.items.get(Item.Type.ROCKET1).getCount() > 0 ? Item.setWeapon(Item.Type.ROCKET1) : Item.setWeapon(Item.Type.GUN1);
          break;
        case ACTION_SELECT_WEAPON_14 :
          this.currentItem = this.items.get(Item.Type.ROCKET2).getCount() > 0 ? Item.setWeapon(Item.Type.ROCKET2) : Item.setWeapon(Item.Type.GUN1);
          break;
        default :
          break;
      }
    }
  }

  /**
   * @param sprites the list of sprites to apply the effects of the bomb on.
   * @param explosions the list of all explosions in the game
   */
  public void useBomb(SpriteVector<Javoid> sprites,SpriteVector<Explosion> explosions)
  {
    if (this.isAlive())
    {
      SpriteVector<Javoid> tempSprites = new SpriteVector<Javoid>(Javoid.MAX_NUMBER);
      SpriteVector<Explosion> tempExplosions = new SpriteVector<Explosion>(Explosion.MAX_NUMBER);
      int points = 0;

      // accumulate points, create the smaller javoids, create an explosion, kill the larger javoids
      for (Javoid javoid : sprites)
      {
        points += javoid.getDamage();
        for (Javoid basicSprite2 : javoid.breakApart())
          tempSprites.add(basicSprite2);
        tempExplosions.add(new Explosion(javoid.getParent(),new Move(javoid.getMove()),javoid.getSize()));
      }
      // remove the dead javoids, add the new smaller javoids, clear the roid container, repaint
      sprites.removeDead();
      for (Javoid javoid : tempSprites)
        sprites.add(javoid);
      tempSprites.clear();
      explosions.removeDead();
      explosions.addAll(tempExplosions);
      tempExplosions.clear();
      this.setPoints(this.getPoints() + points);
    }
  }

  /**
   * @param sprites the list of sprites to apply the effects of the bomb on.
   * @param explosions the list of all explosions in the game
   * @param <E> the type of sprite being exploded
   */
  public <E extends BasicSprite> void useMegaBomb(SpriteVector<E> sprites,SpriteVector<Explosion> explosions)
  {
    if (this.isAlive())
    {
      SpriteVector<E> tempSprites = new SpriteVector<E>(Javoid.MAX_NUMBER);
      SpriteVector<Explosion> tempExplosions = new SpriteVector<Explosion>(Explosion.MAX_NUMBER);
      int points = 0;

      // accumulate points, create the smaller javoids, create an explosion, kill the larger javoids
      for (E basicSprite : sprites)
      {
        points += basicSprite.getDamage();
        tempExplosions.add(new Explosion(basicSprite.getParent(),new Move(basicSprite.getMove()),basicSprite.getSize()));
        if (basicSprite instanceof Ship && !basicSprite.isPlayer())
          basicSprite.kill();
      }
      // remove the dead javoids, add the new smaller javoids, clear the roid container, repaint
      sprites.removeDead();
      for (E basicSprite : tempSprites)
        sprites.add(basicSprite);
      tempSprites.clear();
      explosions.removeDead();
      explosions.addAll(tempExplosions);
      tempExplosions.clear();
      this.setPoints(this.getPoints() + points);
    }
  }

  /**
   * use a hyperspace jump to change position and direction instantly and randomly.
   */
  public void useJumps()
  {
    if (this.isAlive() && this.items.get(Item.Type.JUMP).getCount() > 0)
    {
      this.items.get(Item.Type.JUMP).modifyCount(-1);
      this.setX(Math.random() * this.getScreen().width);
      this.setY(Math.random() * this.getScreen().height);
      this.setDirection(Math.random() * 360);
      this.getShield().setX(this.getX());
      this.getShield().setY(this.getY());
      this.getShield().setDirection(this.getDirection());
    }
  }

  /**
   * activate the ship's shields
   */
  public void useShields()
  {
    if (this.isAlive() && this.items.get(Item.Type.SHIELD).getCount() > 0 && !this.getShield().isAlive())
    {
      this.items.get(Item.Type.SHIELD).modifyCount(-1);
      this.getShield().setDeaths(0);
      this.getShield().setDamage(0);
      this.getShield().setDuration(0);
      this.getShield().setMove(new Move(this.getMove()));
    }
  }

  /**
   * Use the ship's afterburners if possible
   */
  public void useAfterBurner()
  {
    if (this.isAlive() && this.items.get(Item.Type.AFTERBURNER).getCount() > 0)
    {
      this.items.get(Item.Type.AFTERBURNER).modifyCount(-1);
      // alter maximum speed for ship and shield to use afterburner
      this.setMaxVelocity(Ship.DEFAULT_VELOCITY * Ship.AFTERBURNER_MULTIPLIER);
      this.getShield().setMaxVelocity(Ship.DEFAULT_VELOCITY * Ship.AFTERBURNER_MULTIPLIER);
      super.accelerate(Move.DEFAULT_ACCELERATE * Ship.AFTERBURNER_MULTIPLIER);
      this.getShield().setMaxVelocity(Ship.DEFAULT_VELOCITY);
      this.setMaxVelocity(Ship.DEFAULT_VELOCITY);
      if (!this.soundAfterburner)
      {
        this.soundAfterburner = true;
        Media.play(BasicSprite.getPacmanGame() ? PACMAN_EATDOT : SoundMap.Sound.AFTERBURNER);
      }
    }
  }

  /**
   * @return fire the ship's gun
   */
  public SpriteVector<Bullet> useGun()
  {
    SpriteVector<Bullet> bullets = new SpriteVector<Bullet>(Bullet.MAX_NUMBER);
    Item item = this.getItems().get(this.getCurrentItem());

    if (this.isAlive() && item.getCount() > 0)
    {
      synchronized (bullets)
      {
        bullets.addAll(this.spawnBullets());
      }
      Weapon weapon = (Weapon)item;
      if (bullets.size() > 0)
      {
        weapon.modifyCount(-1);
        weapon.modifyFired(bullets.size());
        if (weapon.getShields() > 0)
          weapon.modifyShields(-1);
        if (weapon.getHomings() > 0)
          weapon.modifyHomings(-1);
      }
    }
    else
      this.currentItem = Item.setWeapon(Item.Type.GUN1);
    return bullets;
  }

  /**
   * @return use the ship's gun automatically if not player controlled
   */
  public SpriteVector<Bullet> useAutomaticGun()
  {
    SpriteVector<Bullet> bullets = new SpriteVector<Bullet>(Bullet.MAX_NUMBER);

    this.setWeaponBest(); // select the best weapon
    if (Math.random() < Javoids.getDifficulty() / 10.0d) // only shoot some of the times, more often as difficulty increases
      synchronized (bullets)
      {
        bullets.addAll(this.useGun()); // shoot the gun
      }
    return bullets;
  }

  /**
   * @return create the bullets from one use of the gun
   */
  public SpriteVector<Bullet> spawnBullets()
  {
    SpriteVector<Bullet> bullets = new SpriteVector<Bullet>(Bullet.MAX_NUMBER);
    Weapon weapon = (Weapon)this.items.get(this.currentItem);

    if (weapon.getFired() <= weapon.getMaximumFired() - weapon.getDirections())
    {
      Health tempAlive = new Health(1,weapon.getDamage(),weapon.getRange(),Health.DEFAULT_AGING_RATE,NORMAL);
      Move tempMove = new Move(this.getMove());
      int directions = weapon.getDirections();

      tempMove.setMaxVelocity(weapon.getVelocity());
      tempMove.accelerate(0); // without this bullets shared the momentum and looked odd when being shot
      synchronized (bullets)
      {
        bullets.addAll(Bullet.spawn(this,directions,this.getMove().getDirection(),tempAlive,tempMove,weapon));
      }
      Media.play(BasicSprite.getPacmanGame() ? PACMAN_EATDOT : weapon.getSound());
    }
    return bullets;
  }

  /**
   * Apply the effects of the power up to the ship
   * @param sprite the power up to use
   */
  public void apply(BasicSprite sprite) // change to ship.apply(powerUp);
  {
    PowerUp powerUp = (PowerUp)sprite;
    int difference;

    this.setPoints(this.getPoints() + Math.abs(powerUp.getPoints()));
    this.addItems(powerUp.getItems());
    if (powerUp.getExtraHealth() != null)
    {
      this.setDeaths(this.getDeaths() + powerUp.getExtraHealth().getDeaths());
      // heal ship damage first, then shield damage
      difference = this.getDamage() + powerUp.getExtraHealth().getDamage() - this.getMaxDamage();
      super.modifyDamage(powerUp.getExtraHealth().getDamage());
      if (difference > 0)
        this.getShield().modifyDamage(difference);
      this.getShield().modifyDuration(powerUp.getExtraHealth().getDuration());
      this.getShield().setMove(new Move(this.getMove()));
    }
    this.setGravity(powerUp.getExtraGravity());
    this.setGravityDuration(this.gravityDuration + powerUp.getExtraGravityDuration());
    Media.play(BasicSprite.getPacmanGame() ? PACMAN_EATFRUIT : POWERUP);
    powerUp.kill(); // powerup die!
  }

  /**
   * @param _items the list of items to add the the ship's list
   */
  public void addItems(HashMap<Item.Type,Item> _items)
  {
    for (Item item2 : _items.values())
    {
      Item item1 = this.items.get(item2.getType());
      if (item1.isLimitless() && item2.getCount() > 0)
        item1.setCount(item2.getCount());
      else
        item1.modifyCount(item2.getCount());
      if (_items.get(item2.getType()) instanceof Weapon)
      {
        Weapon weapon1 = (Weapon)this.items.get(item1.getType());
        Weapon weapon2 = (Weapon)item2;
        weapon1.modifyShields(weapon2.getShields());
        weapon1.modifyHomings(weapon2.getHomings());
      }
    }
  }

  /**
   * @return the image name for the image currently being used
   */
  /*
   * (non-Javadoc)
   * @see javoids.Sprite#getImageNumber()
   */
  @Override
  public ImageMap.Name getImageNumber()
  {
    return BasicSprite.getPacmanGame() ? PACMAN_PACMAN : this.getOriginalImage();
  }

  /**
   * reset all items to their default values
   */
  public void resetItems()
  {
    for (Item item : this.getItems().values())
    {
      if (Debug.debugItem)
      {
        item.setLimitless(true);
        item.setCount(item.getDefaultCount());
      }
      if (Debug.debugWeapon)
      {
        item.setLimitless(true);
        item.setCount(item.getDefaultCount());
        if (item instanceof Weapon)
        {
          Weapon weapon = (Weapon)item;
          if (Debug.debugWeaponShield)
            weapon.setShields(weapon.getDefaultCount());
          if (Debug.debugWeaponHoming)
            weapon.setHomings(weapon.getDefaultCount());
        }
        this.setCurrentItem(Item.Type.ROCKET2);
      }
    }
    // restore the primary gun all of the time
    this.items.get(Item.Type.GUN1).setCount(this.items.get(Item.Type.GUN1).getDefaultCount());
  }

  /**
   * Provide a String representation of this object.
   * @return String A representation of the object for debugging.
   */
  @Override
  public String toString()
  {
    return String.format(Messages.getString("Ship.ToString"),super.toString(),this.items,Integer.valueOf(this.freeLives),Integer.valueOf(Ship.count),this.getGravity(),Integer.valueOf(this.gravityDuration)); //$NON-NLS-1$
  }
}
/* Ship----------------------- */
