/*  Saper Game.
    Copyright (C) 2007 - Pawel Bednarek
    <bednarek.pawel@gmail.com>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/> 
 */
package saper;

import java.awt.Color;
import java.awt.Component;
import java.awt.Graphics;
import java.awt.event.MouseListener;

import javax.swing.BorderFactory;
import javax.swing.JLabel;
import javax.swing.border.BevelBorder;
import javax.swing.border.Border;

/**
 * Interfejs oraz implementacja potrzebnych metod dla przycisku
 * wyswietlajacego jedno pole pola minowego
 */
public interface SaperButton {

    /**
     * Komponent wyswietlajacy pojedyncze pole
     * @return
     */
    public Component getComponent();
    /**
     * Wspolrzedna pola minowego reprezentowana przez ten obiekt
     * @return
     */
    public int getSaperX();
    /**
     * Wspolrzedna pola minowego reprezentowana przez ten obiekt
     * @return
     */
    public int getSaperY();
    /**
     * Zapewnia mozliwosc odbierania zdarzen myszy
     * @param listener
     */
    public void addMouseListener(MouseListener listener);
    /**
     * Zapewnia mozliwosc odbierania zdarzen myszy
     * @param listener
     */
    public void removeMouseListener(MouseListener listener);
    
    /**
     * Zaznaczenie ze mozna kliknac na pole. Czyli ze pole jest zakryte
     */
    public void paintHidden();
    /**
     * Zaznaczenie miny na polu.
     */
    public void paintMine();
    /**
     * Zaznaczenie, ze pole sasiaduje z 'number' minami.
     * Jesli number = 0 paintClear zostanie wywolane
     * @param number
     */
    public void paintNumber(int number);
    /**
     * Zaznaczenie ze tutaj jest blednie zaznaczone pole.
     */
    public void paintError();
    /**
     * Rysowanie przycisku
     * @param g
     */
    public void paint(Graphics g);
    /**
     * Implementacja przycisku gry w oparciu o proste pole tekstowe.
     * Metody zwiazane z MouseListener sa juz w JLabel
     */
    public static final class SaperLabelButton extends JLabel implements SaperButton {
        
        // przycisk wypukly
        private static final Border BORDER_UP = BorderFactory.createBevelBorder(BevelBorder.RAISED);
        // przycisk wklesly
        private static final Border BORDER_LOW = BorderFactory.createBevelBorder(BevelBorder.LOWERED);
        // kolor textu
        private static final Color MINE_TEXT_COLOR = Color.red;
        // kolor tekstu
        private static final Color TEXT_COLOR = Color.black;
        
        // wspolrzedne pola gry
        private int sx,sy;
        
        public SaperLabelButton(int x, int y){
            setHorizontalAlignment(CENTER); // rysowanie textu na srodeczku
            paintHidden();
            this.sx = x;
            this.sy = y;
        }
        /**
         * @see saper.SaperButton#paintUncovered()
         */
        public void paintHidden() {
            setBorder(BORDER_UP);
            setForeground(TEXT_COLOR);
            setText(" ");
        }

        /**
         * @see saper.SaperButton#paintMine()
         */
        public void paintMine() {
            setBorder(BORDER_LOW);
            setForeground(MINE_TEXT_COLOR);
            setText("M");
        }

        /**
         * @see saper.SaperButton#paintNumber(int)
         */
        public void paintNumber(int number) {
            setBorder(BORDER_LOW);
            setForeground(TEXT_COLOR);
            setText(number == 0 ? "" : Integer.toString(number));
        }

        /**
         * @see saper.SaperButton#paintError()
         */
        public void paintError() {
            setBorder(BORDER_LOW);
            setForeground(MINE_TEXT_COLOR);
            setText("X");
        }
        
        /**
         * @see saper.SaperButton#getComponent()
         */
        public Component getComponent() {
            return this;
        }

        /**
         * @see saper.SaperButton#getSaperX()
         */
        public int getSaperX() {
            return this.sx;
        }

        /**
         * @see saper.SaperButton#getSaperY()
         */
        public int getSaperY() {
            return this.sy;
        }
    }
}
