/* 
 *	HT Editor
 *	sysdisplay.cc - screen access functions for POSIX
 *
 *	Copyright (C) 1999-2004 Stefan Weyergraf (stefan@weyergraf.de)
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License version 2 as
 *	published by the Free Software Foundation.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program; if not, write to the Free Software
 *	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#define INCL_DOS
#define INCL_DOSERRORS
#define INCL_VIO
#define INCL_KBD
#include <os2.h>
#include <uconv.h>

#include "config.h"


#include <signal.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <malloc.h>

#include "io/display.h"
#include "io/types.h"

#define DISP_DEBUG
#ifdef DISP_DEBUG
#include <stdio.h>
#endif

#define USE_DBLBUF


struct gcharmap {
	uint id;
	uint c;
};

#define GCHARMAP_END (uint)(-1)

#if 0
static gcharmap graphical_text[] = {
	{ GC_1VLINE, '|' },
	{ GC_1HLINE, '-' },
	{ GC_1CORNER0, '+' },
	{ GC_1CORNER1, '+' },
	{ GC_1CORNER2, '+' },
	{ GC_1CORNER3, '+' },
	{ GC_1UTEE, '+' },
	{ GC_1LTEE, '+' },
	{ GC_1DTEE, '+' },
	{ GC_1RTEE, '+' },
	{ GC_1CROSS, '+' },
	{ GC_2VLINE, '|' },
	{ GC_2HLINE, '=' },
	{ GC_2CORNER0, '+' },
	{ GC_2CORNER1, '+' },
	{ GC_2CORNER2, '+' },
	{ GC_2CORNER3, '+' },
	{ GC_LOW, ':' },
	{ GC_MEDIUM, '#' },
	{ GC_HIGH, '#' },
	{ GC_FULL, '#' },
	{ GC_ARROW_UP, '^' },
	{ GC_ARROW_DOWN, 'v' },
	{ GC_ARROW_LEFT, '<' },
	{ GC_ARROW_RIGHT, '>' },
	{ GC_SMALL_ARROW_UP, '^' },
	{ GC_SMALL_ARROW_DOWN, 'v' },
	{ GC_FILLED_CIRCLE, '*' },
	{ GC_FILLED_QUAD, 'x' },
	{ GC_FILLED_UPPER, '~' },
	{ GC_FILLED_LOWER, '_' },
	{ GC_TRANSPARENT, '\0' },
	{ GCHARMAP_END, GCHARMAP_END }
};
#endif
static gcharmap graphical_cp_dbcspc[] = {
	{ GC_1VLINE, 0x1d },
	{ GC_1HLINE, '-' },
	{ GC_1CORNER0, '+' },
	{ GC_1CORNER1, '+' },
	{ GC_1CORNER2, '+' },
	{ GC_1CORNER3, '+' },
	{ GC_1UTEE, '+' },
	{ GC_1LTEE, '+' },
	{ GC_1DTEE, '+' },
	{ GC_1RTEE, '+' },
	{ GC_1CROSS, '+' },
	{ GC_2VLINE, 0x05 },
	{ GC_2HLINE, 0x06 },
	{ GC_2CORNER0, 0x02 },
	{ GC_2CORNER1, 0x04 },
	{ GC_2CORNER2, 0x03 },
	{ GC_2CORNER3, 0x01 },
	{ GC_LOW, 0x1a },
	{ GC_MEDIUM, 0x14 },
	{ GC_HIGH, 0x14 },
	{ GC_FULL, 0x14 },
	{ GC_ARROW_UP, 0x1c },
	{ GC_ARROW_DOWN, 0x07 },
	{ GC_ARROW_LEFT, 0x1f },
	{ GC_ARROW_RIGHT, 0x1e },
	{ GC_SMALL_ARROW_UP, 0x1c },
	{ GC_SMALL_ARROW_DOWN, 0x07 },
	{ GC_FILLED_CIRCLE, 0x0f },
	{ GC_FILLED_QUAD, 0x0e },
	{ GC_FILLED_UPPER, 0x7e },
	{ GC_FILLED_LOWER, 0x5f },
	{ GC_TRANSPARENT, '\0' },
	{ GCHARMAP_END, GCHARMAP_END }
};
static gcharmap graphical_cp_sbcspc[] = {
	{ GC_1VLINE, 0xb3 },
	{ GC_1HLINE, 0xc4 },
	{ GC_1CORNER0, 0xbf },
	{ GC_1CORNER1, 0xd9 },
	{ GC_1CORNER2, 0xc0 },
	{ GC_1CORNER3, 0xda },
	{ GC_1UTEE, 0xc2 },
	{ GC_1LTEE, 0xc3 },
	{ GC_1DTEE, 0xc1 },
	{ GC_1RTEE, 0xb4 },
	{ GC_1CROSS, 0xc5 },
	{ GC_2VLINE, 0xba },
	{ GC_2HLINE, 0xcd },
	{ GC_2CORNER0, 0xbb },
	{ GC_2CORNER1, 0xbc },
	{ GC_2CORNER2, 0xc8 },
	{ GC_2CORNER3, 0xc9 },
	{ GC_LOW, 0xb0 },
	{ GC_MEDIUM, 0xb1 },
	{ GC_HIGH, 0xb2 },
	{ GC_FULL, 0xdb },
	{ GC_ARROW_UP, 0x1e },
	{ GC_ARROW_DOWN, 0x1f },
	{ GC_ARROW_LEFT, 0x11 },
	{ GC_ARROW_RIGHT, 0x10 },
	{ GC_SMALL_ARROW_UP, 0x18 },
	{ GC_SMALL_ARROW_DOWN, 0x19 },
	{ GC_FILLED_CIRCLE, 0x0f },
	{ GC_FILLED_QUAD, 0x16 },
	{ GC_FILLED_UPPER, 0xdf },
	{ GC_FILLED_LOWER, 0xdc },
	{ GC_TRANSPARENT, '\0' },
	{ GCHARMAP_END, GCHARMAP_END }
};


gcharmap *graphical_map = graphical_cp_sbcspc;
HVIO hvioSystem = 0;


static
int max_mbclen(int cp)
{
	UconvObject uo;
	uconv_attribute_t ua;
	CHAR lead[256], trail[256];
	udcrange_t udc[32];
	char s[64];
	UniChar us[64];
	unsigned i;
	int rc;
	
	sprintf(s, "IBM%d@map=display", cp);
	for(i=0; i<sizeof(us)/sizeof(us[0]); i++) {
		us[i] = (UniChar)(unsigned char)s[i];
		if (s[i] == '\0') break;
	}
	ua.mb_max_len = 1; /* just a proof... */
	rc = UniCreateUconvObject(us, &uo);
	if (rc == 0)
		rc = UniQueryUconvObject(uo, &ua, sizeof(ua), lead, trail, udc);
	if (rc == 0) {
		rc = (int)(unsigned char)(ua.mb_max_len);
	}
	else
		rc = 1;
	
	return rc;
}


uint mapToGraphical(uint chr)
{
	int i;	
	for(i=0;;i++) {
		if (graphical_map[i].id == GCHARMAP_END) {
			break;
		}
		if (graphical_map[i].id == (uint)chr) {
			return graphical_map[i].c;
		}
	}
	return '?';
}

uint mapCharToSystemCP(char chr, Codepage codepage)
{
	switch (codepage) {
	case CP_DEVICE: return (byte)chr;
	case CP_GRAPHICAL: return mapToGraphical((byte)chr);
	case CP_INVALID: return '?';
	case CP_WINDOWS: // return (byte)chr;
	case CP_UNICODE:
	default: break;
	}
	return (byte)chr;
}

/*
 *	Class Os2VioSystemDisplay
 */
 

class Os2VioSystemDisplay: public SystemDisplay {
protected:
	PBYTE	viobuf;
#ifdef USE_DBLBUF
	PBYTE	viobuf2;
#endif
	vcp		*colorbuf;
	HVIO	hvio;
	USHORT	cpvio;
	VIOCURSORINFO	org_cursor_info;
	USHORT org_cursor_x, org_cursor_y;
	CursorMode cursor_mode;
	int cursorx, cursory; 
	
	void	setVioCursorMode(CursorMode mode);
public:
					Os2VioSystemDisplay(const char *title);
	virtual				~Os2VioSystemDisplay();
	/* extends Display */
	virtual	void			copyFromDisplay(const Display &display, int x, int y, const Bounds &clipping);
	virtual	void			fill(int x, int y, int nx, int ny, vcp color, char chr, Codepage cp = CP_DEVICE);
	virtual	void			getCursor(int &x, int &y) const;
	virtual	CursorMode		getCursorMode() const;
	virtual	int			nprint(int x, int y, vcp color, const char *str, int maxstrlen, Codepage cp = CP_DEVICE);
	virtual	bool			read(uint &rawchar, vcp &color, int x, int y) const;
	virtual	void			setBounds(const Bounds &b);
	virtual	void			setCursor(int x, int y, CursorMode mode = CURSOR_NORMAL);
	virtual	void			setCursorMode(CursorMode mode = CURSOR_NORMAL);
	/* extends SystemDisplay */
	virtual	void			show();
	uint	vcpToSystem(vcp color);
	void	putChar(int x, int y, uint rawchar, vcp vc);
};

Os2VioSystemDisplay *gDisplay = NULL;

static
bool os2vio_get_screen_size(int &w, int &h, HVIO hvio)
{
	VIOMODEINFO vm = { 0 };
	vm.cb = sizeof(vm);
	if (VioGetMode(&vm, hvio) != NO_ERROR) return false;
	w = vm.col;
	h = vm.row;
	return true;
}

bool sys_get_screen_size(int &w, int &h)
{
	return os2vio_get_screen_size(w, h, hvioSystem);
}

bool sys_set_screen_size(int w, int h)
{
#ifdef DISP_DEBUG
	printf("set_sc w=%d h=%d\n", w, h);
#endif
	return true;
}

Os2VioSystemDisplay::Os2VioSystemDisplay(const char *title)
: SystemDisplay()
{
	int col, row;
	
	viobuf = NULL;
#ifdef USE_DBLBUF
	viobuf2 = NULL;
#endif
	colorbuf = NULL;
	hvio = hvioSystem;
	VioGetCp(0, &cpvio, hvio);
	
	graphical_map = max_mbclen(cpvio) > 1
	                    ? graphical_cp_dbcspc : graphical_cp_sbcspc;
	
	VioGetCurType(&org_cursor_info, hvio);
	VioGetCurPos(&org_cursor_y, &org_cursor_x, hvio);
	os2vio_get_screen_size(col, row, hvio);

	assign(0, 0, col, row);
	// setVioCursorMode(CURSOR_OFF);
	setCursor(0, 0, CURSOR_NORMAL);
	show();
}

Os2VioSystemDisplay::~Os2VioSystemDisplay()
{
	unsigned char s[2];
	
	s[0] = ' '; s[1] = 0x07;
	VioScrollUp(0, 0, h, w, h, s, hvio); /* cls */
	if (viobuf) { free(viobuf); viobuf = NULL; }
#ifdef USE_DBLBUF
	if (viobuf2) { free(viobuf2); viobuf2 = NULL; }
#endif
	if (colorbuf) { free(colorbuf); colorbuf = NULL; }
	VioSetCurPos(org_cursor_y, org_cursor_x, hvio);
	VioSetCurType(&org_cursor_info, hvio);
}

void Os2VioSystemDisplay::copyFromDisplay(const Display &d, int x, int y, const Bounds &clipping)
{
	int ix, iy;
	for (iy=0; iy < d.h; iy++) {
		if (y+iy >= clipping.y+clipping.h) break;
		if (y+iy < clipping.y) break;
		int dest = x;
		for (ix=0; ix < d.w; ix++) {
			uint rawchar;
			vcp color;
			d.read(rawchar, color, ix, iy);
			if ((x+ix < clipping.x+clipping.w) && (x+ix >= clipping.x))
				putChar(dest, y+iy, rawchar, color);
			dest++;
		}
	}
}

void Os2VioSystemDisplay::fill(int x, int y, int nx, int ny, vcp color, char chr, Codepage cp)
{
	int ix, iy;
	uint rawchar;
	
	rawchar = mapCharToSystemCP(chr, cp);
	for(iy = y; iy < (y+ny) && iy < h; iy++) {
		for(ix = x; ix < (x+nx) && ix < w; ix++) {
			if (ix >= 0 && iy >=0)
				putChar(ix, iy, rawchar, color);
		}
	}
}

void Os2VioSystemDisplay::getCursor(int &x, int &y) const
{
	x = cursorx;
	y = cursory;
}

CursorMode Os2VioSystemDisplay::getCursorMode() const
{
	return cursor_mode;
}

int Os2VioSystemDisplay::nprint(int x, int y, vcp color, const char *str, int maxstrlen, Codepage cp)
{
	int n;
	for (n=0; n < maxstrlen && (x+n) < w && str[n]; n++) {
		if (x+n >= 0) putChar(x+n, y, mapCharToSystemCP(str[n], cp), color);
	}
	return n;
}


bool Os2VioSystemDisplay::read(uint &rawchar, vcp &color, int x, int y) const
{
	// if (x < 0 || y < 0 || x >= w || y >= h) return false;
	rawchar = viobuf[(x+y*this->w) * 2];
	color = colorbuf[x+y*this->w];
	return true;
}

void Os2VioSystemDisplay::setBounds(const Bounds &b)
{
	SystemDisplay::setBounds(b);
	if (viobuf) { free(viobuf); viobuf = NULL; }
#ifdef USE_DBLBUF
	if (viobuf2) { free(viobuf2); viobuf2 = NULL; }
#endif
	if (colorbuf) { free(colorbuf); colorbuf = NULL; }
	
	viobuf = ht_malloc(w * h * 2);
#ifdef USE_DBLBUF
	viobuf2 = ht_malloc(w * h * 2);
	{
		int i, n = w*h*2;
		unsigned char attr = (unsigned char)vcpToSystem(VCP(VC_WHITE, VC_BLACK));
		
		for(i=0; i<n; i+=2) {
			viobuf2[i] = ' ';
			viobuf2[i+1] = attr;
		}
	}
#endif
	colorbuf = ht_malloc(w * h * sizeof(vcp));
	fill(x, y, w, h, VCP(VC_WHITE, VC_BLACK), ' ');
}

void Os2VioSystemDisplay::setCursor(int x, int y, CursorMode mode)
{
	cursorx = x;
	cursory = y;
	if (cursorx < 0) cursorx = 0;
	if (cursory < 0) cursory = 0;
	if (cursorx >= w) cursorx = w-1;
	if (cursory >= h) cursory = h-1;
	setCursorMode(mode);
}


void Os2VioSystemDisplay::setVioCursorMode(CursorMode mode)
{
	VIOCURSORINFO vc = {0};
	
	vc.cx = 1;
	switch(mode) {
	case CURSOR_OFF:
		vc.attr = (USHORT)(unsigned)(-1);
		break;
	case CURSOR_BOLD:
		vc.yStart = 0;
		vc.cEnd = (USHORT)(unsigned)(-100);
		vc.attr = 0x0f;
		break;
	case CURSOR_NORMAL:
		vc.yStart = (USHORT)(unsigned)(-(15 * 100)/16);
		vc.cEnd = (USHORT)(unsigned)(-100);
		vc.attr = 0x0f;
		break;
	default:
		return;
	}
	VioSetCurType(&vc, hvio);
}


void Os2VioSystemDisplay::setCursorMode(CursorMode mode)
{
	cursor_mode = mode;
}

void Os2VioSystemDisplay::putChar(int x, int y, uint rawchar, vcp vc)
{
	int ofs;
	if (x < 0 || y < 0 || x >= w || y >= h) return;
	ofs = x + y*w;
	if (rawchar) viobuf[ofs * 2] = rawchar;
	colorbuf[ofs] = mixColors(colorbuf[ofs], vc);
	vc = colorbuf[ofs];
#if 1
	uint fg, bg;
	if (VC_GET_BASECOLOR(VCP_BACKGROUND(vc))==VC_TRANSPARENT) {
		bg=viobuf[ofs*2+1]>>4;
	} else if (VC_GET_LIGHT(VCP_BACKGROUND(vc))) {
		bg=(VC_GET_BASECOLOR(VCP_BACKGROUND(vc)) & 7) | 8;
	} else {
		bg=(VC_GET_BASECOLOR(VCP_BACKGROUND(vc)) & 7);
	}
	if (VC_GET_BASECOLOR(VCP_FOREGROUND(vc))==VC_TRANSPARENT) {
		fg=viobuf[ofs*2+1]&0x0f;
	} else if (VC_GET_LIGHT(VCP_FOREGROUND(vc))) {
		fg=(VC_GET_BASECOLOR(VCP_FOREGROUND(vc)) & 7) | 8;
	} else {
		fg=(VC_GET_BASECOLOR(VCP_FOREGROUND(vc)) & 7);
	}
	viobuf[ofs*2+1] = (unsigned char)((bg << 4) | fg);
#else
	viobuf[ofs * 2 +1] = vcpToSystem(colorbuf[ofs]);
#endif
}


void Os2VioSystemDisplay::show()
{
	setVioCursorMode(CURSOR_OFF);
#ifdef USE_DBLBUF
	int buflim = w * h * 2;
	int ofs, cpycnt;
	bool need_redraw = false;
	
	for(ofs=0; ofs < buflim; ofs +=2) {
		if (*(USHORT *)(viobuf + ofs) != *(USHORT *)(viobuf2 + ofs)) {
			need_redraw = true;
			break;
		}
	}
	if (need_redraw) {
		for(cpycnt = buflim - ofs; cpycnt > 0; cpycnt -= 2) {
			if (*(USHORT *)(viobuf + ofs + cpycnt - 2) != *(USHORT *)(viobuf2 + ofs + cpycnt - 2))
				break;
		}
		if (cpycnt > 0) {
			VioWrtCellStr(viobuf + ofs, cpycnt, ofs / (w*2), (ofs % (w*2)) / 2, hvio);
			memcpy(viobuf2 + ofs, viobuf + ofs, cpycnt);
		}
		
	}
#else
	VioWrtCellStr(viobuf, w*h*2, 0, 0, hvio);
#endif
	VioSetCurPos(cursory, cursorx, hvio);
	setVioCursorMode(cursor_mode);
}

uint Os2VioSystemDisplay::vcpToSystem(vcp vc)
{
	byte fg, bg;
	if (VC_GET_LIGHT(VCP_BACKGROUND(vc))) {
		bg = (VC_GET_BASECOLOR(VCP_BACKGROUND(vc)) & 7);
	} else {
		bg = (VC_GET_BASECOLOR(VCP_BACKGROUND(vc)) & 7);
	}
	if (VC_GET_LIGHT(VCP_FOREGROUND(vc))) {
		fg = (VC_GET_BASECOLOR(VCP_FOREGROUND(vc)) & 7)+8;
	} else {
		fg = (VC_GET_BASECOLOR(VCP_FOREGROUND(vc)) & 7);
	}
	return ((bg<<4)|fg);
}


static int sysdisplay_count = 0;

SystemDisplay *allocSystemDisplay(const char *title)
{
	if (sysdisplay_count) return NULL;
	sysdisplay_count++;
	gDisplay = new Os2VioSystemDisplay(title);
	return gDisplay;
}

/*
void sys_display_enter()
{
	gDisplay->term_on();
}

void sys_display_leave()
{
	gDisplay->term_off();
}
*/


static bool gWinChFlag;


bool sys_get_winch_flag()
{
	return gWinChFlag;
}

void sys_set_winch_flag(bool f)
{
	gWinChFlag = f;
}

