/****************************************************************/
/*                                                              */
/* Fetches articles from an NNTP server and stores them locally */
/*             Also posts outgoing articles                     */
/*                                                              */
/*    Author:       Peter Moylan (peter@ee.newcastle.edu.au)    */
/*    Started:      11 February 2001                            */
/*    Last revised: 16 February 2001                            */
/*                                                              */
/* Usage:                                                       */
/*    getnews cfgfile                                           */
/*                                                              */
/*    The cfgfile parameter is the name of the configuration    */
/*    file.  If it is omitted, we assume "getnews.cfg".         */
/*    See CFG.DOC for the format of the configuration file.     */
/*                                                              */
/* Prerequisite: for this to work you must have RXU.DLL in      */
/* your LIBPATH.  RXU, a Rexx library written by Dave Boll, is  */
/* available from Hobbes with the name rxu1a.zip.               */
/*                                                              */
/****************************************************************/

/****************************************************************/
/*                       MAIN PROGRAM                           */
/****************************************************************/

parse arg cfgfile
IF cfgfile = '' THEN cfgfile = 'getnews.cfg'

CALL RxFuncAdd SysLoadFuncs, rexxutil, sysloadfuncs
CALL SysLoadFuncs
CALL RxFuncAdd "SockLoadFuncs","rxSock","SockLoadFuncs"
CALL SockLoadFuncs nul
RB = ""                                       /* Receive buffer */
Globals = "OurSocket RB mailroot"

outfile = "getnews.tmp"
IF STREAM(outfile, 'C', 'QUERY EXISTS') \= '' THEN '@del 'outfile' >nul'

WeaselDir = STRIP(LINEIN(cfgfile), 'T', '\')||'\'
CALL LINEOUT outfile,WeaselDir
newsserver = LINEIN(cfgfile)
CALL LINEOUT outfile,newsserver

mailroot = SysIni(WeaselDir||'WEASEL.INI', '$SYS', 'MailRoot')
j = POS('00'X,mailroot)
IF j > 0 THEN mailroot = LEFT(mailroot, j-1)
mailroot = TRANSLATE(STRIP(mailroot), '\', '/')

IF ConnectToServer(newsserver) THEN
    DO
        CALL PostOutgoingArticles
        CALL ReadAllGroups cfgfile outfile
        '@del 'cfgfile
        '@rename 'outfile' 'cfgfile
    END
ELSE
    '@del 'outfile

IF OurSocket \= -1 THEN
    DO
        CALL ExecCmd "QUIT"
        CALL SockSoClose(OurSocket)
    END

/* Tell Major Major to re-check its incoming mail. */

call rxfuncadd 'rxuinit','rxu','rxuinit'
call rxuinit
SemName = "\SEM32\WEASEL\RECEIVED"
if RxOpenEventSem(hev, SemName) \= 0 then
    rc = RxCreateEventSem( hev ,'Shared', SemName, 'Reset')
call RxPostEventSem hev
call RxResetEventSem hev
call RxCloseEventSem hev
call RxuTerm

EXIT

/****************************************************************/
/*                    HANDLING ALL GROUPS                       */
/****************************************************************/

ReadAllGroups: PROCEDURE expose (Globals)

    /* Reads the input file, retrieves new articles for all     */
    /* newsgroups specified there, writes an update to the      */
    /* output file.  The two file names are specified as        */
    /* arguments.                                               */

    PARSE ARG infile outfile
    DO FOREVER
        buffer = LINEIN(infile)
        IF STREAM(infile, 'S') \= 'READY' THEN LEAVE
        PARSE VAR buffer groupname number
        IF groupname \= '' THEN
            DO
                result = FetchGroup(groupname, number)
                CALL LINEOUT outfile, groupname result
            END
    END
    CALL STREAM infile, 'C', 'CLOSE'
    CALL STREAM outfile, 'C', 'CLOSE'
    RETURN

/****************************************************************/
/*                     HANDLING ONE GROUP                       */
/****************************************************************/

FetchGroup: PROCEDURE expose (Globals)

    /* Retrieves new articles for one newsgroup. */

    parse arg groupname, N
    CALL SendCmd "GROUP "groupname
    response = ReceiveLine()
    PARSE VAR response code numarticles lownum highnum name
    IF (code < 200) | (code > 299) then
        DO
             SAY "Can't read group "groupname", response code is "code
        END
    ELSE
        DO
            count = 0
            IF N < lownum THEN N = lownum
            DO WHILE N <= highnum
                count = count + FetchArticle(groupname, N)
                N = N+1
            END
            SAY
            SAY groupname': 'count' articles retrieved'
        END
    RETURN N

/****************************************************************/
/*                   HANDLING ONE ARTICLE                       */
/****************************************************************/

FetchArticle: PROCEDURE expose (Globals)

    /* Processes one article from the server. */

    parse arg groupname, N
    dir = mailroot||groupname'\'
    IF ExecCmd("ARTICLE "||N) THEN
        DO
            DO UNTIL STREAM(f, 'C', 'QUERY EXISTS') = ''
                f = SysTempFileName(dir'?????.ART' )
            END
            CALL FetchFile f
            CALL CHAROUT 'STDOUT', '.'
            DO UNTIL STREAM(f1, 'C', 'QUERY EXISTS') = ''
                f1 = SysTempFileName('?????.msg' )
            END
            '@rename 'f' 'f1
            RETURN 1
        END
    ELSE
        RETURN 0

/****************************************************************/
/*                     RECEIVING A FILE                         */
/****************************************************************/

FetchFile: PROCEDURE expose (Globals)

    /* Receives a file from the server, where the function      */
    /* parameter is the name of the file.  The doubled periods  */
    /* as specified in the NNTP protocol are allowed to remain  */
    /* in the file, because we're passing this file straight    */
    /* over to the mail system.                                 */

    PARSE ARG filename
    DO FOREVER
        line = ReceiveLine()
        IF line = '.' THEN LEAVE
        CALL LineOut filename,line
    END
    CALL STREAM filename, 'C', 'CLOSE'
    RETURN

/****************************************************************/
/*                       SENDING A FILE                         */
/****************************************************************/

SendFile: PROCEDURE expose (Globals)

    /* Sends a file to the server.  The function parameter is   */
    /* the name of the file.  We assume (because of the way     */
    /* Weasel stores data) that lines starting with '.' have    */
    /* already had the '.' doubled.                             */

    PARSE ARG f
    DO FOREVER
        buffer = LINEIN(f)
        IF STREAM(f, 'S') \= 'READY' THEN LEAVE
        CALL SendCmd buffer
    END
    CALL SendCmd '.'
    CALL STREAM f, 'C', 'CLOSE'
    RETURN

/****************************************************************/
/*                 CONNECTING TO THE SERVER                     */
/****************************************************************/

ConnectToServer: procedure expose (Globals)

    /* This procedure opens a connection to the NNTP server.    */

    parse arg server
    OurSocket = SockSocket("AF_INET", "SOCK_STREAM", "IPPROTO_TCP")
    IF OurSocket = -1 THEN
      DO
        say "Can't create socket"
        RETURN 0
      END
    IF DATATYPE(LEFT(server,1))='NUM' THEN host.addr = server
    ELSE
      DO
        IF \SockGetHostByName(server,'host.') THEN
          DO
            say "Unknown host"
            DROP host.
            RETURN 0
          END
      END

    target.!family = 'AF_INET'
    target.!port = 119
    target.!addr = host.addr
    DROP host.
    IF SockConnect(OurSocket,"target.!") = -1 THEN
      DO
        say "Failed to connect"
        RETURN 0
      END
    IF \OKReply() THEN
      DO
        say "Server rejected us"
        RETURN 0
      END
    RETURN 1

/****************************************************************/

SendCmd: PROCEDURE expose (Globals)

    /* Sends a one-line command to the server */

    PARSE ARG str
    RETURN (SockSend(OurSocket, str||'0D0A'X) \= -1)

/****************************************************************/

ExecCmd: PROCEDURE expose (Globals)

    /* Like SendCmd, but also checks for a positive reply. */

    PARSE ARG str
    RETURN (SockSend(OurSocket, str||'0D0A'X) \= -1) & OKReply()

/****************************************************************/

OKReply: PROCEDURE expose (Globals)

    /* Checks that server returned an "OK" response */

    RETURN LEFT(ReceiveLine(),1) = '2'

/****************************************************************/
/*              RECEIVING ONE LINE FROM THE SERVER              */
/*                                                              */
/*  Note: the global variable RB is the receive buffer; it      */
/*  holds input that has not yet been processed.                */
/****************************************************************/

ReceiveLine: PROCEDURE expose (Globals)
    line = ''
    DO FOREVER
      len = LENGTH(RB)
      IF len = 0 THEN
        DO
          NullResponseCount = 0
          DO WHILE len = 0
            len = SockRecv(OurSocket, 'RB', 256)
            IF len = 0 THEN
              DO
                NullResponseCount = NullResponseCount+1
                IF NullResponseCount > 20 THEN len = -1
              END
          END /*do-while*/
          IF len = -1 THEN RETURN ''
        END /* if len=0 */
      j0 = POS('0A'X, RB)
      IF j0 = 0 THEN
        DO
          line = STRIP(line||RB, 'T', '0D'X)
          RB = ''
        END
      ELSE
        DO
          line = STRIP(line||LEFT(RB,j0-1), 'T', '0D'X)
          RB = RIGHT(RB,len-j0)
          RETURN line
        END
    END /* do forever */

/****************************************************************/
/*                  POSTING OUTGOING ARTICLES                   */
/****************************************************************/

PostOutgoingArticles: PROCEDURE expose (Globals)

    /* Sends all files in the 'outgoing' directory to the       */
    /* server.                                                  */

    call SysFileTree 'outgoing\*.ART','file','FO'
    count = 0
    DO j=1 TO file.0
        CALL SendCmd 'POST'
        IF LEFT(ReceiveLine(),1) = '3' THEN
            DO
                CALL SendFile file.j
                response = ReceiveLine()
                IF LEFT(response,1) = '2' THEN
                    DO
                        '@del 'file.j
                        count = count+1
                    END
                ELSE
                    SAY "Article rejected, reply was: "response
            END
    END
    SAY count" articles posted"
    RETURN

/****************************************************************/

