#ifndef INCL_ENDIAN_HPP
#define INCL_ENDIAN_HPP

#include <stdlib.h>
#ifdef __GNUC__
#include <sys/builtin.h>
#endif
#include <algorithm>

/*************************************************************************
 *                                                                       *
 * endian.hpp                                                            *
 * Header file for endian conversion class                               *
 *                                                                       *
 *************************************************************************/


// This class will convert little-endian to big-endian and vice versa
template <typename T>
class EndianSwap
{private:
   union SwapUnion
   {  unsigned char Chars[sizeof(T)];
      T             Value;
   };
 public:
   static T       Convert(const T& v)
   {  SwapUnion s;
      s.Value = v;
      std::reverse(s.Chars, s.Chars + sizeof(T));
      return s.Value;
   }
};

// trivial case
template <>
class EndianSwap<unsigned char>
{public:
   static unsigned char Convert(unsigned char v) { return v; }
};
template <>
class EndianSwap<char>
{public:
   static char Convert(char v) { return v; }
};

#ifdef __GNUC__
// faster implementation for short ints if the compiler supports access to the 16 bit rotate instruction of the CPU.
template<>
class EndianSwap<unsigned short>
{public:
   static unsigned short Convert(unsigned short v) { return _srotl(v, 8); }
};
template<>
class EndianSwap<short>
{public:
   static short Convert(short v) { return _srotl(v, 8); }
};
#endif

// This class will operate as functor and covert the endianness, if configured.
struct Endian
{  bool           Swap;

   // Initialize this class with true, if the endianness of the source
   // is different to that of the destination.
   explicit       Endian(bool swap = true)            : Swap(swap) {}
   // Copy constructor. Copy is less expensive than a reference.
                  Endian(const Endian& r)             : Swap(r.Swap) {}

   template <typename T>
   T              operator()(const T& v) const        { return Swap ? EndianSwap<T>::Convert(v) : v; }
};

#endif
