#ifndef INCL_IFD_HPP
#define INCL_IFD_HPP


/*************************************************************************
 *                                                                       *
 * exif.hpp                                                              *
 * Header file for EXIF specific code                                    *
 *                                                                       *
 *************************************************************************/

#include "Endian.hpp"
#include <iostream>
#include <map>
#include <string>


class IFD
{public: // subtypes
   enum IFDTypes
   {  TIFF_UNDEFINED          = 0, // Not any tag, type, or other TIFF structure
      TIFF_Byte               = 1,
      TIFF_Ascii              = 2,
      TIFF_Short              = 3,
      TIFF_Long               = 4,
      TIFF_Rational           = 5,
      TIFF_SByte              = 6,
      TIFF_Undefined          = 7,
      TIFF_SShort             = 8,
      TIFF_SLong              = 9,
      TIFF_SRational          = 10,
      TIFF_Float              = 11,
      TIFF_Double             = 12,
   };
   typedef unsigned char      TtiffByte;
   typedef signed   char      TtiffSByte;
   typedef unsigned short     TtiffShort;
   typedef signed   short     TtiffSShort;
   typedef unsigned int       TtiffLong;
   typedef signed   int       TtiffSLong;
   typedef struct
   {  TtiffLong numerator;
      TtiffLong denominator;
   }                          TtiffRational;
   typedef struct
   {  TtiffSLong numerator;
      TtiffSLong denominator;
   }                          TtiffSRational;
   typedef char               TtiffAscii;
   typedef float              TtiffFloat;
   typedef double             TtiffDouble;
   typedef unsigned char      TtiffUndefined;

   struct Value
   {  static const int TypeLengths[12];
      const Endian   Convert;
      TtiffShort     Type;
      const char*    Data;

      Value()        : Type(0) {}
      Value(const Endian& convert)           : Convert(convert), Type(0) {}
      Value(const Endian& convert, TtiffShort type, const char* data) : Convert(convert), Type(type), Data(data) {}

      unsigned long  GetInt() const;
      long           GetSInt() const;
      TtiffRational  GetRational() const;
      TtiffSRational GetSRational() const;
      double         GetFloat() const;
      const void*    GetRAW() const          { return Data; }
      static void    CancelRational(TtiffRational& r);
      static void    CancelRational(TtiffSRational& r);
    private:
      void           toostream(std::ostream& os) const; // called by ostream << IFD::Value
      friend std::ostream& operator<< (std::ostream& os, const Value& v);
    protected:
      template <typename T>
      const T        Fetch(int subitem = 0) const { return Convert(((T*)Data)[subitem]); }
   };

   struct ValueList : Value
   {  TtiffLong      Count; // number of components
      ValueList() : Count(0) {}
      ValueList(const Endian& convert) : Value(convert), Count(0) {}
      ValueList(const Endian& convert, TtiffShort type, TtiffShort count, const char* data) : Value(convert, type, data), Count(count) {}
      const Value    operator[](unsigned element) const;
      std::string    GetASCII() const;
    private:
      void           toostream(std::ostream& os) const; // called by ostream << IFD::ValueList
      friend std::ostream& operator<< (std::ostream& os, const ValueList& v);
   };

 protected:
   Endian            Convert;
   const char*       Root;
   const char*       DataEnd;
   static const ValueList NoValue;
 private:
   const char*       Data;
   typedef std::map<int, ValueList> IFDListType;
   IFDListType       IFDList;
   static const char* TypeNames[];
 public:
   typedef IFDListType::const_iterator const_iterator;

 public:
   IFD(const Endian& convert, const char* root, const char* data, const char* dataend) { Reset(convert, root, data, dataend); }
   virtual           ~IFD() {}
   bool              Parse();
   const ValueList&  operator[](int tag) const;
   const_iterator    begin() const           { return IFDList.begin(); }
   const_iterator    end() const             { return IFDList.end(); }

 protected:
   IFD()                                     : Convert(false), Root(NULL), DataEnd(NULL), Data(NULL) {}
   void              Reset(const Endian& convert, const char* root, const char* data, const char* dataend);
   void              Reset(TtiffLong offset);
   template <typename T>
   const T           Fetch();
   virtual void      itemtoostream(std::ostream& os, const IFDListType::value_type& item) const;
   virtual void      toostream(std::ostream& os) const;
 private:
   friend std::ostream& operator<< (std::ostream& os, const IFD& ifd);
   void              Check();
};


class TIFF : public IFD
{public:
   enum TIFF_
   {  TIFF_II                          = 0x4949U, // Intel byte order: 76543210 stored as 10325476
      TIFF_MM                          = 0x4d4dU, // Motorola byte order: 76543210 stored as 76543210
      TIFF_42                          = 0x002aU, // TIFF file identifier
   };

 public:
   TIFF(const char* tiffheader, const char* dataend);
   void              Parse();
 protected:
   TIFF()                              : IFD(Endian(false), NULL, NULL, NULL) {}
   using IFD::Reset;
   void Reset(const char* tiffheader, const char* dataend) { IFD::Reset(Endian(false), tiffheader, tiffheader, dataend); }
};


class Exif : public TIFF
{public:
   enum JPEGMarker
   {  JPEG_SOF                         = 0xffc0U,
      JPEG_DHT                         = 0xffc4U,
      JPEG_SOI                         = 0xffd8U,
      JPEG_EOI                         = 0xffd9U,
      JPEG_SOS                         = 0xffdaU,
      JPEG_DQT                         = 0xffdbU,
      JPEG_DRI                         = 0xffddU,
      JPEG_APP0                        = 0xffe0U,
      JPEG_APP1                        = 0xffe1U,
      JPEG_APP2                        = 0xffe2U,
      JPEG_APP3                        = 0xffe3U,
      JPEG_APP4                        = 0xffe4U,
      JPEG_APP5                        = 0xffe5U,
      JPEG_APP6                        = 0xffe6U,
      JPEG_APP7                        = 0xffe7U,
      JPEG_APP8                        = 0xffe8U,
      JPEG_APP9                        = 0xffe9U,
      JPEG_APP10                       = 0xffeaU,
      JPEG_APP11                       = 0xffebU,
      JPEG_APP12                       = 0xffecU,
      JPEG_APP13                       = 0xffedU,
      JPEG_APP14                       = 0xffeeU,
      JPEG_APP15                       = 0xffefU
   };
   enum IFD0Tag
   {  TIFF_NewSubfileType              = 0x00feU,
      TIFF_SubfileType                 = 0x00ffU,
      TIFF_ImageWidth                  = 0x0100U,
      TIFF_ImageLength                 = 0x0101U,
      TIFF_BitsPerSample               = 0x0102U,
      TIFF_Compression                 = 0x0103U,
      TIFF_PhotometricInterpretation   = 0x0106U,
      TIFF_Thresholding                = 0x0107U,
      TIFF_CellWidth                   = 0x0108U,
      TIFF_CellLength                  = 0x0109U,
      TIFF_FillOrder                   = 0x010aU,
      TIFF_DocumentName                = 0x010dU,
      TIFF_ImageDescription            = 0x010eU,
      TIFF_Make                        = 0x010fU,
      TIFF_Model                       = 0x0110U,
      TIFF_StripOffset                 = 0x0111U,
      TIFF_Orientation                 = 0x0112U,
      TIFF_SamplesPerPixel             = 0x0115U,
      TIFF_RowsPerStrip                = 0x0116U,
      TIFF_StripByteCounts             = 0x0117U,
      TIFF_MinSampleValue              = 0x0118U,
      TIFF_MaxSampleValue              = 0x0119U,
      TIFF_XResolution                 = 0x011aU,
      TIFF_YResolution                 = 0x011bU,
      TIFF_PlanarConfiguration         = 0x011cU,
      TIFF_PageName                    = 0x011dU,
      TIFF_XPosition                   = 0x011eU,
      TIFF_YPosition                   = 0x011fU,
      TIFF_FreeOffsets                 = 0x0120U,
      TIFF_FreeByteCounts              = 0x0121U,
      TIFF_GrayResponseUnit            = 0x0122U,
      TIFF_GrayResponseCurve           = 0x0123U,
      TIFF_T4Options                   = 0x0124U,
      TIFF_T6Options                   = 0x0125U,
      TIFF_ResolutionUnit              = 0x0128U,
      TIFF_PageNumber                  = 0x0129U,
      TIFF_TransferFunction            = 0x012dU,
      TIFF_Software                    = 0x0131U,
      TIFF_DateTime                    = 0x0132U,
      TIFF_Artist                      = 0x013bU,
      TIFF_HostComputer                = 0x013cU,
      TIFF_Predictor                   = 0x013dU,
      TIFF_WhitePoint                  = 0x013eU,
      TIFF_PrimaryChromaticities       = 0x013fU,
      TIFF_ColorMap                    = 0x0140U,
      TIFF_HalftoneHints               = 0x0141U,
      TIFF_TileWidth                   = 0x0142U,
      TIFF_TileLength                  = 0x0143U,
      TIFF_TileOffsets                 = 0x0144U,
      TIFF_TileByteCounts              = 0x0145U,
      TIFF_SubIfds                     = 0x014aU,
      TIFF_InkSet                      = 0x014cU,
      TIFF_InkNames                    = 0x014dU,
      TIFF_NumberOfInks                = 0x014eU,
      TIFF_DotRange                    = 0x0150U,
      TIFF_TargetPrinter               = 0x0151U,
      TIFF_ExtraSamples                = 0x0152U,
      TIFF_SampleFormat                = 0x0153U,
      TIFF_SMinSampleValue             = 0x0154U,
      TIFF_SMaxSampleValue             = 0x0155U,
      TIFF_TransferRange               = 0x0156U,
      TIFF_JpegTables                  = 0x015bU,
      TIFF_JpegProc                    = 0x0200U,
      TIFF_JpegInterchangeFormat       = 0x0201U,
      TIFF_JpegInterchangeFormatLength = 0x0202U,
      TIFF_JpegRestartInterval         = 0x0203U,
      TIFF_JpegLosslessPredictors      = 0x0205U,
      TIFF_JpegPointTransforms         = 0x0206U,
      TIFF_JpegQTables                 = 0x0207U,
      TIFF_JpegDcTables                = 0x0208U,
      TIFF_JpegAcTables                = 0x0209U,
      TIFF_YCbCrCoefficients           = 0x0211U,
      TIFF_YCbCrSubSampling            = 0x0212U,
      TIFF_YCbCrPositioning            = 0x0213U,
      TIFF_ReferenceBlackWhite         = 0x0214U,
      TIFF_Copyright                   = 0x8298U,
      TIFF_ExifIFDPointer              = 0x8769U,   /* Offset to EXIF SubIFD */
      TIFF_GPSInfoIFDPointer           = 0x8825U,
      TIFF_InteroperabilityIFDPointer  = 0xa005U,
   };
   enum InteroperabilityTag
   {  EXIF_InteroperabilityIndex       = 0x0001U,
      EXIF_InteroperabilityVersion     = 0x0002U,
   };
   enum ExifTag
   {  EXIF_CfaRepeatPatternDim         = 0x828dU,
      EXIF_CfaPattern                  = 0x828eU,
      EXIF_BatteryLevel                = 0x828fU,
      EXIF_ExposureTime                = 0x829aU,
      EXIF_FNumber                     = 0x829dU,
      EXIF_IptcNaa                     = 0x83bbU,
      EXIF_InterColorProfile           = 0x8773U,
      EXIF_ExposureProgram             = 0x8822U,
      EXIF_SpectralSensitivity         = 0x8824U,
      EXIF_ISOSpeedRatings             = 0x8827U,
      EXIF_OECF                        = 0x8828U,
      EXIF_Interlace                   = 0x8829U,
      EXIF_TimeZoneOffset              = 0x882aU,
      EXIF_SelfTimerMode               = 0x882bU,
      EXIF_ExifVersion                 = 0x9000U,
      EXIF_DateTimeOriginal            = 0x9003U,
      EXIF_DateTimeDigitized           = 0x9004U,
      EXIF_ComponentConfiguration      = 0x9101U,
      EXIF_CompressedBitsPerPixel      = 0x9102U,   /* CompressedBitsPerPixel */
      EXIF_ShutterSpeedValue           = 0x9201U,
      EXIF_ApertureValue               = 0x9202U,
      EXIF_BrightnessValue             = 0x9203U,
      EXIF_ExposureBiasValue           = 0x9204U,
      EXIF_MaxApertureValue            = 0x9205U,
      EXIF_SubjectDistance             = 0x9206U,
      EXIF_MeteringMode                = 0x9207U,
      EXIF_LightSource                 = 0x9208U,
      EXIF_Flash                       = 0x9209U,
      EXIF_FocalLength                 = 0x920aU,
      EXIF_Noise                       = 0x920dU,
      EXIF_ImageNumber                 = 0x9211U,
      EXIF_SecurityClassification      = 0x9212U,
      EXIF_ImageHistory                = 0x9213U,
      EXIF_SubjectArea                 = 0x9214U,
      EXIF_TiffEpStandardId            = 0x9216U,
      EXIF_MakerNote                   = 0x927cU,
      EXIF_UserComment                 = 0x9286U,
      EXIF_SubSecTime                  = 0x9290U,
      EXIF_SubSecTimeOriginal          = 0x9291U,
      EXIF_SubSecTimeDigitized         = 0x9292U,
      EXIF_FlashpixVersion             = 0xa000U,
      EXIF_ColorSpace                  = 0xa001U,
      EXIF_PixelXDimension             = 0xa002U,
      EXIF_PixelYDimension             = 0xa003U,
      EXIF_RelatedSoundFile            = 0xa004U,
      EXIF_FlashEnergy                 = 0xa20bU,
      EXIF_SpatialFrequencyResponse    = 0xa20cU,
      EXIF_FocalPlaneXResolution       = 0xa20eU,
      EXIF_FocalPlaneYResolution       = 0xa20fU,
      EXIF_FocalPlaneResolutionUnit    = 0xa210U,
      EXIF_SubjectLocation             = 0xa214U,
      EXIF_ExposureIndex               = 0xa215U,
      EXIF_SensingMethod               = 0xa217U,
      EXIF_FileSource                  = 0xa300U,
      EXIF_SceneType                   = 0xa301U,
      EXIF_CFAPattern                  = 0xa302U,
      EXIF_CustomRendered              = 0xa401U,
      EXIF_ExposureMode                = 0xa402U,
      EXIF_WhiteBalance                = 0xa403U,
      EXIF_DigitalZoomRatio            = 0xa404U,
      EXIF_FocalLengthIn35mmFilm       = 0xa405U,
      EXIF_SceneCaptureType            = 0xa406U,
      EXIF_GainControl                 = 0xa407U,
      EXIF_Contrast                    = 0xa408U,
      EXIF_Saturation                  = 0xa409U,
      EXIF_Sharpness                   = 0xa40aU,
      EXIF_DeviceSettingDescription    = 0xa40bU,
      EXIF_SubjectDistanceRange        = 0xa40cU,
      EXIF_ImageUniqueID               = 0xa420U
   };
   enum GPSTag
   {  EXIF_GPSVersionID                = 0x0000U,
      EXIF_GPSLatitudeRef              = 0x0001U,
      EXIF_GPSLatitude                 = 0x0002U,
      EXIF_GPSLongitudeRef             = 0x0003U,
      EXIF_GPSLongitude                = 0x0004U,
      EXIF_GPSAltitudeRef              = 0x0005U,
      EXIF_GPSAltitude                 = 0x0006U,
      EXIF_GPSTimeStamp                = 0x0007U,
      EXIF_GPSSatelites                = 0x0008U,
      EXIF_GPSStatus                   = 0x0009U,
      EXIF_GPSMeasureMode              = 0x000aU,
      EXIF_GPSDOP                      = 0x000bU,
      EXIF_GPSSpeedRef                 = 0x000cU,
      EXIF_GPSSpeed                    = 0x000dU,
      EXIF_GPSTrackRef                 = 0x000eU,
      EXIF_GPSTrack                    = 0x000fU,
      EXIF_GPSImgDirectionRef          = 0x0010U,
      EXIF_GPSImgDirection             = 0x0011U,
      EXIF_GPSMapDatum                 = 0x0012U,
      EXIF_GPSDestLatitudeRef          = 0x0013U,
      EXIF_GPSDestLatitude             = 0x0014U,
      EXIF_GPSDestLongitudeRef         = 0x0015U,
      EXIF_GPSDestLongitude            = 0x0016U,
      EXIF_GPSDestBearingRef           = 0x0017U,
      EXIF_GPSDestBearing              = 0x0018U,
      EXIF_GPSDestDistanceRef          = 0x0019U,
      EXIF_GPSDestDistance             = 0x001aU,
      EXIF_GPSProcessingMethod         = 0x001bU,
      EXIF_GPSAreaInformation          = 0x001cU,
      EXIF_GPSDateStamp                = 0x001dU,
      EXIF_GPSDifferential             = 0x001eU
   };

 protected:
   class DecodeEntry
   {public:
      enum           Flag
      {  Basic       = 0x00,
         Detail      = 0x01,
         Internal    = 0x02,
         MoreAfter   = 0x10,
         MoreBefore  = 0x20,
      };
      typedef void (DecodeEntry::*TDisplayFunction)(std::ostream& os, const IFD::ValueList& vl) const;
    public:
      int            Tag;
      int            Flags;
      const char*    Name;
      const char*    Unit;
      TDisplayFunction DisplayFunction;
    private:
      static const DecodeEntry Table[];
      typedef std::map<int, const Exif::DecodeEntry*> SortedDecodeTable;
      static SortedDecodeTable SortedTable;
    private:
      static SortedDecodeTable SortTable();
    private:         // Display functions
      void           DDefault(std::ostream& os, const IFD::ValueList& vl) const;
      void           DasASCII(std::ostream& os, const IFD::ValueList& vl) const;
      void           DToken(std::ostream& os, const IFD::ValueList& vl) const;
      void           DCharSet(std::ostream& os, const IFD::ValueList& vl) const;
    public:
      static const DecodeEntry* Find(int tag);
   };
 private:
   mutable int       LastItemFlags; // internal state of the item decoder
 protected:
   virtual void      itemtoostream(std::ostream& os, const IFDListType::value_type& item) const;
   virtual void      toostream(std::ostream& os) const;
 private:
   static bool       CompareDecodeEntry(const DecodeEntry& de, int tag);
 public:
   bool              ShowUnknown;
 public:
   Exif(const char* fileheader, const char* dataend) throw (std::string);
   //~Exif();
   void              Parse();
 private:
   void              ParseSubIFD(int tag, const char* name) throw();
};




// template/inline functions
inline std::ostream& operator<< (std::ostream& os, const IFD::Value& v)
{  v.toostream(os);
   return os;
}

inline std::ostream& operator<< (std::ostream& os, const IFD::ValueList& v)
{  v.toostream(os);
   return os;
}

inline std::ostream& operator<< (std::ostream& os, const IFD& ifd)
{  ifd.toostream(os);
   return os;
}

template <typename T>
const T IFD::Fetch()
{  if (Data + sizeof(T) > DataEnd)
      throw stringf("IFD error: tried to fetch data element larger than the remaining data.\n (Offset %x, Length %x, Remaining %x)", Data-Root, sizeof(T), DataEnd-Data);
   return Convert(*((const T*&)Data)++);
}
#endif
