/* REXX */
/*
   A mini pipe mixer control REXX script (third edition) based on work of
   Lesha Bogdanow.
   Works at least with my cheap noname Crystal-PCI-CS4614 Sound-Card.
   Mix90.exe or 'lbmix -ps' or something like this has to run in background
   to be able to use this script.

   Usable keys: Cursor up/down left/right,
                Ctrl + Cursor up/down,
                Ctrl + Cursor left/right,
                TAB, Shift+TAB,
                SPACE, m, q.

   History:     6.1.2001  Buxfix and small changes to display muting-state of
                master control correctly, which is supported now in LBMix 0.06.

   Copyright:   >freeware< If you like this little script, find a bug
                or have a good idea to enhance it, you can send email to
                fbakan@gmx.net    Franz Bakan, 6. 1. 2001

   Change       2004.08.05 Veit Kannegieser
                * adapted for newer pipe protocol with additional flags
                  in mute byte
                * try to find if the control is mono (display only)
                * altered controls to suit M2E hardware
                * press shift+letter (repeatedly) to switch between controls
                  e.g. 'M' jumps between Master and Mic
*/

CALL RxFuncAdd 'SysLoadFuncs','RexxUtil','SysLoadFuncs'
CALL SysLoadFuncs

ESC = "1B"x||"["
no  = ESC||"44;0;37m" /* normal */
bd  = ESC||"1m"       /* bold   */
reverse  = ESC||"7m"
muted    = reverse
red      = ESC||"31;1m" /* bold red */
selected = ESC||"44;37;1m"

pipe="\PIPE\MIXER"

controls = 0
/*                Name     , query-string, set-string */
call addcontrol   " Master", "11", "01"
call addcontrol   "   DA  ", "6D", "4D"
call addcontrol   "  Line ", "63", "43"
call addcontrol   "   CD  ", "64", "44"
call addcontrol   "  Mic  ", "62", "42"
call addcontrol   "   3D  ", "6C", "4C"
call addcontrol   " Beeper", "60", "40"


/* increase number of controls and enter additional lines here. For example: */
/* call addcontrol   "  AUX  ", "66", "46" */
/* see ioctl90.h or http://www.pompano.net/~joenord/ioctl90/ for details */

controls = controls+1
lc = controls - 1  /* last control */
ctrl.lc.n = "  "||red||"Rec  " ; ctrl.lc.q = "6F" ; ctrl.lc.set = "4F"


idx = 2 /* Default to Line */
idx = 0 /* Default to Master */


'@MODE 'controls*8 3
CALL SysCls
CALL SysCurState(OFF)

/* query status */
DO i = 0 to controls-1
  CALL queryvolume i
  ctrl.i.atr = no /* text atribut */
  rc = SysCurPos(1,i*8)
  IF ctrl.i.s THEN DO
    control_disp = ''
    IF ismute(i) THEN
      control_disp = control_disp||muted
    control_disp = control_disp||FORMAT(ctrl.i.left,3)
    IF ismono(i) THEN
      control_disp = '  '||control_disp||'  '
    ELSE
      control_disp = control_disp||' '||FORMAT(ctrl.i.right,3)
    IF ismute(i) THEN
      control_disp = control_disp||no
    SAY control_disp
  END
END

ctrl.idx.atr = selected
rc = SysCurPos(2,0)
rc = CHAROUT(, ' 'bd||'m'||no||'=toggle_mute ' bd||'q'||no||'=quit')

CALL output

DO FOREVER
  key=SysGetKey(NOECHO)
  SELECT
    WHEN key == "09"x THEN DO /* select source */
      ctrl.idx.atr = no
      idx = idx + 1
      IF idx == controls THEN idx = 0
      ctrl.idx.atr = selected
      CALL output
    END /* tab */

    WHEN (key == 'm' | key == ' ') THEN DO /* toggle mute */
      IF ismute(idx) THEN ctrl.idx.mute = D2X( X2D(ctrl.idx.mute) - 1)
      ELSE                ctrl.idx.mute = D2X( X2D(ctrl.idx.mute) + 1)
      CALL changevolume
    END /* m */

    WHEN key == 'q' THEN DO
      '@mode 80 25'
      SAY no
      CALL SysCls
      EXIT(0)
    END /* q */

    WHEN key == "00"x | key == "E0"x THEN DO  /* Function Keys */
      key2 = SysGetKey(NOECHO)
      SELECT
        WHEN C2D(key2) == 15 | C2D(key2) == 75 THEN DO /* select source */
          ctrl.idx.atr = no
          idx = idx -1
          IF idx == -1 THEN idx = controls-1
          ctrl.idx.atr = selected
          CALL output
        END /* shift tab  cursor left */

        WHEN C2D(key2) == 77 THEN DO /* select source */
          ctrl.idx.atr = no
          idx = idx +1
          IF idx == controls THEN idx = 0
          ctrl.idx.atr = selected
          CALL output
        END /*  cursor right */

        WHEN C2D(key2) == 72 THEN DO /* Cursor up: increase volume */
          IF ctrl.idx.left < 100 THEN ctrl.idx.left = ctrl.idx.left + 1
          IF ctrl.idx.right < 100 THEN ctrl.idx.right = ctrl.idx.right + 1
          CALL changevolume
        END /* 72 */

        WHEN C2D(key2) == 141 THEN DO /* Ctrl+Cursor up: increase volume +10 */
          ctrl.idx.left = ctrl.idx.left + 10
          IF ctrl.idx.left > 100 THEN ctrl.idx.left = 100
          ctrl.idx.right = ctrl.idx.right + 10
          IF ctrl.idx.right > 100 THEN ctrl.idx.right = 100
          CALL changevolume
        END /* 141 */

        WHEN C2D(key2) == 80 THEN DO /* Cursor down: decrease volume */
          IF ctrl.idx.left > 0 THEN ctrl.idx.left = ctrl.idx.left - 1
          IF ctrl.idx.right > 0 THEN ctrl.idx.right = ctrl.idx.right - 1
          CALL changevolume
        END /* 80 */

        WHEN C2D(key2) == 145 THEN DO /* Ctrl+Cursor down: decrease volume -10 */
          ctrl.idx.left = ctrl.idx.left - 10
          IF ctrl.idx.left < 0 THEN ctrl.idx.left = 0
          ctrl.idx.right = ctrl.idx.right - 10
          IF ctrl.idx.right < 0 THEN ctrl.idx.right = 0
          CALL changevolume
        END /* 145 */

        WHEN C2D(key2) == 115 & idx <> 7 THEN DO /* Ctrl+Cursor left: balance to left */
          IF ctrl.idx.left < 100 THEN ctrl.idx.left = ctrl.idx.left +1
          IF ctrl.idx.right > 0 THEN ctrl.idx.right = ctrl.idx.right -1
          CALL changevolume
        END /* 75 */

        WHEN C2D(key2) == 116 & idx <> 7 THEN DO /* Ctrl+Cursor right: balance to right */
          IF ctrl.idx.left > 0 THEN ctrl.idx.left = ctrl.idx.left -1
          IF ctrl.idx.right < 100 THEN ctrl.idx.right = ctrl.idx.right +1
          CALL changevolume
        END /* 77 */

      otherwise
        nop
      END /* select key2 */
    END /* function Key */


    WHEN (key>='A') & (key<='Z') | (key>='a') & (key<='z') THEN do
      count_j=idx+1
      DO count_i=1 TO controls-1
        IF count_j>=controls THEN count_j=count_j-controls
        /* can not use first key, because of ansi colour markup
        IF Translate(Left(Strip(ctrl.count_j.n),1,' ')) == Translate(key) THEN */
        IF Pos(Translate(key),ctrl.count_j.n) \== 0 THEN
          DO
            ctrl.idx.atr = no
            idx=count_j
            ctrl.idx.atr = selected
            CALL output
            Leave count_i
          END
        count_j=count_j+1
      END
    END

    otherwise
      nop
  END /* select key */
END

EXIT

addcontrol:
  Parse Arg ctrl.controls.n, ctrl.controls.q, ctrl.controls.set
  controls = controls + 1
  Return

topipe:
  IF STREAM(pipe,'c','OPEN') <> 'READY:' THEN DO
    SAY 'Unable to connect to pipe for write' pipe
    Call SysSleep 4
    EXIT -1
  END
  IF lineout(pipe, dataout) <> 0 THEN DO
    SAY 'Unable to write to pipe' pipe
    EXIT -2
  END
  datain = ''
  DO ii=1 to 16
    data.ii = CHARIN(pipe)
  END
  IF STREAM(pipe,'c','CLOSE') <> 'READY:' THEN DO
    SAY 'Unable to close pipe' pipe
    EXIT -3
  END
  DO ii = 1 to 16
    datain = datain||data.ii
  END
  CALL SysSleep 0.01
RETURN

output:
  rc = SysCurPos(0,0)
  CALL SysCurState(OFF)
  DO i = 0 to controls - 1
    rc = CHAROUT(, ctrl.i.atr||ctrl.i.n||no||' ')
  END
RETURN

changevolume:
  dataout = ctrl.idx.set ctrl.idx.left ctrl.idx.right ctrl.idx.mute
  rc = SysCurPos(1,idx*8)
  IF ctrl.idx.s THEN DO
    control_disp = ''
    IF ismute(idx) THEN
      control_disp = control_disp||muted
    control_disp = control_disp||FORMAT(ctrl.idx.left,3)
    IF ismono(idx) THEN
      control_disp = '  '||control_disp||'  '
    ELSE
      control_disp = control_disp||' '||FORMAT(ctrl.idx.right,3)
    IF ismute(idx) THEN
      control_disp = control_disp||no
    SAY control_disp
  END
  CALL topipe
RETURN

ismute:
  Parse ARG ismute_idx
  /* mute flag is 1h */
  Return (X2D(ctrl.ismute_idx.mute) // 2)

ismono:
  Parse ARG ismono_idx
  /* Mono controls are MONOIN(,PHONE,MIC?),3D */
  IF ctrl.ismono_idx.q == '60' /* | ctrl.ismono_idx.q == '61' | ctrl.ismono_idx.q == '62' */ | ctrl.ismono_idx.q == '6C' THEN Return 1
  /* mono flag is 100h */
  Return ((X2D(ctrl.ismono_idx.mute) % 256) // 2)

queryvolume:
  ARG id
  dataout = ctrl.id.q
  CALL topipe
  IF datain == "" THEN datain = "xx 50 50 0" /* in case query status fails */
  ctrl.id.left  = word(datain,2) /* volume left  */
  ctrl.id.right = word(datain,3) /* volume right */
  ctrl.id.mute  = word(datain,4) /* mute         */
  ctrl.id.mute  = word(datain,4)
  IF datatype(ctrl.id.left)='NUM' THEN ctrl.id.s = 1 /* supported    */
  ELSE DO
     ctrl.id.s     = 0
     ctrl.id.left  = 0
     ctrl.id.right = 0
     ctrl.id.mute  = 0
  END
RETURN
