/*

Copyright (C) 1996, 1997 John W. Eaton

This file is part of Octave.

Octave is free software; you can redistribute it and/or modify it
under the terms of the GNU General Public License as published by the
Free Software Foundation; either version 2, or (at your option) any
later version.

Octave is distributed in the hope that it will be useful, but WITHOUT
ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
for more details.

You should have received a copy of the GNU General Public License
along with Octave; see the file COPYING.  If not, write to the Free
Software Foundation, 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

*/

#if defined (__GNUG__)
#pragma implementation
#endif

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "error.h"
#include "oct-obj.h"
#include "oct-lvalue.h"
#include "pager.h"
#include "pt-const.h"
#include "pt-id.h"
#include "pt-walk.h"
#include "symtab.h"
#include "utils.h"
#include "variables.h"

// Symbols from the symbol table.

string
tree_identifier::name (void) const
{
  string retval;
  if (sym)
    retval = sym->name ();
  return retval;
}

tree_identifier *
tree_identifier::define (octave_function *f, unsigned int sym_type)
{
  int status = sym->define (f, sym_type);
  return status ? this : 0;
}

void
tree_identifier::document (const string& s)
{
  if (sym)
    sym->document (s);
}

bool
tree_identifier::is_defined (void)
{
  return (sym && sym->is_defined ());
}

bool
tree_identifier::is_function (void)
{
  return (sym && sym->is_function ());
}

void
tree_identifier::eval_undefined_error (void)
{
  int l = line ();
  int c = column ();

  if (l == -1 && c == -1)
    ::error ("`%s' undefined", name ().c_str ());
  else
    ::error ("`%s' undefined near line %d column %d",
	     name ().c_str (), l, c);
}

// Try to find a definition for an identifier.  Here's how:
//
//   * If the identifier is already defined and is a function defined
//     in an function file that has been modified since the last time 
//     we parsed it, parse it again.
//
//   * If the identifier is not defined, try to find a builtin
//     variable or an already compiled function with the same name.
//
//   * If the identifier is still undefined, try looking for an
//     function file to parse.
//
//   * On systems that support dynamic linking, we prefer .oct files
//     over .m files.

octave_value
tree_identifier::do_lookup (bool& script_file_executed, bool exec_script)
{
  static octave_value foo;

  script_file_executed = lookup (sym, exec_script);

  return script_file_executed ? foo : sym->def ();
}

void
tree_identifier::link_to_global (void)
{
  if (sym)
    link_to_global_variable (sym);
}

void
tree_identifier::mark_as_static (void)
{
  if (sym)
    sym->mark_as_static ();
}

void
tree_identifier::mark_as_formal_parameter (void)
{
  if (sym)
    sym->mark_as_formal_parameter ();
}

octave_value_list
tree_identifier::rvalue (int nargout)
{
  octave_value_list retval;

  if (error_state)
    return retval;

  bool script_file_executed = false;

  octave_value val = do_lookup (script_file_executed);

  if (! script_file_executed)
    {
      if (val.is_defined ())
	{
	  // XXX GAGME XXX -- this would be cleaner if we required
	  // parens to indicate function calls.
	  //
	  // If this identifier refers to a function, we need to know
	  // whether it is indexed so that we can do the same thing
	  // for `f' and `f()'.  If the index is present, return the
	  // function object and let tree_index_expression::rvalue
	  // handle indexing.  Otherwise, arrange to call the function
	  // here, so that we don't return the function definition as
	  // a value.

	  if (val.is_function () && ! is_postfix_indexed ())
	    {
	      octave_value_list tmp_args;

	      retval = val.do_index_op (nargout, tmp_args);
	    }
	  else
	    {
	      if (print_result () && nargout == 0)
		val.print_with_name (octave_stdout, name ());

	      retval = val;
	    }
	}
      else
	eval_undefined_error ();
    }

  return retval;
}

octave_value
tree_identifier::rvalue (void)
{
  octave_value retval;

  octave_value_list tmp = rvalue (1);

  if (! tmp.empty ())
    retval = tmp(0);

  return retval;
}

octave_lvalue
tree_identifier::lvalue (void)
{
  return sym->variable_reference ();
}

void
tree_identifier::accept (tree_walker& tw)
{
  tw.visit_identifier (*this);
}

/*
;;; Local Variables: ***
;;; mode: C++ ***
;;; End: ***
*/
