/*
 *  Rexx/DW
 *  Copyright (C) 2003-2010 Mark Hessling <mark@rexx.org>
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Library General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public
 *  License along with this library; if not, write to the Free
 *  Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
/****h* RexxDW/Introduction
 * DESCRIPTION
 *   RexxDW is an external function package that allows a Rexx programmer to write
 *   cross platform GUI applications, using a lightweight portable GUI API called
 *   Dynamic Windows (dwindows).
 * USAGE
 *   The fundamental building blocks of a RexxDW program consist of two major
 *   components; widgets and event handlers or callbacks.
 *
 *   Various widgets are packed into the required layout and events
 *   are connected to the widgets to carry out the required behaviour.
 *
 *   The structure of a RexxDW program consists of various initialisation; creation
 *   of the widgets, packing them, connecting callbacks, and then calling the
 *   event loop handler to dispatch the event to the appropriate callbacks.
 *   The callbacks are normal Rexx subroutines.
 *
 *   There are two event loop handlers depending on whether your Rexx interpreter
 *   supports the RexxCallback() API. DW_main() is used for interpreters that have
 *   RexxCallback(); DW_main_iteration() is used for those interpretes that don't.
 * DERIVED FROM
 *   All of the functions in RexxDW are based on the same function in dwindows.
 *   Some function arguments will differ due to the nature of the Rexx API
 *   interface.
 *
 *   The format of the function names is dw_(target)_(action)[_(options)].
 *   eg. for tree widgets, functions act on both the tree itself and the tree items
 *   within the tree widget, so DW_tree_new() executes the "new" action on the
 *   tree widget, DW_tree_item_change() executes the "change" action on the
 *   tree item, not the tree.
 *
 *   Some dwindows functions are not available in RexxDW, mainly because they are
 *   not able to be implemented:
 *   TBD, etc...
 *
 *   A number of functions have been added to RexxDW that are not part of dwindows:
 *   dw_and, dw_or, dw_radiobutton_set, dw_radiobutton_get,...
 *
 * TODO
 *   - document dw_exec()??
 *   - add DW_container_get_column_type and DW_filesystem_get_column_type
 *   - can dw_window_maximize() be implemented as dw_window_set_style() with MAXIMIZE
 *     attribute set and MINIMIZE attribute unset?
 *   - no VALUE_CHANGE callback on spinbutton widget
 *   - document that KEY_PRESS needs to be trapped in toplevel window
 *     but BUTTON_PRESS in low-level window (confirm on all platforms)
 *   - DOCO for:
 *     dw_box_pack_at_index( box, item, index, width, height, hsize, vsize, pad)
 *     dw_box_unpack_at_index( box, index)
 *     dw_box_unpack( box )
 * BUGS
 *   - bug in (Windows at least) item_select callback on container. Seems that
 *     the callback gets called twice irrespective of whether 0 or 1 is
 *     returned from the callback.
 *   - bug (Windows) with drawing directly to a render window (ie not via a pixmap)
 *     (see rexxdw.rexx, Utility TAB)
 *   - Under GTK+ 2.0 or greater, a call to dw_window_set_font() causes an expose
 *     event to fire. If dw_window_set_font() is called within the expose callback,
 *     you get an infinite loop. Bit of a bugger if your expose event sets different
 *     fonts for different lines!!!
 * PORTABILITY
 *   RexxDW runs on Windows NT/2k/XP/Vista/W7, OS/2 3.0+, MacOS X, and on any Un*x platform
 *   that has GTK+ 1.x installed (but works better with GTK+ 2.x).
 * SEE ALSO
 *   RexxDW lives at href:http://rexxdw.sf.net
 *   Dynamic Windows lives at href:http://dwindows.netlabs.org
 ******
 */
#ifdef HAVE_CONFIG_H
# include "config.h"
#else
# include "defines.h"
#endif

#if defined(__EMX__) && defined(USE_OS2REXX)
# define INCL_DOS
# define INCL_WIN
# define INCL_GPI
# include "rxpack.h"
# include <dw.h>
#else
# include <dw.h>
# include "rxpack.h"
#endif

#ifdef TIME_WITH_SYS_TIME
# ifdef HAVE_TIME_H
#  include <time.h>
# endif
# ifdef HAVE_SYS_TIME_H
#  include <sys/time.h>
# endif
#else
# ifdef HAVE_TIME_H
#   include <time.h>
# else
#  ifdef HAVE_SYS_TIME_H
#   include <sys/time.h>
#  endif
# endif
#endif

#ifdef HAVE_ICONV_H
# include <iconv.h>
#endif

#if !defined(GTK_MAJOR_VERSION)
# define GTK_MAJOR_VERSION       0
#endif

#define API_REXXCALLBACK_PRESENT 1
#define API_REXXCALLBACK_MISSING 0

#define HAVE_REXXCALLBACK ( RexxDWConstants[0].numeric_value == API_REXXCALLBACK_PRESENT )

static char leftico[1150] = {
    0x00, 0x00, 0x01, 0x00, 0x01, 0x00, 0x10, 0x10, 0x00, 0x00, 0x01, 0x00, 0x20, 0x00, 0x68, 0x04, 0x00, 0x00, 0x16, 0x00, 0x00, 0x00, 0x28, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x20, 0x00, 0x00, 0x00, 0x01, 0x00, 0x20, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x13, 0x0B, 0x00, 0x00, 0x13, 0x0B, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xFF, 0xFF, 0xFF, 0x02, 0xFF, 0xFF, 0xFF, 0x29, 0xFF, 0xFF, 0xFF, 0x5A, 0xFF, 0xFF, 0xFF, 0x46, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xFF, 0xFF, 0xFF, 0x20, 0xEB, 0xEE, 0xEB, 0x5B, 0x7C, 0x7F,
    0x7D, 0xB3, 0x44, 0x47, 0x46, 0xFF, 0xF2, 0xF2, 0xF2, 0x62, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xFF, 0xFF, 0xFF, 0x16, 0xFF, 0xFF, 0xFF, 0x4C, 0x9B, 0x9D, 0x9B, 0x97, 0x59, 0x5C, 0x5B, 0xF1, 0xC3, 0xC4, 0xC4, 0xFC, 0x44, 0x47, 0x46, 0xFF, 0xD8, 0xD8, 0xD8, 0x5B, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0xFF, 0xFF, 0xFF, 0x0E, 0xFF, 0xFF, 0xFF, 0x42, 0xC3, 0xC3, 0xC3, 0x7F, 0x62, 0x64, 0x64, 0xDB, 0x95, 0x97, 0x96, 0xFA, 0xEC, 0xED, 0xEC, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0x44, 0x47, 0x46, 0xFF, 0xBD, 0xBD, 0xBD, 0x55, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xFF, 0xFF, 0xFF, 0x08, 0xFF, 0xFF, 0xFF, 0x39, 0xE7, 0xEA, 0xEA, 0x6C, 0x74, 0x77, 0x75, 0xBF, 0x75, 0x78, 0x77, 0xFA, 0xE2, 0xE3, 0xE2, 0xFF, 0xEB, 0xEC, 0xEC, 0xFF, 0xDA, 0xDC, 0xDB, 0xFF, 0xF1, 0xF1, 0xF1, 0xFF, 0x44, 0x47, 0x46, 0xFF, 0xA3, 0xA3,
    0xA3, 0x4E, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xFF, 0xFF, 0xFF, 0x28, 0xFC, 0xFC, 0xFC, 0x61, 0x8F, 0x91, 0x91, 0xA0, 0x66, 0x68, 0x68, 0xF6, 0xD0, 0xD2, 0xD1, 0xFD, 0xEC, 0xED, 0xEC, 0xFF, 0xE0, 0xE2, 0xE1, 0xFF, 0xD5, 0xD8, 0xD7, 0xFF, 0xCB, 0xCE, 0xCC, 0xFF, 0xEC, 0xED, 0xED, 0xFF, 0x44, 0x47, 0x46, 0xFF, 0x8A, 0x8A, 0x8A, 0x48, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xFF, 0xFF, 0xFF, 0x5F, 0x94, 0x97, 0x96, 0x9B, 0x5E, 0x61, 0x60, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xE9, 0xEA,
    0xEA, 0xFF, 0xDD, 0xDF, 0xDE, 0xFF, 0xD2, 0xD4, 0xD3, 0xFF, 0xC7, 0xCA, 0xC9, 0xFF, 0xBD, 0xC0, 0xBE, 0xFF, 0xE8, 0xE9, 0xE9, 0xFF, 0x44, 0x47, 0x46, 0xFF, 0x72, 0x72, 0x72, 0x41, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xFF, 0xFF, 0xFF, 0x26, 0xE6, 0xE6, 0xE6, 0x5B, 0x7D, 0x7F, 0x7F, 0x9B, 0x61, 0x62, 0x62, 0xF5, 0xCC, 0xCE, 0xCD, 0xFD, 0xE7, 0xE9, 0xE8, 0xFF, 0xCA, 0xCC, 0xCB, 0xFF, 0xB9, 0xBD, 0xBB, 0xFF, 0xAE, 0xB2, 0xB1, 0xFF, 0xE4, 0xE5, 0xE5, 0xFF, 0x44, 0x47, 0x46, 0xFF, 0x5B, 0x5B, 0x5B, 0x3B, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xDB, 0xDB, 0xDB, 0x07, 0xC3, 0xC3, 0xC3, 0x2F, 0xA0, 0xA0, 0xA0, 0x5B, 0x5C, 0x5E, 0x5D, 0xB5, 0x6E, 0x70, 0x70, 0xFA, 0xDF, 0xE0, 0xE0, 0xFE, 0xD2, 0xD5, 0xD4, 0xFF, 0xA3, 0xA8, 0xA6, 0xFF, 0xE0, 0xE2, 0xE1, 0xFF, 0x44, 0x47, 0x46, 0xFF, 0x40, 0x40, 0x40, 0x34, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x99, 0x99, 0x99, 0x0A, 0x8D, 0x8D, 0x8D, 0x2F, 0x64, 0x67, 0x67, 0x63, 0x4E, 0x51,
    0x4F, 0xD4, 0x8E, 0x91, 0x8F, 0xFA, 0xE1, 0xE2, 0xE2, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0x44, 0x47, 0x46, 0xFF, 0x27, 0x27, 0x27, 0x2E, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x62, 0x62, 0x62, 0x0D, 0x59, 0x59, 0x59, 0x2B, 0x45, 0x47, 0x45, 0x76, 0x4F, 0x51, 0x50, 0xEF, 0xB5, 0xB7, 0xB6, 0xFB, 0x44, 0x47, 0x46, 0xFF, 0x0D, 0x0D, 0x0D, 0x27, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x37, 0x37, 0x37, 0x0E, 0x22, 0x28, 0x22, 0x2D, 0x40, 0x42, 0x42, 0x97, 0x44, 0x47, 0x46, 0xFF, 0x00, 0x00, 0x00, 0x24, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x01, 0x00, 0x00, 0x00, 0x0E, 0x00, 0x00, 0x00, 0x1F, 0x00, 0x00, 0x00, 0x19, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xFF, 0xFF, 0x00, 0x00, 0xFF, 0xFF, 0x00, 0x00, 0xFF, 0xC3, 0x00, 0x00, 0xFF, 0x83,
    0x00, 0x00, 0xFE, 0x03, 0x00, 0x00, 0xF8, 0x03, 0x00, 0x00, 0xE0, 0x03, 0x00, 0x00, 0xC0, 0x03, 0x00, 0x00, 0xC0, 0x03, 0x00, 0x00, 0xC0, 0x03, 0x00, 0x00, 0xE0, 0x03, 0x00, 0x00, 0xF8, 0x03, 0x00, 0x00, 0xFE, 0x03, 0x00, 0x00, 0xFF, 0x83, 0x00, 0x00, 0xFF, 0xC3, 0x00, 0x00, 0xFF, 0xFF, 0x00, 0x00,
};

static char rightico[1150] = {
    0x00, 0x00, 0x01, 0x00, 0x01, 0x00, 0x10, 0x10, 0x00, 0x00, 0x01, 0x00, 0x20, 0x00, 0x68, 0x04, 0x00, 0x00, 0x16, 0x00, 0x00, 0x00, 0x28, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x20, 0x00, 0x00, 0x00, 0x01, 0x00, 0x20, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x12, 0x0B, 0x00, 0x00, 0x12, 0x0B, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xFF, 0xFF,
    0xFF, 0x46, 0xFF, 0xFF, 0xFF, 0x5A, 0xFF, 0xFF, 0xFF, 0x29, 0xFF, 0xFF, 0xFF, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xF2, 0xF2, 0xF2, 0x62, 0x44, 0x47, 0x46, 0xFF, 0x7C, 0x7F, 0x7D, 0xB3, 0xEB, 0xEE, 0xEB, 0x5B, 0xFF, 0xFF, 0xFF, 0x20, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xD8, 0xD8, 0xD8, 0x5B, 0x44, 0x47, 0x46, 0xFF, 0xC3, 0xC4, 0xC4, 0xFC, 0x59, 0x5C, 0x5B, 0xF1, 0x9B, 0x9D, 0x9B, 0x97, 0xFF, 0xFF, 0xFF, 0x4C, 0xFF, 0xFF, 0xFF, 0x16, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xBD, 0xBD, 0xBD, 0x55, 0x44, 0x47, 0x46, 0xFF, 0xFF, 0xFF,
    0xFF, 0xFF, 0xEC, 0xED, 0xEC, 0xFF, 0x95, 0x97, 0x96, 0xFA, 0x62, 0x64, 0x64, 0xDB, 0xC3, 0xC3, 0xC3, 0x7F, 0xFF, 0xFF, 0xFF, 0x42, 0xFF, 0xFF, 0xFF, 0x0E, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xA3, 0xA3, 0xA3, 0x4E, 0x44, 0x47, 0x46, 0xFF, 0xF1, 0xF1, 0xF1, 0xFF, 0xDA, 0xDC, 0xDB, 0xFF, 0xEB, 0xEC, 0xEC, 0xFF, 0xE2, 0xE3, 0xE2, 0xFF, 0x75, 0x78, 0x77, 0xFA, 0x74, 0x77, 0x75, 0xBF, 0xE7, 0xEA, 0xEA, 0x6C, 0xFF, 0xFF, 0xFF, 0x39, 0xFF, 0xFF, 0xFF, 0x08, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x8A, 0x8A, 0x8A, 0x48, 0x44, 0x47, 0x46, 0xFF, 0xEC, 0xED, 0xED, 0xFF, 0xCB, 0xCE, 0xCC, 0xFF, 0xD5, 0xD8, 0xD7, 0xFF, 0xE0, 0xE2, 0xE1, 0xFF, 0xEC, 0xED, 0xEC, 0xFF, 0xD0, 0xD2, 0xD1, 0xFD, 0x66, 0x68, 0x68, 0xF6, 0x8F, 0x91, 0x91, 0xA0, 0xFC, 0xFC, 0xFC, 0x61, 0xFF, 0xFF, 0xFF, 0x28, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x72, 0x72, 0x72, 0x41, 0x44, 0x47, 0x46, 0xFF, 0xE8, 0xE9, 0xE9, 0xFF, 0xBD, 0xC0, 0xBE, 0xFF, 0xC7, 0xCA,
    0xC9, 0xFF, 0xD2, 0xD4, 0xD3, 0xFF, 0xDD, 0xDF, 0xDE, 0xFF, 0xE9, 0xEA, 0xEA, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0x5E, 0x61, 0x60, 0xFF, 0x94, 0x97, 0x96, 0x9B, 0xFF, 0xFF, 0xFF, 0x5F, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x5B, 0x5B, 0x5B, 0x3B, 0x44, 0x47, 0x46, 0xFF, 0xE4, 0xE5, 0xE5, 0xFF, 0xAE, 0xB2, 0xB1, 0xFF, 0xB9, 0xBD, 0xBB, 0xFF, 0xCA, 0xCC, 0xCB, 0xFF, 0xE7, 0xE9, 0xE8, 0xFF, 0xCC, 0xCE, 0xCD, 0xFD, 0x61, 0x62, 0x62, 0xF5, 0x7D, 0x7F, 0x7F, 0x9B, 0xE6, 0xE6, 0xE6, 0x5B, 0xFF, 0xFF, 0xFF, 0x26, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x40, 0x40, 0x34, 0x44, 0x47, 0x46, 0xFF, 0xE0, 0xE2, 0xE1, 0xFF, 0xA3, 0xA8, 0xA6, 0xFF, 0xD2, 0xD5, 0xD4, 0xFF, 0xDF, 0xE0, 0xE0, 0xFE, 0x6E, 0x70, 0x70, 0xFA, 0x5C, 0x5E, 0x5D, 0xB5, 0xA0, 0xA0, 0xA0, 0x5B, 0xC3, 0xC3, 0xC3, 0x2F, 0xDB, 0xDB, 0xDB, 0x07, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x27, 0x27, 0x27, 0x2E, 0x44, 0x47, 0x46, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xE1, 0xE2, 0xE2, 0xFF, 0x8E, 0x91, 0x8F, 0xFA, 0x4E, 0x51, 0x4F, 0xD4, 0x64, 0x67,
    0x67, 0x63, 0x8D, 0x8D, 0x8D, 0x2F, 0x99, 0x99, 0x99, 0x0A, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0D, 0x0D, 0x0D, 0x27, 0x44, 0x47, 0x46, 0xFF, 0xB5, 0xB7, 0xB6, 0xFB, 0x4F, 0x51, 0x50, 0xEF, 0x45, 0x47, 0x45, 0x76, 0x59, 0x59, 0x59, 0x2B, 0x62, 0x62, 0x62, 0x0D, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x24, 0x44, 0x47, 0x46, 0xFF, 0x40, 0x42, 0x42, 0x97, 0x22, 0x28, 0x22, 0x2D, 0x37, 0x37, 0x37, 0x0E, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x19, 0x00, 0x00, 0x00, 0x1F, 0x00, 0x00, 0x00, 0x0E, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xFF, 0xFF, 0x00, 0x00, 0xFF, 0xFF, 0x00, 0x00, 0xC3, 0xFF, 0x00, 0x00, 0xC1, 0xFF,
    0x00, 0x00, 0xC0, 0x7F, 0x00, 0x00, 0xC0, 0x1F, 0x00, 0x00, 0xC0, 0x07, 0x00, 0x00, 0xC0, 0x03, 0x00, 0x00, 0xC0, 0x03, 0x00, 0x00, 0xC0, 0x03, 0x00, 0x00, 0xC0, 0x07, 0x00, 0x00, 0xC0, 0x1F, 0x00, 0x00, 0xC0, 0x7F, 0x00, 0x00, 0xC1, 0xFF, 0x00, 0x00, 0xC3, 0xFF, 0x00, 0x00, 0xFF, 0xFF, 0x00, 0x00,
};

static char allleftico[1150] = {
    0x00, 0x00, 0x01, 0x00, 0x01, 0x00, 0x10, 0x10, 0x00, 0x00, 0x01, 0x00, 0x20, 0x00, 0x68, 0x04, 0x00, 0x00, 0x16, 0x00, 0x00, 0x00, 0x28, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x20, 0x00, 0x00, 0x00, 0x01, 0x00, 0x20, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x12, 0x0B, 0x00, 0x00, 0x12, 0x0B, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xFF, 0xFF, 0xFF, 0x04, 0xFF, 0xFF, 0xFF, 0x39, 0xFF, 0xFF, 0xFF, 0x5B, 0xFF, 0xFF, 0xFF, 0x32, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xFF, 0xFF,
    0xFF, 0x04, 0xFF, 0xFF, 0xFF, 0x39, 0xFF, 0xFF, 0xFF, 0x5B, 0xFF, 0xFF, 0xFF, 0x32, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xFF, 0xFF, 0xFF, 0x1A, 0xFC, 0xFC, 0xFC, 0x59, 0x86, 0x87, 0x86, 0xAA, 0x44, 0x47, 0x46, 0xFF, 0xFF, 0xFF, 0xFF, 0x62, 0x00, 0x00, 0x00, 0x00, 0xFF, 0xFF, 0xFF, 0x1A, 0xFC, 0xFC, 0xFC, 0x59, 0x86, 0x87, 0x86, 0xAA, 0x44, 0x47, 0x46, 0xFF, 0xFF, 0xFF, 0xFF, 0x62, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xFF, 0xFF, 0xFF, 0x05, 0xFB, 0xFB, 0xFB, 0x3B, 0xC7, 0xC7,
    0xC7, 0x7B, 0x5F, 0x61, 0x60, 0xE1, 0xB2, 0xB3, 0xB3, 0xFB, 0x44, 0x47, 0x46, 0xFF, 0xFC, 0xFC, 0xFC, 0x65, 0xFB, 0xFB, 0xFB, 0x3B, 0xC7, 0xC7, 0xC7, 0x7B, 0x5F, 0x61, 0x60, 0xE1, 0xB2, 0xB3, 0xB3, 0xFB, 0x44, 0x47, 0x46, 0xFF, 0xFC, 0xFC, 0xFC, 0x65, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xD5, 0xD5, 0xD5, 0x18, 0xD2, 0xD2, 0xD2, 0x50, 0x75, 0x76, 0x75, 0xA4, 0x66, 0x68, 0x68, 0xFA, 0xE5, 0xE6, 0xE5, 0xFE, 0xFD, 0xFE, 0xFE, 0xFF, 0x44, 0x47, 0x46, 0xFF, 0xD3, 0xD3, 0xD3, 0x5C, 0x75, 0x76, 0x75, 0xA4, 0x66, 0x68, 0x68, 0xFA, 0xE5, 0xE6, 0xE5, 0xFE, 0xFD, 0xFE, 0xFE, 0xFF, 0x44, 0x47,
    0x46, 0xFF, 0xD5, 0xD5, 0xD5, 0x5B, 0x00, 0x00, 0x00, 0x00, 0xAC, 0xAC, 0xAC, 0x28, 0x89, 0x89, 0x89, 0x6A, 0x52, 0x54, 0x53, 0xDD, 0xA0, 0xA1, 0xA1, 0xFA, 0xED, 0xEE, 0xEE, 0xFF, 0xE1, 0xE3, 0xE2, 0xFF, 0xF1, 0xF2, 0xF1, 0xFF, 0x44, 0x47, 0x46, 0xFF, 0x52, 0x54, 0x53, 0xDD, 0xA0, 0xA1, 0xA1, 0xFA, 0xED, 0xEE, 0xEE, 0xFF, 0xE1, 0xE3, 0xE2, 0xFF, 0xF1, 0xF2, 0xF1, 0xFF, 0x44, 0x47, 0x46, 0xFF, 0xAD, 0xAD, 0xAD, 0x51, 0x00, 0x00, 0x00, 0x00, 0x8A, 0x8A, 0x8A, 0x3F, 0x44, 0x47, 0x46, 0xFF, 0xA6, 0xA8, 0xA8, 0xFE, 0xFF, 0xFF, 0xFF, 0xFF, 0xB8, 0xBC, 0xBA, 0xFF, 0xC4, 0xC7, 0xC5, 0xFF, 0xEC, 0xED,
    0xED, 0xFF, 0x44, 0x47, 0x46, 0xFF, 0xA6, 0xA8, 0xA8, 0xFE, 0xFF, 0xFF, 0xFF, 0xFF, 0xB8, 0xBC, 0xBA, 0xFF, 0xC4, 0xC7, 0xC5, 0xFF, 0xEC, 0xED, 0xED, 0xFF, 0x44, 0x47, 0x46, 0xFF, 0x88, 0x88, 0x88, 0x47, 0x00, 0x00, 0x00, 0x00, 0x5E, 0x5E, 0x5E, 0x1E, 0x57, 0x57, 0x57, 0x58, 0x4A, 0x4C, 0x4B, 0xD9, 0x9A, 0x9B, 0x9B, 0xFA, 0xE3, 0xE4, 0xE3, 0xFF, 0xB8, 0xBB, 0xBA, 0xFF, 0xE2, 0xE4, 0xE3, 0xFF, 0x44, 0x47, 0x46, 0xFF, 0x4A, 0x4C, 0x4B, 0xD9, 0x9A, 0x9B, 0x9B, 0xFA, 0xE3, 0xE4, 0xE3, 0xFF, 0xB8, 0xBB, 0xBA, 0xFF, 0xE2, 0xE4, 0xE3, 0xFF, 0x44, 0x47, 0x46, 0xFF, 0x60, 0x60, 0x60, 0x3D, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x3B, 0x3B, 0x3B, 0x0D, 0x3D, 0x3D, 0x3D, 0x2E, 0x44, 0x46, 0x44, 0x8E, 0x62, 0x64, 0x64, 0xF8, 0xDA, 0xDB, 0xDB, 0xFE, 0xFB, 0xFB, 0xFB, 0xFF, 0x44, 0x47, 0x46, 0xFF, 0x3F, 0x3F, 0x3F, 0x35, 0x44, 0x46, 0x44, 0x8E, 0x62, 0x64, 0x64, 0xF8, 0xDA, 0xDB, 0xDB, 0xFE, 0xFB, 0xFB, 0xFB, 0xFF, 0x44, 0x47, 0x46, 0xFF, 0x3C, 0x3C, 0x3C, 0x33, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0x15, 0x15, 0x15, 0x18, 0x2F, 0x2F, 0x2F, 0x47, 0x4A, 0x4C, 0x4B, 0xD6, 0xA8, 0xAA, 0xA9, 0xFA, 0x44, 0x47, 0x46, 0xFF, 0x13, 0x13,
    0x13, 0x29, 0x15, 0x15, 0x15, 0x18, 0x2F, 0x2F, 0x2F, 0x47, 0x4A, 0x4C, 0x4B, 0xD6, 0xA8, 0xAA, 0xA9, 0xFA, 0x44, 0x47, 0x46, 0xFF, 0x13, 0x13, 0x13, 0x29, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x09, 0x08, 0x08, 0x08, 0x21, 0x3D, 0x3F, 0x3D, 0x86, 0x44, 0x47, 0x46, 0xFF, 0x00, 0x00, 0x00, 0x23, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x09, 0x08, 0x08, 0x08, 0x21, 0x3D, 0x3F, 0x3D, 0x86, 0x44, 0x47, 0x46, 0xFF, 0x00, 0x00, 0x00, 0x23, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00, 0x21, 0x00, 0x00, 0x00, 0x12, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00, 0x20, 0x00, 0x00, 0x00, 0x12, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xFF, 0xFF, 0x00, 0x00, 0xFF, 0xFF, 0x00, 0x00, 0xFF, 0xFF, 0x00, 0x00, 0xF8, 0x61,
    0x00, 0x00, 0xF0, 0x41, 0x00, 0x00, 0xC0, 0x01, 0x00, 0x00, 0x80, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x80, 0x01, 0x00, 0x00, 0xC0, 0x01, 0x00, 0x00, 0xF0, 0x41, 0x00, 0x00, 0xF8, 0x61, 0x00, 0x00, 0xFF, 0xFF, 0x00, 0x00, 0xFF, 0xFF, 0x00, 0x00,
};

static char allrightico[1150] = {
    0x00, 0x00, 0x01, 0x00, 0x01, 0x00, 0x10, 0x10, 0x00, 0x00, 0x01, 0x00, 0x20, 0x00, 0x68, 0x04, 0x00, 0x00, 0x16, 0x00, 0x00, 0x00, 0x28, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x20, 0x00, 0x00, 0x00, 0x01, 0x00, 0x20, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x12, 0x0B, 0x00, 0x00, 0x12, 0x0B, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xFF, 0xFF, 0xFF, 0x2B, 0xFF, 0xFF, 0xFF, 0x5B, 0xFF, 0xFF, 0xFF, 0x3D, 0xFF, 0xFF, 0xFF, 0x06, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xFF, 0xFF, 0xFF, 0x2B, 0xFF, 0xFF, 0xFF, 0x5B, 0xFF, 0xFF, 0xFF, 0x3D, 0xFF, 0xFF, 0xFF, 0x06, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xFF, 0xFF, 0xFF, 0x5C, 0x44, 0x47, 0x46, 0xFF, 0x7A, 0x7B, 0x7B, 0xB6, 0xF7, 0xF7, 0xF7, 0x5E, 0xFF, 0xFF, 0xFF, 0x1E, 0x00, 0x00, 0x00, 0x00, 0xFF, 0xFF, 0xFF, 0x5C, 0x44, 0x47, 0x46, 0xFF, 0x7A, 0x7B, 0x7B, 0xB6, 0xF7, 0xF7, 0xF7, 0x5E, 0xFF, 0xFF, 0xFF, 0x1E, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xFC, 0xFC, 0xFC, 0x5F, 0x44, 0x47, 0x46, 0xFF, 0xB0, 0xB2, 0xB1, 0xFC, 0x5C, 0x61,
    0x60, 0xE8, 0xBD, 0xBD, 0xBD, 0x80, 0xFF, 0xFF, 0xFF, 0x3F, 0xFC, 0xFC, 0xFC, 0x61, 0x44, 0x47, 0x46, 0xFF, 0xB0, 0xB2, 0xB1, 0xFC, 0x5D, 0x61, 0x60, 0xE8, 0xBD, 0xBD, 0xBD, 0x80, 0xFB, 0xFB, 0xFB, 0x40, 0xFF, 0xFF, 0xFF, 0x07, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xD5, 0xD5, 0xD5, 0x56, 0x49, 0x4E, 0x4B, 0xFA, 0xFB, 0xFB, 0xFB, 0xFF, 0xE8, 0xE9, 0xE8, 0xFF, 0x6D, 0x6F, 0x6E, 0xFA, 0x6F, 0x70, 0x70, 0xAD, 0xCE, 0xCE, 0xCE, 0x5E, 0x49, 0x4E, 0x4B, 0xFA, 0xFB, 0xFB, 0xFB, 0xFF, 0xE8, 0xE9, 0xE8, 0xFF, 0x6D, 0x6F, 0x6E, 0xFA, 0x6F, 0x70, 0x70, 0xAD, 0xCF, 0xCF,
    0xCF, 0x55, 0xD9, 0xD9, 0xD9, 0x1B, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xAE, 0xAE, 0xAE, 0x4C, 0x49, 0x4D, 0x4B, 0xFA, 0xED, 0xEE, 0xEE, 0xFF, 0xE1, 0xE3, 0xE2, 0xFF, 0xED, 0xEE, 0xED, 0xFF, 0xAC, 0xAD, 0xAC, 0xFB, 0x51, 0x54, 0x53, 0xE4, 0x49, 0x4C, 0x4B, 0xFB, 0xED, 0xEE, 0xEE, 0xFF, 0xE1, 0xE3, 0xE2, 0xFF, 0xED, 0xEE, 0xED, 0xFF, 0xAC, 0xAD, 0xAC, 0xFB, 0x51, 0x54, 0x53, 0xE4, 0x85, 0x85, 0x85, 0x6F, 0xAE, 0xAE, 0xAE, 0x2C, 0x00, 0x00, 0x00, 0x00, 0x89, 0x89, 0x89, 0x43, 0x49, 0x4C, 0x4B, 0xF9, 0xE9, 0xEA, 0xEA, 0xFF, 0xC5, 0xC8, 0xC6, 0xFF, 0xB8, 0xBB, 0xBA, 0xFF, 0xF6, 0xF6,
    0xF6, 0xFF, 0xBC, 0xBD, 0xBD, 0xFF, 0x48, 0x4B, 0x4A, 0xFF, 0xE9, 0xEA, 0xEA, 0xFF, 0xC5, 0xC8, 0xC6, 0xFF, 0xB8, 0xBB, 0xBA, 0xFF, 0xF6, 0xF6, 0xF6, 0xFF, 0xBC, 0xBD, 0xBD, 0xFF, 0x44, 0x47, 0x46, 0xFF, 0x88, 0x88, 0x88, 0x47, 0x00, 0x00, 0x00, 0x00, 0x62, 0x62, 0x62, 0x39, 0x48, 0x4B, 0x4A, 0xF9, 0xE0, 0xE1, 0xE1, 0xFF, 0xB6, 0xBA, 0xB8, 0xFF, 0xE1, 0xE2, 0xE2, 0xFF, 0xA5, 0xA6, 0xA5, 0xFB, 0x4A, 0x4D, 0x4C, 0xE1, 0x47, 0x4B, 0x4A, 0xFA, 0xE0, 0xE1, 0xE1, 0xFF, 0xB6, 0xBA, 0xB8, 0xFF, 0xE1, 0xE2, 0xE2, 0xFF, 0xA5, 0xA6, 0xA5, 0xFB, 0x4A, 0x4D, 0x4C, 0xE1, 0x54, 0x54, 0x54, 0x5E, 0x64, 0x64,
    0x64, 0x21, 0x00, 0x00, 0x00, 0x00, 0x3A, 0x3A, 0x3A, 0x30, 0x47, 0x4A, 0x49, 0xF9, 0xF6, 0xF6, 0xF6, 0xFF, 0xDD, 0xDE, 0xDE, 0xFF, 0x69, 0x6B, 0x6A, 0xF8, 0x46, 0x48, 0x48, 0x99, 0x3D, 0x3D, 0x3D, 0x36, 0x47, 0x4A, 0x49, 0xF9, 0xF6, 0xF6, 0xF6, 0xFF, 0xDD, 0xDE, 0xDE, 0xFF, 0x69, 0x6B, 0x6A, 0xF8, 0x46, 0x48, 0x48, 0x99, 0x3E, 0x3E, 0x3E, 0x31, 0x44, 0x44, 0x44, 0x0F, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x14, 0x14, 0x14, 0x27, 0x44, 0x47, 0x46, 0xFF, 0xA8, 0xAA, 0xA9, 0xFB, 0x4C, 0x4F, 0x4E, 0xDE, 0x2E, 0x2E, 0x2E, 0x4E, 0x14, 0x14, 0x14, 0x1A, 0x14, 0x14, 0x14, 0x27, 0x44, 0x47,
    0x46, 0xFF, 0xA8, 0xAA, 0xA9, 0xFB, 0x4C, 0x4F, 0x4E, 0xDE, 0x2E, 0x2E, 0x2E, 0x4E, 0x14, 0x14, 0x14, 0x1A, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x20, 0x44, 0x47, 0x46, 0xFF, 0x3E, 0x40, 0x40, 0x98, 0x07, 0x07, 0x07, 0x23, 0x00, 0x00, 0x00, 0x0A, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x21, 0x44, 0x47, 0x46, 0xFF, 0x3E, 0x40, 0x40, 0x98, 0x07, 0x07, 0x07, 0x23, 0x00, 0x00, 0x00, 0x0A, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x0F, 0x00, 0x00, 0x00, 0x20, 0x00, 0x00, 0x00, 0x15, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x11, 0x00, 0x00, 0x00, 0x21, 0x00, 0x00, 0x00, 0x16, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xFF, 0xFF, 0x00, 0x00, 0xFF, 0xFF, 0x00, 0x00, 0xFF, 0xFF, 0x00, 0x00, 0x86, 0x1F,
    0x00, 0x00, 0x82, 0x0F, 0x00, 0x00, 0x80, 0x03, 0x00, 0x00, 0x80, 0x01, 0x00, 0x00, 0x80, 0x00, 0x00, 0x00, 0x80, 0x00, 0x00, 0x00, 0x80, 0x00, 0x00, 0x00, 0x80, 0x01, 0x00, 0x00, 0x80, 0x03, 0x00, 0x00, 0x82, 0x0F, 0x00, 0x00, 0x86, 0x1F, 0x00, 0x00, 0xFF, 0xFF, 0x00, 0x00, 0xFF, 0xFF, 0x00, 0x00,
};

/*
 * These #defines SHOULD be in dw.h
 */

#define dw_radiobutton_set   dw_checkbox_set

/* dw_box_new() */
/* dw_group_box_new() */
/* dw_splitbar_new() */
/* dw_scrollbar_new() */
/*
#define DW_VERT                   TRUE
#define DW_HORZ                   FALSE
*/

/* dw_notebook_new() */
#define DW_TAB_TO_TOP             TRUE
#define DW_TAB_TO_BOTTOM          FALSE

/* dw_menu_append_item() */
#define DW_MENU_END               TRUE
#define DW_MENU_START             FALSE
#define DW_MENU_CHECKABLE         TRUE
#define DW_MENU_NOT_CHECKABLE     FALSE

/* dw_slider_new() */
#define DW_SLIDER_VERT            TRUE
#define DW_SLIDER_HORZ            FALSE

/* dw_box_pack_start() */
/* dw_box_pack_end() */
/* dw_box_pack_at_index() */
#define DW_EXPAND_HORZ            TRUE
#define DW_DONT_EXPAND_HORZ       FALSE
#define DW_EXPAND_VERT            TRUE
#define DW_DONT_EXPAND_VERT       FALSE

/* dw_notebook_page_new() */
#define DW_PAGE_TO_FRONT          TRUE
#define DW_PAGE_TO_BACK           FALSE

/* dw_listbox_select() */
#define DW_SELECTED               TRUE
#define DW_NOT_SELECTED           FALSE

/* dw_mle_set_editable() */
#define DW_EDITABLE               TRUE
#define DW_READONLY               FALSE

/* dw_mle_set_word_wrap() */
#define DW_WORD_WRAP              TRUE
#define DW_DONT_WORD_WRAP         FALSE

/* dw_checkbox_set() */
/* dw_radiobutton_set() */
/* dw_menu_item_set_state() */
#define DW_CHECKED                TRUE
#define DW_UNCHECKED              FALSE

/* dw_container_clear() */
#define DW_REDRAW                 TRUE
#define DW_DONT_REDRAW            FALSE

/* dw_container_new() */
#define DW_MULTIPLE_SELECTION     TRUE
#define DW_SINGLE_SELECTION       FALSE

/* dw_draw_rect(), dw_draw_polygon() */
#define DW_FILL                   TRUE
#define DW_DONT_FILL              FALSE

/* dw_groupbox_new() */
#define DW_FONT_NORMAL            0
#define DW_FONT_BOLD              1
#define DW_FONT_ITALIC            2

/*
 * The #defines above SHOULD be in dw.h
 */

#define REXXDW_WIDGET_LISTBOX_SELECTOR   1

static char *RxPackageName = "rexxdw";

static int DWSIGNAL rexxdw_configure_event_cb( HWND window, int width, int height, void *data );
static int DWSIGNAL rexxdw_key_press_event_cb( HWND window, char key, int virtual_key, int state, void *data );
static int DWSIGNAL rexxdw_button_press_event_cb( HWND window, int x, int y, int button, void *data );
static int DWSIGNAL rexxdw_button_release_event_cb( HWND window, int x, int y, int button, void *data );
static int DWSIGNAL rexxdw_motion_notify_event_cb( HWND window, int x, int y, int state, void *data );
static int DWSIGNAL rexxdw_delete_event_cb( HWND window, void *data );
static int DWSIGNAL rexxdw_expose_event_cb( HWND window, DWExpose *exp, void *data );
static int DWSIGNAL rexxdw_clicked_event_cb( HWND window, void *data );
static int DWSIGNAL rexxdw_item_enter_event_cb( HWND window, char *text, void *data );
static int DWSIGNAL rexxdw_item_context_event_cb( HWND window, char *text, int x, int y, void *data, void *itemdata );
static int DWSIGNAL rexxdw_list_select_event_cb( HWND window, int item, void *data );
static int DWSIGNAL rexxdw_item_select_event_cb( HWND window, HWND item, char *text, void *itemdata, void *data );
static int DWSIGNAL rexxdw_set_focus_event_cb( HWND window, void *data );
static int DWSIGNAL rexxdw_value_changed_event_cb( HWND window, int val, void *data );
static int DWSIGNAL rexxdw_switch_page_event_cb( HWND window, unsigned long pagenum, void *itemdata );
static int DWSIGNAL rexxdw_column_click_event_cb( HWND window, int column_num, void *itemdata );

/*
 * Package prototypes...
 */

/* widgets */
/*---------*/

/* boxes */
rxfunc( DW_box_new );
rxfunc( DW_groupbox_new );
rxfunc( DW_scrollbox_new );
rxfunc( DW_box_pack_start );
rxfunc( DW_box_pack_end );
rxfunc( DW_box_pack_at_index );
rxfunc( DW_box_unpack );
rxfunc( DW_box_unpack_at_index );

/* buttons */
rxfunc( DW_button_new );
rxfunc( DW_bitmapbutton_new_from_file );
rxfunc( DW_bitmapbutton_new_from_data );

/* calendar */
rxfunc( DW_calendar_new );
rxfunc( DW_calendar_set_date );
rxfunc( DW_calendar_get_date );

/* checkbox radiobuttons */
rxfunc( DW_checkbox_new );
rxfunc( DW_radiobutton_new );
rxfunc( DW_checkbox_set );
rxfunc( DW_checkbox_get );
rxfunc( DW_radiobutton_set );
rxfunc( DW_radiobutton_get );

/* container/filesystem */
rxfunc( DW_container_new );
rxfunc( DW_container_alloc );
rxfunc( DW_container_setup );
rxfunc( DW_filesystem_setup );
rxfunc( DW_container_set_item );
rxfunc( DW_filesystem_set_file );
rxfunc( DW_filesystem_set_item );
rxfunc( DW_container_change_item );
rxfunc( DW_filesystem_change_file );
rxfunc( DW_filesystem_change_item );
rxfunc( DW_container_insert );
rxfunc( DW_container_set_row_title );
rxfunc( DW_container_set_column_width );
rxfunc( DW_container_optimize );
rxfunc( DW_container_clear );
rxfunc( DW_container_delete );
rxfunc( DW_container_delete_row );
rxfunc( DW_container_query_start );
rxfunc( DW_container_query_next );
rxfunc( DW_container_cursor );
rxfunc( DW_container_scroll );
rxfunc( DW_container_set_stripe );

/* rendering */
rxfunc( DW_render_new );
rxfunc( DW_pixmap_new );
rxfunc( DW_pixmap_new_from_file );
rxfunc( DW_pixmap_new_from_data );
rxfunc( DW_pixmap_destroy );
rxfunc( DW_pixmap_set_transparent_color );
rxfunc( DW_pixmap_bitblt );
rxfunc( DW_pixmap_width );
rxfunc( DW_pixmap_height );
rxfunc( DW_draw_text );
rxfunc( DW_draw_line );
rxfunc( DW_draw_rect );
rxfunc( DW_draw_point );
rxfunc( DW_draw_polygon );

/* entryfield */
rxfunc( DW_entryfield_new );
rxfunc( DW_entryfield_password_new );
rxfunc( DW_entryfield_set_limit );

/* embedded html */
rxfunc( DW_html_action );
rxfunc( DW_html_new );
rxfunc( DW_html_raw );
rxfunc( DW_html_url );

/* clipboard */
rxfunc( DW_clipboard_get_text );
rxfunc( DW_clipboard_set_text );

/* listbox/combobox */
rxfunc( DW_listbox_new );
rxfunc( DW_combobox_new );
rxfunc( DW_listbox_selector_new );
rxfunc( DW_listbox_selector_setup );
rxfunc( DW_listbox_selector_clear );
rxfunc( DW_listbox_append );
rxfunc( DW_listbox_list_append );
rxfunc( DW_listbox_clear );
rxfunc( DW_listbox_count );
rxfunc( DW_listbox_delete );
rxfunc( DW_listbox_insert );
rxfunc( DW_listbox_select );
rxfunc( DW_listbox_selected );
rxfunc( DW_listbox_selected_multi );
rxfunc( DW_listbox_set_top );
rxfunc( DW_listbox_set_text );
rxfunc( DW_listbox_get_text );

/* menus */
rxfunc( DW_menu_new );
rxfunc( DW_menubar_new );
rxfunc( DW_menu_append_item );
rxfunc( DW_menu_delete_item );
rxfunc( DW_menu_destroy );
rxfunc( DW_menu_item_set_state );
rxfunc( DW_menu_popup );

/* multiline edit */
rxfunc( DW_mle_new );
rxfunc( DW_mle_delete );
rxfunc( DW_mle_import );
rxfunc( DW_mle_export );
rxfunc( DW_mle_get_size );
rxfunc( DW_mle_search );
rxfunc( DW_mle_set_cursor );
rxfunc( DW_mle_set_editable );
rxfunc( DW_mle_set_visible );
rxfunc( DW_mle_set_word_wrap );
rxfunc( DW_mle_freeze );
rxfunc( DW_mle_thaw );

/* notebook */
rxfunc( DW_notebook_new );
rxfunc( DW_notebook_page_new );
rxfunc( DW_notebook_pack );
rxfunc( DW_notebook_page_destroy );
rxfunc( DW_notebook_page_set );
rxfunc( DW_notebook_page_set_text );
rxfunc( DW_notebook_page_set_status_text );
rxfunc( DW_notebook_page_get );

/* percent */
rxfunc( DW_percent_new );
rxfunc( DW_percent_set_pos );

/* scrollbar */
rxfunc( DW_scrollbar_new );
rxfunc( DW_scrollbar_set_range );
rxfunc( DW_scrollbar_set_pos );
rxfunc( DW_scrollbar_get_pos );

/* slider */
rxfunc( DW_slider_new );
rxfunc( DW_slider_set_pos );
rxfunc( DW_slider_get_pos );

/* spinbutton */
rxfunc( DW_spinbutton_new );
rxfunc( DW_spinbutton_set_limits );
rxfunc( DW_spinbutton_set_pos );
rxfunc( DW_spinbutton_get_pos );

/* splitbar */
rxfunc( DW_splitbar_new );
rxfunc( DW_splitbar_get );
rxfunc( DW_splitbar_set );

/* text */
rxfunc( DW_status_text_new );
rxfunc( DW_text_new );

/* tree */
rxfunc( DW_tree_new );
rxfunc( DW_tree_insert );
rxfunc( DW_tree_insert_after );
rxfunc( DW_tree_clear );
rxfunc( DW_tree_item_delete );
rxfunc( DW_tree_item_collapse );
rxfunc( DW_tree_item_expand );
rxfunc( DW_tree_item_set_data );
rxfunc( DW_tree_item_get_data );
rxfunc( DW_tree_item_select );
rxfunc( DW_tree_item_change );
rxfunc( DW_tree_get_parent );

/* window */
rxfunc( DW_window_new );
rxfunc( DW_mdi_new );
rxfunc( DW_window_destroy );
rxfunc( DW_window_capture );
rxfunc( DW_window_click_default );
rxfunc( DW_window_default );
rxfunc( DW_window_enable );
rxfunc( DW_window_disable );
rxfunc( DW_window_from_id );
rxfunc( DW_window_function ); /* not avaliable */
rxfunc( DW_window_hide );
rxfunc( DW_window_show );
rxfunc( DW_window_lower );
rxfunc( DW_window_minimize );
rxfunc( DW_window_raise );
rxfunc( DW_window_redraw );
rxfunc( DW_window_release );
rxfunc( DW_window_reparent );
rxfunc( DW_window_set_bitmap );
rxfunc( DW_window_set_bitmap_from_data );
rxfunc( DW_window_set_border );
rxfunc( DW_window_set_pointer );
rxfunc( DW_window_set_color );
rxfunc( DW_window_set_font );
rxfunc( DW_window_set_focus );
rxfunc( DW_window_set_icon );
rxfunc( DW_window_set_pos );
rxfunc( DW_window_set_size );
rxfunc( DW_window_set_pos_size );
rxfunc( DW_window_get_pos_size );
rxfunc( DW_window_set_style );
rxfunc( DW_window_set_tooltip );
rxfunc( DW_window_set_text );
rxfunc( DW_window_get_text );
rxfunc( DW_window_get_font );
rxfunc( DW_window_set_gravity );
rxfunc( DW_window_get_preferred_size );

rxfunc( DW_window_set_data );
rxfunc( DW_window_get_data );

/* miscellaneous */
rxfunc( DW_messagebox );
rxfunc( DW_bitmap_new );
rxfunc( DW_bitmap_new_from_file );
rxfunc( DW_bitmap_new_from_data );
rxfunc( DW_icon_load_from_file );
rxfunc( DW_icon_load_from_data );
rxfunc( DW_taskbar_insert );
rxfunc( DW_taskbar_delete );


/* process control */
/*-----------------*/
rxfunc( DW_init );
rxfunc( DW_main );
rxfunc( DW_main_iteration );
rxfunc( DW_main_sleep );
rxfunc( DW_main_quit );
rxfunc( DW_exec );
rxfunc( DW_exit );
rxfunc( DW_shutdown );

/* dialog */
/*--------*/
rxfunc( DW_dialog_new );
rxfunc( DW_dialog_dismiss );
rxfunc( DW_dialog_wait );

/* callback management */
/*---------------------*/
rxfunc( DW_signal_connect );
rxfunc( DW_signal_disconnect );
rxfunc( DW_signal_disconnect_by_window );
rxfunc( DW_timer_connect );
rxfunc( DW_timer_disconnect );
rxfunc( DW_callback_get_timestamp );

/* browsing */
/*----------*/
rxfunc( DW_browse );
rxfunc( DW_file_browse );

/* colour support */
/*----------------*/
rxfunc( DW_color_depth_get );
rxfunc( DW_rgb );
rxfunc( DW_red_value );
rxfunc( DW_green_value );
rxfunc( DW_blue_value );
rxfunc( DW_color_background_set );
rxfunc( DW_color_foreground_set );
rxfunc( DW_color_choose );

/* font support */
/*--------------*/
rxfunc( DW_font_choose );
rxfunc( DW_font_text_extents_get );
rxfunc( DW_font_set_default );

/* module support */
/*----------------*/
rxfunc( DW_module_load );
rxfunc( DW_module_close );
rxfunc( DW_module_symbol );

/* mutex support */
/*---------------*/
rxfunc( DW_mutex_close );
rxfunc( DW_mutex_lock );
rxfunc( DW_mutex_new );
rxfunc( DW_mutex_unlock );

/* events */
/*--------*/
rxfunc( DW_event_new );
rxfunc( DW_event_close );
rxfunc( DW_event_post );
rxfunc( DW_event_reset );
rxfunc( DW_event_wait );

/* thread support */
/*----------------*/
rxfunc( DW_thread_new );
rxfunc( DW_thread_end );
rxfunc( DW_thread_id );

/* pointer position */
/*------------------*/
rxfunc( DW_pointer_set_pos );
rxfunc( DW_pointer_get_pos );

/* utility */
/*---------------*/
rxfunc( DW_app_dir );
rxfunc( DW_beep );
rxfunc( DW_debug );
rxfunc( DW_environment_query );
rxfunc( DW_user_dir );
rxfunc( DW_screen_height );
rxfunc( DW_screen_width );
rxfunc( DW_or );
rxfunc( DW_and );
rxfunc( DW_flush );
rxfunc( DW_free );

/* function package management */
/*-----------------------------*/
rxfunc( DW_DropFuncs );
rxfunc( DW_LoadFuncs );
rxfunc( DW_QueryFunction );
rxfunc( DW_Variable );

#ifdef HAVE_ICONV_H
rxfunc( DW_convert );
#endif

/*
 * Table of Rexx/DW Functions. Used to install/de-install functions.
 * If you change this table, don't forget to change the table at the end
 * of this file.
 */
RexxFunction RexxDWFunctions[] = {
   { "DW_APP_DIR", DW_app_dir ,"DW_app_dir" , 1 },
   { "DW_BEEP", DW_beep ,"DW_beep" , 1 },
   { "DW_BITMAP_NEW", DW_bitmap_new ,"DW_bitmap_new" , 1 },
   { "DW_BITMAP_NEW_FROM_FILE", DW_bitmap_new_from_file ,"DW_bitmap_new_from_file" , 1 },
   { "DW_BITMAP_NEW_FROM_DATA", DW_bitmap_new_from_data ,"DW_bitmap_new_from_data" , 1 },
   { "DW_BITMAPBUTTON_NEW_FROM_FILE", DW_bitmapbutton_new_from_file ,"DW_bitmapbutton_new_from_file" , 1 },
   { "DW_BITMAPBUTTON_NEW_FROM_DATA", DW_bitmapbutton_new_from_data ,"DW_bitmapbutton_new_from_data" , 1 },
   { "DW_BOX_NEW", DW_box_new ,"DW_box_new" , 1 },
   { "DW_BOX_PACK_AT_INDEX", DW_box_pack_at_index ,"DW_box_pack_at_index" , 1 },
   { "DW_BOX_PACK_END", DW_box_pack_end ,"DW_box_pack_end" , 1 },
   { "DW_BOX_PACK_START", DW_box_pack_start ,"DW_box_pack_start" , 1 },
   { "DW_BOX_UNPACK", DW_box_unpack ,"DW_box_unpack" , 1 },
   { "DW_BOX_UNPACK_AT_INDEX", DW_box_unpack_at_index ,"DW_box_unpack_at_index" , 1 },
   { "DW_BROWSE", DW_browse ,"DW_browse" , 1 },
   { "DW_BUTTON_NEW", DW_button_new ,"DW_button_new" , 1 },
   { "DW_CALENDAR_NEW", DW_calendar_new ,"DW_calendar_new" , 1 },
   { "DW_CALENDAR_GET_DATE", DW_calendar_get_date ,"DW_calendar_get_date" , 1 },
   { "DW_CALENDAR_SET_DATE", DW_calendar_set_date ,"DW_calendar_set_date" , 1 },
   { "DW_CALLBACK_GET_TIMESTAMP", DW_callback_get_timestamp ,"DW_callback_get_timestamp" , 1 },
   { "DW_CHECKBOX_NEW", DW_checkbox_new ,"DW_checkbox_new" , 1 },
   { "DW_CHECKBOX_GET", DW_checkbox_get ,"DW_checkbox_get" , 1 },
   { "DW_CHECKBOX_SET", DW_checkbox_set ,"DW_checkbox_set" , 1 },
   { "DW_CLIPBOARD_GET_TEXT", DW_clipboard_get_text,"DW_clipboard_get_text" , 1 },
   { "DW_CLIPBOARD_SET_TEXT", DW_clipboard_set_text,"DW_clipboard_set_text" , 1 },
   { "DW_COLOR_BACKGROUND_SET", DW_color_background_set ,"DW_color_background_set" , 1 },
   { "DW_COLOR_CHOOSE", DW_color_choose ,"DW_color_choose" , 1 },
   { "DW_COLOR_DEPTH_GET", DW_color_depth_get ,"DW_color_depth_get" , 1 },
   { "DW_COLOR_FOREGROUND_SET", DW_color_foreground_set ,"DW_color_foreground_set" , 1 },
   { "DW_COMBOBOX_NEW", DW_combobox_new ,"DW_combobox_new" , 1 },
   { "DW_CONTAINER_ALLOC", DW_container_alloc ,"DW_container_alloc" , 1 },
   { "DW_CONTAINER_CHANGE_ITEM", DW_container_change_item ,"DW_container_change_item" , 1 },
   { "DW_CONTAINER_CLEAR", DW_container_clear ,"DW_container_clear" , 1 },
   { "DW_CONTAINER_CURSOR", DW_container_cursor ,"DW_container_cursor" , 1 },
   { "DW_CONTAINER_DELETE", DW_container_delete ,"DW_container_delete" , 1 },
   { "DW_CONTAINER_DELETE_ROW", DW_container_delete_row ,"DW_container_delete_row" , 1 },
   { "DW_CONTAINER_INSERT", DW_container_insert ,"DW_container_insert" , 1 },
   { "DW_CONTAINER_NEW", DW_container_new ,"DW_container_new" , 1 },
   { "DW_CONTAINER_OPTIMIZE", DW_container_optimize ,"DW_container_optimize" , 1 },
   { "DW_CONTAINER_QUERY_NEXT", DW_container_query_next ,"DW_container_query_next" , 1 },
   { "DW_CONTAINER_QUERY_START", DW_container_query_start ,"DW_container_query_start" , 1 },
   { "DW_CONTAINER_SCROLL", DW_container_scroll ,"DW_container_scroll" , 1 },
   { "DW_CONTAINER_SET_COLUMN_WIDTH", DW_container_set_column_width ,"DW_container_set_column_width" , 1 },
   { "DW_CONTAINER_SET_ITEM", DW_container_set_item ,"DW_container_set_item" , 1 },
   { "DW_CONTAINER_SET_ROW_TITLE", DW_container_set_row_title ,"DW_container_set_row_title" , 1 },
   { "DW_CONTAINER_SET_STRIPE", DW_container_set_stripe ,"DW_container_set_stripe" , 1 },
   { "DW_CONTAINER_SETUP", DW_container_setup ,"DW_container_setup" , 1 },
   { "DW_DEBUG", DW_debug ,"DW_debug" , 1 },
   { "DW_DIALOG_DISMISS", DW_dialog_dismiss ,"DW_dialog_dismiss" , 1 },
   { "DW_DIALOG_NEW", DW_dialog_new ,"DW_dialog_new" , 1 },
   { "DW_DIALOG_WAIT", DW_dialog_wait ,"DW_dialog_wait" , 1 },
   { "DW_DRAW_LINE", DW_draw_line ,"DW_draw_line" , 1 },
   { "DW_DRAW_POINT", DW_draw_point ,"DW_draw_point" , 1 },
   { "DW_DRAW_POLYGON", DW_draw_polygon ,"DW_draw_polygon" , 1 },
   { "DW_DRAW_RECT", DW_draw_rect ,"DW_draw_rect" , 1 },
   { "DW_DRAW_TEXT", DW_draw_text ,"DW_draw_text" , 1 },
   { "DW_ENTRYFIELD_NEW", DW_entryfield_new ,"DW_entryfield_new" , 1 },
   { "DW_ENTRYFIELD_PASSWORD_NEW", DW_entryfield_password_new ,"DW_entryfield_password_new" , 1 },
   { "DW_ENTRYFIELD_SET_LIMIT", DW_entryfield_set_limit ,"DW_entryfield_set_limit" , 1 },
   { "DW_ENVIRONMENT_QUERY", DW_environment_query ,"DW_environment_query" , 1 },
   { "DW_EVENT_CLOSE", DW_event_close ,"DW_event_close" , 1 },
   { "DW_EVENT_NEW", DW_event_new ,"DW_event_new" , 1 },
   { "DW_EVENT_POST", DW_event_post ,"DW_event_post" , 1 },
   { "DW_EVENT_RESET", DW_event_reset ,"DW_event_reset" , 1 },
   { "DW_EVENT_WAIT", DW_event_wait ,"DW_event_wait" , 1 },
   { "DW_EXEC", DW_exec ,"DW_exec" , 1 },
   { "DW_EXIT", DW_exit ,"DW_exit" , 1 },
   { "DW_FILE_BROWSE", DW_file_browse ,"DW_file_browse" , 1 },
   { "DW_FILESYSTEM_CHANGE_FILE", DW_filesystem_change_file ,"DW_filesystem_change_file" , 1 },
   { "DW_FILESYSTEM_CHANGE_ITEM", DW_filesystem_change_item ,"DW_filesystem_change_item" , 1 },
   { "DW_FILESYSTEM_SET_FILE", DW_filesystem_set_file ,"DW_filesystem_set_file" , 1 },
   { "DW_FILESYSTEM_SET_ITEM", DW_filesystem_set_item ,"DW_filesystem_set_item" , 1 },
   { "DW_FILESYSTEM_SETUP", DW_filesystem_setup ,"DW_filesystem_setup" , 1 },
   { "DW_FLUSH", DW_flush ,"DW_flush" , 1 },
   { "DW_FONT_CHOOSE", DW_font_choose ,"DW_font_choose" , 1 },
   { "DW_FONT_SET_DEFAULT", DW_font_set_default ,"DW_font_set_default" , 1 },
   { "DW_FONT_TEXT_EXTENTS_GET", DW_font_text_extents_get ,"DW_font_text_extents_get" , 1 },
   { "DW_FREE", DW_free ,"DW_free" , 1 },
   { "DW_GROUPBOX_NEW", DW_groupbox_new ,"DW_groupbox_new" , 1 },
   { "DW_HTML_ACTION", DW_html_action ,"DW_html_action" , 1 },
   { "DW_HTML_NEW", DW_html_new ,"DW_html_new" , 1 },
   { "DW_HTML_RAW", DW_html_raw ,"DW_html_raw" , 1 },
   { "DW_HTML_URL", DW_html_url ,"DW_html_url" , 1 },
   { "DW_ICON_LOAD_FROM_DATA", DW_icon_load_from_data ,"DW_icon_load_from_data" , 1 },
   { "DW_ICON_LOAD_FROM_FILE", DW_icon_load_from_file ,"DW_icon_load_from_file" , 1 },
   { "DW_INIT", DW_init ,"DW_init" , 1 },
   { "DW_LISTBOX_APPEND", DW_listbox_append ,"DW_listbox_append" , 1 },
   { "DW_LISTBOX_LIST_APPEND", DW_listbox_list_append ,"DW_listbox_list_append" , 1 },
   { "DW_LISTBOX_CLEAR", DW_listbox_clear ,"DW_listbox_clear" , 1 },
   { "DW_LISTBOX_COUNT", DW_listbox_count ,"DW_listbox_count" , 1 },
   { "DW_LISTBOX_DELETE", DW_listbox_delete ,"DW_listbox_delete" , 1 },
   { "DW_LISTBOX_INSERT", DW_listbox_insert ,"DW_listbox_insert" , 1 },
   { "DW_LISTBOX_NEW", DW_listbox_new ,"DW_listbox_new" , 1 },
   { "DW_LISTBOX_GET_TEXT", DW_listbox_get_text ,"DW_listbox_get_text" , 1 },
   { "DW_LISTBOX_SELECT", DW_listbox_select ,"DW_listbox_select" , 1 },
   { "DW_LISTBOX_SELECTED", DW_listbox_selected ,"DW_listbox_selected" , 1 },
   { "DW_LISTBOX_SELECTED_MULTI", DW_listbox_selected_multi ,"DW_listbox_selected_multi" , 1 },
   { "DW_LISTBOX_SELECTOR_NEW", DW_listbox_selector_new ,"DW_listbox_selector_new" , 1 },
   { "DW_LISTBOX_SELECTOR_SETUP", DW_listbox_selector_setup ,"DW_listbox_selector_setup" , 1 },
   { "DW_LISTBOX_SELECTOR_CLEAR", DW_listbox_selector_clear ,"DW_listbox_selector_clear" , 1 },
   { "DW_LISTBOX_SET_TEXT", DW_listbox_set_text ,"DW_listbox_set_text" , 1 },
   { "DW_LISTBOX_SET_TOP", DW_listbox_set_top ,"DW_listbox_set_top" , 1 },
   { "DW_MAIN", DW_main ,"DW_main" , 1 },
   { "DW_MAIN_ITERATION", DW_main_iteration ,"DW_main_iteration" , 1 },
   { "DW_MAIN_QUIT", DW_main_quit ,"DW_main_quit" , 1 },
   { "DW_MAIN_SLEEP", DW_main_sleep ,"DW_main_sleep" , 1 },
   { "DW_MDI_NEW", DW_mdi_new ,"DW_mdi_new" , 1 },
   { "DW_MENU_APPEND_ITEM", DW_menu_append_item ,"DW_menu_append_item" , 1 },
   { "DW_MENU_DELETE_ITEM", DW_menu_delete_item ,"DW_menu_delete_item" , 1 },
   { "DW_MENU_DESTROY", DW_menu_destroy ,"DW_menu_destroy" , 1 },
   { "DW_MENU_ITEM_SET_STATE", DW_menu_item_set_state ,"DW_menu_item_set_state" , 1 },
   { "DW_MENU_NEW", DW_menu_new ,"DW_menu_new" , 1 },
   { "DW_MENU_POPUP", DW_menu_popup ,"DW_menu_popup" , 1 },
   { "DW_MENUBAR_NEW", DW_menubar_new ,"DW_menubar_new" , 1 },
   { "DW_MESSAGEBOX", DW_messagebox ,"DW_messagebox" , 1 },
   { "DW_MLE_DELETE", DW_mle_delete ,"DW_mle_delete" , 1 },
   { "DW_MLE_EXPORT", DW_mle_export ,"DW_mle_export" , 1 },
   { "DW_MLE_FREEZE", DW_mle_freeze ,"DW_mle_freeze" , 1 },
   { "DW_MLE_IMPORT", DW_mle_import ,"DW_mle_import" , 1 },
   { "DW_MLE_NEW", DW_mle_new ,"DW_mle_new" , 1 },
   { "DW_MLE_GET_SIZE", DW_mle_get_size ,"DW_mle_get_size" , 1 },
   { "DW_MLE_SEARCH", DW_mle_search ,"DW_mle_search" , 1 },
   { "DW_MLE_SET_CURSOR", DW_mle_set_cursor ,"DW_mle_set_cursor" , 1 },
   { "DW_MLE_SET_EDITABLE", DW_mle_set_editable ,"DW_mle_set_editable" , 1 },
   { "DW_MLE_SET_VISIBLE", DW_mle_set_visible ,"DW_mle_set_visible" , 1 },
   { "DW_MLE_SET_WORD_WRAP", DW_mle_set_word_wrap ,"DW_mle_set_word_wrap" , 1 },
   { "DW_MLE_THAW", DW_mle_thaw ,"DW_mle_thaw" , 1 },
   { "DW_MODULE_CLOSE", DW_module_close ,"DW_module_close" , 1 },
   { "DW_MODULE_LOAD", DW_module_load ,"DW_module_load" , 1 },
   { "DW_MODULE_SYMBOL", DW_module_symbol ,"DW_module_symbol" , 1 },
   { "DW_MUTEX_CLOSE", DW_mutex_close ,"DW_mutex_close" , 1 },
   { "DW_MUTEX_LOCK", DW_mutex_lock ,"DW_mutex_lock" , 1 },
   { "DW_MUTEX_NEW", DW_mutex_new ,"DW_mutex_new" , 1 },
   { "DW_MUTEX_UNLOCK", DW_mutex_unlock ,"DW_mutex_unlock" , 1 },
   { "DW_NOTEBOOK_NEW", DW_notebook_new ,"DW_notebook_new" , 1 },
   { "DW_NOTEBOOK_PACK", DW_notebook_pack ,"DW_notebook_pack" , 1 },
   { "DW_NOTEBOOK_PAGE_DESTROY", DW_notebook_page_destroy ,"DW_notebook_page_destroy" , 1 },
   { "DW_NOTEBOOK_PAGE_NEW", DW_notebook_page_new ,"DW_notebook_page_new" , 1 },
   { "DW_NOTEBOOK_PAGE_GET", DW_notebook_page_get ,"DW_notebook_page_get" , 1 },
   { "DW_NOTEBOOK_PAGE_SET", DW_notebook_page_set ,"DW_notebook_page_set" , 1 },
   { "DW_NOTEBOOK_PAGE_SET_STATUS_TEXT", DW_notebook_page_set_status_text ,"DW_notebook_page_set_status_text" , 1 },
   { "DW_NOTEBOOK_PAGE_SET_TEXT", DW_notebook_page_set_text ,"DW_notebook_page_set_text" , 1 },
   { "DW_PERCENT_NEW", DW_percent_new ,"DW_percent_new" , 1 },
   { "DW_PERCENT_SET_POS", DW_percent_set_pos ,"DW_percent_set_pos" , 1 },
   { "DW_PIXMAP_BITBLT", DW_pixmap_bitblt ,"DW_pixmap_bitblt" , 1 },
   { "DW_PIXMAP_DESTROY", DW_pixmap_destroy ,"DW_pixmap_destroy" , 1 },
   { "DW_PIXMAP_NEW", DW_pixmap_new ,"DW_pixmap_new" , 1 },
   { "DW_PIXMAP_NEW_FROM_DATA", DW_pixmap_new_from_data ,"DW_pixmap_new_from_data" , 1 },
   { "DW_PIXMAP_NEW_FROM_FILE", DW_pixmap_new_from_file ,"DW_pixmap_new_from_file" , 1 },
   { "DW_PIXMAP_SET_TRANSPARENT_COLOR", DW_pixmap_set_transparent_color,"DW_pixmap_set_transparent_color" , 1 },
   { "DW_PIXMAP_WIDTH", DW_pixmap_width ,"DW_pixmap_width" , 1 },
   { "DW_PIXMAP_HEIGHT", DW_pixmap_height ,"DW_pixmap_height" , 1 },
   { "DW_POINTER_GET_POS", DW_pointer_get_pos ,"DW_pointer_get_pos" , 1 },
   { "DW_POINTER_SET_POS", DW_pointer_set_pos ,"DW_pointer_set_pos" , 1 },
   { "DW_RADIOBUTTON_NEW", DW_radiobutton_new ,"DW_radiobutton_new" , 1 },
   { "DW_RADIOBUTTON_GET", DW_radiobutton_get ,"DW_radiobutton_get" , 1 },
   { "DW_RADIOBUTTON_SET", DW_radiobutton_set ,"DW_radiobutton_set" , 1 },
   { "DW_RENDER_NEW", DW_render_new ,"DW_render_new" , 1 },
   { "DW_SCREEN_HEIGHT", DW_screen_height ,"DW_screen_height" , 1 },
   { "DW_SCREEN_WIDTH", DW_screen_width ,"DW_screen_width" , 1 },
   { "DW_SCROLLBAR_NEW", DW_scrollbar_new ,"DW_scrollbar_new" , 1 },
   { "DW_SCROLLBAR_GET_POS", DW_scrollbar_get_pos ,"DW_scrollbar_get_pos" , 1 },
   { "DW_SCROLLBAR_SET_POS", DW_scrollbar_set_pos ,"DW_scrollbar_set_pos" , 1 },
   { "DW_SCROLLBAR_SET_RANGE", DW_scrollbar_set_range ,"DW_scrollbar_set_range" , 1 },
   { "DW_SCROLLBOX_NEW", DW_scrollbox_new ,"DW_scrollbox_new" , 1 },
   { "DW_SHUTDOWN", DW_shutdown ,"DW_shutdown" , 1 },
   { "DW_SIGNAL_CONNECT", DW_signal_connect ,"DW_signal_connect" , 1 },
   { "DW_SIGNAL_DISCONNECT", DW_signal_disconnect ,"DW_signal_disconnect" , 1 },
   { "DW_SIGNAL_DISCONNECT_BY_WINDOW", DW_signal_disconnect_by_window ,"DW_signal_disconnect_by_window" , 1 },
   { "DW_SLIDER_NEW", DW_slider_new ,"DW_slider_new" , 1 },
   { "DW_SLIDER_GET_POS", DW_slider_get_pos ,"DW_slider_get_pos" , 1 },
   { "DW_SLIDER_SET_POS", DW_slider_set_pos ,"DW_slider_set_pos" , 1 },
   { "DW_SPINBUTTON_NEW", DW_spinbutton_new ,"DW_spinbutton_new" , 1 },
   { "DW_SPINBUTTON_GET_POS", DW_spinbutton_get_pos ,"DW_spinbutton_get_pos" , 1 },
   { "DW_SPINBUTTON_SET_LIMITS", DW_spinbutton_set_limits ,"DW_spinbutton_set_limits" , 1 },
   { "DW_SPINBUTTON_SET_POS", DW_spinbutton_set_pos ,"DW_spinbutton_set_pos" , 1 },
   { "DW_SPLITBAR_GET", DW_splitbar_get ,"DW_splitbar_get" , 1 },
   { "DW_SPLITBAR_NEW", DW_splitbar_new ,"DW_splitbar_new" , 1 },
   { "DW_SPLITBAR_SET", DW_splitbar_set ,"DW_splitbar_set" , 1 },
   { "DW_STATUS_TEXT_NEW", DW_status_text_new ,"DW_status_text_new" , 1 },
   { "DW_TASKBAR_DELETE", DW_taskbar_delete,"DW_taskbar_delete" , 1 },
   { "DW_TASKBAR_INSERT", DW_taskbar_insert,"DW_taskbar_insert" , 1 },
   { "DW_TEXT_NEW", DW_text_new ,"DW_text_new" , 1 },
   { "DW_THREAD_END", DW_thread_end ,"DW_thread_end" , 1 },
   { "DW_THREAD_ID", DW_thread_id ,"DW_thread_id" , 1 },
   { "DW_THREAD_NEW", DW_thread_new ,"DW_thread_new" , 1 },
   { "DW_TIMER_CONNECT", DW_timer_connect ,"DW_timer_connect" , 1 },
   { "DW_TIMER_DISCONNECT", DW_timer_disconnect ,"DW_timer_disconnect" , 1 },
   { "DW_TREE_CLEAR", DW_tree_clear ,"DW_tree_clear" , 1 },
   { "DW_TREE_ITEM_COLLAPSE", DW_tree_item_collapse ,"DW_tree_item_collapse" , 1 },
   { "DW_TREE_ITEM_DELETE", DW_tree_item_delete ,"DW_tree_item_delete" , 1 },
   { "DW_TREE_ITEM_EXPAND", DW_tree_item_expand ,"DW_tree_item_expand" , 1 },
   { "DW_TREE_ITEM_GET_DATA", DW_tree_item_get_data ,"DW_tree_item_get_data" , 1 },
   { "DW_TREE_INSERT", DW_tree_insert ,"DW_tree_insert" , 1 },
   { "DW_TREE_INSERT_AFTER", DW_tree_insert_after ,"DW_tree_insert_after" , 1 },
   { "DW_TREE_ITEM_SELECT", DW_tree_item_select ,"DW_tree_item_select" , 1 },
   { "DW_TREE_NEW", DW_tree_new ,"DW_tree_new" , 1 },
   { "DW_TREE_ITEM_CHANGE", DW_tree_item_change ,"DW_tree_item_change" , 1 },
   { "DW_TREE_ITEM_SET_DATA", DW_tree_item_set_data ,"DW_tree_item_set_data" , 1 },
   { "DW_TREE_GET_PARENT", DW_tree_get_parent ,"DW_tree_get_parent" , 1 },
   { "DW_USER_DIR", DW_user_dir ,"DW_user_dir" , 1 },
   { "DW_WINDOW_CAPTURE", DW_window_capture ,"DW_window_capture" , 1 },
   { "DW_WINDOW_CLICK_DEFAULT", DW_window_click_default ,"DW_window_click_default" , 1 },
   { "DW_WINDOW_DEFAULT", DW_window_default ,"DW_window_default" , 1 },
   { "DW_WINDOW_DESTROY", DW_window_destroy ,"DW_window_destroy" , 1 },
   { "DW_WINDOW_DISABLE", DW_window_disable ,"DW_window_disable" , 1 },
   { "DW_WINDOW_ENABLE", DW_window_enable ,"DW_window_enable" , 1 },
   { "DW_WINDOW_FROM_ID", DW_window_from_id ,"DW_window_from_id" , 1 },
/* { "DW_WINDOW_FUNCTION", DW_window_function ,"DW_window_function" , 1 },*/
   { "DW_WINDOW_GET_DATA", DW_window_get_data ,"DW_window_get_data" , 1 },
   { "DW_WINDOW_GET_POS_SIZE", DW_window_get_pos_size ,"DW_window_get_pos_size" , 1 },
   { "DW_WINDOW_GET_PREFERRED_SIZE", DW_window_get_preferred_size ,"DW_window_get_preferred_size" , 1 },
   { "DW_WINDOW_GET_TEXT", DW_window_get_text ,"DW_window_get_text" , 1 },
   { "DW_WINDOW_GET_FONT", DW_window_get_font ,"DW_window_get_font" , 1 },
   { "DW_WINDOW_HIDE", DW_window_hide ,"DW_window_hide" , 1 },
   { "DW_WINDOW_LOWER", DW_window_lower ,"DW_window_lower" , 1 },
   { "DW_WINDOW_MINIMIZE", DW_window_minimize ,"DW_window_minimize" , 1 },
   { "DW_WINDOW_NEW", DW_window_new ,"DW_window_new" , 1 },
   { "DW_WINDOW_SET_POINTER", DW_window_set_pointer ,"DW_window_set_pointer" , 1 },
   { "DW_WINDOW_RAISE", DW_window_raise ,"DW_window_raise" , 1 },
   { "DW_WINDOW_REDRAW", DW_window_redraw ,"DW_window_redraw" , 1 },
   { "DW_WINDOW_RELEASE", DW_window_release ,"DW_window_release" , 1 },
   { "DW_WINDOW_REPARENT", DW_window_reparent ,"DW_window_reparent" , 1 },
   { "DW_WINDOW_SET_BITMAP", DW_window_set_bitmap ,"DW_window_set_bitmap" , 1 },
   { "DW_WINDOW_SET_BITMAP_FROM_DATA", DW_window_set_bitmap_from_data ,"DW_window_set_bitmap_from_data" , 1 },
   { "DW_WINDOW_SET_BORDER", DW_window_set_border ,"DW_window_set_border" , 1 },
   { "DW_WINDOW_SET_COLOR", DW_window_set_color ,"DW_window_set_color" , 1 },
   { "DW_WINDOW_SET_DATA", DW_window_set_data ,"DW_window_set_data" , 1 },
   { "DW_WINDOW_SET_FOCUS", DW_window_set_focus ,"DW_window_set_focus" , 1 },
   { "DW_WINDOW_SET_FONT", DW_window_set_font ,"DW_window_set_font" , 1 },
   { "DW_WINDOW_SET_GRAVITY", DW_window_set_gravity ,"DW_window_set_gravity" , 1 },
   { "DW_WINDOW_SET_ICON", DW_window_set_icon ,"DW_window_set_icon" , 1 },
   { "DW_WINDOW_SET_POS", DW_window_set_pos ,"DW_window_set_pos" , 1 },
   { "DW_WINDOW_SET_POS_SIZE", DW_window_set_pos_size ,"DW_window_set_pos_size" , 1 },
   { "DW_WINDOW_SET_STYLE", DW_window_set_style ,"DW_window_set_style" , 1 },
   { "DW_WINDOW_SET_TOOLTIP", DW_window_set_tooltip ,"DW_window_set_tooltip" , 1 },
   { "DW_WINDOW_SET_TEXT", DW_window_set_text ,"DW_window_set_text" , 1 },
   { "DW_WINDOW_SET_SIZE", DW_window_set_size ,"DW_window_set_size" , 1 },
   { "DW_WINDOW_SHOW", DW_window_show ,"DW_window_show" , 1 },
   { "DW_VARIABLE", DW_Variable ,"DW_Variable" , 1 },
   { "DW_DROPFUNCS", DW_DropFuncs ,"DW_DropFuncs" , 1 },
   { "DW_LOADFUNCS", DW_LoadFuncs ,"DW_LoadFuncs" , 0 },
   { "DW_QUERYFUNCTION", DW_QueryFunction ,"DW_QueryFunction" , 1 },
   { "DW_OR", DW_or ,"DW_or" , 1 },
   { "DW_AND", DW_and ,"DW_and" , 1 },
   { "DW_RED_VALUE", DW_red_value ,"DW_red_value" , 1 },
   { "DW_GREEN_VALUE", DW_green_value ,"DW_green_value" , 1 },
   { "DW_BLUE_VALUE", DW_blue_value ,"DW_blue_value" , 1 },
   { "DW_RGB", DW_rgb ,"DW_rgb" , 1 },
#ifdef HAVE_ICONV_H
   { "DW_CONVERT", DW_convert ,"DW_convert" , 1 },
#endif
   { NULL, NULL, NULL,0 }
};

typedef struct
{
   char *func;
   int argc;
   PRXSTRING argv;
   RxPackageGlobalDataDef *RxPackageGlobalData;
} REXXDWUserData;

typedef struct
{
   char *func;
   int argc;
   PRXSTRING argv;
   RxPackageGlobalDataDef *RxPackageGlobalData;
} tREXXDWUserData;

/*
 * The values #defined here MUST correspond to the position
 * of their equivalents in the callback structure mapping below
 */
#define REXXDW_CONFIGURE_EVENT_CB            0
#define REXXDW_KEY_PRESS_EVENT_CB            1
#define REXXDW_BUTTON_PRESS_EVENT_CB         2
#define REXXDW_BUTTON_RELEASE_EVENT_CB       3
#define REXXDW_MOTION_NOTIFY_EVENT_CB        4
#define REXXDW_DELETE_EVENT_CB               5
#define REXXDW_EXPOSE_EVENT_CB               6
#define REXXDW_CLICKED_EVENT_CB              7
#define REXXDW_ITEM_ENTER_EVENT_CB           8
#define REXXDW_ITEM_CONTEXT_EVENT_CB         9
#define REXXDW_LIST_SELECT_EVENT_CB         10
#define REXXDW_ITEM_SELECT_EVENT_CB         11
#define REXXDW_SET_FOCUS_EVENT_CB           12
#define REXXDW_VALUE_CHANGED_EVENT_CB       13
#define REXXDW_SWITCH_PAGE_EVENT_CB         14
#define REXXDW_COLUMN_CLICK_EVENT_CB        15
#define REXXDW_MAX_CB                       15 /* max is largest valid CB */

/*
 * Define all the structs for the different callbacks
 */
typedef struct
{
   int width;
   int height;
} REXXDWConfigureCallbackData;

typedef struct
{
   char key;
   int virtual_key;
   int state;
} REXXDWKeyPressCallbackData;

typedef struct
{
   int x;
   int y;
   int button;
} REXXDWButtonPressCallbackData;

typedef struct
{
   int x;
   int y;
   int button;
} REXXDWButtonReleaseCallbackData;

typedef struct
{
   int x;
   int y;
   int state;
} REXXDWMotionNotifyCallbackData;

typedef struct
{
   int x;
   int y;
   int width;
   int height;
} REXXDWExposeCallbackData;

typedef struct
{
   char *text;
} REXXDWItemEnterCallbackData;

typedef struct
{
   char *text;
   int x;
   int y;
   char *itemdata;
} REXXDWItemContextCallbackData;

typedef struct
{
   int item;
} REXXDWListSelectCallbackData;

typedef struct
{
   HWND item;
   char *text;
   char *itemdata;
} REXXDWItemSelectCallbackData;

typedef struct
{
   int val;
} REXXDWValueChangedCallbackData;

typedef struct
{
   int page;
} REXXDWSwitchPageCallbackData;

typedef struct
{
   int column_num;
} REXXDWColumnClickCallbackData;

/*
 * ListboxSelector Widget
 */
typedef struct
{
   void *next;
   HWND Topbox;
   HWND LeftListbox;
   HWND RightListbox;
   HWND LeftButton;
   HWND RightButton;
   HWND AllLeftButton;
   HWND AllRightButton;
   int count;
   char ValueStemName[100];
   char SetStemName[100];
   int *LeftIndex;
   int *RightIndex;
   int l_idx;
   int r_idx;
   RxPackageGlobalDataDef *RxPackageGlobalData;
} REXXDWListboxSelector;

struct REXXDWCALLBACKDATA
{
   struct REXXDWCALLBACKDATA *next;
   int callback_type;
   HWND window;
   REXXDWUserData *userdata;
   union
   {
      REXXDWConfigureCallbackData     ConfigureCallbackData;
      REXXDWKeyPressCallbackData      KeyPressCallbackData;
      REXXDWButtonPressCallbackData   ButtonPressCallbackData;
      REXXDWButtonReleaseCallbackData ButtonReleaseCallbackData;
      REXXDWMotionNotifyCallbackData  MotionNotifyCallbackData;
      REXXDWExposeCallbackData        ExposeCallbackData;
      REXXDWItemEnterCallbackData     ItemEnterCallbackData;
      REXXDWItemContextCallbackData   ItemContextCallbackData;
      REXXDWListSelectCallbackData    ListSelectCallbackData;
      REXXDWItemSelectCallbackData    ItemSelectCallbackData;
      REXXDWValueChangedCallbackData  ValueChangedCallbackData;
      REXXDWSwitchPageCallbackData    SwitchPageCallbackData;
      REXXDWColumnClickCallbackData   ColumnClickCallbackData;
/* timer ?? */
   } event;
};
typedef struct REXXDWCALLBACKDATA REXXDWCallbackData;

struct REXXDWWIDGETDATA
{
   struct REXXDWWIDGETDATA *next;
   int widget_type;
   union
   {
      REXXDWListboxSelector     ListboxSelectorData;
   } widget;
};
typedef struct REXXDWWIDGETDATA REXXDWWidgetData;

typedef struct
{
   /*
    * Data definitions required by the package
    */
   struct REXXDWCALLBACKDATA *FirstCallback;
   struct REXXDWCALLBACKDATA *LastCallback;
   struct REXXDWWIDGETDATA *FirstWidget;
   struct REXXDWWIDGETDATA *LastWidget;
   time_t CallbackTimestamp;
   RxPackageGlobalDataDef *RxPackageGlobalData;
} REXXDWDATA;

/*
 * Callback lookup table
 */
typedef struct
{
   void *sigfunc;
   char *signame;
} REXXDWSignalList;

REXXDWSignalList REXXDWSignalTranslate[REXXDW_MAX_CB+1] =
{
   { rexxdw_configure_event_cb, DW_SIGNAL_CONFIGURE },
   { rexxdw_key_press_event_cb, DW_SIGNAL_KEY_PRESS },
   { rexxdw_button_press_event_cb, DW_SIGNAL_BUTTON_PRESS },
   { rexxdw_button_release_event_cb, DW_SIGNAL_BUTTON_RELEASE },
   { rexxdw_motion_notify_event_cb, DW_SIGNAL_MOTION_NOTIFY },
   { rexxdw_delete_event_cb, DW_SIGNAL_DELETE },
   { rexxdw_expose_event_cb, DW_SIGNAL_EXPOSE },
   { rexxdw_clicked_event_cb, DW_SIGNAL_CLICKED },
   { rexxdw_item_enter_event_cb, DW_SIGNAL_ITEM_ENTER },
   { rexxdw_item_context_event_cb, DW_SIGNAL_ITEM_CONTEXT },
   { rexxdw_list_select_event_cb, DW_SIGNAL_LIST_SELECT },
   { rexxdw_item_select_event_cb, DW_SIGNAL_ITEM_SELECT },
   { rexxdw_set_focus_event_cb, DW_SIGNAL_SET_FOCUS },
   { rexxdw_value_changed_event_cb, DW_SIGNAL_VALUE_CHANGED },
   { rexxdw_switch_page_event_cb, DW_SIGNAL_SWITCH_PAGE },
   { rexxdw_column_click_event_cb, DW_SIGNAL_COLUMN_CLICK },
};

/****h* RexxDW/Constants
 * DESCRIPTION
 *   The following "constants" are defined when RexxDW starts. By default, all constants
 *   are stored in an array with the stem preset to !REXXDW.!
 *   This can be changed by using the 'CONSTANTPREFIX' value of DW_variable().
 *   If you use "Procedure" on your labels, you MUST "EXPOSE !REXXDW." or the
 *   stem you set with DW_variable() will not be visible.
 *   To reference the constants defined below, you must prefix them. So
 *   the "constant" DW_DESKTOP would be, by default, referenced in your code as
 *   !REXXDW.!DW_DESKTOP.
 * SEE ALSO
 *   DW_variable
 ******
 */
RxPackageConstantDef RexxDWConstants[] =
{
/****d* Constants/MiscellaneousFlags
 * NAME
 *   MiscellaneousFlags
 * DESCRIPTION
 *   The following is a list of miscellaneous flags.
 * ATTRIBUTES
 *   - GTK_MAJOR_VERSION        - 0 if NOT GTK+ platform, 1 for GTK+ 1.x, 2 for GTK+ 2.x
 *   - HAVE_REXXCALLBACK        - 1 if the interpreter supports RexxCallback() API
 *   - DW_DESKTOP               - the psuedo "window" which is the desktop
 *   - DIRSEP                   - the Operating System directory separator
 *   - PATHSEP                  - the Operating System path separator
 ******
 */

/*
 * HAVE_REXXCALLBACK MUST BE FIRST!!!! See RexxDWInitHandler()
 */
   { "HAVE_REXXCALLBACK"                , 0, API_REXXCALLBACK_MISSING          , NULL        ,0.0, 0 },
   { "GTK_MAJOR_VERSION"                , 0, GTK_MAJOR_VERSION                 , NULL        ,0.0, 0 },
   { "DW_DESKTOP"                       , 0, (long)DW_DESKTOP                  , NULL        ,0.0, 0 },
   { "DIRSEP"                           , 1, 0                                 , FILE_SEPARATOR_STR ,0.0, 0 },
   { "PATHSEP"                          , 1, 0                                 , PATH_SEPARATOR_STR ,0.0, 0 },
/****d* Constants/FileDialogFlags
 * NAME
 *   FileDialogFlags
 * DESCRIPTION
 *   The following is a list of the pre-defined flags that are used in a call to
 *   DW_file_browse() to determine the dialog type displayed.
 * ATTRIBUTES
 *   - DW_DIRECTORY_OPEN        - a directory selector dialog
 *   - DW_FILE_OPEN             - a file open dialog
 *   - DW_FILE_SAVE             - a file save dialog
 * SEE ALSO
 *   DW_file_browse
 ******
 */
   { "DW_FILE_OPEN"                     , 0, DW_FILE_OPEN                      , NULL        ,0.0, 0 },
   { "DW_FILE_SAVE"                     , 0, DW_FILE_SAVE                      , NULL        ,0.0, 0 },
   { "DW_DIRECTORY_OPEN"                , 0, DW_DIRECTORY_OPEN                 , NULL        ,0.0, 0 },

   /* used by dw_exec() */
   { "DW_EXEC_CON"                      , 0, DW_EXEC_CON                       , NULL        ,0.0, 0 },
   { "DW_EXEC_GUI"                      , 0, DW_EXEC_GUI                       , NULL        ,0.0, 0 },

/****d* Constants/ContainerScrollFlags
 * NAME
 *   ContainerScrollFlags
 * DESCRIPTION
 *   The following is a list of the pre-defined flags that are used in a call to
 *   DW_container_scroll() to indicate the direction of scrolling.
 * ATTRIBUTES
 *   - DW_SCROLL_UP             - scroll up a number of lines
 *   - DW_SCROLL_DOWN           - scroll down a number of lines
 *   - DW_SCROLL_TOP            - scroll to the top of the container
 *   - DW_SCROLL_BOTTOM         - scroll to the bottom of the container
 * SEE ALSO
 *   DW_container_scroll
 ******
 */
   { "DW_SCROLL_UP"                     , 0, DW_SCROLL_UP                      , NULL        ,0.0, 0 },
   { "DW_SCROLL_DOWN"                   , 0, DW_SCROLL_DOWN                    , NULL        ,0.0, 0 },
   { "DW_SCROLL_TOP"                    , 0, DW_SCROLL_TOP                     , NULL        ,0.0, 0 },
   { "DW_SCROLL_BOTTOM"                 , 0, DW_SCROLL_BOTTOM                  , NULL        ,0.0, 0 },
/****d* Constants/WindowAttributeFlags
 * NAME
 *   WindowAttributeFlags
 * DESCRIPTION
 *   The following is a list of the pre-defined flags that are used in a call to
 *   DW_window_set_style() to change the behaviour of widgets as explained below.
 * ATTRIBUTES
 *   The following work on all platforms:
 *    - DW_DT_LEFT               - text in a widget is left-aligned
 *    - DW_DT_RIGHT              - text in a widget is right-aligned
 *    - DW_DT_CENTER             - text in a widget is centered
 *    - DW_DT_CENTRE             - text in a widget is centred
 *    - DW_DT_VCENTER            - text in a widget is centered vertically
 *    - DW_DT_VCENTRE            - text in a widget is centred vertically
 *    - DW_BS_NOBORDER           - remove the border from a BitmapButton
 *
 *   The following attributes only have an effect on OS/2:
 *    - DW_DT_QUERYEXTENT
 *    - DW_DT_UNDERSCORE
 *    - DW_DT_STRIKEOUT
 *    - DW_DT_TEXTATTRS
 *    - DW_DT_EXTERNALLEADING
 *    - DW_DT_TOP
 *    - DW_DT_BOTTOM
 *    - DW_DT_HALFTONE
 *    - DW_DT_MNEMONIC
 *    - DW_DT_WORDBREAK
 *    - DW_DT_ERASERECT
 * SEE ALSO
 *   DW_window_set_style
 ******
 */
   { "DW_DT_LEFT"                       , 0, DW_DT_LEFT                        , NULL        ,0.0, 0 },
   { "DW_DT_RIGHT"                      , 0, DW_DT_RIGHT                       , NULL        ,0.0, 0 },
   { "DW_DT_CENTER"                     , 0, DW_DT_CENTER                      , NULL        ,0.0, 0 },
   { "DW_DT_CENTRE"                     , 0, DW_DT_CENTER                      , NULL        ,0.0, 0 },
   { "DW_DT_VCENTER"                    , 0, DW_DT_VCENTER                     , NULL        ,0.0, 0 },
   { "DW_DT_VCENTRE"                    , 0, DW_DT_VCENTER                     , NULL        ,0.0, 0 },
   { "DW_DT_QUERYEXTENT"                , 0, DW_DT_QUERYEXTENT                 , NULL        ,0.0, 0 },
   { "DW_DT_UNDERSCORE"                 , 0, DW_DT_UNDERSCORE                  , NULL        ,0.0, 0 },
   { "DW_DT_STRIKEOUT"                  , 0, DW_DT_STRIKEOUT                   , NULL        ,0.0, 0 },
   { "DW_DT_TEXTATTRS"                  , 0, DW_DT_TEXTATTRS                   , NULL        ,0.0, 0 },
   { "DW_DT_EXTERNALLEADING"            , 0, DW_DT_EXTERNALLEADING             , NULL        ,0.0, 0 },
   { "DW_DT_TOP"                        , 0, DW_DT_TOP                         , NULL        ,0.0, 0 },
   { "DW_DT_BOTTOM"                     , 0, DW_DT_BOTTOM                      , NULL        ,0.0, 0 },
   { "DW_DT_HALFTONE"                   , 0, DW_DT_HALFTONE                    , NULL        ,0.0, 0 },
   { "DW_DT_MNEMONIC"                   , 0, DW_DT_MNEMONIC                    , NULL        ,0.0, 0 },
   { "DW_DT_WORDBREAK"                  , 0, DW_DT_WORDBREAK                   , NULL        ,0.0, 0 },
   { "DW_DT_ERASERECT"                  , 0, DW_DT_ERASERECT                   , NULL        ,0.0, 0 },
   { "DW_BS_NOBORDER"                   , 0, DW_BS_NOBORDER                    , NULL        ,0.0, 0 },
/****d* Constants/WindowStyleFlags
 * NAME
 *   WindowStyleFlags
 * DESCRIPTION
 *   The following is a list of the pre-defined flags that can be used when
 *   creating a new window.
 *   Any number of values from "Style Types" can be used plus optionally
 *   one value from "Initial State Types" are logically "or"ed together using DW_or().
 * ATTRIBUTES
 *   Style Types:
 *    - DW_FCF_TITLEBAR        - display titlebar
 *    - DW_FCF_SYSMENU
 *    - DW_FCF_MENU
 *    - DW_FCF_SIZEBORDER
 *    - DW_FCF_MINBUTTON       - display minimise button
 *    - DW_FCF_MAXBUTTON       - display maximise button
 *    - DW_FCF_MINMAX
 *    - DW_FCF_DLGBORDER
 *    - DW_FCF_BORDER
 *    - DW_FCF_TASKLIST
 *    - DW_FCF_NOMOVEWITHOWNER
 *    - DW_FCF_SYSMODAL
 *    - DW_FCF_HIDEBUTTON
 *    - DW_FCF_HIDEMAX
 *    - DW_FCF_AUTOICON
 *    - DW_FCF_COMPOSITED
 *    - DW_FCF_TEXTURED        - (only relevant to Mac) brushed metal background
 *
 *   Initial State Types:
 *    - DW_FCF_MAXIMIZE        - start maximised
 *    - DW_FCF_MINIMIZE        - start minimised
 *    - DW_FCF_FULLSCREEN      - start full screen
 * SEE ALSO
 *   DW_window_new
 ******
 */
   { "DW_FCF_TITLEBAR"                  , 0, DW_FCF_TITLEBAR                   , NULL        ,0.0, 0 },
   { "DW_FCF_SYSMENU"                   , 0, DW_FCF_SYSMENU                    , NULL        ,0.0, 0 },
   { "DW_FCF_MENU"                      , 0, DW_FCF_MENU                       , NULL        ,0.0, 0 },
   { "DW_FCF_SIZEBORDER"                , 0, DW_FCF_SIZEBORDER                 , NULL        ,0.0, 0 },
   { "DW_FCF_MINBUTTON"                 , 0, DW_FCF_MINBUTTON                  , NULL        ,0.0, 0 },
   { "DW_FCF_MAXBUTTON"                 , 0, DW_FCF_MAXBUTTON                  , NULL        ,0.0, 0 },
   { "DW_FCF_MINMAX"                    , 0, DW_FCF_MINMAX                     , NULL        ,0.0, 0 },
   { "DW_FCF_DLGBORDER"                 , 0, DW_FCF_DLGBORDER                  , NULL        ,0.0, 0 },
   { "DW_FCF_BORDER"                    , 0, DW_FCF_BORDER                     , NULL        ,0.0, 0 },
   { "DW_FCF_TASKLIST"                  , 0, DW_FCF_TASKLIST                   , NULL        ,0.0, 0 },
   { "DW_FCF_NOMOVEWITHOWNER"           , 0, DW_FCF_NOMOVEWITHOWNER            , NULL        ,0.0, 0 },
   { "DW_FCF_SYSMODAL"                  , 0, DW_FCF_SYSMODAL                   , NULL        ,0.0, 0 },
   { "DW_FCF_HIDEBUTTON"                , 0, DW_FCF_HIDEBUTTON                 , NULL        ,0.0, 0 },
   { "DW_FCF_HIDEMAX"                   , 0, DW_FCF_HIDEMAX                    , NULL        ,0.0, 0 },
   { "DW_FCF_AUTOICON"                  , 0, DW_FCF_AUTOICON                   , NULL        ,0.0, 0 },
   { "DW_FCF_MAXIMIZE"                  , 0, DW_FCF_MAXIMIZE                   , NULL        ,0.0, 0 },
   { "DW_FCF_MINIMIZE"                  , 0, DW_FCF_MINIMIZE                   , NULL        ,0.0, 0 },
   { "DW_FCF_COMPOSITED"                , 0, DW_FCF_COMPOSITED                 , NULL        ,0.0, 0 },
   { "DW_FCF_TEXTURED"                  , 0, DW_FCF_TEXTURED                   , NULL        ,0.0, 0 },
   { "DW_FCF_FULLSCREEN"                , 0, DW_FCF_FULLSCREEN                 , NULL        ,0.0, 0 },
/* these are deprecated */
   { "DW_FCF_VERTSCROLL"                , 0, 0                                 , NULL        ,0.0, 0 },
   { "DW_FCF_HORZSCROLL"                , 0, 0                                 , NULL        ,0.0, 0 },
   { "DW_FCF_NOBYTEALIGN"               , 0, 0                                 , NULL        ,0.0, 0 },
   { "DW_FCF_SHELLPOSITION"             , 0, 0                                 , NULL        ,0.0, 0 },
/****d* Constants/WindowGravityValues
 * NAME
 *   WindowGravityValues
 * DESCRIPTION
 *   The following constants are used in DW_window_set_gravity() to specify the
 *   edge that a window's position is relative to.
 * ATTRIBUTES
 *   - DW_GRAV_CENTER - position of window in both horizontal and vertical
 *   - DW_GRAV_CENTRE - position of window in both horizontal and vertical
 *   - DW_GRAV_LEFT   - window x position relative to left edge of screen
 *   - DW_GRAV_RIGHT  - window x position relative to right edge of screen
 *   - DW_GRAV_TOP    - window y position relative to top edge of screen
 *   - DW_GRAV_BOTTOM - window y position relative to bottom edge of screen
 * SEE ALSO
 *   DW_window_set_gravity
 ******
 */
   { "DW_GRAV_CENTER"               , 0, DW_GRAV_CENTER                    , NULL        ,0.0, 0 },
   { "DW_GRAV_CENTRE"               , 0, DW_GRAV_CENTER                    , NULL        ,0.0, 0 },
   { "DW_GRAV_LEFT"                 , 0, DW_GRAV_LEFT                      , NULL        ,0.0, 0 },
   { "DW_GRAV_RIGHT"                , 0, DW_GRAV_RIGHT                     , NULL        ,0.0, 0 },
   { "DW_GRAV_TOP"                  , 0, DW_GRAV_TOP                       , NULL        ,0.0, 0 },
   { "DW_GRAV_BOTTOM"               , 0, DW_GRAV_BOTTOM                    , NULL        ,0.0, 0 },
/****d* Constants/ContainerColumnFlags
 * NAME
 *   ContainerColumnFlags
 * DESCRIPTION
 *   The following is a list of the pre-defined flags that can be used when
 *   creating a container widget.
 *   When used in DW_container_setup() or DW_filesystem_setup(), one value from
 *   "Column Types", one value from "Alignment" and any value from "Separators"
 *   are logically "or"ed together using DW_or().
 * ATTRIBUTES
 *   Column Types:
 *    - DW_CFA_BITMAPORICON   - the column contains an image
 *    - DW_CFA_STRING         - the column contains an unstructured string
 *    - DW_CFA_ULONG          - the column contains an integer
 *    - DW_CFA_TIME           - the column contains a time in the format Time('N')
 *    - DW_CFA_DATE           - the column contains a date in the format Date('S')
 *    - DW_CFA_STRINGANDICON  - the column contains a string and an image (not implemented)
 *
 *   Alignment:
 *    - DW_CFA_CENTER         - the column is centered
 *    - DW_CFA_CENTRE         - the column is centred
 *    - DW_CFA_LEFT           - the column is left aligned
 *    - DW_CFA_RIGHT          - the column is right aligned
 *
 *   Separators (not applicable on all platforms):
 *    - DW_CFA_HORZSEPARATOR  - the column has a cell border on the top and bottom
 *    - DW_CFA_SEPARATOR      - the column has a cell border on the left and right
 * SEE ALSO
 *   DW_container_setup, DW_filesystem_setup
 ******
 */
   { "DW_CFA_BITMAPORICON"              , 0, DW_CFA_BITMAPORICON               , NULL        ,0.0, 0 },
   { "DW_CFA_STRING"                    , 0, DW_CFA_STRING                     , NULL        ,0.0, 0 },
   { "DW_CFA_ULONG"                     , 0, DW_CFA_ULONG                      , NULL        ,0.0, 0 },
   { "DW_CFA_TIME"                      , 0, DW_CFA_TIME                       , NULL        ,0.0, 0 },
   { "DW_CFA_DATE"                      , 0, DW_CFA_DATE                       , NULL        ,0.0, 0 },
   { "DW_CFA_CENTER"                    , 0, DW_CFA_CENTER                     , NULL        ,0.0, 0 },
   { "DW_CFA_CENTRE"                    , 0, DW_CFA_CENTER                     , NULL        ,0.0, 0 },
   { "DW_CFA_LEFT"                      , 0, DW_CFA_LEFT                       , NULL        ,0.0, 0 },
   { "DW_CFA_RIGHT"                     , 0, DW_CFA_RIGHT                      , NULL        ,0.0, 0 },
   { "DW_CFA_STRINGANDICON"             , 0, DW_CFA_STRINGANDICON              , NULL        ,0.0, 0 },
   { "DW_CFA_HORZSEPARATOR"             , 0, DW_CFA_HORZSEPARATOR              , NULL        ,0.0, 0 },
   { "DW_CFA_SEPARATOR"                 , 0, DW_CFA_SEPARATOR                  , NULL        ,0.0, 0 },
/****d* Constants/ContainerQueryFlags
 * NAME
 *   ContainerQueryFlags
 * DESCRIPTION
 *   The following is a list of the pre-defined flags that can be used when
 *   querying a container widget.
 * ATTRIBUTES
 *   - DW_CRA_ALL       - all rows are queried
 *   - DW_CRA_SELECTED  - only rows selected are queried
 *   - DW_CRA_CURSORED  - only the row(s) currently have focus; ie. highlighted
 * SEE ALSO
 *   DW_container_query_start, DW_container_query_next
 ******
 */
   { "DW_CRA_ALL"                       , 0, 0                                 , NULL        ,0.0, 0 },
   { "DW_CRA_SELECTED"                  , 0, DW_CRA_SELECTED                   , NULL        ,0.0, 0 },
   { "DW_CRA_CURSORED"                  , 0, DW_CRA_CURSORED                   , NULL        ,0.0, 0 },
/****d* Constants/MLESearchFlags
 * NAME
 *   MLESearchFlags
 * DESCRIPTION
 *   The following is a list of the pre-defined flags that can be used in
 *   the SearchFlags argument in DW_mle_search().
 * ATTRIBUTES
 *   - DW_MLE_CASESENSITIVE - search done respecting case
 * SEE ALSO
 *   DW_mle_search
 ******
 */
   { "DW_MLE_CASESENSITIVE"             , 0, DW_MLE_CASESENSITIVE              , NULL        ,0.0, 0 },
/****d* Constants/MLEEditableFlags
 * NAME
 *   MLEEditableFlags
 * DESCRIPTION
 *   The following is a list of the pre-defined flags that can be used in
 *   the EditableFlags argument in DW_mle_set_editable().
 * ATTRIBUTES
 *   - DW_EDITABLE          - MLE is editable
 *   - DW_READONLY          - MLE is not editable; ie readonly
 * SEE ALSO
 *   DW_mle_set_editable
 ******
 */
   { "DW_EDITABLE"                  , 0, TRUE                              , NULL        ,0.0, 0 },
   { "DW_READONLY"                  , 0, FALSE                             , NULL        ,0.0, 0 },
/****d* Constants/MLEWordWrapFlags
 * NAME
 *   MLEWordWrapFlags
 * DESCRIPTION
 *   The following is a list of the pre-defined flags that can be used in
 *   the WordWrapFlags argument in DW_mle_set_word_wrap().
 * ATTRIBUTES
 *   - DW_WORD_WRAP         - words are wrapped in the MLE
 *   - DW_DONT_WORD_WRAP    - no word wrapping occurs in the MLE
 * SEE ALSO
 *   DW_mle_set_word_wrap
 ******
 */
   { "DW_WORD_WRAP"                 , 0, TRUE                              , NULL        ,0.0, 0 },
   { "DW_DONT_WORD_WRAP"            , 0, FALSE                             , NULL        ,0.0, 0 },
/****d* Constants/PointerTypes
 * NAME
 *   PointerTypes
 * DESCRIPTION
 *   The following is a list of the pre-defined mouse pointer types used
 *   in the Pointer argument in DW_window_set_pointer().
 * ATTRIBUTES
 *   - DW_POINTER_DEFAULT   - the default pointer shape
 *   - DW_POINTER_CLOCK     - a clock pointer
 *   - DW_POINTER_ARROW     - an arrow pointing to the top left
 *   - DW_POINTER_QUESTION  - a question mark pointer
 * SEE ALSO
 *   DW_window_set_pointer
 ******
 */
   { "DW_POINTER_DEFAULT"               , 0, DW_POINTER_DEFAULT                , NULL        ,0.0, 0 },
   { "DW_POINTER_ARROW"                 , 0, DW_POINTER_ARROW                  , NULL        ,0.0, 0 },
   { "DW_POINTER_CLOCK"                 , 0, DW_POINTER_CLOCK                  , NULL        ,0.0, 0 },
   { "DW_POINTER_QUESTION"              , 0, DW_POINTER_QUESTION               , NULL        ,0.0, 0 },
/****d* Constants/ButtonMasks
 * NAME
 *   ButtonMasks
 * DESCRIPTION
 *   The following is a list of the pre-defined mouse button mask types returned
 *   in the Button argument in a MotionNotify callback.
 * ATTRIBUTES
 *   - DW_BUTTON1_MASK      - button 1 is being pressed
 *   - DW_BUTTON2_MASK      - button 2 is being pressed
 *   - DW_BUTTON3_MASK      - button 3 is being pressed
 * SEE ALSO
 *   MotionNotifyEventCallback
 ******
 */
   { "DW_BUTTON1_MASK"                  , 0, DW_BUTTON1_MASK                   , NULL        ,0.0, 0 },
   { "DW_BUTTON2_MASK"                  , 0, DW_BUTTON2_MASK                   , NULL        ,0.0, 0 },
   { "DW_BUTTON3_MASK"                  , 0, DW_BUTTON3_MASK                   , NULL        ,0.0, 0 },
/****d* Constants/Colours
 * NAME
 *   Colours
 * DESCRIPTION
 *   The following is a list of the pre-defined colours known to RexxDW.
 *   These colour constants are used in functions that require pre-defined
 *   colours.
 * ATTRIBUTES
 *   - DW_CLR_BLACK
 *   - DW_CLR_DARKRED
 *   - DW_CLR_DARKGREEN
 *   - DW_CLR_BROWN
 *   - DW_CLR_DARKBLUE
 *   - DW_CLR_DARKPINK
 *   - DW_CLR_DARKCYAN
 *   - DW_CLR_PALEGRAY
 *   - DW_CLR_DARKGRAY
 *   - DW_CLR_RED
 *   - DW_CLR_GREEN
 *   - DW_CLR_YELLOW
 *   - DW_CLR_BLUE
 *   - DW_CLR_PINK
 *   - DW_CLR_CYAN
 *   - DW_CLR_WHITE
 *   - DW_CLR_DEFAULT
 *   - DW_CLR_TRANSPARENT (use on background)
 * SEE ALSO
 *   DW_color_background_set, DW_color_foreground_set, DW_color_choose
 ******
 */
   { "DW_CLR_BLACK"                     , 0, DW_CLR_BLACK                      , NULL        ,0.0, 0 },
   { "DW_CLR_DARKRED"                   , 0, DW_CLR_DARKRED                    , NULL        ,0.0, 0 },
   { "DW_CLR_DARKGREEN"                 , 0, DW_CLR_DARKGREEN                  , NULL        ,0.0, 0 },
   { "DW_CLR_BROWN"                     , 0, DW_CLR_BROWN                      , NULL        ,0.0, 0 },
   { "DW_CLR_DARKBLUE"                  , 0, DW_CLR_DARKBLUE                   , NULL        ,0.0, 0 },
   { "DW_CLR_DARKPINK"                  , 0, DW_CLR_DARKPINK                   , NULL        ,0.0, 0 },
   { "DW_CLR_DARKCYAN"                  , 0, DW_CLR_DARKCYAN                   , NULL        ,0.0, 0 },
   { "DW_CLR_PALEGRAY"                  , 0, DW_CLR_PALEGRAY                   , NULL        ,0.0, 0 },
   { "DW_CLR_DARKGRAY"                  , 0, DW_CLR_DARKGRAY                   , NULL        ,0.0, 0 },
   { "DW_CLR_RED"                       , 0, DW_CLR_RED                        , NULL        ,0.0, 0 },
   { "DW_CLR_GREEN"                     , 0, DW_CLR_GREEN                      , NULL        ,0.0, 0 },
   { "DW_CLR_YELLOW"                    , 0, DW_CLR_YELLOW                     , NULL        ,0.0, 0 },
   { "DW_CLR_BLUE"                      , 0, DW_CLR_BLUE                       , NULL        ,0.0, 0 },
   { "DW_CLR_PINK"                      , 0, DW_CLR_PINK                       , NULL        ,0.0, 0 },
   { "DW_CLR_CYAN"                      , 0, DW_CLR_CYAN                       , NULL        ,0.0, 0 },
   { "DW_CLR_WHITE"                     , 0, DW_CLR_WHITE                      , NULL        ,0.0, 0 },
   { "DW_CLR_DEFAULT"                   , 0, DW_CLR_DEFAULT                    , NULL        ,0.0, 0 },
   { "DW_CLR_TRANSPARENT"               , 0, DW_RGB_TRANSPARENT                , NULL        ,0.0, 0 },
   /* events */
   { "DW_CONFIGURE_EVENT"           , 0, REXXDW_CONFIGURE_EVENT_CB         , NULL        ,0.0, 0 },
   { "DW_KEY_PRESS_EVENT"           , 0, REXXDW_KEY_PRESS_EVENT_CB         , NULL        ,0.0, 0 },
   { "DW_BUTTON_PRESS_EVENT"        , 0, REXXDW_BUTTON_PRESS_EVENT_CB      , NULL        ,0.0, 0 },
   { "DW_BUTTON_RELEASE_EVENT"      , 0, REXXDW_BUTTON_RELEASE_EVENT_CB    , NULL        ,0.0, 0 },
   { "DW_MOTION_NOTIFY_EVENT"       , 0, REXXDW_MOTION_NOTIFY_EVENT_CB     , NULL        ,0.0, 0 },
   { "DW_DELETE_EVENT"              , 0, REXXDW_DELETE_EVENT_CB            , NULL        ,0.0, 0 },
   { "DW_EXPOSE_EVENT"              , 0, REXXDW_EXPOSE_EVENT_CB            , NULL        ,0.0, 0 },
   { "DW_CLICKED_EVENT"             , 0, REXXDW_CLICKED_EVENT_CB           , NULL        ,0.0, 0 },
   { "DW_ITEM_ENTER_EVENT"          , 0, REXXDW_ITEM_ENTER_EVENT_CB        , NULL        ,0.0, 0 },
   { "DW_ITEM_CONTEXT_EVENT"        , 0, REXXDW_ITEM_CONTEXT_EVENT_CB      , NULL        ,0.0, 0 },
   { "DW_LIST_SELECT_EVENT"         , 0, REXXDW_LIST_SELECT_EVENT_CB       , NULL        ,0.0, 0 },
   { "DW_ITEM_SELECT_EVENT"         , 0, REXXDW_ITEM_SELECT_EVENT_CB       , NULL        ,0.0, 0 },
   { "DW_SET_FOCUS_EVENT"           , 0, REXXDW_SET_FOCUS_EVENT_CB         , NULL        ,0.0, 0 },
   { "DW_VALUE_CHANGED_EVENT"       , 0, REXXDW_VALUE_CHANGED_EVENT_CB     , NULL        ,0.0, 0 },
   { "DW_SWITCH_PAGE_EVENT"         , 0, REXXDW_SWITCH_PAGE_EVENT_CB       , NULL        ,0.0, 0 },
   { "DW_COLUMN_CLICK_EVENT"        , 0, REXXDW_COLUMN_CLICK_EVENT_CB      , NULL        ,0.0, 0 },
/****d* Constants/WidgetOrientation
 * NAME
 *   WidgetOrientation
 * DESCRIPTION
 *   The following constants are used in widgets that require an orientation
 *   to be specified.
 * ATTRIBUTES
 *   - DW_VERT                - vertical orientation
 *   - DW_HORZ                - horizontal orientation
 * SEE ALSO
 *   DW_box_new, DW_groupbox_new, DW_splitbar_new, DW_scrollbar_new
 ******
 */
   { "DW_VERT"                      , 0, TRUE                              , NULL        ,0.0, 0 },
   { "DW_HORZ"                      , 0, FALSE                             , NULL        ,0.0, 0 },
/****d* Constants/HTMLActions
 * NAME
 *   HTMLActions
 * DESCRIPTION
 *   The following constants are used with DW_html_action to take actions
 *   on an embedded HTML widget.
 * ATTRIBUTES
 *   - DW_HTML_GOBACK         - go back to the previously viewed page
 *   - DW_HTML_GOFORWARD      - go forward to the previously viewed page
 *   - DW_HTML_GOHOME         - go to HOME page
 *   - DW_HTML_RELOAD         - reload the current page
 *   - DW_HTML_STOP           - stop loading the current page
 * SEE ALSO
 *   DW_html_action
 ******
 */
   { "DW_HTML_GOBACK"               , 0, DW_HTML_GOBACK                    , NULL        ,0.0, 0 },
   { "DW_HTML_GOFORWARD"            , 0, DW_HTML_GOFORWARD                 , NULL        ,0.0, 0 },
   { "DW_HTML_GOHOME"               , 0, DW_HTML_GOHOME                    , NULL        ,0.0, 0 },
   { "DW_HTML_SEARCH"               , 0, DW_HTML_SEARCH                    , NULL        ,0.0, 0 },
   { "DW_HTML_RELOAD"               , 0, DW_HTML_RELOAD                    , NULL        ,0.0, 0 },
   { "DW_HTML_STOP"                 , 0, DW_HTML_STOP                      , NULL        ,0.0, 0 },
/****d* Constants/NotebookTabLocation
 * NAME
 *   NotebookTabLocation
 * DESCRIPTION
 *   The following constants are used in the TabLocation argument in a call
 *   to DW_notebook_new() to specify whether notebook tabs are located at the
 *   top or bottom of the window.
 * ATTRIBUTES
 *   - DW_TAB_TO_TOP          - notebook tabs are displayed across the top
 *   - DW_TAB_TO_BOTTOM       - notebook tabs are displayed across the bottom
 * SEE ALSO
 *   DW_notebook_new
 ******
 */
   { "DW_TAB_TO_TOP"                , 0, TRUE                              , NULL        ,0.0, 0 },
   { "DW_TAB_TO_BOTTOM"             , 0, FALSE                             , NULL        ,0.0, 0 },
/****d* Constants/NotebookPageLocation
 * NAME
 *   NotebookPageLocation
 * DESCRIPTION
 *   The following constants are used in the PageLocation argument in a call
 *   to DW_notebook_page_new() to specify if the page is created in front
 *   or behind other pages.
 * ATTRIBUTES
 *   - DW_PAGE_TO_FRONT       - page is created in front of other pages
 *   - DW_PAGE_TO_BACK        - page is created behind other pages
 * SEE ALSO
 *   DW_notebook_page_new
 ******
 */
   { "DW_PAGE_TO_FRONT"             , 0, TRUE                              , NULL        ,0.0, 0 },
   { "DW_PAGE_TO_BACK"              , 0, FALSE                             , NULL        ,0.0, 0 },
/****d* Constants/MenuConstants
 * NAME
 *   MenuConstants
 * DESCRIPTION
 *   The following constants are used when appending a menu item to a menu.
 * ATTRIBUTES
 *   - Title agument:
 *      - DW_MENU_SEPARATOR      - the string to indicate the menu item is a separator
 *   - Id agument:
 *      - DW_MENU_POPUP          - indicates the item is part of a popup menu item
 *      - DW_MENU_AUTO           - automatically assigns a menu id
 *   - End agument:
 *      - DW_MENU_START          - add the item at the start of the menu items
 *      - DW_MENU_END            - add the item at the end of the menu items
 *   - Check agument:
 *      - DW_MENU_CHECKABLE      - the menu item is checkable
 *      - DW_MENU_NOT_CHECKABLE  - the menu item is not checkable
 *   - Submenu agument:
 *      - DW_MENU_NOMENU         - indicates there is no submenu for this menu item
 * SEE ALSO
 *   DW_menu_append_item
 ******
 */
   { "DW_MENU_END"                  , 0, TRUE                              , NULL        ,0.0, 0 },
   { "DW_MENU_START"                , 0, FALSE                             , NULL        ,0.0, 0 },
   { "DW_MENU_CHECKABLE"            , 0, TRUE                              , NULL        ,0.0, 0 },
   { "DW_MENU_NOT_CHECKABLE"        , 0, FALSE                             , NULL        ,0.0, 0 },
   { "DW_MENU_SEPARATOR"            , 1, 0                                 , DW_MENU_SEPARATOR    ,0.0, 0 },
   { "DW_MENU_AUTO"                 , 0, DW_MENU_AUTO                      , NULL        ,0.0, 0 },
   { "DW_MENU_POPUP"                , 0, DW_MENU_POPUP                     , NULL        ,0.0, 0 },
   { "DW_MENU_NOMENU"               , 0, DW_NOMENU                         , NULL        ,0.0, 0 },
/****d* Constants/MenuItemConstants
 * NAME
 *   MenuItemConstants
 * DESCRIPTION
 *   The following constants are used when setting the state of a menu item.
 * ATTRIBUTES
 *   - DW_MENU_CHECKED        - sets the state of the checkable menu item to checked
 *   - DW_MENU_UNCHECKED      - sets the state of the checkable menu item to unchecked
 *   - DW_MENU_ENABLED        - enables the menu item
 *   - DW_MENU_DISABLED       - disables the menu item
 * SEE ALSO
 *   DW_menu_item_set_state, DW_or
 * NOTES
 *   Any of the above attributes can be ORed together.
 ******
 */
   { "DW_MENU_CHECKED"              , 0, DW_MIS_CHECKED                    , NULL        ,0.0, 0 },
   { "DW_MENU_UNCHECKED"            , 0, DW_MIS_UNCHECKED                  , NULL        ,0.0, 0 },
   { "DW_MENU_ENABLED"              , 0, DW_MIS_ENABLED                    , NULL        ,0.0, 0 },
   { "DW_MENU_DISABLED"             , 0, DW_MIS_DISABLED                   , NULL        ,0.0, 0 },
/****d* Constants/BoxExpansionFlags
 * NAME
 *   BoxExpansionFlags
 * DESCRIPTION
 *   The following constants are used in the box packing functions to
 *   specify whether the box expand in a horizontal or vertical direction.
 * ATTRIBUTES
 *   - DW_EXPAND_HORZ         - expand the box horizontally
 *   - DW_DONT_EXPAND_HORZ    - don't expand the box horizontal
 *   - DW_EXPAND_VERT         - expand the box vertically
 *   - DW_DONT_EXPAND_VERT    - don't expand the box vertically
 * SEE ALSO
 *   DW_box_pack_start, DW_box_pack_end, DW_box_pack_at_index
 ******
 */
   { "DW_EXPAND_HORZ"               , 0, TRUE                              , NULL        ,0.0, 0 },
   { "DW_DONT_EXPAND_HORZ"          , 0, FALSE                             , NULL        ,0.0, 0 },
   { "DW_EXPAND_VERT"               , 0, TRUE                              , NULL        ,0.0, 0 },
   { "DW_DONT_EXPAND_VERT"          , 0, FALSE                             , NULL        ,0.0, 0 },
/****d* Constants/BoxSizeValues
 * NAME
 *   BoxSizeValues
 * DESCRIPTION
 *   The following constant is used in the box packing functions where
 *   a pixel size would normally be specified. This constant value indicates that
 *   Rexx/DW should automatically size the widget in the direction indicated.
 * ATTRIBUTES
 *   - DW_SIZE_AUTO - size the box automatically in the indicated direction
 * SEE ALSO
 *   DW_box_pack_start, DW_box_pack_end
 ******
 */
   { "DW_SIZE_AUTO"                 , 0, -1                                , NULL        ,0.0, 0 },
/****d* Constants/GroupBoxFontFlags
 * NAME
 *   GroupBoxFontFlags
 * DESCRIPTION
 *   The following constants can be used when creating a groupbox to indicate
 *   whether the font used for displaying the title text is normal, bold and/or
 *   italic. Use DW_or to specify bold and italic.
 *   This flag is optional and defaults to DW_FONT_NORMAL.
 * ATTRIBUTES
 *   - DW_FONT_NORMAL - use the default font
 *   - DW_FONT_BOLD   - use the default font with bold attribute
 *   - DW_FONT_ITALIC - use the default font with italic attribute
 * SEE ALSO
 *   DW_groupbox_new
 ******
 */
   { "DW_FONT_NORMAL"               , 0, DW_FONT_NORMAL                    , NULL        ,0.0, 0 },
   { "DW_FONT_BOLD"                 , 0, DW_FONT_BOLD                      , NULL        ,0.0, 0 },
   { "DW_FONT_ITALIC"               , 0, DW_FONT_ITALIC                    , NULL        ,0.0, 0 },
/****d* Constants/ListboxSelection
 * NAME
 *   ListboxSelection
 * DESCRIPTION
 *   The following constants are used in the Selection argument to a call to
 *   DW_listbox_new().
 * ATTRIBUTES
 *   - DW_LB_SINGLE_SELECTION    - indicates that only 1 row in the listbox is selectable
 *   - DW_LB_MULTIPLE_SELECTION  - indicates that multiple rows in the listbox are selectable
 * SEE ALSO
 *   DW_listbox_new
 ******
 */
   { "DW_LB_SINGLE_SELECTION"           , 0, 0                                 , NULL        ,0.0, 0 },
   { "DW_LB_MULTIPLE_SELECTION"         , 0, DW_LS_MULTIPLESEL                 , NULL        ,0.0, 0 },
/****d* Constants/ListboxSelected
 * NAME
 *   ListboxSelected
 * DESCRIPTION
 *   The following is a list of the pre-defined flags that can be used when
 *   setting the State of a listbox or combobox.
 * ATTRIBUTES
 *   - DW_LB_SELECTED      - set the item to be selected
 *   - DW_LB_UNSELECTED    - set the item to NOT be selected
 * SEE ALSO
 *   DW_listbox_select
 ******
 */
   { "DW_LB_SELECTED"               , 0, TRUE                              , NULL        ,0.0, 0 },
   { "DW_LB_UNSELECTED"             , 0, FALSE                             , NULL        ,0.0, 0 },
/****d* Constants/ListboxReturn
 * NAME
 *   ListboxReturn
 * DESCRIPTION
 *   The following is the return value from DW_listbox_selected() or
 *   DW_listbox_selected_multi() if no item in the listbox has been selected.
 * ATTRIBUTES
 *   - DW_LB_NONE          - no listbox item selected
 * SEE ALSO
 *   DW_listbox_selected(), DW_listbox_selected_multi()
 ******
 */
   { "DW_LB_NONE"                   , 0, DW_LIT_NONE                       , NULL        ,0.0, 0 },
/****d* Constants/WidgetChecked
 * NAME
 *   WidgetChecked
 * DESCRIPTION
 *   The following constants are used to indicate the state of a widget
 *   that can have a boolean state.
 * ATTRIBUTES
 *   - DW_CHECKED             - widget is "on" or set
 *   - DW_UNCHECKED           - widget is "off" or unset
 * SEE ALSO
 *   DW_checkbox_set, DW_radiobutton_set
 * NOTES
 *   For
 ******
 */
   { "DW_CHECKED"                   , 0, TRUE                              , NULL        ,0.0, 0 },
   { "DW_UNCHECKED"                 , 0, FALSE                             , NULL        ,0.0, 0 },
/****d* Constants/ContainerClearFlags
 * NAME
 *   ContainerClearFlags
 * DESCRIPTION
 *   The following constants are used in the Redraw argument to a call to
 *   DW_container_clear().
 * ATTRIBUTES
 *   - DW_REDRAW              - redraw the cleared container
 *   - DW_DONT_REDRAW         - don't redraw the cleared container
 * SEE ALSO
 *   DW_container_new
 ******
 */
   { "DW_REDRAW"                    , 0, TRUE                              , NULL        ,0.0, 0 },
   { "DW_DONT_REDRAW"               , 0, FALSE                             , NULL        ,0.0, 0 },
/****d* Constants/ContainerSelection
 * NAME
 *   ContainerSelection
 * DESCRIPTION
 *   The following constants are used in the Selection argument to a call to
 *   DW_container_new().
 * ATTRIBUTES
 *   - DW_SINGLE_SELECTION    - indicates that only 1 row in the container is selectable
 *   - DW_MULTIPLE_SELECTION  - indicates that multiple rows in the container are selectable
 * SEE ALSO
 *   DW_container_new
 ******
 */
   { "DW_MULTIPLE_SELECTION"        , 0, TRUE                              , NULL        ,0.0, 0 },
   { "DW_SINGLE_SELECTION"          , 0, FALSE                             , NULL        ,0.0, 0 },
/****d* Constants/DrawingFlags
 * NAME
 *   DrawingFlags
 * DESCRIPTION
 *   The following constants are used in the Flags argument to a call to
 *   DW_draw_rect(), DW_draw_arc(), or DW_draw_polygon.
 * ATTRIBUTES
 *   - DW_DRAW_DEFAULT - draws an outline shape 1 pixel in width
 *   - DW_DRAW_FILL    - draws a filled shape
 *   - DW_DRAW_FULL    - draws a circle when DW_draw_arc() is called
 *   - DW_DRAW_NOAA    - turns off anti-aliasing
 *   - DW_FILL         - draws a filled shape (deprecated)
 *   - DW_DONT_FILL    - draws an outline shape 1 pixel in width (deprecated)
 * SEE ALSO
 *   DW_draw_rect, DW_draw_arc, DW_draw_polygon
 ******
 */
   { "DW_DRAW_DEFAULT"              , 0, DW_DRAW_DEFAULT                   , NULL        ,0.0, 0 },
   { "DW_DRAW_FILL"                 , 0, DW_DRAW_FILL                      , NULL        ,0.0, 0 },
   { "DW_DRAW_FULL"                 , 0, DW_DRAW_FULL                      , NULL        ,0.0, 0 },
   { "DW_DRAW_NOAA"                 , 0, DW_DRAW_NOAA                      , NULL        ,0.0, 0 },
   { "DW_FILL"                      , 0, TRUE                              , NULL        ,0.0, 0 },
   { "DW_DONT_FILL"                 , 0, FALSE                             , NULL        ,0.0, 0 },
/****d* Constants/MessageboxFlags
 * NAME
 *   MessageboxFlags
 * DESCRIPTION
 *   The following constants are used in the Flags argument to a call to
 *   DW_messagebox(). Logically "or" one value from Buttons and one value from Icons
 * ATTRIBUTES
 *   Buttons:
 *    - DW_MB_OK
 *    - DW_MB_OKCANCEL
 *    - DW_MB_YESNO
 *    - DW_MB_YESNOCANCEL
 *
 *   Icons:
 *    - DW_MB_WARNING
 *    - DW_MB_ERROR
 *    - DW_MB_INFORMATION
 *    - DW_MB_QUESTION
 * SEE ALSO
 *   DW_messagebox
 ******
 */
   { "DW_MB_OK"                     , 0, DW_MB_OK                          , NULL        ,0.0, 0 },
   { "DW_MB_OKCANCEL"               , 0, DW_MB_OKCANCEL                    , NULL        ,0.0, 0 },
   { "DW_MB_YESNO"                  , 0, DW_MB_YESNO                       , NULL        ,0.0, 0 },
   { "DW_MB_YESNOCANCEL"            , 0, DW_MB_YESNOCANCEL                 , NULL        ,0.0, 0 },
   { "DW_MB_WARNING"                , 0, DW_MB_WARNING                     , NULL        ,0.0, 0 },
   { "DW_MB_ERROR"                  , 0, DW_MB_ERROR                       , NULL        ,0.0, 0 },
   { "DW_MB_INFORMATION"            , 0, DW_MB_INFORMATION                 , NULL        ,0.0, 0 },
   { "DW_MB_QUESTION"               , 0, DW_MB_QUESTION                    , NULL        ,0.0, 0 },
/****d* Constants/MessageboxResults
 * NAME
 *   MessageboxResults
 * DESCRIPTION
 *   The following constants are returned from a call to DW_messagebox().
 * ATTRIBUTES
 *   - DW_MB_RETURN_OK
 *   - DW_MB_RETURN_YES
 *   - DW_MB_RETURN_NO
 *   - DW_MB_RETURN_CANCEL
 * SEE ALSO
 *   DW_messagebox
 ******
 */
   { "DW_MB_RETURN_OK"              , 0, DW_MB_RETURN_OK                   , NULL        ,0.0, 0 },
   { "DW_MB_RETURN_YES"             , 0, DW_MB_RETURN_YES                  , NULL        ,0.0, 0 },
   { "DW_MB_RETURN_NO"              , 0, DW_MB_RETURN_NO                   , NULL        ,0.0, 0 },
   { "DW_MB_RETURN_CANCEL"          , 0, DW_MB_RETURN_CANCEL               , NULL        ,0.0, 0 },
/****d* Constants/VirtualKeys
 * NAME
 *   VirtualKeys
 * DESCRIPTION
 *   The following is a list of the mnemonic key values known to RexxDW.
 *   These keys are returned in a KeyPressEvent Callback, in the VirtualKey
 *   argument.
 * ATTRIBUTES
 *   - DW_VK_CANCEL
 *   - DW_VK_BACK
 *   - DW_VK_TAB
 *   - DW_VK_CLEAR
 *   - DW_VK_RETURN
 *   - DW_VK_MENU
 *   - DW_VK_PAUSE
 *   - DW_VK_CAPITAL
 *   - DW_VK_ESCAPE
 *   - DW_VK_SPACE
 *   - DW_VK_PRIOR
 *   - DW_VK_NEXT
 *   - DW_VK_END
 *   - DW_VK_HOME
 *   - DW_VK_LEFT
 *   - DW_VK_UP
 *   - DW_VK_RIGHT
 *   - DW_VK_DOWN
 *   - DW_VK_SELECT
 *   - DW_VK_PRINT
 *   - DW_VK_EXECUTE
 *   - DW_VK_SNAPSHOT
 *   - DW_VK_INSERT
 *   - DW_VK_DELETE
 *   - DW_VK_HELP
 *   - DW_VK_LWIN
 *   - DW_VK_RWIN
 *   - DW_VK_NUMPAD0
 *   - DW_VK_NUMPAD1
 *   - DW_VK_NUMPAD2
 *   - DW_VK_NUMPAD3
 *   - DW_VK_NUMPAD4
 *   - DW_VK_NUMPAD5
 *   - DW_VK_NUMPAD6
 *   - DW_VK_NUMPAD7
 *   - DW_VK_NUMPAD8
 *   - DW_VK_NUMPAD9
 *   - DW_VK_MULTIPLY
 *   - DW_VK_ADD
 *   - DW_VK_SEPARATOR
 *   - DW_VK_SUBTRACT
 *   - DW_VK_DECIMAL
 *   - DW_VK_DIVIDE
 *   - DW_VK_F1
 *   - DW_VK_F2
 *   - DW_VK_F3
 *   - DW_VK_F4
 *   - DW_VK_F5
 *   - DW_VK_F6
 *   - DW_VK_F7
 *   - DW_VK_F8
 *   - DW_VK_F9
 *   - DW_VK_F10
 *   - DW_VK_F11
 *   - DW_VK_F12
 *   - DW_VK_F13
 *   - DW_VK_F14
 *   - DW_VK_F15
 *   - DW_VK_F16
 *   - DW_VK_F17
 *   - DW_VK_F18
 *   - DW_VK_F19
 *   - DW_VK_F20
 *   - DW_VK_F21
 *   - DW_VK_F22
 *   - DW_VK_F23
 *   - DW_VK_F24
 *   - DW_VK_NUMLOCK
 *   - DW_VK_SCROLL
 *   - DW_VK_LSHIFT
 *   - DW_VK_RSHIFT
 *   - DW_VK_LCONTROL
 *   - DW_VK_RCONTROL
 *   - DW_VK_LMENU
 *   - DW_VK_RMENU
 * SEE ALSO
 *   KeyPressEventCallback
 ******
 */
   { "DW_VK_CANCEL"                 , 0, VK_CANCEL                         , NULL        ,0.0, 0 },
   { "DW_VK_BACK"                   , 0, VK_BACK                           , NULL        ,0.0, 0 },
   { "DW_VK_TAB"                    , 0, VK_TAB                            , NULL        ,0.0, 0 },
   { "DW_VK_CLEAR"                  , 0, VK_CLEAR                          , NULL        ,0.0, 0 },
   { "DW_VK_RETURN"                 , 0, VK_RETURN                         , NULL        ,0.0, 0 },
   { "DW_VK_MENU"                   , 0, VK_MENU                           , NULL        ,0.0, 0 },
   { "DW_VK_PAUSE"                  , 0, VK_PAUSE                          , NULL        ,0.0, 0 },
   { "DW_VK_CAPITAL"                , 0, VK_CAPITAL                        , NULL        ,0.0, 0 },
   { "DW_VK_ESCAPE"                 , 0, VK_ESCAPE                         , NULL        ,0.0, 0 },
   { "DW_VK_SPACE"                  , 0, VK_SPACE                          , NULL        ,0.0, 0 },
   { "DW_VK_PRIOR"                  , 0, VK_PRIOR                          , NULL        ,0.0, 0 },
   { "DW_VK_NEXT"                   , 0, VK_NEXT                           , NULL        ,0.0, 0 },
   { "DW_VK_END"                    , 0, VK_END                            , NULL        ,0.0, 0 },
   { "DW_VK_HOME"                   , 0, VK_HOME                           , NULL        ,0.0, 0 },
   { "DW_VK_LEFT"                   , 0, VK_LEFT                           , NULL        ,0.0, 0 },
   { "DW_VK_UP"                     , 0, VK_UP                             , NULL        ,0.0, 0 },
   { "DW_VK_RIGHT"                  , 0, VK_RIGHT                          , NULL        ,0.0, 0 },
   { "DW_VK_DOWN"                   , 0, VK_DOWN                           , NULL        ,0.0, 0 },
   { "DW_VK_SELECT"                 , 0, VK_SELECT                         , NULL        ,0.0, 0 },
   { "DW_VK_PRINT"                  , 0, VK_PRINT                          , NULL        ,0.0, 0 },
   { "DW_VK_EXECUTE"                , 0, VK_EXECUTE                        , NULL        ,0.0, 0 },
   { "DW_VK_SNAPSHOT"               , 0, VK_SNAPSHOT                       , NULL        ,0.0, 0 },
   { "DW_VK_INSERT"                 , 0, VK_INSERT                         , NULL        ,0.0, 0 },
   { "DW_VK_DELETE"                 , 0, VK_DELETE                         , NULL        ,0.0, 0 },
   { "DW_VK_HELP"                   , 0, VK_HELP                           , NULL        ,0.0, 0 },
   { "DW_VK_LWIN"                   , 0, VK_LWIN                           , NULL        ,0.0, 0 },
   { "DW_VK_RWIN"                   , 0, VK_RWIN                           , NULL        ,0.0, 0 },
   { "DW_VK_NUMPAD0"                , 0, VK_NUMPAD0                        , NULL        ,0.0, 0 },
   { "DW_VK_NUMPAD1"                , 0, VK_NUMPAD1                        , NULL        ,0.0, 0 },
   { "DW_VK_NUMPAD2"                , 0, VK_NUMPAD2                        , NULL        ,0.0, 0 },
   { "DW_VK_NUMPAD3"                , 0, VK_NUMPAD3                        , NULL        ,0.0, 0 },
   { "DW_VK_NUMPAD4"                , 0, VK_NUMPAD4                        , NULL        ,0.0, 0 },
   { "DW_VK_NUMPAD5"                , 0, VK_NUMPAD5                        , NULL        ,0.0, 0 },
   { "DW_VK_NUMPAD6"                , 0, VK_NUMPAD6                        , NULL        ,0.0, 0 },
   { "DW_VK_NUMPAD7"                , 0, VK_NUMPAD7                        , NULL        ,0.0, 0 },
   { "DW_VK_NUMPAD8"                , 0, VK_NUMPAD8                        , NULL        ,0.0, 0 },
   { "DW_VK_NUMPAD9"                , 0, VK_NUMPAD9                        , NULL        ,0.0, 0 },
   { "DW_VK_MULTIPLY"               , 0, VK_MULTIPLY                       , NULL        ,0.0, 0 },
   { "DW_VK_ADD"                    , 0, VK_ADD                            , NULL        ,0.0, 0 },
   { "DW_VK_SEPARATOR"              , 0, VK_SEPARATOR                      , NULL        ,0.0, 0 },
   { "DW_VK_SUBTRACT"               , 0, VK_SUBTRACT                       , NULL        ,0.0, 0 },
   { "DW_VK_DECIMAL"                , 0, VK_DECIMAL                        , NULL        ,0.0, 0 },
   { "DW_VK_DIVIDE"                 , 0, VK_DIVIDE                         , NULL        ,0.0, 0 },
   { "DW_VK_F1"                     , 0, VK_F1                             , NULL        ,0.0, 0 },
   { "DW_VK_F2"                     , 0, VK_F2                             , NULL        ,0.0, 0 },
   { "DW_VK_F3"                     , 0, VK_F3                             , NULL        ,0.0, 0 },
   { "DW_VK_F4"                     , 0, VK_F4                             , NULL        ,0.0, 0 },
   { "DW_VK_F5"                     , 0, VK_F5                             , NULL        ,0.0, 0 },
   { "DW_VK_F6"                     , 0, VK_F6                             , NULL        ,0.0, 0 },
   { "DW_VK_F7"                     , 0, VK_F7                             , NULL        ,0.0, 0 },
   { "DW_VK_F8"                     , 0, VK_F8                             , NULL        ,0.0, 0 },
   { "DW_VK_F9"                     , 0, VK_F9                             , NULL        ,0.0, 0 },
   { "DW_VK_F10"                    , 0, VK_F10                            , NULL        ,0.0, 0 },
   { "DW_VK_F11"                    , 0, VK_F11                            , NULL        ,0.0, 0 },
   { "DW_VK_F12"                    , 0, VK_F12                            , NULL        ,0.0, 0 },
   { "DW_VK_F13"                    , 0, VK_F13                            , NULL        ,0.0, 0 },
   { "DW_VK_F14"                    , 0, VK_F14                            , NULL        ,0.0, 0 },
   { "DW_VK_F15"                    , 0, VK_F15                            , NULL        ,0.0, 0 },
   { "DW_VK_F16"                    , 0, VK_F16                            , NULL        ,0.0, 0 },
   { "DW_VK_F17"                    , 0, VK_F17                            , NULL        ,0.0, 0 },
   { "DW_VK_F18"                    , 0, VK_F18                            , NULL        ,0.0, 0 },
   { "DW_VK_F19"                    , 0, VK_F19                            , NULL        ,0.0, 0 },
   { "DW_VK_F20"                    , 0, VK_F20                            , NULL        ,0.0, 0 },
   { "DW_VK_F21"                    , 0, VK_F21                            , NULL        ,0.0, 0 },
   { "DW_VK_F22"                    , 0, VK_F22                            , NULL        ,0.0, 0 },
   { "DW_VK_F23"                    , 0, VK_F23                            , NULL        ,0.0, 0 },
   { "DW_VK_F24"                    , 0, VK_F24                            , NULL        ,0.0, 0 },
   { "DW_VK_NUMLOCK"                , 0, VK_NUMLOCK                        , NULL        ,0.0, 0 },
   { "DW_VK_SCROLL"                 , 0, VK_SCROLL                         , NULL        ,0.0, 0 },
   { "DW_VK_LSHIFT"                 , 0, VK_LSHIFT                         , NULL        ,0.0, 0 },
   { "DW_VK_RSHIFT"                 , 0, VK_RSHIFT                         , NULL        ,0.0, 0 },
   { "DW_VK_LCONTROL"               , 0, VK_LCONTROL                       , NULL        ,0.0, 0 },
   { "DW_VK_RCONTROL"               , 0, VK_RCONTROL                       , NULL        ,0.0, 0 },
   { "DW_VK_LMENU"                  , 0, VK_LMENU                          , NULL        ,0.0, 0 },
   { "DW_VK_RMENU"                  , 0, VK_RMENU                          , NULL        ,0.0, 0 },
/****d* Constants/KeyStates
 * NAME
 *   KeyStates
 * DESCRIPTION
 *   The following is a list of the mnemonic key state values known to Rexx/DW.
 *   These keys are returned in a KeyPressEvent Callback, in the KeyState
 *   argument.
 * ATTRIBUTES
 *   - DW_KC_CTRL
 *   - DW_KC_SHIFT
 *   - DW_KC_ALT
 * SEE ALSO
 *   KeyPressEventCallback
 ******
 */
   { "DW_KC_CTRL"                   , 0, KC_CTRL                           , NULL        ,0.0, 0 },
   { "DW_KC_SHIFT"                  , 0, KC_SHIFT                          , NULL        ,0.0, 0 },
   { "DW_KC_ALT"                    , 0, KC_ALT                            , NULL        ,0.0, 0 },
/****d* Constants/MouseButtons
 * NAME
 *   MouseButtons
 * DESCRIPTION
 *   The following is a list of the mnemonic mouse button values known to RexxDW.
 *   These buttons are returned in a ButtonPressEvent Callback, in the Button
 *   argument.
 * ATTRIBUTES
 *   - DW_VK_LBUTTON
 *   - DW_VK_RBUTTON
 *   - DW_VK_MBUTTON
 * SEE ALSO
 *   ButtonPressEventCallback, ButtonReleaseEventCallback
 ******
 */
 /*
  * The following values do NOT use the equivalent dwindows #defines as these #defines
  * have different values to the actual code. eg under Windows VK_MBUTTON is #defined (in winuser.h)
  * to 4, but the Windows dw.c code returns 3.
  */
   { "DW_VK_LBUTTON"                , 0, 1                                 , NULL        ,0.0, 0 },
   { "DW_VK_RBUTTON"                , 0, 2                                 , NULL        ,0.0, 0 },
   { "DW_VK_MBUTTON"                , 0, 3                                 , NULL        ,0.0, 0 },
/****d* Constants/PercentValue
 * NAME
 *   PercentValue
 * DESCRIPTION
 *   The following constant is used as the value argument to DW_percent_set_pos()
 *   for displaying a moving bar which is not relative to a fixed value.
 * ATTRIBUTES
 *   - DW_PERCENT_INDETERMINATE
 * SEE ALSO
 *   DW_percent_set_pos()
 ******
 */
   { "DW_PERCENT_INDETERMINATE"     , 0, DW_PERCENT_INDETERMINATE          , NULL        ,0.0, 0 },
   { NULL                           , 0, 0                                 , NULL        ,0.0, 0 },
};

/****h* RexxDW/Fontnames
 * NAME
 *   Fontnames
 * DESCRIPTION
 *   Fonts are specified in Rexx/DW in Dynamic Windows font format on all platforms.
 *   This format is:
 *      fontsize.fontname [bold] [italic]
 *   For example to set the font of a widget to 14 point Courier Bold:
 *      Call dw_window_set_font win, '14.Courier Bold'
 * NOTES
 *   While the format of the font specification is the same on all platforms, the
 *   fontnames available on each platform will be different.
 *
 *   On GTK+ 2.x+ it is also possible to specify fonts in Pango format:
 *      fontname [bold] [italic] fontsize
 *   when setting a font. However, dw_window_get_font() will always return the
 *   Dynamic Windows font format.
 * SEE ALSO
 *   DW_window_set_font, DW_window_get_font
 ******
 */

/****h* RexxDW/Packing
 * NAME
 *   Packing
 * DESCRIPTION
 *   One of the most important aspects of laying out widgets in a Rexx/DW program is
 *   how packing works.  This section explains the Rexx/DW packing mechanism.
 * NOTES
 *   |html <img src ="images/packing.png">
 *   The above ??????
 *   TODO - horz/vert boxes, pack empty spaces, expand/dont_expand
 ******
 */

/*
 * These functions used by loader.c to obtain package-specific info
 */
int RexxDWInitialiser( RxPackageGlobalDataDef *MyGlob )
{
   return 0;
}

int RexxDWTerminator( RxPackageGlobalDataDef *MyGlob )
{
   return 0;
}

/*
 * Every package MUST have one of these, otherwise ooRexx will crash
 */
LONG APIENTRY RexxDWInitHandler( LONG ExitNum, LONG Subfun, PEXIT PBlock )
{
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();
   DEBUGDUMP(fprintf(stderr,"%s-%d: In RexxDWInitHandler() ExitNum %ld Subfun %ld\n",__FILE__,__LINE__,ExitNum, Subfun);)
   /*
    * Determine if the Rexx interpreter has RexxCallBack()
    * If USE_REGINA then we have it.
    * If USE_REXXTRANS then we MIGHT have it; we need to call RxTransHaveRexxCallBack()
    * to check.
    * If REXXCALLBACK defined, then we also have it.
    * All other situations, we DON'T have it.
    * We need to determine this BEFORE calling SetPackageConstants()
    */
#if defined( USE_REGINA )
   RexxDWConstants[0].numeric_value = API_REXXCALLBACK_PRESENT;
#elif defined ( USE_REXXTRANS )
   if ( RexxTransHaveRexxCallBack() )
      RexxDWConstants[0].numeric_value = API_REXXCALLBACK_PRESENT;
#elif defined( REXXCALLBACK )
   RexxDWConstants[0].numeric_value = API_REXXCALLBACK_PRESENT;
#else
   /*
    * The default value for HAVE_REXXCALLBACK variable is not set, so we don't have to change anything
    */
#endif
   SetPackageConstants( RxPackageGlobalData, RexxDWConstants, 0 );
   return 0L;
}

LONG APIENTRY RexxDWTermHandler( LONG ExitNum, LONG Subfun, PEXIT PBlock )
{
   DEBUGDUMP(fprintf(stderr,"%s-%d: In RexxDWTermHandler() ExitNum %ld Subfun %ld\n",__FILE__,__LINE__,ExitNum, Subfun);)
   /*
    * Under OS/2 change our process from a PM App back to the type it was when we started.
    */
   return 0L;
}

RexxSubcomHandler *getRexxDWSubcomHandler( void )
{
   return NULL;
}

RexxExitHandler *getRexxDWInitHandler( void )
{
   return RexxDWInitHandler;
}

RexxExitHandler *getRexxDWTermHandler( void )
{
   return RexxDWTermHandler;
}

RexxFunction *getRexxDWFunctions( void )
{
   return RexxDWFunctions;
}

RxPackageConstantDef *getRexxDWConstants( void )
{
   return RexxDWConstants;
}

PackageInitialiser *getRexxDWInitialiser( void )
{
   return RexxDWInitialiser;
}

PackageTerminator *getRexxDWTerminator( void )
{
   return RexxDWTerminator;
}

void *getRexxDWFunctionAddress( char *name )
{
   int i, size = sizeof( RexxDWFunctions ) / sizeof( RexxFunction ) ;

   for (i = 0; i < size && RexxDWFunctions[i].InternalName; i++)
   {
      if ( strcmp( RexxDWFunctions[i].InternalName, name) == 0 )
         return RexxDWFunctions[i].EntryPoint;
   }
   return NULL;
}

/*
 * If we don't have the RexxFreeMemory() API, then emulate it
 */
#ifndef REXXFREEMEMORY
static APIRET APIENTRY RexxFreeMemory( PVOID ptr )
{
#if defined( WIN32 )
   GlobalFree( ptr );
#elif defined( __OS2__ )
   DosFreeMem( ptr );
#else
   free( ptr );
#endif
   return 0;
}
# define REXXFREEMEMORY
#endif

/*
 * If we don't have the RexxAllocateMemory() API, then emulate it
 */
#ifndef REXXALLOCATEMEMORY
static PVOID APIENTRY RexxAllocateMemory( ULONG size )
{
   PVOID ret;
# if defined( WIN32 )
   ret = (PVOID)( GlobalLock( GlobalAlloc ( GMEM_FIXED, size ) ) );
   return ret;
# elif defined( __OS2__ )
   if ( ( BOOL )DosAllocMem( &ret, size, fPERM|PAG_COMMIT ) )
      return NULL;
   else
      return ret;
# else
   ret = (PVOID)malloc( size );
   return ret;
# endif
}
# define REXXALLOCATEMEMORY
#endif

/*
 * Frees the callback args memory allocated in dw_signal_connect()
 * and dw_timer_connect() when RexxCallBack() is present
 */
void free_cb_argv( REXXDWUserData *curr )
{
#if 0
   int i;

   for ( i = 0; i < curr->argc; i++ )
   {
      free( curr->argv[i].strptr );
   }
   if ( curr->argv )
      free( curr->argv );
#endif
}

void *INIT_RXPACKAGE ( RxPackageGlobalDataDef *RxPackageGlobalData )
{
   /*
    * This needs to allocate memory for RexxDWData structure,
    */
   RxPackageGlobalData->RXPACKAGE_tsd = (REXXDWDATA *)MALLOC_TSD( RxPackageGlobalData, sizeof(REXXDWDATA) );
   if ( RxPackageGlobalData->RXPACKAGE_tsd )
      memset( RxPackageGlobalData->RXPACKAGE_tsd , 0, sizeof(REXXDWDATA) );
   /*
    * TODO
    * If any global variable that require initialisation or allocation of memory, add them here
    */
   return (void *)RxPackageGlobalData->RXPACKAGE_tsd;
}

void TERM_RXPACKAGE ( RxPackageGlobalDataDef *RxPackageGlobalData )
{
   /*
   REXXEECDATA *RexxDWData=(REXXEECDATA *)RxPackageGlobalData->RXPACKAGE_tsd;
    * This needs to free memory allocated in INIT_RXPACKAGE()
    * and call any database-specific termination code
    */
   /*
    * TODO
    * Free any items in the RexxDWData struct that were allocated in INIT_RXPACKAGE
    */
   /*
    * Free the RexxDWData struct
    */
   FREE_TSD( RxPackageGlobalData, RxPackageGlobalData->RXPACKAGE_tsd );
   RxPackageGlobalData = NULL;
}

/*
 * Adds a new callback entry to the queue
 */
static REXXDWCallbackData *addCallback( RxPackageGlobalDataDef *RxPackageGlobalData, int callback_type )
{
   REXXDWCallbackData *curr;
   REXXDWDATA *RexxDWData;

   InternalTrace( RxPackageGlobalData, "addCallback", "Starting..." );

   RexxDWData=(REXXDWDATA *)RxPackageGlobalData->RXPACKAGE_tsd;
   curr = (REXXDWCallbackData *)calloc( 1, sizeof(REXXDWCallbackData) );
   if ( curr == NULL )
      return NULL;
   curr->next = NULL;
   curr->callback_type = callback_type;
   if ( RexxDWData->FirstCallback == NULL )
   {
      RexxDWData->FirstCallback = RexxDWData->LastCallback = curr;
   }
   else
   {
      RexxDWData->LastCallback->next = curr;
      RexxDWData->LastCallback = curr;
   }
   return curr;
}

/*
 * Adds a new RexxDW widget to the list
 */
static REXXDWWidgetData *addWidget( RxPackageGlobalDataDef *RxPackageGlobalData, int widget_type )
{
   REXXDWWidgetData *curr;
   REXXDWDATA *RexxDWData;

   InternalTrace( RxPackageGlobalData, "addWidget", "Starting..." );

   RexxDWData=(REXXDWDATA *)RxPackageGlobalData->RXPACKAGE_tsd;
   curr = (REXXDWWidgetData *)calloc( 1, sizeof(REXXDWWidgetData) );
   if ( curr == NULL )
      return NULL;
   curr->next = NULL;
   curr->widget_type = widget_type;
   if ( RexxDWData->FirstWidget == NULL )
   {
      RexxDWData->FirstWidget = RexxDWData->LastWidget = curr;
   }
   else
   {
      RexxDWData->LastWidget->next = curr;
      RexxDWData->LastWidget = curr;
   }
   return curr;
}

/*
 * Always look for a match between handle, and widget->TopBox
 */
static REXXDWWidgetData *findWidget( RxPackageGlobalDataDef *RxPackageGlobalData, HWND handle )
{
   REXXDWWidgetData *curr;
   REXXDWDATA *RexxDWData;

   InternalTrace( RxPackageGlobalData, "findWidget", "Starting..." );

   RexxDWData=(REXXDWDATA *)RxPackageGlobalData->RXPACKAGE_tsd;
   curr =  RexxDWData->FirstWidget;
   while( curr )
   {
      switch( curr->widget_type )
      {
         case REXXDW_WIDGET_LISTBOX_SELECTOR:
            if ( handle == curr->widget.ListboxSelectorData.Topbox )
               return curr;
            break;
      }
      curr = curr->next;
   }
   return curr;
}

/*
 * Always look for a match between handle, and widget->TopBox
 */
static void deleteWidget( RxPackageGlobalDataDef *RxPackageGlobalData, REXXDWWidgetData *widget )
{
   REXXDWWidgetData *curr;
   REXXDWDATA *RexxDWData;

   InternalTrace( RxPackageGlobalData, "deleteWidget", "Starting..." );

   RexxDWData=(REXXDWDATA *)RxPackageGlobalData->RXPACKAGE_tsd;
   curr = RexxDWData->FirstWidget;
   switch( widget->widget_type )
   {
      case REXXDW_WIDGET_LISTBOX_SELECTOR:
         free( widget->widget.ListboxSelectorData.LeftIndex );
         widget->widget.ListboxSelectorData.LeftIndex = NULL;
         free( widget->widget.ListboxSelectorData.RightIndex );
         widget->widget.ListboxSelectorData.RightIndex = NULL;
         break;
   }
   /*
    * Remove from linked list
    */
   if ( widget == RexxDWData->FirstWidget
   &&   widget == RexxDWData->LastWidget )
   {
      RexxDWData->FirstWidget = RexxDWData->LastWidget = NULL;
   }
   else if ( widget == RexxDWData->FirstWidget )
   {
      RexxDWData->FirstWidget = widget->next;
   }
   else if ( widget == RexxDWData->LastWidget )
   {
      while( curr )
      {
         if ( curr->next == widget )
         {
            RexxDWData->LastWidget = curr;
            curr->next = NULL;
            break;
         }
         curr = curr->next;
      }
   }
   else
   {
      while( curr )
      {
         if ( curr->next == widget )
         {
            curr->next = widget->next;
            break;
         }
         curr = curr->next;
      }
   }
   free( widget );
   return;
}

static char *myquote( char *str, int len )
{
   int i;
   int squote = 0, dquote = 0, quote;
   char *qstr;

   qstr = (char *)malloc( len+3 );
   if ( qstr == NULL )
      return str;

   for ( i = 0; i < len; i++ )
   {
      if ( str[i] == '\'' )
         squote++;
      else if ( str[i] == '"' )
         dquote++;
      qstr[i+1] = str[i];
   }
   if ( squote == 0 )
      quote = '\'';
   else
      quote = '"';
   qstr[0] = quote;
   qstr[len+1] = quote;
   qstr[len+2] = '\0';
   return qstr;
}

static int rexxdw_thread_new_cb( void *data )
{
   PRXSTRING argv;
   tREXXDWUserData *curr=(tREXXDWUserData *)data;
   RXSTRING RetStr;
   int min_args=0;
   int i,rc,num_args=min_args+curr->argc;
   SHORT rcode;
   char buf[100];
   REXXDWCallbackData *cbd;

   if ( HAVE_REXXCALLBACK )
   {
      argv = (PRXSTRING)malloc( num_args*sizeof(RXSTRING) );
      if ( argv == NULL )
         return -1;
      for ( i = 0; i < curr->argc; i++ )
      {
         argv[i+min_args] = curr->argv[i];
      }

      sprintf( buf, "DW_THREAD_NEW callback: %s() args: %d", curr->func, num_args );
      FunctionPrologue( curr->RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, buf, num_args, argv );

      MAKERXSTRING( RetStr, NULL, 0 );
#if defined( REXXCALLBACK )
      rc = RexxCallBack( curr->func, num_args, argv, &rcode, &RetStr );
#else
      rc = rcode = 0;
#endif
      if ( RetStr.strptr )
         RexxFreeMemory( RetStr.strptr );
      free( argv );
/*      free_cb_argv( curr ); */
      return rcode;
   }
   else
   {
      cbd = addCallback( curr->RxPackageGlobalData, REXXDW_MAX_CB+2 );
      if ( cbd == NULL )
         return -1;
      /*
       * Specify the data for this callback type
       */
      cbd->window = NULL;
      cbd->userdata = data;
      DEBUGDUMP(fprintf(stderr,"%s-%d: In rexxdw_timer_cb() Func: %s Argc: %d\n",
                               __FILE__,__LINE__,
                               cbd->userdata->func,
                               cbd->userdata->argc );)
      return 0;
   }
}

static int RexxDWListboxSelectorPopulate( REXXDWListboxSelector *widget )
{
   int i,ind,l_idx=0,r_idx=0;
   RXSTRING value;
   RxPackageGlobalDataDef *RxPackageGlobalData=widget->RxPackageGlobalData;

   for ( i = 1; i <= widget->count; i++ )
   {
      if ( GetRexxVariable( RxPackageGlobalData, widget->ValueStemName, &value, i ) == NULL )
      {
         dw_messagebox("Error!", DW_MB_OK | DW_MB_WARNING, "Error getting value of \"%s%d\". Did you EXPOSE this stem?", widget->ValueStemName, i );
         return 1;
      }
      if ( GetRexxVariableInteger( RxPackageGlobalData, widget->SetStemName, &ind, i ) == NULL )
      {
         dw_messagebox("Error!", DW_MB_OK | DW_MB_WARNING, "Error getting value of \"%s%d\". Did you EXPOSE this stem?", widget->SetStemName, i );
         return 1;
      }
      InternalTrace( RxPackageGlobalData, "RexxDWListboxSelectorPopulate", "Variable: <%s%d> Value: <%s> Set: %d", widget->ValueStemName, i, value.strptr, ind );

      if ( ind )
      {
         dw_listbox_append( widget->LeftListbox, RXSTRPTR(value) );
         widget->LeftIndex[l_idx] = i;
         l_idx++;
      }
      else
      {
         dw_listbox_append( widget->RightListbox, RXSTRPTR(value) );
         widget->RightIndex[r_idx] = i;
         r_idx++;
      }
   }
   widget->l_idx = l_idx;
   widget->r_idx = r_idx;
   return 0;
}

static int RexxDWListboxSelectorClear( REXXDWListboxSelector *widget )
{
   RxPackageGlobalDataDef *RxPackageGlobalData=widget->RxPackageGlobalData;

   InternalTrace( RxPackageGlobalData, "RexxDWListboxSelectorClear", "Value Stem: %s, SetStemName: %s", widget->ValueStemName, widget->SetStemName );

   widget->count = 0;

   dw_listbox_clear( widget->LeftListbox );
   dw_listbox_clear( widget->RightListbox );

   free( widget->LeftIndex );
   free( widget->RightIndex );
   widget->LeftIndex = NULL;
   widget->RightIndex = NULL;
   return 0;
}

/*
 * Move a listbox item from left listbox to right listbox
 */
static int DWSIGNAL RexxDWListBoxSelectorMoveItemLeftCallback( HWND window, void *data )
{
   REXXDWListboxSelector *widget = (REXXDWListboxSelector *)data;
   int idx=-1,len,tail;
   char buf[100];
   RxPackageGlobalDataDef *RxPackageGlobalData=widget->RxPackageGlobalData;

   InternalTrace( RxPackageGlobalData, "RexxDWListBoxSelectorMoveItemLeftCallback", "Value Stem: %s, SetStemName: %s", widget->ValueStemName, widget->SetStemName );

   while ( 1 )
   {
      idx = dw_listbox_selected_multi( widget->RightListbox, idx );
      if ( idx == DW_LIT_NONE )
         break;
      tail = widget->RightIndex[idx];
      len = sprintf( buf, "%s%d", widget->SetStemName, tail );
      if ( SetRexxVariable( RxPackageGlobalData, buf, len, (char *)"1", 1 ) == 1 )
      {
       /* todo */
      }
   }
   dw_listbox_clear( widget->LeftListbox );
   dw_listbox_clear( widget->RightListbox );
   RexxDWListboxSelectorPopulate( widget );
   return 0;
}

/*
 * Move a listbox item from right listbox to left listbox
 */
static int DWSIGNAL RexxDWListBoxSelectorMoveItemRightCallback( HWND window, void *data )
{
   REXXDWListboxSelector *widget = (REXXDWListboxSelector *)data;
   int idx=-1,len,tail;
   char buf[100];
   RxPackageGlobalDataDef *RxPackageGlobalData=widget->RxPackageGlobalData;

   InternalTrace( RxPackageGlobalData, "RexxDWListBoxSelectorMoveItemRightCallback", "Value Stem: %s, SetStemName: %s", widget->ValueStemName, widget->SetStemName );

   while ( 1 )
   {
      idx = dw_listbox_selected_multi( widget->LeftListbox, idx );
      if ( idx == DW_LIT_NONE )
         break;
      tail = widget->LeftIndex[idx];
      len = sprintf( buf, "%s%d", widget->SetStemName, tail );
      if ( SetRexxVariable( RxPackageGlobalData, buf, len, (char *)"0", 1 ) == 1 )
      {
       /* todo */
      }
   }
   dw_listbox_clear( widget->LeftListbox );
   dw_listbox_clear( widget->RightListbox );
   RexxDWListboxSelectorPopulate( widget );
   return 0;
}

/*
 * Move all listbox items from left listbox to right listbox
 */
static int DWSIGNAL RexxDWListBoxSelectorMoveAllItemLeftCallback( HWND window, void *data )
{
   REXXDWListboxSelector *widget = (REXXDWListboxSelector *)data;
   int i,len;
   char buf[100];
   RxPackageGlobalDataDef *RxPackageGlobalData=widget->RxPackageGlobalData;

   InternalTrace( RxPackageGlobalData, "RexxDWListBoxSelectorMoveAllItemLeftCallback", "Variable: %s", widget->ValueStemName );

   for ( i = 1; i <= widget->count; i++ )
   {
      len = sprintf( buf, "%s%d", widget->SetStemName, i );
      if ( SetRexxVariable( RxPackageGlobalData, buf, len, (char *)"1", 1 ) == 1 )
      {
       /* todo */
      }
   }
   dw_listbox_clear( widget->LeftListbox );
   dw_listbox_clear( widget->RightListbox );
   RexxDWListboxSelectorPopulate( widget );
   return 0;
}

/*
 * Move all listbox items from right listbox to left listbox
 */
static int DWSIGNAL RexxDWListBoxSelectorMoveAllItemRightCallback( HWND window, void *data )
{
   REXXDWListboxSelector *widget = (REXXDWListboxSelector *)data;
   int i,len;
   char buf[100];
   RxPackageGlobalDataDef *RxPackageGlobalData=widget->RxPackageGlobalData;

   InternalTrace( RxPackageGlobalData, "RexxDWListBoxSelectorMoveAllItemRightCallback", "Variable: %s", widget->ValueStemName );

   for ( i = 1; i <= widget->count; i++ )
   {
      len = sprintf( buf, "%s%d", widget->SetStemName, i );
      if ( SetRexxVariable( RxPackageGlobalData, buf, len, (char *)"0", 1 ) == 1 )
      {
       /* todo */
      }
   }
   dw_listbox_clear( widget->LeftListbox );
   dw_listbox_clear( widget->RightListbox );
   RexxDWListboxSelectorPopulate( widget );
   return 0;
}

static int DWSIGNAL RexxDWListBoxSelectorDeleteCallback( HWND window )
{
//   REXXDWListboxSelector *widget = (REXXDWListboxSelector *)data;
//   int i,len;
//   char buf[100];
//   RxPackageGlobalDataDef *RxPackageGlobalData=widget->RxPackageGlobalData;

//   InternalTrace( RxPackageGlobalData, "RexxDWListBoxSelectorDeleteCallback" );
fprintf( stderr,"%s %d\n",__FILE__,__LINE__);

   return 0;
}

/****h* RexxDW/Callbacks
 * NAME
 *   Callbacks
 * DESCRIPTION
 *   Rexx/DW is an event driven application. What this means is that internally various
 *   events occur, such as a key press, a window becoming exposed. Rexx/DW allows you
 *   to call an internal Rexx procedure when various events occur.
 *
 *   Each widget you create has certain valid events associated with it. As part of your
 *   code you associate a particular event with a widget. This is done by calling
 *   DW_signal_connect(). This associates an event on a particular widget with an
 *   internal Rexx procedure, so that when the particular event occurs on that widget
 *   your procedure is executed and certain parameters are passed to it. Details on what
 *   parameters are passed is detailed below depending on the type of event.
 *
 *   The internal Rexx procedure is called in the context of where DW_main() or
 *   DW_main_iteration() is called. This is important when considering what Rexx
 *   variables may or may not be in scope. It is probably better to always use
 *   PROCEDURE EXPOSE (myglobals) on procedures so you know that any variables you
 *   use within your procedure will not affect an others outside without you explicitly
 *   listing them.
 *
 *   Once you have handled an event in your code, you can return 1 to inform Dynamic Windows
 *   that all processing has been handled for the particular event. Dynamic Windows will
 *   not execute its default event handling code.
 *   If your event handler returns 0, then Dynamic Windows will execute its default event
 *   handling code.
 * ATTRIBUTES
 *   The values of the supported events are:
 *   - REXXDW_CONFIGURE_EVENT_CB
 *   - REXXDW_KEY_PRESS_EVENT_CB
 *   - REXXDW_BUTTON_PRESS_EVENT_CB
 *   - REXXDW_BUTTON_RELEASE_EVENT_CB
 *   - REXXDW_MOTION_NOTIFY_EVENT_CB
 *   - REXXDW_DELETE_EVENT_CB
 *   - REXXDW_EXPOSE_EVENT_CB
 *   - REXXDW_CLICKED_EVENT_CB
 *   - REXXDW_ITEM_ENTER_EVENT_CB
 *   - REXXDW_ITEM_CONTEXT_EVENT_CB
 *   - REXXDW_LIST_SELECT_EVENT_CB
 *   - REXXDW_ITEM_SELECT_EVENT_CB
 *   - REXXDW_SET_FOCUS_EVENT_CB
 *   - REXXDW_VALUE_CHANGED_EVENT_CB
 *   - REXXDW_SWITCH_PAGE_EVENT_CB
 *   - REXXDW_COLUMN_CLICK_EVENT_CB
 * NOTES
 *   With Rexx interpreters that don't provide RexxCallBack() it is not possible to
 *   inform Dynamic Windows to execute the default callback functionality after
 *   your event handler has executed its code, because the return value is returned
 *   to the Rexx program; via DW_main_iteration(), not to Dynamic Windows.
 ******
 */

/****** Callbacks/ConfigureEventCallback
 * NAME
 *   ConfigureEventCallback
 * USAGE
 *   rcode = ConfigureEventCallback(Window, Width, Height [,UserData[,...]])
 * DESCRIPTION
 *   Called when a DW_CONFIGURE_EVENT signal fires on a window. This event usually
 *   occurs when a window's size is changed.
 * ARGUMENTS
 *   - Window   - value of the window handle that has been resized
 *   - Width    - the new width of Window in pixels
 *   - Height   - the new height of Window in pixels
 *   - UserData - following optional values that the user supplied when the signal event
 *              was connected
 * RETURN VALUE
 *   Return 1 to ensure that the default callback is not called.
 *   Return 0 to enable the default callback to be called.
 * SOURCE
...
Call dw_signal_connect window, !REXXDW.!DW_CONFIGURE_EVENT, 'configure_cb', ,
   'fred', 'mary'
...
configure_cb: Procedure Expose !REXXDW.
Parse Arg win, width, height, var1, var2
Say 'New width:' width 'height:' height 'for Window:' win 'UserVar1:' var1 ,
   'UserVar2:' var2
Return 1
 ******
 */
static int DWSIGNAL rexxdw_configure_event_cb( HWND window, int width, int height, void *data )
{
   char rx_window[16];
   char rx_width[16];
   char rx_height[16];
   PRXSTRING argv;
   REXXDWUserData *curr=(REXXDWUserData *)data;
   RXSTRING RetStr;
   int min_args=3;
   int i,rc,num_args=min_args+curr->argc;
   SHORT rcode;
   char buf[100];
   REXXDWCallbackData *cbd;
   REXXDWDATA *RexxDWData=(REXXDWDATA *)curr->RxPackageGlobalData->RXPACKAGE_tsd;

   RexxDWData->CallbackTimestamp = time(NULL);
   if ( HAVE_REXXCALLBACK )
   {
      argv = (PRXSTRING)malloc( num_args*sizeof(RXSTRING) );
      if ( argv == NULL )
         return -1;
      sprintf( rx_window, "%lu", (unsigned long)window );
      sprintf( rx_width, "%ld", (long)width );
      sprintf( rx_height, "%ld", (long)height );
      MAKERXSTRING( argv[0], rx_window, strlen( rx_window ) );
      MAKERXSTRING( argv[1], rx_width, strlen( rx_width ) );
      MAKERXSTRING( argv[2], rx_height, strlen( rx_height ) );
      for ( i = 0; i < curr->argc; i++ )
      {
         argv[i+min_args] = curr->argv[i];
      }

      sprintf( buf, "DW_CONFIGURE_EVENT callback: %s()", curr->func );
      FunctionPrologue( curr->RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, buf, num_args, argv );

      MAKERXSTRING( RetStr, NULL, 0 );
#if defined( REXXCALLBACK )
      rc = RexxCallBack( curr->func, num_args, argv, &rcode, &RetStr );
#else
      rc = rcode = 0;
#endif
      if ( RetStr.strptr )
         RexxFreeMemory( RetStr.strptr );
      free( argv );
      free_cb_argv( curr );
      return rcode;
   }
   else
   {
      cbd = addCallback( curr->RxPackageGlobalData, REXXDW_CONFIGURE_EVENT_CB );
      if ( cbd == NULL )
         return -1;
      /*
       * Specify the data for this callback type
       */
      cbd->window = window;
      cbd->event.ConfigureCallbackData.width = width;
      cbd->event.ConfigureCallbackData.height = height;
      cbd->userdata = data;
      DEBUGDUMP(fprintf(stderr,"%s-%d: In rexxdw_configure_event_cb() Window: %ld Width: %d Height: %d Func: %s Argc: %d\n",
                               __FILE__,__LINE__,
                               cbd->window,
                               cbd->event.ConfigureCallbackData.width,
                               cbd->event.ConfigureCallbackData.height,
                               cbd->userdata->func,
                               cbd->userdata->argc );)
      return 0;
   }
}

/****** Callbacks/KeyPressEventCallback
 * NAME
 *   KeyPressEventCallback
 * USAGE
 *   rcode = KeyPressEventCallback(Window, ASCIIKey, VirtualKey, KeyState [,UserData[,...]])
 * DESCRIPTION
 *   Called when a DW_KEY_PRESS_EVENT signal fires on a window. Occurs when the user
 *   presses a key.
 * ARGUMENTS
 *   - Window     - value of the window handle in which the key was pressed
 *   - ASCIIKey   - the Hex representation of the key pressed. If not as ASCII key
 *                  then the empty string is returned and VirtualKey should be used
 *   - VirtualKey - the mnemonic key value of the key pressed
 *   - KeyState   - indicates whether SHIFT, CTRL or ALT modifiers are active
 *   - UserData   - following optional values that the user supplied when the signal event
 *                  was connected
 * RETURN VALUE
 *   Return 1 to ensure that the default callback is not called.
 *   Return 0 to enable the default callback to be called.
 * SEE ALSO
 *   VirtualKeys, KeyStates
 * SOURCE
...
Call dw_signal_connect window, !REXXDW.!DW_KEY_PRESS_EVENT, 'keypress_cb', ,
   'fred', 'mary'
...
keypress_cb: Procedure Expose !REXXDW.
Parse Arg win, ascii, virtual, state, var1, var2
Say 'ASCII Key:' ascii 'Virtual Key:' virtual 'State:' state 'UserVar1:' ,
   var1 'UserVar2:' var2
If virtual = !REXXDW.!DW_VK_F1 Then Return 1
Return 1
 ******
 */
static int DWSIGNAL rexxdw_key_press_event_cb( HWND window, char key, int virtual_key, int state, void *data )
{
   char rx_window[16];
   char rx_key[16];
   char rx_virtual_key[16];
   char rx_state[16];
   PRXSTRING argv;
   REXXDWUserData *curr=(REXXDWUserData *)data;
   RXSTRING RetStr;
   int min_args=4;
   int i,rc,num_args=min_args+curr->argc;
   SHORT rcode;
   char buf[100];
   REXXDWCallbackData *cbd;
   REXXDWDATA *RexxDWData=(REXXDWDATA *)curr->RxPackageGlobalData->RXPACKAGE_tsd;

   RexxDWData->CallbackTimestamp = time(NULL);
   if ( HAVE_REXXCALLBACK )
   {
      argv = (PRXSTRING)malloc( num_args*sizeof(RXSTRING) );
      if ( argv == NULL )
         return -1;
      sprintf( rx_window, "%lu", (unsigned long)window );
      MAKERXSTRING( argv[0], rx_window, strlen( rx_window ) );
      if ( key )
         sprintf( rx_key, "\"%X\"", key );
      else
         strcpy( rx_key, "" );
      MAKERXSTRING( argv[1], rx_key, strlen( rx_key ) );
      sprintf( rx_virtual_key, "%lu", (unsigned long)virtual_key );
      MAKERXSTRING( argv[2], rx_virtual_key, strlen( rx_virtual_key ) );
      sprintf( rx_state, "%lu", (unsigned long)state );
      MAKERXSTRING( argv[3], rx_state, strlen( rx_state ) );
      for ( i = 0; i < curr->argc; i++ )
      {
         argv[i+min_args] = curr->argv[i];
      }

      sprintf( buf, "DW_KEY_PRESS_EVENT callback: %s()", curr->func );
      FunctionPrologue( curr->RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, buf, num_args, argv );

      MAKERXSTRING( RetStr, NULL, 0 );
#if defined( REXXCALLBACK )
      rc = RexxCallBack( curr->func, num_args, argv, &rcode, &RetStr );
#else
      rc = rcode = 0;
#endif
      if ( RetStr.strptr )
         RexxFreeMemory( RetStr.strptr );
      free( argv );
      free_cb_argv( curr );
      return rcode;
   }
   else
   {
      cbd = addCallback( curr->RxPackageGlobalData, REXXDW_KEY_PRESS_EVENT_CB );
      if ( cbd == NULL )
         return -1;
      /*
       * Specify the data for this callback type
       */
      cbd->window = window;
      cbd->event.KeyPressCallbackData.key = key;
      cbd->event.KeyPressCallbackData.virtual_key = virtual_key;
      cbd->event.KeyPressCallbackData.state = state;
      cbd->userdata = data;
      DEBUGDUMP(fprintf(stderr,"%s-%d: In rexxdw_key_press_event_cb() Window: %ld Key: \"%X\"(%d) VirtualKey: %d State: %d Func: %s Argc: %d\n",
                               __FILE__,__LINE__,
                               cbd->window,
                               cbd->event.KeyPressCallbackData.key, cbd->event.KeyPressCallbackData.key,
                               cbd->event.KeyPressCallbackData.virtual_key,
                               cbd->event.KeyPressCallbackData.state,
                               cbd->userdata->func,
                               cbd->userdata->argc );)
   }
   return 0;
}

/****** Callbacks/ButtonPressEventCallback
 * NAME
 *   ButtonPressEventCallback
 * USAGE
 *   rcode = ButtonPressEventCallback(Window, X, Y, Button [,UserData[,...]])
 * DESCRIPTION
 *   Called when a DW_BUTTON_PRESS_EVENT signal fires on a window. This occurs when
 *   the user clicks a button with the mouse.
 * ARGUMENTS
 *   - Window     - value of the window handle in which the mouse button was pressed
 *   - X          - the X coordinate where the mouse button was pressed, relative
 *                  to the top left corner of the Window
 *   - Y          - the Y coordinate where the mouse button was pressed, relative
 *                  to the top left corner of the Window
 *   - Button     - which mouse button was pressed
 *   - UserData   - following optional values that the user supplied when the signal event
 *                  was connected
 * RETURN VALUE
 *   Return 1 to ensure that the default callback is not called.
 *   Return 0 to enable the default callback to be called.
 * SEE ALSO
 *   MouseButtons
 * SOURCE
...
Call dw_signal_connect window, !REXXDW.!DW_BUTTON_PRESS_EVENT, ,
   'buttonpress_cb', 'fred'
...
buttonpress_cb:
Parse Arg win, x, y, button, data
Say 'Button:' button 'pressed at:' x'/'y 'in Window:' win
Return 1
 ******
 */
static int DWSIGNAL rexxdw_button_press_event_cb( HWND window, int x, int y, int button, void *data )
{
   char rx_window[16];
   char rx_x[16];
   char rx_y[16];
   char rx_button[16];
   PRXSTRING argv;
   REXXDWUserData *curr=(REXXDWUserData *)data;
   RXSTRING RetStr;
   int min_args=4;
   int i,rc,num_args=min_args+curr->argc;
   SHORT rcode;
   char buf[100];
   REXXDWCallbackData *cbd;
   REXXDWDATA *RexxDWData=(REXXDWDATA *)curr->RxPackageGlobalData->RXPACKAGE_tsd;

   RexxDWData->CallbackTimestamp = time(NULL);
   if ( HAVE_REXXCALLBACK )
   {
      argv = (PRXSTRING)malloc( num_args*sizeof(RXSTRING) );
      if ( argv == NULL )
         return -1;
      sprintf( rx_window, "%lu", (unsigned long)window );
      sprintf( rx_x, "%d", x );
      sprintf( rx_y, "%d", y );
      sprintf( rx_button, "%d", button );
      MAKERXSTRING( argv[0], rx_window, strlen( rx_window ) );
      MAKERXSTRING( argv[1], rx_x, strlen( rx_x ) );
      MAKERXSTRING( argv[2], rx_y, strlen( rx_y ) );
      MAKERXSTRING( argv[3], rx_button, strlen( rx_button ) );
      for ( i = 0; i < curr->argc; i++ )
      {
         argv[i+min_args] = curr->argv[i];
      }

      sprintf( buf, "DW_BUTTON_PRESS_EVENT callback: %s()", curr->func );
      FunctionPrologue( curr->RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, buf, num_args, argv );

      MAKERXSTRING( RetStr, NULL, 0 );
#if defined( REXXCALLBACK )
      rc = RexxCallBack( curr->func, num_args, argv, &rcode, &RetStr );
#else
      rc = rcode = 0;
#endif
      if ( RetStr.strptr )
         RexxFreeMemory( RetStr.strptr );
      free( argv );
      free_cb_argv( curr );
      return rcode;
   }
   else
   {
      cbd = addCallback( curr->RxPackageGlobalData, REXXDW_BUTTON_PRESS_EVENT_CB );
      if ( cbd == NULL )
         return -1;
      /*
       * Specify the data for this callback type
       */
      cbd->window = window;
      cbd->event.ButtonPressCallbackData.x = x;
      cbd->event.ButtonPressCallbackData.y = y;
      cbd->event.ButtonPressCallbackData.button = button;
      cbd->userdata = data;
      DEBUGDUMP(fprintf(stderr,"%s-%d: In rexxdw_button_press_event_cb() Window: %ld X: %d Y: %d Button: %d Func: %s Argc: %d\n",
                               __FILE__,__LINE__,
                               cbd->window,
                               cbd->event.ButtonPressCallbackData.x,
                               cbd->event.ButtonPressCallbackData.y,
                               cbd->event.ButtonPressCallbackData.button,
                               cbd->userdata->func,
                               cbd->userdata->argc );)
      return 0;
   }
}

/****** Callbacks/ButtonReleaseEventCallback
 * NAME
 *   ButtonReleaseEventCallback
 * USAGE
 *   rcode = ButtonReleaseEventCallback(Window, X, Y, Button [,UserData[,...]])
 * DESCRIPTION
 *   Called when a DW_BUTTON_RELEASE_EVENT signal fires on a window. This occurs when
 *   the user releases a button on the mouse.
 * ARGUMENTS
 *   - Window     - value of the window handle in which the mouse button was released
 *   - X          - the X coordinate where the mouse button was released, relative
 *                  to the top left corner of the Window
 *   - Y          - the Y coordinate where the mouse button was released, relative
 *                  to the top left corner of the Window
 *   - Button     - which mouse button was released
 *   - UserData   - following optional values that the user supplied when the signal event
 *                  was connected
 * RETURN VALUE
 *   Return 1 to ensure that the default callback is not called.
 *   Return 0 to enable the default callback to be called.
 * SEE ALSO
 *   MouseButtons
 * SOURCE
...
Call dw_signal_connect window, !REXXDW.!DW_BUTTON_RELEASE_EVENT, ,
   'buttonrelease_cb', 'fred'
...
buttonreleases_cb:
Parse Arg win, x, y, button, data
Say 'Button:' button 'released at:' x'/'y 'in Window:' win
Return 1
 ******
 */
static int DWSIGNAL rexxdw_button_release_event_cb( HWND window, int x, int y, int button, void *data )
{
   char rx_window[16];
   char rx_x[16];
   char rx_y[16];
   char rx_button[16];
   PRXSTRING argv;
   REXXDWUserData *curr=(REXXDWUserData *)data;
   RXSTRING RetStr;
   int min_args=4;
   int i,rc,num_args=min_args+curr->argc;
   SHORT rcode;
   char buf[100];
   REXXDWCallbackData *cbd;
   REXXDWDATA *RexxDWData=(REXXDWDATA *)curr->RxPackageGlobalData->RXPACKAGE_tsd;

   RexxDWData->CallbackTimestamp = time(NULL);
   if ( HAVE_REXXCALLBACK )
   {
      argv = (PRXSTRING)malloc( num_args*sizeof(RXSTRING) );
      if ( argv == NULL )
         return -1;
      sprintf( rx_window, "%lu", (unsigned long)window );
      sprintf( rx_x, "%d", x );
      sprintf( rx_y, "%d", y );
      sprintf( rx_button, "%d", button );
      MAKERXSTRING( argv[0], rx_window, strlen( rx_window ) );
      MAKERXSTRING( argv[1], rx_x, strlen( rx_x ) );
      MAKERXSTRING( argv[2], rx_y, strlen( rx_y ) );
      MAKERXSTRING( argv[3], rx_button, strlen( rx_button ) );
      for ( i = 0; i < curr->argc; i++ )
      {
         argv[i+min_args] = curr->argv[i];
      }

      sprintf( buf, "DW_BUTTON_RELEASE_EVENT callback: %s()", curr->func );
      FunctionPrologue( curr->RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, buf, num_args, argv );

      MAKERXSTRING( RetStr, NULL, 0 );
#if defined( REXXCALLBACK )
      rc = RexxCallBack( curr->func, num_args, argv, &rcode, &RetStr );
#else
      rc = rcode = 0;
#endif
      if ( RetStr.strptr )
         RexxFreeMemory( RetStr.strptr );
      free( argv );
      free_cb_argv( curr );
      return rcode;
   }
   else
   {
      cbd = addCallback( curr->RxPackageGlobalData, REXXDW_BUTTON_RELEASE_EVENT_CB );
      if ( cbd == NULL )
         return -1;
      /*
       * Specify the data for this callback type
       */
      cbd->window = window;
      cbd->event.ButtonReleaseCallbackData.x = x;
      cbd->event.ButtonReleaseCallbackData.y = y;
      cbd->event.ButtonReleaseCallbackData.button = button;
      cbd->userdata = data;
      DEBUGDUMP(fprintf(stderr,"%s-%d: In rexxdw_button_release_event_cb() Window: %ld X: %d Y: %d Button: %d Func: %s Argc: %d\n",
                               __FILE__,__LINE__,
                               cbd->window,
                               cbd->event.ButtonReleaseCallbackData.x,
                               cbd->event.ButtonReleaseCallbackData.y,
                               cbd->event.ButtonReleaseCallbackData.button,
                               cbd->userdata->func,
                               cbd->userdata->argc );)
      return 0;
   }
}

/****** Callbacks/MotionNotifyEventCallback
 * NAME
 *   MotionNotifyEventCallback
 * USAGE
 *   rcode = MotionNotifyEventCallback(Window, X, Y, Button [,UserData[,...]])
 * DESCRIPTION
 *   Called when a DW_MOTION_NOTIFY_EVENT signal fires on a window. This occurs when
 *   the user moves the mouse.
 * ARGUMENTS
 *   - Window     - value of the window handle in which the mouse is being moved
 *   - X          - the X coordinate where the mouse cursor is currently displayed, relative
 *                  to the top left corner of the Window
 *   - Y          - the Y coordinate where the mouse cursor is currently displayed, relative
 *                  to the top left corner of the Window
 *   - Button     - Indicates which button(s) are being pressed during the mouse
 *   - UserData   - following optional values that the user supplied when the signal event
 *                  was connected
 * RETURN VALUE
 *   Return 1 to ensure that the default callback is not called.
 *   Return 0 to enable the default callback to be called.
 * SEE ALSO
 *   ButtonMasks, ButtonPressEventCallback, ButtonReleaseEventCallback
 * SOURCE
 ...
Call dw_signal_connect window, !REXXDW.!DW_MOTION_NOTIFY_EVENT, ,
   'motionnotify_cb', 'fred'
...
motionnotify_cb:
Parse Arg win, x, y, state, data
Say 'Mouse movied to' x'/'y 'in Window:' win 'with state:' state
Return 1
 ******
 */
static int DWSIGNAL rexxdw_motion_notify_event_cb( HWND window, int x, int y, int state, void *data )
{
   char rx_window[16];
   char rx_x[16];
   char rx_y[16];
   char rx_state[16];
   PRXSTRING argv;
   REXXDWUserData *curr=(REXXDWUserData *)data;
   RXSTRING RetStr;
   int min_args=4;
   int i,rc,num_args=min_args+curr->argc;
   SHORT rcode;
   char buf[100];
   REXXDWCallbackData *cbd;
   REXXDWDATA *RexxDWData=(REXXDWDATA *)curr->RxPackageGlobalData->RXPACKAGE_tsd;

   RexxDWData->CallbackTimestamp = time(NULL);
   if ( HAVE_REXXCALLBACK )
   {
      argv = (PRXSTRING)malloc( num_args*sizeof(RXSTRING) );
      if ( argv == NULL )
         return -1;
      sprintf( rx_window, "%lu", (unsigned long)window );
      sprintf( rx_x, "%d", x );
      sprintf( rx_y, "%d", y );
      sprintf( rx_state, "%d", state );
      MAKERXSTRING( argv[0], rx_window, strlen( rx_window ) );
      MAKERXSTRING( argv[1], rx_x, strlen( rx_x ) );
      MAKERXSTRING( argv[2], rx_y, strlen( rx_y ) );
      MAKERXSTRING( argv[3], rx_state, strlen( rx_state ) );
      for ( i = 0; i < curr->argc; i++ )
      {
         argv[i+min_args] = curr->argv[i];
      }

      sprintf( buf, "DW_MOTION_NOTIFY_EVENT callback: %s()", curr->func );
      FunctionPrologue( curr->RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, buf, num_args, argv );

      MAKERXSTRING( RetStr, NULL, 0 );
#if defined( REXXCALLBACK )
      rc = RexxCallBack( curr->func, num_args, argv, &rcode, &RetStr );
#else
      rc = rcode = 0;
#endif
      if ( RetStr.strptr )
         RexxFreeMemory( RetStr.strptr );
      free( argv );
      free_cb_argv( curr );
      return rcode;
   }
   else
   {
      cbd = addCallback( curr->RxPackageGlobalData, REXXDW_MOTION_NOTIFY_EVENT_CB );
      if ( cbd == NULL )
         return -1;
      /*
       * Specify the data for this callback type
       */
      cbd->window = window;
      cbd->event.MotionNotifyCallbackData.x = x;
      cbd->event.MotionNotifyCallbackData.y = y;
      cbd->event.MotionNotifyCallbackData.state = state;
      cbd->userdata = data;
      DEBUGDUMP(fprintf(stderr,"%s-%d: In rexxdw_motion_notify_event_cb() Window: %ld X: %d Y: %d State: %d Func: %s Argc: %d\n",
                               __FILE__,__LINE__,
                               cbd->window,
                               cbd->event.MotionNotifyCallbackData.x,
                               cbd->event.MotionNotifyCallbackData.y,
                               cbd->event.MotionNotifyCallbackData.state,
                               cbd->userdata->func,
                               cbd->userdata->argc );)
      return 0;
   }
}

/****** Callbacks/DeleteEventCallback
 * NAME
 *   DeleteEventCallback
 * USAGE
 *   rcode = DeleteEventCallback(Window [,UserData[,...]])
 * DESCRIPTION
 *   Called when a DW_DELETE_EVENT signal fires on a window. This occurs when
 *   the user closes the window by clicking the close icon on the window title bar,
 *   or when dw_window_destroy() ???? is called on the window.
 * ARGUMENTS
 *   - Window     - value of the window handle in which the mouse button was released
 *   - UserData   - following optional values that the user supplied when the signal event
 *                  was connected
 * RETURN VALUE
 *   Return 1 to ensure that the default callback is not called.
 *   Return 0 to enable the default callback to be called.
 * SEE ALSO
 *   DW_window_destroy
 * SOURCE
...
Call dw_signal_connect window, !REXXDW.!DW_DELETE_EVENT, 'delete_cb', ,
   'fred'
...
delete_cb:
Parse Arg win, data
Say 'Window:' win 'deleted'
Return 1
 ******
 */
static int DWSIGNAL rexxdw_delete_event_cb( HWND window, void *data )
{
   char rx_window[16];
   PRXSTRING argv;
   REXXDWUserData *curr=(REXXDWUserData *)data;
   RXSTRING RetStr;
   int min_args=1;
   int i,rc,num_args=min_args+curr->argc;
   SHORT rcode;
   char buf[100];
   REXXDWCallbackData *cbd;
   REXXDWDATA *RexxDWData=(REXXDWDATA *)curr->RxPackageGlobalData->RXPACKAGE_tsd;

   RexxDWData->CallbackTimestamp = time(NULL);
   if ( HAVE_REXXCALLBACK )
   {
      argv = (PRXSTRING)malloc( num_args*sizeof(RXSTRING) );
      if ( argv == NULL )
         return -1;
      sprintf( rx_window, "%lu", (unsigned long)window );
      MAKERXSTRING( argv[0], rx_window, strlen( rx_window ) );
      for ( i = 0; i < curr->argc; i++ )
      {
         argv[i+min_args] = curr->argv[i];
      }

      sprintf( buf, "DW_DELETE_EVENT callback: %s()", curr->func );
      FunctionPrologue( curr->RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, buf, num_args, argv );

      MAKERXSTRING( RetStr, NULL, 0 );
#if defined( REXXCALLBACK )
      rc = RexxCallBack( curr->func, num_args, argv, &rcode, &RetStr );
#else
      rc = rcode = 0;
#endif
      if ( RetStr.strptr )
         RexxFreeMemory( RetStr.strptr );
      free( argv );
      free_cb_argv( curr );
      return rcode;
   }
   else
   {
      cbd = addCallback( curr->RxPackageGlobalData, REXXDW_DELETE_EVENT_CB );
      if ( cbd == NULL )
         return -1;
      /*
       * Specify the data for this callback type
       */
      cbd->window = window;
      cbd->userdata = data;
      DEBUGDUMP(fprintf(stderr,"%s-%d: In rexxdw_delete_event_cb() Window: %ld Func: %s Argc: %d\n",
                               __FILE__,__LINE__,
                               cbd->window,
                               cbd->userdata->func,
                               cbd->userdata->argc );)
      return TRUE; /* really need to return TRUE here */
   }
}

/****** Callbacks/ExposeEventCallback
 * NAME
 *   ExposeEventCallback
 * USAGE
 *   rcode = ExposeEventCallback(Window, X, Y, Width, Height [,UserData[,...]])
 * DESCRIPTION
 *   Called when a DW_EXPOSE_EVENT signal fires on a window. This occurs when
 *   the window is exposed.
 * ARGUMENTS
 *   - Window     - value of the window handle which is being exposed
 *   - X          - the top left X coordinate of the exposed area
 *   - Y          - the top left Y coordinate of the exposed area
 *   - Width      - the width of the exposed area
 *   - Height     - the height of the exposed area
 *   - UserData   - following optional values that the user supplied when the signal event
 *                was connected
 * RETURN VALUE
 *   Return 1 to ensure that the default callback is not called.
 *   Return 0 to enable the default callback to be called.
 * SOURCE
...
Call dw_signal_connect window, !REXXDW.!DW_EXPOSE_EVENT, 'expose_cb', ,
   'fred'
...
expose_cb:
Parse Arg win, x, y, width, height, data
Say 'Window:' win 'exposed at' x'/'y 'size:' width'/'height
Return 1
 ******
 */
static int DWSIGNAL rexxdw_expose_event_cb( HWND window, DWExpose *exp, void *data )
{
   char rx_window[16];
   char rx_x[16];
   char rx_y[16];
   char rx_width[16];
   char rx_height[16];
   PRXSTRING argv;
   REXXDWUserData *curr=(REXXDWUserData *)data;
   RXSTRING RetStr;
   int min_args=5;
   int i,rc,num_args=min_args+curr->argc;
   SHORT rcode;
   char buf[100];
   REXXDWCallbackData *cbd;
   REXXDWDATA *RexxDWData=(REXXDWDATA *)curr->RxPackageGlobalData->RXPACKAGE_tsd;

   RexxDWData->CallbackTimestamp = time(NULL);
   if ( HAVE_REXXCALLBACK )
   {
      argv = (PRXSTRING)malloc( num_args*sizeof(RXSTRING) );
      if ( argv == NULL )
         return -1;
      sprintf( rx_window, "%lu", (unsigned long)window );
      sprintf( rx_x, "%d", exp->x );
      sprintf( rx_y, "%d", exp->y );
      sprintf( rx_width, "%d", exp->width );
      sprintf( rx_height, "%d", exp->height );
      MAKERXSTRING( argv[0], rx_window, strlen( rx_window ) );
      MAKERXSTRING( argv[1], rx_x, strlen( rx_x ) );
      MAKERXSTRING( argv[2], rx_y, strlen( rx_y ) );
      MAKERXSTRING( argv[3], rx_width, strlen( rx_width ) );
      MAKERXSTRING( argv[4], rx_height, strlen( rx_height ) );
      for ( i = 0; i < curr->argc; i++ )
      {
         argv[i+min_args] = curr->argv[i];
      }
      DEBUGDUMP(fprintf(stderr,"%s-%d: In rexxdw_expose_event_cb() Window: %ld X: %d Y: %d Width: %d Height: %d Func: %s Argc: %d\n",
                               __FILE__,__LINE__,
                               rx_window,
                               rx_x,
                               rx_y,
                               rx_width,
                               rx_height,
                               curr->func,
                               curr->argc );)

      sprintf( buf, "DW_EXPOSE_EVENT callback: %s()", curr->func );
      FunctionPrologue( curr->RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, buf, num_args, argv );

      MAKERXSTRING( RetStr, NULL, 0 );
#if defined( REXXCALLBACK )
      rc = RexxCallBack( curr->func, num_args, argv, &rcode, &RetStr );
#else
      rc = rcode = 0;
#endif
      if ( RetStr.strptr )
         RexxFreeMemory( RetStr.strptr );
      free( argv );
      free_cb_argv( curr );
      return rcode;
   }
   else
   {
      cbd = addCallback( curr->RxPackageGlobalData, REXXDW_EXPOSE_EVENT_CB );
      if ( cbd == NULL )
         return -1;
      /*
       * Specify the data for this callback type
       */
      cbd->window = window;
      cbd->event.ExposeCallbackData.x = exp->x;
      cbd->event.ExposeCallbackData.y = exp->y;
      cbd->event.ExposeCallbackData.width = exp->width;
      cbd->event.ExposeCallbackData.height = exp->height;
      cbd->userdata = data;
      DEBUGDUMP(fprintf(stderr,"%s-%d: In rexxdw_expose_event_cb() Window: %ld X: %d Y: %d Width: %d Height: %d Func: %s Argc: %d\n",
                               __FILE__,__LINE__,
                               cbd->window,
                               cbd->event.ExposeCallbackData.x,
                               cbd->event.ExposeCallbackData.y,
                               cbd->event.ExposeCallbackData.width,
                               cbd->event.ExposeCallbackData.height,
                               cbd->userdata->func,
                               cbd->userdata->argc );)
      return 0;
   }
}

/****** Callbacks/ClickedEventCallback
 * NAME
 *   ClickedEventCallback
 * USAGE
 *   rcode = ClickedEventCallback(Window [,UserData[,...]])
 * DESCRIPTION
 *   Called when a DW_CLICKED_EVENT signal fires on a window. This occurs when
 *   the user clicks in the window; usually a button window.
 * ARGUMENTS
 *   - Window     - value of the window handle that was clicked
 *   - UserData   - following optional values that the user supplied when the signal event
 *                  was connected
 * RETURN VALUE
 *   Return 1 to ensure that the default callback is not called.
 *   Return 0 to enable the default callback to be called.
 * SOURCE
...
Call dw_signal_connect window, !REXXDW.!DW_CLICKED_EVENT, 'clicked_cb', ,
   'fred'
...
clicked_cb:
Parse Arg win, data
Say 'Window:' win 'clicked'
Return 1
 ******
 */
static int DWSIGNAL rexxdw_clicked_event_cb( HWND window, void *data )
{
   char buf[100];
   char rx_window[16];
   PRXSTRING argv;
   REXXDWUserData *curr=(REXXDWUserData *)data;
   RXSTRING RetStr;
   int min_args=1;
   int i,rc,num_args=min_args+curr->argc;
   SHORT rcode;
   REXXDWCallbackData *cbd;
   REXXDWDATA *RexxDWData=(REXXDWDATA *)curr->RxPackageGlobalData->RXPACKAGE_tsd;

   RexxDWData->CallbackTimestamp = time(NULL);
   if ( HAVE_REXXCALLBACK )
   {
      argv = (PRXSTRING)malloc( num_args*sizeof(RXSTRING) );
      if ( argv == NULL )
         return -1;
      sprintf( rx_window, "%lu", (unsigned long)window );
      MAKERXSTRING( argv[0], rx_window, strlen( rx_window ) );
      for ( i = 0; i < curr->argc; i++ )
      {
         argv[i+min_args] = curr->argv[i];
      }

      sprintf( buf, "DW_CLICKED_EVENT callback: %s()", curr->func );
      FunctionPrologue( curr->RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, buf, num_args, argv );

      MAKERXSTRING( RetStr, NULL, 0 );
#if defined( REXXCALLBACK )
      rc = RexxCallBack( curr->func, num_args, argv, &rcode, &RetStr );
#else
      rc = rcode = 0;
#endif
      if ( RetStr.strptr )
         RexxFreeMemory( RetStr.strptr );
      free( argv );
      free_cb_argv( curr );
      return rcode;
   }
   else
   {
      cbd = addCallback( curr->RxPackageGlobalData, REXXDW_CLICKED_EVENT_CB );
      if ( cbd == NULL )
         return -1;
      /*
       * Specify the data for this callback type
       */
      cbd->window = window;
      cbd->userdata = data;
      DEBUGDUMP(fprintf(stderr,"%s-%d: In rexxdw_clicked_event_cb() Window: %ld Func: %s Argc: %d\n",
                               __FILE__,__LINE__,
                               cbd->window,
                               cbd->userdata->func,
                               cbd->userdata->argc );)
      return 0;
   }
}

/****** Callbacks/ItemEnterEventCallback
 * NAME
 *   ItemEnterEventCallback
 * USAGE
 *   rcode = ItemEnterEventCallback(Window, Text [,UserData[,...]])
 * DESCRIPTION
 *   Called when a DW_ITEM_ENTER_EVENT signal fires on a window. This occurs when
 *   the user double-clicks or presses ENTER key in a container window.
 * ARGUMENTS
 *   - Window     - value of the window handle that was clicked
 *   - Text       - the string set by a call to DW_container_set_row_title()
 *   - UserData   - following optional values that the user supplied when the signal event
 *                  was connected
 * RETURN VALUE
 *   Return 1 to ensure that the default callback is not called.
 *   Return 0 to enable the default callback to be called.
 * SEE ALSO
 *   DW_container_set_row_title
 * SOURCE
...
Call dw_signal_connect window, !REXXDW.!DW_ITEM_ENTER_EVENT, 'itementer_cb', ,
   'fred'
...
itementer_cb:
Parse Arg win, text, data
Say 'Item text' text 'entered in Window:' win 'userdata:' data
Return 1
 ******
 */
static int DWSIGNAL rexxdw_item_enter_event_cb( HWND window, char *text, void *data )
{
   char rx_window[16];
   PRXSTRING argv;
   REXXDWUserData *curr=(REXXDWUserData *)data;
   RXSTRING RetStr;
   int min_args=2;
   int i,rc,num_args=min_args+curr->argc;
   SHORT rcode;
   char *tmptext=NULL;
   char buf[100];
   REXXDWCallbackData *cbd;
   REXXDWDATA *RexxDWData=(REXXDWDATA *)curr->RxPackageGlobalData->RXPACKAGE_tsd;

   RexxDWData->CallbackTimestamp = time(NULL);
   if ( HAVE_REXXCALLBACK )
   {
      argv = (PRXSTRING)malloc( num_args*sizeof(RXSTRING) );
      if ( argv == NULL )
         return -1;
      sprintf( rx_window, "%lu", (unsigned long)window );
      MAKERXSTRING( argv[0], rx_window, strlen( rx_window ) );
      /*
       * Need to allocate memory for "text" argument
       */
      if ( text )
      {
         int len = strlen( text );
         tmptext = malloc( len + 1 );
         if ( tmptext )
         {
            strcpy( tmptext, text );
            MAKERXSTRING( argv[1], tmptext, len );
         }
      }
      else
      {
         MAKERXSTRING( argv[1], "", 0 );
      }
      for ( i = 0; i < curr->argc; i++ )
      {
         argv[i+min_args] = curr->argv[i];
      }

      sprintf( buf, "DW_ITEM_ENTER_EVENT callback: %s()", curr->func );
      FunctionPrologue( curr->RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, buf, num_args, argv );

      MAKERXSTRING( RetStr, NULL, 0 );
#if defined( REXXCALLBACK )
      rc = RexxCallBack( curr->func, num_args, argv, &rcode, &RetStr );
#else
      rc = rcode = 0;
#endif
      if ( RetStr.strptr )
         RexxFreeMemory( RetStr.strptr );
      free( argv );
      free_cb_argv( curr );
      if ( tmptext ) free( tmptext );
      return rcode;
   }
   else
   {
      cbd = addCallback( curr->RxPackageGlobalData, REXXDW_ITEM_ENTER_EVENT_CB );
      if ( cbd == NULL )
         return -1;
      /*
       * Specify the data for this callback type
       */
      cbd->window = window;
      cbd->event.ItemEnterCallbackData.text = text;
      cbd->userdata = data;
      DEBUGDUMP(fprintf(stderr,"%s-%d: In rexxdw_item_enter_event_cb() Window: %ld Text: \"%s\" Func: %s Argc: %d\n",
                               __FILE__,__LINE__,
                               cbd->window,
                               cbd->event.ItemEnterCallbackData.text,
                               cbd->userdata->func,
                               cbd->userdata->argc );)
      return 0;
   }
}

/****** Callbacks/ItemContextEventCallback
 * NAME
 *   ItemContextEventCallback
 * USAGE
 *   rcode = ItemContextEventCallback(Window, Text, X, Y, ItemData [,UserData[,...]])
 * DESCRIPTION
 *   Called when a DW_ITEM_CONTEXT_EVENT signal fires on a window. This occurs when
 *   the user clicks the right mouse button in a container or tree window.
 * ARGUMENTS
 *   - Window     - value of the window handle that was clicked
 *   - Text       - the string set by a call to DW_container_set_row_title()
 *                  or the third argument ???? to DW_tree_item_change()
 *   - X          - the X coordinate where the mouse button was pressed, relative
 *                  to the top left corner of the Window
 *   - Y          - the Y coordinate where the mouse button was pressed, relative
 *                  to the top left corner of the Window
 *   - ItemData   - ????
 *   - UserData   - following optional values that the user supplied when the signal event
 *                  was connected
 * RETURN VALUE
 *   Return 1 to ensure that the default callback is not called.
 *   Return 0 to enable the default callback to be called.
 * SEE ALSO
 *   DW_container_set_row_title, DW_tree_item_change
 * NOTES
 *   When connecting this signal to a Tree widget, DW_signal_connect() MUST
 *   be called BEFORE the Tree is populated.
 *   When connecting this signal to a Container widget, DW_signal_connect() MUST
 *   be called AFTER the Container is populated.
 * SOURCE
...
Call dw_signal_connect window, !REXXDW.!DW_ITEM_CONTEXT_EVENT, ,
   'itemcontext_cb', 'fred'
...
itemcontext_cb:
Parse Arg win, text, x, y, itemdata, userdata
Say 'Item text' text 'entered in Window:' win 'at' x'/'y 'userdata:' userdata
Return 1
 ******
 */
static int DWSIGNAL rexxdw_item_context_event_cb( HWND window, char *text, int x, int y, void *data, void *itemdata )
{
   char rx_window[16];
   char rx_x[16];
   char rx_y[16];
   PRXSTRING argv;
   REXXDWUserData *curr=(REXXDWUserData *)data;
   RXSTRING RetStr;
   int min_args=5;
   int i,rc,num_args=min_args+curr->argc;
   SHORT rcode;
   char *tmptext=NULL;
   char buf[100];
   REXXDWCallbackData *cbd;
   REXXDWDATA *RexxDWData=(REXXDWDATA *)curr->RxPackageGlobalData->RXPACKAGE_tsd;

   RexxDWData->CallbackTimestamp = time(NULL);
   if ( HAVE_REXXCALLBACK )
   {
      argv = (PRXSTRING)malloc( num_args*sizeof(RXSTRING) );
      if ( argv == NULL )
         return -1;
      sprintf( rx_window, "%lu", (unsigned long)window );
      sprintf( rx_x, "%d", x );
      sprintf( rx_y, "%d", y );
      MAKERXSTRING( argv[0], rx_window, strlen( rx_window ) );
      /*
       * Need to allocate memory for "text" argument
       */
      if ( text )
      {
         int len = strlen( text );
         tmptext = malloc( len + 1 );
         if ( tmptext )
         {
            strcpy( tmptext, text );
            MAKERXSTRING( argv[1], tmptext, len );
         }
      }
      else
      {
         MAKERXSTRING( argv[1], "", 0 );
      }
      MAKERXSTRING( argv[2], rx_x, strlen( rx_x ) );
      MAKERXSTRING( argv[3], rx_y, strlen( rx_y ) );
      if ( itemdata )
      {
         MAKERXSTRING( argv[4], (char *)itemdata, strlen( (char *)itemdata ) );
      }
      else
      {
         MAKERXSTRING( argv[4], "", 0 );
      }
      for ( i = 0; i < curr->argc; i++ )
      {
         argv[i+min_args] = curr->argv[i];
      }

      sprintf( buf, "DW_ITEM_CONTEXT_EVENT callback: %s()", curr->func );
      FunctionPrologue( curr->RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, buf, num_args, argv );

      MAKERXSTRING( RetStr, NULL, 0 );
#if defined( REXXCALLBACK )
      rc = RexxCallBack( curr->func, num_args, argv, &rcode, &RetStr );
#else
      rc = rcode = 0;
#endif
      if ( RetStr.strptr )
         RexxFreeMemory( RetStr.strptr );
      free( argv );
      free_cb_argv( curr );
      if ( tmptext ) free( tmptext );
      return rcode;
   }
   else
   {
      cbd = addCallback( curr->RxPackageGlobalData, REXXDW_ITEM_CONTEXT_EVENT_CB );
      if ( cbd == NULL )
         return -1;
      /*
       * Specify the data for this callback type
       */
      cbd->window = window;
      cbd->event.ItemContextCallbackData.text = text;
      cbd->event.ItemContextCallbackData.x = x;
      cbd->event.ItemContextCallbackData.y = y;
      cbd->event.ItemContextCallbackData.itemdata = (char *)itemdata;
      cbd->userdata = data;
      DEBUGDUMP(fprintf(stderr,"%s-%d: In rexxdw_item_context_event_cb() Window: %ld Text: \"%s\" X: %d Y: %d Itemdata: \"%s\" Func: %s Argc: %d\n",
                               __FILE__,__LINE__,
                               cbd->window,
                               cbd->event.ItemContextCallbackData.text,
                               cbd->event.ItemContextCallbackData.x,
                               cbd->event.ItemContextCallbackData.y,
                               cbd->event.ItemContextCallbackData.itemdata,
                               cbd->userdata->func,
                               cbd->userdata->argc );)
      return 0;
   }
}

/****** Callbacks/ListSelectEventCallback
 * NAME
 *   ListSelectEventCallback
 * USAGE
 *   rcode = ListSelectEventCallback(Window, Item [,UserData[,...]])
 * DESCRIPTION
 *   Called when a DW_LIST_SELECT_EVENT signal fires on a window. This occurs when
 *   the user clicks an entry in a listbox or combobox; ie an item in a list is
 *   selected.
 * ARGUMENTS
 *   - Window     - value of the window handle that was clicked
 *   - Item       - the index of the item in the list that is selected. This
 *                  index is 0-based.
 *   - UserData   - following optional values that the user supplied when the signal event
 *                  was connected
 * RETURN VALUE
 *   Return 1 to ensure that the default callback is not called.
 *   Return 0 to enable the default callback to be called.
 * SOURCE
...
Call dw_signal_connect window, !REXXDW.!DW_LIST_SELECT_EVENT, ,
   'listselect_cb', 'fred'
...
listselect_cb:
Parse Arg win, item, userdata
Say 'Item number' item 'selected in Window:' win 'userdata:' userdata
Return 1
 ******
 */
static int DWSIGNAL rexxdw_list_select_event_cb( HWND window, int item, void *data )
{
   char rx_window[16];
   char rx_item[16];
   PRXSTRING argv;
   REXXDWUserData *curr=(REXXDWUserData *)data;
   RXSTRING RetStr;
   int min_args=2;
   int i,rc,num_args=min_args+curr->argc;
   SHORT rcode;
   char buf[100];
   REXXDWCallbackData *cbd;
   REXXDWDATA *RexxDWData=(REXXDWDATA *)curr->RxPackageGlobalData->RXPACKAGE_tsd;

   RexxDWData->CallbackTimestamp = time(NULL);
   if ( HAVE_REXXCALLBACK )
   {
      argv = (PRXSTRING)malloc( num_args*sizeof(RXSTRING) );
      if ( argv == NULL )
         return -1;
      sprintf( rx_window, "%lu", (unsigned long)window );
      sprintf( rx_item, "%d", item );
      MAKERXSTRING( argv[0], rx_window, strlen( rx_window ) );
      MAKERXSTRING( argv[1], rx_item, strlen( rx_item ) );
      for ( i = 0; i < curr->argc; i++ )
      {
         argv[i+min_args] = curr->argv[i];
      }

      sprintf( buf, "DW_LIST_SELECT_EVENT callback: %s()", curr->func );
      FunctionPrologue( curr->RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, buf, num_args, argv );

      MAKERXSTRING( RetStr, NULL, 0 );
#if defined( REXXCALLBACK )
      rc = RexxCallBack( curr->func, num_args, argv, &rcode, &RetStr );
#else
      rc = rcode = 0;
#endif
      if ( RetStr.strptr )
         RexxFreeMemory( RetStr.strptr );
      free( argv );
      free_cb_argv( curr );
      return rcode;
   }
   else
   {
      cbd = addCallback( curr->RxPackageGlobalData, REXXDW_LIST_SELECT_EVENT_CB );
      if ( cbd == NULL )
         return -1;
      /*
       * Specify the data for this callback type
       */
      cbd->window = window;
      cbd->event.ListSelectCallbackData.item = item;
      cbd->userdata = data;
      DEBUGDUMP(fprintf(stderr,"%s-%d: In rexxdw_list_select_event_cb() Window: %ld Item: %d Func: %s Argc: %d\n",
                               __FILE__,__LINE__,
                               cbd->window,
                               cbd->event.ListSelectCallbackData.item,
                               cbd->userdata->func,
                               cbd->userdata->argc );)
      return 0;
   }
}

/****** Callbacks/ItemSelectEventCallback
 * NAME
 *   ItemSelectEventCallback
 * USAGE
 *   rcode = ItemSelectEventCallback(Window, Item, Text, Itemdata [,UserData[,...]])
 * DESCRIPTION
 *   Called when a DW_ITEM_SELECT_EVENT signal fires on a window. This occurs when
 *   the user clicks the left button on an item in a container or tree window.
 *   selected.
 * ARGUMENTS
 *   - Window     - value of the window handle that was clicked
 *   - Item       - the index of the item in the list that is selected. This
 *                  index is 0-based.
 *   - Text       - the string set by a call to DW_container_set_row_title()
 *                  or the second argument to DW_tree_insert()
 *   - ItemData   - the string set by a the fifth argument to a call to DW_tree_insert()
 *   - UserData   - following optional values that the user supplied when the signal event
 *                  was connected
 * RETURN VALUE
 *   Return 1 to ensure that the default callback is not called.
 *   Return 0 to enable the default callback to be called.
 * NOTES
 *   When connecting this signal to a Tree widget, DW_signal_connect() MUST
 *   be called BEFORE the Tree is populated otherwise the signal will never
 *   fire.
 *   When connecting this signal to a Container widget, DW_signal_connect() MUST
 *   be called AFTER the Container is populated.
 *
 * SOURCE
...
Call dw_signal_connect window, !REXXDW.!DW_ITEM_SELECT_EVENT, ,
   'itemselect_cb', 'fred'
...
itemselect_cb:
Parse Arg win, item, text, userdata
Say 'Item number' item 'selected in Window:' win 'text is:' text ,
   'userdata:' userdata
Return 1
 ******
 */
static int DWSIGNAL rexxdw_item_select_event_cb( HWND window, HWND item, char *text, void *data, void *itemdata )
{
   char rx_window[16];
   char rx_item[16];
   PRXSTRING argv;
   REXXDWUserData *curr=(REXXDWUserData *)data;
   RXSTRING RetStr;
   int min_args=4;
   int i,rc,num_args=min_args+curr->argc;
   SHORT rcode;
   char *tmptext=NULL;
   char buf[100];
   REXXDWCallbackData *cbd;
   REXXDWDATA *RexxDWData=(REXXDWDATA *)curr->RxPackageGlobalData->RXPACKAGE_tsd;

   RexxDWData->CallbackTimestamp = time(NULL);
   if ( HAVE_REXXCALLBACK )
   {
      argv = (PRXSTRING)malloc( num_args*sizeof(RXSTRING) );
      if ( argv == NULL )
         return -1;
      sprintf( rx_window, "%lu", (unsigned long)window );
      sprintf( rx_item, "%lu", (unsigned long)item );
      MAKERXSTRING( argv[0], rx_window, strlen( rx_window ) );
      MAKERXSTRING( argv[1], rx_item, strlen( rx_item ) );
      /*
       * Need to allocate memory for "text" argument
       */
      if ( text )
      {
         int len = strlen( text );
         tmptext = malloc( len + 1 );
         if ( tmptext )
         {
            strcpy( tmptext, text );
            MAKERXSTRING( argv[2], tmptext, len );
         }
      }
      else
      {
         MAKERXSTRING( argv[2], "", 0 );
      }
      if ( itemdata )
      {
         MAKERXSTRING( argv[3], (char *)itemdata, strlen( (char *)itemdata ) );
      }
      else
      {
         MAKERXSTRING( argv[3], "", 0 );
      }
      for ( i = 0; i < curr->argc; i++ )
      {
         argv[i+min_args] = curr->argv[i];
      }

      sprintf( buf, "DW_ITEM_SELECT_EVENT callback: %s()", curr->func );
      FunctionPrologue( curr->RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, buf, num_args, argv );

      MAKERXSTRING( RetStr, NULL, 0 );
#if defined( REXXCALLBACK )
      rc = RexxCallBack( curr->func, num_args, argv, &rcode, &RetStr );
#else
      rc = rcode = 0;
#endif
      if ( RetStr.strptr )
         RexxFreeMemory( RetStr.strptr );
      free( argv );
      free_cb_argv( curr );
      if ( tmptext ) free( tmptext );
      return rcode;
   }
   else
   {
      cbd = addCallback( curr->RxPackageGlobalData, REXXDW_ITEM_SELECT_EVENT_CB );
      if ( cbd == NULL )
         return -1;
      /*
       * Specify the data for this callback type
       */
      cbd->window = window;
      cbd->event.ItemSelectCallbackData.item = item;

      if ( text )
         cbd->event.ItemSelectCallbackData.text = text;
      else
         cbd->event.ItemSelectCallbackData.text = "";

      if ( itemdata )
         cbd->event.ItemSelectCallbackData.itemdata = (char *)itemdata;
      else
         cbd->event.ItemSelectCallbackData.itemdata = "";
      cbd->userdata = data;
      DEBUGDUMP(fprintf(stderr,"%s-%d: In rexxdw_item_select_event_cb() Window: %ld Item: %ld Text: \"%s\" Itemdata: \"%s\" Func: %s Argc: %d\n",
                               __FILE__,__LINE__,
                               cbd->window,
                               cbd->event.ItemSelectCallbackData.item,
                               cbd->event.ItemSelectCallbackData.text,
                               cbd->event.ItemSelectCallbackData.itemdata,
                               cbd->userdata->func,
                               cbd->userdata->argc );)
      return 0;
   }
}

/****** Callbacks/SetFocusEventCallback
 * NAME
 *   SetFocusEventCallback
 * USAGE
 *   rcode = SetFocusEventCallback(Window [,UserData[,...]])
 * DESCRIPTION
 *   Called when a DW_SET_FOCUS_EVENT signal fires on a window. This occurs when
 *   the user gives focus to a window, either by clicking the left button in
 *   the window or by tabbing to the window.
 *   selected.
 * ARGUMENTS
 *   - Window     - value of the window handle that was clicked
 *   - UserData   - following optional values that the user supplied when the signal event
 *                  was connected
 * RETURN VALUE
 *   Return 1 to ensure that the default callback is not called.
 *   Return 0 to enable the default callback to be called.
 * SOURCE
...
Call dw_signal_connect window, !REXXDW.!DW_SET_FOCUS_EVENT, 'setfocus_cb', ,
   'fred'
...
setfocus_cb:
Parse Arg win, userdata
Say 'Window:' win 'now has focus userdata:' userdata
Return 1
 ******
 */
static int DWSIGNAL rexxdw_set_focus_event_cb( HWND window, void *data )
{
   char rx_window[16];
   PRXSTRING argv;
   REXXDWUserData *curr=(REXXDWUserData *)data;
   RXSTRING RetStr;
   int min_args=1;
   int i,rc,num_args=min_args+curr->argc;
   SHORT rcode;
   char buf[100];
   REXXDWCallbackData *cbd;
   REXXDWDATA *RexxDWData=(REXXDWDATA *)curr->RxPackageGlobalData->RXPACKAGE_tsd;

   RexxDWData->CallbackTimestamp = time(NULL);
   if ( HAVE_REXXCALLBACK )
   {
      argv = (PRXSTRING)malloc( num_args*sizeof(RXSTRING) );
      if ( argv == NULL )
         return -1;
      sprintf( rx_window, "%lu", (unsigned long)window );
      MAKERXSTRING( argv[0], rx_window, strlen( rx_window ) );
      for ( i = 0; i < curr->argc; i++ )
      {
         argv[i+min_args] = curr->argv[i];
      }

      sprintf( buf, "DW_SET_FOCUS_EVENT callback: %s()", curr->func );
      FunctionPrologue( curr->RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, buf, num_args, argv );

      MAKERXSTRING( RetStr, NULL, 0 );
#if defined( REXXCALLBACK )
      rc = RexxCallBack( curr->func, num_args, argv, &rcode, &RetStr );
#else
      rc = rcode = 0;
#endif
      if ( RetStr.strptr )
         RexxFreeMemory( RetStr.strptr );
      free( argv );
      free_cb_argv( curr );
      return rcode;
   }
   else
   {
      cbd = addCallback( curr->RxPackageGlobalData, REXXDW_SET_FOCUS_EVENT_CB );
      if ( cbd == NULL )
         return -1;
      /*
       * Specify the data for this callback type
       */
      cbd->window = window;
      cbd->userdata = data;
      DEBUGDUMP(fprintf(stderr,"%s-%d: In rexxdw_set_focus_event_cb() Window: %ld Func: %s Argc: %d\n",
                               __FILE__,__LINE__,
                               cbd->window,
                               cbd->userdata->func,
                               cbd->userdata->argc );)
      return 0;
   }
}

/****** Callbacks/ValueChangedEventCallback
 * NAME
 *   ValueChangedEventCallback
 * USAGE
 *   rcode = ValueChangedEventCallback(Window, Value [,UserData[,...]])
 * DESCRIPTION
 *   Called when a DW_VALUE_CHANGED_EVENT signal fires on a window. This occurs when
 *   the user adjusts a scrollbar, or a slider control.
 * ARGUMENTS
 *   - Window     - value of the window handle that was clicked
 *   - Value      - the new value of the top of the scrollbar thumb in units defined
 *                  by DW_scrollbar_set_range()
 *   - UserData   - following optional values that the user supplied when the signal event
 *                  was connected
 * RETURN VALUE
 *   Return 1 to ensure that the default callback is not called.
 *   Return 0 to enable the default callback to be called.
 * SEE ALSO
 *   DW_scrollbar_set_range, DW_slider_set_pos
 * SOURCE
...
Call dw_signal_connect window, !REXXDW.!DW_VALUE_CHANGED_EVENT, ,
   'valuechanged_cb', 'fred'
...
valuechanged_cb:
Parse Arg win, val, userdata
Say 'The top of the scrollbar thumb is now' val 'in Window:' win ,
   'userdata:' userdata
Return 1
 ******
 */
static int DWSIGNAL rexxdw_value_changed_event_cb( HWND window, int val, void *data )
{
   char rx_window[16];
   char rx_val[16];
   PRXSTRING argv;
   REXXDWUserData *curr=(REXXDWUserData *)data;
   RXSTRING RetStr;
   int min_args=2;
   int i,rc,num_args=min_args+curr->argc;
   SHORT rcode;
   char buf[100];
   REXXDWCallbackData *cbd;
   REXXDWDATA *RexxDWData=(REXXDWDATA *)curr->RxPackageGlobalData->RXPACKAGE_tsd;

   RexxDWData->CallbackTimestamp = time(NULL);
   if ( HAVE_REXXCALLBACK )
   {
      argv = (PRXSTRING)malloc( num_args*sizeof(RXSTRING) );
      if ( argv == NULL )
         return -1;
      sprintf( rx_window, "%lu", (unsigned long)window );
      sprintf( rx_val, "%d", val );
      MAKERXSTRING( argv[0], rx_window, strlen( rx_window ) );
      MAKERXSTRING( argv[1], rx_val, strlen( rx_val ) );
      for ( i = 0; i < curr->argc; i++ )
      {
         argv[i+min_args] = curr->argv[i];
      }

      sprintf( buf, "DW_VALUE_CHANGED_EVENT callback: %s()", curr->func );
      FunctionPrologue( curr->RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, buf, num_args, argv );

      MAKERXSTRING( RetStr, NULL, 0 );
#if defined( REXXCALLBACK )
      rc = RexxCallBack( curr->func, num_args, argv, &rcode, &RetStr );
#else
      rc = rcode = 0;
#endif
      if ( RetStr.strptr )
         RexxFreeMemory( RetStr.strptr );
      free( argv );
      free_cb_argv( curr );
      return rcode;
   }
   else
   {
      cbd = addCallback( curr->RxPackageGlobalData, REXXDW_VALUE_CHANGED_EVENT_CB );
      if ( cbd == NULL )
         return -1;
      /*
       * Specify the data for this callback type
       */
      cbd->window = window;
      cbd->event.ValueChangedCallbackData.val = val;
      cbd->userdata = data;
      DEBUGDUMP(fprintf(stderr,"%s-%d: In rexxdw_value_changed_event_cb() Window: %ld Value: %d Func: %s Argc: %d\n",
                               __FILE__,__LINE__,
                               cbd->window,
                               cbd->event.ValueChangedCallbackData.val,
                               cbd->userdata->func,
                               cbd->userdata->argc );)
      return 0;
   }
}

/****** Callbacks/SwitchPageEventCallback
 * NAME
 *   SwitchPageEventCallback
 * USAGE
 *   rcode = SwitchPageEventCallback(Window, Page [,UserData[,...]])
 * DESCRIPTION
 *   Called when a DW_SWITCH_PAGE_EVENT signal fires on a window. This occurs when
 *   the user clicks on a tab in a notebook.
 * ARGUMENTS
 *   - Window     - value of the window handle that was clicked
 *   - Page       - the index of the page of the notebook selected. Page numbers are
 *                  0 based.
 *   - UserData   - following optional values that the user supplied when the signal event
 *                  was connected
 * RETURN VALUE
 *   Return 1 to ensure that the default callback is not called.
 *   Return 0 to enable the default callback to be called.
 * SOURCE
...
Call dw_signal_connect window, !REXXDW.!DW_SWITCH_PAGE_EVENT, ,
   'switchpage_cb', 'fred'
...
switchpage_cb:
Parse Arg win, page, userdata
Say 'Page' page 'of Window:' win 'selected. userdata:' userdata
Return 1
 ******
 */
static int DWSIGNAL rexxdw_switch_page_event_cb( HWND window, unsigned long page, void *data )
{
   char rx_window[16];
   char rx_page[16];
   PRXSTRING argv;
   REXXDWUserData *curr=(REXXDWUserData *)data;
   RXSTRING RetStr;
   int min_args=2;
   int i,rc,num_args=min_args+curr->argc;
   SHORT rcode;
   char buf[100];
   REXXDWCallbackData *cbd;
   REXXDWDATA *RexxDWData=(REXXDWDATA *)curr->RxPackageGlobalData->RXPACKAGE_tsd;

   RexxDWData->CallbackTimestamp = time(NULL);
   if ( HAVE_REXXCALLBACK )
   {
      argv = (PRXSTRING)malloc( num_args*sizeof(RXSTRING) );
      if ( argv == NULL )
         return -1;
      sprintf( rx_window, "%lu", (unsigned long)window );
      sprintf( rx_page, "%lu", page );
      MAKERXSTRING( argv[0], rx_window, strlen( rx_window ) );
      MAKERXSTRING( argv[1], rx_page, strlen( rx_page ) );
      for ( i = 0; i < curr->argc; i++ )
      {
         argv[i+min_args] = curr->argv[i];
      }

      sprintf( buf, "DW_SWITCH_PAGE_EVENT callback: %s()", curr->func );
      FunctionPrologue( curr->RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, buf, num_args, argv );

      MAKERXSTRING( RetStr, NULL, 0 );
#if defined( REXXCALLBACK )
      rc = RexxCallBack( curr->func, num_args, argv, &rcode, &RetStr );
#else
      rc = rcode = 0;
#endif
      if ( RetStr.strptr )
         RexxFreeMemory( RetStr.strptr );
      free( argv );
      free_cb_argv( curr );
      return rcode;
   }
   else
   {
      cbd = addCallback( curr->RxPackageGlobalData, REXXDW_SWITCH_PAGE_EVENT_CB );
      if ( cbd == NULL )
         return -1;
      /*
       * Specify the data for this callback type
       */
      cbd->window = window;
      cbd->event.SwitchPageCallbackData.page = page;
      cbd->userdata = data;
      DEBUGDUMP(fprintf(stderr,"%s-%d: In rexxdw_switch_page_event_cb() Window: %ld Page: %ld Func: %s Argc: %d\n",
                               __FILE__,__LINE__,
                               cbd->window,
                               cbd->event.SwitchPageCallbackData.page,
                               cbd->userdata->func,
                               cbd->userdata->argc );)
      return 0;
   }
}

/****** Callbacks/ColumnClickEventCallback
 * NAME
 *   ColumnClickEventCallback
 * USAGE
 *   rcode = ColumnClickEventCallback(Window, Column [,UserData[,...]])
 * DESCRIPTION
 *   Called when a DW_COLUMN_CLICK_EVENT signal fires on a window. This occurs when
 *   the user clicks on a column heading of a container window.
 *   selected.
 * ARGUMENTS
 *   - Window     - value of the window handle that was clicked
 *   - Column     - the index of the column of the container selected. Column numbers are
 *                  0 based.
 *   - UserData   - following optional values that the user supplied when the signal event
 *                  was connected
 * RETURN VALUE
 *   Return 1 to ensure that the default callback is not called.
 *   Return 0 to enable the default callback to be called.
 * SOURCE
...
Call dw_signal_connect window, !REXXDW.!DW_COLUMN_CLICK_EVENT, ,
   'columnclick_cb', 'fred'
...
columnclick_cb:
Parse Arg win, column, userdata
Say 'Column' column 'of Window:' win 'selected. userdata:' userdata
Return 1
 ******
 */
static int DWSIGNAL rexxdw_column_click_event_cb( HWND window, int column_num, void *data )
{
   char rx_window[16];
   char rx_column_num[16];
   PRXSTRING argv;
   REXXDWUserData *curr=(REXXDWUserData *)data;
   RXSTRING RetStr;
   int min_args=2;
   int i,rc,num_args=min_args+curr->argc;
   SHORT rcode;
   char buf[100];
   REXXDWCallbackData *cbd;
   REXXDWDATA *RexxDWData=(REXXDWDATA *)curr->RxPackageGlobalData->RXPACKAGE_tsd;

   RexxDWData->CallbackTimestamp = time(NULL);
   if ( HAVE_REXXCALLBACK )
   {
      argv = (PRXSTRING)malloc( num_args*sizeof(RXSTRING) );
      if ( argv == NULL )
         return -1;
      sprintf( rx_window, "%lu", (unsigned long)window );
      sprintf( rx_column_num, "%d", column_num );
      MAKERXSTRING( argv[0], rx_window, strlen( rx_window ) );
      MAKERXSTRING( argv[1], rx_column_num, strlen( rx_column_num ) );
      for ( i = 0; i < curr->argc; i++ )
      {
         argv[i+min_args] = curr->argv[i];
      }

      sprintf( buf, "DW_COLUMN_CLICK_EVENT callback: %s()", curr->func );
      FunctionPrologue( curr->RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, buf, num_args, argv );

      MAKERXSTRING( RetStr, NULL, 0 );
#if defined( REXXCALLBACK )
      rc = RexxCallBack( curr->func, num_args, argv, &rcode, &RetStr );
#else
      rc = rcode = 0;
#endif
      if ( RetStr.strptr )
         RexxFreeMemory( RetStr.strptr );
      free( argv );
      free_cb_argv( curr );
      return rcode;
   }
   else
   {
      cbd = addCallback( curr->RxPackageGlobalData, REXXDW_SWITCH_PAGE_EVENT_CB );
      if ( cbd == NULL )
         return -1;
      /*
       * Specify the data for this callback type
       */
      cbd->window = window;
      cbd->event.ColumnClickCallbackData.column_num = column_num;
      cbd->userdata = data;
      DEBUGDUMP(fprintf(stderr,"%s-%d: In rexxdw_column_click_event_cb() Window: %ld Column: %d Func: %s Argc: %d\n",
                               __FILE__,__LINE__,
                               cbd->window,
                               cbd->event.ColumnClickCallbackData.column_num,
                               cbd->userdata->func,
                               cbd->userdata->argc );)
      return 0;
   }
}

/****** Callbacks/TimerEventCallback
 * NAME
 *   TimerEventCallback
 * USAGE
 *   rcode = TimerEventCallback([UserData[,...]])
 * DESCRIPTION
 *   Called when a generic timer goes off. This occurs after the period specified
 *   by DW_timer_connect().
 *   selected.
 * ARGUMENTS
 *   - UserData   - optional values that the user supplied when the signal event
 *                  was connected
 * RETURN VALUE
 *   Return 1 to ensure that the timer is rearmed.
 *   Return 0 to ensure that the timer is not rearmed.
 * NOTES
 *   The return code from the function called as a result of the timer firing determines
 *   whether the timer continues to fire after the specified period.
 * SEE ALSO
 *   DW_timer_connect
 * SOURCE
...
Call dw_timer_connect window, 5000, 'timer_cb', 'fred'
...
timer_cb:
Parse Arg userdata
Say 'Our timer went off after 5 seconds' 'userdata:' userdata
Return 1
 ******
 */
/*
 * This is the common rexxdw timer callback. All timer callbacks go through this function.
 * "data" consists of the Rexx subroutine name, followed by the parameters
 *        that the user passed in dw_signal_connect()
 * All of this data is added to a queue which is cleared by dw_main_iteration()
 */
int DWSIGNAL rexxdw_timer_cb( void *data )
{
   PRXSTRING argv;
   REXXDWUserData *curr=(REXXDWUserData *)data;
   RXSTRING RetStr;
   int min_args=0;
   int i,rc,num_args=min_args+curr->argc;
   SHORT rcode;
   char buf[100];
   REXXDWCallbackData *cbd;

   if ( HAVE_REXXCALLBACK )
   {
      argv = (PRXSTRING)malloc( num_args*sizeof(RXSTRING) );
      if ( argv == NULL )
         return -1;
      for ( i = 0; i < curr->argc; i++ )
      {
         argv[i+min_args] = curr->argv[i];
      }

      sprintf( buf, "Timer EVENT callback: %s()", curr->func );
      FunctionPrologue( curr->RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, buf, num_args, argv );

      MAKERXSTRING( RetStr, NULL, 0 );
#if defined( REXXCALLBACK )
      rc = RexxCallBack( curr->func, num_args, argv, &rcode, &RetStr );
#else
      rc = rcode = 0;
#endif
      if ( RetStr.strptr )
         RexxFreeMemory( RetStr.strptr );
      free( argv );
      free_cb_argv( curr );
      return rcode;
   }
   else
   {
      cbd = addCallback( curr->RxPackageGlobalData, REXXDW_MAX_CB+1 );
      if ( cbd == NULL )
         return -1;
      cbd->window = NULL;
      cbd->userdata = data;
      DEBUGDUMP(fprintf(stderr,"%s-%d: In rexxdw_timer_cb() Func: %s Argc: %d\n",
                               __FILE__,__LINE__,
                               cbd->userdata->func,
                               cbd->userdata->argc );)
      /*
       * returning 0 here means that the timer will not go off again; you have to
       * explicitly call dw_timer_connect() again
       */
      return 0;
   }
}

/****h* RexxDW/Functions
 * DESCRIPTION
 *   The following external functions comprise Rexx/DW.
 ******
 */

/****h* Functions/Widgets
 * DESCRIPTION
 *   Widgets are the fundamental display items in RexxDW. RexxDW supports a range
 *   of widgets to enable a rich GUI interface for your Rexx programs.
 *
 * NOTES
 *   RexxDW supports all common widget types as follows:
 *      box
 *      button
 *      checkbox/radiobutton
 *      container/filesystem
 *      dialog
 *      rendering
 *      entryfield
 *      listbox/combobox
 *      menu
 *      multiline edit
 *      notebook
 *      percent
 *      scrollbar
 *      slider
 *      spinbutton
 *      splitbar
 *      text
 *      tree
 *      window
 ******
 */

/****h* Widgets/Box
 * DESCRIPTION
 *   Boxes are used to layout the GUI display. Widgets are packed into a
 *   box to position that widget. A box can be horizontal or vertical, which
 *   determines the arrangement of the widgets packed into the box.
 *   So to arrange two widgets side-by-side, you would create a horizontal box,
 *   your two widgets, and pack those two widgets into the box and they appear
 *   beside each other.
 ******
 */

/****f* Box/DW_box_new
 * NAME
 *   DW_box_new
 * SYNOPSIS
 *   win = dw_box_new( Orientation )
 * FUNCTION
 *   Creates a new generic box window.
 *   A box is the basic widget for arranging the location and position of
 *   other widgets.
 * ARGUMENTS
 *   - Orientation- !REXXDW.!DW_VERT to indicate a vertical box
 *                  !REXXDW.!DW_HORZ to indicate a horizontal box
 * RESULT
 *   A window identifier.
 * SEE ALSO
 *   DW_groupbox_new, WidgetOrientation
 * NOTES
 *   The orientation of a box determines the orientation of the widgets that are
 *   packed into this box. Creating a horizontal box and packing two buttons into it
 *   will result in the two buttons being placed side by side.
 * SOURCE
 *
...
box = dw_box_new( !REXXDW.!DW_VERT )
Call dw_box_pack_start win, box, 0, 0, !REXXDW.!DW_EXPAND_HORZ, ,
   !REXXDW.!DW_EXPAND_VERT, 0
 ******
 */
rxfunc( DW_box_new )
{
   HWND rc; /* function return variable */
   int type; /* local variable for arg[0] */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 1, 1 ) )
      return( 1 );
   if ( RxStrToInt( RxPackageGlobalData, &argv[0], (RX_INT *)&type ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   rc = dw_box_new( type, 0 );
   return RxReturnUnsignedNumber( RxPackageGlobalData, retstr, (ULONG)rc );
}

/****** Box/DW_scrollbox_new
 * NAME
 *   DW_scrollbox_new
 * SYNOPSIS
 *   win = dw_scrollbox_new( Orientation )
 * FUNCTION
 *   Creates a new generic box window with scrollbars.
 *   A box is the basic widget for arranging the location and position of
 *   other widgets.
 * ARGUMENTS
 *   - Orientation- !REXXDW.!DW_VERT to indicate a vertical box
 *                  !REXXDW.!DW_HORZ to indicate a horizontal box
 * RESULT
 *   A window identifier.
 * SEE ALSO
 *   DW_box_new, DW_groupbox_new, WidgetOrientation
 * NOTES
 *   The orientation of a box determines the orientation of the widgets that are
 *   packed into this box. Creating a horizontal box and packing two buttons into it
 *   will result in the two buttons being placed side by side.
 * SOURCE
 *
...
box = dw_scrollbox_new( !REXXDW.!DW_VERT )
Call dw_box_pack_start win, box, 0, 0, !REXXDW.!DW_EXPAND_HORZ, ,
   !REXXDW.!DW_EXPAND_VERT, 0
 ******
 */
rxfunc( DW_scrollbox_new )
{
   HWND rc; /* function return variable */
   int type; /* local variable for arg[0] */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 1, 1 ) )
      return( 1 );
   if ( RxStrToInt( RxPackageGlobalData, &argv[0], (RX_INT *)&type ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   rc = dw_scrollbox_new( type, 0 );
   return RxReturnUnsignedNumber( RxPackageGlobalData, retstr, (ULONG)rc );
}

/****f* Box/DW_groupbox_new
 * NAME
 *   DW_groupbox_new
 * SYNOPSIS
 *   win = dw_groupbox_new( Orientation, Title[, FontFlag] )
 * FUNCTION
 *   Creates a new generic groupbox window.
 *   A box is the basic widget for arranging the location and position of
 *   other widgets. The difference between a box and a groupbox is that
 *   a groupbox contains a border and a Title in the top left corner.
 * ARGUMENTS
 *   - Orientation- !REXXDW.!DW_VERT to indicate a vertical box
 *                  !REXXDW.!DW_HORZ to indicate a horizontal box
 *   - Title      - the text to display as the box title
 *   - FontFlag   - Optional. Specify !REXXDW.!DW_FONT_BOLD or !REXXDW.!DW_FONT_ITALIC
 *                  to set the font attributes of Title. If tou require bold and italic
 *                  or them together with DW_or.
 *                  Specify !REXXDW.!DW_FONT_NORMAL or don't specify a
 *                  value to use the normal, default font.
 * RESULT
 *   A window identifier.
 * SEE ALSO
 *   DW_box_new, WidgetOrientation
 * NOTES
 *   The orientation of a box determines the orientation of the widgets that are
 *   packed into this box. Creating a horizontal box and packing two buttons into it
 *   will result in the two buttons being placed side by side.
 * SOURCE
 *
...
box = dw_groupbox_new( !REXXDW.!DW_VERT, 'Editor', dw_or( !REXXDW.!DW_FONT_BOLD, !REXXDW.!DW_FONT_ITALIC) )
Call dw_box_pack_start win, box, 0, 0, !REXXDW.!DW_EXPAND_HORZ, ,
   !REXXDW.!DW_EXPAND_VERT, 0
 ******
 */
rxfunc( DW_groupbox_new )
{
   HWND rc; /* function return variable */
   int type; /* local variable for arg[0] */
   char *title; /* local variable for arg[1] */
   int bold = DW_FONT_NORMAL; /* local variable for arg[2] */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 2, 3 ) )
      return( 1 );
   if ( RxStrToInt( RxPackageGlobalData, &argv[0], (RX_INT *)&type ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   if ( RXVALIDSTRING(argv[1]) )
      title = RXSTRPTR(argv[1]);
   else
      title = "";
   if ( argc == 3 )
   {
      if ( RxStrToInt( RxPackageGlobalData, &argv[2], (RX_INT *)&bold ) == -1 )
      {
         RxDisplayError( RxPackageGlobalData, name, "*ERROR* Invalid value for argument %d.", 3 );
         return(1);
      }
      if ( bold < 0
      ||   bold > (DW_FONT_BOLD | DW_FONT_ITALIC ) )
      {
         RxDisplayError( RxPackageGlobalData, name, "*ERROR* Invalid value for argument %d.", 3 );
         return(1);
      }
   }
   rc = dw_groupbox_new( type, 0, title );
   if ( rc != 0
   &&   bold != DW_FONT_NORMAL )
   {
      char *oldfont;
      char *newfont;
      oldfont = dw_window_get_font( rc );
      if ( oldfont )
      {
         newfont = malloc( strlen(oldfont) + 13 );
         if ( newfont )
         {
            strcpy( newfont, oldfont );
            if ( (bold & DW_FONT_BOLD) == DW_FONT_BOLD )
               strcat( newfont, " Bold" );
            if ( (bold & DW_FONT_ITALIC) == DW_FONT_ITALIC )
               strcat( newfont, " Italic" );
            dw_window_set_font( rc, newfont );
            free( newfont);
         }
      }
   }
   return RxReturnUnsignedNumber( RxPackageGlobalData, retstr, (ULONG)rc );
}

/****f* Box/DW_box_pack_start
 * NAME
 *   DW_box_pack_start
 * SYNOPSIS
 *   dw_box_pack_start( ParentBox, BoxToPack, BoxWidth, BoxHeight, HorzExpand, VertExpand, Padding )
 * FUNCTION
 *   Packs the BoxToPack at the start of the ParentBox; ie at the left of a horizontal
 *   ParentBox, or the top of a vertical ParentBox. The height and width of the box can
 *   be specified as is the ability to allow the box to expand horizontally or vertically.
 * ARGUMENTS
 *   - ParentBox  - the box into which the BoxToPack is packed
 *   - BoxToPack  - the box that requires packing
 *   - BoxWidth   - the initial width of the box in pixels, or !REXXDW.!DW_SIZE_AUTO
 *   - BoxHeight  - the initial height of the box in pixels, or !REXXDW.!DW_SIZE_AUTO
 *   - HorzExpand - indicate if the box can be expanded horizontally.
 *                  Specify either !REXXDW.!DW_EXPAND_HORZ or !REXXDW.!DW_DONT_EXPAND_HORZ
 *   - VertExpand - indicate if the box can be expanded vertically.
 *                  Specify either !REXXDW.!DW_EXPAND_VERT or !REXXDW.!DW_DONT_EXPAND_VERT
 *   - Padding    - The number of pixels of padding to add around all sides of BoxToPack
 * RESULT
 *   No return result.
 * SEE ALSO
 *   DW_box_pack_end, DW_Box_pack_at_index, BoxSizeValues
 * NOTES
 *   The BoxToPack argument can be 0, which will result in empty space being packed.
 *   !REXXDW.!DW_SIZE_AUTO is used to enable the dwindows packing algorith to size the box
 *   automatically
 * SOURCE
 *
...
button = dw_bitmapbutton_new_from_file( 'Quit', 0, 'quit' )
Call dw_box_pack_start win, button, 0, 32, !REXXDW.!DW_EXPAND_HORZ, ,
   !REXXDW.!DW_DONT_EXPAND_VERT, 0
 ******
 */
rxfunc( DW_box_pack_start )
{
   HWND box; /* local variable for arg[0] */
   HWND item; /* local variable for arg[1] */
   int width; /* local variable for arg[2] */
   int height; /* local variable for arg[3] */
   int hsize; /* local variable for arg[4] */
   int vsize; /* local variable for arg[5] */
   int pad; /* local variable for arg[6] */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 7, 7 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&box ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   if ( RxStrToULong( RxPackageGlobalData, &argv[1], (RX_ULONG *)&item ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 2 );
      return(1);
   }
   if ( RxStrToInt( RxPackageGlobalData, &argv[2], (RX_INT *)&width ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 3 );
      return(1);
   }
   if ( RxStrToInt( RxPackageGlobalData, &argv[3], (RX_INT *)&height ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 4 );
      return(1);
   }
   if ( RxStrToInt( RxPackageGlobalData, &argv[4], (RX_INT *)&hsize ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 5 );
      return(1);
   }
   if ( RxStrToInt( RxPackageGlobalData, &argv[5], (RX_INT *)&vsize ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 6 );
      return(1);
   }
   if ( RxStrToInt( RxPackageGlobalData, &argv[6], (RX_INT *)&pad ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 7 );
      return(1);
   }
   dw_box_pack_start( box, item, width, height, hsize, vsize, pad );
   return RxReturnString( RxPackageGlobalData, retstr, "" );
}

/****f* Box/DW_box_pack_end
 * NAME
 *   DW_box_pack_end
 * SYNOPSIS
 *   dw_box_pack_end( ParentBox, BoxToPack, BoxWidth, BoxHeight, HorzExpand, VertExpand, Padding )
 * FUNCTION
 *   Packs the BoxToPack at the end of the ParentBox; ie at the right of a horizontal
 *   ParentBox, or the bottom of a vertical ParentBox. The height and width of the box can
 *   be specified as is the ability to allow the box to expand horizontally or vertically.
 * ARGUMENTS
 *   - ParentBox  - the box into which the BoxToPack is packed
 *   - BoxToPack  - the box that requires packing
 *   - BoxWidth   - the initial width of the box in pixels, or !REXXDW.!DW_SIZE_AUTO
 *   - BoxHeight  - the initial height of the box in pixels, or !REXXDW.!DW_SIZE_AUTO
 *   - HorzExpand - indicate if the box can be expanded horizontally.
 *                  Specify either !REXXDW.!DW_EXPAND_HORZ or !REXXDW.!DW_DONT_EXPAND_HORZ
 *   - VertExpand - indicate if the box can be expanded vertically.
 *                  Specify either !REXXDW.!DW_EXPAND_VERT or !REXXDW.!DW_DONT_EXPAND_VERT
 *   - Padding    - The number of pixels of padding to add around all sides of BoxToPack
 * RESULT
 *   No return result.
 * SEE ALSO
 *   DW_box_pack_start, DW_box_pack_at_index, BoxSizeValues
 * NOTES
 *   The BoxToPack argument can be 0, which will result in empty space being packed.
 *   !REXXDW.!DW_SIZE_AUTO is used to enable the dwindows packing algorith to size the box
 *   automatically
 * SOURCE
 *
...
button = dw_bitmapbutton_new_from_file( 'Quit', 0, 'quit' )
Call dw_box_pack_end win, button, 0, 32, !REXXDW.!DW_EXPAND_HORZ, ,
   !REXXDW.!DW_DONT_EXPAND_VERT, 0
 ******
 */
rxfunc( DW_box_pack_end )
{
   HWND box; /* local variable for arg[0] */
   HWND item; /* local variable for arg[1] */
   int width; /* local variable for arg[2] */
   int height; /* local variable for arg[3] */
   int hsize; /* local variable for arg[4] */
   int vsize; /* local variable for arg[5] */
   int pad; /* local variable for arg[6] */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 7, 7 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&box ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   if ( RxStrToULong( RxPackageGlobalData, &argv[1], (RX_ULONG *)&item ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 2 );
      return(1);
   }
   if ( RxStrToInt( RxPackageGlobalData, &argv[2], (RX_INT *)&width ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 3 );
      return(1);
   }
   if ( RxStrToInt( RxPackageGlobalData, &argv[3], (RX_INT *)&height ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 4 );
      return(1);
   }
   if ( RxStrToInt( RxPackageGlobalData, &argv[4], (RX_INT *)&hsize ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 5 );
      return(1);
   }
   if ( RxStrToInt( RxPackageGlobalData, &argv[5], (RX_INT *)&vsize ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 6 );
      return(1);
   }
   if ( RxStrToInt( RxPackageGlobalData, &argv[6], (RX_INT *)&pad ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 7 );
      return(1);
   }
   dw_box_pack_end( box, item, width, height, hsize, vsize, pad );
   return RxReturnString( RxPackageGlobalData, retstr, "" );
}

/****f* Box/DW_box_pack_at_index
 * NAME
 *   DW_box_pack_at_index
 * SYNOPSIS
 *   dw_box_pack_at_index( ParentBox, BoxToPack, Index, BoxWidth, BoxHeight, HorzExpand, VertExpand, Padding )
 * FUNCTION
 *   Packs the BoxToPack after the ?????????at the end of the ParentBox
 *   The height and width of the box can be specified as is the ability to allow the box to
 *   expand horizontally or vertically.
 * ARGUMENTS
 *   - ParentBox  - the box into which the BoxToPack is packed
 *   - BoxToPack  - the box that requires packing
 *   - Index      - ????
 *   - BoxWidth   - the initial width of the box in pixels, or !REXXDW.!DW_SIZE_AUTO
 *   - BoxHeight  - the initial height of the box in pixels, or !REXXDW.!DW_SIZE_AUTO
 *   - HorzExpand - indicate if the box can be expanded horizontally.
 *                  Specify either !REXXDW.!DW_EXPAND_HORZ or !REXXDW.!DW_DONT_EXPAND_HORZ
 *   - VertExpand - indicate if the box can be expanded vertically.
 *                  Specify either !REXXDW.!DW_EXPAND_VERT or !REXXDW.!DW_DONT_EXPAND_VERT
 *   - Padding    - The number of pixels of padding to add around all sides of BoxToPack
 * RESULT
 *   No return result.
 * SEE ALSO
 *   DW_box_pack_start, DW_box_pack_end, BoxSizeValues
 * NOTES
 *   The BoxToPack argument can be 0, which will result in empty space being packed.
 *   !REXXDW.!DW_SIZE_AUTO is used to enable the dwindows packing algorith to size the box
 *   automatically
 * SOURCE
 *
...
button = dw_bitmapbutton_new_from_file( 'Quit', 0, 'quit' )
Call dw_box_pack_at_index win, button, 2, 0, 32, !REXXDW.!DW_EXPAND_HORZ, ,
   !REXXDW.!DW_DONT_EXPAND_VERT, 0
 ******
 */
rxfunc( DW_box_pack_at_index )
{
   HWND box; /* local variable for arg[0] */
   HWND item; /* local variable for arg[1] */
   int width; /* local variable for arg[2] */
   int index; /* local variable for arg[2] */
   int height; /* local variable for arg[4] */
   int hsize; /* local variable for arg[5] */
   int vsize; /* local variable for arg[6] */
   int pad; /* local variable for arg[7] */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 8, 8 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&box ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name, "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   if ( RxStrToULong( RxPackageGlobalData, &argv[1], (RX_ULONG *)&item ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name, "*ERROR* Invalid number for argument %d.", 2 );
      return(1);
   }
   if ( RxStrToInt( RxPackageGlobalData, &argv[2], (RX_INT *)&index ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name, "*ERROR* Invalid number for argument %d.", 3 );
      return(1);
   }
   if ( RxStrToInt( RxPackageGlobalData, &argv[3], (RX_INT *)&width ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name, "*ERROR* Invalid number for argument %d.", 4 );
      return(1);
   }
   if ( RxStrToInt( RxPackageGlobalData, &argv[4], (RX_INT *)&height ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name, "*ERROR* Invalid number for argument %d.", 5 );
      return(1);
   }
   if ( RxStrToInt( RxPackageGlobalData, &argv[5], (RX_INT *)&hsize ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name, "*ERROR* Invalid number for argument %d.", 6 );
      return(1);
   }
   if ( RxStrToInt( RxPackageGlobalData, &argv[6], (RX_INT *)&vsize ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name, "*ERROR* Invalid number for argument %d.", 7 );
      return(1);
   }
   if ( RxStrToInt( RxPackageGlobalData, &argv[7], (RX_INT *)&pad ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name, "*ERROR* Invalid number for argument %d.", 8 );
      return(1);
   }
   dw_box_pack_at_index( box, item, index, width, height, hsize, vsize, pad );
   return RxReturnString( RxPackageGlobalData, retstr, "" );
}

/****f* Box/DW_box_unpack
 * NAME
 *   DW_box_unpack
 * SYNOPSIS
 *   win = dw_box_unpack( BoxToUnpack )
 * FUNCTION
 *   Unpacks the BoxToUnpack from the current parent box.
 * ARGUMENTS
 *   - BoxToUnpack - the box that is to be unpacked
 * RESULT
 *   The window identifier of the box to be unpacked.
 * SEE ALSO
 *   DW_box_unpack_at_index, DW_window_reparent
 * NOTES
 *   The window identifier returned can be used to pack into another box.
 * SOURCE
 *
...
box = dw_box_new( !REXXDW.!DW_VERT )
Call dw_box_pack_start win, box, 0, 0, !REXXDW.!DW_EXPAND_HORZ, ,
   !REXXDW.!DW_EXPAND_VERT, 0
...
win = dw_box_unpack( box )
 ******
 */
rxfunc( DW_box_unpack )
{
   HWND box; /* local variable for arg[0] */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 1, 1 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&box ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name, "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   dw_box_unpack( box );
   return RxReturnString( RxPackageGlobalData, retstr, "" );
}

/****f* Box/DW_box_unpack_at_index
 * NAME
 *   DW_box_unpack_at_index
 * SYNOPSIS
 *   win = dw_box_unpack_at_index( BoxToUnpack, Index )
 * FUNCTION
 *   Unpacks the BoxToUnpack ?????????
 * ARGUMENTS
 *   - BoxToUnpack - the box that is to be unpacked
 * RESULT
 *   The window identifier of the box to be unpacked.
 * SEE ALSO
 *   DW_box_unpack, DW_window_reparent
 * NOTES
 *   The window identifier returned can be used to pack into another box.
 * SOURCE
 *
...
box = dw_box_new( !REXXDW.!DW_VERT )
Call dw_box_pack_start win, box, 0, 0, !REXXDW.!DW_EXPAND_HORZ, ,
   !REXXDW.!DW_EXPAND_VERT, 0
...
win = dw_box_unpack_at_index( box, index )
 ******
 */
rxfunc( DW_box_unpack_at_index )
{
   HWND box; /* local variable for arg[0] */
   int index; /* local variable for arg[1] */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 2, 2 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&box ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name, "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   if ( RxStrToInt( RxPackageGlobalData, &argv[1], (RX_INT *)&index ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name, "*ERROR* Invalid number for argument %d.", 2 );
      return(1);
   }
   dw_box_unpack_at_index( box, index );
   return RxReturnString( RxPackageGlobalData, retstr, "" );
}

/****h* Widgets/Button
 * DESCRIPTION
 *   Buttons enable an action to be carried out by the user clicking the
 *   left mouse button on the widget.
 *   RexxDW supports two types of button; plain text buttons and bitmap
 *   buttons.
 ******
 */

/****f* Button/DW_button_new
 * NAME
 *   DW_button_new
 * SYNOPSIS
 *   win = dw_button_new( ButtonText, Id )
 * FUNCTION
 *   Creates a new text button.
 *   A button is a widget which contains text and can be clicked to execute
 *   some action.
 * ARGUMENTS
 *   - ButtonText - the text to display in the button
 *   - Id         - a numeric identifier used to identify the window
 * RESULT
 *   A window identifier.
 * SEE ALSO
 *   DW_bitmapbutton_new_from_file, DW_bitmapbutton_new_from_data
 * SOURCE
 *
...
button = dw_button_new( "Quit", 10 )
 ******
 */
rxfunc( DW_button_new )
{
   HWND rc; /* function return variable */
   char * text; /* local variable for arg[0] */
   unsigned long id; /* local variable for arg[1] */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 2, 2 ) )
      return( 1 );
   if ( RXVALIDSTRING(argv[0]) )
      text = RXSTRPTR(argv[0]);
   else
      text = "";
   if ( RxStrToULong( RxPackageGlobalData, &argv[1], (RX_ULONG *)&id ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 2 );
      return(1);
   }
   rc = dw_button_new( text, id );
   return RxReturnUnsignedNumber( RxPackageGlobalData, retstr, (ULONG)rc );
}

/****f* Button/DW_bitmapbutton_new_from_file
 * NAME
 *   DW_bitmapbutton_new_from_file
 * SYNOPSIS
 *   win = dw_bitmapbutton_new_from_file( BubbleText, Id, Filename )
 * FUNCTION
 *   Creates a new button window using a bitmap image.
 * ARGUMENTS
 *   - BubbleText - the text to display when the mouse is over the button
 *   - Id         - a numeric identifier used to identify the window
 *   - Filename   - the name of a file containing a valid bitmap image (.BMP, or .XPM)
 * RESULT
 *   A window identifier.
 * SEE ALSO
 *   DW_button_new, DW_bitmapbutton_new_from_data
 * SOURCE
 *
...
button = dw_bitmapbutton_new_from_file( 'Quit', 0, 'quit' )
Call dw_box_pack_start win, button, 32, 32, !REXXDW.!DW_DONT_EXPAND_HORZ, ,
   !REXXDW.!DW_DONT_EXPAND_VERT, 0
 ******
 */
rxfunc( DW_bitmapbutton_new_from_file )
{
   HWND rc; /* function return variable */
   char * text; /* local variable for arg[0] */
   unsigned long id; /* local variable for arg[1] */
   char * filename; /* local variable for arg[2] */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 3, 3 ) )
      return( 1 );
   if ( RXVALIDSTRING(argv[0]) )
      text = RXSTRPTR(argv[0]);
   else
      text = "";
   if ( RxStrToULong( RxPackageGlobalData, &argv[1], (RX_ULONG *)&id ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 2 );
      return(1);
   }
   if ( RXVALIDSTRING(argv[2]) )
      filename = RXSTRPTR(argv[2]);
   else
      filename = "";
   rc = dw_bitmapbutton_new_from_file( text, id, filename );
   return RxReturnUnsignedNumber( RxPackageGlobalData, retstr, (ULONG)rc );
}

/****f* Button/DW_bitmapbutton_new_from_data
 * NAME
 *   DW_bitmapbutton_new_from_data
 * SYNOPSIS
 *   win = dw_bitmapbutton_new_from_data( BubbleText, Id, Data )
 * FUNCTION
 *   Creates a new button window using a bitmap image.
 * ARGUMENTS
 *   - BubbleText - the text to display when the mouse is over the button
 *   - Id         - a numeric identifier used to identify the window
 *   - Data       - the exact contents of the image.
 * RESULT
 *   A window identifier.
 * SEE ALSO
 *   DW_button_new, DW_bitmapbutton_new_from_file
 * SOURCE
 *
...
ico = "0000A6D7007E"
...
button = dw_bitmapbutton_new_from_data( 'Quit', 0, x2c(ico) )
Call dw_box_pack_start win, button, 32, 32, !REXXDW.!DW_DONT_EXPAND_HORZ, ,
   !REXXDW.!DW_DONT_EXPAND_VERT, 0
 ******
 */
rxfunc( DW_bitmapbutton_new_from_data )
{
   HWND rc; /* function return variable */
   char * text; /* local variable for arg[0] */
   unsigned long id; /* local variable for arg[1] */
   char * string; /* local variable for arg[2] */
   int len;
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 3, 3 ) )
      return( 1 );
   if ( RXVALIDSTRING(argv[0]) )
      text = RXSTRPTR(argv[0]);
   else
      text = "";
   if ( RxStrToULong( RxPackageGlobalData, &argv[1], (RX_ULONG *)&id ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 2 );
      return(1);
   }
   if ( RXVALIDSTRING(argv[2]) )
   {
      string = RXSTRPTR(argv[2]);
      len = RXSTRLEN(argv[2]);
   }
   else
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Supplied image data is empty." );
      return(1);
   }
   rc = dw_bitmapbutton_new_from_data( text, id, string, len );
   return RxReturnUnsignedNumber( RxPackageGlobalData, retstr, (ULONG)rc );
}

/****h* Widgets/Calendar
 * DESCRIPTION
 *   The Calendar widget displays a month calendar from which the user can
 *   select a date.
 ******
 */

/****f* Calendar/DW_calendar_new
 * NAME
 *   DW_calendar_new
 * SYNOPSIS
 *   win = dw_calendar_new( Id )
 * FUNCTION
 *   Creates a new calendar.
 *   A calendar is a widget which allows the user to select a date.
 * ARGUMENTS
 *   - Id         - a numeric identifier used to identify the window
 * RESULT
 *   A window identifier.
 * SEE ALSO
 *   DW_calendar_get_date, DW_calendar_set_date
 * NOTES
 *   On OS/2 this is a dummy text widget. On GTK, only 1 month is ever displayed.
 *   On Windows, more than 1 month is displayed depending on whether the widget is
 *   packed with expanding on.
 * SOURCE
 *
...
calendar = dw_calendar_new( 10 )
 ******
 */
rxfunc( DW_calendar_new )
{
   HWND rc; /* function return variable */
   unsigned long id; /* local variable for arg[0] */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 1, 1 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&id ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   rc = dw_calendar_new( id );
   return RxReturnUnsignedNumber( RxPackageGlobalData, retstr, (ULONG)rc );
}

/****f* Calendar/DW_calendar_get_date
 * NAME
 *   DW_calendar_get_date
 * SYNOPSIS
 *   date = dw_calendar_get_date( Win )
 * FUNCTION
 *   Gets the selected date of the widget.
 * ARGUMENTS
 *   - Win        - the window identifier returned from DW_calendar_new()
 * RESULT
 *   The date selected in Rexx's DATE('S') format.
 * SEE ALSO
 *   DW_calendar_new, DW_calendar_set_date
 * SOURCE
 *
...
calendar = dw_calendar_new( 10 )
...
date = dw_calendar_get_date( calendar )
 ******
 */
rxfunc( DW_calendar_get_date )
{
   HWND handle; /* local variable for arg[0] */
   unsigned int year, month, day;
   char buf[9];
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 1, 1 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&handle ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   dw_calendar_get_date( handle, &year, &month, &day );
   sprintf( buf, "%4.4d%2.2d%2.2d", year, month, day );
   return RxReturnString( RxPackageGlobalData, retstr, buf );
}

/****f* Calendar/DW_calendar_set_date
 * NAME
 *   DW_calendar_set_date
 * SYNOPSIS
 *   dw_calendar_set_date( Win, Date )
 * FUNCTION
 *   Sets the date for the widget.
 * ARGUMENTS
 *   - Win        - the window identifier returned from DW_calendar_new()
 *   - Date       - the date to set the widget to. Must be in Rexx DATE( 'S' ) format.
 * RESULT
 *   None
 * SEE ALSO
 *   DW_calendar_new, DW_calendar_get_date
 * SOURCE
 *
...
calendar = dw_calendar_new( 10 )
...
date = dw_calendar_set_date( calendar, Date( 'S' ) )
 ******
 */
rxfunc( DW_calendar_set_date )
{
   HWND handle; /* local variable for arg[0] */
   int year, month, day;
   char buf[5];
   char *tmp;
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 2, 2 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&handle ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   if ( strlen( argv[1].strptr) != 8 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid length for DATE argument %d.", 2 );
      return(1);
   }
   tmp = argv[1].strptr;
   memcpy( buf, tmp, 4 );
   buf[4] = '\0';
   year = atoi( buf );
   memcpy( buf, tmp+4, 2 );
   buf[2] = '\0';
   month = atoi( buf );
   memcpy( buf, tmp+6, 2 );
   buf[2] = '\0';
   day = atoi( buf );
   dw_calendar_set_date( handle, year, month, day );
   return RxReturnString( RxPackageGlobalData, retstr, "" );
}

/****h* Widgets/CheckboxAndRadiobutton
 * DESCRIPTION
 *   Checkboxes provide a simple mechanism for a user to indicate a boolean
 *   value response.
 *   Radiobuttons provide the user with a means to choose from a list of
 *   options.
 ******
 */

/****f* CheckboxAndRadiobutton/DW_checkbox_new
 * NAME
 *   DW_checkbox_new
 * SYNOPSIS
 *   win = dw_checkbox_new( CheckboxText, Id )
 * FUNCTION
 *   Creates a new checkbox.
 *   A checkbox is a widget that allows for a boolean value to be set. It
 *   contains text to describe the boolean value being set.
 * ARGUMENTS
 *   - CheckboxText - the text to display next to the checkbox
 *   - Id           - a numeric identifier used to identify the window
 * RESULT
 *   A window identifier.
 * SEE ALSO
 *   DW_checkbox_get, DW_checkbox_set
 * SOURCE
 *
...
checkbox = dw_checkbox_new( "Allow user to delete", 0 )
 ******
 */
rxfunc( DW_checkbox_new )
{
   HWND rc; /* function return variable */
   char * text; /* local variable for arg[0] */
   unsigned long id; /* local variable for arg[1] */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 2, 2 ) )
      return( 1 );
   if ( RXVALIDSTRING(argv[0]) )
      text = RXSTRPTR(argv[0]);
   else
      text = "";
   if ( RxStrToULong( RxPackageGlobalData, &argv[1], (RX_ULONG *)&id ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 2 );
      return(1);
   }
   rc = dw_checkbox_new( text, id );
   return RxReturnUnsignedNumber( RxPackageGlobalData, retstr, (ULONG)rc );
}

/****f* CheckboxAndRadiobutton/DW_radiobutton_new
 * NAME
 *   DW_radiobutton_new
 * SYNOPSIS
 *   win = dw_radiobutton_new( ButtonText, Id )
 * FUNCTION
 *   Creates a new radiobutton.
 *   A radiobutton is a widget that allows for one of a number of values to be set. It
 *   contains text to describe the value being set. A number of radiobuttons are
 *   created together with one of them set at any point in time.
 * ARGUMENTS
 *   - ButtonText - the text to display next to the radiobutton
 *   - Id         - a numeric identifier used to identify the window
 * RESULT
 *   A window identifier.
 * SEE ALSO
 *   DW_checkbox_new, DW_groupbox_new
 * NOTES
 *   For radiobuttons to work properly, all radiobuttons that are to be grouped together
 *   MUST be packed into the same groupbox and they MUST have a unique Id within the
 *   groupbox.
 * SOURCE
 *
langs = 'Rexx Perl Tcl C COBOL'
groupbox = dw_groupbox_new( !REXXDW.!DW_VERT, 0, 'Favourite Language' )
Do i = 1 To Words( langs )
   rb.i = dw_radiobutton_new( Word( langs, i ), i*10 )
   Call dw_box_pack_start groupbox, rb.i, 150, 15, !REXXDW.!DW_EXPAND_HORZ, ,
      !REXXDW.!DW_DONT_EXPAND_VERT, 0
End
Call dw_radiobutton_set rb.1, !REXXDW.!DW_CHECKED
 ******
 */
rxfunc( DW_radiobutton_new )
{
   HWND rc; /* function return variable */
   char * text; /* local variable for arg[0] */
   ULONG id; /* local variable for arg[1] */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 2, 2 ) )
      return( 1 );
   if ( RXVALIDSTRING(argv[0]) )
      text = RXSTRPTR(argv[0]);
   else
      text = "";
   if ( RxStrToULong( RxPackageGlobalData, &argv[1], (RX_ULONG *)&id ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 2 );
      return(1);
   }
   rc = dw_radiobutton_new( text, id );
   return RxReturnUnsignedNumber( RxPackageGlobalData, retstr, (ULONG)rc );
}

/****f* CheckboxAndRadiobutton/DW_checkbox_set
 * NAME
 *   DW_checkbox_set
 * SYNOPSIS
 *   dw_checkbox_set( Win, State )
 * FUNCTION
 *   Sets the state of the specified checkbox.
 * ARGUMENTS
 *   - Win        - the window identifier returned from DW_checkbox_new()
 *   - State      - the initial state of the checkbox;
 *                  !REXXDW.!DW_CHECKED or !REXXDW.!DW_UNCHECKED
 * RESULT
 *   No return result.
 * SEE ALSO
 *   DW_checkbox_new, DW_checkbox_get, WidgetChecked
 * SOURCE
 *
...
checkbox = dw_checkbox_new( "Allow access", 0 )
Call dw_checkbox_set checkbox, !REXXDW.!DW_CHECKED
 ******
 */
rxfunc( DW_checkbox_set )
{
   HWND handle; /* local variable for arg[0] */
   int value; /* local variable for arg[1] */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 2, 2 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&handle ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   if ( RxStrToInt( RxPackageGlobalData, &argv[1], (RX_INT *)&value ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 2 );
      return(1);
   }
   dw_checkbox_set( handle, value );
   return RxReturnString( RxPackageGlobalData, retstr, "" );
}

/****f* CheckboxAndRadiobutton/DW_radiobutton_set
 * NAME
 *   DW_radiobutton_set
 * SYNOPSIS
 *   dw_radiobutton_set( Win, State )
 * FUNCTION
 *   Sets the state of the radiobutton to checked or not.
 * ARGUMENTS
 *   - Win        - the window identifier returned from DW_radiobutton_new()
 *   - State      - set to DW_CHECKED or DW_UNCHECKED
 * RESULT
 *   No return result.
 * SEE ALSO
 *   DW_radiobutton_get
 * SOURCE
 *
langs = 'Rexx Perl Tcl C COBOL'
groupbox = dw_groupbox_new( !REXXDW.!DW_VERT, 0, 'Favourite Language' )
Do i = 1 To Words( langs )
   rb.i = dw_radiobutton_new( Word( langs, i ), i*10 )
   Call dw_box_pack_start groupbox, rb.i, 150, 15, !REXXDW.!DW_EXPAND_HORZ, ,
      !REXXDW.!DW_DONT_EXPAND_VERT, 0
End
Call dw_radiobutton_set rb.1, !REXXDW.!DW_CHECKED
 ******
 */
rxfunc( DW_radiobutton_set )
{
   HWND handle; /* local variable for arg[0] */
   int value; /* local variable for arg[1] */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 2, 2 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&handle ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   if ( RxStrToInt( RxPackageGlobalData, &argv[1], (RX_INT *)&value ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 2 );
      return(1);
   }
   dw_checkbox_set( handle, value );
   return RxReturnString( RxPackageGlobalData, retstr, "" );
}

/****f* CheckboxAndRadiobutton/DW_checkbox_get
 * NAME
 *   DW_checkbox_get
 * SYNOPSIS
 *   win = dw_checkbox_get( Win )
 * FUNCTION
 *   Queries the state of the specified checkbox.
 * ARGUMENTS
 *   - Win        - the window identifier returned from DW_checkbox_new()
 * RESULT
 *   1 if the checkbox is checked, 0 if not checked.
 * SEE ALSO
 *   DW_checkbox_new, DW_checkbox_set
 * SOURCE
 *
...
checkbox = dw_checkbox_new( "Allow access", 0 )
...
rc = dw_checkbox_get( checkbox )
cb_result = 'Unchecked Checked'
Say 'Allow access checkbox' Word( cb_result, rc+1 )
 ******
 */
rxfunc( DW_checkbox_get )
{
   int rc; /* function return variable */
   HWND handle; /* local variable for arg[0] */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 1, 1 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&handle ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   rc = dw_checkbox_get( handle );
   return RxReturnNumber( RxPackageGlobalData, retstr, rc );
}

/****f* CheckboxAndRadiobutton/DW_radiobutton_get
 * NAME
 *   DW_radiobutton_get
 * SYNOPSIS
 *   boolean = dw_radiobutton_get( Win )
 * FUNCTION
 *   Determines if a radiobutton has been checked or not.
 * ARGUMENTS
 *   - Win        - the window identifier returned from DW_radiobutton_new()
 * RESULT
 *   1 if the radiobutton is checked, 0 if unchecked
 * SEE ALSO
 *   DW_radiobutton_set
 * SOURCE
 *
langs = 'Rexx Perl Tcl C COBOL'
groupbox = dw_groupbox_new( !REXXDW.!DW_VERT, 0, 'Favourite Language' )
Do i = 1 To Words( langs )
   rb.i = dw_radiobutton_new( Word( langs, i ), i*10 )
   Call dw_box_pack_start groupbox, rb.i, 150, 15, !REXXDW.!DW_EXPAND_HORZ, ,
      !REXXDW.!DW_DONT_EXPAND_VERT, 0
End
Call dw_radiobutton_set rb.1, !REXXDW.!DW_CHECKED
...
Do i = 1 To Words( langs )
   If dw_radiobutton_get( rb.i ) Then
      Do
         Say 'Your favourite language is' Word( langs, i )
         Leave
      End
End
 ******
 */
rxfunc( DW_radiobutton_get )
{
   int rc; /* function return variable */
   HWND handle; /* local variable for arg[0] */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 1, 1 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&handle ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   rc = dw_checkbox_get( handle );
   return RxReturnNumber( RxPackageGlobalData, retstr, rc );
}

/****h* Widgets/ContainerAndFilesystem
 * DESCRIPTION
 *   These widgets provide a two-dimensional display of read-only data.
 *   Data are displayed in rows and columns, with automatically created scrollbars
 *   to display the complete set of data. Columns can be resized by the user at runtime.
 *   A Filesystem is the same as a normal Container, except that the first column
 *   consists of a column of string and icon items, which usually displays a filename.
 *
 *   All DW_container*() functions (for which there is no specific "Filename" equivalent)
 *   also works with a "Filename" widget.
 ******
 */

/****f* ContainerAndFilesystem/DW_container_new
 * NAME
 *   DW_container_new
 * SYNOPSIS
 *   win = dw_container_new( Id, Selection )
 * FUNCTION
 *   Creates a new container.
 *   A container is a widget that provides a grid of cells with column headings.
 * ARGUMENTS
 *   - Id         - a numeric identifier used to identify the window
 *   - Selection  - !REXXDW.!DW_SINGLE_SELECTION indicates that only 1 row in the container is selectable
 *                  !REXXDW.!DW_MULTIPLE_SELECTION indicates that multiple rowsin the container are selectable
 * RESULT
 *   A window identifier.
 * SEE ALSO
 *   ContainerSelection
 * SOURCE
 *
...
container = dw_container_new( 1, !REXXDW.!DW_SINGLE_SELECTION )
 ******
 */
rxfunc( DW_container_new )
{
   HWND rc; /* function return variable */
   unsigned long id; /* local variable for arg[0] */
   int multi; /* local variable for arg[1] */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 2, 2 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&id ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   if ( RxStrToInt( RxPackageGlobalData, &argv[1], (RX_INT *)&multi ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 2 );
      return(1);
   }
   rc = dw_container_new( id, multi );
   /* force the use of text comparisons of row titles */
   dw_window_set_data( rc, "_dw_textcomp", DW_INT_TO_POINTER(1) );
   return RxReturnUnsignedNumber( RxPackageGlobalData, retstr, (ULONG)rc );
}

/****f* ContainerAndFilesystem/DW_container_alloc
 * NAME
 *   DW_container_alloc
 * SYNOPSIS
 *   alloc = dw_container_alloc( Win, RowCount )
 * FUNCTION
 *   Allocates working memory for a container window.
 * ARGUMENTS
 *   - Win        - the window identifier returned from DW_container_new()
 *   - RowCount   - the number of rows of data that the container will initially contain
 * RESULT
 *   An identifier for the working memory.
 * SEE ALSO
 *   DW_container_new, DW_container_clear
 * NOTES
 *   After a container window is created, you need to allocate memory for the initial
 *   number of rows that the container will contain. This call does that and must
 *   be called after creating the container. The allocated memory will be freed when
 *   the container is destroyed and by a call to DW_container_clear().
 * SOURCE
 *
...
container = dw_container_new( 1, !REXXDW.!DW_SINGLE_SELECTION )
alloc = dw_container_alloc( container, 10 )
...
 ******
 */
rxfunc( DW_container_alloc )
{
   void * rc; /* function return variable */
   HWND handle; /* local variable for arg[0] */
   int rowcount; /* local variable for arg[1] */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 2, 2 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&handle ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   if ( RxStrToInt( RxPackageGlobalData, &argv[1], (RX_INT *)&rowcount ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 2 );
      return(1);
   }
   rc = dw_container_alloc( handle, rowcount );
   return RxReturnPointer( RxPackageGlobalData, retstr, (void *)rc );
}

/****f* ContainerAndFilesystem/DW_container_setup
 * NAME
 *   DW_container_setup
 * SYNOPSIS
 *   dw_container_setup( Win, FlagsArray, TitlesArray[,Separator] )
 * FUNCTION
 *   Defines properties for the columns in a container.
 * ARGUMENTS
 *   - Win        - the window identifier returned from DW_container_new()
 *   - FlagsArray - an array containing column flags
 *   - TitlesArray- an array containing column heading titles
 *   - Separator  - An optional flag under OS/2 ?????
 * RESULT
 *   No return result.
 * SEE ALSO
 *   DW_container_change_item, DW_filesystem_setup, ContainerColumnFlags
 * NOTES
 *   The FlagsArray is a Rexx stem variable name, including a trailing period. The 0th
 *   compound variable (eg Flags.0) contains a count of the number of items in the
 *   array. Each item in the array can contain a combination of the values
 *   described in ContainerColumnFlags.
 * SOURCE
 *
...
flags.0 = 2
flags.1 = dw_or( !REXXDW.!DW_CFA_STRINGANDICON, !REXXDW.!DW_CFA_LEFT, ,
   !REXXDW.!DW_CFA_SEPARATOR, !REXXDW.!DW_CFA_HORZSEPARATOR )
flags.2 = dw_or( !REXXDW.!DW_CFA_STRING, !REXXDW.!DW_CFA_LEFT, ,
   !REXXDW.!DW_CFA_SEPARATOR, !REXXDW.!DW_CFA_HORZSEPARATOR )
titles.0 = 2
titles.1 = 'Title'
titles.2 = 'Command'
...
container = dw_container_new( 1, !REXXDW.!DW_MULTIPLE_SELECTION )
Call dw_container_setup container, 'flags.', 'titles.', 0
alloc = dw_container_alloc( container, numrows )
...
Do i = 1 To numrows
   Call dw_container_set_item container, alloc, 0, i-1, title.i, icon.i
   Call dw_container_set_item container, alloc, 1, i-1, command.i
   Call dw_container_set_row_title alloc, i-1, "Row"i
End
 ******
 */
rxfunc( DW_container_setup )
{
   int rc; /* function return variable */
   HWND handle; /* local variable for arg[0] */
   unsigned long * flags; /* local variable for arg[1] */
   char ** titles; /* local variable for arg[2] */
   int separator=0; /* local variable for arg[3] */
   int num_args;
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 3, 4 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&handle ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   if ( ( num_args = RxStemToULongArray( RxPackageGlobalData, &argv[1], (unsigned long * *)&flags ) ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Unable to convert Flags stem: \"%s\".", RXSTRPTR(argv[1]) );
      return(1);
   }
   if ( ( num_args = RxStemToCharArray( RxPackageGlobalData, &argv[2], (char ** *)&titles ) ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Unable to convert Titles stem: \"%s\".", RXSTRPTR(argv[2]) );
      return(1);
   }
   if ( argc == 4 )
   {
      if ( RxStrToInt( RxPackageGlobalData, &argv[3], (RX_INT *)&separator ) == -1 )
      {
         RxDisplayError( RxPackageGlobalData, name,
                         "*ERROR* Invalid number for argument %d.", 4 );
         return(1);
      }
   }
   rc = dw_container_setup( handle, flags, titles, num_args, separator );
   RxFreeULongArray( RxPackageGlobalData, flags  );
   RxFreeCharArray( RxPackageGlobalData, titles, num_args );
   return RxReturnNumber( RxPackageGlobalData, retstr, rc );
}

/****f* ContainerAndFilesystem/DW_filesystem_setup
 * NAME
 *   DW_filesystem_setup
 * SYNOPSIS
 *   dw_filesystem_setup( Win, FlagsArray, TitlesArray, Separator )
 * FUNCTION
 *   Defines properties for the columns in a filesystem container.
 * ARGUMENTS
 *   - Win        - the window identifier returned from DW_container_new()
 *   - FlagsArray - an array containing column flags
 *   - TitlesArray- an array containing column heading titles
 * RESULT
 *   No return result.
 * SEE ALSO
 *   DW_filesystem_change_item, DW_container_setup, ContainerColumnFlags
 * NOTES
 *   The FlagsArray is a Rexx stem variable name, including a trailing period. The 0th
 *   compound variable (eg Flags.0) contains a count of the number of items in the
 *   array. Each item in the array can contain a combination of the values
 *   described in ContainerColumnFlags.
 * SOURCE
 *
...
flags.0 = 2
flags.1 = dw_or( !REXXDW.!DW_CFA_STRINGANDICON, !REXXDW.!DW_CFA_LEFT, ,
   !REXXDW.!DW_CFA_SEPARATOR, !REXXDW.!DW_CFA_HORZSEPARATOR )
flags.2 = dw_or( !REXXDW.!DW_CFA_STRING, !REXXDW.!DW_CFA_LEFT, ,
   !REXXDW.!DW_CFA_SEPARATOR, !REXXDW.!DW_CFA_HORZSEPARATOR )
titles.0 = 2
titles.1 = 'Title'
titles.2 = 'Command'
...
container = dw_container_new( 1, !REXXDW.!DW_MULTIPLE_SELECTION )
alloc = dw_container_alloc( container, numrows )
Call dw_filesystem_setup container, 'flags.', 'titles.'
...
normalfileicon = dw_icon_load_from_file( '/home/mark/normalfile' )
Do i = 1 To numrows
   Call dw_filesystem_set_file container, alloc, '/home/mark/myfile', myicon
   Call dw_filesystem_set_item container, alloc, 0, i-1, title.i, icon.i
   Call dw_filesystem_set_item container, alloc, 1, i-1, command.i
   Call dw_container_set_row_title alloc, i-1, "Row"i
End
 ******
 */
rxfunc( DW_filesystem_setup )
{
   int rc; /* function return variable */
   HWND handle; /* local variable for arg[0] */
   unsigned long * flags; /* local variable for arg[1] */
   char ** titles; /* local variable for arg[2] */
   int num_args;
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 3, 3 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&handle ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name, "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   if ( ( num_args = RxStemToULongArray( RxPackageGlobalData, &argv[1], (unsigned long * *)&flags ) ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name, "*ERROR* Unable to convert Flags stem: \"%s\".", RXSTRPTR(argv[1]) );
      return(1);
   }
   if ( ( num_args = RxStemToCharArray( RxPackageGlobalData, &argv[2], (char ** *)&titles ) ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name, "*ERROR* Unable to convert Titles stem: \"%s\".", RXSTRPTR(argv[1]) );
      return(1);
   }
   rc = dw_filesystem_setup( handle, flags, titles, num_args );
   RxFreeULongArray( RxPackageGlobalData, flags  );
   RxFreeCharArray( RxPackageGlobalData, titles, num_args );
   return RxReturnNumber( RxPackageGlobalData, retstr, rc );
}

/****f* ContainerAndFilesystem/DW_container_set_item
 * NAME
 *   DW_container_set_item
 * SYNOPSIS
 *   dw_container_set_item( Win, Memory, Column, Row, Data [,Data2] )
 * FUNCTION
 *   Adds  the initial data into a cell of a container.
 * ARGUMENTS
 *   - Win        - the window identifier returned from DW_container_new()
 *   - Memory     - the memory identifier returned by a call to DW_container_alloc()
 *   - Column     - the column of the data to be changed, 0 based
 *   - Row        - the row of the data to be changed, 0 based
 *   - Data       - the new data value for the cell
 *   - Data2      - extra data for the cell if the column type is !REXXDW.!DW_CFA_STRINGANDICON
 *                  (not implemented yet)
 * RESULT
 *   No return result.
 * SEE ALSO
 *   DW_container_change_item, DW_container_setup, ContainerColumnFlags
 *   DW_filesystem_change_item, DW_filesystem_set_item
 * NOTES
 *   The Data (and Data2) arguments must match the data type of the column when
 *   defined in the call to DW_container_setup().
 *   To clear a displayed icon in a column of type DW_CFA_BITMAPORICON set
 *   Data argument to 0.
 * SOURCE
 *
...
flags.0 = 2
flags.1 = dw_or( !REXXDW.!DW_CFA_STRINGANDICON, !REXXDW.!DW_CFA_LEFT, ,
   !REXXDW.!DW_CFA_SEPARATOR, !REXXDW.!DW_CFA_HORZSEPARATOR )
flags.2 = dw_or( !REXXDW.!DW_CFA_STRING, !REXXDW.!DW_CFA_LEFT, ,
   !REXXDW.!DW_CFA_SEPARATOR, !REXXDW.!DW_CFA_HORZSEPARATOR )
titles.0 = 2
titles.1 = 'Title'
titles.2 = 'Command'
...
container = dw_container_new( 1, !REXXDW.!DW_MULTIPLE_SELECTION )
alloc = dw_container_alloc( container, numrows )
Call dw_container_setup container, 'flags.', 'titles.', 0
...
Do i = 1 To numrows
   Call dw_container_set_item container, alloc, 0, i-1, title.i, icon.i
   Call dw_container_set_item container, alloc, 1, i-1, command.i
   Call dw_container_set_row_title alloc, i-1, "Row"i
End
 ******
 */
rxfunc( DW_container_set_item )
{
   HWND handle; /* local variable for arg[0] */
   void * pointer; /* local variable for arg[1] */
   int column; /* local variable for arg[2] */
   int row; /* local variable for arg[3] */
   int flag;
   void * data = NULL; /* local generic variable for arg[4] */
   long icon; /* local variable for arg[4] */
   unsigned long number;
   char * str; /* local variable for arg[4] */
   CTIME time;
   CDATE date;
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 5, 6 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&handle ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   if ( RxStrToULong( RxPackageGlobalData, &argv[1], (RX_ULONG *)&pointer ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 2 );
      return(1);
   }
   if ( RxStrToInt( RxPackageGlobalData, &argv[2], (RX_INT *)&column ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 3 );
      return(1);
   }
   if ( RxStrToInt( RxPackageGlobalData, &argv[3], (RX_INT *)&row ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 4 );
      return(1);
   }
   /*
    * We have to do some mucking around here.
    * The last argument to dw_container_set_item() is a void*, but the actual
    * data format is dependent on the flags that were passed to dw_container_setup().
    */
   flag = dw_container_get_column_type( handle, column );
   if ( flag == 0 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Unable to determine column type for column %d.", column );
      return(1);
   }
   if ( flag == DW_CFA_BITMAPORICON )
   {
      if ( RxStrToULong( RxPackageGlobalData, &argv[4], (RX_ULONG *)&icon ) == -1 )
      {
         RxDisplayError( RxPackageGlobalData, name,
                         "*ERROR* Invalid number for argument %d.", 5 );
         return(1);
      }
      if ( icon )
         data = (void *)&icon;
   }
   else if ( flag == DW_CFA_STRINGANDICON )
   {
      if ( argc != 6 ) /* must be 6 args for this flag */
         return 1;
      if ( RxStrToULong( RxPackageGlobalData, &argv[5], (RX_ULONG *)&icon ) == -1 )
      {
         RxDisplayError( RxPackageGlobalData, name,
                         "*ERROR* Invalid number for argument %d.", 6 );
         return(1);
      }
      if ( icon )
         data = (void *)&icon;
      if ( RXVALIDSTRING(argv[4]) )
         str = RXSTRPTR(argv[4]);
      else
         str = "";
/*      data = (void *)&str; */
   }
   else if ( flag == DW_CFA_STRING )
   {
      if ( RXVALIDSTRING(argv[4]) )
         str = RXSTRPTR(argv[4]);
      else
         str = "";
      data = (void *)&str;
   }
   else if ( flag == DW_CFA_ULONG )
   {
      if ( RxStrToULong( RxPackageGlobalData, &argv[4], (RX_ULONG *)&number ) == -1 )
      {
         RxDisplayError( RxPackageGlobalData, name,
                         "*ERROR* Invalid number for argument %d.", 5 );
         return(1);
      }
      data = (void *)&number;
   }
   else if ( flag == DW_CFA_DATE )
   {
      char tmpday[3];
      char tmpmonth[3];
      char tmpyear[5];
      char *ptr = RXSTRPTR(argv[4]);
      if ( RXSTRLEN(argv[4]) != 8 )
      {
         RxDisplayError( RxPackageGlobalData, name,
                         "*ERROR* Invalid date length; must be 8." );
         return(1);
      }
      memcpy( tmpday, ptr+6, 2 ); tmpday[2] = '\0';
      memcpy( tmpmonth, ptr+4, 2 ); tmpmonth[2] = '\0';
      memcpy( tmpyear, ptr, 4 ); tmpyear[4] = '\0';
      date.day = atoi( tmpday );
      date.month = atoi( tmpmonth );
      date.year = atoi( tmpyear );
      data = (void *)&date;
   }
   else if ( flag == DW_CFA_TIME )
   {
      char tmptime[9];
      if ( RXSTRLEN(argv[4]) != 8 )
      {
         RxDisplayError( RxPackageGlobalData, name,
                         "*ERROR* Invalid time length; must be 8." );
         return(1);
      }
      strcpy( tmptime, RXSTRPTR(argv[4]) );
      tmptime[2] = '\0';
      tmptime[5] = '\0';
      time.seconds = atoi( tmptime+6 );
      time.minutes = atoi( tmptime+3 );
      time.hours = atoi( tmptime );
      data = (void *)&time;
   }
   else
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Unhandled column format: %d.", flag );
      return(1);
   }
   dw_container_set_item( handle, pointer, column, row, data );
   return RxReturnString( RxPackageGlobalData, retstr, "" );
}

/****f* ContainerAndFilesystem/DW_filesystem_set_file
 * NAME
 *   DW_filesystem_set_file
 * SYNOPSIS
 *   dw_filesystem_set_file( Win, Memory, Row, Filename, Icon )
 * FUNCTION
 *   Adds the data in the first cell of a filesystem container.
 * ARGUMENTS
 *   - Win        - the window identifier returned from DW_container_new()
 *   - Memory     - the memory identifier returned by a call to DW_container_alloc()
 *   - Row        - the row of the data to be changed, 0 based
 *   - Filename   - the new filename for the cell
 *   - Icon       - the icon identifier returned from DW_icon_load_from_file()
 *                  or DW_icon_load_from_data() to be the new icon for the cell
 * RESULT
 *   No return result.
 * SEE ALSO
 *   DW_filesystem_change_file, DW_filesystem_set_item
 * SOURCE
 *
...
normalfileicon = dw_icon_load_from_file( '/home/mark/normalfile' )
container = dw_container_new( 1, !REXXDW.!DW_SINGLE_SELECTION )
alloc = dw_container_alloc( container, 10 )
...
Call dw_filesystem_set_file container, alloc, 10, '/home/mark/newfile.txt', ,
   normalfileicon
 ******
 */
rxfunc( DW_filesystem_set_file )
{
   HWND handle; /* local variable for arg[0] */
   void * pointer; /* local variable for arg[1] */
   int row; /* local variable for arg[2] */
   char * filename; /* local variable for arg[3] */
   HICN icon; /* local variable for arg[4] */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 5, 5 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&handle ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   if ( RxStrToULong( RxPackageGlobalData, &argv[1], (RX_ULONG *)&pointer ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 2 );
      return(1);
   }
   if ( RxStrToInt( RxPackageGlobalData, &argv[2], (RX_INT *)&row ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 3 );
      return(1);
   }
   if ( RXVALIDSTRING(argv[3]) )
      filename = RXSTRPTR(argv[3]);
   else
      filename = "";
   if ( RxStrToULong( RxPackageGlobalData, &argv[4], (RX_ULONG *)&icon ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 5 );
      return(1);
   }
   dw_filesystem_set_file( handle, pointer, row, filename, icon );
   return RxReturnString( RxPackageGlobalData, retstr, "" );
}

/****f* ContainerAndFilesystem/DW_filesystem_set_item
 * NAME
 *   DW_filesystem_set_item
 * SYNOPSIS
 *   dw_filesystem_set_item( Win, Memory, Column, Row, Data [,Data2] )
 * FUNCTION
 *   Adds  the initial data into a cell of a filesystem container.
 * ARGUMENTS
 *   - Win        - the window identifier returned from DW_container_new()
 *   - Memory     - the memory identifier returned by a call to DW_container_alloc()
 *   - Column     - the column of the data to be changed, 0 based
 *   - Row        - the row of the data to be changed, 0 based
 *   - Data       - the new data value for the cell
 *   - Data2      - extra data for the cell if the column type is !REXXDW.!DW_CFA_STRINGANDICON
 *                  (not implemented yet)
 * RESULT
 *   No return result.
 * SEE ALSO
 *   DW_container_change_item, DW_container_setup, ContainerColumnFlags
 *   DW_filesystem_change_item, DW_container_set_item
 * NOTES
 *   The Data (and Data2) arguments must match the data type of the column when
 *   defined in the call to DW_container_setup().
 * SOURCE
 *
...
flags.0 = 2
flags.1 = dw_or( !REXXDW.!DW_CFA_STRINGANDICON, !REXXDW.!DW_CFA_LEFT, ,
   !REXXDW.!DW_CFA_SEPARATOR, !REXXDW.!DW_CFA_HORZSEPARATOR )
flags.2 = dw_or( !REXXDW.!DW_CFA_STRING, !REXXDW.!DW_CFA_LEFT, ,
   !REXXDW.!DW_CFA_SEPARATOR, !REXXDW.!DW_CFA_HORZSEPARATOR )
titles.0 = 2
titles.1 = 'Title'
titles.2 = 'Command'
...
container = dw_container_new( 1, !REXXDW.!DW_MULTIPLE_SELECTION )
alloc = dw_container_alloc( container, numrows )
Call dw_filesystem_setup container, 'flags.', 'titles.', 0
...
Do i = 1 To numrows
   Call dw_filesystem_set_file container, alloc, i-1, fname.i, icon.i
   Call dw_filesystem_set_item container, alloc, 0, i-1, title.i, icon.i
   Call dw_filesystem_set_item container, alloc, 1, i-1, command.i
   Call dw_container_set_row_title alloc, i-1, "Row"i
End
 ******
 */
rxfunc( DW_filesystem_set_item )
{
   HWND handle; /* local variable for arg[0] */
   void * pointer; /* local variable for arg[1] */
   int column; /* local variable for arg[2] */
   int row; /* local variable for arg[3] */
   int flag;
   void * data=NULL; /* local generic variable for arg[4] */
   long icon; /* local variable for arg[4] */
   unsigned long number;
   char * str; /* local variable for arg[4] */
   CTIME time;
   CDATE date;
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 5, 6 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&handle ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   if ( RxStrToULong( RxPackageGlobalData, &argv[1], (RX_ULONG *)&pointer ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 2 );
      return(1);
   }
   if ( RxStrToInt( RxPackageGlobalData, &argv[2], (RX_INT *)&column ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 3 );
      return(1);
   }
   if ( RxStrToInt( RxPackageGlobalData, &argv[3], (RX_INT *)&row ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 4 );
      return(1);
   }
   /*
    * We have to do some mucking around here.
    * The last argument to dw_filesystem_set_item() is a void*, but the actual
    * data format is dependent on the flags that were passed to dw_container_setup().
    */
   flag = dw_filesystem_get_column_type( handle, column );
   if ( flag == 0 )
   {
      RxDisplayError( RxPackageGlobalData, name, "*ERROR* Unable to determine column type for column %d.", column );
      return(1);
   }
   if ( flag == DW_CFA_BITMAPORICON )
   {
      if ( RxStrToULong( RxPackageGlobalData, &argv[4], (RX_ULONG *)&icon ) == -1 )
      {
         RxDisplayError( RxPackageGlobalData, name, "*ERROR* Invalid number for type DW_CFA_BITMAPORICON in argument %d.", 5 );
         return(1);
      }
      if ( icon )
         data = (void *)&icon;
   }
   else if ( flag == DW_CFA_STRINGANDICON )
   {
      if ( argc != 6 ) /* must be 6 args for this flag */
         return 1;
      if ( RxStrToULong( RxPackageGlobalData, &argv[5], (RX_ULONG *)&icon ) == -1 )
      {
         RxDisplayError( RxPackageGlobalData, name, "*ERROR* Invalid number for type DW_CFA_STRINGANDICON in argument %d.", 5 );
         return(1);
      }
      if ( icon )
           data = (void *)&icon;
      if ( RXVALIDSTRING(argv[4]) )
         str = RXSTRPTR(argv[4]);
      else
         str = "";
/*      data = (void *)&str; */
   }
   else if ( flag == DW_CFA_STRING )
   {
      if ( RXVALIDSTRING(argv[4]) )
         str = RXSTRPTR(argv[4]);
      else
         str = "";
      data = (void *)&str;
   }
   else if ( flag == DW_CFA_ULONG )
   {
      if ( RxStrToULong( RxPackageGlobalData, &argv[4], (RX_ULONG *)&number ) == -1 )
      {
         RxDisplayError( RxPackageGlobalData, name, "*ERROR* Invalid number for type DW_CFA_STRING in argument %d.", 5 );
         return(1);
      }
      data = (void *)&number;
   }
   else if ( flag == DW_CFA_DATE )
   {
      char tmpday[3];
      char tmpmonth[3];
      char tmpyear[5];
      char *ptr = RXSTRPTR(argv[4]);
      if ( RXSTRLEN(argv[4]) != 8 )
      {
         RxDisplayError( RxPackageGlobalData, name, "*ERROR* Invalid date length; must be 8." );
         return(1);
      }
      memcpy( tmpday, ptr+6, 2 ); tmpday[2] = '\0';
      memcpy( tmpmonth, ptr+4, 2 ); tmpmonth[2] = '\0';
      memcpy( tmpyear, ptr, 4 ); tmpyear[4] = '\0';
      date.day = atoi( tmpday );
      date.month = atoi( tmpmonth );
      date.year = atoi( tmpyear );
      data = (void *)&date;
   }
   else if ( flag == DW_CFA_TIME )
   {
      char tmptime[9];
      if ( RXSTRLEN(argv[4]) != 8 )
      {
         RxDisplayError( RxPackageGlobalData, name, "*ERROR* Invalid time length; must be 8." );
         return(1);
      }
      strcpy( tmptime, RXSTRPTR(argv[4]) );
      tmptime[2] = '\0';
      tmptime[5] = '\0';
      time.seconds = atoi( tmptime+6 );
      time.minutes = atoi( tmptime+3 );
      time.hours = atoi( tmptime );
      data = (void *)&time;
   }
   else
   {
      RxDisplayError( RxPackageGlobalData, name, "*ERROR* Unhandled column format: %d.", flag );
      return(1);
   }
   dw_filesystem_set_item( handle, pointer, column, row, data );
   return RxReturnString( RxPackageGlobalData, retstr, "" );
}

/****f* ContainerAndFilesystem/DW_container_change_item
 * NAME
 *   DW_container_change_item
 * SYNOPSIS
 *   dw_container_change_item( Win, Column, Row, Data [,Data2] )
 * FUNCTION
 *   Changes the data in a cell of a container.
 * ARGUMENTS
 *   - Win        - the window identifier returned from DW_container_new()
 *   - Column     - the column of the data to be changed, 0 based
 *   - Row        - the row of the data to be changed, 0 based
 *   - Data       - the new data value for the cell
 *   - Data2      - extra data for the cell if the column type is !REXXDW.!DW_CFA_STRINGANDICON
 *                  (not implemented yet)
 * RESULT
 *   No return result.
 * SEE ALSO
 *   DW_container_set_item
 * NOTES
 *   The Data (and Data2) arguments must match the data type of the column when
 *   defined in the call to DW_container_setup().
 *   To clear a displayed icon in a column of type DW_CFA_BITMAPORICON set
 *   Data argument to 0.
 * SOURCE
 *
...
container = dw_container_new( 1, !REXXDW.!DW_SINGLE_SELECTION )
...
Call dw_container_change_item container, 0, 10, 'New string data'
 ******
 */
rxfunc( DW_container_change_item )
{
   HWND handle; /* local variable for arg[0] */
   int column; /* local variable for arg[1] */
   int row; /* local variable for arg[2] */
   int flag;
   void * data = NULL; /* local generic variable for arg[3] */
   long icon; /* local variable for arg[3] */
   unsigned long number;
   char * str; /* local variable for arg[3] */
   CTIME time;
   CDATE date;
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 4, 5 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&handle ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   if ( RxStrToInt( RxPackageGlobalData, &argv[1], (RX_INT *)&column ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 2 );
      return(1);
   }
   if ( RxStrToInt( RxPackageGlobalData, &argv[2], (RX_INT *)&row ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 3 );
      return(1);
   }
   /*
    * We have to do some mucking around here.
    * The last argument to dw_container_set_item() is a void*, but the actual
    * data format is dependent on the flags that were passed to dw_container_setup().
    */
   flag = dw_container_get_column_type( handle, column );
   if ( flag == 0 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Unable to determine column type for column %d.", column );
      return(1);
   }
   if ( flag == DW_CFA_BITMAPORICON )
   {
      if ( RxStrToULong( RxPackageGlobalData, &argv[3], (RX_ULONG *)&icon ) == -1 )
      {
         RxDisplayError( RxPackageGlobalData, name,
                         "*ERROR* Invalid number for argument %d.", 4 );
         return(1);
      }
      if ( icon )
         data = (void *)&icon;
   }
   else if ( flag == DW_CFA_STRINGANDICON )
   {
      if ( argc != 5 ) /* must be 5 args for this flag */
         return 1;
      if ( RxStrToULong( RxPackageGlobalData, &argv[4], (RX_ULONG *)&icon ) == -1 )
      {
         RxDisplayError( RxPackageGlobalData, name,
                         "*ERROR* Invalid number for argument %d.", 3 );
         return(1);
      }
      if ( icon )
           data = (void *)&icon;
      if ( RXVALIDSTRING(argv[3]) )
         str = RXSTRPTR(argv[3]);
      else
         str = "";
/*      data = (void *)&str; */
   }
   else if ( flag == DW_CFA_STRING )
   {
      if ( RXVALIDSTRING(argv[3]) )
         str = RXSTRPTR(argv[3]);
      else
         str = "";
      data = (void *)&str;
   }
   else if ( flag == DW_CFA_ULONG )
   {
      if ( RxStrToULong( RxPackageGlobalData, &argv[3], (RX_ULONG *)&number ) == -1 )
      {
         RxDisplayError( RxPackageGlobalData, name,
                         "*ERROR* Invalid number for argument %d.", 4 );
         return(1);
      }
      data = (void *)&number;
   }
   else if ( flag == DW_CFA_DATE )
   {
      char tmpday[3];
      char tmpmonth[3];
      char tmpyear[5];
      char *ptr = RXSTRPTR(argv[3]);
      if ( RXSTRLEN(argv[3]) != 8 )
      {
         RxDisplayError( RxPackageGlobalData, name,
                         "*ERROR* Invalid date length; must be 8." );
         return(1);
      }
      memcpy( tmpday, ptr+6, 2 ); tmpday[2] = '\0';
      memcpy( tmpmonth, ptr+4, 2 ); tmpmonth[2] = '\0';
      memcpy( tmpyear, ptr, 4 ); tmpyear[4] = '\0';
      date.day = atoi( tmpday );
      date.month = atoi( tmpmonth );
      date.year = atoi( tmpyear );
      data = (void *)&date;
   }
   else if ( flag == DW_CFA_TIME )
   {
      char tmptime[9];
      if ( RXSTRLEN(argv[3]) != 8 )
      {
         RxDisplayError( RxPackageGlobalData, name,
                         "*ERROR* Invalid time length; must be 8." );
         return(1);
      }
      strcpy( tmptime, RXSTRPTR(argv[3]) );
      tmptime[2] = '\0';
      tmptime[5] = '\0';
      time.seconds = atoi( tmptime+6 );
      time.minutes = atoi( tmptime+3 );
      time.hours = atoi( tmptime );
      data = (void *)&time;
   }
   else
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Unhandled column format: %d.", flag );
      return(1);
   }
   dw_container_change_item( handle, column, row, data );
   return RxReturnString( RxPackageGlobalData, retstr, "" );
}

/****f* ContainerAndFilesystem/DW_filesystem_change_file
 * NAME
 *   DW_filesystem_change_file
 * SYNOPSIS
 *   dw_filesystem_change_file( Win, Row, Filename, Icon )
 * FUNCTION
 *   Changes the data in the first cell of a filesystem container.
 * ARGUMENTS
 *   - Win        - the window identifier returned from DW_container_new()
 *   - Row        - the row of the data to be changed, 0 based
 *   - Filename   - the new filename for the cell
 *   - Icon       - the icon identifier returned from DW_icon_load_from_file()
 *                  or DW_icon_load_from_data() to be the new icon for the cell
 * RESULT
 *   No return result.
 * SEE ALSO
 *   DW_filesystem_change_item, DW_filesystem_set_file
 * SOURCE
 *
...
normalfileicon = dw_icon_load_from_file( '/home/mark/normalfile' )
...
container = dw_container_new( 1, !REXXDW.!DW_SINGLE_SELECTION )
...
Call dw_filesystem_change_file container, 10, '/home/mark/newfile.txt', ,
   normalfileicon
 ******
 */
rxfunc( DW_filesystem_change_file )
{
   HWND handle; /* local variable for arg[0] */
   int row; /* local variable for arg[1] */
   char * filename; /* local variable for arg[2] */
   HICN icon; /* local variable for arg[3] */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 4, 4 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&handle ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   if ( RxStrToInt( RxPackageGlobalData, &argv[1], (RX_INT *)&row ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 2 );
      return(1);
   }
   if ( RXVALIDSTRING(argv[2]) )
      filename = RXSTRPTR(argv[2]);
   else
      filename = "";
   if ( RxStrToULong( RxPackageGlobalData, &argv[3], (RX_ULONG *)&icon ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 4 );
      return(1);
   }
   dw_filesystem_change_file( handle, row, filename, icon );
   return RxReturnString( RxPackageGlobalData, retstr, "" );
}

/****f* ContainerAndFilesystem/DW_filesystem_change_item
 * NAME
 *   DW_filesystem_change_item
 * SYNOPSIS
 *   dw_filesystem_change_item( Win, Column, Row, Data [,Data2] )
 * FUNCTION
 *   Changes the data in a cell of a container.
 * ARGUMENTS
 *   - Win        - the window identifier returned from DW_container_new()
 *   - Column     - the column of the data to be changed, 0 based
 *   - Row        - the row of the data to be changed, 0 based
 *   - Data       - the new data value for the cell
 *   - Data2      - extra data for the cell if the column type is !REXXDW.!DW_CFA_STRINGANDICON
 *                  (not implemented yet)
 * RESULT
 *   No return result.
 * SEE ALSO
 *   DW_container_set_item
 * NOTES
 *   The Data (and Data2) arguments must match the data type of the column when
 *   defined in the call to DW_container_setup().
 * SOURCE
 *
...
container = dw_container_new( 1, !REXXDW.!DW_SINGLE_SELECTION )
...
Call dw_filesystem_change_item container, 0, 10, 'New string data'
 ******
 */
rxfunc( DW_filesystem_change_item )
{
   HWND handle; /* local variable for arg[0] */
   int column; /* local variable for arg[1] */
   int row; /* local variable for arg[2] */
   int flag;
   void * data = NULL; /* local generic variable for arg[3] */
   long icon; /* local variable for arg[3] */
   unsigned long number;
   char * str; /* local variable for arg[3] */
   CTIME time;
   CDATE date;
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 4, 5 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&handle ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   if ( RxStrToInt( RxPackageGlobalData, &argv[1], (RX_INT *)&column ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 2 );
      return(1);
   }
   if ( RxStrToInt( RxPackageGlobalData, &argv[2], (RX_INT *)&row ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 3 );
      return(1);
   }
   /*
    * We have to do some mucking around here.
    * The last argument to dw_container_set_item() is a void*, but the actual
    * data format is dependent on the flags that were passed to dw_container_setup().
    */
   flag = dw_filesystem_get_column_type( handle, column );
   if ( flag == 0 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Unable to determine column type for column %d.", column );
      return(1);
   }
   if ( flag == DW_CFA_BITMAPORICON )
   {
      if ( RxStrToULong( RxPackageGlobalData, &argv[3], (RX_ULONG *)&icon ) == -1 )
      {
         RxDisplayError( RxPackageGlobalData, name,
                         "*ERROR* Invalid number for argument %d.", 4 );
         return(1);
      }
      if ( icon )
         data = (void *)&icon;
   }
   else if ( flag == DW_CFA_STRINGANDICON )
   {
      if ( argc != 5 ) /* must be 5 args for this flag */
         return 1;
      if ( RxStrToULong( RxPackageGlobalData, &argv[4], (RX_ULONG *)&icon ) == -1 )
      {
         RxDisplayError( RxPackageGlobalData, name,
                         "*ERROR* Invalid number for argument %d.", 5 );
         return(1);
      }
      if ( icon )
         data = (void *)&icon;
      if ( RXVALIDSTRING(argv[3]) )
         str = RXSTRPTR(argv[3]);
      else
         str = "";
/*      data = (void *)&str; */
   }
   else if ( flag == DW_CFA_STRING )
   {
      if ( RXVALIDSTRING(argv[3]) )
         str = RXSTRPTR(argv[3]);
      else
         str = "";
      data = (void *)&str;
   }
   else if ( flag == DW_CFA_ULONG )
   {
      if ( RxStrToULong( RxPackageGlobalData, &argv[3], (RX_ULONG *)&number ) == -1 )
      {
         RxDisplayError( RxPackageGlobalData, name,
                         "*ERROR* Invalid number for argument %d.", 4 );
         return(1);
      }
      data = (void *)&number;
   }
   else if ( flag == DW_CFA_DATE )
   {
      char tmpday[3];
      char tmpmonth[3];
      char tmpyear[5];
      char *ptr = RXSTRPTR(argv[3]);
      if ( RXSTRLEN(argv[3]) != 8 )
      {
         RxDisplayError( RxPackageGlobalData, name,
                         "*ERROR* Invalid date length; must be 8." );
         return(1);
      }
      memcpy( tmpday, ptr+6, 2 ); tmpday[2] = '\0';
      memcpy( tmpmonth, ptr+4, 2 ); tmpmonth[2] = '\0';
      memcpy( tmpyear, ptr, 4 ); tmpyear[4] = '\0';
      date.day = atoi( tmpday );
      date.month = atoi( tmpmonth );
      date.year = atoi( tmpyear );
      data = (void *)&date;
   }
   else if ( flag == DW_CFA_TIME )
   {
      char tmptime[9];
      if ( RXSTRLEN(argv[3]) != 8 )
      {
         RxDisplayError( RxPackageGlobalData, name,
                         "*ERROR* Invalid time length; must be 8." );
         return(1);
      }
      strcpy( tmptime, RXSTRPTR(argv[3]) );
      tmptime[2] = '\0';
      tmptime[5] = '\0';
      time.seconds = atoi( tmptime+6 );
      time.minutes = atoi( tmptime+3 );
      time.hours = atoi( tmptime );
      data = (void *)&time;
   }
   else
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Unhandled column format: %d.", flag );
      return(1);
   }
   dw_filesystem_change_item( handle, column, row, data );
   return RxReturnString( RxPackageGlobalData, retstr, "" );
}

/****f* ContainerAndFilesystem/DW_container_insert
 * NAME
 *   DW_container_insert
 * SYNOPSIS
 *   dw_container_insert( Win, Memory, RowCount )
 * FUNCTION
 *   Inserts the allocated memory into the container.
 * ARGUMENTS
 *   - Win        - the window identifier returned from DW_container_new()
 *   - Memory     - the memory identifier returned by a call to DW_container_alloc()
 *   - RowCount   - the number of rows of data that the container will initially contain
 * RESULT
 *   No return result.
 * SEE ALSO
 *   DW_container_new, DW_container_alloc
 * SOURCE
 *
...
container = dw_container_new( 1, !REXXDW.!DW_SINGLE_SELECTION )
alloc = dw_container_alloc( container, numrows )
...
Call dw_container_insert container, alloc, numrows
 ******
 */
rxfunc( DW_container_insert )
{
   HWND handle; /* local variable for arg[0] */
   void * pointer; /* local variable for arg[1] */
   int rowcount; /* local variable for arg[2] */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 3, 3 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&handle ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   if ( RxStrToULong( RxPackageGlobalData, &argv[1], (RX_ULONG *)&pointer ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 2 );
      return(1);
   }
   if ( RxStrToInt( RxPackageGlobalData, &argv[2], (RX_INT *)&rowcount ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 3 );
      return(1);
   }
   dw_container_insert( handle, pointer, rowcount );
   return RxReturnString( RxPackageGlobalData, retstr, "" );
}

/****f* ContainerAndFilesystem/DW_container_set_row_title
 * NAME
 *   DW_container_set_row_title
 * SYNOPSIS
 *   dw_container_set_row_title( Memory, Row, TitleText )
 * FUNCTION
 *   Assigns a textual string for the specified Row. Used to identify this
 *   row in other container functions
 * ARGUMENTS
 *   - Memory     - the memory identifier returned by a call to DW_container_alloc()
 *   - Row        - the row of data to be assigned the value; 0 based
 *   - TitleText  - the text to be associated with the row
 * RESULT
 *   No return result.
 * SEE ALSO
 *   DW_container_cursor, DW_container_delete_row,
 *   DW_container_query_next, DW_container_query_start
 * SOURCE
 *
...
container = dw_container_new( 1, !REXXDW.!DW_SINGLE_SELECTION )
alloc = dw_container_alloc( container, numrows )
...
Do i = 1 To numrows
   Call dw_container_set_row_title alloc, i-1, "Row"i
End
...
Call dw_container_delete_row container, "Row10"
 ******
 */
rxfunc( DW_container_set_row_title )
{
   void * pointer; /* local variable for arg[0] */
   int row; /* local variable for arg[1] */
   char * title; /* local variable for arg[2] */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 3, 3 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&pointer ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   if ( RxStrToInt( RxPackageGlobalData, &argv[1], (RX_INT *)&row ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 2 );
      return(1);
   }
   /* TODO - memory leak !! */
   if ( RXVALIDSTRING(argv[2]) )
      title = strdup( RXSTRPTR(argv[2]) );
   else
      title = strdup( "" );
   dw_container_set_row_title( pointer, row, title );
   return RxReturnString( RxPackageGlobalData, retstr, "" );
}

/****f* ContainerAndFilesystem/DW_container_set_stripe
 * NAME
 *   DW_container_set_stripe
 * SYNOPSIS
 *   dw_container_set_stripe( Win, OddColour, EvenColour )
 * FUNCTION
 *   Sets the background colours for alternate rows in a container.
 * ARGUMENTS
 *   - Win        - the window identifier returned from DW_container_new()
 *   - OddColour  - the internal RexxDW colour for odd numbered rows
 *   - EvenColour - the internal RexxDW colour for even numbered rows
 * RESULT
 *   No return result.
 * SEE ALSO
 *   Colours, DW_rgb
 * SOURCE
 *
...
container = dw_container_new( 1, !REXXDW.!DW_SINGLE_SELECTION )
alloc = dw_container_alloc( container, numrows )
Call dw_container_set_stripe container, !REXXDW.!DW_CLR_YELLOW, !REXXDW.!DW_CLR_DEFAULT
...
 ******
 */
rxfunc( DW_container_set_stripe )
{
   HWND handle; /* local variable for arg[0] */
   unsigned long odd; /* local variable for arg[1] */
   unsigned long even; /* local variable for arg[2] */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 3, 3 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&handle ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name, "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   if ( RxStrToULong( RxPackageGlobalData, &argv[1], (RX_ULONG *)&odd ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name, "*ERROR* Invalid number for argument %d.", 2 );
      return(1);
   }
   if ( RxStrToULong( RxPackageGlobalData, &argv[2], (RX_ULONG *)&even ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name, "*ERROR* Invalid number for argument %d.", 3 );
      return(1);
   }
   dw_container_set_stripe( handle, odd, even );
   return RxReturnString( RxPackageGlobalData, retstr, "" );
}

/****f* ContainerAndFilesystem/DW_container_set_column_width
 * NAME
 *   DW_container_set_column_width
 * SYNOPSIS
 *   dw_container_set_column_width( Win, Column, Width )
 * FUNCTION
 *   Sets the width of the column in the container to Width pixels.
 * ARGUMENTS
 *   - Win        - the window identifier returned from DW_container_new()
 *   - Column     - the column that requires a specific width; 0 based
 *   - Width      - the width of the column in pixels
 * RESULT
 *   No return result.
 * SEE ALSO
 *   DW_container_optimize
 * NOTES
 *   Columns with a set fixed width are not changed when DW_container_optimize()
 *   is called.
 * SOURCE
 *
...
container = dw_container_new( 1, !REXXDW.!DW_MULTIPLE_SELECTION )
...
Call dw_container_set_width( container, 0, 25 )
 ******
 */
rxfunc( DW_container_set_column_width )
{
   HWND handle; /* local variable for arg[0] */
   int column; /* local variable for arg[1] */
   int width; /* local variable for arg[2] */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 3, 3 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&handle ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   if ( RxStrToInt( RxPackageGlobalData, &argv[1], (RX_INT *)&column ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 2 );
      return(1);
   }
   if ( RxStrToInt( RxPackageGlobalData, &argv[2], (RX_INT *)&width ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 3 );
      return(1);
   }
   dw_container_set_column_width( handle, column, width );
   return RxReturnString( RxPackageGlobalData, retstr, "" );
}

/****f* ContainerAndFilesystem/DW_container_optimize
 * NAME
 *   DW_container_optimize
 * SYNOPSIS
 *   dw_container_optimize( Win )
 * FUNCTION
 *   Rearranges the width of the container columns to provide the optimum
 *   display of the columns.
 * ARGUMENTS
 *   - Win        - the window identifier returned from DW_container_new()
 * RESULT
 *   No return result.
 * SOURCE
 *
...
container = dw_container_new( 1, !REXXDW.!DW_SINGLE_SELECTION )
alloc = dw_container_alloc( container, numrows )
...
Call dw_container_insert container, alloc, numrows
Call dw_container_optimize container
 ******
 */
rxfunc( DW_container_optimize )
{
   HWND handle; /* local variable for arg[0] */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 1, 1 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&handle ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   dw_container_optimize( handle );
   return RxReturnString( RxPackageGlobalData, retstr, "" );
}

/****f* ContainerAndFilesystem/DW_container_clear
 * NAME
 *   DW_container_clear
 * SYNOPSIS
 *   dw_container_clear( Win, Redraw )
 * FUNCTION
 *   Clears the container of all rows and frees the memory allocated with
 *   DW_container_alloc().
 * ARGUMENTS
 *   - Win        - the window identifier returned from DW_container_new()
 *   - Redraw     - see ContainerClearFlags
 * RESULT
 *   No return result.
 * SEE ALSO
 *   DW_container_delete, ContainerClearFlags
 * SOURCE
 *
...
container = dw_container_new( 1, !REXXDW.!DW_SINGLE_SELECTION )
...
Call dw_container_clear container, !REXXDW.!DW_DONT_REDRAW
 ******
 */
rxfunc( DW_container_clear )
{
   HWND handle; /* local variable for arg[0] */
   int redraw; /* local variable for arg[1] */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 2, 2 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&handle ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   if ( RxStrToInt( RxPackageGlobalData, &argv[1], (RX_INT *)&redraw ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 2 );
      return(1);
   }
   dw_container_clear( handle, redraw );
   return RxReturnString( RxPackageGlobalData, retstr, "" );
}

/****f* ContainerAndFilesystem/DW_container_delete
 * NAME
 *   DW_container_delete
 * SYNOPSIS
 *   dw_container_delete( Win, NumRows )
 * FUNCTION
 *   Deletes NumRows from the the container starting with the first row.
 * ARGUMENTS
 *   - Win        - the window identifier returned from DW_container_new()
 *   - NumRows    - the number of rows to delete
 * RESULT
 *   No return result.
 * SEE ALSO
 *   DW_container_delete_row
 * SOURCE
 *
...
container = dw_container_new( 1, !REXXDW.!DW_SINGLE_SELECTION )
...
Call dw_container_delete container, 5
 ******
 */
rxfunc( DW_container_delete )
{
   HWND handle; /* local variable for arg[0] */
   int rowcount; /* local variable for arg[1] */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 2, 2 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&handle ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   if ( RxStrToInt( RxPackageGlobalData, &argv[1], (RX_INT *)&rowcount ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 2 );
      return(1);
   }
   dw_container_delete( handle, rowcount );
   return RxReturnString( RxPackageGlobalData, retstr, "" );
}

/****f* ContainerAndFilesystem/DW_container_delete_row
 * NAME
 *   DW_container_delete_row
 * SYNOPSIS
 *   dw_container_delete_row( Win, TitleText )
 * FUNCTION
 *   Deletes the row from the container which has the specified TitleText
 * ARGUMENTS
 *   - Win        - the window identifier returned from DW_container_new()
 *   - TitleText  - the text that was set by a call to DW_container_set_row_title()
 * RESULT
 *   No return result.
 * SEE ALSO
 *   DW_container_delete, DW_container_set_row_title
 * SOURCE
 *
...
container = dw_container_new( 1, !REXXDW.!DW_SINGLE_SELECTION )
alloc = dw_container_alloc( container, numrows )
...
Do i = 1 To numrows
   Call dw_container_set_row_title alloc, i-1, "Row"i
End
...
Call dw_container_delete_row container, "Row10"
 ******
 */
rxfunc( DW_container_delete_row )
{
   HWND handle; /* local variable for arg[0] */
   char * text; /* local variable for arg[1] */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 2, 2 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&handle ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   if ( RXVALIDSTRING(argv[1]) )
      text = RXSTRPTR(argv[1]);
   else
      text = "";
   dw_container_delete_row( handle, text );
   return RxReturnString( RxPackageGlobalData, retstr, "" );
}

/****f* ContainerAndFilesystem/DW_container_query_start
 * NAME
 *   DW_container_query_start
 * SYNOPSIS
 *   text = dw_container_query_start( Win, Flags )
 * FUNCTION
 *   Initiates a query on a container for subsequent calls to DW_container_query_next().
 *   Returns the title text associated with the first row in the container based on
 *   Flags.
 *   The title text having previously been set with a call to DW_container_set_row_title().
 * ARGUMENTS
 *   - Win        - the window identifier returned from DW_container_new()
 *   - Flags      - !REXXDW.!DW_CRA_ALL indicates that all rows in the container
 *                  should be queried.
 *                  !REXXDW.!DW_CRA_SELECTED indicates that only those rows that have been
 *                  selected should be queried.
 *                  !REXXDW.!DW_CRA_CURSORED indicates that only those items that
 *                  have focus should be queried.
 * RESULT
 *   The title text of the row or the empty string to indicate the end of the
 *   query. ie no more rows are available for querying.
 * SEE ALSO
 *   DW_container_query_next, DW_container_set_row_title
 * SOURCE
 *
...
container = dw_container_new( 1, !REXXDW.!DW_MULTIPLE_SELECTION )
...
selected = dw_container_query_start( container, !REXXDW.!DW_CRA_SELECTED )
Do While selected \= ''
   Say 'Title of row selected is' selected
   selected = dw_container_query_next( container, !REXXDW.!DW_CRA_SELECTED )
End
 ******
 */
rxfunc( DW_container_query_start )
{
   char * rc; /* function return variable */
   int rcode;
   HWND handle; /* local variable for arg[0] */
   unsigned long flags; /* local variable for arg[1] */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 2, 2 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&handle ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   if ( RxStrToULong( RxPackageGlobalData, &argv[1], (RX_ULONG *)&flags ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 2 );
      return(1);
   }
   rc = dw_container_query_start( handle, flags );
   rcode = RxReturnString( RxPackageGlobalData, retstr, rc );
#if DW_MAJOR_VERSION >= 3
   free (rc );
#endif
   return rcode;
}

/****f* ContainerAndFilesystem/DW_container_query_next
 * NAME
 *   DW_container_query_next
 * SYNOPSIS
 *   text = dw_container_query_next( Win, Flags )
 * FUNCTION
 *   Returns the title text associated with the next row in the container.
 *   The title text having previously been set with a call to DW_container_set_row_title().
 * ARGUMENTS
 *   - Win        - the window identifier returned from DW_container_new()
 *   - Flags      - !REXXDW.!DW_CRA_ALL indicates that all rows in the container
 *                  should be queried.
 *                  !REXXDW.!DW_CRA_SELECTED indicates that only those rows that have been
 *                  selected should be queried.
 *                  !REXXDW.!DW_CRA_CURSORED indicates that only those items that
 *                  have focus should be queried.
 * RESULT
 *   The title text of the row or the empty string to indicate the end of the
 *   query. ie no more rows are available for querying.
 * NOTES
 *   As the only way of knowing that the query has returned all rows in subsequent calls
 *   to this function, all rows MUST have a non-blank title text.
 * SEE ALSO
 *   DW_container_query_start, DW_container_set_row_title
 * SOURCE
 *
...
container = dw_container_new( 1, !REXXDW.!DW_MULTIPLE_SELECTION )
...
selected = dw_container_query_start( container, !REXXDW.!DW_CRA_SELECTED )
Do While selected \= ''
   Say 'Title of row selected is' selected
   selected = dw_container_query_next( container, !REXXDW.!DW_CRA_SELECTED )
End
 ******
 */
rxfunc( DW_container_query_next )
{
   char * rc; /* function return variable */
   int rcode;
   HWND handle; /* local variable for arg[0] */
   unsigned long flags; /* local variable for arg[1] */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 2, 2 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&handle ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   if ( RxStrToULong( RxPackageGlobalData, &argv[1], (RX_ULONG *)&flags ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 2 );
      return(1);
   }
   rc = dw_container_query_next( handle, flags );
   rcode = RxReturnString( RxPackageGlobalData, retstr, rc );
#if DW_MAJOR_VERSION >= 3
   free (rc );
#endif
   return rcode;
}

/****f* ContainerAndFilesystem/DW_container_cursor
 * NAME
 *   DW_container_cursor
 * SYNOPSIS
 *   dw_container_cursor( Win, TitleText )
 * FUNCTION
 *   Rearranges the view of the container so that the row with the specified
 *   TitleText is visible.
 * ARGUMENTS
 *   - Win        - the window identifier returned from DW_container_new()
 *   - TitleText  - the text that was set by a call to DW_container_set_row_title()
 * RESULT
 *   No return result.
 * SEE ALSO
 *   DW_container_set_row_title
 * SOURCE
 *
...
container = dw_container_new( 1, !REXXDW.!DW_SINGLE_SELECTION )
alloc = dw_container_alloc( container, numrows )
...
Do i = 1 To numrows
   Call dw_container_set_row_title alloc, i-1, "Row"i
End
...
Call dw_container_cursor container, "Row10"
 ******
 */
rxfunc( DW_container_cursor )
{
   HWND handle; /* local variable for arg[0] */
   char * text; /* local variable for arg[1] */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 2, 2 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&handle ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   if ( RXVALIDSTRING(argv[1]) )
      text = RXSTRPTR(argv[1]);
   else
      text = "";
   dw_container_cursor( handle, text );
   return RxReturnString( RxPackageGlobalData, retstr, "" );
}

/****f* ContainerAndFilesystem/DW_container_scroll
 * NAME
 *   DW_container_scroll
 * SYNOPSIS
 *   dw_container_scroll( Win, Direction, NumRows )
 * FUNCTION
 *   Scrolls the current view of the container by NumRows rows in the
 *   direction specified by Direction.
 * ARGUMENTS
 *   - Win        - the window identifier returned from DW_container_new()
 *   - Direction:
 *      o !REXXDW.!DW_SCROLL_TOP shows the first rows in the container
 *      o !REXXDW.!DW_SCROLL_BOTTOM shows the last rows in the container
 *      o !REXXDW.!DW_SCROLL_UP scrolls up NumRows rows
 *      o !REXXDW.!DW_SCROLL_DOWN scrolls down NumRows rows
 * RESULT
 *   No return result.
 * SEE ALSO
 *   ContainerScrollFlags
 * NOTES
 *   The NumRows argument is ignored for !REXXDW.!DW_SCROLL_TOP and
 *   !REXXDW.!DW_SCROLL_BOTTOM.
 * SOURCE
 *
...
container = dw_container_new( 1, !REXXDW.!DW_MULTIPLE_SELECTION )
...
Call dw_container_scroll( container, !REXXDW.!DW_SCROLL_UP, 10 )
 ******
 */
rxfunc( DW_container_scroll )
{
   HWND handle; /* local variable for arg[0] */
   int direction; /* local variable for arg[1] */
   long rows; /* local variable for arg[2] */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 3, 3 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&handle ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   if ( RxStrToInt( RxPackageGlobalData, &argv[1], (RX_INT *)&direction ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 2 );
      return(1);
   }
   if ( RxStrToLong( RxPackageGlobalData, &argv[2], (RX_LONG *)&rows ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 3 );
      return(1);
   }
   dw_container_scroll( handle, direction, rows );
   return RxReturnString( RxPackageGlobalData, retstr, "" );
}

/****h* Widgets/Rendering
 * DESCRIPTION
 *   A rendering widget allows low-level graphics drawing to be done.
 *   Two widget types can be rendered; a render window and a pixmap.
 *
 *   The "normal" way to write low-level graphics is to draw into an
 *   off-screen pixmap, and then "bitblt" the off-screen pixmap into
 *   the render window. This is much faster that drawing directly into
 *   the render window itself.
 * NOTES
 *   Two callbacks can be used with Render windows; DW_EXPOSE_EVENT and
 *   DW_CONFIGURE_EVENT.
 *   Whenever the Render window's size is changed a "configure" event is fired.
 *   Whenever a portion of the window is made visible, an "expose" event fires.
 ******
 */

/****f* Rendering/DW_render_new
 * NAME
 *   DW_render_new
 * SYNOPSIS
 *   win = dw_render_new( Id )
 * FUNCTION
 *   Creates a new render window.
 *   A render window is used to draw primitive graphics on, or to be the window in
 *   which a pixmap is rendered.
 * ARGUMENTS
 *   - Id         - a numeric identifier used to identify the window
 * RESULT
 *   A render identifier.
 * SOURCE
 *
render = dw_render_new( 10 )
depth = dw_color_depth_get()
pixmap = dw_pixmap_new( render, 64, 64, depth )
 ******
 */
rxfunc( DW_render_new )
{
   HWND rc; /* function return variable */
   unsigned long id; /* local variable for arg[0] */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 1, 1 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&id ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   rc = dw_render_new( id );
   return RxReturnUnsignedNumber( RxPackageGlobalData, retstr, (ULONG)rc );
}

/****f* Rendering/DW_pixmap_new
 * NAME
 *   DW_pixmap_new
 * SYNOPSIS
 *   win = dw_pixmap_new( Win, Width, Height, Depth )
 * FUNCTION
 *   Creates a new pixmap widget of the given dimensions.
 * ARGUMENTS
 *   - Win        - the window identifier returned from DW_render_new()
 *                  that the pixmap is associated with (or zero)
 *   - Width      - the width of the pixmap in pixels
 *   - Height     - the height of the pixmap in pixels
 *   - Depth      - the colour depth of the pixmap as returned from DW_color_depth_get()
 * RESULT
 *   A pixmap identifier.
 * NOTES
 *   The Win argument can be specified as 0. This will result in the pixmap being written
 *   "off screen". This is usually done to avoid flicker if several drawing
 *   actions are done to the pixmap before rendering it to the screen.
 * SOURCE
 *
render = dw_render_new( 10 )
depth = dw_color_depth_get()
pixmap = dw_pixmap_new( render, 64, 64, depth )
 ******
 */
rxfunc( DW_pixmap_new )
{
   HPIXMAP rc; /* function return variable */
   HWND handle; /* local variable for arg[0] */
   unsigned long width; /* local variable for arg[1] */
   unsigned long height; /* local variable for arg[2] */
   int depth; /* local variable for arg[3] */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 4, 4 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&handle ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   if ( RxStrToULong( RxPackageGlobalData, &argv[1], (RX_ULONG *)&width ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 2 );
      return(1);
   }
   if ( RxStrToULong( RxPackageGlobalData, &argv[2], (RX_ULONG *)&height ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 3 );
      return(1);
   }
   if ( RxStrToInt( RxPackageGlobalData, &argv[3], (RX_INT *)&depth ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 4 );
      return(1);
   }
   rc = dw_pixmap_new( handle, width, height, depth );
   return RxReturnNumber( RxPackageGlobalData, retstr, (ULONG)rc );
}

/****f* Rendering/DW_pixmap_new_from_file
 * NAME
 *   DW_pixmap_new_from_file
 * SYNOPSIS
 *   win = dw_pixmap_new_from_file( Win, Filename )
 * FUNCTION
 *   Creates a new pixmap widget from the specified Filename.
 * ARGUMENTS
 *   - Win        - the window identifier returned from DW_render_new()
 *                  that the pixmap is associated with (or zero)
 *   - Filename   - the name of a file containing a valid image
 * RESULT
 *   A pixmap identifier.
 * SEE ALSO
 *   DW_pixmap_new_from_data
 * NOTES
 *   Under Windows and OS/2 a Bitmap file (.BMP) must be presented. Under Linux
 *   various image file formats are valid; eg. XPM, PNG, JPG
 *   The size of the pixmap is determined from the dimensions of the image.
 *   The Win argument can be specified as 0. This will result in the pixmap being written
 *   "off screen". This is usually done to avoid to avoid flicker if several drawing
 *   actions are done to the pixmap before rendering it to the screen.
 * SOURCE
 *
render = dw_render_new( 10 )
pixmap = dw_pixmap_new_from_file( render, '/home/mark/mypixmap' )
 ******
 */
rxfunc( DW_pixmap_new_from_file )
{
   HPIXMAP rc; /* function return variable */
   HWND handle; /* local variable for arg[0] */
   char * filename; /* local variable for arg[1] */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 2, 2 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&handle ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   if ( RXVALIDSTRING(argv[1]) )
      filename = RXSTRPTR(argv[1]);
   else
      filename = "";
   rc = dw_pixmap_new_from_file( handle, filename );
   return RxReturnNumber( RxPackageGlobalData, retstr, (ULONG)rc );
}

/****f* Rendering/DW_pixmap_new_from_data
 * NAME
 *   DW_pixmap_new_from_data
 * SYNOPSIS
 *   win = dw_pixmap_new_from_data( Win, Filename )
 * FUNCTION
 *   Creates a new pixmap widget from the specified Filename.
 * ARGUMENTS
 *   - Win        - the window identifier returned from DW_render_new()
 *                  that the pixmap is associated with (or zero)
 *   - Data       - the exact contents of the image.
 * RESULT
 *   A pixmap identifier.
 * SEE ALSO
 *   DW_pixmap_new_from_file
 * NOTES
 *   Under Windows and OS/2 a Bitmap file (.BMP) must be presented. Under Linux
 *   various image file formats are valid; eg. XPM, PNG, JPG
 *   The size of the pixmap is determined from the dimensions of the image.
 *   The Win argument can be specified as 0. This will result in the pixmap being written
 *   "off screen". This is usually done to avoid to avoid flicker if several drawing
 *   actions are done to the pixmap before rendering it to the screen.
 * SOURCE
 *
...
bmp = "0000A6D7007E"
...
render = dw_render_new( 10 )
pixmap = dw_pixmap_new_from_data( render, x2c( bmp ) )
 ******
 */
rxfunc( DW_pixmap_new_from_data )
{
   HPIXMAP rc; /* function return variable */
   HWND handle; /* local variable for arg[0] */
   char * data; /* local variable for arg[1] */
   int len;
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 2, 2 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&handle ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   if ( RXVALIDSTRING(argv[1]) )
   {
      data = RXSTRPTR(argv[1]);
      len = RXSTRLEN(argv[1]);
   }
   else
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Supplied image data is empty." );
      return(1);
   }
   rc = dw_pixmap_new_from_data( handle, data, len );
   return RxReturnNumber( RxPackageGlobalData, retstr, (ULONG)rc );
}

/****f* Rendering/DW_pixmap_set_transparent_color
 * NAME
 *   DW_pixmap_set_transparent_color
 * SYNOPSIS
 *   DW_pixmap_set_transparent_color( Pixmap, Color )
 * FUNCTION
 *   Specifies the color to be used as the transparent background
 *   when this pixmap is bitblt'ed
 * ARGUMENTS
 *   - Pixmap     - the pixmap identifier returned from DW_pixmap_new(),
 *                  DW_pixmap_new_from_file() or DW_pixmap_new_from_data.
 *   - Color -    - the color to be rendered transparent.
 * RESULT
 *   No return result.
 * SEE ALSO
 *   DW_pixmap_new, DW_pixmap_new_from_file, DW_pixmap_new_from_data
 * SOURCE
 *
...
bmp = "0000A6D7007E"
...
render = dw_render_new( 10 )
pixmap = dw_pixmap_new_from_data( render, x2c( bmp ) )
Call dw_pixmap_set_transparent_color pixmap, !REXXDW.!DW_RED
 ******
 */
rxfunc( DW_pixmap_set_transparent_color )
{
   HPIXMAP pixmap; /* local variable for arg[0] */
   unsigned long color; /* local variable for arg[1] */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 2, 2 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&pixmap ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name, "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   if ( RxStrToULong( RxPackageGlobalData, &argv[1], (RX_ULONG *)&color ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name, "*ERROR* Invalid number for argument %d.", 2 );
      return(1);
   }
   dw_pixmap_set_transparent_color ( pixmap, color );
   return RxReturnString( RxPackageGlobalData, retstr, "" );
}

/****f* Rendering/DW_pixmap_destroy
 * NAME
 *   DW_pixmap_destroy
 * SYNOPSIS
 *   dw_pixmap_destroy( Pixmap )
 * FUNCTION
 *   Destroys a Pixmap.
 * ARGUMENTS
 *   - Pixmap     - the pixmap identifier returned from DW_pixmap_new(),
 *                  DW_pixmap_new_from_file() or DW_pixmap_new_from_data.
 * RESULT
 *   No return result.
 * SEE ALSO
 *   DW_pixmap_new, DW_pixmap_new_from_file, DW_pixmap_new_from_data
 * SOURCE
 *
render = dw_render_new( 10 )
pixmap = dw_pixmap_new_from_file( render, '/home/mark/mypixmap' )
...
Call dw_pixmap_destroy( pixmap )
 ******
 */
rxfunc( DW_pixmap_destroy )
{
   HPIXMAP pixmap; /* local variable for arg[0] */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 1, 1 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&pixmap ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   dw_pixmap_destroy( pixmap );
   return RxReturnString( RxPackageGlobalData, retstr, "" );
}

/****f* Rendering/DW_pixmap_bitblt
 * NAME
 *   DW_pixmap_bitblt
 * SYNOPSIS
 *   dw_pixmap_bitblt( DestWin, DestPixmap, DestX, DestY, Width, Height, SrcWin, SrcPixmap, SrcX, SrcY )
 * FUNCTION
 *   Copies a portion of a render window or pixmap to another render window or pixmap.
 * ARGUMENTS
 *   - DestWin    - the destination window identifier returned from DW_render_new()
 *   - DestPixmap - the destination pixmap identifier returned from DW_pixmap_new(),
 *                  DW_pixmap_new_from_file() or DW_pixmap_new_from_data
 *   - DestX      - the X coordinate of the destination measured in pixels
 *   - DestY      - the Y coordinate of the destination measured in pixels
 *   - Width      - the width of the portion to copy measured in pixels
 *   - Height     - the height of the portion to copy measured in pixels
 *   - SrcWin     - the source window identifier returned from DW_render_new()
 *   - SrcPixmap  - the source pixmap identifier returned from DW_pixmap_new()
 *   - SrcX       - the X coordinate of the source measured in pixels
 *   - SrcY       - the Y coordinate of the source measured in pixels
 * RESULT
 *   No return result.
 * SEE ALSO
 *   DW_pixmap_new, DW_pixmap_new_from_file, DW_pixmap_new_from_data, DW_render_new
 * NOTES
 *   Only one of DestWin, DestPixmap is specified; the other should be 0
 *   Only one of SrcWin, SrcPixmap is specified; the other should be 0
 * SOURCE
 *
render = dw_render_new( 10 )
depth = dw_color_depth_get()
pixmap = dw_pixmap_new( render, 64, 64, depth )
...
Call dw_pixmap_bitblt( render, 0, 0, 0, 64, 64, 0, pixmap, 0, 0 )
 ******
 */
rxfunc( DW_pixmap_bitblt )
{
   HWND dest; /* local variable for arg[0] */
   HPIXMAP destp; /* local variable for arg[1] */
   int xdest; /* local variable for arg[2] */
   int ydest; /* local variable for arg[3] */
   int width; /* local variable for arg[4] */
   int height; /* local variable for arg[5] */
   HWND src; /* local variable for arg[6] */
   HPIXMAP srcp; /* local variable for arg[7] */
   int xsrc; /* local variable for arg[8] */
   int ysrc; /* local variable for arg[9] */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 10, 10 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&dest ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   if ( RxStrToULong( RxPackageGlobalData, &argv[1], (RX_ULONG *)&destp ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 2 );
      return(1);
   }
   if ( RxStrToInt( RxPackageGlobalData, &argv[2], (RX_INT *)&xdest ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 3 );
      return(1);
   }
   if ( RxStrToInt( RxPackageGlobalData, &argv[3], (RX_INT *)&ydest ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 4 );
      return(1);
   }
   if ( RxStrToInt( RxPackageGlobalData, &argv[4], (RX_INT *)&width ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 5 );
      return(1);
   }
   if ( RxStrToInt( RxPackageGlobalData, &argv[5], (RX_INT *)&height ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 6 );
      return(1);
   }
   if ( RxStrToULong( RxPackageGlobalData, &argv[6], (RX_ULONG *)&src ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 7 );
      return(1);
   }
   if ( RxStrToULong( RxPackageGlobalData, &argv[7], (RX_ULONG *)&srcp ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 8 );
      return(1);
   }
   if ( RxStrToInt( RxPackageGlobalData, &argv[8], (RX_INT *)&xsrc ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 9 );
      return(1);
   }
   if ( RxStrToInt( RxPackageGlobalData, &argv[9], (RX_INT *)&ysrc ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 10 );
      return(1);
   }
   dw_pixmap_bitblt( dest, destp, xdest, ydest, width, height, src, srcp, xsrc, ysrc );
   return RxReturnString( RxPackageGlobalData, retstr, "" );
}

/****f* Rendering/DW_pixmap_width
 * NAME
 *   DW_pixmap_width
 * SYNOPSIS
 *   width = dw_pixmap_width( Win )
 * FUNCTION
 *   returns the width of the specified pixmap.
 * ARGUMENTS
 *   - Win        - the window identifier returned from DW_pixmap_new(),
 *                  DW_pixmap_new_from_file() or DW_pixmap_new_from_data().
 * RESULT
 *   The pixmap's width
 * SEE ALSO
 *   DW_pixmap_height
 * SOURCE
 *
render = dw_render_new( 10 )
pixmap = dw_pixmap_new_from_file( render, '/home/mark/mypixmap' )
Say 'The width of the pixmap is' dw_pixmap_width( pixmap )
 ******
 */
rxfunc( DW_pixmap_width )
{
   unsigned long rc; /* function return variable */
   HPIXMAP handle; /* local variable for arg[0] */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 1, 1 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&handle ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   rc = DW_PIXMAP_WIDTH( handle );
   return RxReturnUnsignedNumber( RxPackageGlobalData, retstr, (ULONG)rc );
}

/****f* Rendering/DW_pixmap_height
 * NAME
 *   DW_pixmap_height
 * SYNOPSIS
 *   width = dw_pixmap_height( Win )
 * FUNCTION
 *   returns the height of the specified pixmap.
 * ARGUMENTS
 *   - Win        - the window identifier returned from DW_pixmap_new(),
 *                  DW_pixmap_new_from_file() or DW_pixmap_new_from_data().
 * RESULT
 *   The pixmap's height
 * SEE ALSO
 *   DW_pixmap_width
 * SOURCE
 *
render = dw_render_new( 10 )
pixmap = dw_pixmap_new_from_file( render, '/home/mark/mypixmap' )
Say 'The height of the pixmap is' dw_pixmap_height( pixmap )
 ******
 */
rxfunc( DW_pixmap_height )
{
   unsigned long rc; /* function return variable */
   HPIXMAP handle; /* local variable for arg[0] */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 1, 1 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&handle ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   rc = DW_PIXMAP_HEIGHT( handle );
   return RxReturnUnsignedNumber( RxPackageGlobalData, retstr, (ULONG)rc );
}

/****f* Rendering/DW_draw_text
 * NAME
 *   DW_draw_text
 * SYNOPSIS
 *   dw_draw_text( Win, Pixmap, Fill, X, Y, Text )
 * FUNCTION
 *   Draws the specified Text at the coordinates X/Y.
 *   The colour of the Text is set by a call to DW_color_foreground_set().
 *   The font of the text is set by a call to DW_window_set_font() against
 *   the render box in which the text is drawn eith directly, or via a pixmap.
 * ARGUMENTS
 *   - Win        - the window identifier returned from DW_render_new()
 *   - Pixmap     - the pixmap identifier returned from DW_pixmap_new()
 *   - X          - the x coordinate in pixels of the top left corner
 *   - Y          - the y coordinate in pixels of the top left corner
 *   - Text       - the text to be drawn
 * RESULT
 *   No return result
 * SEE ALSO
 *   DW_render_new, DW_pixmap_new, DW_color_foreground_set
 *   DW_draw_line, DW_draw_point, DW_draw_rect, DW_window_set_font
 * NOTES
 *   Only one of Win or Pixmap is required. The other argument should be set to 0.
 * SOURCE
 *
...
win = dw_render_new( 0 )
Call dw_window_set_font win, myfont
...
Call dw_draw_text win, 0, 10, 34, 'A string at 10/34'
 ******
 */
rxfunc( DW_draw_text )
{
   HWND handle; /* local variable for arg[0] */
   HPIXMAP pixmap; /* local variable for arg[1] */
   int x; /* local variable for arg[2] */
   int y; /* local variable for arg[3] */
   char * text; /* local variable for arg[4] */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 5, 5 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&handle ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   if ( RxStrToULong( RxPackageGlobalData, &argv[1], (RX_ULONG *)&pixmap ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 2 );
      return(1);
   }
   if ( RxStrToInt( RxPackageGlobalData, &argv[2], (RX_INT *)&x ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 3 );
      return(1);
   }
   if ( RxStrToInt( RxPackageGlobalData, &argv[3], (RX_INT *)&y ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 4 );
      return(1);
   }
   if ( RXVALIDSTRING(argv[4]) )
      text = RXSTRPTR(argv[4]);
   else
      text = "";
   dw_draw_text( handle, pixmap, x, y, text );
   return RxReturnString( RxPackageGlobalData, retstr, "" );
}

/****f* Rendering/DW_draw_line
 * NAME
 *   DW_draw_line
 * SYNOPSIS
 *   dw_draw_line( Win, Pixmap, X1, Y1, X2, Y2 )
 * FUNCTION
 *   Draws a line between the coordinates X1/Y1 and X2/Y2 in either the render
 *   window identfied by Win or the pixmap identified by Pixmap. The line is 1 pixel
 *   in width and the colour of the line is set by a call to DW_color_foreground_set().
 * ARGUMENTS
 *   - Win        - the window identifier returned from DW_render_new()
 *   - Pixmap     - the pixmap identifier returned from DW_pixmap_new()
 *   - X1         - the x coordinate in pixels of one end of the line
 *   - Y1         - the y coordinate in pixels of one end of the line
 *   - X2         - the x coordinate in pixels of the other end of the line
 *   - Y2         - the y coordinate in pixels of the other end of the line
 * RESULT
 *   No return result
 * SEE ALSO
 *   DW_render_new, DW_pixmap_new, DW_color_foreground_set,
 *   DW_draw_point, DW_draw_rect, DW_draw_text
 * NOTES
 *   Only one of Win or Pixmap is required. The other argument should be set to 0.
 * SOURCE
 *
...
win = dw_render_new( 0 )
...
Call dw_draw_line win, 0, 10, 10, 34, 56
 ******
 */
rxfunc( DW_draw_line )
{
   HWND handle; /* local variable for arg[0] */
   HPIXMAP pixmap; /* local variable for arg[1] */
   int x1; /* local variable for arg[2] */
   int y1; /* local variable for arg[3] */
   int x2; /* local variable for arg[4] */
   int y2; /* local variable for arg[5] */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 6, 6 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&handle ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   if ( RxStrToULong( RxPackageGlobalData, &argv[1], (RX_ULONG *)&pixmap ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 2 );
      return(1);
   }
   if ( RxStrToInt( RxPackageGlobalData, &argv[2], (RX_INT *)&x1 ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 3 );
      return(1);
   }
   if ( RxStrToInt( RxPackageGlobalData, &argv[3], (RX_INT *)&y1 ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 4 );
      return(1);
   }
   if ( RxStrToInt( RxPackageGlobalData, &argv[4], (RX_INT *)&x2 ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 5 );
      return(1);
   }
   if ( RxStrToInt( RxPackageGlobalData, &argv[5], (RX_INT *)&y2 ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 6 );
      return(1);
   }
   dw_draw_line( handle, pixmap, x1, y1, x2, y2 );
   return RxReturnString( RxPackageGlobalData, retstr, "" );
}

/****f* Rendering/DW_draw_rect
 * NAME
 *   DW_draw_rect
 * SYNOPSIS
 *   dw_draw_rect( Win, Pixmap, Fill, X, Y, Width, Height )
 * FUNCTION
 *   Draws a rectangle with the top left corner at the coordinates X/Y with
 *   the specified Width and Height. All measurements are in pixels.
 *   The colour of the rectangle is set by a call to DW_color_foreground_set().
 * ARGUMENTS
 *   - Win        - the window identifier returned from DW_render_new()
 *   - Pixmap     - the pixmap identifier returned from DW_pixmap_new()
 *   - Fill       - indicate if a filled or outline rectangle is drawn
 *   - X          - the x coordinate in pixels of the top left corner
 *   - Y          - the y coordinate in pixels of the top left corner
 *   - Width      - the width of the rectangle
 *   - Height     - the height of the rectangle
 * RESULT
 *   No return result
 * SEE ALSO
 *   DW_render_new, DW_pixmap_new, DW_color_foreground_set
 *   DW_draw_line, DW_draw_point, DW_draw_text, DW_draw_polygon, DrawingFlags
 * NOTES
 *   Only one of Win or Pixmap is required. The other argument should be set to 0.
 * SOURCE
 *
...
win = dw_render_new( 0 )
...
Call dw_draw_rect win, 0, !REXXDW.!DW_DRAW_FILL, 10, 34, 40, 40
 ******
 */
rxfunc( DW_draw_rect )
{
   HWND handle; /* local variable for arg[0] */
   HPIXMAP pixmap; /* local variable for arg[1] */
   int fill; /* local variable for arg[2] */
   int x; /* local variable for arg[3] */
   int y; /* local variable for arg[4] */
   int width; /* local variable for arg[5] */
   int height; /* local variable for arg[6] */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 7, 7 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&handle ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   if ( RxStrToULong( RxPackageGlobalData, &argv[1], (RX_ULONG *)&pixmap ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 2 );
      return(1);
   }
   if ( RxStrToInt( RxPackageGlobalData, &argv[2], (RX_INT *)&fill ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 3 );
      return(1);
   }
   if ( RxStrToInt( RxPackageGlobalData, &argv[3], (RX_INT *)&x ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 4 );
      return(1);
   }
   if ( RxStrToInt( RxPackageGlobalData, &argv[4], (RX_INT *)&y ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 5 );
      return(1);
   }
   if ( RxStrToInt( RxPackageGlobalData, &argv[5], (RX_INT *)&width ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 6 );
      return(1);
   }
   if ( RxStrToInt( RxPackageGlobalData, &argv[6], (RX_INT *)&height ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 7 );
      return(1);
   }
   dw_draw_rect( handle, pixmap, fill, x, y, width, height );
   return RxReturnString( RxPackageGlobalData, retstr, "" );
}

/****f* Rendering/DW_draw_point
 * NAME
 *   DW_draw_point
 * SYNOPSIS
 *   dw_draw_point( Win, Pixmap, X, Y )
 * FUNCTION
 *   Draws a point 1 pixel in size at the coordinates X/Y. The colour of the point
 *   is set by a call to DW_color_foreground_set().
 * ARGUMENTS
 *   - Win        - the window identifier returned from DW_render_new()
 *   - Pixmap     - the pixmap identifier returned from DW_pixmap_new()
 *   - X          - the x coordinate in pixels of the point
 *   - Y          - the y coordinate in pixels of the point
 * RESULT
 *   No return result
 * SEE ALSO
 *   DW_render_new, DW_pixmap_new, DW_color_foreground_set
 *   DW_draw_line, DW_draw_rect, DW_draw_text
 * NOTES
 *   Only one of Win or Pixmap is required. The other argument should be set to 0.
 * SOURCE
 *
...
win = dw_render_new( 0 )
...
Call dw_draw_point win, 0, 10, 34
 ******
 */
rxfunc( DW_draw_point )
{
   HWND handle; /* local variable for arg[0] */
   HPIXMAP pixmap; /* local variable for arg[1] */
   int x; /* local variable for arg[2] */
   int y; /* local variable for arg[3] */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 4, 4 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&handle ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   if ( RxStrToULong( RxPackageGlobalData, &argv[1], (RX_ULONG *)&pixmap ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 2 );
      return(1);
   }
   if ( RxStrToInt( RxPackageGlobalData, &argv[2], (RX_INT *)&x ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 3 );
      return(1);
   }
   if ( RxStrToInt( RxPackageGlobalData, &argv[3], (RX_INT *)&y ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 4 );
      return(1);
   }
   dw_draw_point( handle, pixmap, x, y );
   return RxReturnString( RxPackageGlobalData, retstr, "" );
}

/****f* Rendering/DW_draw_polygon
 * NAME
 *   DW_draw_polygon
 * SYNOPSIS
 *   dw_draw_polygon( Win, Pixmap, Fill, X, Y )
 * FUNCTION
 *   Draws a closed polygon with the points defined in the X and Y stems.
 *   All measurements are in pixels.
 *   The colour of the polygon is set by a call to DW_color_foreground_set().
 * ARGUMENTS
 *   - Win        - the window identifier returned from DW_render_new()
 *   - Pixmap     - the pixmap identifier returned from DW_pixmap_new()
 *   - Fill       - indicate if a filled or outline polygon is drawn
 *   - X          - a stem name of the x coordinate in pixels
 *   - Y          - a stem name of the y coordinate in pixels
 * RESULT
 *   No return result
 * SEE ALSO
 *   DW_render_new, DW_pixmap_new, DW_color_foreground_set
 *   DW_draw_line, DW_draw_point, DW_draw_text, DW_draw_rect, DrawingFlags
 * NOTES
 *   Only one of Win or Pixmap is required. The other argument should be set to 0.
 *   The ploygon is closed. ie a line is drawn from the first coordinates to the
 *   last corrdinates.
 * SOURCE
 *
...
win = dw_render_new( 0 )
...
x.0 = 3
x.1 = 10
x.2 = 20
x.3 = 30
y.0 = 3
y.1 = 15
y.2 = 45
y.3 = 77
Call dw_draw_polygon win, 0, !REXXDW.!DW_DRAW_FILL, 'x.', 'y.'
 ******
 */
rxfunc( DW_draw_polygon )
{
   HWND handle; /* local variable for arg[0] */
   HPIXMAP pixmap; /* local variable for arg[1] */
   int fill; /* local variable for arg[2] */
   int *x; /* local variable for arg[3] */
   int *y; /* local variable for arg[4] */
   int num_x, num_y;
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 5, 5 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&handle ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name, "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   if ( RxStrToULong( RxPackageGlobalData, &argv[1], (RX_ULONG *)&pixmap ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name, "*ERROR* Invalid number for argument %d.", 2 );
      return(1);
   }
   if ( RxStrToInt( RxPackageGlobalData, &argv[2], (RX_INT *)&fill ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name, "*ERROR* Invalid number for argument %d.", 3 );
      return(1);
   }
   if ( ( num_x = RxStemToIntArray( RxPackageGlobalData, &argv[3], (int * *)&x ) ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name, "*ERROR* Unable to convert X stem: \"%s\".", RXSTRPTR(argv[3]) );
      return(1);
   }
   if ( ( num_y = RxStemToIntArray( RxPackageGlobalData, &argv[4], (int * *)&y ) ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name, "*ERROR* Unable to convert Y stem: \"%s\".", RXSTRPTR(argv[4]) );
      return(1);
   }
   if ( num_x != num_y )
   {
      RxDisplayError( RxPackageGlobalData, name, "*ERROR* X and Y stems not same size." );
      return(1);
   }
   dw_draw_polygon( handle, pixmap, fill, num_x, x, y );
   RxFreeIntArray( RxPackageGlobalData, x  );
   RxFreeIntArray( RxPackageGlobalData, y  );
   return RxReturnString( RxPackageGlobalData, retstr, "" );
}

/****h* Widgets/EntryField
 * DESCRIPTION
 *   Entryfields allow a user to enter a single line of text.
 ******
 */

/****f* EntryField/DW_entryfield_new
 * NAME
 *   DW_entryfield_new
 * SYNOPSIS
 *   win = dw_entryfield_new( CurrentText, Id )
 * FUNCTION
 *   Creates a new entryfield.
 *   An entryfield is a single line window into which text can be entered.
 * ARGUMENTS
 *   - CurrentText - the initial text to display in the entryfield
 *   - Id          - a numeric identifier used to identify the window
 * RESULT
 *   A window identifier.
 * SEE ALSO
 *   DW_entryfield_password_new, DW_entryfield_set_limit
 * SOURCE
 *
...
entryfield = dw_entryfield_new( "Initial value", 0 )
 ******
 */
rxfunc( DW_entryfield_new )
{
   HWND rc; /* function return variable */
   char * text; /* local variable for arg[0] */
   unsigned long id; /* local variable for arg[1] */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 2, 2 ) )
      return( 1 );
   if ( RXVALIDSTRING(argv[0]) )
      text = RXSTRPTR(argv[0]);
   else
      text = "";
   if ( RxStrToULong( RxPackageGlobalData, &argv[1], (RX_ULONG *)&id ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 2 );
      return(1);
   }
   rc = dw_entryfield_new( text, id );
   return RxReturnUnsignedNumber( RxPackageGlobalData, retstr, (ULONG)rc );
}

/****f* EntryField/DW_entryfield_password_new
 * NAME
 *   DW_entryfield_password_new
 * SYNOPSIS
 *   win = dw_entryfield_password_new( CurrentText, Id )
 * FUNCTION
 *   Creates a new password entryfield.
 *   A password entryfield is similar to an entryfiled, but any character displayed
 *   in the entryfield is replaced by an asterisk to mask the contents.
 * ARGUMENTS
 *   - CurrentText - the initial text to display in the entryfield
 *   - Id          - a numeric identifier used to identify the window
 * RESULT
 *   A window identifier.
 * SEE ALSO
 *   DW_entryfield_new, DW_entryfield_set_limit
 * SOURCE
 *
...
entryfield = dw_entryfield_password_new( "mypassword", 0 )
 ******
 */
rxfunc( DW_entryfield_password_new )
{
   HWND rc; /* function return variable */
   char * text; /* local variable for arg[0] */
   ULONG id; /* local variable for arg[1] */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 2, 2 ) )
      return( 1 );
   if ( RXVALIDSTRING(argv[0]) )
      text = RXSTRPTR(argv[0]);
   else
      text = "";
   if ( RxStrToULong( RxPackageGlobalData, &argv[1], (RX_ULONG *)&id ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 2 );
      return(1);
   }
   rc = dw_entryfield_password_new( text, id );
   return RxReturnUnsignedNumber( RxPackageGlobalData, retstr, (ULONG)rc );
}

/****f* EntryField/DW_entryfield_set_limit
 * NAME
 *   DW_entryfield_set_limit
 * SYNOPSIS
 *   dw_entryfield_set_limit( Win, Length )
 * FUNCTION
 *   Sets the maximum number of characters that can be entered into an entryfield.
 * ARGUMENTS
 *   - Win        - the window identifier returned from DW_entryfield_new()
 *                  or DW_entryfield_password_new()
 *   - Length     - the maximum number of characters allowed
 * RESULT
 *   No return result.
 * SEE ALSO
 *   DW_entryfield_new, DW_entryfield_password_new
 * SOURCE
 *
...
entryfield = dw_entryfield_password_new( "mypassword", 0 )
Call dw_entryfield_set_limit entryfield, 10
 ******
 */
rxfunc( DW_entryfield_set_limit )
{
   HWND handle; /* local variable for arg[0] */
   ULONG limit; /* local variable for arg[1] */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 2, 2 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&handle ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   if ( RxStrToULong( RxPackageGlobalData, &argv[1], (RX_ULONG *)&limit ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 2 );
      return(1);
   }
   dw_entryfield_set_limit( handle, limit );
   return RxReturnString( RxPackageGlobalData, retstr, "" );
}

/****h* Widgets/ListboxAndCombobox
 * DESCRIPTION
 *   Listboxes provide a list of selectable items. A combobox is a listbox
 *   plus an entry field.
 ******
 */

/****f* ListboxAndCombobox/DW_listbox_new
 * NAME
 *   DW_listbox_new
 * SYNOPSIS
 *   win = dw_listbox_new( Id, Selection )
 * FUNCTION
 *   Creates a new listbox.
 *   A listbox contains a list of item values.
 * ARGUMENTS
 *   - Id         - a numeric identifier used to identify the window
 *   - Selection  - indicates if one or more items are selectable at once
 * RESULT
 *   A window identifier.
 * SEE ALSO
 *   DW_combobox_new, ListboxSelection
 * SOURCE
 *
...
listbox = dw_listbox_new( 10, !REXXDW.!DW_LB_SINGLE_SELECTION )
 ******
 */
rxfunc( DW_listbox_new )
{
   HWND rc; /* function return variable */
   unsigned long id; /* local variable for arg[0] */
   int multi; /* local variable for arg[1] */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 2, 2 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&id ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   if ( RxStrToInt( RxPackageGlobalData, &argv[1], (RX_INT *)&multi ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 2 );
      return(1);
   }
   rc = dw_listbox_new( id, multi );
   return RxReturnUnsignedNumber( RxPackageGlobalData, retstr, (ULONG)rc );
}

/****f* ListboxAndCombobox/DW_combobox_new
 * NAME
 *   DW_combobox_new
 * SYNOPSIS
 *   win = dw_combobox_new( ComboboxText, Id )
 * FUNCTION
 *   Creates a new combobox.
 *   A combobox contains a single line text entry field plus a selectable item
 *   from a list of pre set item values.
 * ARGUMENTS
 *   - ButtonText - the text to display as the default value of the combobox
 *   - Id         - a numeric identifier used to identify the window
 * RESULT
 *   A window identifier.
 * SEE ALSO
 *   DW_listbox_new
 * SOURCE
 *
...
combobox = dw_combobox_new( "Default", 0 )
 ******
 */
rxfunc( DW_combobox_new )
{
   HWND rc; /* function return variable */
   char * text; /* local variable for arg[0] */
   unsigned long id; /* local variable for arg[1] */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 2, 2 ) )
      return( 1 );
   if ( RXVALIDSTRING(argv[0]) )
      text = RXSTRPTR(argv[0]);
   else
      text = "";
   if ( RxStrToULong( RxPackageGlobalData, &argv[1], (RX_ULONG *)&id ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 2 );
      return(1);
   }
   rc = dw_combobox_new( text, id );
   return RxReturnUnsignedNumber( RxPackageGlobalData, retstr, (ULONG)rc );
}

/****f* ListboxAndCombobox/DW_listbox_selector_new
 * NAME
 *   DW_listbox_selector_new
 * SYNOPSIS
 *   win = dw_listbox_selector_new( LeftHeading, RightHeading, Width, Height )
 * FUNCTION
 *   Creates a new Listbox Selector widget.
 *   A Listbox Selector widget provides two listboxes side-by-side with buttons to move items
 *   from one listbox to another.
 * ARGUMENTS
 *   - LeftHeading  - the text to appear above the left Listbox
 *   - RightHeading - the text to appear above the right Listbox
 *   - Width        - the minumum width of the Listboxes
 *   - Height       - the minumum height of the Listboxes
 * RESULT
 *   A window identifier, which is actually a Box widget suitable for packing.
 * SOURCE
 *
ls = dw_listbox_selector_new( 'Selected', 'Available', 100, 200 )
value.0 = 3
value.1 = 'line 1'
value.2 = 'line 2'
value.3 = 'line 3'
ind.0 = 3
ind.1 = 1
ind.2 = 1
ind.3 = 0
Call dw_listbox_selector_setup( ls, 'value.', 'ind.' )
...
Do i = 1 To value.0
   If ind.i = 1 Then Say 'Value:' value.i 'has been selected'
End
 ******
 */
rxfunc( DW_listbox_selector_new )
{
   char *LeftHeading; /* local variable for arg[0] */
   char *RightHeading; /* local variable for arg[1] */
   int width, height;
   HWND tmp,heading;
   REXXDWWidgetData *widget;
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 4, 4 ) )
      return( 1 );
   if ( RXVALIDSTRING(argv[0]) )
      LeftHeading = RXSTRPTR(argv[0]);
   else
      LeftHeading = "";
   if ( RXVALIDSTRING(argv[1]) )
      RightHeading = RXSTRPTR(argv[1]);
   else
      RightHeading = "";
   if ( RxStrToInt( RxPackageGlobalData, &argv[2], (RX_INT *)&width ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 3 );
      return(1);
   }
   if ( RxStrToInt( RxPackageGlobalData, &argv[3], (RX_INT *)&height ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 4 );
      return(1);
   }

   widget = addWidget( RxPackageGlobalData, REXXDW_WIDGET_LISTBOX_SELECTOR );
   if ( widget == NULL )
      return(1);

   widget->widget.ListboxSelectorData.Topbox = dw_box_new( DW_HORZ, 0 ); /* packed by caller */
   /*
    * Left box
    */
   tmp = dw_box_new( DW_VERT, 0 );
   dw_box_pack_start( widget->widget.ListboxSelectorData.Topbox, tmp, 0, 0, DW_EXPAND_HORZ, DW_EXPAND_VERT, 0 );
   heading = dw_text_new( LeftHeading, 0 );
   dw_box_pack_start( tmp, heading, width, 20, DW_EXPAND_HORZ, DW_DONT_EXPAND_VERT, 2 );
   dw_window_set_style( heading, DW_DT_CENTER | DW_DT_VCENTER , DW_DT_CENTER | DW_DT_VCENTER );
   widget->widget.ListboxSelectorData.LeftListbox = dw_listbox_new( 101, DW_MULTIPLE_SELECTION );
   dw_box_pack_start( tmp, widget->widget.ListboxSelectorData.LeftListbox, width, height, DW_EXPAND_HORZ, DW_EXPAND_VERT, 2 );
   /*
    * Middle box
    */
   tmp = dw_box_new( DW_VERT, 0 );
   dw_box_pack_start( widget->widget.ListboxSelectorData.Topbox, tmp, 30, 0, DW_DONT_EXPAND_HORZ, DW_EXPAND_VERT, 2 );

   heading = dw_text_new( " ", 0 );
   dw_box_pack_start( tmp, heading, 1, 20, DW_EXPAND_HORZ, DW_DONT_EXPAND_VERT, 2 );
   dw_box_pack_start( tmp, 0, 0, 0, DW_EXPAND_HORZ, DW_EXPAND_VERT, 0 ); /* blank space */
   widget->widget.ListboxSelectorData.LeftButton = dw_bitmapbutton_new_from_data( "Move selected to left", 0, leftico, sizeof(leftico) );
   dw_box_pack_start( tmp, widget->widget.ListboxSelectorData.LeftButton, 30, 30, DW_DONT_EXPAND_HORZ, DW_DONT_EXPAND_VERT, 0 );
   dw_window_set_style( widget->widget.ListboxSelectorData.LeftButton, DW_BS_NOBORDER, DW_BS_NOBORDER );
   dw_box_pack_start( tmp, 0, 0, 2, DW_EXPAND_HORZ, DW_DONT_EXPAND_VERT, 0 ); /* blank space */
   widget->widget.ListboxSelectorData.RightButton = dw_bitmapbutton_new_from_data( "Move selected to right", 0, rightico, sizeof(rightico) );
   dw_box_pack_start( tmp, widget->widget.ListboxSelectorData.RightButton, 30, 30, DW_DONT_EXPAND_HORZ, DW_DONT_EXPAND_VERT, 0 );
   dw_window_set_style( widget->widget.ListboxSelectorData.RightButton, DW_BS_NOBORDER, DW_BS_NOBORDER );
   dw_box_pack_start( tmp, 0, 0, 0, DW_EXPAND_HORZ, DW_EXPAND_VERT, 0 ); /* blank space */
   widget->widget.ListboxSelectorData.AllLeftButton = dw_bitmapbutton_new_from_data( "Move all to left", 0, allleftico, sizeof(allleftico) );
   dw_box_pack_start( tmp, widget->widget.ListboxSelectorData.AllLeftButton, 30, 30, DW_DONT_EXPAND_HORZ, DW_DONT_EXPAND_VERT, 0 );
   dw_window_set_style( widget->widget.ListboxSelectorData.AllLeftButton, DW_BS_NOBORDER, DW_BS_NOBORDER );
   dw_box_pack_start( tmp, 0, 0, 2, DW_EXPAND_HORZ, DW_DONT_EXPAND_VERT, 0 ); /* blank space */
   widget->widget.ListboxSelectorData.AllRightButton = dw_bitmapbutton_new_from_data( "Move all to right", 0, allrightico, sizeof(allrightico) );
   dw_box_pack_start( tmp, widget->widget.ListboxSelectorData.AllRightButton, 30, 30, DW_DONT_EXPAND_HORZ, DW_DONT_EXPAND_VERT, 0 );
   dw_window_set_style( widget->widget.ListboxSelectorData.AllRightButton, DW_BS_NOBORDER, DW_BS_NOBORDER );
   dw_box_pack_start( tmp, 0, 0, 0, DW_EXPAND_HORZ, DW_EXPAND_VERT, 0 ); /* blank space */
   /*
    * Right box
    */
   tmp = dw_box_new( DW_VERT, 0 );
   dw_box_pack_start( widget->widget.ListboxSelectorData.Topbox, tmp, 0, 0, DW_EXPAND_HORZ, DW_EXPAND_VERT, 0 );
   heading = dw_text_new( RightHeading, 0 );
   dw_box_pack_start( tmp, heading, width, 20, DW_EXPAND_HORZ, DW_DONT_EXPAND_VERT, 2 );
   dw_window_set_style( heading, DW_DT_CENTER | DW_DT_VCENTER , DW_DT_CENTER | DW_DT_VCENTER );
   widget->widget.ListboxSelectorData.RightListbox = dw_listbox_new( 102, DW_MULTIPLE_SELECTION );
   dw_box_pack_start( tmp, widget->widget.ListboxSelectorData.RightListbox, width, height, DW_EXPAND_HORZ, DW_EXPAND_VERT, 2 );
   /*
    * Add pointer to thread-safe data
    */
   widget->widget.ListboxSelectorData.RxPackageGlobalData = RxPackageGlobalData;

   dw_signal_connect( widget->widget.ListboxSelectorData.LeftButton, DW_SIGNAL_CLICKED, DW_SIGNAL_FUNC(RexxDWListBoxSelectorMoveItemLeftCallback), &widget->widget.ListboxSelectorData );
   dw_signal_connect( widget->widget.ListboxSelectorData.RightButton, DW_SIGNAL_CLICKED, DW_SIGNAL_FUNC(RexxDWListBoxSelectorMoveItemRightCallback), &widget->widget.ListboxSelectorData );
   dw_signal_connect( widget->widget.ListboxSelectorData.AllLeftButton, DW_SIGNAL_CLICKED, DW_SIGNAL_FUNC(RexxDWListBoxSelectorMoveAllItemLeftCallback), &widget->widget.ListboxSelectorData );
   dw_signal_connect( widget->widget.ListboxSelectorData.AllRightButton, DW_SIGNAL_CLICKED, DW_SIGNAL_FUNC(RexxDWListBoxSelectorMoveAllItemRightCallback), &widget->widget.ListboxSelectorData );
   dw_signal_connect( widget->widget.ListboxSelectorData.Topbox, DW_SIGNAL_DELETE, DW_SIGNAL_FUNC(RexxDWListBoxSelectorDeleteCallback), &widget->widget.ListboxSelectorData );

   return RxReturnUnsignedNumber( RxPackageGlobalData, retstr, (ULONG)widget->widget.ListboxSelectorData.Topbox );
}

/****f* ListboxAndCombobox/DW_listbox_selector_setup
 * NAME
 *   DW_listbox_selector_setup
 * SYNOPSIS
 *   dw_listbox_selector_setup( Win, ValueArray, SetArray )
 * FUNCTION
 *   Populates a listbox selector widget.
 * ARGUMENTS
 *   - Win          - the window identifier returned from DW_listbox_selector_new()
 *   - ValueArray   - an array containing the text to be displayed in the Listboxes
 *   - SetArray     - an array specifying which values in ValueArray are displayed
 *                    in which Listbox
 * RESULT
 *   No return result.
 * NOTES
 *   The ValueArray is a Rexx "array" containing the items that are to be displayed
 *   in the Listboxes. The SetArray is a Rexx "array" with values set to 1 or 0.
 *   When creating the Listbox Selector, if an item in the SetArray has a value of 1
 *   the corresponding item in the ValueArray is displayed in the left Listbox.
 *   When an item is moved from the left Listbox to the right Listbox, the corresponding
 *   item in the SetArray is set to 1.
 *   When an item is moved from the right Listbox to the left Listbox, the corresponding
 *   item in the SetArray is set to 0.
 *   The ValueArray and SetArray stem variable names should be global, so that they can
 *   be referenced inside RexxDW callbacks.
 * SOURCE
 *
ls = dw_listbox_selector_new( 'Selected', 'Available', 100, 200 )
value.0 = 3
value.1 = 'line 1'
value.2 = 'line 2'
value.3 = 'line 3'
ind.0 = 3
ind.1 = 1
ind.2 = 1
ind.3 = 0
Call dw_listbox_selector_setup( ls, 'value.', 'ind.' )
...
Do i = 1 To value.0
   If ind.i = 1 Then Say 'Value:' value.i 'has been selected'
End
 ******
 */
rxfunc( DW_listbox_selector_setup )
{
   HWND handle; /* local variable for arg[0] */
   REXXDWWidgetData *widget;
   int count1,count2,len;
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 3, 3 ) )
      return( 1 );

   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&handle ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name, "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }

   /*
    * Validate the stem names and that they have .0 values that are equal
    */
   if ( !RXVALIDSTRING( argv[1] ) )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid Stem Name for argument %d.", 2 );
      return(1);
   }
   len = RXSTRLEN( argv[1] );
   if ( argv[1].strptr[len-1] != '.' )
   {
      RxDisplayError( RxPackageGlobalData, name, "*ERROR* Argument %d is not a Stem Name (no trailing period).", 2 );
      return(1);
   }
   if ( GetRexxVariableInteger( RxPackageGlobalData, RXSTRPTR(argv[1]), &count1, 0 ) == NULL )
   {
      RxDisplayError( RxPackageGlobalData, name, "*ERROR* Invalid number for %s0.", RXSTRPTR(argv[1]) );
      return(1);
   }

   if ( !RXVALIDSTRING( argv[2] ) )
   {
      RxDisplayError( RxPackageGlobalData, name, "*ERROR* Invalid Stem Name for argument %d.", 3 );
      return(1);
   }
   len = RXSTRLEN( argv[2] );
   if ( argv[2].strptr[len-1] != '.' )
   {
      RxDisplayError( RxPackageGlobalData, name, "*ERROR* Argument %d is not a Stem Name (no trailing period).", 3 );
      return(1);
   }
   if ( GetRexxVariableInteger( RxPackageGlobalData, RXSTRPTR(argv[2]), &count2, 0 ) == NULL )
   {
      RxDisplayError( RxPackageGlobalData, name, "*ERROR* Invalid number for %s0.", RXSTRPTR(argv[2]) );
      return(1);
   }

   if ( count1 != count2 )
   {
      RxDisplayError( RxPackageGlobalData, name, "*ERROR* Size of Value stem: %d is not the same as Indicator stem: %d.", count1, count2 );
      return(1);
   }

   widget = findWidget( RxPackageGlobalData, handle );
   if ( widget == NULL )
   {
      RxDisplayError( RxPackageGlobalData, name, "*ERROR* Window handle passed is not a Listbox Selector." );
      return(1);
   }
   /*
    * In case we didn't call dw_listbox_selector_clear() before resetting the contents...
    */
   RexxDWListboxSelectorClear( &widget->widget.ListboxSelectorData );

   /*
    * Specify the data for this widget type
    */
   widget->widget.ListboxSelectorData.count = count1;
   widget->widget.ListboxSelectorData.LeftIndex = calloc( count1, sizeof(int) );
   widget->widget.ListboxSelectorData.RightIndex = calloc( count1, sizeof(int) );

   strcpy( widget->widget.ListboxSelectorData.ValueStemName, RXSTRPTR(argv[1]) );
   strcpy( widget->widget.ListboxSelectorData.SetStemName, RXSTRPTR(argv[2]) );

   RexxDWListboxSelectorPopulate( &widget->widget.ListboxSelectorData );

   return RxReturnString( RxPackageGlobalData, retstr, "" );
}

/****f* ListboxAndCombobox/DW_listbox_selector_clear
 * NAME
 *   DW_listbox_selector_clear
 * SYNOPSIS
 *   dw_listbox_selector_clear( Win )
 * FUNCTION
 *   Clears all data from a listbox selector widget.
 * ARGUMENTS
 *   - Win          - the window identifier returned from DW_listbox_selector_new()
 * RESULT
 *   No return result.
 * SOURCE
 *
ls = dw_listbox_selector_new( 'Selected', 'Available', 100, 200 )
value.0 = 3
value.1 = 'line 1'
value.2 = 'line 2'
value.3 = 'line 3'
ind.0 = 3
ind.1 = 1
ind.2 = 1
ind.3 = 0
Call dw_listbox_selector_setup( ls, 'value.', 'ind.' )
...
Do i = 1 To value.0
   If ind.i = 1 Then Say 'Value:' value.i 'has been selected'
End
...
Call dw_listbox_selector_clear( ls )
 ******
 */
rxfunc( DW_listbox_selector_clear )
{
   HWND handle; /* local variable for arg[0] */
   REXXDWWidgetData *widget;
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 1, 1 ) )
      return( 1 );

   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&handle ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name, "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }

   widget = findWidget( RxPackageGlobalData, handle );
   if ( widget == NULL )
   {
      RxDisplayError( RxPackageGlobalData, name, "*ERROR* Window handle passed is not a Listbox Selector." );
      return(1);
   }

   /*
    * Clear the data...
    */
   RexxDWListboxSelectorClear( &widget->widget.ListboxSelectorData );

   return RxReturnString( RxPackageGlobalData, retstr, "" );
}

/****f* ListboxAndCombobox/DW_listbox_append
 * NAME
 *   DW_listbox_append
 * SYNOPSIS
 *   dw_listbox_append( Win, Text )
 * FUNCTION
 *   Appends the specified Text to the end of the current items in the listbox
 *   or combobox specified by Win.
 * ARGUMENTS
 *   - Win        - the window identifier returned from DW_listbox_new()
 *                  or DW_combobox_new
 *   - Text       - the text of the item to append to the list
 * RESULT
 *   No return result.
 * SEE ALSO
 *   DW_listbox_list_append, DW_listbox_insert
 * SOURCE
 *
listbox = dw_listbox_new( 0, !REXXDW.!DW_LB_SINGLE_SELECTION )
Do i = 1 To 5
   Call dw_listbox_append listbox, text.i
End
 ******
 */
rxfunc( DW_listbox_append )
{
   HWND handle; /* local variable for arg[0] */
   char * text; /* local variable for arg[1] */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 2, 2 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&handle ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   if ( RXVALIDSTRING(argv[1]) )
      text = RXSTRPTR(argv[1]);
   else
      text = "";
   dw_listbox_append( handle, text );
   return RxReturnString( RxPackageGlobalData, retstr, "" );
}

/****f* ListboxAndCombobox/DW_listbox_list_append
 * NAME
 *   DW_listbox_list_append
 * SYNOPSIS
 *   dw_listbox_list_append( Win, TextArray )
 * FUNCTION
 *   Appends each item in the specified TextArray to the end of the current
 * items in the listbox or combobox specified by Win.
 * ARGUMENTS
 *   - Win        - the window identifier returned from DW_listbox_new()
 *                  or DW_combobox_new
 *   - TextArray  - an array containing text of the items to append to the list
 * RESULT
 *   No return result.
 * SEE ALSO
 *   DW_listbox_append, DW_listbox_insert
 * SOURCE
 *
listbox = dw_listbox_new( 0, !REXXDW.!DW_LB_SINGLE_SELECTION )
text.0 = 3
text.1 = 'line 1'
text.2 = 'line 2'
text.3 = 'line 3'
Call dw_listbox_list_append listbox, 'text.'
 ******
 */
rxfunc( DW_listbox_list_append )
{
   HWND handle; /* local variable for arg[0] */
   char ** text; /* local variable for arg[1] */
   int num_args;
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 2, 2 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&handle ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   if ( ( num_args = RxStemToCharArray( RxPackageGlobalData, &argv[1], (char ** *)&text ) ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Unable to convert Text stem: \"%s\".", RXSTRPTR(argv[1]) );
      return(1);
   }
   dw_listbox_list_append( handle, text, num_args );
   RxFreeCharArray( RxPackageGlobalData, text, num_args );
   return RxReturnString( RxPackageGlobalData, retstr, "" );
}

/****f* ListboxAndCombobox/DW_listbox_insert
 * NAME
 *   DW_listbox_insert
 * SYNOPSIS
 *   dw_listbox_insert( Win, Text, Pos )
 * FUNCTION
 *   Appends the specified Text into the listbox
 *   or combobox specified by Win at the specified Pos.
 * ARGUMENTS
 *   - Win        - the window identifier returned from DW_listbox_new()
 *                  or DW_combobox_new
 *   - Text       - the text of the item to insert into the list
 *   - Pos        - the position at which the text is to be inserted.
 *                  If zero, negative, or greater than the number of existing
 *                  items, Text will be appended
 * RESULT
 *   No return result.
 * SEE ALSO
 *   DW_listbox_append
 * SOURCE
 *
listbox = dw_listbox_new( 0, !REXXDW.!DW_LB_SINGLE_SELECTION )
Call dw_listbox_insert listbox, 'first', 1
Call dw_listbox_insert listbox, 'last', 1
 ******
 */
rxfunc( DW_listbox_insert )
{
   HWND handle; /* local variable for arg[0] */
   char * text; /* local variable for arg[1] */
   int pos;
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 3, 3 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&handle ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   if ( RXVALIDSTRING(argv[1]) )
      text = RXSTRPTR(argv[1]);
   else
      text = "";
   if ( RxStrToInt( RxPackageGlobalData, &argv[2], (RX_INT *)&pos ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 3 );
      return(1);
   }
   /*
    * Make pos zero-based per API
    */
   pos--;
   dw_listbox_insert( handle, text, pos );
   return RxReturnString( RxPackageGlobalData, retstr, "" );
}

/****f* ListboxAndCombobox/DW_listbox_clear
 * NAME
 *   DW_listbox_clear
 * SYNOPSIS
 *   dw_listbox_clear( Win )
 * FUNCTION
 *   Removes all items from the listbox or combobox specified by Win.
 * ARGUMENTS
 *   - Win        - the window identifier returned from DW_listbox_new()
 *                  or DW_combobox_new
 * RESULT
 *   No return result.
 * SEE ALSO
 *   DW_listbox_delete
 * SOURCE
 *
listbox = dw_listbox_new( 0, !REXXDW.!DW_LB_SINGLE_SELECTION )
Do i = 1 To 5
   Call dw_listbox_append listbox, text.i
End
...
Call dw_listbox_clear( listbox )
 ******
 */
rxfunc( DW_listbox_clear )
{
   HWND handle; /* local variable for arg[0] */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 1, 1 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&handle ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   dw_listbox_clear( handle );
   return RxReturnString( RxPackageGlobalData, retstr, "" );
}

/****f* ListboxAndCombobox/DW_listbox_count
 * NAME
 *   DW_listbox_count
 * SYNOPSIS
 *   count = dw_listbox_count( Win )
 * FUNCTION
 *   Returns the number of items in the listbox or combobox specified by Win.
 * ARGUMENTS
 *   - Win        - the window identifier returned from DW_listbox_new()
 *                  or DW_combobox_new
 * RESULT
 *   The number of items in the list.
 * SOURCE
 *
combobox = dw_combobox_new( "Default", 0 )
Do i = 1 To 5
   Call dw_listbox_append combobox, text.i
End
Say 'There are' dw_listbox_count( combobox ) 'items in the list'
 ******
 */
rxfunc( DW_listbox_count )
{
   int rc; /* function return variable */
   HWND handle; /* local variable for arg[0] */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 1, 1 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&handle ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   rc = dw_listbox_count( handle );
   return RxReturnNumber( RxPackageGlobalData, retstr, rc );
}

/****f* ListboxAndCombobox/DW_listbox_delete
 * NAME
 *   DW_listbox_delete
 * SYNOPSIS
 *   dw_listbox_delete( Win, Index )
 * FUNCTION
 *   Deletes the item specified by Index from the list of items in the listbox
 *   or combobox specified by Win.
 * ARGUMENTS
 *   - Win        - the window identifier returned from DW_listbox_new()
 *                  or DW_combobox_new
 *   - Index      - the 0 based index of the item to delete
 * RESULT
 *   No return result.
 * SEE ALSO
 *   DW_listbox_clear
 * SOURCE
 *
combobox = dw_combobox_new( "Default", 0 )
Do i = 1 To 5
   Call dw_listbox_append combobox, text.i
End
...
Call dw_listbox_delete combobox, 4
 ******
 */
rxfunc( DW_listbox_delete )
{
   HWND handle; /* local variable for arg[0] */
   int index; /* local variable for arg[1] */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 2, 2 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&handle ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   if ( RxStrToInt( RxPackageGlobalData, &argv[1], (RX_INT *)&index ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 2 );
      return(1);
   }
   dw_listbox_delete( handle, index );
   return RxReturnString( RxPackageGlobalData, retstr, "" );
}

/****f* ListboxAndCombobox/DW_listbox_select
 * NAME
 *   DW_listbox_select
 * SYNOPSIS
 *   dw_listbox_select( Win, Index, State )
 * FUNCTION
 *   Sets the State of the item specified by Index in the list of items in the listbox
 *   or combobox specified by Win.
 * ARGUMENTS
 *   - Win        - the window identifier returned from DW_listbox_new()
 *                  or DW_combobox_new
 *   - Index      - the 0 based index of the item to change its state
 *   - State      - indicates whether the item is to be selected or not
 * RESULT
 *   No return result.
 * SEE ALSO
 *   DW_listbox_selected, ListboxSelected
 * NOTES
 *   If the listbox was created with DW_LB_SINGLE_SELECTION, then setting an item
 *   to be selected will unselect an item that is currently selected.
 *   If the listbox was created with DW_LB_MULTIPLE_SELECTION, then setting an item
 *   to be selected will leave any other item already selected alone.
 * SOURCE
 *
listbox = dw_listbox_new( 10, !REXXDW.!DW_LB_SINGLE_SELECTION )
Do i = 1 To 5
   Call dw_listbox_append listbox, text.i
End
...
Call dw_listbox_select listbox, 4, !REXXDW.!DW_LB_SELECTED
 ******
 */
rxfunc( DW_listbox_select )
{
   HWND handle; /* local variable for arg[0] */
   int index; /* local variable for arg[1] */
   int state; /* local variable for arg[2] */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 3, 3 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&handle ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   if ( RxStrToInt( RxPackageGlobalData, &argv[1], (RX_INT *)&index ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 2 );
      return(1);
   }
   if ( RxStrToInt( RxPackageGlobalData, &argv[2], (RX_INT *)&state ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 3 );
      return(1);
   }
   dw_listbox_select( handle, index, state );
   return RxReturnString( RxPackageGlobalData, retstr, "" );
}

/****f* ListboxAndCombobox/DW_listbox_selected
 * NAME
 *   DW_listbox_selected
 * SYNOPSIS
 *   index = dw_listbox_selected( Win )
 * FUNCTION
 *   Returns the index of the item in the listbox or combobox currently
 *   selected or !REXXDW.!DW_LB_NONE if none selected.
 * ARGUMENTS
 *   - Win        - the window identifier returned from DW_listbox_new()
 *                  or DW_combobox_new
 * RESULT
 *   The 0 based index of the item selected.
 * SEE ALSO
 *   ListboxReturn, DW_listbox_select, DW_listbox_selected_multi
 * SOURCE
 *
listbox = dw_listbox_new( 10, !REXXDW.!DW_LB_SINGLE_SELECTION )
Do i = 1 To 5
   Call dw_listbox_append listbox, text.i
End
...
Call dw_listbox_select listbox, 4, !REXXDW.!DW_LB_SELECTED
...
Say 'Item' dw_listbox_selected( listbox ) 'is currently selected'
 ******
 */
rxfunc( DW_listbox_selected )
{
   int rc; /* function return variable */
   HWND handle; /* local variable for arg[0] */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 1, 1 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&handle ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   rc = dw_listbox_selected( handle );
   return RxReturnNumber( RxPackageGlobalData, retstr, rc );
}

/****f* ListboxAndCombobox/DW_listbox_selected_multi
 * NAME
 *   DW_listbox_selected_multi
 * SYNOPSIS
 *   index = dw_listbox_selected_multi( Win, Start )
 * FUNCTION
 *   Returns the index of the next item in the listbox currently
 *   selected or !REXXDW.!DW_LB_NONE if none selected.
 * ARGUMENTS
 *   - Win        - the window identifier returned from DW_listbox_new()
 *   - Start      - the index to start looking from
 * RESULT
 *   The 0 based index of the next item selected.
 * SEE ALSO
 *   ListboxReturn, DW_listbox_select, DW_listbox_selected
 * NOTES
 *   This call only works on listboxes created with DW_LB_MULTIPLE_SELECTION;
 *   and not on comboboxes.
 *   The first call should use -1 as Start, and on subsequent calls, the
 *   index returned from the previous call.
 * SOURCE
 *
listbox = dw_listbox_new( 10, !REXXDW.!DW_LB_MULTIPLE_SELECTION )
Do i = 1 To 5
   Call dw_listbox_append listbox, text.i
End
...
Call dw_listbox_select listbox, 2, !REXXDW.!DW_LB_SELECTED
Call dw_listbox_select listbox, 4, !REXXDW.!DW_LB_SELECTED
...
idx = -1
Do Forever
   idx = dw_listbox_selected_multi( listbox, idx )
   If idx = !REXXDW.!DW_LB_NONE Then Leave
   Say 'Item' idx 'is currently selected'
End
 ******
 */
rxfunc( DW_listbox_selected_multi )
{
   int rc; /* function return variable */
   HWND handle; /* local variable for arg[0] */
   int where; /* local variable for arg[1] */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 2, 2 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&handle ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   if ( RxStrToInt( RxPackageGlobalData, &argv[1], (RX_INT *)&where ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 2 );
      return(1);
   }
   rc = dw_listbox_selected_multi( handle, where );
   return RxReturnNumber( RxPackageGlobalData, retstr, rc );
}

/****f* ListboxAndCombobox/DW_listbox_set_top
 * NAME
 *   DW_listbox_set_top
 * SYNOPSIS
 *   dw_listbox_set_top( Win, Index )
 * FUNCTION
 *   Sets the item with Index to be the first item displayed in the viewport of
 *   the listbox.
 * ARGUMENTS
 *   - Win        - the window identifier returned from DW_listbox_new()
 *   - Index      - the 0 based index of the item to display at the top of the viewport
 * RESULT
 *   No return result.
 * NOTES
 *   This does not work on comboboxes.
 * SOURCE
 *
listbox = dw_listbox_new( 10, !REXXDW.!DW_LB_SINGLE_SELECTION )
Do i = 1 To 5
   Call dw_listbox_append listbox, text.i
End
...
Call dw_listbox_set_top listbox, 4
 ******
 */
rxfunc( DW_listbox_set_top )
{
   HWND handle; /* local variable for arg[0] */
   int top; /* local variable for arg[1] */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 2, 2 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&handle ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   if ( RxStrToInt( RxPackageGlobalData, &argv[1], (RX_INT *)&top ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 2 );
      return(1);
   }
   dw_listbox_set_top( handle, top );
   return RxReturnString( RxPackageGlobalData, retstr, "" );
}

/****f* ListboxAndCombobox/DW_listbox_set_text
 * NAME
 *   DW_listbox_set_text
 * SYNOPSIS
 *   dw_listbox_set_text( Win, Index, Text )
 * FUNCTION
 *   Sets the Text of the Index item in the listbox or combobox
 * ARGUMENTS
 *   - Win        - the window identifier returned from DW_listbox_new()
 *                  or DW_combobox_new
 *   - Index      - the 0 based index of the item to change its text
 *   - Text       - the new text of the item
 * RESULT
 *   No return result.
 * SEE ALSO
 *   DW_listbox_append
 * SOURCE
 *
listbox = dw_listbox_new( 10, !REXXDW.!DW_LB_SINGLE_SELECTION )
Do i = 1 To 5
   Call dw_listbox_append listbox, text.i
End
...
Call dw_listbox_set_text listbox, 4, 'New Text'
 ******
 */
rxfunc( DW_listbox_set_text )
{
   HWND handle; /* local variable for arg[0] */
   unsigned int index; /* local variable for arg[1] */
   char * buffer; /* local variable for arg[2] */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 3, 3 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&handle ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   if ( RxStrToUInt( RxPackageGlobalData, &argv[1], (RX_UINT *)&index ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 2 );
      return(1);
   }
   if ( RXVALIDSTRING(argv[2]) )
      buffer = RXSTRPTR(argv[2]);
   else
      buffer = "";
   dw_listbox_set_text( handle, index, buffer );
   return RxReturnString( RxPackageGlobalData, retstr, "" );
}

/****f* ListboxAndCombobox/DW_listbox_get_text
 * NAME
 *   DW_listbox_get_text
 * SYNOPSIS
 *   text = dw_listbox_get_text( Win, Index )
 * FUNCTION
 *   Returns the text of the item with Index in the listbox or combobox
 * ARGUMENTS
 *   - Win        - the window identifier returned from DW_listbox_new()
 *                  or DW_combobox_new
 *   - Index      - 0 based index into the list of items for which the text is required
 * RESULT
 *   The text of the item.
 * SOURCE
 *
combobox = dw_combobox_new( "Default", 0 )
Do i = 1 To 5
   Call dw_listbox_append combobox, text.i
End
Say 'Item 3 is' dw_listbox_get_text( combobox, 3 )
 ******
 */
rxfunc( DW_listbox_get_text )
{
   HWND handle; /* local variable for arg[0] */
   unsigned int index; /* local variable for arg[1] */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 2, 2 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&handle ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   if ( RxStrToUInt( RxPackageGlobalData, &argv[1], (RX_UINT *)&index ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 2 );
      return(1);
   }
   dw_listbox_get_text( handle, index, retstr->strptr, RXAUTOBUFLEN );
   retstr->strlength = strlen( retstr->strptr );
   return RxReturn( RxPackageGlobalData, retstr );
}

/****h* Widgets/EmbeddedHTML
 * DESCRIPTION
 *   The embedded HTML widget provides a means to display HTML in a window.
 ******
 */

/****f* EmbeddedHTML/DW_html_new
 * NAME
 *   DW_html_new
 * SYNOPSIS
 *   win = dw_html_new( Id )
 * FUNCTION
 *   Creates a new HTML widget.
 * ARGUMENTS
 *   - Id         - a numeric identifier used to identify the window
 * RESULT
 *   A window identifier.
 * SOURCE
 *
...
html = dw_html_new( 10 )
 ******
 */
rxfunc( DW_html_new )
{
   HWND rc; /* function return variable */
   unsigned long id; /* local variable for arg[0] */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 1, 1 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&id ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   rc = dw_html_new( id );
   return RxReturnUnsignedNumber( RxPackageGlobalData, retstr, (ULONG)rc );
}

/****f* EmbeddedHTML/DW_html_url
 * NAME
 *   DW_html_url
 * SYNOPSIS
 *   rc = dw_html_url( Win, Url )
 * FUNCTION
 *   Displays a URL in a HTML widget.
 * ARGUMENTS
 *   - Win        - the window identifier returned from DW_html_new()
 *   - Url        - the URL of the HTML page to display
 * RESULT
 *   0 if successful; any other value is an error
 * SOURCE
 *
...
html = dw_html_new( 10 )
Call dw_html_url( html, 'http://rexxdw.sf.net' )
 ******
 */
rxfunc( DW_html_url )
{
   HWND handle; /* local variable for arg[0] */
   char * url; /* local variable for arg[1] */
   int rc;
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 2, 2 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&handle ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   // TODO - memory leak
   if ( RXVALIDSTRING(argv[1]) )
      url = strdup( RXSTRPTR(argv[1]) );
   else
   {
      RxDisplayError( RxPackageGlobalData, name, "*ERROR* Supplied URL is empty." );
      return(1);
   }
   rc = dw_html_url( handle, url );
   return RxReturnUnsignedNumber( RxPackageGlobalData, retstr, (ULONG)rc );
}

/****f* EmbeddedHTML/DW_html_raw
 * NAME
 *   DW_html_raw
 * SYNOPSIS
 *   rc = dw_html_raw( Win, Raw )
 * FUNCTION
 *   DIsplays raw HTML in a HTML widget.
 * ARGUMENTS
 *   - Win        - the window identifier returned from DW_html_new()
 *   - Raw        - the raw HTML
 * RESULT
 *   0 if successful; any other value is an error
 * SOURCE
 *
...
html = dw_html_new( 10 )
Call dw_html_raw( html, '<html><body><h1>Rexx/DW rules</h1></body></html>' )
 ******
 */
rxfunc( DW_html_raw )
{
   HWND handle; /* local variable for arg[0] */
   char * raw; /* local variable for arg[1] */
   int rc;
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 2, 2 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&handle ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   // TODO - memory leak
   if ( RXVALIDSTRING(argv[1]) )
      raw = strdup( RXSTRPTR(argv[1]) );
   else
   {
      RxDisplayError( RxPackageGlobalData, name, "*ERROR* Supplied HTML data is empty." );
      return(1);
   }
   rc = dw_html_raw( handle, raw );
   return RxReturnUnsignedNumber( RxPackageGlobalData, retstr, (ULONG)rc );
}

/****f* EmbeddedHTML/DW_html_action
 * NAME
 *   DW_html_action
 * SYNOPSIS
 *   rc = dw_html_action( Win, Action
 * FUNCTION
 *   Invoke specified action on HTML widget.
 * ARGUMENTS
 *   - Win        - the window identifier returned from DW_html_new()
 *   - Action     - the action to execute. see HTMLActions
 * RESULT
 *   0 if successful; any other value is an error
 * SEE ALSO
 *   HTMLActions
 * SOURCE
 *
...
html = dw_html_new( 10 )
Call dw_html_action( html, !REXXDW.!DW_HTML_GOHOME )
 ******
 */
rxfunc( DW_html_action )
{
   HWND handle; /* local variable for arg[0] */
   int action;  /* local variable for arg[1] */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 2, 2 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&handle ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   if ( RxStrToInt( RxPackageGlobalData, &argv[1], (RX_INT *)&action ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 2 );
      return(1);
   }
   dw_html_action( handle, action );
   return RxReturnString( RxPackageGlobalData, retstr, "" );
}

/****h* Widgets/Menu
 * DESCRIPTION
 *   Menus provide a list of textual options to the user to select
 *   an action. RexxDW provides standard drop-down menus located directly
 *   under the window's titlebar, or floating popup menus.
 ******
 */

/****f* Menu/DW_menu_new
 * NAME
 *   DW_menu_new
 * SYNOPSIS
 *   win = dw_menu_new( Id )
 * FUNCTION
 *   Creates a new menu.
 *   This menu can be used as a submenu linked to a menubar, or as a popup menu.
 * ARGUMENTS
 *   - Id         - a numeric identifier used to identify the window
 * RESULT
 *   A window identifier.
 * SEE ALSO
 *   DW_menubar_new
 * SOURCE
 *
...
menu = dw_menu_new( 10 )
 ******
 */
rxfunc( DW_menu_new )
{
   HMENUI rc; /* function return variable */
   unsigned long id; /* local variable for arg[0] */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 1, 1 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&id ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   rc = dw_menu_new( id );
   return RxReturnUnsignedNumber( RxPackageGlobalData, retstr, (ULONG)rc );
}

/****f* Menu/DW_menubar_new
 * NAME
 *   DW_menubar_new
 * SYNOPSIS
 *   win = dw_menubar_new( Win )
 * FUNCTION
 *   Creates a new menubar.
 *   A menubar is located at the top of the specified toplevel window
 *   identified by Win.
 * ARGUMENTS
 *   - Win        - a window identifier returned from a call to DW_window_new().
 * RESULT
 *   A menubar identifier.
 * SEE ALSO
 *   DW_menu_new, DW_window_new
 * NOTES
 *   Currently under GTK+ you need to call this function AFTER other widgets
 *   have been created in the window that this menubar is to reside.
 * SOURCE
 *
...
win = dw_window_new( !REXXDW.!DW_DESKTOP, 'Window on desktop', style )
...
menubar = dw_menubar_new( win )
 ******
 */
rxfunc( DW_menubar_new )
{
   HMENUI rc; /* function return variable */
   HWND location; /* local variable for arg[0] */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 1, 1 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&location ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   rc = dw_menubar_new( location );
   return RxReturnUnsignedNumber( RxPackageGlobalData, retstr, (ULONG)rc );
}

/****f* Menu/DW_menu_append_item
 * NAME
 *   DW_menu_append_item
 * SYNOPSIS
 *   index = dw_menu_append_item( Menu, Title, Id, Flags, End, Check, Submenu )
 * FUNCTION
 *   Append a menu item to an existing menu or menubar. The menu item can be an
 *   individual item or another submenu.
 * ARGUMENTS
 *   - Menu       - the menu identifier returned from DW_menu_new() or DW_menubar_new()
 *   - Title      - the text of the menu item or DW_MENU_SEPARATOR
 *   - Id         - a numeric identifier used to identify the menu item or one of DW_MENU_POPUP,
 *                  DW_MENU_AUTO
 *   - Flags      - effectively ignored
 *   - End        - indicates if the menu item appears at the start or end of the
 *                  existing list. One of DW_MENU_START or DW_MENU_END
 *   - Check      - indicates if the menu item is checkable or not via DW_menu_item_set_state()
 *                  one of DW_MENU_CHECKABLE or DW_MENU_NOT_CHECKABLE
 *   - Submenu    - a menu identifier returned from a a call to DW_menu_new() or DW_MENU_NOMENU
 *                  if this is NOT a sub-menu
 * RESULT
 *   A menu identifier.
 * SEE ALSO
 *   DW_menu_new, DW_menubar_new, DW_menu_item_set_state, MenuConstants
 * NOTES
 *   Preceding a letter of the Title, will enable that letter to be typed on the
 *   keyboard to invoke the callback associated with that menu item.
 *   In the source below, with the menu item displayed, typing 'a' will execute
 *   'AddRepositoryCallback'
 *   The Title can also be set to DW_MENU_SEPARATOR which will display a line instead
 *   of text.
 *
 *   A menu item MUST have a unique Id within a particular menu, otherwise the
 *   callbacks will not work.
 * SOURCE
 *
menu = dw_menu_new( 0 )
menuitem = dw_menu_append_item( menu, '~Add Repository', 1011, 0, ,
   !REXXDW.!DW_MENU_END, !REXXDW.!DW_MENU_CHECKABLE, !REXXDW.!DW_MENU_NOMENU0 )
Call dw_signal_connect menuitem, !REXXDW.!DW_CLICKED_EVENT, ,
   'AddRepositoryCallback'
...
Call dw_menu_item_set_state menu, 1011, !REXXDW.!DW_CHECKED
 ******
 */
rxfunc( DW_menu_append_item )
{
   HWND rc; /* function return variable */
   HMENUI menu; /* local variable for arg[0] */
   char * title; /* local variable for arg[1] */
   long id; /* local variable for arg[2] */
   unsigned long flags; /* local variable for arg[3] */
   int end; /* local variable for arg[4] */
   int check; /* local variable for arg[5] */
   HMENUI submenu; /* local variable for arg[6] */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 7, 7 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&menu ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   if ( RXVALIDSTRING(argv[1]) )
      title = RXSTRPTR(argv[1]);
   else
      title = "";
   if ( RxStrToLong( RxPackageGlobalData, &argv[2], (RX_LONG *)&id ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 3 );
      return(1);
   }
   if ( RxStrToULong( RxPackageGlobalData, &argv[3], (RX_ULONG *)&flags ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 4 );
      return(1);
   }
   if ( RxStrToInt( RxPackageGlobalData, &argv[4], (RX_INT *)&end ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 5 );
      return(1);
   }
   if ( RxStrToInt( RxPackageGlobalData, &argv[5], (RX_INT *)&check ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 6 );
      return(1);
   }
   if ( RxStrToULong( RxPackageGlobalData, &argv[6], (RX_ULONG *)&submenu ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 7 );
      return(1);
   }
   rc = dw_menu_append_item( menu, title, id, flags, end, check, submenu );
   return RxReturnUnsignedNumber( RxPackageGlobalData, retstr, (ULONG)rc );
}

/****f* Menu/DW_menu_delete_item
 * NAME
 *   DW_menu_delete_item
 * SYNOPSIS
 *   dw_menu_delete_item( Menu, Id )
 * FUNCTION
 *   Deletes a menu item from an existing menu or menubar. The menu item can be an
 *   individual item or another submenu.
 * ARGUMENTS
 *   - Menu       - the menu identifier returned from DW_menu_new() or DW_menubar_new()
 *   - Id         - a numeric identifier used to identify the menu item
 * RESULT
 *   No return result.
 * SEE ALSO
 *   DW_menu_new, DW_menubar_new, DW_menu_append_item, MenuConstants
 * SOURCE
 *
menu = dw_menu_new( 0 )
menuitem = dw_menu_delete_item( menu, '~Add Repository', 1011, 0, ,
   !REXXDW.!DW_MENU_END, !REXXDW.!DW_MENU_NOT_CHECKABLE, 0 )
Call dw_signal_connect menuitem, !REXXDW.!DW_CLICKED_EVENT, ,
   'AddRepositoryCallback'
...
Call dw_menu_delete_item, menu, 1011
 ******
 */
rxfunc( DW_menu_delete_item )
{
   HMENUI menu; /* local variable for arg[0] */
   unsigned long id; /* local variable for arg[1] */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 2, 2 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&menu ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   if ( RxStrToULong( RxPackageGlobalData, &argv[1], (RX_ULONG *)&id ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 2 );
      return(1);
   }
   dw_menu_delete_item( menu, id );
   return RxReturnString( RxPackageGlobalData, retstr, "" );
}

/****f* Menu/DW_menu_destroy
 * NAME
 *   DW_menu_destroy
 * SYNOPSIS
 *   dw_menu_destroy( Menu )
 * FUNCTION
 *   Destroys a menu or menubar.
 * ARGUMENTS
 *   - Menu       - the menu identifier returned from DW_menu_new() or DW_menubar_new()
 * RESULT
 *   No return result.
 * SEE ALSO
 *   DW_menu_new, DW_menubar_new
 * SOURCE
 *
menu = dw_menu_new( 0 )
...
Call dw_menu_destroy menu
...
 ******
 */
rxfunc( DW_menu_destroy )
{
   HMENUI menu; /* local variable for arg[0] */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 1, 1 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&menu ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   dw_menu_destroy( &menu );
   return RxReturnString( RxPackageGlobalData, retstr, "" );
}

/****f* Menu/DW_menu_item_set_state
 * NAME
 *   DW_menu_item_set_state
 * SYNOPSIS
 *   dw_menu_item_set_state( Menu, Id, State )
 * FUNCTION
 *   Sets the state of the menu item identified by Menu and Id. The state of a menu item
 *   can be checked/unchecked and/or enabled/disabled.
 * ARGUMENTS
 *   - Menu       - the menu identifier returned from DW_menu_new() or DW_menubar_new()
 *   - Id         - a numeric identifier used to identify the menu
 *   - State      - the state of the menu item; the checked state ORed with the enabled state
 * RESULT
 *   No return result.
 * SEE ALSO
 *   DW_menu_new, MenuItemConstants
 * NOTES
 *   The State is cumulative.  Just setting DW_CHECKED does not affect whether the menu item
 *   is enabled or disabled.
 * SOURCE
 *
menu = dw_menu_new( 0 )
menuitem = dw_menu_append_item( menu, '~Add Repository', 1011, 0, ,
   !REXXDW.!DW_MENU_END, !REXXDW.!DW_MENU_CHECKABLE, 0 )
...
Call dw_menu_item_set_state menu, 1011, dw_or( !REXXDW.!DW_MENU_CHECKED, !REXXDW.!DW_MENU_ENABLED)
 ******
 */
rxfunc( DW_menu_item_set_state )
{
   HMENUI menu; /* local variable for arg[0] */
   unsigned long id; /* local variable for arg[1] */
   int state; /* local variable for arg[2] */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 3, 3 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&menu ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name, "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   if ( RxStrToULong( RxPackageGlobalData, &argv[1], (RX_ULONG *)&id ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name, "*ERROR* Invalid number for argument %d.", 2 );
      return(1);
   }
   if ( RxStrToInt( RxPackageGlobalData, &argv[2], (RX_INT *)&state ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name, "*ERROR* Invalid number for argument %d.", 3 );
      return(1);
   }
   dw_menu_item_set_state( menu, id, state );
   return RxReturnString( RxPackageGlobalData, retstr, "" );
}

/****f* Menu/DW_menu_popup
 * NAME
 *   DW_menu_popup
 * SYNOPSIS
 *   dw_menu_popup( Menu, Win, X, Y )
 * FUNCTION
 *   Displays the popup Menu in the specified window at the
 *   location specified by X/Y in pixels relative to ?????.
 * ARGUMENTS
 *   - Menu       - the menu identifier returned from DW_menu_new()
 *   - Win        - the window identifier of the toplevel window
 *   - X          - the X coordinate where the top left corner of the menu
 *                  will appear
 *   - Y          - the Y coordinate where the top left corner of the menu
 *                  will appear
 * RESULT
 *   No return result.
 * SEE ALSO
 *   DW_menu_new
 * NOTES
 *   The popup menu associated with the Menu identifier is destroyed each
 *   time DW_menu_popup() is called.
 *   The Win argument must be the window identifier of the topelevel window.
 *   ie. The window identifier returned from DW_window_new() called with
 *   !REXXDW.!DW_DESKTOP.
 * SOURCE
 *
win = dw_window_new( !REXXDW.!DW_DESKTOP, 'Window on desktop', style )
...
menu = dw_menu_new( 0 )
menuitem = dw_menu_append_item( menu, '~Add Repository', 1011, 0, ,
   !REXXDW.!DW_MENU_END, !REXXDW.!DW_MENU_NOT_CHECKABLE, 0 )
...
Call dw_menu_popup menu, win, 30, 130
 ******
 */
rxfunc( DW_menu_popup )
{
   HMENUI menu; /* local variable for arg[0] */
   HWND parent; /* local variable for arg[1] */
   int x; /* local variable for arg[2] */
   int y; /* local variable for arg[3] */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 4, 4 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&menu ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   if ( RxStrToULong( RxPackageGlobalData, &argv[1], (RX_ULONG *)&parent ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 2 );
      return(1);
   }
   if ( RxStrToInt( RxPackageGlobalData, &argv[2], (RX_INT *)&x ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 3 );
      return(1);
   }
   if ( RxStrToInt( RxPackageGlobalData, &argv[3], (RX_INT *)&y ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 4 );
      return(1);
   }
   dw_menu_popup( &menu, parent, x, y );
   return RxReturnString( RxPackageGlobalData, retstr, "" );
}

/****h* Widgets/MultiLineEdit
 * DESCRIPTION
 *   A multi-line edit widget is similar to an Entryfield, except that
 *   more than one line of text can be manipulated.
 ******
 */

/****f* MultiLineEdit/DW_mle_new
 * NAME
 *   DW_mle_new
 * SYNOPSIS
 *   win = dw_mle_new( Id )
 * FUNCTION
 *   Creates a new Multi-Line edit widget.
 *   An MLE is used to display and edit multiple lines of text.
 * ARGUMENTS
 *   - Id         - a numeric identifier used to identify the window
 * RESULT
 *   An MLE identifier.
 * SOURCE
 *
mle = dw_mle_new( 0 )
 ******
 */
rxfunc( DW_mle_new )
{
   HWND rc; /* function return variable */
   unsigned long id; /* local variable for arg[0] */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 1, 1 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&id ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   rc = dw_mle_new( id );
   return RxReturnUnsignedNumber( RxPackageGlobalData, retstr, (ULONG)rc );
}

/****f* MultiLineEdit/DW_mle_delete
 * NAME
 *   DW_mle_delete
 * SYNOPSIS
 *   dw_mle_delete( Win[, StartPoint, Length] )
 * FUNCTION
 *   Deletes the text from a Multi-Line Entry widget starting at StartPoint
 *   for Length characters
 * ARGUMENTS
 *   - Win        - the window identifier returned from DW_mle_new()
 *   - StartPoint - 0 based offset into the MLE counted in characters
 *   - Length     - the number of characters to delete
 * RESULT
 *   No return result.
 * NOTES
 *   If no Startpoint is specified, it defaults to 0 (the start of the MLE)
 *   If no Length is specified, it defaults to the total number of characters in
 *   the MLE.
 * SOURCE
 *
...
mle = dw_mle_new( 0 )
...
Call dw_mle_delete( mle, 0, 100 )
 ******
 */
rxfunc( DW_mle_delete )
{
   HWND handle; /* local variable for arg[0] */
   int startpoint; /* local variable for arg[1] */
   int length; /* local variable for arg[2] */
   unsigned long bytes, lines;
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 1, 3 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&handle ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   /*
    * Use dw_mle_clear() if only 1 argument
    */
   if ( argc == 1 )
   {
      dw_mle_clear( handle );
   }
   else
   {
      /*
       * Determine the size of the text in case we want to use default values
       */
      dw_mle_get_size( handle, &bytes, &lines );
      if ( RxStrToInt( RxPackageGlobalData, &argv[1], (RX_INT *)&startpoint ) == -1 )
      {
         RxDisplayError( RxPackageGlobalData, name,
                         "*ERROR* Invalid number for argument %d.", 2 );
         return(1);
      }
      if ( argc > 2 )
      {
         if ( RxStrToInt( RxPackageGlobalData, &argv[2], (RX_INT *)&length ) == -1 )
         {
            RxDisplayError( RxPackageGlobalData, name,
                            "*ERROR* Invalid number for argument %d.", 3 );
            return(1);
         }
      }
      else
         length = bytes;
      dw_mle_delete( handle, startpoint, length );
   }
   return RxReturnString( RxPackageGlobalData, retstr, "" );
}

/****f* MultiLineEdit/DW_mle_import
 * NAME
 *   DW_mle_import
 * SYNOPSIS
 *   Location = dw_mle_import( Win, Text, StartPoint )
 * FUNCTION
 *   Inserts the Text into a Multi-Line Entry widget starting at StartPoint.
 * ARGUMENTS
 *   - Win        - the window identifier returned from DW_mle_new()
 *   - Text       - the text to insert into the MLE
 *   - StartPoint - 0 based offset into the MLE counted in characters
 * RESULT
 *   The point into the MLE after the last character in Text was inserted counted
 *   in characters.
 * SEE ALSO
 *   DW_mle_export
 * NOTES
 *   You need to append a CRLF ('0d0a'x) to the end of text to insert lines.
 *   The first call to dw_mle_import() must specify -1 as StartPoint.
 * SOURCE
 *
...
mle = dw_mle_new( 0 )
...
eol = '0d0a'x
loc = -1
Do i = 1 To 10
   loc = dw_mle_import( mle, 'Inserting line' i 'here'||eol, loc )
End
...
 ******
 */
rxfunc( DW_mle_import )
{
   unsigned int rc; /* function return variable */
   HWND handle; /* local variable for arg[0] */
   char * buffer; /* local variable for arg[1] */
   int startpoint; /* local variable for arg[2] */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 3, 3 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&handle ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name, "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   if ( RXVALIDSTRING(argv[1]) )
      buffer = RXSTRPTR(argv[1]);
   else
      buffer = "";
   if ( RxStrToInt( RxPackageGlobalData, &argv[2], (RX_INT *)&startpoint ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name, "*ERROR* Invalid number for argument %d.", 3 );
      return(1);
   }
   rc = dw_mle_import( handle, buffer, startpoint );
   return RxReturnNumber( RxPackageGlobalData, retstr, (ULONG)rc );
}

/****f* MultiLineEdit/DW_mle_export
 * NAME
 *   DW_mle_export
 * SYNOPSIS
 *   text = dw_mle_export( Win[, StartPoint, Length] )
 * FUNCTION
 *   Returns the text from a Multi-Line Entry widget starting at StartPoint
 *   for Length characters.
 * ARGUMENTS
 *   - Win        - the window identifier returned from DW_mle_new()
 *   - StartPoint - 0 based offset into the MLE counted in characters
 *   - Length     - the number of characters to export
 * RESULT
 *   The text exported.
 * SEE ALSO
 *   DW_mle_import
 * NOTES
 *   If no Startpoint is specified, it defaults to 0 (the start of the MLE)
 *   If no Length is specified, it defaults to the total number of characters in
 *   the MLE.
 * SOURCE
 *
...
mle = dw_mle_new( 0 )
...
contents = dw_mle_export( mle, 0, 100 )
 ******
 */
rxfunc( DW_mle_export )
{
   HWND handle; /* local variable for arg[0] */
   int startpoint; /* local variable for arg[1] */
   int length; /* local variable for arg[2] */
   char * buffer;
   unsigned long bytes, lines;
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 1, 3 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&handle ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   /*
    * We have to determine how much memory is required to capture
    * the complete contents of the MLE. Its easier to allocate space
    * for the complete MLE rather than work out how much is being
    * requested.
    */
   dw_mle_get_size( handle, &bytes, &lines );
   buffer = (char *)RexxAllocateMemory( bytes+1 );
   if ( buffer == NULL )
   {
      RxDisplayError( RxPackageGlobalData, name, "*ERROR* Out of memory." );
      return(1);
   }
   if ( argc > 1 )
   {
      if ( RxStrToInt( RxPackageGlobalData, &argv[1], (RX_INT *)&startpoint ) == -1 )
      {
         RxDisplayError( RxPackageGlobalData, name, "*ERROR* Invalid number for argument %d.", 2 );
         RexxFreeMemory( buffer );
         return(1);
      }
   }
   else
      startpoint = 0;
   if ( argc > 2 )
   {
      if ( RxStrToInt( RxPackageGlobalData, &argv[2], (RX_INT *)&length ) == -1 )
      {
         RxDisplayError( RxPackageGlobalData, name, "*ERROR* Invalid number for argument %d.", 3 );
         RexxFreeMemory( buffer );
         return(1);
      }
   }
   else
      length = bytes;
   dw_mle_export( handle, buffer, startpoint, length );
   return RxReturnStringAndFree( RxPackageGlobalData, retstr, buffer, 1 );
}

/****f* MultiLineEdit/DW_mle_get_size
 * NAME
 *   DW_mle_get_size
 * SYNOPSIS
 *   Bytes Lines = dw_mle_get_size( Win )
 * FUNCTION
 *   Returns the number of Bytes and Lines currently in the Multi-Line Entry widget.
 * ARGUMENTS
 *   - Win        - the window identifier returned from DW_mle_new()
 * RESULT
 *   Bytes and Lines are returned as two words. PARSE VALUE is the easiest
 *   way to obtain both values.
 * SEE ALSO
 *   DW_mle_new
 * SOURCE
 *
...
mle = dw_mle_new( 0 )
...
Parse Value dw_mle_get_size( mle ) With bytes lines
 ******
 */
rxfunc( DW_mle_get_size )
{
   HWND handle; /* local variable for arg[0] */
   char buf1[100]; /* enough for two numbers */
   unsigned long bytes;
   unsigned long lines;
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 1, 1 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&handle ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name, "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   dw_mle_get_size( handle, &bytes, &lines );
   sprintf( buf1, "%ld %ld", bytes, lines );
   return RxReturnString( RxPackageGlobalData, retstr, buf1 );
}

/****f* MultiLineEdit/DW_mle_search
 * NAME
 *   DW_mle_search
 * SYNOPSIS
 *   Location = dw_mle_search( Win, Text, StartPoint, SearchFlags )
 * FUNCTION
 *   Searches the MLE for Text starting at StartPoint.
 * ARGUMENTS
 *   - Win        - the window identifier returned from DW_mle_new()
 *   - Text       - the text to search for in the MLE
 *   - StartPoint - 0 based offset into the MLE counted in characters
 *   - SearchFlags- optional flag to search respecting case if set to
 *                  DW_MLE_CASESENSITIVE. default is to search irrespective
 *                  of case
 * RESULT
 *   The point into the MLE where the matching Text starts counted
 *   in characters.
 * SEE ALSO
 *   MLESearchFlags
 * SOURCE
 *
...
mle = dw_mle_new( 0 )
...
loc = dw_mle_import( mle, 'More text here', 100 )
...
pos = dw_mle_search( mle, 'text', 0, 0 )
 ******
 */
rxfunc( DW_mle_search )
{
   int rc; /* function return variable */
   HWND handle; /* local variable for arg[0] */
   char * text; /* local variable for arg[1] */
   int point; /* local variable for arg[2] */
   unsigned long flags; /* local variable for arg[3] */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 4, 4 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&handle ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   if ( RXVALIDSTRING(argv[1]) )
      text = RXSTRPTR(argv[1]);
   else
      text = "";
   if ( RxStrToInt( RxPackageGlobalData, &argv[2], (RX_INT *)&point ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 3 );
      return(1);
   }
   if ( RxStrToULong( RxPackageGlobalData, &argv[3], (RX_ULONG *)&flags ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 4 );
      return(1);
   }
   rc = dw_mle_search( handle, text, point, flags );
   return RxReturnNumber( RxPackageGlobalData, retstr, rc );
}

/****f* MultiLineEdit/DW_mle_set_cursor
 * NAME
 *   DW_mle_set_curesor
 * SYNOPSIS
 *   dw_mle_set_cursor( Win, Point )
 * FUNCTION
 *   Sets the cursor at Point in the Multi-Line Entry widget.
 * ARGUMENTS
 *   - Win        - the window identifier returned from DW_mle_new()
 *   - Point      - the character at which to display the cursor in the MLE
 * RESULT
 *   No return result.
 * SOURCE
 *
...
mle = dw_mle_new( 0 )
...
Call dw_mle_set_cursor( mle, 10 )
 ******
 */
rxfunc( DW_mle_set_cursor )
{
   HWND handle; /* local variable for arg[0] */
   int point; /* local variable for arg[1] */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 2, 2 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&handle ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   if ( RxStrToInt( RxPackageGlobalData, &argv[1], (RX_INT *)&point ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 2 );
      return(1);
   }
   dw_mle_set_cursor( handle, point );
   return RxReturnString( RxPackageGlobalData, retstr, "" );
}

/****f* MultiLineEdit/DW_mle_set_editable
 * NAME
 *   DW_mle_set_editable
 * SYNOPSIS
 *   dw_mle_set_editable( Win, EditableFlags )
 * FUNCTION
 *   Sets the state of the Multi-Line Entry widget to be editable or readonly.
 * ARGUMENTS
 *   - Win           - the window identifier returned from DW_mle_new()
 *   - EditableFlags - one of DW_EDITABLE or DW_READONLY
 * RESULT
 *   No return result.
 * SEE ALSO
 *   MLEEditableFlags
 * NOTES
 *   By default MLEs are editable
 * SOURCE
 *
...
mle = dw_mle_new( 0 )
...
Call dw_mle_set_editable( mle, !REXXDW.!DW_READONLY )
 ******
 */
rxfunc( DW_mle_set_editable )
{
   HWND handle; /* local variable for arg[0] */
   int state; /* local variable for arg[1] */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 2, 2 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&handle ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   if ( RxStrToInt( RxPackageGlobalData, &argv[1], (RX_INT *)&state ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 2 );
      return(1);
   }
   dw_mle_set_editable( handle, state );
   return RxReturnString( RxPackageGlobalData, retstr, "" );
}

/****f* MultiLineEdit/DW_mle_set_visible
 * NAME
 *   DW_mle_set_visible
 * SYNOPSIS
 *   dw_mle_set_visible( Win, Line )
 * FUNCTION
 *   Sets the Line in the Multi-Line Entry widget to be visible.
 * ARGUMENTS
 *   - Win        - the window identifier returned from DW_mle_new()
 *   - Line       - the Line in the MLE to be visible
 * RESULT
 *   No return result.
 * SOURCE
 *
...
mle = dw_mle_new( 0 )
...
Call dw_mle_set_visible( mle, 10 )
 ******
 */
rxfunc( DW_mle_set_visible )
{
   HWND handle; /* local variable for arg[0] */
   int line; /* local variable for arg[1] */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 2, 2 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&handle ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   if ( RxStrToInt( RxPackageGlobalData, &argv[1], (RX_INT *)&line ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 2 );
      return(1);
   }
   dw_mle_set_visible( handle, line );
   return RxReturnString( RxPackageGlobalData, retstr, "" );
}

/****f* MultiLineEdit/DW_mle_set_word_wrap
 * NAME
 *   DW_mle_set_word_wrap
 * SYNOPSIS
 *   dw_mle_set_editable( Win, WordWrapFlags )
 * FUNCTION
 *   Sets the state of the Multi-Line Entry widget to allow lines to wrap on
 *   word boundaries or be truncated at the edge of the viewport.
 * ARGUMENTS
 *   - Win           - the window identifier returned from DW_mle_new()
 *   - WordWrapFlags - one of DW_WORD_WRAP or DW_DONT_WORD_WRAP
 * RESULT
 *   No return result.
 * SEE ALSO
 *   MLEWordWrapFlags
 * SOURCE
 *
...
mle = dw_mle_new( 0 )
...
Call dw_mle_set_word_wrap( mle, !REXXDW.!DW_WORD_WRAP )
 ******
 */
rxfunc( DW_mle_set_word_wrap )
{
   HWND handle; /* local variable for arg[0] */
   int state; /* local variable for arg[1] */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 2, 2 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&handle ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   if ( RxStrToInt( RxPackageGlobalData, &argv[1], (RX_INT *)&state ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 2 );
      return(1);
   }
   dw_mle_set_word_wrap( handle, state );
   return RxReturnString( RxPackageGlobalData, retstr, "" );
}

/****f* MultiLineEdit/DW_mle_freeze
 * NAME
 *   DW_mle_freeze
 * SYNOPSIS
 *   dw_mle_freeze( Win )
 * FUNCTION
 *   Stops the screen output to a Multi-Line Entry widget.
 * ARGUMENTS
 *   - Win        - the window identifier returned from DW_mle_new()
 * RESULT
 *   No return result.
 * SEE ALSO
 *   DW_mle_thaw
 * NOTES
 *   On some platforms, there can be excessive scrolling and redraws when text is
 *   imported into an MLE. This function reduces these effects.
 * SOURCE
 *
...
mle = dw_mle_new( 0 )
...
Call dw_mle_freeze( mle )
loc = -1
Do i = 1 To text.0
   loc = dw_mle_import( mle, text.i, loc )
End
Call dw_mle_thaw( mle )
 ******
 */
rxfunc( DW_mle_freeze )
{
   HWND handle; /* local variable for arg[0] */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 1, 1 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&handle ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   dw_mle_freeze( handle );
   return RxReturnString( RxPackageGlobalData, retstr, "" );
}

/****f* MultiLineEdit/DW_mle_thaw
 * NAME
 *   DW_mle_thaw
 * SYNOPSIS
 *   dw_mle_thaw( Win )
 * FUNCTION
 *   Restarts the screen output to a Multi-Line Entry widget.
 * ARGUMENTS
 *   - Win        - the window identifier returned from DW_mle_new()
 * RESULT
 *   No return result.
 * SEE ALSO
 *   DW_mle_freeze
 * NOTES
 *   On some platforms, there can be excessive scrolling and redraws when text is
 *   imported into an MLE. This function reduces these effects.
 * SOURCE
 *
...
mle = dw_mle_new( 0 )
...
Call dw_mle_freeze( mle )
loc = -1
Do i = 1 To text.0
   loc = dw_mle_import( mle, text.i, loc )
End
Call dw_mle_thaw( mle )
 ******
 */
rxfunc( DW_mle_thaw )
{
   HWND handle; /* local variable for arg[0] */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 1, 1 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&handle ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   dw_mle_thaw( handle );
   return RxReturnString( RxPackageGlobalData, retstr, "" );
}

/****h* Widgets/Notebook
 * DESCRIPTION
 *   Strictly speaking these widgets are "tabbed notebooks". This widget
 *   provides another sort of frame in which other widgets are packed.
 *   Notebook pages are frames that overlap the same screen space, but can
 *   be brought to the front by the user clicking on the tab.
 ******
 */

/****f* Notebook/DW_notebook_new
 * NAME
 *   DW_notebook_new
 * SYNOPSIS
 *   win = dw_notebook_new( Id, TabLocation )
 * FUNCTION
 *   Creates a new tabbed notebook widget.
 *   A notebook is a frame in which pages containing other widgets can
 *   be packed. Each page can be selected by clicking that page's tab.
 * ARGUMENTS
 *   - Id         - a numeric identifier used to identify the window
 *   - TabLocation- indicates if the tabs appear across the top or bottom
 * RESULT
 *   A notebook identifier.
 * SEE ALSO
 *   NotebookTabLocation
 * SOURCE
 *
notebook = dw_notebook_new( 0, !REXXDW.!DW_TAB_TO_TOP )
 ******
 */
rxfunc( DW_notebook_new )
{
   HWND rc; /* function return variable */
   unsigned long id; /* local variable for arg[0] */
   int top; /* local variable for arg[1] */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 2, 2 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&id ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   if ( RxStrToInt( RxPackageGlobalData, &argv[1], (RX_INT *)&top ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 2 );
      return(1);
   }
   rc = dw_notebook_new( id, top );
   return RxReturnUnsignedNumber( RxPackageGlobalData, retstr, (ULONG)rc );
}

/****f* Notebook/DW_notebook_page_new
 * NAME
 *   DW_notebook_page_new
 * SYNOPSIS
 *   page = dw_notebook_page_new( Win, Flags, PageLocation )
 * FUNCTION
 *   Creates a new page within a tabbed notebook widget.
 * ARGUMENTS
 *   - Win          - the window identifier returned from DW_notebook_new()
 *   - Flags        - ignored
 *   - PageLocation - indicates if the new page is shown in front or behind current tabs
 * RESULT
 *   A notebook page identifier.
 * SEE ALSO
 *   DW_notebook_new, NotebookPageLocation
 * NOTES
 *   A maximum of 256 pages can be created in a notebook.
 *   On Windows, the PageLocation does nothing, so it is best to
 *   specify which page is in front once all pages have been created by calling
 *   DW_notebook_page_set().
 * SOURCE
 *
notebook = dw_notebook_new( 0, !REXXDW.!DW_TAB_TO_TOP )
notebookbox = dw_box_new( !REXXDW.!DW_VERT, 0 )
notebookpage = dw_notebook_page_new( notebook, 0, !REXXDW.!DW_PAGE_TO_BACK )
Call dw_notebook_pack notebook, notebookpage, notebookpagebox
 ******
 */
rxfunc( DW_notebook_page_new )
{
   unsigned long rc; /* function return variable */
   HWND handle; /* local variable for arg[0] */
   unsigned long flags; /* local variable for arg[1] */
   int front; /* local variable for arg[2] */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 3, 3 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&handle ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   if ( RxStrToULong( RxPackageGlobalData, &argv[1], (RX_ULONG *)&flags ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 2 );
      return(1);
   }
   if ( RxStrToInt( RxPackageGlobalData, &argv[2], (RX_INT *)&front ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 3 );
      return(1);
   }
   rc = dw_notebook_page_new( handle, flags, front );
   return RxReturnUnsignedNumber( RxPackageGlobalData, retstr, rc );
}

/****f* Notebook/DW_notebook_pack
 * NAME
 *   DW_notebook_pack
 * SYNOPSIS
 *   dw_notebook_pack( Win, Page, Box )
 * FUNCTION
 *   Packs a generic box into a page in the notebook.
 * ARGUMENTS
 *   - Win        - the window identifier returned from DW_notebook_new()
 *   - Page       - the page identifier returned from DW_notebook_page_new()
 *   - Box        - the box identifier returned from a DW_box_new() or DW_groupbox_new()
 * RESULT
 *   No return result.
 * SEE ALSO
 *   DW_notebook_new, DW_notebook_page_new
 * SOURCE
 *
notebook = dw_notebook_new( 0, !REXXDW.!DW_TAB_TO_TOP )
notebookbox = dw_box_new( !REXXDW.!DW_VERT, 0 )
notebookpage = dw_notebook_page_new( notebook, 0, !REXXDW.!DW_PAGE_TO_BACK )
Call dw_notebook_pack notebook, notebookpage, notebookpagebox
 ******
 */
rxfunc( DW_notebook_pack )
{
   HWND handle; /* local variable for arg[0] */
   unsigned long pageid; /* local variable for arg[1] */
   HWND page; /* local variable for arg[2] */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 3, 3 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&handle ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   if ( RxStrToULong( RxPackageGlobalData, &argv[1], (RX_ULONG *)&pageid ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 2 );
      return(1);
   }
   if ( RxStrToULong( RxPackageGlobalData, &argv[2], (RX_ULONG *)&page ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 3 );
      return(1);
   }
   dw_notebook_pack( handle, pageid, page );
   return RxReturnString( RxPackageGlobalData, retstr, "" );
}

/****f* Notebook/DW_notebook_page_destroy
 * NAME
 *   DW_notebook_page_destroy
 * SYNOPSIS
 *   dw_notebook_page_destroy( Win, Page )
 * FUNCTION
 *   Destroys the specified Page of a notebook. All widgets inside the Page
 *   are also destroyed.
 * ARGUMENTS
 *   - Win        - the window identifier returned from DW_notebook_new()
 *   - Page       - the page identifier returned from DW_notebook_page_new()
 * RESULT
 *   No return result.
 * SEE ALSO
 *   DW_notebook_new, DW_notebook_page_new
 * SOURCE
 *
notebook = dw_notebook_new( 0, !REXXDW.!DW_TAB_TO_TOP )
notebookbox = dw_box_new( !REXXDW.!DW_VERT, 0 )
notebookpage = dw_notebook_page_new( notebook, 0, !REXXDW.!DW_PAGE_TO_BACK )
Call dw_notebook_pack notebook, notebookpage, notebookpagebox
...
Call dw_notebook_page_destoy notebook, notebookpage
 ******
 */
rxfunc( DW_notebook_page_destroy )
{
   HWND handle; /* local variable for arg[0] */
   unsigned int pageid; /* local variable for arg[1] */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 2, 2 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&handle ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   if ( RxStrToUInt( RxPackageGlobalData, &argv[1], (RX_UINT *)&pageid ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 2 );
      return(1);
   }
   dw_notebook_page_destroy( handle, pageid );
   return RxReturnString( RxPackageGlobalData, retstr, "" );
}

/****f* Notebook/DW_notebook_page_set
 * NAME
 *   DW_notebook_page_set
 * SYNOPSIS
 *   dw_notebook_page_set( Win, Page )
 * FUNCTION
 *   Sets Page to be the notebook page displayed at the front.
 * ARGUMENTS
 *   - Win        - the window identifier returned from DW_notebook_new()
 *   - Page       - the page identifier returned from DW_notebook_page_new()
 * RESULT
 *   No return result.
 * SEE ALSO
 *   DW_notebook_new, DW_notebook_page_get
 * SOURCE
 *
notebook = dw_notebook_new( 0, !REXXDW.!DW_TAB_TO_TOP )
notebookbox = dw_box_new( !REXXDW.!DW_VERT, 0 )
notebookpage = dw_notebook_page_new( notebook, 0, !REXXDW.!DW_PAGE_TO_BACK )
Call dw_notebook_pack notebook, notebookpage, notebookpagebox
...
Call dw_notebook_page_set notebook, notebookpage
 ******
 */
rxfunc( DW_notebook_page_set )
{
   HWND handle; /* local variable for arg[0] */
   unsigned int pageid; /* local variable for arg[1] */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 2, 2 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&handle ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   if ( RxStrToUInt( RxPackageGlobalData, &argv[1], (RX_UINT *)&pageid ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 2 );
      return(1);
   }
   dw_notebook_page_set( handle, pageid );
   return RxReturnString( RxPackageGlobalData, retstr, "" );
}

/****f* Notebook/DW_notebook_page_set_text
 * NAME
 *   DW_notebook_page_set_text
 * SYNOPSIS
 *   dw_notebook_page_set_text( Win, Page, Text )
 * FUNCTION
 *   Sets the text on the notebook tab on the Page to Text.
 * ARGUMENTS
 *   - Win        - the window identifier returned from DW_notebook_new()
 *   - Page       - the page identifier returned from DW_notebook_page_new()
 *   - Text       - the text to display on the tab
 * RESULT
 *   No return result.
 * SOURCE
 *
notebook = dw_notebook_new( 0, !REXXDW.!DW_TAB_TO_TOP )
notebookbox = dw_box_new( !REXXDW.!DW_VERT, 0 )
notebookpage = dw_notebook_page_new( notebook, 0, !REXXDW.!DW_PAGE_TO_BACK )
Call dw_notebook_pack notebook, notebookpage, notebookpagebox
...
Call dw_notebook_page_set_text notebook, notebookpage, 'Tab Text'
 ******
 */
rxfunc( DW_notebook_page_set_text )
{
   HWND handle; /* local variable for arg[0] */
   unsigned long pageid; /* local variable for arg[1] */
   char * text; /* local variable for arg[2] */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 3, 3 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&handle ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   if ( RxStrToULong( RxPackageGlobalData, &argv[1], (RX_ULONG *)&pageid ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 2 );
      return(1);
   }
   if ( RXVALIDSTRING(argv[2]) )
      text = RXSTRPTR(argv[2]);
   else
      text = "";
   dw_notebook_page_set_text( handle, pageid, text );
   return RxReturnString( RxPackageGlobalData, retstr, "" );
}

/****f* Notebook/DW_notebook_page_set_status_text
 * NAME
 *   DW_notebook_page_set_status_text
 * SYNOPSIS
 *   dw_notebook_page_set_status_text( Win, Page, Text )
 * FUNCTION
 *   Sets the status text area on the Page to Text.
 * ARGUMENTS
 *   - Win        - the window identifier returned from DW_notebook_new()
 *   - Page       - the page identifier returned from DW_notebook_page_new()
 *   - Text       - the text to display in the status text area
 * RESULT
 *   No return result.
 * NOTES
 *   Only OS/2 has a status text area. Other platforms ignore this function.
 * SOURCE
 *
notebook = dw_notebook_new( 0, !REXXDW.!DW_TAB_TO_TOP )
notebookbox = dw_box_new( !REXXDW.!DW_VERT, 0 )
notebookpage = dw_notebook_page_new( notebook, 0, !REXXDW.!DW_PAGE_TO_BACK )
Call dw_notebook_pack notebook, notebookpage, notebookpagebox
...
Call dw_notebook_page_set_status_text notebook, notebookpage, 'Subtitle Text'
 ******
 */
rxfunc( DW_notebook_page_set_status_text )
{
   HWND handle; /* local variable for arg[0] */
   unsigned long pageid; /* local variable for arg[1] */
   char * text; /* local variable for arg[2] */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 3, 3 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&handle ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   if ( RxStrToULong( RxPackageGlobalData, &argv[1], (RX_ULONG *)&pageid ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 2 );
      return(1);
   }
   if ( RXVALIDSTRING(argv[2]) )
      text = RXSTRPTR(argv[2]);
   else
      text = "";
   dw_notebook_page_set_status_text( handle, pageid, text );
   return RxReturnString( RxPackageGlobalData, retstr, "" );
}

/****f* Notebook/DW_notebook_page_get
 * NAME
 *   DW_notebook_page_get
 * SYNOPSIS
 *   page = dw_notebook_page_get( Win, Flags, PageLocation )
 * FUNCTION
 *   Return the page identifier of the notebook page currently at the front.
 * ARGUMENTS
 *   - Win          - the window identifier returned from DW_notebook_new()
 * RESULT
 *   A notebook page identifier.
 * SEE ALSO
 *   DW_notebook_new, DW_notebook_page_set
 * SOURCE
 *
notebook = dw_notebook_new( 0, !REXXDW.!DW_TAB_TO_TOP )
notebookbox = dw_box_new( !REXXDW.!DW_VERT, 0 )
notebookpage = dw_notebook_page_new( notebook, 0, !REXXDW.!DW_PAGE_TO_BACK )
Call dw_notebook_pack notebook, notebookpage, notebookpagebox
...
Say 'The pageid of the notebook is:' dw_notebook_page_get( notebook )
 ******
 */
rxfunc( DW_notebook_page_get )
{
   unsigned long rc; /* function return variable */
   HWND handle; /* local variable for arg[0] */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 1, 1 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&handle ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   rc = dw_notebook_page_get( handle );
   return RxReturnNumber( RxPackageGlobalData, retstr, (ULONG)rc );
}

/****h* Widgets/Percent
 * DESCRIPTION
 *   A Percent widget is simply a progress bar.
 * NOTES
 *   Under OS/2 only 1 instance of a percent or slider widget can exist
 *   within the same box. So for complete portability, all percent and
 *   slider widgets should be enclosed in their own box.
 ******
 */

/****f* Percent/DW_percent_new
 * NAME
 *   DW_percent_new
 * SYNOPSIS
 *   win = dw_percent_new( Id )
 * FUNCTION
 *   Creates a new percent progress status bar widget.
 * ARGUMENTS
 *   - Id         - a numeric identifier used to identify the window
 * RESULT
 *   A percent progress status bar identifier.
 * SOURCE
 *
percent = dw_percent_new( 0 )
 ******
 */
rxfunc( DW_percent_new )
{
   HWND rc; /* function return variable */
   unsigned long id; /* local variable for arg[0] */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 1, 1 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&id ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   rc = dw_percent_new( id );
   return RxReturnUnsignedNumber( RxPackageGlobalData, retstr, (ULONG)rc );
}

/****f* Percent/DW_percent_set_pos
 * NAME
 *   DW_percent_set_pos
 * SYNOPSIS
 *   dw_percent_set_pos( Win, Position )
 * FUNCTION
 *   Sets the current position of the percent bar to Position.
 * ARGUMENTS
 *   - Win        - the window identifier returned from DW_percent_new()
 *   - Position   - the position of the bar as a whole number percentage
 *                  or !REXXDW.!DW_PERCENT_INDETERMINATE to display an infinitely
 *                  moving bar not relative to a fixed value
 * RESULT
 *   No return result.
 * SOURCE
 *
percent = dw_percent_new( 0 )
...
now = (bytes_sent*100) % total_bytes
Call dw_percent_set_pos percent, now
 ******
 */
rxfunc( DW_percent_set_pos )
{
   HWND handle; /* local variable for arg[0] */
   int position; /* local variable for arg[1] */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 2, 2 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&handle ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name, "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   if ( RxStrToInt( RxPackageGlobalData, &argv[1], (RX_INT *)&position ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name, "*ERROR* Invalid number for argument %d.", 2 );
      return(1);
   }
   dw_percent_set_pos( handle, position );
   return RxReturnString( RxPackageGlobalData, retstr, "" );
}

/****h* Widgets/Scrollbar
 * DESCRIPTION
 *   A Scrollbar widget is used in conjunction with other widgets that are not
 *   by default scrollable, to allow details that are not completely visible in
 *   a window to be viewable by scrolling.
 *   Both horizontal and vertical scrollbars are supported.
 ******
 */

/****f* Scrollbar/DW_scrollbar_new
 * NAME
 *   DW_scrollbar_new
 * SYNOPSIS
 *   win = dw_scrollbar_new( Orientation, Id )
 * FUNCTION
 *   Creates a new scrollbar.
 *   A scrollbar is used to scroll various other widgets like Multi-Line edit
 *   windows or listboxes.
 * ARGUMENTS
 *   - Orientation- !REXXDW.!DW_VERT to indicate a vertical scrollbar
 *                  !REXXDW.!DW_HORZ to indicate a horizontal scrollbar
 *   - Id         - a numeric identifier used to identify the window
 * RESULT
 *   A scrollbar identifier.
 * SOURCE
 *
vscrollbar = dw_scrollbar_new( !REXXDW.!DW_VERT, 0 )
 ******
 */
rxfunc( DW_scrollbar_new )
{
   HWND rc; /* function return variable */
   int vertical; /* local variable for arg[0] */
   ULONG id; /* local variable for arg[1] */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 2, 2 ) )
      return( 1 );
   if ( RxStrToInt( RxPackageGlobalData, &argv[0], (RX_INT *)&vertical ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   if ( RxStrToULong( RxPackageGlobalData, &argv[1], (RX_ULONG *)&id ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 2 );
      return(1);
   }
   rc = dw_scrollbar_new( vertical, id );
   return RxReturnUnsignedNumber( RxPackageGlobalData, retstr, (ULONG)rc );
}

/****f* Scrollbar/DW_scrollbar_set_range
 * NAME
 *   DW_scrollbar_set_range
 * SYNOPSIS
 *   dw_scrollbar_set_range( Win, Range, Thumbwidth )
 * FUNCTION
 *   Sets the maximum value and width of the scrollbar thumb of the scrollbar.
 * ARGUMENTS
 *   - Win        - the window identifier returned from DW_scrollbar_new()
 *   - Range      - the maximum range of the scrollbar
 *   - Thumbwidth - the width of the scrollbar's thumb
 * RESULT
 *   No return result.
 * SOURCE
 *
vscrollbar = dw_scrollbar_new( !REXXDW.!DW_VERT, 0 )
Call dw_scrollbar_set_range vscrollbar, 200, 40
...
Call dw_scrollbar_set_pos vscrollbar, 150
...
Say 'The scrollbar is at' dw_scrollbar_get_pos( vscrollbar )
 ******
 */
rxfunc( DW_scrollbar_set_range )
{
   HWND handle; /* local variable for arg[0] */
   unsigned int range; /* local variable for arg[1] */
   unsigned int visible; /* local variable for arg[2] */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 3, 3 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&handle ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   if ( RxStrToUInt( RxPackageGlobalData, &argv[1], (RX_UINT *)&range ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 2 );
      return(1);
   }
   if ( RxStrToUInt( RxPackageGlobalData, &argv[2], (RX_UINT *)&visible ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 3 );
      return(1);
   }
   dw_scrollbar_set_range( handle, range, visible );
   return RxReturnString( RxPackageGlobalData, retstr, "" );
}

/****f* Scrollbar/DW_scrollbar_set_pos
 * NAME
 *   DW_scrollbar_set_pos
 * SYNOPSIS
 *   dw_scrollbar_set_pos( Win, Position )
 * FUNCTION
 *   Sets the position of the scrollbar.
 * ARGUMENTS
 *   - Win        - the window identifier returned from DW_scrollbar_new()
 *   - Position   - the position within the scrollbar's range
 * RESULT
 *   No return result.
 * SEE ALSO
 *   DW_scrollbar_get_pos
 * SOURCE
 *
vscrollbar = dw_scrollbar_new( !REXXDW.!DW_VERT, 0 )
Call dw_scrollbar_set_range vscrollbar, 200, 40
...
Call dw_scrollbar_set_pos vscrollbar, 150
...
Say 'The scrollbar is at' dw_scrollbar_get_pos( vscrollbar )
 ******
 */
rxfunc( DW_scrollbar_set_pos )
{
   HWND handle; /* local variable for arg[0] */
   unsigned int position; /* local variable for arg[1] */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 2, 2 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&handle ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   if ( RxStrToUInt( RxPackageGlobalData, &argv[1], (RX_UINT *)&position ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 2 );
      return(1);
   }
   dw_scrollbar_set_pos( handle, position );
   return RxReturnString( RxPackageGlobalData, retstr, "" );
}

/****f* Scrollbar/DW_scrollbar_get_pos
 * NAME
 *   DW_scrollbar_get_pos
 * SYNOPSIS
 *   pos = dw_scrollbar_get_pos( Win )
 * FUNCTION
 *   Returns the current position of the scrollbar.
 * ARGUMENTS
 *   - Win        - the window identifier returned from DW_scrollbar_new()
 * RESULT
 *   The position of the scrollbar.
 * SEE ALSO
 *   DW_scrollbar_set_pos
 * SOURCE
 *
vscrollbar = dw_scrollbar_new( !REXXDW.!DW_VERT, 0 )
Call dw_scrollbar_set_range vscrollbar, 200, 40
Say 'The scrollbar is at' dw_scrollbar_get_pos( vscrollbar )
 ******
 */
rxfunc( DW_scrollbar_get_pos )
{
   unsigned int rc; /* function return variable */
   HWND handle; /* local variable for arg[0] */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 1, 1 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&handle ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   rc = dw_scrollbar_get_pos( handle );
   return RxReturnNumber( RxPackageGlobalData, retstr, (ULONG)rc );
}

/****h* Widgets/Slider
 * DESCRIPTION
 *   A Slider widget is similar to a Scrollbar, but it allows the user to select
 *   a numeric value by dragging a thumb until the desired numeric value is reached.
 * NOTES
 *   Under OS/2 only 1 instance of a percent or slider widget can exist
 *   within the same box. So for complete portability, all percent and
 *   slider widgets should be enclosed in their own box.
 ******
 */

/****f* Slider/DW_slider_new
 * NAME
 *   DW_slider_new
 * SYNOPSIS
 *   win = dw_slider_new( Orientation, Range, Id )
 * FUNCTION
 *   Creates a new slider.
 *   A slider provides a widget for selecting a numeric values from a range.
 * ARGUMENTS
 *   - Orientation- !REXXDW.!DW_VERT to indicate a vertical scrollbar
 *                  !REXXDW.!DW_HORZ to indicate a horizontal scrollbar
 *   - Range      - the maximum number of the numeric range
 *   - Id         - a numeric identifier used to identify the window
 * RESULT
 *   A slider identifier.
 * SOURCE
 *
slider = dw_slider_new( !REXXDW.!DW_VERT, 100, 0 )
 ******
 */
rxfunc( DW_slider_new )
{
   HWND rc; /* function return variable */
   int vertical; /* local variable for arg[0] */
   int increments; /* local variable for arg[1] */
   ULONG id; /* local variable for arg[2] */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 3, 3 ) )
      return( 1 );
   if ( RxStrToInt( RxPackageGlobalData, &argv[0], (RX_INT *)&vertical ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   if ( RxStrToInt( RxPackageGlobalData, &argv[1], (RX_INT *)&increments ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 2 );
      return(1);
   }
   if ( RxStrToULong( RxPackageGlobalData, &argv[2], (RX_ULONG *)&id ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 3 );
      return(1);
   }
   rc = dw_slider_new( vertical, increments, id );
   return RxReturnUnsignedNumber( RxPackageGlobalData, retstr, (ULONG)rc );
}

/****f* Slider/DW_slider_set_pos
 * NAME
 *   DW_slider_set_pos
 * SYNOPSIS
 *   dw_slider_set_pos( Win, Position )
 * FUNCTION
 *   Sets the position of the slider.
 * ARGUMENTS
 *   - Win        - the window identifier returned from DW_slider_new()
 *   - Position   - the position within the slider's range
 * RESULT
 *   No return result.
 * SEE ALSO
 *   DW_slider_get_pos
 * SOURCE
 *
vslider = dw_slider_new( !REXXDW.!DW_VERT, 100, 0 )
...
Call dw_slider_set_pos vslider, 50
...
Say 'The slider is at' dw_slider_get_pos( vslider )
 ******
 */
rxfunc( DW_slider_set_pos )
{
   HWND handle; /* local variable for arg[0] */
   unsigned int position; /* local variable for arg[1] */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 2, 2 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&handle ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   if ( RxStrToUInt( RxPackageGlobalData, &argv[1], (RX_UINT *)&position ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 2 );
      return(1);
   }
   dw_slider_set_pos( handle, position );
   return RxReturnString( RxPackageGlobalData, retstr, "" );
}

/****f* Slider/DW_slider_get_pos
 * NAME
 *   DW_slider_get_pos
 * SYNOPSIS
 *   pos = dw_slider_get_pos( Win )
 * FUNCTION
 *   Returns the current position of the slider.
 * ARGUMENTS
 *   - Win        - the window identifier returned from DW_slider_new()
 * RESULT
 *   The position of the slider.
 * SEE ALSO
 *   DW_slider_set_pos
 * SOURCE
 *
vslider = dw_slider_new( !REXXDW.!DW_VERT, 100, 0 )
Say 'The slider is at' dw_slider_get_pos( vslider )
 ******
 */
rxfunc( DW_slider_get_pos )
{
   unsigned int rc; /* function return variable */
   HWND handle; /* local variable for arg[0] */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 1, 1 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&handle ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   rc = dw_slider_get_pos( handle );
   return RxReturnNumber( RxPackageGlobalData, retstr, (ULONG)rc );
}

/****h* Widgets/Spinbutton
 * DESCRIPTION
 *   A Spinbutton widget is similar to a Slider, but the user selects a numeric
 *   value by clicking an up or down arrow, or by manually entering the value
 *   into an entry field.
 *   A Spinbutton will wrap around at each end of its value limits.
 ******
 */

/****f* Spinbutton/DW_spinbutton_new
 * NAME
 *   DW_spinbutton_new
 * SYNOPSIS
 *   win = dw_spinbutton_new( Text, Id )
 * FUNCTION
 *   Creates a new spinbutton.
 *   A spinbutton provides a widget for selecting a numeric values from a range
 *   or by manually entering a value.
 * ARGUMENTS
 *   - Text       - the text describing the spinbutton
 *   - Id         - a numeric identifier used to identify the window
 * RESULT
 *   A spinbutton identifier.
 * SOURCE
 *
spinbutton = dw_spinbutton_new( 'Select Number', 0 )
 ******
 */
rxfunc( DW_spinbutton_new )
{
   HWND rc; /* function return variable */
   char * text; /* local variable for arg[0] */
   unsigned long id; /* local variable for arg[1] */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 2, 2 ) )
      return( 1 );
   if ( RXVALIDSTRING(argv[0]) )
      text = RXSTRPTR(argv[0]);
   else
      text = "";
   if ( RxStrToULong( RxPackageGlobalData, &argv[1], (RX_ULONG *)&id ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 2 );
      return(1);
   }
   rc = dw_spinbutton_new( text, id );
   return RxReturnUnsignedNumber( RxPackageGlobalData, retstr, (ULONG)rc );
}

/****f* Spinbutton/DW_spinbutton_set_limits
 * NAME
 *   DW_spinbutton_set_limits
 * SYNOPSIS
 *   dw_spinbutton_set_limits( Win, Upper, Lower )
 * FUNCTION
 *   Sets the maximum and minimum values the spinbutton can support.
 * ARGUMENTS
 *   - Win        - the window identifier returned from DW_spinbutton_new()
 *   - Upper      - the maximum number of the spinbutton
 *   - Lower      - the minimum number of the spinbutton
 * RESULT
 *   No return result.
 * SEE ALSO
 *   DW_spinbutton_set_pos, DW_spinbutton_get_pos
 * SOURCE
 *
spinbutton = dw_spinbutton_new( 'Select Number', 0 )
Call dw_spinbutton_set_limits spinbutton, 200, 40
...
Call dw_spinbutton_set_pos spinbutton, 150
...
Say 'The spinbutton is at' dw_spinbutton_get_pos( spinbutton )
 ******
 */
rxfunc( DW_spinbutton_set_limits )
{
   HWND handle; /* local variable for arg[0] */
   long upper; /* local variable for arg[1] */
   long lower; /* local variable for arg[2] */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 3, 3 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&handle ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   if ( RxStrToLong( RxPackageGlobalData, &argv[1], (RX_LONG *)&upper ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 2 );
      return(1);
   }
   if ( RxStrToLong( RxPackageGlobalData, &argv[2], (RX_LONG *)&lower ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 3 );
      return(1);
   }
   dw_spinbutton_set_limits( handle, upper, lower );
   return RxReturnString( RxPackageGlobalData, retstr, "" );
}

/****f* Spinbutton/DW_spinbutton_set_pos
 * NAME
 *   DW_spinbutton_set_pos
 * SYNOPSIS
 *   dw_spinbutton_set_pos( Win, Position )
 * FUNCTION
 *   Sets the position of the spinbutton.
 * ARGUMENTS
 *   - Win        - the window identifier returned from DW_spinbutton_new()
 *   - Position   - the position within the spinbutton's range
 * RESULT
 *   No return result.
 * SEE ALSO
 *   DW_spinbutton_get_pos
 * SOURCE
 *
spinbutton = dw_spinbutton_new( 'Select Number', 0 )
Call dw_spinbutton_set_limits spinbutton, 200, 40
...
Call dw_spinbutton_set_pos spinbutton, 50
...
Say 'The spinbutton is at' dw_spinbutton_get_pos( spinbutton )
 ******
 */
rxfunc( DW_spinbutton_set_pos )
{
   HWND handle; /* local variable for arg[0] */
   long position; /* local variable for arg[1] */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 2, 2 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&handle ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   if ( RxStrToLong( RxPackageGlobalData, &argv[1], (RX_LONG *)&position ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 2 );
      return(1);
   }
   dw_spinbutton_set_pos( handle, position );
   return RxReturnString( RxPackageGlobalData, retstr, "" );
}

/****f* Spinbutton/DW_spinbutton_get_pos
 * NAME
 *   DW_spinbutton_get_pos
 * SYNOPSIS
 *   pos = dw_spinbutton_get_pos( Win )
 * FUNCTION
 *   Returns the current position of the spinbutton.
 * ARGUMENTS
 *   - Win        - the window identifier returned from DW_spinbutton_new()
 * RESULT
 *   The position of the spinbutton.
 * SEE ALSO
 *   DW_spinbutton_set_pos
 * SOURCE
 *
spinbutton = dw_spinbutton_new( 'Select Number', 0 )
Call dw_spinbutton_set_limits spinbutton, 200, 40
...
Call dw_spinbutton_set_pos spinbutton, 150
...
Say 'The spinbutton is at' dw_spinbutton_get_pos( spinbutton )
 ******
 */
rxfunc( DW_spinbutton_get_pos )
{
   long rc; /* function return variable */
   HWND handle; /* local variable for arg[0] */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 1, 1 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&handle ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   rc = dw_spinbutton_get_pos( handle );
   return RxReturnNumber( RxPackageGlobalData, retstr, rc );
}

/****h* Widgets/Splitbar
 * DESCRIPTION
 *   A Splitbar is another frame in which other widgets are packed. The Splitbar
 *   consists of two halves and a divider, which can be dragged by the user to
 *   resize each side of the splitbar. The widgets within each side of the
 *   Splitbar are automatically resized.
 ******
 */

/****f* Splitbar/DW_splitbar_new
 * NAME
 *   DW_splitbar_new
 * SYNOPSIS
 *   win = dw_splitbar_new( Orientation, FirstBox, SecondBox, Id )
 * FUNCTION
 *   Creates a new splitbar.
 *   A splitbar provides a widget for packing two boxes which can be dynamically
 *   resized by the user. This function also packs the boxes.
 * ARGUMENTS
 *   - Orientation- !REXXDW.!DW_VERT to indicate a vertical box
 *                  !REXXDW.!DW_HORZ to indicate a horizontal box
 *   - FirstBox   - the box to pack into the left or top half of the splitbar
 *   - SecondBox  - the box to pack into the right or bottom half of the splitbar
 *   - Id         - a numeric identifier used to identify the window
 * RESULT
 *   A splitbar identifier.
 * SEE ALSO
 *   WidgetOrientation
 * SOURCE
 *
splitbar = dw_splitbar_new( !REXXDW.!DW_HORZ, box1, box2, 0 )
Call dw_splitbar_set splitbar, 40.0
 ******
 */
rxfunc( DW_splitbar_new )
{
   HWND rc; /* function return variable */
   int type; /* local variable for arg[0] */
   HWND topleft; /* local variable for arg[1] */
   HWND bottomright; /* local variable for arg[2] */
   unsigned long id; /* local variable for arg[3] */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 4, 4 ) )
      return( 1 );
   if ( RxStrToInt( RxPackageGlobalData, &argv[0], (RX_INT *)&type ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   if ( RxStrToULong( RxPackageGlobalData, &argv[1], (RX_ULONG *)&topleft ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 2 );
      return(1);
   }
   if ( RxStrToULong( RxPackageGlobalData, &argv[2], (RX_ULONG *)&bottomright ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 3 );
      return(1);
   }
   if ( RxStrToULong( RxPackageGlobalData, &argv[3], (RX_ULONG *)&id ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 4 );
      return(1);
   }
   rc = dw_splitbar_new( type, topleft, bottomright, id );
   return RxReturnUnsignedNumber( RxPackageGlobalData, retstr, (ULONG)rc );
}

/****f* Splitbar/DW_splitbar_set
 * NAME
 *   DW_splitbar_set
 * SYNOPSIS
 *   dw_splitbar_set( Win, Percent )
 * FUNCTION
 *   Sets the relative sizes of the halves of the splitbar, by setting the
 *   percentage of the left or top half.
 * ARGUMENTS
 *   - Win        - the window identifier returned from DW_splitbar_new()
 *   - Percent    - the relative size of the left or top half of the splitbar
 * RESULT
 *   No return result.
 * SOURCE
 *
splitbar = dw_splitbar_new( !REXXDW.!DW_HORZ, box1, box2, 0 )
Call dw_splitbar_set splitbar, 40.0
 ******
 */
rxfunc( DW_splitbar_set )
{
   HWND handle; /* local variable for arg[0] */
   double percent; /* local variable for arg[1] */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 2, 2 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&handle ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   if ( RxStrToDouble( RxPackageGlobalData, &argv[1], (double *)&percent ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 2 );
      return(1);
   }
   dw_splitbar_set( handle, (float)percent );
   return RxReturnString( RxPackageGlobalData, retstr, "" );
}

/****f* Splitbar/DW_splitbar_get
 * NAME
 *   DW_splitbat_get
 * SYNOPSIS
 *   perc = dw_splitbar_get( Win )
 * FUNCTION
 *   Returns the current percentage that the first half of the splitbar occupies.
 * ARGUMENTS
 *   - Win        - the window identifier returned from DW_splitbar_new()
 * RESULT
 *   The percentage of the first half of the splitbar.
 * SEE ALSO
 *   DW_splitbar_new, DW_splitbar_set
 * SOURCE
 *
splitbar = dw_splitbar_new( !REXXDW.!DW_HORZ, box1, box2, 0 )
Call dw_splitbar_set splitbar, 40.0
...
Say 'The splitbar percentage is' dw_splitbar_get( splitbar )
 ******
 */
rxfunc( DW_splitbar_get )
{
   double rc; /* function return variable */
   HWND handle; /* local variable for arg[0] */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 1, 1 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&handle ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   rc = dw_splitbar_get( handle );
   return RxReturnDouble( RxPackageGlobalData, retstr, (double)rc );
}

/****h* Widgets/StaticText
 * DESCRIPTION
 *   StaticText widgets simply display static text.
 ******
 */

/****f* StaticText/DW_text_new
 * NAME
 *   DW_text_new
 * SYNOPSIS
 *   win = dw_text_new( Text, Id )
 * FUNCTION
 *   Creates a new text window.
 *   A text window is a readonly window containing text.
 * ARGUMENTS
 *   - Text       - the text to appear in the window
 *   - Id         - a numeric identifier used to identify the window
 * RESULT
 *   A text window identifier.
 * SEE ALSO
 *   DW_status_text_new
 * SOURCE
 *
text = dw_text_new( 'This is Rexx/DW', 0 )
 ******
 */
rxfunc( DW_text_new )
{
   HWND rc; /* function return variable */
   char * text; /* local variable for arg[0] */
   unsigned long id; /* local variable for arg[1] */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 2, 2 ) )
      return( 1 );
   /* TODO - memory leak !! */
   if ( RXVALIDSTRING(argv[0]) )
      text = strdup( RXSTRPTR(argv[0]) );
   else
      text = strdup( "" );
   if ( RxStrToULong( RxPackageGlobalData, &argv[1], (RX_ULONG *)&id ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 2 );
      return(1);
   }
   rc = dw_text_new( text, id );
   return RxReturnUnsignedNumber( RxPackageGlobalData, retstr, (ULONG)rc );
}

/****f* StaticText/DW_status_text_new
 * NAME
 *   DW_status_text_new
 * SYNOPSIS
 *   win = dw_status_text_new( Text, Id )
 * FUNCTION
 *   Creates a new status text window.
 *   A status text window is a readonly window containing text with a sunken appearence.
 * ARGUMENTS
 *   - Text       - the text to appear in the window
 *   - Id         - a numeric identifier used to identify the window
 * RESULT
 *   A status text window identifier.
 * SEE ALSO
 *   DW_text_new
 * SOURCE
 *
stext = dw_status_text_new( 'The current status is' status, 0 )
 ******
 */
rxfunc( DW_status_text_new )
{
   HWND rc; /* function return variable */
   char * text; /* local variable for arg[0] */
   unsigned long id; /* local variable for arg[1] */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 2, 2 ) )
      return( 1 );
   if ( RXVALIDSTRING(argv[0]) )
      text = RXSTRPTR(argv[0]);
   else
      text = "";
   if ( RxStrToULong( RxPackageGlobalData, &argv[1], (RX_ULONG *)&id ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 2 );
      return(1);
   }
   rc = dw_status_text_new( text, id );
   return RxReturnUnsignedNumber( RxPackageGlobalData, retstr, (ULONG)rc );
}

/****h* Widgets/Tree
 * DESCRIPTION
 *   A Tree widget provides a hierachical display of items. Paricularly
 *   suited to a filesystem display.
 ******
 */

/****f* Tree/DW_tree_new
 * NAME
 *   DW_tree_new
 * SYNOPSIS
 *   win = dw_tree_new( Id )
 * FUNCTION
 *   Creates a new tree widget.
 *   A tree widget allows the user to display a hierarchy in a tree format.
 * ARGUMENTS
 *   - Id         - a numeric identifier used to identify the window
 * RESULT
 *   A tree identifier.
 * SOURCE
 *
tree = dw_tree_new( 0 )
Do i = 1 To text.0
   item.i = dw_tree_insert_after( tree, topitem, Word( text.i ), ,
      fileicon, parent, 'item'i )
End
 ******
 */
rxfunc( DW_tree_new )
{
   HWND rc; /* function return variable */
   unsigned long id; /* local variable for arg[0] */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 1, 1 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&id ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   rc = dw_tree_new( id );
   return RxReturnUnsignedNumber( RxPackageGlobalData, retstr, (ULONG)rc );
}

/****f* Tree/DW_tree_insert
 * NAME
 *   DW_tree_insert
 * SYNOPSIS
 *   data = dw_tree_insert( Win, Title[, Icon[, ParentItem[, ItemData]]] )
 * FUNCTION
 *   Inserts the tree item into the Tree at the end of any other items
 *   in the Tree with the same ParentItem. Intended to be used when populating
 *   and empty Tree
 * ARGUMENTS
 *   - Win        - the window identifier returned from DW_tree_new()
 *   - Title      - the text to appear next to the item
 *   - Icon       - the icon identifier returned from DW_icon_load_from_file()
 *                  or DW_icon_load_from_data() to appear with the item
 *   - ParentItem - the item identifier returned from DW_tree_insert()
 *                  under which this item is to appear
 *   - ItemData   - The user data that is passed to the callback as ItemData
 *                  argument
 * RESULT
 *   A tree item identifier.
 * SEE ALSO
 *   DW_tree_new, DW_tree_insert_after
 * SOURCE
 *
tree = dw_tree_new( 0 )
Do i = 1 To text.0
   item.i = dw_tree_insert( tree, Word( text.i ), fileicon, parent, ,
      'item'i )
End
 ******
 */
rxfunc( DW_tree_insert )
{
   HTREEITEM rc; /* function return variable */
   HWND handle; /* local variable for arg[0] */
   char * title; /* local variable for arg[1] */
   HICN icon=0L; /* local variable for arg[2] */
   HTREEITEM parent=0; /* local variable for arg[3] */
   char * itemdata=NULL; /* local variable for arg[4] */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 2, 5 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&handle ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   /* TODO - memory leak!! */
   if ( RXVALIDSTRING(argv[1]) )
      title = strdup( RXSTRPTR(argv[1]) );
   else
      title = strdup( "" );
   if ( argc > 2
   &&   RxStrToULong( RxPackageGlobalData, &argv[2], (RX_ULONG *)&icon ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 3 );
      return(1);
   }
   if ( argc > 3
   &&   RxStrToULong( RxPackageGlobalData, &argv[3], (RX_ULONG *)&parent ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 4 );
      return(1);
   }
   if ( argc > 4 )
   {
   /* TODO - memory leak!! */
      if ( RXVALIDSTRING(argv[4]) )
         itemdata = strdup( RXSTRPTR(argv[4]) );
      else
         itemdata = strdup( "" );
   }
   rc = dw_tree_insert( handle, title, icon, parent, (void *)itemdata );
   return RxReturnUnsignedNumber( RxPackageGlobalData, retstr, (ULONG)rc );
}

/****f* Tree/DW_tree_insert_after
 * NAME
 *   DW_tree_insert_after
 * SYNOPSIS
 *   data = dw_tree_insert_after( Win, Item, Title[, Icon[, ParentItem[, ItemData]]] )
 * FUNCTION
 *   Inserts the tree item into an existing Tree physically after the item specified
 *   in Item and at the level identified by ParentItem
 * ARGUMENTS
 *   - Win        - the window identifier returned from DW_tree_new()
 *   - Item       - the item in the tree where this item is to follow
 *   - Title      - the text to appear next to the item
 *   - Icon       - the icon identifier returned from DW_icon_load_from_file()
 *                  or DW_icon_load_from_data() to appear with the item
 *   - ParentItem - the item identifier returned from a previous call to
 *                  DW_tree_insert_after() or DW_tree_insert()
 *                  If not specified, the item will be added at the top level.
 *   - ItemData   - The user data that is passed to the callback as ItemData
 *                argument
 * RESULT
 *   A tree item identifier.
 * SEE ALSO
 *   DW_tree_new, DW_tree_insert
 * SOURCE
 *
tree = dw_tree_new( 0 )
topitem = dw_tree_insert( tree, 'Parent' )
Do i = 1 To text.0
   item.i = dw_tree_insert( tree, topitem, Word( text.i ), fileicon, ,
      parent, 'item'i )
End
...
Call dw_tree_insert_after( tree, item.5, "New Item", fileicon, item.3 )
 ******
 */
rxfunc( DW_tree_insert_after )
{
   HTREEITEM rc; /* function return variable */
   HWND handle; /* local variable for arg[0] */
   HTREEITEM item; /* local variable for arg[1] */
   char * title; /* local variable for arg[2] */
   HICN icon=0L; /* local variable for arg[3] */
   HTREEITEM parent=0; /* local variable for arg[4] */
   char * itemdata=NULL; /* local variable for arg[5] */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 3, 6 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&handle ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   if ( RxStrToULong( RxPackageGlobalData, &argv[1], (RX_ULONG *)&item ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 2 );
      return(1);
   }
   /* TODO - memory leak!! */
   if ( RXVALIDSTRING(argv[2]) )
      title = strdup( RXSTRPTR(argv[2]) );
   else
      title = strdup( "" );
   if ( argc > 3
   &&   RxStrToULong( RxPackageGlobalData, &argv[3], (RX_ULONG *)&icon ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 4 );
      return(1);
   }
   if ( argc > 4
   &&   RxStrToULong( RxPackageGlobalData, &argv[4], (RX_ULONG *)&parent ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 5 );
      return(1);
   }
   if ( argc > 5 )
   {
   /* TODO - memory leak!! */
      if ( RXVALIDSTRING(argv[5]) )
         itemdata = strdup( RXSTRPTR(argv[5]) );
      else
         itemdata = strdup( "" );
   }
   rc = dw_tree_insert_after( handle, item, title, icon, parent, (void *)itemdata );
   return RxReturnUnsignedNumber( RxPackageGlobalData, retstr, (ULONG)rc );
}

/****f* Tree/DW_tree_clear
 * NAME
 *   DW_tree_clear
 * SYNOPSIS
 *   dw_tree_clear( Win )
 * FUNCTION
 *   Removes all items from a tree widget.
 * ARGUMENTS
 *   - Win        - the window identifier returned from DW_tree_new()
 * RESULT
 *   No return result.
 * SEE ALSO
 *   DW_tree_new, DW_tree_item_delete
 * SOURCE
 *
tree = dw_tree_new( 0 )
...
Call dw_tree_clear tree
 ******
 */
rxfunc( DW_tree_clear )
{
   HWND handle; /* local variable for arg[0] */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 1, 1 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&handle ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   dw_tree_clear( handle );
   return RxReturnString( RxPackageGlobalData, retstr, "" );
}

/****f* Tree/DW_tree_item_delete
 * NAME
 *   DW_tree_item_delete
 * SYNOPSIS
 *   dw_tree_item_delete( Win, Item )
 * FUNCTION
 *   Deletes the specified Item in the tree. If the item is a parent item
 *   then all child items under it are also deleted. (Is this true???)
 * ARGUMENTS
 *   - Win        - the window identifier returned from DW_tree_new()
 *   - Item       - the item identifier returned from DW_tree_insert()
 *                  or DW_tree_insert_after()
 * RESULT
 *   No return result.
 * SEE ALSO
 *   DW_tree_new, DW_tree_insert, DW_tree_insert_after
 * SOURCE
 *
tree = dw_tree_new( 0 )
Do i = 1 To text.0
   item.i = dw_tree_insert( tree, Word( text.i ), fileicon )
End
...
Call dw_tree_item_delete tree, item.5
 ******
 */
rxfunc( DW_tree_item_delete )
{
   HWND handle; /* local variable for arg[0] */
   HTREEITEM item; /* local variable for arg[1] */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 2, 2 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&handle ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   if ( RxStrToULong( RxPackageGlobalData, &argv[1], (RX_ULONG *)&item ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 2 );
      return(1);
   }
   dw_tree_item_delete( handle, item );
   return RxReturnString( RxPackageGlobalData, retstr, "" );
}

/****f* Tree/DW_tree_item_collapse
 * NAME
 *   DW_tree_item_collapse
 * SYNOPSIS
 *   dw_tree_item_collapse( Win, Item )
 * FUNCTION
 *   Collapses the specified Item in the tree.
 * ARGUMENTS
 *   - Win        - the window identifier returned from DW_tree_new()
 *   - Item       - the item identifier returned from DW_tree_insert()
 *                  or DW_tree_insert_after()
 * RESULT
 *   No return result.
 * SEE ALSO
 *   DW_tree_new, DW_tree_insert, DW_tree_insert_after, DW_tree_item_expand
 * SOURCE
 *
tree = dw_tree_new( 0 )
Do i = 1 To text.0
   item.i = dw_tree_insert( tree, Word( text.i ), fileicon )
End
...
Call dw_tree_item_collapse tree, item.5
 ******
 */
rxfunc( DW_tree_item_collapse )
{
   HWND handle; /* local variable for arg[0] */
   HTREEITEM item; /* local variable for arg[1] */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 2, 2 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&handle ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   if ( RxStrToULong( RxPackageGlobalData, &argv[1], (RX_ULONG *)&item ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 2 );
      return(1);
   }
   dw_tree_item_collapse( handle, item );
   return RxReturnString( RxPackageGlobalData, retstr, "" );
}

/****f* Tree/DW_tree_item_expand
 * NAME
 *   DW_tree_item_expand
 * SYNOPSIS
 *   dw_tree_item_expand( Win, Item )
 * FUNCTION
 *   Expands the specified Item in the tree.
 * ARGUMENTS
 *   - Win        - the window identifier returned from DW_tree_new()
 *   - Item       - the item identifier returned from DW_tree_insert()
 *                  or DW_tree_insert_after()
 * RESULT
 *   No return result.
 * SEE ALSO
 *   DW_tree_new, DW_tree_insert, DW_tree_insert_after, DW_tree_item_collapse
 * SOURCE
 *
tree = dw_tree_new( 0 )
Do i = 1 To text.0
   item.i = dw_tree_insert( tree, Word( text.i ), fileicon )
End
...
Call dw_tree_item_expand tree, item.5
 ******
 */
rxfunc( DW_tree_item_expand )
{
   HWND handle; /* local variable for arg[0] */
   HTREEITEM item; /* local variable for arg[1] */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 2, 2 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&handle ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   if ( RxStrToULong( RxPackageGlobalData, &argv[1], (RX_ULONG *)&item ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 2 );
      return(1);
   }
   dw_tree_item_expand( handle, item );
   return RxReturnString( RxPackageGlobalData, retstr, "" );
}

/****f* Tree/DW_tree_item_set_data
 * NAME
 *   DW_tree_item_set_data
 * SYNOPSIS
 *   dw_tree_item_set_data( Win, Item, UserData )
 * FUNCTION
 *   Sets the userdata for the specified tree Item.
 * ARGUMENTS
 *   - Win        - the window identifier returned from DW_tree_new()
 *   - Item       - the item in the tree to which the data is to be attached
 *   - UserData   - the string to associate with the tree item's user data
 * RESULT
 *   No return result.
 * SEE ALSO
 *   DW_tree_new, DW_tree_item_get_data
 * SOURCE
 *
tree = dw_tree_new( 0 )
Do i = 1 To text.0
   item.i = dw_tree_insert( tree, Word( text.i ), fileicon )
   Call dw_tree_item_set_data tree, item.i, 'item'i
End
...
Say 'The data from this item is' dw_tree_item_get_data( tree, item.5 ) ,
   'and should be "item5"'
 ******
 */
rxfunc( DW_tree_item_set_data )
{
   HWND handle; /* local variable for arg[0] */
   HTREEITEM item; /* local variable for arg[1] */
   char * itemdata=NULL; /* local variable for arg[2] */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 3, 3 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&handle ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   if ( RxStrToULong( RxPackageGlobalData, &argv[1], (RX_ULONG *)&item ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 2 );
      return(1);
   }
   /* TODO - memory leak!! */
   if ( RXVALIDSTRING(argv[2]) )
      itemdata = strdup( RXSTRPTR(argv[2]));
   else
      itemdata = strdup("");
   dw_tree_item_set_data( handle, item, (void *)itemdata );
   return RxReturnString( RxPackageGlobalData, retstr, "" );
}

/****f* Tree/DW_tree_item_get_data
 * NAME
 *   DW_tree_item_get_data
 * SYNOPSIS
 *   data = dw_tree_item_get_data( Win, Item )
 * FUNCTION
 *   Returns the data the user set for this Item when calling DW_tree_item_set_data().
 * ARGUMENTS
 *   - Win        - the window identifier returned from DW_tree_new()
 *   - Item       - the item identifier returned from DW_tree_insert()
 * RESULT
 *   The user supplied data.
 * SEE ALSO
 *   DW_tree_new, DW_tree_item_set_data
 * SOURCE
 *
tree = dw_tree_new( 0 )
Do i = 1 To text.0
   item.i = dw_tree_insert( tree, Word( text.i ), fileicon )
   Call dw_tree_item_set_data tree, item.i, 'item'i
End
...
Say 'The data from this item is' dw_tree_item_get_data( tree, item.5 ) ,
   'and should be "item5"'
 ******
 */
rxfunc( DW_tree_item_get_data )
{
   char * rc; /* function return variable */
   HWND handle; /* local variable for arg[0] */
   HTREEITEM item; /* local variable for arg[1] */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 2, 2 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&handle ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   if ( RxStrToULong( RxPackageGlobalData, &argv[1], (RX_ULONG *)&item ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 2 );
      return(1);
   }
   rc = (char *)dw_tree_item_get_data( handle, item );
   return RxReturnString( RxPackageGlobalData, retstr, rc );
}

/****f* Tree/DW_tree_item_select
 * NAME
 *   DW_tree_item_select
 * SYNOPSIS
 *   dw_tree_item_select( Win, Item )
 * FUNCTION
 *   Makes the specified tree Item the currently selected item.
 * ARGUMENTS
 *   - Win        - the window identifier returned from DW_tree_new()
 *   - Item       - the item identifier returned from DW_tree_insert()
 *                  or DW_tree_insert_after()
 * RESULT
 *   No return result.
 * SEE ALSO
 *   DW_tree_new, DW_tree_item_set_data, DW_tree_insert, DW_tree_insert_after
 * SOURCE
 *
tree = dw_tree_new( 0 )
Do i = 1 To text.0
   item.i = dw_tree_insert( tree, Word( text.i ), fileicon )
   Call dw_tree_item_set_data tree, item.i, 'item'i
End
...
Call dw_tree_item_select( tree, item.5 )
 ******
 */
rxfunc( DW_tree_item_select )
{
   HWND handle; /* local variable for arg[0] */
   HTREEITEM item; /* local variable for arg[1] */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 2, 2 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&handle ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   if ( RxStrToULong( RxPackageGlobalData, &argv[1], (RX_ULONG *)&item ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 2 );
      return(1);
   }
   dw_tree_item_select( handle, item );
   return RxReturnString( RxPackageGlobalData, retstr, "" );
}

/****f* Tree/DW_tree_item_change
 * NAME
 *   DW_tree_item_change
 * SYNOPSIS
 *   dw_tree_item_change( Win, Item, Title, Icon )
 * FUNCTION
 *   Changes the Title and Icon for the specified tree Item.
 * ARGUMENTS
 *   - Win        - the window identifier returned from DW_tree_new()
 *   - Item       - the item in the tree where this item is to follow
 *   - Title      - the text to appear next to the item
 *   - Icon       - the icon identifier returned from DW_icon_load_from_file()
 *                  or DW_icon_load_from_data() to appear with the item
 * RESULT
 *   No return result.
 * SEE ALSO
 *   DW_tree_new, DW_tree_insert, DW_tree_insert_after
 * SOURCE
 *
tree = dw_tree_new( 0 )
Do i = 1 To text.0
   item.i = dw_tree_insert( tree, Word( text.i ), fileicon )
   Call dw_tree_item_set_data tree, item.i, 'item'i
End
...
Call dw_tree_item_change tree, item.5, 'New Title', diricon
 ******
 */
rxfunc( DW_tree_item_change )
{
   HWND handle; /* local variable for arg[0] */
   HTREEITEM item; /* local variable for arg[1] */
   char * title; /* local variable for arg[2] */
   HICN icon; /* local variable for arg[3] */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 4, 4 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&handle ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   if ( RxStrToULong( RxPackageGlobalData, &argv[1], (RX_ULONG *)&item ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 2 );
      return(1);
   }
   /* TODO - memory leak!! */
   if ( RXVALIDSTRING(argv[2]) )
      title = strdup( RXSTRPTR(argv[2]) );
   else
      title = strdup( "" );
   if ( RxStrToULong( RxPackageGlobalData, &argv[3], (RX_ULONG *)&icon ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 4 );
      return(1);
   }
   dw_tree_item_change( handle, item, title, icon );
   return RxReturnString( RxPackageGlobalData, retstr, "" );
}

/****f* Tree/DW_tree_get_parent
 * NAME
 *   DW_tree_get_parent
 * SYNOPSIS
 *   data = dw_tree_get_parent( Win, Item )
 * FUNCTION
 *   Returns the parent of Item
 * ARGUMENTS
 *   - Win        - the window identifier returned from DW_tree_new()
 *   - Item       - the item identifier returned from DW_tree_insert()
 * RESULT
 *   The parent of the supplied Item.
 * SEE ALSO
 *   DW_tree_new, DW_tree_item_set_data
 * SOURCE
 *
tree = dw_tree_new( 0 )
Do i = 1 To text.0
   item.i = dw_tree_insert( tree, Word( text.i ), fileicon )
   Call dw_tree_item_set_data tree, item.i, 'item'i
End
...
Say 'The parent of item.5 is' dw_tree_get_parent( tree, item.5 ) ,
   'and should be' item.4
 ******
 */
rxfunc( DW_tree_get_parent )
{
   HTREEITEM rc; /* function return variable */
   HWND handle; /* local variable for arg[0] */
   HTREEITEM item; /* local variable for arg[1] */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 2, 2 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&handle ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   if ( RxStrToULong( RxPackageGlobalData, &argv[1], (RX_ULONG *)&item ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 2 );
      return(1);
   }
   rc = dw_tree_get_parent( handle, item );
   return RxReturnUnsignedNumber( RxPackageGlobalData, retstr, (ULONG)rc );
}

/****h* Widgets/Window
 * DESCRIPTION
 *   A Window is frame in which other widgets are contained. It is also a generic
 *   term for a number of other widgets that can be manipulated by a number of
 *   functions in this section.
 ******
 */

/****f* Window/DW_window_new
 * NAME
 *   DW_window_new
 * SYNOPSIS
 *   win = dw_window_new( Win, Title, Flags )
 * FUNCTION
 *   Creates a new window.
 *   A window is a visible frame containing a border and titlebar. It can also
 *   contain minimise/maximise/close buttons etc depending on the value of
 *   Flags.
 * ARGUMENTS
 *   - Win        - the window identifier returned from various functions that
 *                  return a window identifier or DW_DESKTOP. This is this window's
 *                  parent window
 *   - Title      - the text to appear in the titlebar of the window (if it has one)
 *   - Flags      - the flags defining the style and initial behaviour of the window
 * RESULT
 *   A window identifier.
 * SEE ALSO
 *   WindowStyleFlags
 * SOURCE
 *
win = dw_window_new( !REXXDW.!DW_DESKTOP, 'Edit User Preferences', ,
   windowstyle )
 ******
 */
rxfunc( DW_window_new )
{
   HWND rc; /* function return variable */
   HWND hwndOwner; /* local variable for arg[0] */
   char * title; /* local variable for arg[1] */
   unsigned long flStyle; /* local variable for arg[2] */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 3, 3 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&hwndOwner ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   if ( RXVALIDSTRING(argv[1]) )
      title = RXSTRPTR(argv[1]);
   else
      title = "";
   if ( RxStrToULong( RxPackageGlobalData, &argv[2], (RX_ULONG *)&flStyle ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 3 );
      return(1);
   }
   rc = dw_window_new( hwndOwner, title, flStyle );
   return RxReturnUnsignedNumber( RxPackageGlobalData, retstr, (ULONG)rc );
}

/****f* Window/DW_mdi_new
 * NAME
 *   DW_mdi_new
 * SYNOPSIS
 *   win = dw_mdi_new( Id )
 * FUNCTION
 *   Creates a new Multi-Document Interface (MDI) frame.
 *   An MDI frame is used as a container for multiple windows.
 * ARGUMENTS
 *   - Id        - a numeric identifier used to identify the window
 * RESULT
 *   An MDI frame identifier.
 * SEE ALSO
 *   DW_window_new
 * SOURCE
 *
...
mdi = dw_mdi_new( 100 )
...
mdiwin1 = dw_window_new( mdi, 'First MDI window', style )
mdiwin2 = dw_window_new( mdi, 'Second MDI window', style )
...
 ******
 */
rxfunc( DW_mdi_new )
{
   HWND rc; /* function return variable */
   unsigned long id; /* local variable for arg[0] */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 1, 1 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&id ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   rc = dw_mdi_new( id );
   return RxReturnUnsignedNumber( RxPackageGlobalData, retstr, (ULONG)rc );
}

/****f* Window/DW_window_destroy
 * NAME
 *   DW_window_destroy
 * SYNOPSIS
 *   dw_window_destroy( Win )
 * FUNCTION
 *   Destroys a window and all other items created within it.
 * ARGUMENTS
 *   - Win        - the window identifier returned from various functions that
 *                  return a window identifier
 * RESULT
 *   0 if successful, any other value is an error
 * SEE ALSO
 *   DW_window_new
 * SOURCE
 *
win = dw_window_new( !REXXDW.!DW_DESKTOP, 'Window on desktop', style )
...
Call dw_window_destroy( win )
 ******
 */
rxfunc( DW_window_destroy )
{
   int rc; /* function return variable */
   HWND handle; /* local variable for arg[0] */
   REXXDWWidgetData *widget;
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 1, 1 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&handle ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   rc = dw_window_destroy( handle );
   /*
    * cleanup our own widgets last
    */
   widget = findWidget( RxPackageGlobalData, handle );
   if ( widget != NULL )
   {
      deleteWidget( RxPackageGlobalData, widget );
   }
   return RxReturnNumber( RxPackageGlobalData, retstr, rc );
}

/****f* Window/DW_window_capture
 * NAME
 *   DW_window_capture
 * SYNOPSIS
 *   dw_window_capture( Win )
 * FUNCTION
 *   Captures the mouse pointer to the specified window.
 * ARGUMENTS
 *   - Win        - the window identifier returned from various functions that
 *                  return a window identifier
 * RESULT
 *   No return result.
 * SEE ALSO
 *   DW_window_release
 * NOTES
 *   Only 1 window can capture the mouse pointer at any point in time.
 * SOURCE
 *
Call dw_window_capture( win )
 ******
 */
rxfunc( DW_window_capture )
{
   HWND handle; /* local variable for arg[0] */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 1, 1 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&handle ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   dw_window_capture( handle );
   return RxReturnString( RxPackageGlobalData, retstr, "" );
}

/****f* Window/DW_window_click_default
 * NAME
 *   DW_window_click_default
 * SYNOPSIS
 *   dw_window_click_default( Win, DefaultWin )
 * FUNCTION
 *   Defines the window; DefaultWin that is to receive a button click when the ENTER
 *   key is pressed in Win.
 * ARGUMENTS
 *   - Win        - the window identifier returned from a call to dw_window_new()
 *                  used as the source of the ENTER key
 *   - DefaultWin - the window identifier returned from various functions that
 *                  return a window identifier which receives button click
 * RESULT
 *   No return result.
 * NOTES
 *   The source window; Win, should be a top-level window.
 *   Only one widget within a top-level window can be the widget to receive the
 *   click event.
 *   The DefaultWin should be a widget that accepts the DW_CLICKED_EVENT signal.
 * SOURCE
 *
b1 = dw_button_new( 'OK', 0 )
...
Call dw_signal_connect b1, !REXXDW.!DW_CLICKED_EVENT, 'QuitCallback', !global.!mainwindow
...
Call dw_window_click_default( win, b1 )
 ******
 */
rxfunc( DW_window_click_default )
{
   HWND window; /* local variable for arg[0] */
   HWND next; /* local variable for arg[1] */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 2, 2 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&window ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   if ( RxStrToULong( RxPackageGlobalData, &argv[1], (RX_ULONG *)&next ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 2 );
      return(1);
   }
   dw_window_click_default( window, next );
   return RxReturnString( RxPackageGlobalData, retstr, "" );
}

/****f* Window/DW_window_default
 * NAME
 *   DW_window_default
 * SYNOPSIS
 *   dw_window_default( Win, DefaultWin )
 * FUNCTION
 *   Sets the default focus item; DefaultWin for the specified window.
 * ARGUMENTS
 *   - Win        - the window identifier returned from various functions that
 *                  return a window identifier containing DefaultWin
 *   - DefaultWin - the window identifier returned from various functions that
 *                  return a window identifier which receives default focus
 * RESULT
 *   No return result.
 * SOURCE
 *
Call dw_window_default( win, mywin )
 ******
 */
rxfunc( DW_window_default )
{
   HWND window; /* local variable for arg[0] */
   HWND defaultitem; /* local variable for arg[1] */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 2, 2 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&window ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   if ( RxStrToULong( RxPackageGlobalData, &argv[1], (RX_ULONG *)&defaultitem ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 2 );
      return(1);
   }
   dw_window_default( window, defaultitem );
   return RxReturnString( RxPackageGlobalData, retstr, "" );
}

/****f* Window/DW_window_enable
 * NAME
 *   DW_window_enable
 * SYNOPSIS
 *   dw_window_enable( Win )
 * FUNCTION
 *   Enables a window.
 * ARGUMENTS
 *   - Win        - the window identifier returned from various functions that
 *                  return a window identifier
 * RESULT
 *   No return result.
 * SEE ALSO
 *   DW_window_disable
 * SOURCE
 *
win = dw_window_new( !REXXDW.!DW_DESKTOP, 'Window on desktop', style )
...
Call dw_window_disable( win )
...
Call dw_window_enable( win )
 ******
 */
rxfunc( DW_window_enable )
{
   HWND handle; /* local variable for arg[0] */
   REXXDWWidgetData *widget;
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 1, 1 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&handle ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   /*
    * look for our own widgets first
    */
   widget = findWidget( RxPackageGlobalData, handle );
   if ( widget != NULL )
   {
      switch( widget->widget_type )
      {
         case REXXDW_WIDGET_LISTBOX_SELECTOR:
            dw_window_enable( widget->widget.ListboxSelectorData.LeftListbox );
            dw_window_enable( widget->widget.ListboxSelectorData.RightListbox );
            dw_window_enable( widget->widget.ListboxSelectorData.LeftButton );
            dw_window_enable( widget->widget.ListboxSelectorData.RightButton );
            dw_window_enable( widget->widget.ListboxSelectorData.AllLeftButton );
            dw_window_enable( widget->widget.ListboxSelectorData.AllRightButton );
            break;
         default:
            break;
      }
   }
   else
   {
      dw_window_enable( handle );
   }
   return RxReturnString( RxPackageGlobalData, retstr, "" );
}

/****f* Window/DW_window_disable
 * NAME
 *   DW_window_disable
 * SYNOPSIS
 *   dw_window_disable( Win )
 * FUNCTION
 *   Disables a window.
 * ARGUMENTS
 *   - Win        - the window identifier returned from various functions that
 *                  return a window identifier
 * RESULT
 *   No return result.
 * SEE ALSO
 *   DW_window_enable
 * SOURCE
 *
win = dw_window_new( !REXXDW.!DW_DESKTOP, 'Window on desktop', style )
...
Call dw_window_disable( win )
 ******
 */
rxfunc( DW_window_disable )
{
   HWND handle; /* local variable for arg[0] */
   REXXDWWidgetData *widget;
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 1, 1 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&handle ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   /*
    * look for our own widgets first
    */
   widget = findWidget( RxPackageGlobalData, handle );
   if ( widget != NULL )
   {
      switch( widget->widget_type )
      {
         case REXXDW_WIDGET_LISTBOX_SELECTOR:
            dw_window_disable( widget->widget.ListboxSelectorData.LeftListbox );
            dw_window_disable( widget->widget.ListboxSelectorData.RightListbox );
            dw_window_disable( widget->widget.ListboxSelectorData.LeftButton );
            dw_window_disable( widget->widget.ListboxSelectorData.RightButton );
            dw_window_disable( widget->widget.ListboxSelectorData.AllLeftButton );
            dw_window_disable( widget->widget.ListboxSelectorData.AllRightButton );
            break;
         default:
            break;
      }
   }
   else
   {
      dw_window_disable( handle );
   }
   return RxReturnString( RxPackageGlobalData, retstr, "" );
}

/****f* Window/DW_window_from_id
 * NAME
 *   DW_window_from_id
 * SYNOPSIS
 *   win = dw_window_from_id( Win, Id )
 * FUNCTION
 *   Returns the window identifier from a widget within Win with the given Id.
 * ARGUMENTS
 *   - Win        - the window identifier returned from various functions that
 *                  return a window identifier
 *   - Id         - the window id provided by the user in various function calls
 * RESULT
 *   The window identifier.
 * NOTES
 *   For this function to work, all Ids for widgets packed into the one window
 *   must be unique.
 * SOURCE
 *
box = dw_box_new( !REXXDW.!DW_VERT, 0 )
bitmap = dw_bitmap_new( 100 )
Call dw_box_pack_start box, bitmap, 10, 10, !REXXDW.!DW_DONT_EXPAND_HORZ, ,
   !REXXDW.!DW_DONT_EXPAND_VERT, 0
button = dw_button_new( "Quit", 10 )
Call dw_box_pack_start box, button, 10, 10, !REXXDW.!DW_DONT_EXPAND_HORZ, ,
   !REXXDW.!DW_DONT_EXPAND_VERT, 0
...
Say 'Id' dw_window_from_id( box, 100 ) 'should be the same as' bitmap
 ******
 */
rxfunc( DW_window_from_id )
{
   HWND rc; /* function return variable */
   HWND handle; /* local variable for arg[0] */
   int id; /* local variable for arg[1] */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 2, 2 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&handle ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   if ( RxStrToInt( RxPackageGlobalData, &argv[1], (RX_INT *)&id ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 2 );
      return(1);
   }
   rc = dw_window_from_id( handle, id );
   return RxReturnUnsignedNumber( RxPackageGlobalData, retstr, (ULONG)rc );
}

/*
 * void : dw_window_function [HWND handle, void *function, void *data] ( 3 )
 */
rxfunc( DW_window_function )
{
   HWND handle; /* local variable for arg[0] */
   void * function; /* local variable for arg[1] */
   void * data; /* local variable for arg[2] */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 3, 3 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&handle ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   if ( RxStrToULong( RxPackageGlobalData, &argv[1], (RX_ULONG *)&function ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 2 );
      return(1);
   }
   if ( RxStrToULong( RxPackageGlobalData, &argv[2], (RX_ULONG *)&data ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 3 );
      return(1);
   }
   dw_window_function( handle, function, data );
   return RxReturnString( RxPackageGlobalData, retstr, "" );
}

/****f* Window/DW_window_show
 * NAME
 *   DW_window_show
 * SYNOPSIS
 *   rc = dw_window_show( Win )
 * FUNCTION
 *   Makes the specified window visible and sets focus to that window.
 * ARGUMENTS
 *   - Win        - the window identifier returned from various functions that
 *                  return a window identifier
 * RESULT
 *   0 if function succeeds, error for any other value
 * SEE ALSO
 *   DW_window_hide
 * SOURCE
 *
win = dw_window_new( !REXXDW.!DW_DESKTOP, 'Edit User Preferences', ,
   windowstyle )
...
Call dw_window_show( win )
 ******
 */
rxfunc( DW_window_show )
{
   int rc; /* function return variable */
   HWND handle; /* local variable for arg[0] */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 1, 1 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&handle ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   rc = dw_window_show( handle );
   return RxReturnNumber( RxPackageGlobalData, retstr, rc );
}

/****f* Window/DW_window_hide
 * NAME
 *   DW_window_hide
 * SYNOPSIS
 *   dw_window_hide( Win )
 * FUNCTION
 *   Makes the specified window invisible.
 * ARGUMENTS
 *   - Win        - the window identifier returned from various functions that
 *                  return a window identifier
 * RESULT
 *   No return result.
 * SEE ALSO
 *   DW_window_show
 * SOURCE
 *
win = dw_window_new( !REXXDW.!DW_DESKTOP, 'Edit User Preferences', ,
   windowstyle )
...
Call dw_window_hide( win )
 ******
 */
rxfunc( DW_window_hide )
{
   int rc; /* function return variable */
   HWND handle; /* local variable for arg[0] */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 1, 1 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&handle ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   rc = dw_window_hide( handle );
   return RxReturnNumber( RxPackageGlobalData, retstr, rc );
}

/****f* Window/DW_window_lower
 * NAME
 *   DW_window_lower
 * SYNOPSIS
 *   dw_window_lower( Win )
 * FUNCTION
 *   Makes the specified window appear behind all others.
 * ARGUMENTS
 *   - Win        - the window identifier returned from various functions that
 *                  return a window identifier
 * RESULT
 *   No return result.
 * SEE ALSO
 *   DW_window_raise
 * SOURCE
 *
win = dw_window_new( !REXXDW.!DW_DESKTOP, 'Edit User Preferences', ,
   windowstyle )
...
Call dw_window_lower( win )
 ******
 */
rxfunc( DW_window_lower )
{
   int rc; /* function return variable */
   HWND handle; /* local variable for arg[0] */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 1, 1 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&handle ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   rc = dw_window_lower( handle );
   return RxReturnNumber( RxPackageGlobalData, retstr, rc );
}

/****f* Window/DW_window_minimize
 * NAME
 *   DW_window_minimize
 * SYNOPSIS
 *   dw_window_minimize( Win )
 * FUNCTION
 *   Minimises the specified window.
 * ARGUMENTS
 *   - Win        - the window identifier returned from various functions that
 *                  return a window identifier
 * RESULT
 *   No return result.
 * SEE ALSO
 *   DW_window_maximize; but there isn't one!!
 * SOURCE
 *
win = dw_window_new( !REXXDW.!DW_DESKTOP, 'Edit User Preferences', ,
   windowstyle )
...
Call dw_window_minimize( win )
 ******
 */
rxfunc( DW_window_minimize )
{
   int rc; /* function return variable */
   HWND handle; /* local variable for arg[0] */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 1, 1 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&handle ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   rc = dw_window_minimize( handle );
   return RxReturnNumber( RxPackageGlobalData, retstr, rc );
}

/****f* Window/DW_window_raise
 * NAME
 *   DW_window_raise
 * SYNOPSIS
 *   dw_window_raise( Win )
 * FUNCTION
 *   Makes the specified window appear in front of all others.
 * ARGUMENTS
 *   - Win        - the window identifier returned from various functions that
 *                  return a window identifier
 * RESULT
 *   No return result.
 * SEE ALSO
 *   DW_window_lower
 * SOURCE
 *
win = dw_window_new( !REXXDW.!DW_DESKTOP, 'Edit User Preferences', ,
   windowstyle )
...
Call dw_window_raise( win )
 ******
 */
rxfunc( DW_window_raise )
{
   int rc; /* function return variable */
   HWND handle; /* local variable for arg[0] */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 1, 1 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&handle ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   rc = dw_window_raise( handle );
   return RxReturnNumber( RxPackageGlobalData, retstr, rc );
}

/****f* Window/DW_window_redraw
 * NAME
 *   DW_window_redraw
 * SYNOPSIS
 *   dw_window_redraw( Win )
 * FUNCTION
 *   Redraws the specified window and all of widgets contained within it.
 *   This should be called after re-packing any widgets within the window.
 * ARGUMENTS
 *   - Win        - the window identifier returned from various functions that
 *                  return a window identifier
 * RESULT
 *   No return result.
 * SEE ALSO
 *   DW_window_show
 * SOURCE
 *
win = dw_window_new( !REXXDW.!DW_DESKTOP, 'Edit User Preferences', ,
   windowstyle )
...
Call dw_window_redraw( win )
 ******
 */
rxfunc( DW_window_redraw )
{
   HWND handle; /* local variable for arg[0] */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 1, 1 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&handle ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   dw_window_redraw( handle );
   return RxReturnString( RxPackageGlobalData, retstr, "" );
}

/****f* Window/DW_window_release
 * NAME
 *   DW_window_release
 * SYNOPSIS
 *   dw_window_release( )
 * FUNCTION
 *   Releases the captured mouse pointer from a previous call to DW_window_captture().
 * ARGUMENTS
 *   None
 * RESULT
 *   No return result.
 * SEE ALSO
 *   DW_window_capture
 * NOTES
 *   Only 1 window can capture the mouse pointer at any point in time.
 * SOURCE
 *
Call dw_window_capture( win )
...
Call dw_window_release( )
 ******
 */
rxfunc( DW_window_release )
{
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 0, 0 ) )
      return( 1 );
   dw_window_release( );
   return RxReturnString( RxPackageGlobalData, retstr, "" );
}

/****f* Window/DW_window_reparent
 * NAME
 *   DW_window_reparent
 * SYNOPSIS
 *   dw_window_reparent( Win, ParentWin )
 * FUNCTION
 *   Changes the parent window; ParentWin of the specified Win.
 * ARGUMENTS
 *   - Win        - the window identifier returned from various functions that
 *                  return a window identifier to be reparented
 *   - ParentWin  - the window identifier returned from various functions that
 *                  return a window to becoem the new parent. May also be DW_DESKTOP
 * RESULT
 *   No return result.
 * SOURCE
 *
mdi = dw_mdi_new( 100 )
win = dw_window_new( mdi, 'Edit User Preferences', windowstyle )
...
Call dw_window_reparent( win, !REXXDW.!DW_DESKTOP )
...
box1 = dw_box_new( !REXXDW.!DW_VERT, 0 )
Call dw_box_pack_start win, box1, 10, 10, !REXXDW.!DW_DONT_EXPAND_HORZ, ,
   !REXXDW.!DW_DONT_EXPAND_VERT, 0
button = dw_button_new( "Quit", 10 )
Call dw_box_pack_start box1, button, 10, 10, !REXXDW.!DW_DONT_EXPAND_HORZ, ,
   !REXXDW.!DW_DONT_EXPAND_VERT, 0
...
box2 = dw_box_new( !REXXDW.!DW_VERT, 0 )
Call dw_window_reparent box2, button
 ******
 */
rxfunc( DW_window_reparent )
{
   HWND handle; /* local variable for arg[0] */
   HWND newparent; /* local variable for arg[1] */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 2, 2 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&handle ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   if ( RxStrToULong( RxPackageGlobalData, &argv[1], (RX_ULONG *)&newparent ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 2 );
      return(1);
   }
   dw_window_reparent( handle, newparent );
   return RxReturnString( RxPackageGlobalData, retstr, "" );
}

/****f* Window/DW_window_set_bitmap
 * NAME
 *   DW_window_set_bitmap
 * SYNOPSIS
 *   dw_window_set_bitmap( Win, Id, Filename )
 * FUNCTION
 *   Associates the bitmap in Filename with the specified window.
 * ARGUMENTS
 *   - Win        - the window identifier returned from various functions that
 *                  return a window identifier
 *   - Id         - a numeric identifier used to identify the window
 *   - Filename   - the name of a file containing a valid bitmap image (.BMP, or .XPM)
 * RESULT
 *   No return result.
 * SEE ALSO
 *   DW_bitmap_new_from_file, DW_bitmap_new_from_data
 * SOURCE
 *
...
bitmap = dw_bitmap_new( 100 )
Call dw_box_pack_start win, bitmap, 10, 10, !REXXDW.!DW_DONT_EXPAND_HORZ, ,
   !REXXDW.!DW_DONT_EXPAND_VERT, 0
Call dw_window_set_bitmap bitmap, 0, 'mypicture'
 ******
 */
rxfunc( DW_window_set_bitmap )
{
   HWND handle; /* local variable for arg[0] */
   unsigned long id; /* local variable for arg[1] */
   char * filename; /* local variable for arg[2] */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 3, 3 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&handle ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   if ( RxStrToULong( RxPackageGlobalData, &argv[1], (RX_ULONG *)&id ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 2 );
      return(1);
   }
   if ( RXVALIDSTRING(argv[2]) )
      filename = RXSTRPTR(argv[2]);
   else
      filename = "";
   dw_window_set_bitmap( handle, id, filename );
   return RxReturnString( RxPackageGlobalData, retstr, "" );
}

/****f* Window/DW_window_set_bitmap_from_data
 * NAME
 *   DW_window_set_bitmap_from_data
 * SYNOPSIS
 *   dw_window_set_bitmap_from_data( Win, Id, Data )
 * FUNCTION
 *   Associates the bitmap in Data with the specified window.
 * ARGUMENTS
 *   - Win        - the window identifier returned from various functions that
 *                  return a window identifier
 *   - Id         - a numeric identifier used to identify the window
 *   - Data       - the contents of the image.
 * RESULT
 *   No return result.
 * SOURCE
 *
...
ico = "0000A6D7007E"
...
bitmap = dw_bitmap_new( 100 )
Call dw_box_pack_start win, bitmap, 10, 10, !REXXDW.!DW_DONT_EXPAND_HORZ, ,
   !REXXDW.!DW_DONT_EXPAND_VERT, 0
Call dw_window_set_bitmap_from_data bitmap, 0, x2c( ico )
 ******
 */
rxfunc( DW_window_set_bitmap_from_data )
{
   HWND handle; /* local variable for arg[0] */
   unsigned long id; /* local variable for arg[1] */
   char * data; /* local variable for arg[2] */
   int len;
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 3, 3 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&handle ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   if ( RxStrToULong( RxPackageGlobalData, &argv[1], (RX_ULONG *)&id ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 2 );
      return(1);
   }
   if ( RXVALIDSTRING(argv[2]) )
   {
      data = RXSTRPTR(argv[2]);
      len = RXSTRLEN(argv[2]);
   }
   else
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Supplied image data is empty." );
      return(1);
   }
   dw_window_set_bitmap_from_data( handle, id, data, len );
   return RxReturnString( RxPackageGlobalData, retstr, "" );
}

/****f* Window/DW_window_set_border
 * NAME
 *   DW_window_set_border
 * SYNOPSIS
 *   dw_window_set_border( Win, Width )
 * FUNCTION
 *   Sets the width of the border for the specified window.
 * ARGUMENTS
 *   - Win        - the window identifier returned from various functions that
 *                  return a window identifier
 *   - Width      - the width in pixels of the window's border
 * RESULT
 *   0 if successful, any other value is an error
 * NOTES
 *   This function only works on OS/2.
 * SOURCE
 *
win = dw_window_new( mdi, 'Edit User Preferences', windowstyle )
Call dw_window_set_border win, 20
 ******
 */
rxfunc( DW_window_set_border )
{
   int rc; /* function return variable */
   HWND handle; /* local variable for arg[0] */
   int border; /* local variable for arg[1] */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 2, 2 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&handle ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   if ( RxStrToInt( RxPackageGlobalData, &argv[1], (RX_INT *)&border ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 2 );
      return(1);
   }
   rc = dw_window_set_border( handle, border );
   return RxReturnNumber( RxPackageGlobalData, retstr, rc );
}

/****f* Window/DW_window_set_pointer
 * NAME
 *   DW_window_set_pointer
 * SYNOPSIS
 *   dw_window_set_pointer( Win, Pointer )
 * FUNCTION
 *   Changes the mouse pointer to Pointer.
 * ARGUMENTS
 *   - Win        - the window identifier returned from various functions that
 *                  return a window identifier
 *   - Pointer    - the new pointer; one of PointerTypes
 * RESULT
 *   No return result.
 * SEE ALSO
 *   PointerTypes
 * NOTES
 *   To ensure that the pointer is changed immediately, call DW_main_sleep()
 *   afterwards for a short period, like 10 milliseconds
 * SOURCE
 *
win = dw_window_new( !REXXDW.!DW_DESKTOP, 'Edit User Preferences', ,
   windowstyle )
...
Call dw_window_set_pointer( win, !REXXDW.!DW_POINTER_CLOCK )
Call dw_main_sleep 10
 ******
 */
rxfunc( DW_window_set_pointer )
{
   HWND handle; /* local variable for arg[0] */
   int pointertype; /* local variable for arg[1] */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 2, 2 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&handle ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   if ( RxStrToInt( RxPackageGlobalData, &argv[1], (RX_INT *)&pointertype ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 2 );
      return(1);
   }
   dw_window_set_pointer( handle, pointertype );
   return RxReturnString( RxPackageGlobalData, retstr, "" );
}

/****f* Window/DW_window_set_color
 * NAME
 *   DW_window_set_color
 * SYNOPSIS
 *   dw_window_set_color( Win, Fore, Back )
 * FUNCTION
 *   Sets the foreground and background colour of the window.
 * ARGUMENTS
 *   - Win        - the window identifier returned from various functions that
 *                  return a window identifier
 *   - Fore       - the internal RexxDW colour for the foreground
 *   - Back       - the internal RexxDW colour for the background
 * RESULT
 *   0 if successful, any other value is an error
 * SEE ALSO
 *   Colours, DW_rgb
 * SOURCE
 *
text = dw_text_new( 'File to browse', 0 )
Call dw_window_set_color text, !REXXDW.!DW_CLR_BLACK, !REXXDW.!DW_CLR_YELLOW
 ******
 */
rxfunc( DW_window_set_color )
{
   int rc; /* function return variable */
   HWND handle; /* local variable for arg[0] */
   unsigned long fore; /* local variable for arg[1] */
   unsigned long back; /* local variable for arg[2] */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 3, 3 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&handle ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   if ( RxStrToULong( RxPackageGlobalData, &argv[1], (RX_ULONG *)&fore ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 2 );
      return(1);
   }
   if ( RxStrToULong( RxPackageGlobalData, &argv[2], (RX_ULONG *)&back ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 3 );
      return(1);
   }
   rc = dw_window_set_color( handle, fore, back );
   return RxReturnNumber( RxPackageGlobalData, retstr, rc );
}

/****f* Window/DW_window_set_focus
 * NAME
 *   DW_window_set_focus
 * SYNOPSIS
 *   dw_window_set_focus( Win )
 * FUNCTION
 *   Makes the specified Win the active window.
 * ARGUMENTS
 *   - Win        - the window identifier returned from various functions that
 *                  return a window identifier
 * RESULT
 *   No return result.
 * SOURCE
 *
entry = dw_entryfield_new( '', 0 )
...
Call dw_window_set_focus entry
 ******
 */
rxfunc( DW_window_set_focus )
{
   HWND handle; /* local variable for arg[0] */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 1, 1 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&handle ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name, "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   dw_window_set_focus( handle );
   return RxReturnString( RxPackageGlobalData, retstr, "" );
}

/****f* Window/DW_window_set_font
 * NAME
 *   DW_window_set_font
 * SYNOPSIS
 *   dw_window_set_font( Win, Fontname )
 * FUNCTION
 *   Sets the text font of the window.
 * ARGUMENTS
 *   - Win        - the window identifier returned from various functions that
 *                  return a window identifier
 *   - Fontname   - the font name of the desired font
 * RESULT
 *   0 if successful, any other value is an error
 * SEE ALSO
 *   Fontnames, DW_window_get_font
 * NOTES
 *   Font names in RexxDW are platform dependent. See Fontnames for details on
 *   how to specify fonts on each platform.
 * SOURCE
 *
Parse Source os . prog
Select
   When os = 'OS2' | os = 'OS/2' Then  FIXEDFONT = "5.System VIO"
   When os = 'WIN32' Then FIXEDFONT = "10.Terminal"
   When !REXXDW.!GTK_MAJOR_VERSION > 1 Then FIXEDFONT = "monospace 10"
   Otherwise FIXEDFONT = "fixed"
End
text = dw_text_new( 'File to browse', 0 )
Call dw_window_set_font text, FIXEDFONT
 ******
 */
rxfunc( DW_window_set_font )
{
   int rc; /* function return variable */
   HWND handle; /* local variable for arg[0] */
   char * fontname; /* local variable for arg[1] */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 2, 2 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&handle ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   if ( RXVALIDSTRING(argv[1]) )
      fontname = RXSTRPTR(argv[1]);
   else
      fontname = "";
   rc = dw_window_set_font( handle, fontname );
   return RxReturnNumber( RxPackageGlobalData, retstr, rc );
}

/****f* Window/DW_window_get_font
 * NAME
 *   DW_window_get_font
 * SYNOPSIS
 *   font = dw_window_get_font( Win )
 * FUNCTION
 *   Returns the font details from the specified window.
 * ARGUMENTS
 *   - Win        - the window identifier returned from various functions that
 *                  return a window identifier
 * RESULT
 *   The value of the font details of the window in Dynamic Windows font format:
 *   fontsize.fontname [bold] [italic]
 * SEE ALSO
 *   Fontnames, DW_window_set_font
 * SOURCE
 *
entryfield = dw_entryfield_new( "Initial value", 0 )
...
Say 'The font for entryfield is' dw_window_get_font( entryfield )
 ******
 */
rxfunc( DW_window_get_font )
{
   char * rc; /* function return variable */
   char * tmp = "";
   HWND handle; /* local variable for arg[0] */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 1, 1 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&handle ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name, "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   rc = dw_window_get_font( handle );
   if ( rc )
   {
      tmp = strdup( rc );
      dw_free( rc );
   }
   return RxReturnString( RxPackageGlobalData, retstr, tmp );
}

/****f* Window/DW_window_set_icon
 * NAME
 *   DW_window_set_icon
 * SYNOPSIS
 *   dw_window_set_icon( Win, Icon )
 * FUNCTION
 *   Sets the text font of the window.
 * ARGUMENTS
 *   - Win        - the window identifier returned from various functions that
 *                  return a window identifier
 *   - Icon       - the icon identifier returned from DW_icon_load_from_file()
 *                  or DW_icon_load_from_data()
 * RESULT
 *   No return result.
 * SEE ALSO
 *   DW_icon_load_from_file, DW_icon_load_from_data
 * SOURCE
 *
rexxdwicon = dw_icon_load_from_file( '/home/mark/rexxdw' )
win = dw_window_new( mdi, 'Edit User Preferences', windowstyle )
Call dw_window_set_icon win, rexxdwicon
 ******
 */
rxfunc( DW_window_set_icon )
{
   HWND handle; /* local variable for arg[0] */
   HICN icon; /* local variable for arg[1] */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 2, 2 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&handle ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   if ( RxStrToULong( RxPackageGlobalData, &argv[1], (RX_ULONG *)&icon ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 2 );
      return(1);
   }
   dw_window_set_icon( handle, icon );
   return RxReturnString( RxPackageGlobalData, retstr, "" );
}

/****f* Window/DW_window_set_gravity
 * NAME
 *   DW_window_set_gravity
 * SYNOPSIS
 *   dw_window_set_gravity( Win, Horz, Vert )
 * FUNCTION
 *   Sets the horizontal and vertical positions from which dw_window_set_pos()
 *   values are relative to.
 * ARGUMENTS
 *   - Win  - the window identifier returned from DW_window_new()
 *   - Horz - Any value specified in WindowGravityValues for x position
 *   - Vert - Any value specified in WindowGravityValues for y position
 * RESULT
 *   No return result.
 * SEE ALSO
 *   WindowGravityValues, DW_window_set_pos, DW_window_set_pos_size
 * SOURCE
 *
win = dw_window_new( !REXXDW.!DW_DESKTOP, 'Window on desktop', style )
Call dw_window_set_gravity win, !REXXDW.!DW_GRAV_LEFT, !REXXDW.!DW_GRAV_LEFT
Call dw_window_set_pos win, 0, 0 -- window positioned in top left corner of screen
...
 ******
 */
rxfunc( DW_window_set_gravity )
{
   HWND handle; /* local variable for arg[0] */
   unsigned long horz; /* local variable for arg[1] */
   unsigned long vert; /* local variable for arg[2] */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 3, 3 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&handle ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   if ( RxStrToULong( RxPackageGlobalData, &argv[1], (RX_ULONG *)&horz ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 2 );
      return(1);
   }
   if ( RxStrToULong( RxPackageGlobalData, &argv[2], (RX_ULONG *)&vert ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 3 );
      return(1);
   }
   dw_window_set_gravity( handle, horz|DW_GRAV_OBSTACLES, vert|DW_GRAV_OBSTACLES );
   return RxReturnString( RxPackageGlobalData, retstr, "" );
}

/****f* Window/DW_window_set_pos
 * NAME
 *   DW_window_set_pos
 * SYNOPSIS
 *   dw_window_set_pos( Win, X, Y )
 * FUNCTION
 *   Sets the position of the top left corner of the window relative to the
 *   window's parent.
 * ARGUMENTS
 *   - Win        - the window identifier returned from various functions that
 *                  return a window identifier
 *   - X          - the x coordinate in pixels of the top left corner
 *   - Y          - the y coordinate in pixels of the top left corner
 * RESULT
 *   No return result.
 * SEE ALSO
 *   DW_window_set_pos_size, DW_window_set_size
 * NOTES
 *   Some window managers (at least under X11) will not respect a request for
 *   the position of a window.
 * SOURCE
 *
win = dw_window_new( !REXXDW.!DW_DESKTOP, 'Edit User Preferences', ,
   windowstyle )
Call dw_window_set_pos win, 50, 90
 ******
 */
rxfunc( DW_window_set_pos )
{
   HWND handle; /* local variable for arg[0] */
   long x; /* local variable for arg[1] */
   long y; /* local variable for arg[2] */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 3, 3 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&handle ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   if ( RxStrToLong( RxPackageGlobalData, &argv[1], (RX_LONG *)&x ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 2 );
      return(1);
   }
   if ( RxStrToLong( RxPackageGlobalData, &argv[2], (RX_LONG *)&y ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 3 );
      return(1);
   }
   dw_window_set_pos( handle, x, y );
   return RxReturnString( RxPackageGlobalData, retstr, "" );
}

/****f* Window/DW_window_set_size
 * NAME
 *   DW_window_set_size
 * SYNOPSIS
 *   dw_window_set_size( Win, Width, Height )
 * FUNCTION
 *   Sets the size of the window in pixels.
 * ARGUMENTS
 *   - Win        - the window identifier returned from various functions that
 *                  return a window identifier
 *   - Width      - the requested width of Window in pixels
 *   - Height     - the requested height of Window in pixels
 * RESULT
 *   No return result.
 * SEE ALSO
 *   DW_window_set_pos, DW_window_set_pos_size
 * NOTES
 *   Some window managers (at least under X11) will not respect a request for
 *   the size of a window.
 * SOURCE
 *
win = dw_window_new( !REXXDW.!DW_DESKTOP, 'Edit User Preferences', ,
   windowstyle )
Call dw_window_set_size win, 640, 480
 ******
 */
rxfunc( DW_window_set_size )
{
   HWND handle; /* local variable for arg[0] */
   unsigned long width; /* local variable for arg[1] */
   unsigned long height; /* local variable for arg[2] */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 3, 3 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&handle ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   if ( RxStrToULong( RxPackageGlobalData, &argv[1], (RX_ULONG *)&width ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 2 );
      return(1);
   }
   if ( RxStrToULong( RxPackageGlobalData, &argv[2], (RX_ULONG *)&height ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 3 );
      return(1);
   }
   dw_window_set_size( handle, width, height );
   return RxReturnString( RxPackageGlobalData, retstr, "" );
}

/****f* Window/DW_window_set_pos_size
 * NAME
 *   DW_window_set_pos_size
 * SYNOPSIS
 *   dw_window_set_pos_size( Win, X, Y, Width, Height )
 * FUNCTION
 *   Sets the position of the top left corner of the window relative to the
 *   window's parent and the size of the window in pixels.
 * ARGUMENTS
 *   - Win        - the window identifier returned from various functions that
 *                  return a window identifier
 *   - X          - the x coordinate in pixels of the top left corner
 *   - Y          - the y coordinate in pixels of the top left corner
 *   - Width      - the requested width of Window in pixels
 *   - Height     - the requested height of Window in pixels
 * RESULT
 *   No return result.
 * SEE ALSO
 *   DW_window_set_pos, DW_window_set_size
 * NOTES
 *   Some window managers (at least under X11) will not respect a request for
 *   the position of a window.
 * SOURCE
 *
win = dw_window_new( !REXXDW.!DW_DESKTOP, 'Edit User Preferences', ,
   windowstyle )
Call dw_window_set_pos_size win, 50, 90, 640, 480
 ******
 */
rxfunc( DW_window_set_pos_size )
{
   HWND handle; /* local variable for arg[0] */
   long x; /* local variable for arg[1] */
   long y; /* local variable for arg[2] */
   unsigned long width; /* local variable for arg[3] */
   unsigned long height; /* local variable for arg[4] */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 5, 5 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&handle ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   if ( RxStrToLong( RxPackageGlobalData, &argv[1], (RX_LONG *)&x ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 2 );
      return(1);
   }
   if ( RxStrToLong( RxPackageGlobalData, &argv[2], (RX_LONG *)&y ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 3 );
      return(1);
   }
   if ( RxStrToULong( RxPackageGlobalData, &argv[3], (RX_ULONG *)&width ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 4 );
      return(1);
   }
   if ( RxStrToULong( RxPackageGlobalData, &argv[4], (RX_ULONG *)&height ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 5 );
      return(1);
   }
   dw_window_set_pos_size( handle, x, y, width, height );
   return RxReturnString( RxPackageGlobalData, retstr, "" );
}

/****f* Window/DW_window_get_pos_size
 * NAME
 *   DW_window_get_pos_size
 * SYNOPSIS
 *   - X Y Width Height = dw_window_get_pos_size( Win )
 * FUNCTION
 *   Returns the X and Y coordinates of the specified window, and the Width
 *   and Height of the window.
 * ARGUMENTS
 *   - Win        - the window identifier returned from various functions that
 *                  return a window identifier
 * RESULT
 *   X, Y, Width and Height are returned as four words. PARSE VALUE is the easiest
 *   way to obtain all values.
 * SOURCE
 *
...
win = dw_window_new( !REXXDW.!DW_DESKTOP, 'Edit User Preferences', ,
   windowstyle )
...
Parse Value dw_window_get_pos_size( win ) With x y width height
 ******
 */
rxfunc( DW_window_get_pos_size )
{
   HWND handle; /* local variable for arg[0] */
   long x=0;
   long y=0;
   unsigned long width=0;
   unsigned long height=0;
   char buf1[200]; /* enough for four numbers */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 1, 1 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&handle ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   dw_window_get_pos_size( handle, &x, &y, &width, &height );
   sprintf( buf1, "%ld %ld %ld %ld", x, y, width, height );
   return RxReturnString( RxPackageGlobalData, retstr, buf1 );
}

/****f* Window/DW_window_get_preferred_size
 * NAME
 *   DW_window_get_preferred_size
 * SYNOPSIS
 *   - Width Height = dw_window_get_preferred_size( Win )
 * FUNCTION
 *   Returns the Width and Height the system thinks the window should be.
 * ARGUMENTS
 *   - Win        - the window identifier returned from various functions that
 *                  return a window identifier
 * RESULT
 *   Width and Height are returned as two words. PARSE VALUE is the easiest
 *   way to obtain all values.
 * SOURCE
 *
...
win = dw_window_new( !REXXDW.!DW_DESKTOP, 'Edit User Preferences', ,
   windowstyle )
...
Parse Value dw_window_get_preferred_size( win ) With width height
 ******
 */
rxfunc( DW_window_get_preferred_size )
{
   HWND handle; /* local variable for arg[0] */
   int width=0;
   int height=0;
   char buf1[100]; /* enough for two numbers */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 1, 1 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&handle ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   dw_window_get_preferred_size( handle, &width, &height );
   sprintf( buf1, "%d %d", width, height );
   return RxReturnString( RxPackageGlobalData, retstr, buf1 );
}

/****f* Window/DW_window_set_style
 * NAME
 *   DW_window_set_style
 * SYNOPSIS
 *   dw_window_set_style( Win, Style, Mask )
 * FUNCTION
 *   Changes the attributes of text within the specified window.
 * ARGUMENTS
 *   - Win        - the window identifier returned from various functions that
 *                  return a window identifier
 *   - Style      - Any value specified in WindowAttributeFlags ORed together
 *   - Mask       - Any value specified in WindowAttributeFlags ORed together
 * RESULT
 *   No return result.
 * SEE ALSO
 *   WindowAttributeFlags
 * SOURCE
 *
win = dw_text_new( 'Edit User Preferences' )
style = dw_or( !REXXDW.!DW_DT_RIGHT, !REXXDW.!DW_DT_VCENTER )
mask =  dw_or( !REXXDW.!DW_DT_RIGHT, !REXXDW.!DW_DT_VCENTER )
Call dw_window_set_style win, style, mask
 ******
 */
rxfunc( DW_window_set_style )
{
   HWND handle; /* local variable for arg[0] */
   unsigned long style; /* local variable for arg[1] */
   unsigned long mask; /* local variable for arg[2] */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 3, 3 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&handle ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   if ( RxStrToULong( RxPackageGlobalData, &argv[1], (RX_ULONG *)&style ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 2 );
      return(1);
   }
   if ( RxStrToULong( RxPackageGlobalData, &argv[2], (RX_ULONG *)&mask ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 3 );
      return(1);
   }
   dw_window_set_style( handle, style, mask );
   return RxReturnString( RxPackageGlobalData, retstr, "" );
}

/****f* Window/DW_window_set_tooltip
 * NAME
 *   DW_window_set_tooltip
 * SYNOPSIS
 *   dw_window_set_tooltip( Win[, Text] )
 * FUNCTION
 *   Sets the floating help text for the specified window.
 * ARGUMENTS
 *   - Win        - the window identifier returned from various functions that
 *                  return a window identifier
 *   - Text       - the value of the floating help text for the window or if omitted
 *                  disables the floating help text
 * RESULT
 *   No return result.
 * SOURCE
 *
entryfield = dw_entryfield_new( "Initial value", 0 )
...
Call dw_window_set_tooltip entryfield, 'Rexx is king'
 ******
 */
rxfunc( DW_window_set_tooltip )
{
   HWND handle; /* local variable for arg[0] */
   char * text; /* local variable for arg[1] */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 1, 2 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&handle ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name, "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   if ( argc == 2 && RXVALIDSTRING(argv[1]) )
      text = RXSTRPTR(argv[1]);
   else
      text = NULL;
   dw_window_set_tooltip( handle, text );
   return RxReturnString( RxPackageGlobalData, retstr, "" );
}

/****f* Window/DW_window_set_text
 * NAME
 *   DW_window_set_text
 * SYNOPSIS
 *   dw_window_set_text( Win, Text )
 * FUNCTION
 *   Sets the text for the specified window. If the window is a top level
 *   window and it has a title bar, the title bar text is changed.
 * ARGUMENTS
 *   - Win        - the window identifier returned from various functions that
 *                  return a window identifier
 *   - Text       - the value of the text attribute the window is to have
 * RESULT
 *   No return result.
 * SEE ALSO
 *   DW_window_get_text
 * NOTES
 *   Not all window types have a meaningful text attribute. Those that do are:
 *      Text                 DW_text_new
 *      StatusText           DW_statustext_new
 *      Combobox             DW_combobox_new()
 *      EntryField           DW_entryfield_new()
 *      EntryFieldPassword   DW_entryfield_password_new()
 * SOURCE
 *
entryfield = dw_entryfield_new( "Initial value", 0 )
...
Call dw_window_set_text entryfield, 'Rexx is king'
 ******
 */
rxfunc( DW_window_set_text )
{
   HWND handle; /* local variable for arg[0] */
   char * text; /* local variable for arg[1] */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 2, 2 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&handle ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   if ( RXVALIDSTRING(argv[1]) )
      text = RXSTRPTR(argv[1]);
   else
      text = "";
   dw_window_set_text( handle, text );
   return RxReturnString( RxPackageGlobalData, retstr, "" );
}

/****f* Window/DW_window_get_text
 * NAME
 *   DW_window_get_text
 * SYNOPSIS
 *   text = dw_window_get_text( Win )
 * FUNCTION
 *   Returns the text attribute from the specified window.
 * ARGUMENTS
 *   - Win        - the window identifier returned from various functions that
 *                  return a window identifier
 * RESULT
 *   The value of the text attribute of the window.
 * SEE ALSO
 *   DW_window_set_text
 * NOTES
 *   Not all windows have a meaningful text attribute. Those that do are:
 *      Text                 DW_text_new
 *      StatusText           DW_statustext_new
 *      Combobox             DW_combobox_new()
 *      EntryField           DW_entryfield_new()
 *      EntryFieldPassword   DW_entryfield_password_new()
 * SOURCE
 *
entryfield = dw_entryfield_new( "Initial value", 0 )
...
Say 'The text is now' dw_window_get_text( entryfield )
 ******
 */
rxfunc( DW_window_get_text )
{
   char * rc; /* function return variable */
   char * tmp = "";
   HWND handle; /* local variable for arg[0] */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 1, 1 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&handle ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   rc = dw_window_get_text( handle );
   if ( rc )
   {
      tmp = strdup( rc );
      dw_free( rc );
   }
   return RxReturnString( RxPackageGlobalData, retstr, tmp );
}

/****f* Window/DW_window_set_data
 * NAME
 *   DW_window_set_data
 * SYNOPSIS
 *   dw_window_set_data( Win, Name, Value )
 * FUNCTION
 *   Sets the user variable; Name to the specified Value for the specified
 *   window.
 * ARGUMENTS
 *   - Win        - the window identifier returned from various functions that
 *                  return a window identifier
 *   - Name       - the user variable name
 *   - Value      - the value that Name is to be set to
 * RESULT
 *   No return result.
 * SEE ALSO
 *   DW_window_get_data
 * NOTES
 *   Don't use variable names that start with "_dw"; they are used internally
 *   by Dynamic Windows.
 * SOURCE
 *
entryfield = dw_entryfield_new( 'Initial value', 0 )
Call dw_window_set_data entryfield, 'MyVar', 'my useful data'
...
Say 'The value of "MyVar" is:' dw_window_get_data( entryfield, 'MyVar' )
 ******
 */

rxfunc( DW_window_set_data )
{
   HWND window; /* local variable for arg[0] */
   char * dataname; /* local variable for arg[1] */
   char * data; /* local variable for arg[2] */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 3, 3 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&window ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   if ( RXVALIDSTRING(argv[1]) )
      dataname = RXSTRPTR(argv[1]);
   else
      dataname = "";
   /* TODO - memory leak !! */
   if ( RXVALIDSTRING(argv[2]) )
      data = strdup( RXSTRPTR(argv[2]) );
   else
      data = strdup( "" );
   dw_window_set_data( window, dataname, (void *)data );
   return RxReturnString( RxPackageGlobalData, retstr, "" );
}

/****f* Window/DW_window_get_data
 * NAME
 *   DW_window_get_data
 * SYNOPSIS
 *   data = dw_window_get_data( Win, Name )
 * FUNCTION
 *   Returns the user variable data for the window with the given Name
 *   created by a call to DW_window_set_data();
 * ARGUMENTS
 *   - Win        - the window identifier returned from various functions that
 *                  return a window identifier
 *   - Name       - the user variable name
 * RESULT
 *   The value of the user variable Name for the window
 * SEE ALSO
 *   DW_window_set_data
 * NOTES
 *   Don't use variable names that start with "_dw"; they are used internally
 *   by Dynamic Windows.
 * SOURCE
 *
entryfield = dw_entryfield_new( 'Initial value', 0 )
Call dw_window_set_data entryfield, 'MyVar', 'my useful data'
...
Say 'The value of "MyVar" is:' dw_window_get_data( entryfield, 'MyVar' )
 ******
 */
rxfunc( DW_window_get_data )
{
   char * rc; /* function return variable */
   HWND window; /* local variable for arg[0] */
   char * dataname; /* local variable for arg[1] */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 2, 2 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&window ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   if ( RXVALIDSTRING(argv[1]) )
      dataname = RXSTRPTR(argv[1]);
   else
      dataname = "";
   rc = (char *)dw_window_get_data( window, dataname );
   return RxReturnString( RxPackageGlobalData, retstr, rc );
}

/****h* Widgets/Miscellaneous
 * DESCRIPTION
 *   There always has to be things that can't be logically grouped together.
 *   These are those.
 ******
 */

/****f* Miscellaneous/DW_messagebox
 * NAME
 *   DW_messagebox
 * SYNOPSIS
 *   action = dw_messagebox( Title, Flags, Text )
 * FUNCTION
 *   Displays a modal messagebox in the middle of the desktop.
 * ARGUMENTS
 *   - Title      - the window title for the messagebox
 *   - Flags      - indicates the buttons to be incorporated into the messagebox
 *                  and the icon to be displayed
 *   - Text       - the text inside the messagebox
 * RESULT
 *   Depending on the button clicked, one of the values defined in MessageboxResults
 * NOTES
 *   The maximum length of Text is 1023 characters.
 * SEE ALSO
 *   MessageboxFlags, MessageboxResults
 * SOURCE
 *
...
action = dw_messagebox( 'Are you sure?', dw_or( !REXXDW.!DW_MB_YESNO, ,
   !REXXDW.!DW_MB_QUESTION ), 'Delete selected files' )
If action = !REXXDW.!DW_MB_RESULT_YES Then Call DeleteFiles
 ******
 */
rxfunc( DW_messagebox )
{
   int rc; /* function return variable */
   char * title; /* local variable for arg[0] */
   int flags; /* local variable for arg[1] */
   char * format; /* local variable for arg[2] */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 3, 3 ) )
      return( 1 );
   if ( RXVALIDSTRING(argv[0]) )
      title = RXSTRPTR(argv[0]);
   else
      title = "";
   if ( RxStrToInt( RxPackageGlobalData, &argv[1], (RX_INT *)&flags ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name, "*ERROR* Invalid number for argument %d.", 2 );
      return(1);
   }
   if ( RXVALIDSTRING(argv[2]) )
      format = RXSTRPTR(argv[2]);
   else
      format = "";
   rc = dw_messagebox( title, flags, "%s", format );
   return RxReturnNumber( RxPackageGlobalData, retstr, rc );
}

/****f* Miscellaneous/DW_bitmap_new
 * NAME
 *   DW_bitmap_new
 * SYNOPSIS
 *   win = dw_bitmap_new( Id )
 * FUNCTION
 *   Creates a new bitmap window used as the container for a bitmap image.
 * ARGUMENTS
 *   - Id        - a numeric identifier used to identify the window
 * RESULT
 *   A window identifier.
 * SEE ALSO
 *   DW_window_set_bitmap, DW_bitmap_new_from_file, DW_bitmap_new_from_data
 * SOURCE
 *
...
bitmap = dw_bitmap_new( 100 )
Call dw_box_pack_start win, bitmap, 10, 10, !REXXDW.!DW_DONT_EXPAND_HORZ, ,
   !REXXDW.!DW_DONT_EXPAND_VERT, 0
Call dw_window_set_bitmap bitmap, 0, 'mypicture'
 ******
 */
rxfunc( DW_bitmap_new )
{
   HWND rc; /* function return variable */
   unsigned long id; /* local variable for arg[0] */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 1, 1 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&id ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   rc = dw_bitmap_new( id );
   return RxReturnUnsignedNumber( RxPackageGlobalData, retstr, (ULONG)rc );
}

/****f* Miscellaneous/DW_bitmap_new_from_file
 * NAME
 *   DW_bitmap_new_from_file
 * SYNOPSIS
 *   win = dw_bitmap_new_from_file( Id, Filename )
 * FUNCTION
 *   Creates a new bitmap window including a bitmap image.
 * ARGUMENTS
 *   - Id         - a numeric identifier used to identify the window
 *   - Filename   - the filename of the bitmap image
 * RESULT
 *   A window identifier.
 * SEE ALSO
 *   DW_window_set_bitmap, DW_bitmap_new, DW_bitmap_new_from_data
 * SOURCE
 *
...
bitmap = dw_bitmap_new_from_file( 100, 'mypicture' )
Call dw_box_pack_start win, bitmap, 10, 10, !REXXDW.!DW_DONT_EXPAND_HORZ, ,
   !REXXDW.!DW_DONT_EXPAND_VERT, 0
 ******
 */
rxfunc( DW_bitmap_new_from_file )
{
   HWND rc; /* function return variable */
   unsigned long id; /* local variable for arg[0] */
   char * filename; /* local variable for arg[1] */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 2, 2 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&id ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   if ( RXVALIDSTRING(argv[1]) )
      filename = RXSTRPTR(argv[1]);
   else
      filename = "";
   rc = dw_bitmap_new( id );
   dw_window_set_bitmap( rc, 0, filename );
   return RxReturnUnsignedNumber( RxPackageGlobalData, retstr, (ULONG)rc );
}

/****f* Miscellaneous/DW_bitmap_new_from_data
 * NAME
 *   DW_bitmap_new_from_data
 * SYNOPSIS
 *   win = dw_bitmap_new_from_data( Id, Data )
 * FUNCTION
 *   Creates a new bitmap window including a bitmap image from a memory image.
 * ARGUMENTS
 *   - Id   - a numeric identifier used to identify the window
 *   - Data - the contents of the image.
 * RESULT
 *   A window identifier.
 * SEE ALSO
 *   DW_bitmap_new_from_file
 * SOURCE
 *
...
ico = "0000A6D7007E"
...
bitmap = dw_bitmap_new_from_data( 100, x2c( ico ) )
Call dw_box_pack_start win, bitmap, 10, 10, !REXXDW.!DW_DONT_EXPAND_HORZ, ,
   !REXXDW.!DW_DONT_EXPAND_VERT, 0
 ******
 */
rxfunc( DW_bitmap_new_from_data )
{
   HWND rc; /* function return variable */
   unsigned long id; /* local variable for arg[0] */
   char * data; /* local variable for arg[1] */
   int len;
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 2, 2 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&id ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   if ( RXVALIDSTRING(argv[1]) )
   {
      data = RXSTRPTR(argv[1]);
      len = RXSTRLEN(argv[1]);
   }
   else
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Supplied image data is empty." );
      return(1);
   }
   rc = dw_bitmap_new( id );
   dw_window_set_bitmap_from_data( rc, 0, data, len );
   return RxReturnUnsignedNumber( RxPackageGlobalData, retstr, (ULONG)rc );
}

/****f* Miscellaneous/DW_icon_load_from_file
 * NAME
 *   DW_icon_load_from_file
 * SYNOPSIS
 *   win = dw_icon_load_from_file( Filename )
 * FUNCTION
 *   Creates a new icon identifier for use in other functions from the file
 *   specified in Filename.
 *   Supported icon types are Windows icons (.ico), OS/2 icons(.ico) and
 *   pixmaps(.xpm) on GTK. GTK 2.x also supports Windows icons (.ico).
 * ARGUMENTS
 *   - Filename   - the filename of the icon file
 * RESULT
 *   An icon identifier.
 * NOTES
 *   The Filename can contain a path component. For portability, do not
 *   specify the file's extension; let dwindows work this out.
 * SOURCE
 *
normalfileicon = dw_icon_load_from_file( '/home/mark/normalfile' )
 ******
 */
rxfunc( DW_icon_load_from_file )
{
   HICN rc; /* function return variable */
   char * filename; /* local variable for arg[0] */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 1, 1 ) )
      return( 1 );
   if ( RXVALIDSTRING(argv[0]) )
      filename = RXSTRPTR(argv[0]);
   else
      filename = "";
   rc = dw_icon_load_from_file( filename );
   return RxReturnUnsignedNumber( RxPackageGlobalData, retstr, (unsigned long)rc );
}

/****f* Miscellaneous/DW_icon_load_from_data
 * NAME
 *   DW_icon_load_from_data
 * SYNOPSIS
 *   win = dw_icon_load_from_data( Data )
 * FUNCTION
 *   Creates a new icon identifier for use in other functions from the file
 *   specified in Filename.
 *   Supported icon types are Windows icons (.ico), OS/2 icons(.ico) and
 *   pixmaps(.xpm) on GTK. GTK 2.x also supports Windows icons (.ico).
 * ARGUMENTS
 *   - Data - the contents of the image.
 * RESULT
 *   An icon identifier.
 * NOTES
 *   The Data argument is to contain an exact image of the icon.
 * SOURCE
 *
...
ico = "0000A6D7007E"
...
normalfileicon = dw_icon_load_from_data( xc2( ico ) )
 ******
 */
rxfunc( DW_icon_load_from_data )
{
   HICN rc; /* function return variable */
   char * data; /* local variable for arg[0] */
   int len;
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 1, 1 ) )
      return( 1 );
   if ( RXVALIDSTRING(argv[0]) )
   {
      data = RXSTRPTR(argv[0]);
      len = RXSTRLEN(argv[0]);
   }
   else
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Supplied image data is empty." );
      return(1);
   }
   rc = dw_icon_load_from_data( data, len );
   return RxReturnUnsignedNumber( RxPackageGlobalData, retstr, (unsigned long)rc );
}

/****f* Miscellaneous/DW_taskbar_insert
 * NAME
 *   DW_taskbar_insert
 * SYNOPSIS
 *   win = dw_taskbar_insert( Toplevel, Icon[, Text[, Callback[, UserData]]] )
 * FUNCTION
 *   Inserts an icon into the window manager's taskbar.
 * ARGUMENTS
 *   - Toplevel - the window handle for the application's toplevel window
 *   - Icon     - the icon handle returned from calls like DW_icon_from_data to be displayed on the taskbar
 *   - Text     - (optional) the text to be displayed when the mouse hovers over the icon
 *   - Function - (optional) the Rexx procedure that is called when the a mouse button is pressed on the taskbar icon
 *   - UserData - (optional) user supplied data to pass to the Function
 * RESULT
 *   Returns a window identifier for the taskbar icon
 * NOTES
 *   If you want a context menu displayed when a button is clicked on the taskbar icon, you need
 *   to specify Function, that Function must conform to ButtonPressEventCallback, and the callback
 *   must display a popup menu
 * SEE ALSO
 *   DW_taskbar_delete, ButtonPressEventCallback
 * SOURCE
 *
...
ico = "0000A6D7007E"
...
icon = dw_icon_load_from_data( xc2( ico ) )
...
win = dw_taskbar_insert( toplevel, icon, ,'contextmenu_cb', 'fred', 'mary' )
...
contextmenu_cb:
Parse Arg win, x, y, button
Say 'Button:' button 'pressed at:' x'/'y 'in Window:' win
If button \= 1 Then Return 0
menu = dw_menu_new( 0 )
menuitem = dw_menu_append_item( menu, '~Show Window', 1011, 0, ,
   !REXXDW.!DW_MENU_END, !REXXDW.!DW_MENU_NOT_CHECKABLE, 0 )
...
Call dw_menu_popup menu, win, x, y
Return 1
 ******
 */
rxfunc( DW_taskbar_insert )
{
   HWND rc; /* function return variable */
   HWND win;    /* local variable for arg[0] */
   HICN icon;   /* local variable for arg[1] */
   char *text=NULL; /* local variable for arg[2] */
   char *cb=NULL; /* local variable for arg[3] */
   unsigned int signum; /* local variable for arg[1] */
   REXXDWUserData * rexx_data;
   char * signame;
   void * sigfunc;
   int i;
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 2, 0 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&win ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name, "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   if ( RxStrToULong( RxPackageGlobalData, &argv[1], (RX_ULONG *)&icon ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name, "*ERROR* Invalid number for argument %d.", 2 );
      return(1);
   }
   if ( argc > 2 )
   {
      if ( RXVALIDSTRING(argv[2]) )
      {
         text = RXSTRPTR(argv[2]);
      }
   }
   if ( argc > 3 )
   {
      if ( RXVALIDSTRING(argv[3]) )
      {
         cb = RXSTRPTR(argv[3]);
      }
   }
   rc = dw_render_new( rand() );
   if ( rc )
   {
      /* pack the render window into the toplevel, making it 1 pixels in size */
      dw_box_pack_start( win, rc, 1, 1, FALSE, FALSE, 0 );
      dw_taskbar_insert( rc, icon, text );
      /* if we have a callback, connect it up */
      if ( cb )
      {
         /*
          * Allocate memory for a RexxDW data struct
          */
         rexx_data = (REXXDWUserData *)calloc( 1, sizeof(REXXDWUserData) );
         if ( rexx_data == NULL )
         {
            RxDisplayError( RxPackageGlobalData, name, "*ERROR* Out of memory." );
            return(1);
         }
         /*
          * Set the PackageGlobalData so our callbacks can reference it
          */
         rexx_data->RxPackageGlobalData = RxPackageGlobalData;
         /*
          * Allocate space for the rexx procedure name
          * Make sure we have a valid argument...
          */
         rexx_data->func = (char *)malloc( RXSTRLEN(argv[3]) + 1 );
         if ( rexx_data->func == NULL )
         {
            free( rexx_data );
            RxDisplayError( RxPackageGlobalData, name, "*ERROR* Out of memory." );
            return(1);
         }
         strcpy( rexx_data->func, RXSTRPTR(argv[3]) );
         if ( argc > 4 )
         {
            rexx_data->argc = argc - 4;
            rexx_data->argv = (PRXSTRING)malloc( rexx_data->argc * sizeof(RXSTRING) );
            if ( rexx_data->argv == NULL )
            {
               free( rexx_data );
               free( rexx_data->func );
               RxDisplayError( RxPackageGlobalData, name, "*ERROR* Out of memory." );
               return(1);
            }
            for ( i = 0; i < rexx_data->argc; i++ )
            {
               rexx_data->argv[i].strlength = RXSTRLEN(argv[i+4]);
               rexx_data->argv[i].strptr = (char *)malloc( 1+RXSTRLEN(argv[i+4]) );
               if ( rexx_data->argv[i].strptr == NULL )
               {
                  free( rexx_data );
                  free( rexx_data->func );
                  RxDisplayError( RxPackageGlobalData, name, "*ERROR* Out of memory." );
                  return(1);
               }
               strcpy( rexx_data->argv[i].strptr, RXSTRPTR(argv[i+4]) );
               DEBUGDUMP(fprintf(stderr,"%s-%d:    Arg: %d \"%s\"\n",__FILE__,__LINE__,i+1,rexx_data->argv[i].strptr);)
            }
         }
         else
         {
            rexx_data->argc = 0;
            rexx_data->argv = NULL;
         }
         DEBUGDUMP(fprintf(stderr,"%s-%d: In DW_taskbar_insert() Func: \"%s\"\n",__FILE__,__LINE__,rexx_data->func);)
      }
      /*
       * Lookup the signal name from the supplied signal number, and then
       * link the appropriate callback type in dw_signal_connect()
       */
      signame = REXXDWSignalTranslate[REXXDW_BUTTON_PRESS_EVENT_CB].signame;
      sigfunc = REXXDWSignalTranslate[REXXDW_BUTTON_PRESS_EVENT_CB].sigfunc;
      dw_signal_connect( rc, signame, DW_SIGNAL_FUNC(sigfunc), (void *)rexx_data );
   }
   return RxReturnUnsignedNumber( RxPackageGlobalData, retstr, (unsigned long)rc );
}

/****f* Miscellaneous/DW_taskbar_delete
 * NAME
 *   DW_taskbar_delete
 * SYNOPSIS
 *   dw_taskbar_delete( Win, Icon )
 * FUNCTION
 *   Deletes an icon into the window manager's taskbar.
 * ARGUMENTS
 *   - Win  - the window handle returned from a call to DW_taskbar_insert
 *   - Icon - the icon handle used in the call to DW_taskbar_insert
 * RESULT
 *   None
 * SEE ALSO
 *   DW_taskbar_insert
 * SOURCE
 *
...
win = dw_taskbar_insert( toplevel, icon )
...
Call dw_taskbar_delete win, icon
 ******
 */
rxfunc( DW_taskbar_delete )
{
   HWND win;    /* local variable for arg[0] */
   HICN icon;   /* local variable for arg[1] */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 2, 2 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&win ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name, "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   if ( RxStrToULong( RxPackageGlobalData, &argv[1], (RX_ULONG *)&icon ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name, "*ERROR* Invalid number for argument %d.", 2 );
      return(1);
   }
   dw_taskbar_delete( win, icon );
   return RxReturnString( RxPackageGlobalData, retstr, "" );
}

/****h* Functions/ProcessControl
 * DESCRIPTION
 *   These functions control the execution of the current process or instigate
 *   other processes.
 ******
 */

/****f* ProcessControl/DW_init
 * NAME
 *   DW_init
 * SYNOPSIS
 *   rc = dw_init( )
 * FUNCTION
 *   Initialises the Rexx/DW interface. This MUST be called before
 *   any other Rexx/DW function.
 * ARGUMENTS
 *   None
 * RESULT
 *   0 if all OK, non-zero if an error occurred
 * SOURCE
 *
Call dw_init
 ******
 */
rxfunc( DW_init )
{
   int rc; /* function return variable */
   int x_argc;
   int i;
   char **x_argv;
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 0, 10 ) )
      return( 1 );
   x_argv = (char **)malloc( sizeof(char *) * (2+argc) );
   if ( x_argv == NULL )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Out of memory." );
      return(1);
   }
   x_argc = 1+argc;
   x_argv[0] = "rexxdw";
   for ( i = 0; i < argc; i++ )
   {
      x_argv[i+1] = strdup( RXSTRPTR(argv[i]) );
   }
   rc = dw_init( TRUE, x_argc, x_argv );
   free( x_argv );
   return RxReturnNumber( RxPackageGlobalData, retstr, rc );
}

/****f* ProcessControl/DW_main
 * NAME
 *   DW_main
 * SYNOPSIS
 *   dw_main( )
 * FUNCTION
 *   This is the main event dispatcher function for Rexx interpreters which
 *   support callbacks.
 * ARGUMENTS
 *   None
 * RESULT
 *   No return result.
 * NOTES
 *   This function can only be called if your Rexx interpreter supports the
 *   RexxCallback() API. This can be determined by the value of the variable
 *   !REXXDW.!HAVE_REXXCALLBACK being 1.
 * SEE ALSO
 *   DW_main_iteration
 * SOURCE
 *
...
If !REXXDW.!HAVE_REXXCALLBACK Then Call dw_main()
Else
   Do Forever
      cmd = dw_main_iteration()
      If cmd \= '' Then Interpret 'Call' cmd
   End
Return 0
 ******
 */
rxfunc( DW_main )
{
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();
   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 0, 0 ) )
      return( 1 );
   dw_main( );
   return RxReturnString( RxPackageGlobalData, retstr, "" );
}

/****f* ProcessControl/DW_main_iteration
 * NAME
 *   DW_main_iteration
 * SYNOPSIS
 *   cmd = dw_main_iteration( )
 * FUNCTION
 *   This is the main event dispatcher function for Rexx interpreters which
 *   DO NOT support callbacks.
 * ARGUMENTS
 *   None
 * RESULT
 *   The Rexx function called back, follwoed by parameters
 * NOTES
 *   This function is supplied for interpreters that do not support the
 *   RexxCallback() API. This can be determined by the value of the variable
 *   !REXXDW.!HAVE_REXXCALLBACK being 1.
 *   You have to be very careful with any optional data you pass to a callback
 *   function as the data could be "interpreted" with syntax errors.
 * SEE ALSO
 *   DW_main
 * SOURCE
 *
...
If !REXXDW.!HAVE_REXXCALLBACK Then Call dw_main()
Else
   Do Forever
      cmd = dw_main_iteration()
      If cmd \= '' Then Interpret 'Call' cmd
   End
Return 0
 ******
 */
rxfunc( DW_main_iteration )
{
   char *qstr;
   REXXDWCallbackData *last_first;
   int data_size = 0;
   int i;
   REXXDWDATA *RexxDWData;
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );

   RexxDWData=(REXXDWDATA *)RxPackageGlobalData->RXPACKAGE_tsd;

   if ( my_checkparam( RxPackageGlobalData, name, argc, 0, 0 ) )
      return( 1 );

   dw_main_iteration( );
   strcpy( retstr->strptr, "" );
   retstr->strlength = 0;
   if ( !HAVE_REXXCALLBACK )
   {
      /*
       * Now determine if there is any callback data to return...
       */
      if ( RexxDWData->FirstCallback != NULL )
      {
         /*
          * All functions have the function name. Add 1 for NUL
          */
         data_size = 1 + strlen( RexxDWData->FirstCallback->userdata->func );
         /*
          * Determine the size of any user-supplied data
          */
         for ( i = 0; i < RexxDWData->FirstCallback->userdata->argc; i++ )
         {
            /*
             * Extra 3 bytes for comma, and quotes
             */
            data_size += 3 + RXSTRLEN( RexxDWData->FirstCallback->userdata->argv[i] );
         }
         /*
          * Determine the size of the callback specific data.
          */
         switch( RexxDWData->FirstCallback->callback_type )
         {
            case REXXDW_CONFIGURE_EVENT_CB:
               data_size += (3 * 16); /* window, width, height */
               break;
            case REXXDW_KEY_PRESS_EVENT_CB:
               data_size += (4 * 16); /* window, key, virtual_key, state */
               break;
            case REXXDW_BUTTON_PRESS_EVENT_CB:
            case REXXDW_BUTTON_RELEASE_EVENT_CB:
               data_size += (4 * 16); /* window, x, y, button */
               break;
            case REXXDW_MOTION_NOTIFY_EVENT_CB:
               data_size += (4 * 16); /* window, x, y, state */
               break;
            case REXXDW_DELETE_EVENT_CB:
               data_size += (1 * 16); /* window */
               break;
            case REXXDW_EXPOSE_EVENT_CB:
               data_size += (5 * 16); /* window, x, y, width, height */
               break;
            case REXXDW_CLICKED_EVENT_CB:
               data_size += (1 * 16); /* window */
               break;
            case REXXDW_ITEM_ENTER_EVENT_CB:
               data_size += (1 * 16) + 2; /* window */
               if ( RexxDWData->FirstCallback->event.ItemEnterCallbackData.text )
                  data_size += strlen( RexxDWData->FirstCallback->event.ItemEnterCallbackData.text );
               break;
            case REXXDW_ITEM_CONTEXT_EVENT_CB:
               data_size += (3 * 16); /* window, x, y */
               if ( RexxDWData->FirstCallback->event.ItemContextCallbackData.text )
                  data_size += strlen( RexxDWData->FirstCallback->event.ItemContextCallbackData.text );
               if ( RexxDWData->FirstCallback->event.ItemContextCallbackData.itemdata )
                  data_size += strlen( RexxDWData->FirstCallback->event.ItemContextCallbackData.itemdata );
               break;
            case REXXDW_LIST_SELECT_EVENT_CB:
               data_size += (2 * 16); /* window, item */
               break;
            case REXXDW_ITEM_SELECT_EVENT_CB:
               data_size += (2 * 16) + 2; /* window, item */
               if ( RexxDWData->FirstCallback->event.ItemSelectCallbackData.text )
                  data_size += strlen( RexxDWData->FirstCallback->event.ItemSelectCallbackData.text );
               if ( RexxDWData->FirstCallback->event.ItemSelectCallbackData.itemdata )
                  data_size += strlen( RexxDWData->FirstCallback->event.ItemSelectCallbackData.itemdata );
               break;
            case REXXDW_SET_FOCUS_EVENT_CB:
               data_size += (1 * 16); /* window */
               break;
            case REXXDW_VALUE_CHANGED_EVENT_CB:
               data_size += (2 * 16); /* window, val */
               break;
            case REXXDW_SWITCH_PAGE_EVENT_CB:
               data_size += (2 * 16); /* window, page */
               break;
            case REXXDW_COLUMN_CLICK_EVENT_CB:
               data_size += (2 * 16); /* window, column_num */
               break;
            default: /* timer callback */
               break;
         }
         /*
          * Allocate the memory now
          */
         retstr->strptr = (char *)RexxAllocateMemory( data_size );
         if ( retstr->strptr == NULL )
         {
         }
         switch( RexxDWData->FirstCallback->callback_type )
         {
            case REXXDW_CONFIGURE_EVENT_CB:
               retstr->strlength = sprintf( retstr->strptr, "%s %lu,%d,%d",
                                            RexxDWData->FirstCallback->userdata->func,
                                            (ULONG)RexxDWData->FirstCallback->window,
                                            RexxDWData->FirstCallback->event.ConfigureCallbackData.width,
                                            RexxDWData->FirstCallback->event.ConfigureCallbackData.height );
               break;
            case REXXDW_KEY_PRESS_EVENT_CB:
               retstr->strlength = sprintf( retstr->strptr, "%s %lu,\"%X\",%d,%d",
                                            RexxDWData->FirstCallback->userdata->func,
                                            (ULONG)RexxDWData->FirstCallback->window,
                                            RexxDWData->FirstCallback->event.KeyPressCallbackData.key,
                                            RexxDWData->FirstCallback->event.KeyPressCallbackData.virtual_key,
                                            RexxDWData->FirstCallback->event.KeyPressCallbackData.state );
               break;
            case REXXDW_BUTTON_PRESS_EVENT_CB:
               retstr->strlength = sprintf( retstr->strptr, "%s %lu,%d,%d,%d",
                                            RexxDWData->FirstCallback->userdata->func,
                                            (ULONG)RexxDWData->FirstCallback->window,
                                            RexxDWData->FirstCallback->event.ButtonPressCallbackData.x,
                                            RexxDWData->FirstCallback->event.ButtonPressCallbackData.y,
                                            RexxDWData->FirstCallback->event.ButtonPressCallbackData.button );
            case REXXDW_BUTTON_RELEASE_EVENT_CB:
               retstr->strlength = sprintf( retstr->strptr, "%s %lu,%d,%d,%d",
                                            RexxDWData->FirstCallback->userdata->func,
                                            (ULONG)RexxDWData->FirstCallback->window,
                                            RexxDWData->FirstCallback->event.ButtonReleaseCallbackData.x,
                                            RexxDWData->FirstCallback->event.ButtonReleaseCallbackData.y,
                                            RexxDWData->FirstCallback->event.ButtonReleaseCallbackData.button );
               break;
            case REXXDW_MOTION_NOTIFY_EVENT_CB:
               retstr->strlength = sprintf( retstr->strptr, "%s %lu,%d,%d,%d",
                                            RexxDWData->FirstCallback->userdata->func,
                                            (ULONG)RexxDWData->FirstCallback->window,
                                            RexxDWData->FirstCallback->event.MotionNotifyCallbackData.x,
                                            RexxDWData->FirstCallback->event.MotionNotifyCallbackData.y,
                                            RexxDWData->FirstCallback->event.MotionNotifyCallbackData.state );
               break;
            case REXXDW_DELETE_EVENT_CB:
               retstr->strlength = sprintf( retstr->strptr, "%s %lu",
                                            RexxDWData->FirstCallback->userdata->func,
                                            (ULONG)RexxDWData->FirstCallback->window );
               break;
            case REXXDW_EXPOSE_EVENT_CB:
               retstr->strlength = sprintf( retstr->strptr, "%s %lu,%d,%d,%d,%d",
                                            RexxDWData->FirstCallback->userdata->func,
                                            (ULONG)RexxDWData->FirstCallback->window,
                                            RexxDWData->FirstCallback->event.ExposeCallbackData.x,
                                            RexxDWData->FirstCallback->event.ExposeCallbackData.y,
                                            RexxDWData->FirstCallback->event.ExposeCallbackData.width,
                                            RexxDWData->FirstCallback->event.ExposeCallbackData.height );
               break;
            case REXXDW_CLICKED_EVENT_CB:
               retstr->strlength = sprintf( retstr->strptr, "%s %lu",
                                            RexxDWData->FirstCallback->userdata->func,
                                            (ULONG)RexxDWData->FirstCallback->window );
               break;
            case REXXDW_ITEM_ENTER_EVENT_CB:
               retstr->strlength = sprintf( retstr->strptr, "%s %lu,\"%s\"",
                                            RexxDWData->FirstCallback->userdata->func,
                                            (ULONG)RexxDWData->FirstCallback->window,
                                            (RexxDWData->FirstCallback->event.ItemEnterCallbackData.text) ? RexxDWData->FirstCallback->event.ItemEnterCallbackData.text : "" );
               break;
            case REXXDW_ITEM_CONTEXT_EVENT_CB:
               retstr->strlength = sprintf( retstr->strptr, "%s %lu,\"%s\",%d,%d,\"%s\"",
                                            RexxDWData->FirstCallback->userdata->func,
                                            (ULONG)RexxDWData->FirstCallback->window,
                                            (RexxDWData->FirstCallback->event.ItemContextCallbackData.text) ? RexxDWData->FirstCallback->event.ItemContextCallbackData.text : "",
                                            RexxDWData->FirstCallback->event.ItemContextCallbackData.x,
                                            RexxDWData->FirstCallback->event.ItemContextCallbackData.y,
                                            (RexxDWData->FirstCallback->event.ItemContextCallbackData.itemdata) ? RexxDWData->FirstCallback->event.ItemContextCallbackData.itemdata : "" );
               break;
            case REXXDW_LIST_SELECT_EVENT_CB:
               retstr->strlength = sprintf( retstr->strptr, "%s %lu,%d",
                                            RexxDWData->FirstCallback->userdata->func,
                                            (ULONG)RexxDWData->FirstCallback->window,
                                            RexxDWData->FirstCallback->event.ListSelectCallbackData.item );
               break;
            case REXXDW_ITEM_SELECT_EVENT_CB:
               retstr->strlength = sprintf( retstr->strptr, "%s %lu,%lu,\"%s\",\"%s\"",
                                            RexxDWData->FirstCallback->userdata->func,
                                            (ULONG)RexxDWData->FirstCallback->window,
                                            (unsigned long)RexxDWData->FirstCallback->event.ItemSelectCallbackData.item,
                                            (RexxDWData->FirstCallback->event.ItemSelectCallbackData.text) ? RexxDWData->FirstCallback->event.ItemSelectCallbackData.text : "",
                                            (RexxDWData->FirstCallback->event.ItemSelectCallbackData.itemdata) ? RexxDWData->FirstCallback->event.ItemSelectCallbackData.itemdata : "" );
               break;
            case REXXDW_SET_FOCUS_EVENT_CB:
               retstr->strlength = sprintf( retstr->strptr, "%s %lu",
                                            RexxDWData->FirstCallback->userdata->func,
                                            (ULONG)RexxDWData->FirstCallback->window );
               break;
            case REXXDW_VALUE_CHANGED_EVENT_CB:
               retstr->strlength = sprintf( retstr->strptr, "%s %lu,%d",
                                            RexxDWData->FirstCallback->userdata->func,
                                            (ULONG)RexxDWData->FirstCallback->window,
                                            RexxDWData->FirstCallback->event.ValueChangedCallbackData.val );
               break;
            case REXXDW_SWITCH_PAGE_EVENT_CB:
               retstr->strlength = sprintf( retstr->strptr, "%s %lu,%d",
                                            RexxDWData->FirstCallback->userdata->func,
                                            (ULONG)RexxDWData->FirstCallback->window,
                                            RexxDWData->FirstCallback->event.SwitchPageCallbackData.page );
               break;
            case REXXDW_COLUMN_CLICK_EVENT_CB:
               retstr->strlength = sprintf( retstr->strptr, "%s %lu,%d",
                                            RexxDWData->FirstCallback->userdata->func,
                                            (ULONG)RexxDWData->FirstCallback->window,
                                            RexxDWData->FirstCallback->event.ColumnClickCallbackData.column_num );
               break;
            default: /* timer callback */
               retstr->strlength = sprintf( retstr->strptr, "%s",
                                            RexxDWData->FirstCallback->userdata->func );
               break;
         }
         /*
          * Now append the user's arguments
          */
         for ( i = 0; i < RexxDWData->FirstCallback->userdata->argc; i++ )
         {
            strcat( retstr->strptr, "," );
            qstr = myquote( RXSTRPTR( RexxDWData->FirstCallback->userdata->argv[i] ), RXSTRLEN( RexxDWData->FirstCallback->userdata->argv[i] ) );
            strcat( retstr->strptr, qstr );
            /*
             * In case we had problems allocating memory, qstr will equal RSTRPTR
             */
            if ( qstr != RXSTRPTR( RexxDWData->FirstCallback->userdata->argv[i] ) )
               free( qstr );
         }
         retstr->strlength = strlen( retstr->strptr );
         /*
          * Set our pointers
          */
         last_first = RexxDWData->FirstCallback;
         RexxDWData->FirstCallback = RexxDWData->FirstCallback->next;
         free( last_first );
      }
      else
      {
         strcpy( retstr->strptr, "" );
         retstr->strlength = 0;
      }
   }
   return RxReturn( RxPackageGlobalData, retstr );
}

/****f* ProcessControl/DW_main_sleep
 * NAME
 *   DW_main_sleep
 * SYNOPSIS
 *   dw_main_sleep( Period )
 * FUNCTION
 *   This function dispatches any events in the event queue for the number of
 *   milliseconds specified by Period.
 * ARGUMENTS
 *   - Period - the number of milliseconds the event dispatcher runs for
 * RESULT
 *   No return result.
 * NOTES
 *   This function is useful to ensure that any outstanding event is executed
 *   before continuing execution within the code and not having to wait until
 *   control passes back to DW_Main() or DW_main_iteration().
 * SEE ALSO
 *   DW_main, DW_main_iteration
 * SOURCE
 *
...
Call dw_window_set_pointer mainwindow, !REXXDW.!DW_POINTER_CLOCK
Call dw_main_sleep 10
 ******
 */
rxfunc( DW_main_sleep )
{
   int milliseconds; /* local variable for arg[0] */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 1, 1 ) )
      return( 1 );
   if ( RxStrToInt( RxPackageGlobalData, &argv[0], (RX_INT *)&milliseconds ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   dw_main_sleep( milliseconds );
   return RxReturnString( RxPackageGlobalData, retstr, "" );
}

/****f* ProcessControl/DW_main_quit
 * NAME
 *   DW_main_quit
 * SYNOPSIS
 *   dw_main_quit( )
 * FUNCTION
 *   This function terminates the call to DW_Main().
 * ARGUMENTS
 *   None
 * RESULT
 *   No return result.
 * NOTES
 *   This function is useful to allow program control to continue after the call to DW_Main().
 *   It would normally be called in a callback associated with a toplevel window close event.
 * SEE ALSO
 *   DW_main, DW_main_iteration
 * SOURCE
 *
...
Call dw_main
Say 'do cleanup tasks here'
Exit 0
...
QuitCallback: Procedure Expose !REXXDW.
Call dw_main_quit
Return 0
...
 ******
 */
rxfunc( DW_main_quit )
{
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 0, 0 ) )
      return( 1 );
   dw_main_quit();
   return RxReturnString( RxPackageGlobalData, retstr, "" );
}

/*
 * int : dw_exec [char *program, int type, char **params] ( 3 )
 */
rxfunc( DW_exec )
{
   int rc; /* function return variable */
   char * program; /* local variable for arg[0] */
   int type; /* local variable for arg[1] */
   char ** params; /* local variable for arg[2] */
   int num_args;
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 3, 3 ) )
      return( 1 );
   if ( RXVALIDSTRING(argv[0]) )
      program = RXSTRPTR(argv[0]);
   else
      program = "";
   if ( RxStrToInt( RxPackageGlobalData, &argv[1], (RX_INT *)&type ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 2 );
      return(1);
   }
   if ( ( num_args = RxStemToCharArray( RxPackageGlobalData, &argv[2], (char ** *)&params ) ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Unable to convert Parameters stem: \"%s\".", RXSTRPTR(argv[2]) );
      return(1);
   }
   rc = dw_exec( program, type, params );
   RxFreeCharArray( RxPackageGlobalData, params, num_args );
   return RxReturnNumber( RxPackageGlobalData, retstr, rc );
}

/****f* ProcessControl/DW_shutdown
 * NAME
 *   DW_shutdown
 * SYNOPSIS
 *   dw_shutdown( )
 * FUNCTION
 *   This function cleans up all dwindows resources in preparation for a Rexx Exit
 * ARGUMENTS
 *   None
 * RESULT
 *   No return result.
 * SEE ALSO
 *   DW_exit
 * SOURCE
 *
...
Call dw_main
Say 'do cleanup tasks here'
Call dw_shutdown
Exit 0
...
QuitCallback: Procedure Expose !REXXDW.
Call dw_main_quit
Return 0
...
 ******
 */
rxfunc( DW_shutdown )
{
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 0, 0 ) )
      return( 1 );
   dw_shutdown();
   return RxReturnString( RxPackageGlobalData, retstr, "" );
}

/****f* ProcessControl/DW_exit
 * NAME
 *   DW_exit
 * SYNOPSIS
 *   dw_exit( Rcode )
 * FUNCTION
 *   This function cleans up all dwindows resources and exits the process with a return code of Rcode.
 * ARGUMENTS
 *   Rcode - numeric return value
 * RESULT
 *   No return result.
 * NOTES
 *   This function is a call to DW_shutdown followed by EXIT.
 * SEE ALSO
 *   DW_shutdown
 * SOURCE
 *
...
Call dw_main
Say 'do cleanup tasks here'
Call dw_exit 0
Exit 0 -- not reached
...
QuitCallback: Procedure Expose !REXXDW.
Call dw_main_quit
Return 0
...
 ******
 */
rxfunc( DW_exit )
{
   int exitcode; /* local variable for arg[0] */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 1, 1 ) )
      return( 1 );
   if ( RxStrToInt( RxPackageGlobalData, &argv[0], (RX_INT *)&exitcode ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   dw_exit( exitcode );
   return RxReturnString( RxPackageGlobalData, retstr, "" );
}

/****h* Functions/Dialog
 * DESCRIPTION
 *   A Dialog allows the programmer to create an interface with the user that
 *   is controlled outside of the normal main message dispatcher.
 *   other processes.
 ******
 */

/****f* Dialog/DW_dialog_new
 * NAME
 *   DW_dialog_new
 * SYNOPSIS
 *   dialog = dw_dialog_new( )
 * FUNCTION
 *   Creates a new dialog.
 *   A dialog is a control widget (as opposed to a display widget) that
 *   allows the programmer to create a new toplevel window and have the user
 *   interact with the contents of the window, independently of any other
 *   toplevel window.
 * ARGUMENTS
 *   None
 * RESULT
 *   Returns a dialog identifier
 * SEE ALSO
 *   DW_dialog_dismiss, DW_dialog_wait
 * NOTES
 *   Due to the lack of real callback capabilities with Open Object Rexx (ooRexx 3.x)
 *   any callbacks specified using dw_signal_connect() from a dialog window will not
 *   work. Only Regina 3.3 or above will operate correctly with dialogs.
 * SOURCE
 *
...
dialog_wait = dw_dialog_new( )
Call dw_signal_connect win, !REXXDW.!DW_DELETE_EVENT, ,
   'GenericCloseCallback', win, dialog_wait
...
result = dw_dialog_wait( dialog_wait )
If result = 'cancel' Then Return 1
...
GenericCloseCallback: Procedure Expose !REXXDW.
Parse Arg ., window, dialog
Call dw_window_destroy window
If dialog \= '' Then Call dw_dialog_dismiss dialog, 'close'
Return 0
 ******
 */
rxfunc( DW_dialog_new )
{
   DWDialog * rc; /* function return variable */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 0, 0 ) )
      return( 1 );
   rc = dw_dialog_new( NULL );
   return RxReturnPointer( RxPackageGlobalData, retstr, (void *)rc );
}

/****f* Dialog/DW_dialog_dismiss
 * NAME
 *   DW_dialog_dismiss
 * SYNOPSIS
 *   dw_dialog_dismiss( Dialog, ReturnString )
 * FUNCTION
 *   Prepares a dialog window for exiting and returns ReturnString
 *   to the waiting DW_dialog_wait().
 * ARGUMENTS
 *   - Dialog       - an identifier returned from a call to DW_dialog_new()
 *   - ReturnString - the string to return to DW_dialog_wait()
 * RESULT
 *   Always returns 0
 * SEE ALSO
 *   DW_dialog_new, DW_dialog_wait
 * SOURCE
 *
...
dialog_wait = dw_dialog_new( )
Call dw_signal_connect win, !REXXDW.!DW_DELETE_EVENT, ,
   'GenericCloseCallback', win, dialog_wait
...
result = dw_dialog_wait( dialog_wait )
If result = 'cancel' Then Return 1
...
GenericCloseCallback: Procedure Expose !REXXDW.
Parse Arg ., window, dialog
Call dw_window_destroy window
If dialog \= '' Then Call dw_dialog_dismiss dialog, 'close'
Return 0
 ******
 */
rxfunc( DW_dialog_dismiss )
{
   int rc; /* function return variable */
   DWDialog * dialog; /* local variable for arg[0] */
   char * result; /* local variable for arg[1] */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 2, 2 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&dialog ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   /* TODO - memory leak !! */
   if ( RXVALIDSTRING(argv[1]) )
      result = strdup( RXSTRPTR(argv[1]) );
   else
      result = strdup( "" );
   rc = dw_dialog_dismiss( dialog, (void *)result );
   return RxReturnNumber( RxPackageGlobalData, retstr, rc );
}

/****f* Dialog/DW_dialog_wait
 * NAME
 *   DW_dialog_wait
 * SYNOPSIS
 *   returnstring = dw_dialog_wait( Dialog )
 * FUNCTION
 *   Creates a new dialog for handling modal dialogs.
 * ARGUMENTS
 *   - Dialog       - an identifier returned from a call to DW_dialog_new()
 * RESULT
 *   Returns the string set by the call to DW_dialog_dismiss()
 * SEE ALSO
 *   DW_dialog_dismiss, DW_dialog_new
 * SOURCE
 *
...
dialog_wait = dw_dialog_new( )
Call dw_signal_connect win, !REXXDW.!DW_DELETE_EVENT, ,
   'GenericCloseCallback', win, dialog_wait
...
result = dw_dialog_wait( dialog_wait )
If result = 'cancel' Then Return 1
...
GenericCloseCallback: Procedure Expose !REXXDW.
Parse Arg ., window, dialog
Call dw_window_destroy window
If dialog \= '' Then Call dw_dialog_dismiss dialog, 'close'
Return 0
 ******
 */
rxfunc( DW_dialog_wait )
{
   char * rc; /* function return variable */
   DWDialog * dialog; /* local variable for arg[0] */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 1, 1 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&dialog ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   rc = (char *)dw_dialog_wait( dialog );
   return RxReturnString( RxPackageGlobalData, retstr, rc );
}

/****h* Functions/CallbackManagement
 * DESCRIPTION
 *   These functions manage callbacks in RexxDW.
 ******
 */

/****f* CallbackManagement/DW_signal_connect
 * NAME
 *   DW_signal_connect
 * SYNOPSIS
 *   dw_signal_connect( Win, Signal, Function[, UserData1[, UserData2[, ...]]] )
 * FUNCTION
 *   Connects an event to a callback function.
 * ARGUMENTS
 *   - Win      - the window identifier to which the callback is assigned
 *   - Signal   - the event to be connected
 *   - Function - the Rexx procedure that is called when the event fires
 *   - UserData - (optional) user supplied data to pass to the Function
 * RESULT
 *   No return result.
 * SEE ALSO
 *   DW_signal_disconnect, DW_signal_disconnect_by_window, Callbacks
 * SOURCE
 *
Call dw_signal_connect window, !REXXDW.!DW_CONFIGURE_EVENT, ,
   'configure_cb', 'fred', 'mary'
 ******
 */
rxfunc( DW_signal_connect )
{
   HWND window; /* local variable for arg[0] */
   unsigned int signum; /* local variable for arg[1] */
   REXXDWUserData * rexx_data;
   char * signame;
   void * sigfunc;
   int i;
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 3, 0 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&window ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   if ( RxStrToUInt( RxPackageGlobalData, &argv[1], (RX_UINT *)&signum ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name, "*ERROR* Invalid number for argument %d.", 2 );
      return(1);
   }
   if ( signum > REXXDW_MAX_CB )
   {
      RxDisplayError( RxPackageGlobalData, name, "*ERROR* Signal number invalid. Must be <= %d.", REXXDW_MAX_CB );
      return(1);
   }
   /*
    * Allocate memory for a RexxDW data struct
    */
   rexx_data = (REXXDWUserData *)calloc( 1, sizeof(REXXDWUserData) );
   if ( rexx_data == NULL )
   {
      RxDisplayError( RxPackageGlobalData, name, "*ERROR* Out of memory." );
      return(1);
   }
   /*
    * Set the PackageGlobalData so our callbacks can reference it
    */
   rexx_data->RxPackageGlobalData = RxPackageGlobalData;
   /*
    * Allocate space for the rexx procedure name
    * Make sure we have a valid argument...
    */
   if ( !RXVALIDSTRING( argv[2]) )
   {
      RxDisplayError( RxPackageGlobalData, name, "*ERROR* No Function Name (arg 3) supplied." );
      return(1);
   }
   rexx_data->func = (char *)malloc( RXSTRLEN(argv[2]) + 1 );
   if ( rexx_data->func == NULL )
   {
      free( rexx_data );
      RxDisplayError( RxPackageGlobalData, name, "*ERROR* Out of memory." );
      return(1);
   }
   strcpy( rexx_data->func, RXSTRPTR(argv[2]) );
   if ( argc > 3 )
   {
      rexx_data->argc = argc - 3;
      rexx_data->argv = (PRXSTRING)malloc( rexx_data->argc * sizeof(RXSTRING) );
      if ( rexx_data->argv == NULL )
      {
         free( rexx_data );
         free( rexx_data->func );
         RxDisplayError( RxPackageGlobalData, name, "*ERROR* Out of memory." );
         return(1);
      }
      for ( i = 0; i < rexx_data->argc; i++ )
      {
         rexx_data->argv[i].strlength = RXSTRLEN(argv[i+3]);
         rexx_data->argv[i].strptr = (char *)malloc( 1+RXSTRLEN(argv[i+3]) );
         if ( rexx_data->argv[i].strptr == NULL )
         {
            free( rexx_data );
            free( rexx_data->func );
            RxDisplayError( RxPackageGlobalData, name, "*ERROR* Out of memory." );
            return(1);
         }
         strcpy( rexx_data->argv[i].strptr, RXSTRPTR(argv[i+3]) );
         DEBUGDUMP(fprintf(stderr,"%s-%d:    Arg: %d \"%s\"\n",__FILE__,__LINE__,i+1,rexx_data->argv[i].strptr);)
      }
   }
   else
   {
      rexx_data->argc = 0;
      rexx_data->argv = NULL;
   }
   DEBUGDUMP(fprintf(stderr,"%s-%d: In DW_signal_connect() Func: \"%s\"\n",__FILE__,__LINE__,rexx_data->func);)
   /*
    * Lookup the signal name from the supplied signal number, and then
    * link the appropriate callback type in dw_signal_connect()
    */
   signame = REXXDWSignalTranslate[signum].signame;
   sigfunc = REXXDWSignalTranslate[signum].sigfunc;
   DEBUGDUMP(fprintf(stderr,"%s-%d: In DW_signal_connect() signum: %d signame: %s\n",__FILE__,__LINE__,signum, signame);)
   dw_signal_connect( window, signame, DW_SIGNAL_FUNC(sigfunc), (void *)rexx_data );
   return RxReturnString( RxPackageGlobalData, retstr, "" );
}

/****f* CallbackManagement/DW_signal_disconnect
 * NAME
 *   DW_signal_disconnect
 * SYNOPSIS
 *   dw_signal_disconnect( Win, Signal )
 * FUNCTION
 *   Disconnects a specific event callback.
 * ARGUMENTS
 *   - Win        - the window identifier that has the callback assigned
 *   - Signal     - the event to be disconnected
 * RESULT
 *   No return result.
 * SEE ALSO
 *   DW_signal_connect, DW_signal_disconnect_by_window, Callbacks
 * SOURCE
 *
Call dw_signal_connect window, !REXXDW.!DW_CONFIGURE_EVENT, ,
   'configure_cb', 'fred', 'mary'
...
Call dw_signal_disconnect window, !REXXDW.!DW_CONFIGURE_EVENT
 ******
 */
rxfunc( DW_signal_disconnect )
{
   HWND window; /* local variable for arg[0] */
   unsigned int signum; /* local variable for arg[1] */
   char * signame;
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 2, 2 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&window ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   if ( RxStrToUInt( RxPackageGlobalData, &argv[1], (RX_UINT *)&signum ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 2 );
      return(1);
   }
   if ( signum > REXXDW_MAX_CB )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Signal number invalid. Must be <= %d.", REXXDW_MAX_CB );
      return(1);
   }
   /*
    * Lookup the signal name from the supplied signal number, and then
    * link the appropriate callback type in dw_signal_connect()
    */
   signame = REXXDWSignalTranslate[signum].signame;
   dw_signal_disconnect_by_name( window, signame );
   return RxReturnString( RxPackageGlobalData, retstr, "" );
}

/****f* CallbackManagement/DW_signal_disconnect_by_window
 * NAME
 *   DW_signal_disconnect_by_window
 * SYNOPSIS
 *   dw_signal_disconnect_by_window( Win )
 * FUNCTION
 *   Disconnects all signals for a window.
 * ARGUMENTS
 *   - Win        - the window identifier that has the callbacks assigned
 * RESULT
 *   No return result.
 * SEE ALSO
 *   DW_signal_connect, DW_signal_disconnect, Callbacks
 * SOURCE
 *
Call dw_signal_connect window, !REXXDW.!DW_CONFIGURE_EVENT, ,
   'configure_cb', 'fred', 'mary'
...
Call dw_signal_disconnect_by_window window
 ******
 */
rxfunc( DW_signal_disconnect_by_window )
{
   HWND window; /* local variable for arg[0] */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 1, 1 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&window ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   dw_signal_disconnect_by_window( window );
   return RxReturnString( RxPackageGlobalData, retstr, "" );
}

/****f* CallbackManagement/DW_timer_connect
 * NAME
 *   DW_timer_connect
 * SYNOPSIS
 *   timer = dw_timer_connect( Interval, Function[, UserData1[, UserData2[, ...]]] )
 * FUNCTION
 *   Creates a new timer callback.
 *   A timer callback will fire after the specified Interval and call the
 *   Rexx procedure Function with the optional arguments.
 * ARGUMENTS
 *   - Interval   - the interval in milliseconds that the timer will fire
 *   - Function   - the Rexx procedure that is called when the timer fires
 *   - UserData*  - optional user supplied data to pass to the Function
 * RESULT
 *   A timer identifier.
 * SEE ALSO
 *   DW_timer_disconnect, Callbacks
 * NOTES
 *   There are two ways to stop a timer from firing. One is explicitly by
 *   calling DW_timer_disconnect() using the timer identifier returned from
 *   this function. The other way is to return 0 from the callback connected
 *   to the timer. Returning 1 from the callback will re-arm the timer.
 * SOURCE
 *
timer = dw_timer_connect( 5000, 'timer_cb', 'fred', 'mary' )
 ******
 */
rxfunc( DW_timer_connect )
{
   int rc; /* function return variable */
   int i;
   int interval; /* local variable for arg[0] */
   REXXDWUserData * rexx_data;
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 2, 0 ) )
      return( 1 );
   if ( RxStrToInt( RxPackageGlobalData, &argv[0], (RX_INT *)&interval ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name, "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   /*
    * Allocate memory for a RexxDW data struct
    */
   rexx_data = (REXXDWUserData *)calloc( 1, sizeof(REXXDWUserData) );
   if ( rexx_data == NULL )
   {
      RxDisplayError( RxPackageGlobalData, name, "*ERROR* Out of memory." );
      return(1);
   }
   /*
    * Allocate space for the rexx procedure name
    * Make sure we have a valid argument...
    */
   if ( !RXVALIDSTRING( argv[1]) )
   {
      RxDisplayError( RxPackageGlobalData, name, "*ERROR* No Function Name (arg 2) supplied." );
      return(1);
   }
   rexx_data->func = (char *)malloc( RXSTRLEN(argv[1]) + 1 );
   if ( rexx_data->func == NULL )
   {
      free( rexx_data );
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Out of memory." );
      return(1);
   }
   strcpy( rexx_data->func, RXSTRPTR(argv[1]) );
   if ( argc > 2 )
   {
      rexx_data->argc = argc - 2;
      rexx_data->argv = (PRXSTRING)malloc( rexx_data->argc * sizeof(RXSTRING) );
      if ( rexx_data->argv == NULL )
      {
         free( rexx_data );
         free( rexx_data->func );
         RxDisplayError( RxPackageGlobalData, name, "*ERROR* Out of memory." );
         return(1);
      }
      for ( i = 0; i < rexx_data->argc; i++ )
      {
         rexx_data->argv[i].strlength = RXSTRLEN(argv[i+2]);
         rexx_data->argv[i].strptr = (char *)malloc( 1+RXSTRLEN(argv[i+2]) );
         if ( rexx_data->argv[i].strptr == NULL )
         {
            free( rexx_data );
            free( rexx_data->func );
            RxDisplayError( RxPackageGlobalData, name, "*ERROR* Out of memory." );
            return(1);
         }
         strcpy( rexx_data->argv[i].strptr, RXSTRPTR(argv[i+2]) );
         DEBUGDUMP(fprintf(stderr,"%s-%d:    Arg: %d \"%s\"\n",__FILE__,__LINE__,i+1,rexx_data->argv[i].strptr);)
      }
   }
   else
   {
      rexx_data->argc = 0;
      rexx_data->argv = NULL;
   }
   DEBUGDUMP(fprintf(stderr,"%s-%d: In DW_timer_connect() Func: %s Argc: %d\n",__FILE__,__LINE__,rexx_data->func, rexx_data->argc);)
   rc = dw_timer_connect( interval, rexxdw_timer_cb, (void *)rexx_data );
   return RxReturnNumber( RxPackageGlobalData, retstr, rc );
}

/****f* CallbackManagement/DW_timer_disconnect
 * NAME
 *   DW_timer_disconnect
 * SYNOPSIS
 *   dw_timer_disconnect( Timer )
 * FUNCTION
 *   Terminates the timer so the timer no longer files.
 * ARGUMENTS
 *   - Timer      - the timer identifier returned from a call to DW_timer_connect()
 * RESULT
 *   No return result.
 * SEE ALSO
 *   DW_timer_connect, Callbacks
 * SOURCE
 *
timer = dw_timer_connect( 5000, 'timer_cb', 'fred', 'mary' )
...
Call dw_timer_disconnect timer
 ******
 */
rxfunc( DW_timer_disconnect )
{
   int id; /* local variable for arg[0] */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 1, 1 ) )
      return( 1 );
   if ( RxStrToInt( RxPackageGlobalData, &argv[0], (RX_INT *)&id ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   dw_timer_disconnect( id );
   return RxReturnString( RxPackageGlobalData, retstr, "" );
}

/****f* CallbackManagement/DW_callback_get_timestamp
 * NAME
 *   DW_callback_get_timestamp
 * SYNOPSIS
 *   timestamp = dw_callback_get_timestamp()
 * FUNCTION
 *   Returns the timestamp (time_t value) of when the last callback fired.
 * ARGUMENTS
 *   - None
 * RESULT
 *   The timestamp of the last callback.
 * SEE ALSO
 *   Callbacks
 * SOURCE
 *
if dw_callback_get_timestamp() + 600 < time('T)' then say 'No activity in the application for 10 minutes.'
 ******
 */
rxfunc( DW_callback_get_timestamp )
{
   REXXDWDATA *RexxDWData;
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RexxDWData=(REXXDWDATA *)RxPackageGlobalData->RXPACKAGE_tsd;

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 0, 0 ) )
      return( 1 );
   return RxReturnUnsignedNumber( RxPackageGlobalData, retstr, (ULONG)RexxDWData->CallbackTimestamp );
}

/****h* Functions/Browsing
 * DESCRIPTION
 *   These functions provide browsing capabilties.
 ******
 */

/****f* Browsing/DW_browse
 * NAME
 *   DW_browse
 * SYNOPSIS
 *   rc = dw_browse( URL )
 * FUNCTION
 *   Starts up the default Web browser as a new process with the specified URL.
 *   When using GTK, the browser started is "netscape". To use another browser
 *   set the environment variable DW_BROWSER to the executable to run.
 * ARGUMENTS
 *   - URL        - the Uniform Resource Locatr to open
 *   - Padding    - The number of pixels of padding to add around all sides of BoxToPack
 * RESULT
 *   The return code from the attempt to execute the browser. Usually 0 indicates
 *   success and any other value an error.
 * SOURCE
 *
...
rc = dw_browse( "http://rexxdw.sf.net" )
 ******
 */
rxfunc( DW_browse )
{
   int rc; /* function return variable */
   char * url; /* local variable for arg[0] */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 1, 1 ) )
      return( 1 );
   if ( RXVALIDSTRING(argv[0]) )
      url = RXSTRPTR(argv[0]);
   else
      url = "";
   rc = dw_browse( url );
   return RxReturnNumber( RxPackageGlobalData, retstr, rc );
}

/****f* Browsing/DW_file_browse
 * NAME
 *   DW_file_browse
 * SYNOPSIS
 *   filename = dw_file_browse( Title, DefaultDir, Extension, Flag )
 * FUNCTION
 *   Creates a file open/save or directory browser dialog.
 * ARGUMENTS
 *   - Title      - the title of the dialog
 *   - DefaultDir - the directory in which the dialog starts
 *   - Ext        - a file extension to filter the files displayed, or the
 *                  file extension to save. Ignored for directory browser.
 *   - Flag       - indicates if the browse dialog is for a file open, file save
 *                  or directory browse. One of !REXXDW.!DW_DIRECTORY_OPEN,
 *                  !REXXDW.!DW_FILE_OPEN or !REXXDW.!DW_FILE_SAVE
 * RESULT
 *   The fully qualified selected file/directory or blank if the dialog is cancelled.
 * SEE ALSO
 *   FileDialogFlags
 * NOTES
 *   On some platforms the current directory is changed when this function is called.
 * SOURCE
 *
...
here = Directory()
dir = dw_file_browse( "Select Directory", "/home/mark", , ,
   !REXXDW.!DW_DIRECTORY_OPEN )
Say "Directory selected was:" dir
Call Directory( here )
 ******
 */
rxfunc( DW_file_browse )
{
   char * rc; /* function return variable */
   char * tmp = "";
   char * title; /* local variable for arg[0] */
   char * defpath; /* local variable for arg[1] */
   char * ext; /* local variable for arg[2] */
   int flags; /* local variable for arg[3] */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 4, 4 ) )
      return( 1 );
   if ( RXVALIDSTRING(argv[0]) )
      title = RXSTRPTR(argv[0]);
   else
      title = "";
   if ( RXVALIDSTRING(argv[1]) )
      defpath = RXSTRPTR(argv[1]);
   else
      defpath = "";
   if ( RXVALIDSTRING(argv[2]) )
      ext = RXSTRPTR(argv[2]);
   else
      ext = NULL;
   if ( RxStrToInt( RxPackageGlobalData, &argv[3], (RX_INT *)&flags ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 4 );
      return(1);
   }
   rc = dw_file_browse( title, defpath, ext, flags );
   if ( rc )
   {
      tmp = strdup( rc );
      dw_free( rc );
   }
   return RxReturnString( RxPackageGlobalData, retstr, tmp );
}

/****h* Functions/ColourSupport
 * DESCRIPTION
 *   These functions allow colours to be manipulated.
 ******
 */

/****f* ColourSupport/DW_color_depth_get
 * NAME
 *   DW_color_depth_get
 * SYNOPSIS
 *   depth = dw_color_depth_get()
 * FUNCTION
 *   Returns the colour depth of the current screen.
 * ARGUMENTS
 *   None
 * RESULT
 *   The colour depth.
 * SOURCE
 *
...
Say 'Colour depth is:' dw_color_depth_get()
 ******
 */
rxfunc( DW_color_depth_get )
{
   unsigned long rc; /* function return variable */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 0, 0 ) )
      return( 1 );
   rc = dw_color_depth_get( );
   return RxReturnUnsignedNumber( RxPackageGlobalData, retstr, rc );
}

/****f* ColourSupport/DW_rgb
 * NAME
 *   DW_rgb
 * SYNOPSIS
 *   colour = dw_rgb( Red, Green, Blue )
 * FUNCTION
 *   Mixes shades of red, green and blue to make an internal RexxDW colour.
 * ARGUMENTS
 *   - Red      - the amount of Red to use (0 to 255 )
 *   - Green    - the amount of Green to use (0 to 255 )
 *   - Blue     - the amount of Blue to use (0 to 255 )
 * RESULT
 *   An internal RexxDW colour.
 * SEE ALSO
 *   DW_red_value, DW_green_value, DW_red_value
 * SOURCE
 *
...
colour = dw_rgb( 45, 134, 200 )
 ******
 */
rxfunc( DW_rgb )
{
   unsigned long color, red, green, blue;
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 3, 3 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&red ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   if ( RxStrToULong( RxPackageGlobalData, &argv[1], (RX_ULONG *)&green ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 2 );
      return(1);
   }
   if ( RxStrToULong( RxPackageGlobalData, &argv[2], (RX_ULONG *)&blue ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 3 );
      return(1);
   }
   color = DW_RGB( red, green, blue );
   return RxReturnUnsignedNumber( RxPackageGlobalData, retstr, (ULONG)color );
}

/****f* ColourSupport/DW_red_value
 * NAME
 *   DW_red_value
 * SYNOPSIS
 *   red = dw_red_value( Colour )
 * FUNCTION
 *   Obtains the red component from a RexxDW colour.
 * ARGUMENTS
 *   - Colour   - the internal RexxDW colour containing a red component
 * RESULT
 *   The red component of the colour.
 * SEE ALSO
 *   DW_green_value, DW_blue_value, DW_rgb
 * SOURCE
 *
...
Say 'The red component of DARKCYAN is:' ,
   dw_red_value( !REXXDW.!DW_CLR_DARKCYAN )
 ******
 */
rxfunc( DW_red_value )
{
   unsigned long color, red;
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 1, 1 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&color ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   red = DW_RED_VALUE( color );
   return RxReturnUnsignedNumber( RxPackageGlobalData, retstr, (ULONG)red );
}

/****f* ColourSupport/DW_green_value
 * NAME
 *   DW_green_value
 * SYNOPSIS
 *   green = dw_green_value( Colour )
 * FUNCTION
 *   Obtains the green component from a RexxDW colour.
 * ARGUMENTS
 *   - Colour   - the internal RexxDW colour containing a green component
 * RESULT
 *   The green component of the colour.
 * SEE ALSO
 *   DW_red_value, DW_blue_value, DW_rgb
 * SOURCE
 *
...
Say 'The green component of DARKCYAN is:' ,
   dw_green_value( !REXXDW.!DW_CLR_DARKCYAN )
 ******
 */
rxfunc( DW_green_value )
{
   unsigned long color, green;
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 1, 1 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&color ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   green = DW_GREEN_VALUE( color );
   return RxReturnUnsignedNumber( RxPackageGlobalData, retstr, (ULONG)green );
}

/****f* ColourSupport/DW_blue_value
 * NAME
 *   DW_blue_value
 * SYNOPSIS
 *   blue = dw_blue_value( Colour )
 * FUNCTION
 *   Obtains the blue component from a RexxDW colour.
 * ARGUMENTS
 *   - Colour   - the internal RexxDW colour containing a blue component
 * RESULT
 *   The blue component of the colour.
 * SEE ALSO
 *   DW_red_value, DW_green_value, DW_rgb
 * SOURCE
 *
...
Say 'The blue component of DARKCYAN is:' ,
   dw_blue_value( !REXXDW.!DW_CLR_DARKCYAN )
 ******
 */
rxfunc( DW_blue_value )
{
   unsigned long color, blue;
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 1, 1 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&color ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   blue = DW_BLUE_VALUE( color );
   return RxReturnUnsignedNumber( RxPackageGlobalData, retstr, (ULONG)blue );
}

/****f* ColourSupport/DW_color_background_set
 * NAME
 *   DW_color_background_set
 * SYNOPSIS
 *   dw_color_background_set( Colour )
 * FUNCTION
 *   Sets the background colour of the specified window. Only applicable
 *   to render??? windows.
 * ARGUMENTS
 *   - Colour     - either one of the pre-defined colour constatnts, or
 *                  a colour returned from a call to DW_rgb()
 * RESULT
 *   No return result.
 * SEE ALSO
 *   Colours, DW_color_foreground_set, DW_rgb
 * SOURCE
 *
...
Call dw_color_background_set( renderbox, dw_rgb( 10, 45, 123 ) )
 ******
 */
rxfunc( DW_color_background_set )
{
   unsigned long value; /* local variable for arg[0] */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 1, 1 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&value ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   dw_color_background_set( value );
   return RxReturnString( RxPackageGlobalData, retstr, "" );
}

/****f* ColourSupport/DW_color_foreground_set
 * NAME
 *   DW_color_foreground_set
 * SYNOPSIS
 *   dw_color_foreground_set( Colour )
 * FUNCTION
 *   Sets the foreground colour of the specified window. Only applicable
 *   to render??? windows.
 * ARGUMENTS
 *   - Colour     - either one of the pre-defined colour constatnts, or
 *                  a colour returned from a call to DW_rgb()
 * RESULT
 *   No return result.
 * SEE ALSO
 *   Colours, DW_color_background_set, DW_rgb
 * SOURCE
 *
...
Call dw_color_foreground_set( renderbox, dw_rgb( 10, 45, 123 ) )
 ******
 */
rxfunc( DW_color_foreground_set )
{
   unsigned long value; /* local variable for arg[0] */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 1, 1 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&value ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   dw_color_foreground_set( value );
   return RxReturnString( RxPackageGlobalData, retstr, "" );
}

/****f* ColourSupport/DW_color_choose
 * NAME
 *   DW_color_choose
 * SYNOPSIS
 *   color = dw_color_choose( InitialColour )
 * FUNCTION
 *   Displays a system dialog for selecting a color.
 * ARGUMENTS
 *   - InitialColour     - either one of the pre-defined colour constatnts, or
 *                         a colour returned from a call to DW_rgb()
 * RESULT
 *   The selected color, or the same value as InitialColor if the user cancels.
 * SEE ALSO
 *   Colours, DW_color_background_set, DW_rgb
 * SOURCE
 *
...
newcolor = dw_color_choose( dw_rgb( 10, 45, 123 )
 ******
 */
rxfunc( DW_color_choose )
{
   unsigned long value; /* local variable for arg[0] */
   unsigned long rc;
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 1, 1 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&value ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   rc = dw_color_choose( value );
   return RxReturnUnsignedNumber( RxPackageGlobalData, retstr, (ULONG)rc );
}

/****h* Functions/FontSupport
 * DESCRIPTION
 *   These functions support manipulation of fonts.
 ******
 */

/****f* FontSupport/DW_font_choose
 * NAME
 *   DW_font_choose
 * SYNOPSIS
 *   font = dw_font_choose( [CurrentFont] )
 * FUNCTION
 *   Displays a font chooser dialog
 * ARGUMENTS
 *   - CurrentFont - the current font to set as the basis for choosing a font
 * RESULT
 *   The selected font or blank if no font chosen
 * SOURCE
 *
font = dw_font_choose( '8.Monaco' )
 ******
 */
rxfunc( DW_font_choose )
{
   char * fontname = NULL; /* local variable for arg[1] */
   char *rc = NULL;
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 0, 1 ) )
      return( 1 );
   if ( argc == 1 )
   {
      if ( RXVALIDSTRING(argv[0]) )
         fontname = RXSTRPTR(argv[0]);
      else
         fontname = NULL;
   }
   rc = dw_font_choose( fontname );
   if ( rc )
      return RxReturnStringAndFree( RxPackageGlobalData, retstr, rc, 1 );
   else
      return RxReturnString( RxPackageGlobalData, retstr, "" );
}

/****f* FontSupport/DW_font_set_default
 * NAME
 *   DW_font_set_default
 * SYNOPSIS
 *   dw_font_set_default( Font )
 * FUNCTION
 *   Sets the default font for all widgets with a text component.
 * ARGUMENTS
 *   - Font - the font to be used as the default in Dynamic Windows format
 * RESULT
 *   None
 * SOURCE
 *
Call dw_font_set_default, '8.Monaco'
 ******
 */
rxfunc( DW_font_set_default )
{
   char * fontname; /* local variable for arg[0] */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 1, 1 ) )
      return( 1 );
   if ( RXVALIDSTRING(argv[0]) )
      fontname = RXSTRPTR(argv[0]);
   else
      return 1;
   dw_font_set_default( fontname );
   return RxReturnString( RxPackageGlobalData, retstr, "" );
}

/****f* FontSupport/DW_font_text_extents_get
 * NAME
 *   DW_font_text_extents_get
 * SYNOPSIS
 *   Width Height = dw_font_text_extents_get( Win, Pixmap, Text )
 * FUNCTION
 *   Returns the width and height of the Text in the current font.
 * ARGUMENTS
 *   - Win        - the window identifier returned from DW_container_new()
 *   - Pixmap     - the pixmap identifier returned from DW_pixmap_new()
 *   - Text       - the string to measure the font width and height
 * RESULT
 *   Width and Height are returned as two words. PARSE VALUE is the easiest
 *   way to obtain both values.
 * NOTES
 *   When calculating the size of the maximum width and height of a font, use a
 *   value like 'g(' for Text to ensure the maximum height of the Text
 *   can be calculated.
 *   Only one of Win or Pixmap is required. The other argument should be set to 0.
 * SOURCE
 *
win = dw_render_new( 0 )
Call dw_window_set_font win, myfont
Parse Value dw_font_text_extents_get( win, 0, 'g(' ) With width height
 ******
 */
rxfunc( DW_font_text_extents_get )
{
   HWND handle; /* local variable for arg[0] */
   HPIXMAP pixmap; /* local variable for arg[1] */
   char * text; /* local variable for arg[2] */
   int width=0;
   int height=0;
   char buf[50]; /* enough for two numbers */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 3, 3 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&handle ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   if ( RxStrToULong( RxPackageGlobalData, &argv[1], (RX_ULONG *)&pixmap ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 2 );
      return(1);
   }
   if ( RXVALIDSTRING(argv[2]) )
      text = RXSTRPTR(argv[2]);
   else
      text = "";
   dw_font_text_extents_get( handle, pixmap, text, &width, &height );
   sprintf( buf, "%d %d", width, height );
   return RxReturnString( RxPackageGlobalData, retstr, buf );
}

/****h* Functions/ModuleSupport
 * DESCRIPTION
 *   These functions support the loading and unloading of modules in DLLs.
 *   Not applicable to RexxDW.
 ******
 */

/*
 * int : dw_module_close [HMOD handle] ( 1 )
 */
rxfunc( DW_module_close )
{
   int rc; /* function return variable */
   HMOD handle; /* local variable for arg[0] */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 1, 1 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&handle ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   rc = dw_module_close( handle );
   return RxReturnNumber( RxPackageGlobalData, retstr, rc );
}

/*
 * int : dw_module_load [char *name, HMOD *handle] ( 2 )
 */
rxfunc( DW_module_load )
{
   int rc; /* function return variable */
   char * x_name; /* local variable for arg[0] */
   HMOD * handle; /* local variable for arg[1] */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 2, 2 ) )
      return( 1 );
   if ( RXVALIDSTRING(argv[0]) )
      x_name = RXSTRPTR(argv[0]);
   else
      x_name = "";
   if ( RxStrToULong( RxPackageGlobalData, &argv[1], (RX_ULONG *)&handle ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 2 );
      return(1);
   }
   rc = dw_module_load( x_name, handle );
   return RxReturnNumber( RxPackageGlobalData, retstr, rc );
}

/*
 * int : dw_module_symbol [HMOD handle, char *name, void**func] ( 3 )
 */
rxfunc( DW_module_symbol )
{
   int rc; /* function return variable */
   HMOD handle; /* local variable for arg[0] */
   char * x_name; /* local variable for arg[1] */
   void ** func; /* local variable for arg[2] */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 3, 3 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&handle ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   if ( RXVALIDSTRING(argv[1]) )
      x_name = RXSTRPTR(argv[1]);
   else
      x_name = "";
   if ( RxStrToULong( RxPackageGlobalData, &argv[2], (RX_ULONG *)&func ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 3 );
      return(1);
   }
   rc = dw_module_symbol( handle, x_name, func );
   return RxReturnNumber( RxPackageGlobalData, retstr, rc );
}

/****h* Functions/MutexSupport
 * DESCRIPTION
 *   These functions support the processing of semaphores.
 *   Not applicable to RexxDW.
 ******
 */

/*
 * void : dw_mutex_close [HMTX mutex] ( 1 )
 */
rxfunc( DW_mutex_close )
{
   HMTX mutex; /* local variable for arg[0] */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 1, 1 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&mutex ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   dw_mutex_close( mutex );
   return RxReturnString( RxPackageGlobalData, retstr, "" );
}

/*
 * void : dw_mutex_lock [HMTX mutex] ( 1 )
 */
rxfunc( DW_mutex_lock )
{
   HMTX mutex; /* local variable for arg[0] */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 1, 1 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&mutex ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   dw_mutex_lock( mutex );
   return RxReturnString( RxPackageGlobalData, retstr, "" );
}

/*
 * HMTX : dw_mutex_new [void] ( 0 )
 */
rxfunc( DW_mutex_new )
{
   HMTX rc; /* function return variable */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 0, 0 ) )
      return( 1 );
   rc = dw_mutex_new( );
   return RxReturnNumber( RxPackageGlobalData, retstr, (ULONG)rc );
}

/*
 * void : dw_mutex_unlock [HMTX mutex] ( 1 )
 */
rxfunc( DW_mutex_unlock )
{
   HMTX mutex; /* local variable for arg[0] */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 1, 1 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&mutex ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   dw_mutex_unlock( mutex );
   return RxReturnString( RxPackageGlobalData, retstr, "" );
}

/****h* Functions/EventSupport
 * DESCRIPTION
 *   These functions support events.
 *   Not applicable to RexxDW.
 ******
 */

/*
 * int : dw_event_close [HEV *eve] ( 1 )
 */
rxfunc( DW_event_close )
{
   int rc; /* function return variable */
   HEV * eve; /* local variable for arg[0] */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 1, 1 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&eve ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   rc = dw_event_close( eve );
   return RxReturnNumber( RxPackageGlobalData, retstr, rc );
}

/*
 * HEV : dw_event_new [void] ( 0 )
 */
rxfunc( DW_event_new )
{
   HEV rc; /* function return variable */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 0, 0 ) )
      return( 1 );
   rc = dw_event_new( );
   return RxReturnNumber( RxPackageGlobalData, retstr, (ULONG)rc );
}

/*
 * int : dw_event_post [HEV eve] ( 1 )
 */
rxfunc( DW_event_post )
{
   int rc; /* function return variable */
   HEV eve; /* local variable for arg[0] */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 1, 1 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&eve ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   rc = dw_event_post( eve );
   return RxReturnNumber( RxPackageGlobalData, retstr, rc );
}

/*
 * int : dw_event_reset [HEV eve] ( 1 )
 */
rxfunc( DW_event_reset )
{
   int rc; /* function return variable */
   HEV eve; /* local variable for arg[0] */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 1, 1 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&eve ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   rc = dw_event_reset( eve );
   return RxReturnNumber( RxPackageGlobalData, retstr, rc );
}

/*
 * int : dw_event_wait [HEV eve, unsigned long timeout] ( 2 )
 */
rxfunc( DW_event_wait )
{
   int rc; /* function return variable */
   HEV eve; /* local variable for arg[0] */
   unsigned long timeout; /* local variable for arg[1] */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 2, 2 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&eve ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   if ( RxStrToULong( RxPackageGlobalData, &argv[1], (RX_ULONG *)&timeout ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 2 );
      return(1);
   }
   rc = dw_event_wait( eve, timeout );
   return RxReturnNumber( RxPackageGlobalData, retstr, rc );
}

/****h* Functions/ThreadSupport
 * DESCRIPTION
 *   These functions support threads.
 *   Not applicable to RexxDW.
 ******
 */

/*
 * void : dw_thread_end [void] ( 0 )
 */
rxfunc( DW_thread_end )
{
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 0, 0 ) )
      return( 1 );
   dw_thread_end( );
   return RxReturnString( RxPackageGlobalData, retstr, "" );
}

/*
 * DWTID : dw_thread_id [void] ( 0 )
 */
rxfunc( DW_thread_id )
{
   DWTID rc; /* function return variable */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 0, 0 ) )
      return( 1 );
   rc = dw_thread_id( );
   return RxReturnPointer( RxPackageGlobalData, retstr, (void *)rc );
}

/*
 * DWTID : dw_thread_new [void *func, [void *data] ( 2 )
 */
rxfunc( DW_thread_new )
{
   DWTID rc; /* function return variable */
   tREXXDWUserData * rexx_data;
   int stack=0; /* local variable for arg[1] */
   int i;
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 1, 0 ) )
      return( 1 );
   /*
    * Allocate memory for a RexxDW data struct
    */
   rexx_data = (tREXXDWUserData *)calloc( 1, sizeof(tREXXDWUserData) );
   if ( rexx_data == NULL )
   {
      RxDisplayError( RxPackageGlobalData, name, "*ERROR* Out of memory." );
      return(1);
   }
   /*
    * Allocate space for the rexx procedure name
    * Make sure we have a valid argument...
    */
   if ( !RXVALIDSTRING( argv[0]) )
   {
      RxDisplayError( RxPackageGlobalData, name, "*ERROR* No Function Name (arg 1) supplied." );
      free( rexx_data );
      return(1);
   }
   rexx_data->func = (char *)malloc( RXSTRLEN(argv[0]) + 1 );
   if ( rexx_data->func == NULL )
   {
      free( rexx_data );
      RxDisplayError( RxPackageGlobalData, name, "*ERROR* Out of memory." );
      return(1);
   }
   strcpy( rexx_data->func, RXSTRPTR(argv[0]) );
   if ( argc > 1 )
   {
      rexx_data->argc = argc - 1;
      rexx_data->argv = (PRXSTRING)malloc( rexx_data->argc * sizeof(RXSTRING) );
      if ( rexx_data->argv == NULL )
      {
         free( rexx_data );
         free( rexx_data->func );
         RxDisplayError( RxPackageGlobalData, name, "*ERROR* Out of memory." );
         return(1);
      }
      for ( i = 0; i < rexx_data->argc; i++ )
      {
         rexx_data->argv[i].strlength = RXSTRLEN(argv[i+1]);
         rexx_data->argv[i].strptr = (char *)malloc( 1+RXSTRLEN(argv[i+1]) );
         if ( rexx_data->argv[i].strptr == NULL )
         {
            free( rexx_data );
            free( rexx_data->func );
            RxDisplayError( RxPackageGlobalData, name, "*ERROR* Out of memory." );
            return(1);
         }
         strcpy( rexx_data->argv[i].strptr, RXSTRPTR(argv[i+1]) );
      }
   }
   else
   {
      rexx_data->argc = 0;
      rexx_data->argv = NULL;
   }

   DEBUGDUMP(fprintf(stderr,"%s-%d: In DW_thread_new() Func: \"%s\"\n",__FILE__,__LINE__,rexx_data->func);)
   rc = dw_thread_new( rexxdw_thread_new_cb, (void *)rexx_data, stack );
   return RxReturnNumber( RxPackageGlobalData, retstr, (ULONG)rc );
}

/****h* Functions/PointerPosition
 * DESCRIPTION
 *   These functions allow positioning of the mouse pointer.
 ******
 */

/****f* PointerPosition/DW_pointer_set_pos
 * NAME
 *   DW_pointer_set_pos
 * SYNOPSIS
 *   dw_pointer_set_pos( X, Y )
 * FUNCTION
 *   Moves the mouse pointer to the specified X and Y coordinates.
 * ARGUMENTS
 *   - X          - the X coordinate of the mouse pointer
 *   - Y          - the Y coordinate of the mouse pointer
 * RESULT
 *   No return result.
 * SEE ALSO
 *   DW_pointer_get_pos
 * NOTES
 *   The coordinates are absolute screen positions, with 0/0 being the
 *   the top left corner of the screen. (Windows!!)
 * SOURCE
 *
...
Call dw_pointer_set_pos 100, 130
 ******
 */
rxfunc( DW_pointer_set_pos )
{
   long x; /* local variable for arg[0] */
   long y; /* local variable for arg[1] */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 2, 2 ) )
      return( 1 );
   if ( RxStrToLong( RxPackageGlobalData, &argv[0], (RX_LONG *)&x ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   if ( RxStrToLong( RxPackageGlobalData, &argv[1], (RX_LONG *)&y ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 2 );
      return(1);
   }
   dw_pointer_set_pos( x, y );
   return RxReturnString( RxPackageGlobalData, retstr, "" );
}

/****f* PointerPosition/DW_pointer_get_pos
 * NAME
 *   DW_pointer_get_pos
 * SYNOPSIS
 *   X Y = dw_pointer_get_pos( )
 * FUNCTION
 *   Returns the X and Y coordinates of the mouse pointer.
 * ARGUMENTS
 *   None
 * RESULT
 *   - X and Y are returned as two words. PARSE VALUE is the easiest
 *   way to obtain both values.
 * SEE ALSO
 *   DW_pointer_set_pos
 * NOTES
 *   The coordinates are absolute screen positions, with 0/0 being the
 *   the top left corner of the screen. (Windows!!)
 * SOURCE
 *
...
Parse Value dw_pointer_get_pos( ) With x y
Say 'The mouse pointer is at' x'/'y
 ******
 */
rxfunc( DW_pointer_get_pos )
{
   long x;
   long y;
   char buf1[100]; /* enough for two numbers */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 0, 0 ) )
      return( 1 );
   dw_pointer_query_pos( &x, &y );
   sprintf( buf1, "%ld %ld", x, y );
   return RxReturnString( RxPackageGlobalData, retstr, buf1 );
}

/****h* Functions/Utility
 * DESCRIPTION
 *   These functions are grouped together becasue they don't fit into any other group.
 ******
 */

/****f* Utility/DW_app_dir
 * NAME
 *   DW_app_dir
 * SYNOPSIS
 *   dir = dw_app_dir( )
 * FUNCTION
 *   Attempts to return the applications data directory.
 * ARGUMENTS
 *   None
 * RESULT
 *   The application data directory.
 * SOURCE
 *
...
dir = dw_app_dir()
Say 'App data dir is:' dir
 ******
 */
rxfunc( DW_app_dir )
{
   char * rc; /* function return variable */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 0, 0 ) )
      return( 1 );
   rc = dw_app_dir( );
   return RxReturnString( RxPackageGlobalData, retstr, rc );
}

/****f* Utility/DW_beep
 * NAME
 *   DW_beep
 * SYNOPSIS
 *   dw_beep( Frequency, Duration )
 * FUNCTION
 *   Makes a sound of the given Frequency for the given Duration.
 * ARGUMENTS
 *   - Frequency - the frequency of the sound
 *   - Duration  - the duration of the sound in milliseconds
 * RESULT
 *   No return result.
 * SOURCE
 *
...
Call dw_beep(5000, 1000)
 ******
 */
rxfunc( DW_beep )
{
   int freq; /* local variable for arg[0] */
   int dur; /* local variable for arg[1] */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 2, 2 ) )
      return( 1 );
   if ( RxStrToInt( RxPackageGlobalData, &argv[0], (RX_INT *)&freq ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   if ( RxStrToInt( RxPackageGlobalData, &argv[1], (RX_INT *)&dur ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 2 );
      return(1);
   }
   dw_beep( freq, dur );
   return RxReturnString( RxPackageGlobalData, retstr, "" );
}

/****f* Utility/DW_debug
 * NAME
 *   DW_debug
 * SYNOPSIS
 *   dw_debug( String )
 * FUNCTION
 *   Displays the supplied String to the system's debugging location.
 *   This is usually the standard error stream.
 * ARGUMENTS
 *   - String - the string to display
 * RESULT
 *   No return result.
 * SOURCE
 *
...
Call dw_debug('This is a debug string')
 ******
 */
rxfunc( DW_debug )
{
   char *str; /* local variable for arg[0] */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 1, 1 ) )
      return( 1 );
   if ( RXVALIDSTRING(argv[0]) )
      str = RXSTRPTR(argv[0]);
   else
      str = "";
   dw_debug( "%s", str );
   return RxReturnString( RxPackageGlobalData, retstr, "" );
}


/****f* Utility/DW_environment_query
 * NAME
 *   DW_environment_query
 * SYNOPSIS
 *   dw_environment_query( Stem )
 * FUNCTION
 *   Returns lots of information about the platform and Dynamic Windows in an array.
 * ARGUMENTS
 *   - Stem      - stem (with trailing period) to place the results
 * RESULT
 *   No return result.
 *   Value of array items:
 *    - Stem.0  - 10 (number of items in array)
 *    - Stem.1  - Operating System Name
 *    - Stem.2  - Operating System Build Date
 *    - Stem.3  - Operating System Build Time
 *    - Stem.4  - Operating System Major Version
 *    - Stem.5  - Operating System Minor Version
 *    - Stem.6  - Operating System Major Build Number
 *    - Stem.7  - Operating System Minor Build Number
 *    - Stem.8  - Dynamic Windows Major Version
 *    - Stem.9  - Dynamic Windows Minor Version
 *    - -Stem.10 - Dynamic Windows Sub Version
 * SOURCE
 *
Call dw_environment_query( 'stem.' )
Say 'OS name is:' stem.1
 ******
 */
rxfunc( DW_environment_query )
{
   DWEnv env;
   char * stem; /* local variable for arg[0] */
   int len,numlen;
   char buf[100];
   char num[20];
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 1, 1 ) )
      return( 1 );
   /*
    * Validate that 'name' is a stem name.
    */
   if ( !RXVALIDSTRING(argv[0]) )
      return(1);

   stem = RXSTRPTR(argv[0]);
   len = RXSTRLEN(argv[0]);

   if ( stem[len-1] != '.' )
      return(1);

   dw_environment_query( &env );

   len = sprintf( buf, "%s1", stem );
   if ( SetRexxVariable( RxPackageGlobalData, buf, len, (char *)env.osName, strlen(env.osName) ) == 1 )
      return(1);
   len = sprintf( buf, "%s2", stem );
   if ( SetRexxVariable( RxPackageGlobalData, buf, len, (char *)env.buildDate, strlen(env.buildDate) ) == 1 )
      return(1);
   len = sprintf( buf, "%s3", stem );
   if ( SetRexxVariable( RxPackageGlobalData, buf, len, (char *)env.buildTime, strlen(env.buildTime) ) == 1 )
      return(1);
   len = sprintf( buf, "%s4", stem );
   numlen = sprintf( num, "%d", env.MajorVersion );
   if ( SetRexxVariable( RxPackageGlobalData, buf, len, num, numlen ) == 1 )
      return(1);
   len = sprintf( buf, "%s5", stem );
   numlen = sprintf( num, "%d", env.MinorVersion );
   if ( SetRexxVariable( RxPackageGlobalData, buf, len, num, numlen ) == 1 )
      return(1);
   len = sprintf( buf, "%s6", stem );
   numlen = sprintf( num, "%d", env.MajorBuild );
   if ( SetRexxVariable( RxPackageGlobalData, buf, len, num, numlen ) == 1 )
      return(1);
   len = sprintf( buf, "%s7", stem );
   numlen = sprintf( num, "%d", env.MinorBuild );
   if ( SetRexxVariable( RxPackageGlobalData, buf, len, num, numlen ) == 1 )
      return(1);
   len = sprintf( buf, "%s8", stem );
   numlen = sprintf( num, "%d", env.DWMajorVersion );
   if ( SetRexxVariable( RxPackageGlobalData, buf, len, num, numlen ) == 1 )
      return(1);
   len = sprintf( buf, "%s9", stem );
   numlen = sprintf( num, "%d", env.DWMinorVersion );
   if ( SetRexxVariable( RxPackageGlobalData, buf, len, num, numlen ) == 1 )
      return(1);
   len = sprintf( buf, "%s10", stem );
   numlen = sprintf( num, "%d", env.DWSubVersion );
   if ( SetRexxVariable( RxPackageGlobalData, buf, len, num, numlen ) == 1 )
      return(1);
   /* set number in array */
   len = sprintf( buf, "%s0", stem );
   numlen = sprintf( num, "%d", 10 );
   if ( SetRexxVariable( RxPackageGlobalData, buf, len, num, numlen ) == 1 )
      return(1);
   return RxReturnString( RxPackageGlobalData, retstr, "" );
}

/****f* Utility/DW_user_dir
 * NAME
 *   DW_user_dir
 * SYNOPSIS
 *   dir = dw_user_dir( )
 * FUNCTION
 *   Returns the user's "home" directory. If the environment variable HOME
 *   is set, this is returned, otherwise "C:\" is returned.
 * ARGUMENTS
 *   None
 * RESULT
 *   The user's home directory.
 * SOURCE
 *
Say 'home is' dw_user_dir()
 ******
 */
rxfunc( DW_user_dir )
{
   char * rc; /* function return variable */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 0, 0 ) )
      return( 1 );
   rc = dw_user_dir( );
   return RxReturnString( RxPackageGlobalData, retstr, rc );
}

/****f* Utility/DW_screen_height
 * NAME
 *   DW_screen_height
 * SYNOPSIS
 *   height = dw_screen_height( )
 * FUNCTION
 *   Returns the height of the physical screen in pixels.
 * ARGUMENTS
 *   None
 * RESULT
 *   The screen height
 * SEE ALSO
 *   DW_screen_width
 * SOURCE
 *
Say 'The height of the screen is' dw_screen_height()
 ******
 */
rxfunc( DW_screen_height )
{
   int rc; /* function return variable */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 0, 0 ) )
      return( 1 );
   rc = dw_screen_height( );
   return RxReturnNumber( RxPackageGlobalData, retstr, rc );
}

/****f* Utility/DW_screen_width
 * NAME
 *   DW_screen_width
 * SYNOPSIS
 *   width = dw_screen_width( )
 * FUNCTION
 *   Returns the width of the physical screen in pixels.
 * ARGUMENTS
 *   None
 * RESULT
 *   The screen width
 * SEE ALSO
 *   DW_screen_height
 * SOURCE
 *
Say 'The width of the screen is' dw_screen_width()
 ******
 */
rxfunc( DW_screen_width )
{
   int rc; /* function return variable */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 0, 0 ) )
      return( 1 );
   rc = dw_screen_width( );
   return RxReturnNumber( RxPackageGlobalData, retstr, rc );
}

/****f* Utility/DW_clipboard_get_text
 * NAME
 *   DW_clipboard_get_text
 * SYNOPSIS
 *   str = dw_clipboard_get_text( )
 * FUNCTION
 *   Returns the textual contents of the default clipboard.
 * ARGUMENTS
 *   None
 * RESULT
 *   The contents of the default clipboard if it can be converted to text.
 * SEE ALSO
 *   DW_clipboard_set_text
 * SOURCE
 *
Say 'The clipboard contains' dw_clipboard_get_text()
 ******
 */
rxfunc( DW_clipboard_get_text )
{
   char *buf;
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 0, 0 ) )
      return( 1 );
   buf = dw_clipboard_get_text( );
   if ( buf == NULL )
      return RxReturnString( RxPackageGlobalData, retstr, "" );
   else
      return RxReturnString( RxPackageGlobalData, retstr, buf );
}

/****f* Utility/DW_clipboard_set_text
 * NAME
 *   DW_clipboard_set_text
 * SYNOPSIS
 *   dw_clipboard_set_text( Str )
 * FUNCTION
 *   Sets the contents of the default clipboard to the specified text.
 * ARGUMENTS
 *   - Str - new clipboard contents
 * RESULT
 *   No return result.
 * SEE ALSO
 *   DW_clipboard_get_text
 * SOURCE
 *
Call dw_clipboard_set_text 'The clipboard contains this.'
 ******
 */
rxfunc( DW_clipboard_set_text )
{
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 1, 1 ) )
      return( 1 );
   dw_clipboard_set_text( RXSTRPTR(argv[0]), RXSTRLEN(argv[0]) );
   return RxReturnString( RxPackageGlobalData, retstr, "" );
}

/****f* Utility/DW_or
 * NAME
 *   DW_or
 * SYNOPSIS
 *   val = dw_or( Val1, Val2[,Val3...]])
 * FUNCTION
 *   Logically "or" multiple values together.
 * ARGUMENTS
 *   - Val1     - value of first numeric value to logically "or"
 *   - Val2     - value of second numeric value to logically "or"
 *   - ...      - optional other numeric values to logically "or"
 * RESULT
 *   The combined value of all arguments logically "or"ed together.
 * SEE ALSO
 *   DW_and
 * SOURCE
 *
...
style = dw_or( !REXXDW.!DW_FCF_SYSMENU, !REXXDW.!DW_FCF_TITLEBAR, ,
   !REXXDW.!DW_FCF_SHELLPOSITION, !REXXDW.!DW_FCF_TASKLIST, ,
   !REXXDW.!DW_FCF_DLGBORDER, !REXXDW.!DW_FCF_SIZEBORDER, ,
   !REXXDW.!DW_FCF_MINMAX )
win = dw_window_new( !REXXDW.!DW_DESKTOP, 'Window on desktop', style )
 ******
 */
rxfunc( DW_or )
{
   unsigned long sum=0L,num,i;
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 2, 0 ) )
      return( 1 );
   for ( i = 0; i < argc; i++ )
   {
      if ( RxStrToULong( RxPackageGlobalData, &argv[i], (RX_ULONG *)&num ) == -1 )
      {
         RxDisplayError( RxPackageGlobalData, name,
                         "*ERROR* Invalid number for argument %d.", i+1 );
         return(1);
      }
      sum |= num;
   }
   return RxReturnNumber( RxPackageGlobalData, retstr, sum );
}

/****f* Utility/DW_and
 * NAME
 *   DW_and
 * SYNOPSIS
 *   val = dw_and( Val1, Val2[,Val3...]])
 * FUNCTION
 *   Logically "and" multiple values together.
 * ARGUMENTS
 *   - Val1     - value of first numeric value to logically "and"
 *   - Val2     - value of second numeric value to logically "and"
 *   - ...      - optional other numeric values to logically "and"
 * RESULT
 *   The combined value of all arguments logically "and"ed together.
 * SEE ALSO
 *   DW_or
 * SOURCE
 *
...
val = dw_and(10, 512345)
 ******
 */
rxfunc( DW_and )
{
   unsigned long sum=0L,num,i;
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 2, 0 ) )
      return( 1 );
   for ( i = 0; i < argc; i++ )
   {
      if ( RxStrToULong( RxPackageGlobalData, &argv[i], (RX_ULONG *)&num ) == -1 )
      {
         RxDisplayError( RxPackageGlobalData, name,
                         "*ERROR* Invalid number for argument %d.", i+1 );
         return(1);
      }
      sum &= num;
   }
   return RxReturnNumber( RxPackageGlobalData, retstr, sum );
}

#ifdef HAVE_ICONV_H
/*
 * Function to convert a string between different character sets based on iconv
 */
rxfunc( DW_convert )
{
   char *type_from, *type_to, *str_from, *str_to;
   iconv_t conv;
   size_t rc,len_from,len_to;
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 3, 3 ) )
      return( 1 );
   if ( RXVALIDSTRING(argv[0]) )
      type_from = RXSTRPTR(argv[0]);
   else
      return(1);
   if ( RXVALIDSTRING(argv[1]) )
      type_to = RXSTRPTR(argv[1]);
   else
      return(1);
   if ( RXVALIDSTRING(argv[2]) )
   {
      str_from = RXSTRPTR(argv[2]);
      len_from = RXSTRLEN(argv[2]);
   }
   else
   {
      str_from = "";
      len_from = 0;
   }
   conv = iconv_open( type_from, type_to );
   if ( conv == (iconv_t)(-1) )
      return(1);
   len_to = 2+(len_from*2);
   str_to = RexxAllocateMemory( len_to );

len_to = len_from;
   if ( str_to == NULL )
      return(1);
   rc = iconv( conv, &str_from, &len_from, &str_to, &len_to );
   if ( rc == (-1) )
   {
      free( str_to );
      iconv_close( conv );
      return(1);
   }
   iconv_close( conv );
   return RxReturnStringAndFree( RxPackageGlobalData, retstr, str_to, 1 );
}
#endif

/****f* Utility/DW_flush
 * NAME
 *   DW_flush
 * SYNOPSIS
 *   dw_flush()
 * FUNCTION
 *   Ensures that any changes drawn to the screen are immediately visible.
 * ARGUMENTS
 *   None
 * RESULT
 *   No return result.
 * SOURCE
 *
...
Call dw_pixmap_bitblt win, , 0, 0, width, height, , pixmap, x, 0
Call dw_flush()
 ******
 */
rxfunc( DW_flush )
{
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 0, 0 ) )
      return( 1 );
   dw_flush( );
   return RxReturnString( RxPackageGlobalData, retstr, "" );
}

/*
 * void : dw_free [void *ptr] ( 1 )
 */
rxfunc( DW_free )
{
   void * ptr; /* local variable for arg[0] */
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );
   if ( my_checkparam( RxPackageGlobalData, name, argc, 1, 1 ) )
      return( 1 );
   if ( RxStrToULong( RxPackageGlobalData, &argv[0], (RX_ULONG *)&ptr ) == -1 )
   {
      RxDisplayError( RxPackageGlobalData, name,
                      "*ERROR* Invalid number for argument %d.", 1 );
      return(1);
   }
   dw_free( ptr );
   return RxReturnString( RxPackageGlobalData, retstr, "" );
}

/****h* Functions/PackageManagement
 * DESCRIPTION
 *   These functions are common to most Rexx external function packages.
 ******
 */

/****f* PackageManagement/DW_loadfuncs
 * NAME
 *   DW_loadfuncs
 * SYNOPSIS
 *   rcode = dw_loadfuncs()
 * FUNCTION
 *   Loads all other RexxDW external functions
 * ARGUMENTS
 *   None
 * RESULT
 *   0 in all cases
 * SEE ALSO
 *   DW_dropfuncs
 * NOTES
 *   Under OS/2, DW_loadfuncs() will set the current process to a PM
 *   application to allow interaction with the OS/2 message queue.
 *   DW_dropfuncs() sets the process back to the type of application
 *   it was started as. If you run a Rexx/DW program as a .CMD file
 *   from a CMD.EXE window, and DO NOT call DW_dropfuncs(), your
 *   CMD.EXE window will be unusable!
 ******
 */
rxfunc( DW_LoadFuncs )
{
   int rc = 0L;
   /*
    * Get thread-safe-data for DLL. EXE global data is not thread-safe
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = GLOBAL_ENTRY_POINT();
   InitRxPackage( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, &rc );
   if ( !QueryRxFunction( RxPackageGlobalData, "DW_MAIN" ) )
   {
      /*
       * Register all external functions
       */
      if ( !rc )
      {
         rc = RegisterRxFunctions( RxPackageGlobalData, RexxDWFunctions, RxPackageName );
      }
   }
   rc = RexxDWInitHandler( 0, 0, NULL );
   return RxReturnNumber( RxPackageGlobalData, retstr, rc );
}

/****f* PackageManagement/DW_dropfuncs
 * NAME
 *   DW_dropfuncs
 * SYNOPSIS
 *   rcode = dw_dropfuncs(["UNLOAD"])
 * FUNCTION
 *   Cleans up RexxDW environment and optionally will drop the
 *   external functions.
 * ARGUMENTS
 *   - UNLOAD - causes the external functions to be dropped.
 * RESULT
 *   0 in all cases
 * SEE ALSO
 *   DW_loadfuncs
 * NOTES
 *   Under OS/2, DW_loadfuncs() will set the current process to a PM
 *   application to allow interaction with the OS/2 message queue.
 *   DW_dropfuncs() sets the process back to the type of application
 *   it was started as. If you run a Rexx/DW program as a .CMD file
 *   from a CMD.EXE window, and DO NOT call DW_dropfuncs(), your
 *   CMD.EXE window will be unusable!
 ******
 */
rxfunc( DW_DropFuncs )
{
   RFH_RETURN_TYPE rc=0;
   int unload=0;
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, ( char* )name, argc, argv );

   if ( my_checkparam( RxPackageGlobalData, name, argc, 0, 1 ) )
      return( 1 );
   if ( argv[0].strlength == 6
   &&   memcmpi( RxPackageGlobalData, argv[0].strptr, "UNLOAD", 6 ) == 0 )
      unload = 1;
   TermRxPackage( RxPackageGlobalData, RexxDWTerminator, RexxDWFunctions, RxPackageName, unload );
   return RxReturnNumber( NULL, retstr, rc );
}

/****f* PackageManagement/DW_variable
 * NAME
 *   DW_variable
 * SYNOPSIS
 *   rcode = dw_variable(Variable [,NewValue])
 * FUNCTION
 *   Get or set an internal RexxDW variable.
 * ARGUMENTS
 *   - Variable - name of the variable to get or set. See NOTES for
 *   - NewValue - the new value of "Variable", if the variable is settable
 * RESULT
 *   When setting a variable, then 0 if success, any other value is an error
 *   When getting a variable, then the value of the variable is returned.
 * NOTES
 *   The "Variable" argument can be one of:
 *      DEBUG (settable)
 *         0 - no debugging
 *         1 - all Rexx variables set by RexxDW are displayed as they are set
 *         2 - all RexxDW functions are traced on entry with argument values and
 *             on exit with the return value
 *         4 - all internal RexxDW functions are traced with their arguments
 *             (really only useful for developers)
 *         The values can be added together for a combination of the above details.
 *      DEBUGFILE (settable)
 *         Where any debugging output is written. By default this goes to
 *         the system's error stream; usually 'stderr'.
 *      CONSTANTPREFIX (settable)
 *         The variable name prefix for all RexxDW constants. By default this is
 *         '!REXXDW.!'. If you change this, it is useful to make the prefix result
 *         in stemmed variables; this makes it far easier to EXPOSE these constants.
 *      VERSION (readonly)
 *         The full version details of RexxDW in the format:
 *         package version version_date
 *         Where:
 *            package      - the string 'rexxdw'
 *            version      - package version in n.n format; eg 1.0
 *            version_date - date package was released in DATE('N') format
 * SOURCE
 *
...
Say 'We are running at debug level:' dw_variable( 'DEBUG' )
 ******
 */
rxfunc( DW_Variable )
{
   RX_ULONG rc = 0L;
   char buf[50];
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );

   if ( my_checkparam( RxPackageGlobalData, name, argc, 1, 2 ) )
      return( 1 );
   if ( argv[0].strlength == 9 && memcmpi( RxPackageGlobalData, "DEBUGFILE", argv[0].strptr, argv[0].strlength ) == 0 )
   {
      if ( argc == 1 )
         return RxReturnString( RxPackageGlobalData, retstr, RxGetTraceFile( RxPackageGlobalData ) );
      else
      {
         rc = RxSetTraceFile( RxPackageGlobalData, (char *)argv[1].strptr );
         return RxReturnNumber( RxPackageGlobalData, retstr, rc );
      }
   }
   else if ( argv[0].strlength == 5 && memcmpi( RxPackageGlobalData, "DEBUG", argv[0].strptr, argv[0].strlength ) == 0 )
   {
      if ( argc == 1 )
      {
         sprintf( buf, "%d", RxGetRunFlags( RxPackageGlobalData ) );
         return RxReturnString( RxPackageGlobalData, retstr, buf );
      }
      else
      {
         if ( RxStrToULong( RxPackageGlobalData, &argv[1], (RX_ULONG *)&rc ) == -1 )
            return RxReturnString( RxPackageGlobalData, retstr, "ERROR: Invalid DEBUG value. Cannot set variable; DEBUG" );
         RxSetRunFlags ( RxPackageGlobalData, rc );
         return RxReturnNumber( RxPackageGlobalData, retstr, 0 );
      }
   }
   else if ( argv[0].strlength == 7 && memcmpi( RxPackageGlobalData, "VERSION", argv[0].strptr, argv[0].strlength ) == 0 )
   {
      if ( argc == 1 )
      {
         sprintf( buf, "%s %s %s", RxPackageName, REXXDW_VERSION, REXXDW_DATE );
         return RxReturnString( RxPackageGlobalData, retstr, buf );
      }
      else
         return RxReturnString( RxPackageGlobalData, retstr, "ERROR: Cannot set variable; VERSION" );
   }
   else if ( argv[0].strlength == 14 && memcmpi( RxPackageGlobalData, "CONSTANTPREFIX", argv[0].strptr, argv[0].strlength ) == 0 )
   {
      if ( argc == 1 )
      {
         return RxReturnString( RxPackageGlobalData, retstr, RxGetConstantPrefix( RxPackageGlobalData ) );
      }
      else
      {
         rc = RxSetConstantPrefix( RxPackageGlobalData, (char *)argv[1].strptr );
         /*
          * Now we have a new prefix, we should reset all our constants
          */
         SetPackageConstants( RxPackageGlobalData, RexxDWConstants, 1 );
         return RxReturnNumber( RxPackageGlobalData, retstr, rc );
      }
   }
   sprintf( buf, "ERROR: Invalid variable; %s", argv[0].strptr );
   return RxReturnString( RxPackageGlobalData, retstr, buf );
}

/****f* PackageManagement/DW_QueryFunction
 * NAME
 *   DW_QueryFunction
 * SYNOPSIS
 *   rcode = dw_QueryFunction(FunctionName|ResultArray[, Option])
 * FUNCTION
 *   Populates an array of all functions supplied by this package depending
 *   on Option
 * ARGUMENTS
 *   - FunctionName - the name of a function to query (no trailing period)
 *   - ResultArray - the stem (trailing period) in which the list of functions is returned
 *   - Option - one of 'R' (the default) for "registered" functions or 'A' for
 *     "available" functions
 * RESULT
 *   0 if successful or 1 if the FunctionName is not found
 * NOTES
 *   To determine if a FunctionName can be executed in your code, pass
 *   the function name as the first argument, and 'R' as the second. If the
 *   function can be called the function returns 0, otherwise it returns 1
 ******
 */
rxfunc( DW_QueryFunction )
{
   ULONG rc = 0L;
   int i,len,tmplen,idx;
   char *stem;
   char optionchar;
   char varname[255], value[10];
   char stem_name[255];
   /*
    * Get thread-safe-data
    */
   RxPackageGlobalDataDef *RxPackageGlobalData = __rxpack_get_tsd();

   RxPackageGlobalData = FunctionPrologue( RxPackageGlobalData, RexxDWInitialiser, RXPACKAGE_CONSTANT_PREFIX, (char *)name, argc, argv );

   if ( my_checkparam( RxPackageGlobalData, name, argc, 1, 2 ) )
      return( 1 );

   stem = RXSTRPTR(argv[0]);
   len = RXSTRLEN(argv[0]);

   if ( argc == 1 )
      optionchar = 'R';
   else
      optionchar = toupper( *argv[1].strptr );

   if ( stem[len-1] != '.' )
   {
      /*
       * Check that the specified function name is available/registered
       */
      rc = 1L; /* not found by default */
      for ( i = 0; RexxDWFunctions[i].ExternalName != NULL; i++ )
      {
         tmplen = strlen( (char *)RexxDWFunctions[i].ExternalName );
         if ( tmplen == len
         &&   memcmpi( RxPackageGlobalData, (char *)RexxDWFunctions[i].ExternalName, stem, len ) == 0 )
         {
            /*
             * Function is available...
             */
            rc = 0L;
            if ( RexxQueryFunction( RexxDWFunctions[i].InternalName )
            &&   optionchar == 'R' )
            {
               /*
                * Function is not registered and we wanted a registered function
                */
               rc = 1L;
            }
            break;
         }
      }
   }
   else
   {
      strcpy( stem_name, stem );
      make_upper( RxPackageGlobalData, stem_name );
      idx = 0;
      for ( i = 0; RexxDWFunctions[i].ExternalName != NULL; i++ )
      {
         if ( RexxQueryFunction( RexxDWFunctions[i].InternalName )
         &&   optionchar == 'R' )
         {
            /*
             * Function is not registered, and we wanted only registered functions
             */
            ;
         }
         else
         {
            idx++;
            len = sprintf( varname, "%s%u", stem_name, idx );
            if ( SetRexxVariable( RxPackageGlobalData, varname, len, (char *)RexxDWFunctions[i].ExternalName, strlen( RexxDWFunctions[i].ExternalName ) ) == 1 )
               return(1);
         }
      }
      i = sprintf( value, "%u", idx);
      len = sprintf(varname, "%s0", stem_name);
      rc = SetRexxVariable( RxPackageGlobalData, varname, len, value, i );
   }
   return RxReturnNumber( NULL, retstr, rc );
}

/*
 * The following functions are used in rxpackage.c
 */

/*-----------------------------------------------------------------------------
 * Print a usage message.
 *----------------------------------------------------------------------------*/
void RexxDWUsage

#ifdef HAVE_PROTO
   (void)
#else
   ()
#endif

{
   (void)fprintf(stderr,
      "\nVersion: %s %s %s\n\nUsage:   %s [-h]\n         %s [-idvf<trace file>] [Rexx program name]\n\n",
      RxPackageName,
      REXXDW_VERSION,
      REXXDW_DATE,
      RxPackageName,
      RxPackageName);
   return;
}

