//
//                     TxWin, Textmode Windowing Library
//
//   Original code Copyright (c) 1995-2021 Fsys Software and Jan van Wijk
//
// ==========================================================================
//
//   TxLib, released under MIT License
//
//   Copyright (c) 1995-2021  Fsys Software and Jan Van Wijk
//
//   Permission is hereby granted, free of charge, to any person obtaining a copy
//   of this software and associated documentation files (the "Software"), to deal
//   in the Software without restriction, including without limitation the rights
//   to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
//   copies of the Software, and to permit persons to whom the Software is
//   furnished to do so, subject to the following conditions:
//
//   The above copyright notice and this permission notice shall be included in all
//   copies or substantial portions of the Software.
//
//   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
//   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
//   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
//   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
//   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
//   OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
//   SOFTWARE.
//
//
//   Questions on TxWin licensing can be directed to: info@dfsee.com
//
// ==========================================================================
//
// TxPrint hook mechanism handling
//
// Author: J. van Wijk
//
// JvW  28-02-2021 LICENSING: Changed from LGPL to the more liberal MIT license
// JvW  19-08-2005 Initial version, split off from TXCON.C

#include <txlib.h>
#include <txtpriv.h>


/*****************************************************************************/
// Perform operations on TxPrint HOOK chain
/*****************************************************************************/
BOOL TxPrintHook                                // RET   success
(
   TXH_TYPE            type,                    // IN    type of handler
   TXH_OPERATION       operation,               // IN    requested operation
   TXH_INFO           *hinfo                    // IN    handler info
)
{
   BOOL                rc = TRUE;
   TXH_INFO          **root;                    // root of handler chain
   TXH_INFO           *this;                    // existing handler if found
   TXH_INFO           *last;                    // insertion point in chain

   ENTER();
   TRARGS(("type: %s, oper %s, hinfo: %8.8lX\n",
          (type      == TXH_T_RAW   ) ? "RAW"      : "CLEAN",
          (operation == TXH_REGISTER) ? "REGISTER" : "DEREGISTER", hinfo));

   switch (type)
   {
      case TXH_T_RAW: root = &txh_raw;   break;
      default:        root = &txh_clean; break;
   }
   for (last = NULL, this  = *root;
                     this != NULL;
        last = this, this  = this->next)
   {
      if (this == hinfo)                        // handler already present
      {
         break;
      }
   }
   switch (operation)
   {
      case TXH_REGISTER:
         if (this == NULL)                      // new handler
         {
            if (last == NULL)                   // first handler
            {
               *root = hinfo;
               hinfo->prev = NULL;
            }
            else                                // another handler
            {
               last->next  = hinfo;
               hinfo->prev = last;
            }
            hinfo->next    = NULL;
         }
         else                                   // already registred
         {
            rc = FALSE;
         }
         break;

      case TXH_DEREGISTER:
         if (this != NULL)                      // existing handler ?
         {
            if (this == *root)                  // first in chain
            {
               *root = this->next;
            }
            else
            {
               this->prev->next = this->next;
            }
            if (this->next != NULL)             // not last in chain ?
            {
               this->next->prev = this->prev;
            }
            hinfo->prev = NULL;
            hinfo->next = NULL;
         }
         else                                   // not registred
         {
            rc = FALSE;
         }
         break;

      default:
         TRACES(("Invalid hook operation: %lu\n", operation));
         break;
   }
   BRETURN(rc);
}                                               // end 'TxPrintHook'
/*---------------------------------------------------------------------------*/

