//
//                     TxWin, Textmode Windowing Library
//
//   Original code Copyright (c) 1995-2021 Fsys Software and Jan van Wijk
//
// ==========================================================================
//
//   TxLib, released under MIT License
//
//   Copyright (c) 1995-2021  Fsys Software and Jan Van Wijk
//
//   Permission is hereby granted, free of charge, to any person obtaining a copy
//   of this software and associated documentation files (the "Software"), to deal
//   in the Software without restriction, including without limitation the rights
//   to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
//   copies of the Software, and to permit persons to whom the Software is
//   furnished to do so, subject to the following conditions:
//
//   The above copyright notice and this permission notice shall be included in all
//   copies or substantial portions of the Software.
//
//   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
//   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
//   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
//   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
//   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
//   OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
//   SOFTWARE.
//
//
//   Questions on TxWin licensing can be directed to: info@dfsee.com
//
// ==========================================================================
//
// CRC32 implementation, compatible with OS/2 LVM data-structure checksums
//
// Author: J. van Wijk
//
//
// JvW  28-02-2021 LICENSING: Changed from LGPL to the more liberal MIT license

#include <txlib.h>                              // TxLib interface

#define DFS_LVM_CRC_RANGE  256
#define DFS_LVM_CRC_POLY   0xEDB88320L
#define DFS_LVM_CRC_INIT   0xFFFFFFFFL

static ULONG          *crcTable = NULL;         // for LVM type CRC

// Allocate and prepare the CRC-table needed for LVM CRC calculation
static void TxCreateLvmCrcTable
(
   void
);


/*****************************************************************************/
// Allocate and prepare the CRC-table needed for LVM CRC calculation
/*****************************************************************************/
static void TxCreateLvmCrcTable
(
   void
)
{
   crcTable = TxAlloc(DFS_LVM_CRC_RANGE, sizeof(ULONG));
   if (crcTable != NULL)
   {
      ULONG            i;
      ULONG            j;
      ULONG            value;

      for ( i = 0; i < DFS_LVM_CRC_RANGE ; i++ )
      {
        value = i;

        for ( j = 8 ; j > 0; j-- )
        {
           if (value & 1)
           {
              value = (value >> 1) ^ DFS_LVM_CRC_POLY;
           }
           else
           {
              value >>= 1;
           }
        }
        crcTable[i] = value;
      }
   }
}                                               // end 'TxCreateLvmCrcTable'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Calculate 32-bit CRC value using LVM compatible algorithm and polynom
/*****************************************************************************/
ULONG TxCalculateLvmCrc
(
   BYTE               *area,                    // IN    data area needing CRC
   ULONG               size                     // IN    size of the data area
)
{
   ULONG               rc = DFS_LVM_CRC_INIT;   // function return, initial

   if (crcTable == NULL)
   {
      TxCreateLvmCrcTable();
   }
   if (crcTable != NULL)
   {
      BYTE            *this = area;             // current byte handled
      ULONG            t1;
      ULONG            t2;                      // intermediate values
      ULONG            i;

      for (this = area, i = 0; i < size; i++, this++)
      {
        t1 = (rc >> 8) & 0xffffff;
        t2 = crcTable[ (rc ^ (ULONG) *this) & (ULONG) 0xff];
        rc = t1 ^ t2;
      }
   }
   return (rc);
}                                               // end 'TxCalculateLvmCrc'
/*---------------------------------------------------------------------------*/

