//
//                     TxWin, Textmode Windowing Library
//
//   Original code Copyright (c) 1995-2021 Fsys Software and Jan van Wijk
//
// ==========================================================================
//
//   TxLib, released under MIT License
//
//   Copyright (c) 1995-2021  Fsys Software and Jan Van Wijk
//
//   Permission is hereby granted, free of charge, to any person obtaining a copy
//   of this software and associated documentation files (the "Software"), to deal
//   in the Software without restriction, including without limitation the rights
//   to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
//   copies of the Software, and to permit persons to whom the Software is
//   furnished to do so, subject to the following conditions:
//
//   The above copyright notice and this permission notice shall be included in all
//   copies or substantial portions of the Software.
//
//   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
//   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
//   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
//   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
//   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
//   OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
//   SOFTWARE.
//
//
//   Questions on TxWin licensing can be directed to: info@dfsee.com
//
// ==========================================================================
//
// TxLib filesystem functions, find file/path section
//
// JvW  28-02-2021 LICENSING: Changed from LGPL to the more liberal MIT license

#include <txlib.h>                              // TxLib interface


/*****************************************************************************/
// Find file in a path specified by environment variable (like PATH)
/*****************************************************************************/
char *TxFindByPath                              // RET   ptr to filename
(
   char               *fname,                   // IN    filename string
   char               *path,                    // IN    env-var with path
   char               *found                    // OUT   found file
)
{
   char               *rc  = NULL;
   char               *p;
   char               *this;
   TXLN                try;
   FILE               *fp;
   char               *ep;
   char               *ec;                      // modifyable copy

   ENTER();
   TRACES(( "Find '%s' in paths specified by: '%s'\n", fname, path));

   if ((ep = getenv( path)) != NULL)
   {
      TRACES(( "path is: '%s'\n", ep));

      if ((ec = TxAlloc( 1, strlen( ep) +1)) != NULL)
      {
         p = strcpy( ec, ep);
         do
         {
            this = p;
            p = strchr( p, FS_ENVP_SEP);
            if (p != NULL)
            {
               *p++ = '\0';
            }
            strcpy( try, this);
            strcat( try, FS_PATH_STR);
            strcat( try, fname);
            TRACES(( "Trying fspec: '%s'\n", try));
            if ((fp = fopen( try, "r")) != NULL)
            {
               fclose( fp);
               strcpy( found, try);
               rc = found;
            }
         } while ((p != NULL) && (rc == NULL));
         TxFreeMem( ec);
      }
   }
   RETURN( rc);
}                                               // end 'TxFindByPath'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Find file in current-dir, exe-dir or PATH, and when found, open for read
/*****************************************************************************/
FILE *TxFindAndOpenFile                         // RET   opened file handle
(
   char               *fname,                   // IN    filename string
   char               *path,                    // IN    env-var with path
   char               *found                    // OUT   found file
)
{
   TXLN                try;
   FILE               *fp;
   char               *pp;

   ENTER();

   TRACES(( "Find file: '%s' using path: '%s'\n", fname, path));

   strcpy(  try, fname);
   TxRepl(  try, FS_PALT_SEP, FS_PATH_SEP);     // fixup ALT separators
   if ((fp = fopen( try, "r")) == NULL)         // try full-spec/current dir
   {
      TRACES(( "FAIL in current dir or absolute path: '%s'\n", try));

      if (TxaExeSwitchSet(TXA_O_DIR))           // explicit files dir given
      {
         strcpy( try, TxaExeSwitchStr(TXA_O_DIR, "", ""));
         if (try[strlen(try)-1] != FS_PATH_SEP) // fspec does not end in SEP
         {
            strcat( try, FS_PATH_STR);          // append directory separator
         }
         strcat( try, fname);                   // append fname
         fp = fopen( try, "r");
      }
      if (fp == NULL)                           // not found yet ...
      {
         strcpy( try, TxaExeArgv(0));
         if ((pp = strrchr( try, FS_PATH_SEP)) != NULL)
         {
            *(pp +1) = 0;                       // try executable directory
         }
         else                                   // if no executable DIR known,
         {
            strcpy( try, FS_PATH_STR);          // try root directory
         }
         strcat( try, fname);                   // append fname

         if ((fp = fopen( try, "r")) == NULL)   // not here either, try PATH
         {
            TRACES(( "FAIL in EXE-directory or root:'%s'\n", try));
            if (TxFindByPath( fname, "PATH", try) != NULL)
            {
               fp = fopen( try, "r");
            }
            else
            {
               TRACES(( "FAIL in paths for env-var: '%s'\n", "PATH"));
            }
         }
      }
   }
   if (fp!= NULL)                               // found the file
   {
      strcpy( found, try);                      // set the found filename
      TRACES(("found file: '%s'\n", found));
   }                                            // and return the (read) fp
   RETURN (fp);
}                                               // end 'TxFindAndOpenFile'
/*---------------------------------------------------------------------------*/

